#!/usr/bin/perl
#
# See COPYRIGHT
#
# Script to generate a pod file from an html source (the same one as for text files too)
# and later this pod file it passed through pod2man
#
# Use:
#  html2man [ <man-dir> [<version-dir>] ] <file.html
#
# <Man-dir> is the directory where the man pages will be created
# (current directory by default). If a file name is given instead of
# directory then the directory of that file is used.
# <Version-dir> is the directory containing the ttf2pt1 files version.h
# and CHANGES.html which are used to generate the release name and date
# for the man page (by default looks in current directory and then in up to
# 5 ancestor directories).
# If the version files can not be found then the release defaults to
# "current" and the date defaults to today.
#
# Special formatting in the html file is:
# All controls are hidden within HTML comments that must occupy a whole separate line
# Such a line looks like:
# <!-- =<html2man_directive> <arguments> -->
# <!-- ==<pod_directive> <arguments> -->
# Any sort of directive must be followed by a space. The pod directives are
# automatically surrounded by empty lines in the output file.
# The html2man directives are:
#
# <!-- =defdoc <docid> <file> <section> -->
# Define a man page. Multiple man pages can be defined in the same HTML
# file. <Docid> is a short name by which this man page will be referred in the
# other directives. <File> is the name of the man page, and <section> is the
# section of the manual (do not confuse with sections within a man page).
#
# <!-- =section <docid> <page_section_name> -->
# All the text following this directive is copied (with translation)
# into the specified section of the specified man page. The sections
# may appear in arbitrary order, they will be rearranged to the standard
# order before output. Only standard section names are permitted (see @stdsect
# below). The pod directives which occur outside of man sections are ignored,
# just like the common text. The translation of HTML tags is:
#
# <br> - to paragraph break
# <b> - to B<>
# <i> - to I<>
# <tt> - to C<>
# <a href> - to F<>
# <ul>, <li>, </ul> - to =over 2, =item *, =back
# &nbsp;, &amp;, &lt;, &gt - to their symbols, appropriately encoded
#
# The rest of HTML tags is removed
#
# If the same section is started more than once, the text from the
# second appearance will be added to the first, etc.
#
# <!-- =stop -->
# Stop copying text to the man page.
#
# <!-- =cont -->
# Continue copying text to the man page, same section as before.
#
# <!-- =text <text> -->
# Insert this <text> into the man page (works only when copying is enabled).
# Characters &lt;, &gt;, &amp; are converted as usual.

use strict;
use Getopt::Std;
use vars qw($opt_d $opt_m);

my @mons = qw(January February March April May June July August September October November December);

getopts('d:m:');

my $dir = $opt_d;
my $maindir = $opt_m;

if($dir eq "") {
	$dir = ".";
} elsif( ! -d $dir ) {
	if( ! ($dir =~ s|\/[^/]*$||) ) {
		$dir = ".";
	}
}
if($maindir eq "") {
	$maindir = ".";
	for(my $i=0; $i<5; $i++) {
		if(-f "$maindir/version.h") {
			last;
		}
		$maindir = "../$maindir";
	}
}

my $release;
my $date;
if( open(VERFILE, "<$maindir/version.h") ) {
	while(<VERFILE>) {
		if( /^\s*\#define\s+TTF2PT1_VERSION\s+\"(.*)\"/ ) {
			$release = "version $1";
		}
	}
	close(VERFILE);
	if( $release =~ /SNAP-([0-9][0-9])([0-9][0-9])([0-9][0-9])/ ) {
		$date = sprintf("%s %d, 20%02d", $mons[$2-1], $3, $1);
	} elsif( open(CFILE, "<$maindir/CHANGES.html") ) {
		while(<CFILE>) {
			if( /\<H4\>/) {
				last;
			}
		}
		$_ = <CFILE>;
		chomp;
		if( $_ =~ s/^.*?-- // ) {
			$date = $_;
		}
		close(CFILE);
	}
}

if($release eq "") {
	$release = "current";
}
if($date eq "") {
	my @lt = localtime(time);
	$date = sprintf("%s %d, %d", $mons[$lt[4]], $lt[3], 1900+$lt[5]);
}

#printf(STDERR "date=%s release=%s\n", $date, $release);

my $writemode = 0;
my %text;
my @allids;
my $tosect;
my %file;
my %mansect;

if($ARGV[0] =~ /\.pl$/) {
#  print "Ignoring $ARGV[0]\n";
  shift;
}

while(<>) {
	if( s/^\<\!\-\- \=(\S+)\s+//) { 
		my $cmd = $1;
		s/\s*--\>\s*$//;
		#printf(STDERR "cmd=%s args=%s\n", $cmd, $_);
		if($cmd =~ /^=/) {
			if($writemode) {
				$text{$tosect} .= "\n\n$cmd $_\n\n";
			}
		} elsif($cmd eq "defdoc") {
			my @sl = split;
			push(@allids, $sl[0]);
			$file{$sl[0]} = $sl[1];
			$mansect{$sl[0]} = $sl[2];
		} elsif($cmd eq "section") {
			# tosect includes the file id
			$tosect = $_;
			$text{$tosect} .= "\n\n";
			$writemode = 1;
		} elsif($cmd eq "stop") {
			$writemode = 0;
			$text{$tosect} .= "\n";
		} elsif($cmd eq "cont") {
			$writemode = 1;
		} elsif($cmd eq "text") {
			if($writemode) {
				s/\&lt\;/</gi;
				s/\&gt\;/>/gi;
				s/\&amp\;/\&/gi;
				$text{$tosect} .= "$_\n";
			}
		}
	} elsif($writemode) {
		s/^\s+//;

		s/\{/\&lbr;/g;
		s/\}/\&rbr;/g;

		s/\<br\>/\n\n/gi;
		#s/\<blockquote\>/\n\n=over 4\n\n/gi;
		#s/\<\/blockquote\>/\n\n=back\n\n/gi;
		s/\<ul\>/\n\n=over 4\n\n/gi;
		s/\<\/ul\>/\n\n=back\n\n/gi;
		s/\<dl\>/\n\n=over 4\n\n/gi;
		s/\<p\>/\n\n/gi;
		s/\<\/dl\>/\n\n=back\n\n/gi;
		s/\<li\>\s*/\n\n=item \*\n\n/gi;
		s/\<dt\>\s*(.*)/\n\n=item $1\n\n/gi;
		s/\<h2\>(.*)\<\/h2\>/\n\n=head2 $1\n\n/gi;
		s/\<i\>(.*?)\<\/i\>/I\{$1\}/gi;
		s/\<b\>(.*?)\<\/b\>/B\{$1\}/gi;
		s/\<tt\>(.*?)\<\/tt\>/C\{$1\}/gi;
		s/\<a href\=\.*?\>(.*?)\<\/a\>/F\{$1\}/gi;
		s/\<.*?\>//g;
		s/\{/\</g;
		s/\}/\>/g;

		s/\&nbsp\;/S< >/gi;
		s/\&amp\;/\&/gi;
		s/\&lt\;/E<lt>/gi;
		s/\&gt\;/E<gt>/gi;
		#s/\|/E<verbar>/g;
		#s/\//E<sol>/g;
		s/\&lbr\;/\{/g;
		s/\&rbr\;/\}/g;

		#printf(STDERR "section=%s add=%s", $tosect, $_);
		$text{$tosect} .= $_;
	}
}

my @stdsect = (
	"NAME", 
	"SYNOPSIS",
	"DESCRIPTION",
	"OPTIONS",
	"RETURN VALUE",
	"ERRORS",
	"EXAMPLES",
	"ENVIRONMENT",
	"FILES",
	"SEE ALSO",
	"NOTES",
	"CAVEATS",
	"DIAGNOSTICS",
	"BUGS", 
	"RESTRICTIONS",
	"AUTHOR",
	"HISTORY" );

#printf(STDERR "allids= @allids\n");
for my $id (@allids) {
	#print(STDERR "creating man page $id $file{$id} $mansect{$id}\n\n");
	die "Unable to create pod file $dir/$file{$id}.pod"
		unless open(PODF, ">$dir/$file{$id}.pod");
	print(PODF "=pod\n\n");
	for my $sect (@stdsect) {
		my $sid = "$id $sect";
		#printf(STDERR "trying %s\n", $sid);
		if(defined $text{$sid}) {
			print(PODF "=head1 $sect\n\n$text{$sid}\n\n");
		}
	}
	print(PODF "=cut\n");
	close(PODF);
	die "Unable to generate the man page $dir/$file{$id}.1"
		if system("pod2man --section=\"$mansect{$id}\" --release=\"$release\" "
		. "--center=\"Udpcast\" --date=\"$date\" "
		. "$dir/$file{$id}.pod >$dir/$file{$id}.1");

	unlink("$dir/$file{$id}.pod");
}
