'''
 ====================================================================
 Copyright (c) 2006 Barry A Scott.  All rights reserved.

 This software is licensed as described in the file LICENSE.txt,
 which you should have received as part of this distribution.

 ====================================================================

    wb_subversion_checkin.py

'''
import wx
import sys
import wb_images
import time
import pysvn

import wb_ids
import wb_images
import wb_exceptions
import wb_list_panel_common
import wb_subversion_utils
import wb_subversion_list_handler_common
import wb_platform_specific

id_exclude = wx.NewId()
id_include = wx.NewId()

class CheckinFrame(wx.Frame):
    def __init__( self, app, project_info, all_files ):
        wx.Frame.__init__( self, None, -1, 'Check in for %s' % project_info.wc_path, size=(700,500) )

        self.app = app

        self.menu_actions = wx.Menu()
        self.menu_actions.Append(  wb_ids.id_File_Edit, 'Edit', 'Edit' )
        if wx.Platform in ['__WXMSW__','__WXMAC__']:
            self.menu_actions.Append(  wb_ids.id_Shell_Open, 'Open', 'Open' )
        self.menu_actions.AppendSeparator()
        self.menu_actions.Append(  wb_ids.id_SP_DiffWorkBase, 'Diff WC vs. BASE...', 'Diff WC vs. BASE...' )
        self.menu_actions.Append(  wb_ids.id_SP_DiffWorkHead, 'Diff WC vs. HEAD...', 'Diff WC vs. HEAD...' )
        self.menu_actions.Append(  wb_ids.id_SP_DiffWorkBranchOriginBase, 'Diff WC vs. branch origin BASE...', 'Diff WC vs. branch origin BASE...' )
        self.menu_actions.Append(  wb_ids.id_SP_DiffWorkBranchOriginHead, 'Diff WC vs. branch origin HEAD...', 'Diff WC vs. branch origin HEAD...' )

        self.menu_actions.AppendSeparator()
        self.menu_actions.Append( wb_ids.id_SP_Annotate, 'Annotate...', 'Annotate...' )
        self.menu_actions.Append( wb_ids.id_SP_History, 'Log history...', 'Log history...' )
        self.menu_actions.Append( wb_ids.id_SP_Info, 'Information...', 'Information...' )
        self.menu_actions.Append( wb_ids.id_SP_Properties, 'Properties...', 'Properties...' )

        self.menu_actions.AppendSeparator()
        self.menu_actions.Append( id_exclude, 'Exclude...', 'Exclude from check in' )
        self.menu_actions.Append( id_include, 'Include...', 'Include from check in' )

        self.menu_bar = wx.MenuBar()
        self.menu_bar.Append( self.menu_actions, "&Actions" )

        self.SetMenuBar( self.menu_bar )

        # Add tool bar
        t = self.CreateToolBar( name="main",
                                style=wx.TB_HORIZONTAL ) # | wx.NO_BORDER | wx.TB_TEXT )

        bitmap_size = (32,32)
        t.SetToolBitmapSize( bitmap_size )
        t.AddSimpleTool( wb_ids.id_File_Edit,
            wb_images.getBitmap( 'toolbar_images/edit.png', bitmap_size ),
            'Edit File', 'Edit File' )
        if wx.Platform in ['__WXMSW__','__WXMAC__']:
            t.AddSimpleTool( wb_ids.id_Shell_Open,
                wb_images.getBitmap( 'toolbar_images/open.png', bitmap_size ),
                'Open File', 'Open File' )
        t.AddSeparator()
        t.AddSimpleTool( wb_ids.id_SP_DiffWorkBase,
            wb_images.getBitmap( 'toolbar_images/diff.png', bitmap_size ),
            'Diff changes against base', 'Diff changes against base' )
        t.AddSimpleTool( wb_ids.id_SP_History,
            wb_images.getBitmap( 'toolbar_images/history.png', bitmap_size ),
            'Show History log', 'Show History log' )
        t.AddSimpleTool( wb_ids.id_SP_Info,
            wb_images.getBitmap( 'toolbar_images/info.png', bitmap_size ),
            'File Information', 'File Information' )
        t.AddSimpleTool( wb_ids.id_SP_Properties,
            wb_images.getBitmap( 'toolbar_images/property.png', bitmap_size ),
            'File Properties', 'File Properties' )
        t.AddSeparator()
        t.AddSimpleTool( id_exclude,
            wb_images.getBitmap( 'toolbar_images/exclude.png', bitmap_size ),
            'Exclude from check in', 'Exclude from check in' )
        t.AddSimpleTool( id_include,
            wb_images.getBitmap( 'toolbar_images/include.png', bitmap_size ),
            'Include in check in', 'Include in check in' )
        t.Realize()


        try_wrapper = wb_exceptions.TryWrapperFactory( self.app.log )

        # Set the application icon
        self.SetIcon( wb_images.getIcon( 'wb.png' ) )

        self.message_filename = wb_platform_specific.getLastCheckinMessageFilename()
        self.last_log_message_text = None

        try:
            f = file( self.message_filename, 'r' )
            self.last_log_message_text = f.read().decode('utf-8').strip()
            f.close()
        except EnvironmentError:
            self.last_log_message_text = ''

        # Create the splitter windows
        self.splitter = wx.SplitterWindow( self, -1 )

        # Make sure the splitters can't be removed by setting a minimum size
        self.splitter.SetMinimumPaneSize( 100 )   # list, log

        # create the individule panels
        self.panel_list = CheckinListPanel( app, self, self.splitter )

        # Create the log message panel
        self.panel_log = wx.Panel( self.splitter, -1 )
        self.v_sizer = wx.BoxSizer( wx.VERTICAL )
        self.h_sizer = wx.BoxSizer( wx.HORIZONTAL )

        self.log_message_ctrl = wx.TextCtrl( self.panel_log, -1, style=wx.TE_MULTILINE )

        self.button_ok = wx.Button( self.panel_log, wx.ID_OK, " Check In " )
        self.button_cancel = wx.Button( self.panel_log, wx.ID_CANCEL, " Cancel " )
        self.button_ok.SetDefault()

        self.button_last_log_message = wx.Button( self.panel_log, -1, "Insert Last Message" )
        self.button_last_log_message.Enable( len(self.last_log_message_text) > 0 )
        self.h_sizer.Add( self.button_last_log_message )

        self.button_ok.Enable( False )

        wx.EVT_BUTTON( self.panel_log, self.button_last_log_message.GetId(), try_wrapper( self.OnInsertLastLogMessage ) )
        wx.EVT_TEXT( self.panel_log, self.log_message_ctrl.GetId(), try_wrapper( self.OnLogMessageChanged ) )

        self.h_sizer.Add( (60, 20), 1, wx.EXPAND)
        self.h_sizer.Add( self.button_ok, 0, wx.EXPAND|wx.EAST, 15)
        self.h_sizer.Add( self.button_cancel, 0, wx.EXPAND|wx.EAST, 2 )

        self.v_sizer.Add( self.log_message_ctrl, 1, wx.EXPAND|wx.ALL, 5 )

        self.v_sizer.Add( self.h_sizer, 0, wx.EXPAND|wx.ALL, 5 )

        self.splitter.SplitHorizontally( self.panel_list, self.panel_log, -150 )

        self.panel_log.SetAutoLayout( True )
        self.panel_log.SetSizer( self.v_sizer )
        self.v_sizer.Fit( self.panel_log )
        self.panel_log.Layout()

        wx.EVT_CLOSE( self, self.OnCloseWindow )

        wx.EVT_BUTTON( self.panel_log, wx.ID_OK, self.app.eventWrapper( self.OnOk ) )
        wx.EVT_BUTTON( self.panel_log, wx.ID_CANCEL, try_wrapper( self.OnCancel ) )

        self.project_info = CheckinProjectInfo( project_info, all_files )
        self.list_handler = CheckinListHandler( self.app, self.panel_list, self.project_info )

        # draw the list - its updates the status info
        self.panel_list.setHandler( self.list_handler )

        wx.EVT_MENU( self, wb_ids.id_File_Edit, try_wrapper( self.OnFileEdit ) )
        wx.EVT_MENU( self, wb_ids.id_Shell_Open, try_wrapper( self.OnShellOpen ) )

        wx.EVT_MENU( self, wb_ids.id_SP_Annotate, self.app.eventWrapper( self.OnSpAnnotate ) )
        wx.EVT_MENU( self, wb_ids.id_SP_DiffWorkBase, self.app.eventWrapper( self.OnSpDiffWorkBase ) )
        wx.EVT_MENU( self, wb_ids.id_SP_DiffWorkHead, self.app.eventWrapper( self.OnSpDiffWorkHead ) )
        wx.EVT_MENU( self, wb_ids.id_SP_DiffWorkBranchOriginBase, self.app.eventWrapper( self.OnSpDiffWorkBranchOriginBase ) )
        wx.EVT_MENU( self, wb_ids.id_SP_DiffWorkBranchOriginHead, self.app.eventWrapper( self.OnSpDiffWorkBranchOriginHead ) )
        wx.EVT_MENU( self, wb_ids.id_SP_History, self.app.eventWrapper( self.OnSpHistory ) )
        wx.EVT_MENU( self, wb_ids.id_SP_Info, self.app.eventWrapper( self.OnSpInfo ) )
        wx.EVT_MENU( self, wb_ids.id_SP_Properties, self.app.eventWrapper( self.OnSpProperties ) )

        wx.EVT_MENU( self, id_exclude, self.app.eventWrapper( self.OnExcludeItem ) )
        wx.EVT_MENU( self, id_include, self.app.eventWrapper( self.OnIncludeItem ) )

        self.log_message_ctrl.SetFocus()

    def OnExcludeItem( self, event ):
        self.list_handler.Cmd_Checkin_ExcludeItem( self.panel_list.getSelectedRows() )
        self.panel_list.drawList()

    def OnIncludeItem( self, event ):
        self.list_handler.Cmd_Checkin_IncludeItem( self.panel_list.getSelectedRows() )
        self.panel_list.drawList()

    def clearUpdateUiState( self ):
        pass

    def getUpdateUiState( self ):
        pass

    def setEventHandler( self, handler ):
        self.handler = handler

    def OnInsertLastLogMessage( self, event ):
        self.log_message_ctrl.WriteText( self.last_log_message_text )
        self.button_ok.Enable( True )

    def OnLogMessageChanged( self, event ):
        self.button_ok.Enable( len( self.log_message_ctrl.GetValue().strip() ) > 0 )

    def OnCloseWindow( self, event ):
        self.Destroy()

    def OnCancel( self, event ):
        self.Destroy()

    def OnOk( self, event ):
        self.Hide()

        message = self.log_message_ctrl.GetValue().encode('utf-8')
        try:
            f = file( self.message_filename, 'w' )
            f.write( message )
            f.close()
        except EnvironmentError:
            pass

        all_filenames = self.list_handler.getCheckinFiles()
        self.app.setAction( 'Check in %s...' % self.project_info.wc_path )
        self.app.setProgress( 'Sent %(count)d of %(total)d', len( all_filenames ) )

        yield self.app.backgroundProcess

        ok = False
        try:
            rev = self.project_info.client_bg.checkin( all_filenames, message )
            ok = True
        except pysvn.ClientError, e:
            self.app.log_client_error( e )

        yield self.app.foregroundProcess

        if ok:
            if rev:
                self.app.log.info( 'Checkin created revision %d' % rev.number )
            else:
                self.app.log.warning( 'No changes to checkin ' )

        self.app.refreshFrame()
        self.app.setAction( 'Ready' )
        self.app.clearProgress()

        self.Destroy()

    # command events
    def OnFileEdit( self, event ):
        return self.panel_list.OnFileEdit()

    def OnShellOpen( self, event ):
        return self.panel_list.OnShellOpen()

    def OnSpAnnotate( self, event ):
        return self.panel_list.OnSpAnnotate()

    def OnSpDiffWorkBase( self, event ):
        return self.panel_list.OnSpDiffWorkBase()

    def OnSpDiffWorkHead( self, event ):
        return self.panel_list.OnSpDiffWorkHead()

    def OnSpDiffWorkBranchOriginBase( self, event ):
        return self.panel_list.OnSpDiffWorkBranchOriginBase()

    def OnSpDiffWorkBranchOriginHead( self, event ):
        return self.panel_list.OnSpDiffWorkBranchOriginHead()

    def OnSpHistory( self, event ):
        return self.panel_list.OnSpHistory()

    def OnSpInfo( self, event ):
        return self.panel_list.OnSpInfo()

    def OnSpProperties( self, event ):
        return self.panel_list.OnSpProperties()

class CheckinListHandler(wb_subversion_list_handler_common.SubversionListHandlerCommon):
    def __init__( self, app, parent, project_info ):
        wb_subversion_list_handler_common.SubversionListHandlerCommon.__init__( self, app, parent, project_info )

        self.all_excluded_files = {}

    def getContextMenu( self ):
        menu_template = \
            [('', wb_ids.id_File_Edit, 'Edit' )]
        if wx.Platform in ['__WXMSW__','__WXMAC__']:
            menu_template += \
                [('', wb_ids.id_Shell_Open, 'Open' )]
        menu_template += \
            [('-', 0, 0 )
            ,('', wb_ids.id_SP_DiffWorkBase, 'Diff WC vs. BASE...' )
            ,('', wb_ids.id_SP_DiffWorkHead, 'Diff WC vs. HEAD...' )
            ,('', wb_ids.id_SP_DiffWorkBranchOriginBase, 'Diff WC vs. branch origin BASE...' )
            ,('', wb_ids.id_SP_DiffWorkBranchOriginHead, 'Diff WC vs. branch origin HEAD...' )
            ,('-', 0, 0 )
            ,('', wb_ids.id_SP_Annotate, 'Annotate...' )
            ,('', wb_ids.id_SP_History, 'Log history...' )
            ,('', wb_ids.id_SP_Info, 'Information...' )
            ,('', wb_ids.id_SP_Properties, 'Properties...' )
            ]

        return wb_subversion_utils.populateMenu( wx.Menu(), menu_template )

    def Cmd_Checkin_ExcludeItem( self, all_rows ):
        #print 'Cmd_Checkin_ExcludeItem',all_rows
        for row in all_rows:
            self.all_excluded_files[ self.getFilename( row ) ] = None

    def Cmd_Checkin_IncludeItem( self, all_rows ):
        #print 'Cmd_Checkin_IncludeItem',all_rows
        for row in all_rows:
            if self.getFilename( row ) in self.all_excluded_files:
                del self.all_excluded_files[ self.getFilename( row ) ]

    def getAllGreyFilenames( self ):
        # show all excluded files in grey
        return self.all_excluded_files

    def getCheckinFiles( self ):
        return [entry.path for entry in self.project_info.all_files
                if entry.path not in self.all_excluded_files]

class CheckinProjectInfo:
    def __init__( self, project_info, all_files ):
        self.all_files = all_files
        self.need_properties = False
        self.project_name = project_info.project_name
        self.url = project_info.url
        self.wc_path = project_info.wc_path
        self.need_checkout = False

        self.client_fg = project_info.client_fg
        self.client_bg = project_info.client_bg

    def getTagsUrl( self, rel_url ):
        return None

    def setNeedProperties( self, need_properties ):
        self.need_properties = need_properties

    def updateStatus( self ):
        pass

    def getFilesStatus( self ):
        return self.all_files

    def getProperty( self, filename, prop_name ):
        return ''

    def getWorkingDir( self ):
        return self.wc_path

class CheckinListPanel(wb_list_panel_common.WbListPanelCommon):
    def __init__( self, app, frame, parent ):
        wb_list_panel_common.WbListPanelCommon.__init__( self, app, frame, parent )

    def getAcceleratorTableInit( self ):
        if wx.Platform == '__WXMAC__':
            acc_init =[
                (wx.ACCEL_ALT, ord('D'), wb_ids.id_SP_DiffWorkBase),
                (wx.ACCEL_ALT, ord('E'), wb_ids.id_File_Edit),
                (wx.ACCEL_ALT, ord('L'), wb_ids.id_SP_History),
                (wx.ACCEL_ALT, ord('I'), wb_ids.id_SP_Info),
                (wx.ACCEL_ALT, ord('P'), wb_ids.id_SP_Properties),
                (wx.ACCEL_ALT, ord('O'), wb_ids.id_Shell_Open),
                ]
        elif wx.Platform == '__WXMSW__':
            acc_init =[
                (wx.ACCEL_CTRL, ord('D'), wb_ids.id_SP_DiffWorkBase),
                (wx.ACCEL_CTRL, ord('E'), wb_ids.id_File_Edit),
                (wx.ACCEL_CTRL, ord('L'), wb_ids.id_SP_History),
                (wx.ACCEL_CTRL, ord('I'), wb_ids.id_SP_Info),
                (wx.ACCEL_CTRL, ord('P'), wb_ids.id_SP_Properties),
                (wx.ACCEL_CTRL, ord('O'), wb_ids.id_Shell_Open),
                ]
        else:
            # Unix
            acc_init =[
                (wx.ACCEL_CTRL, ord('D'), wb_ids.id_SP_DiffWorkBase),
                (wx.ACCEL_CTRL, ord('E'), wb_ids.id_File_Edit),
                (wx.ACCEL_CTRL, ord('L'), wb_ids.id_SP_History),
                (wx.ACCEL_CTRL, ord('I'), wb_ids.id_SP_Info),
                (wx.ACCEL_CTRL, ord('P'), wb_ids.id_SP_Properties),
                ]

        return acc_init
