'''
 ====================================================================
 Copyright (c) 2003-2005 Barry A Scott.  All rights reserved.

 This software is licensed as described in the file LICENSE.txt,
 which you should have received as part of this distribution.

 ====================================================================

    wb_subversion_list_handler.py

'''
import os
import sys
import pysvn
import wx
import wb_subversion_history
import wb_subversion_annotate
import wb_list_panel
import wb_ids
import wb_exceptions
import wb_subversion_info_dialog
import wb_subversion_properties_dialog
import wb_subversion_utils
import wb_read_file

col_labels = [    ('Name',   25, 10, 100, wx.LIST_FORMAT_LEFT),
        ('State',   4,  2,   8, wx.LIST_FORMAT_LEFT),
        ('Date',   12,  4,  20, wx.LIST_FORMAT_RIGHT),
        ('Rev',     4,  2,   8, wx.LIST_FORMAT_RIGHT),
        ('Author', 10,  4,  80, wx.LIST_FORMAT_LEFT),
#        ('Size',    6,  4,  10, wx.LIST_FORMAT_RIGHT),
        ('Mimetype',10,  6,  50, wx.LIST_FORMAT_LEFT),
        ('EOL',     6,  2,  10, wx.LIST_FORMAT_LEFT),
        ('Type',    4,  4,  10, wx.LIST_FORMAT_LEFT)]

class ColumnInfo:
    def __init__( self, name, width, min_width, max_width, alignment ):
        self.name = name
        self.width = int(width)
        self.min_width = int(min_width)
        self.max_width = int(max_width)
        self.alignment = alignment
        self.included = False
        self.column = 0

class ViewColumnInfo:
    def __init__( self ):
        self.column_info = {}
        for col_info in col_labels:
            name, width, min_width, max_width, alignment = col_info
            self.column_info[ name ] = ColumnInfo( name, width, min_width, max_width, alignment )

    def setFromPreferenceData( self, p ):
        column_order = p.column_order[:]
        column_widths = p.column_widths[:]

        # must have Name
        if 'Name' not in column_order:
            column_order.insert( 0, 'Name' )
            column_widths.insert( 0, '0' )

        for index, name in enumerate( column_order ):
            if self.column_info.has_key( name ):
                if len(column_widths) > index:
                    try:
                        width = int(column_widths[index])
                        if( width >= self.column_info[ name ].min_width
                        and width <= self.column_info[ name ].max_width ):
                            self.column_info[ name ].width = width

                    except ValueError:
                        pass

        self.setColumnOrder( column_order )

    def getInfo( self, name ):
        return self.column_info[ name ]

    def excludedInfo( self ):
        return [info for info in self.column_info.values() if not info.included]

    def setColumnOrder( self, column_order ):
        self.column_order = column_order[:]
        for index, name in enumerate( self.column_order ):
            self.column_info[ name ].column = index

    def getColumnOrder( self ):
        return self.column_order

    def getColumnWidths( self ):
        return [str(self.column_info[name].width) for name in self.column_order]

class SubversionListHandler(wb_list_panel.ListHandler):
    col_name = 'Name'
    col_state = 'State'
    col_date = 'Date'
    col_revision = 'Rev'
    col_author = 'Author'
    col_type = 'Type'
    col_size = 'Size'
    col_mime_type = 'Mimetype'
    col_eol_style = 'EOL'

    def __init__( self, app, list_panel, project_info ):
        wb_list_panel.ListHandler.__init__( self, list_panel )
        self.project_info = project_info
        self.app = app

        self.all_files = []
        self.date_font = wx.Font( 8, wx.DEFAULT, wx.NORMAL, wx.NORMAL, False, 'Courier New' )

        self.column_info = ViewColumnInfo()
        self.need_properties = False

        self.is_project_parent = False

    def setIsProjectParent( self ):
        self.is_project_parent = True

    def isProjectParent( self ):
        return self.is_project_parent

    def updateStatus( self ):
        self.project_info.updateStatus()

    def getProjectInfo( self ):
        return self.project_info

    def setupColumns( self ):
        self.column_info.setFromPreferenceData( self.app.prefs.getView() )

        g = self.list_panel.list_ctrl

        # should update in place in case the list was used by other code
        while g.GetColumnCount() > 0:
            g.DeleteColumn( 0 )

        need_properties = False

        char_width = 11
        for index, name in enumerate( self.column_info.getColumnOrder() ):
            info = self.column_info.getInfo( name )
            g.InsertColumn( index, info.name, info.alignment, info.width*char_width )
            if name in [self.col_mime_type, self.col_eol_style]:
                need_properties = True

        self.project_info.setNeedProperties( need_properties )

    def initList( self, sort_data, filter_field, filter_text ):
        self.app.trace.info( 'SubversionListHandler.initList %s', self.project_info.project_name )

        self.list_panel.updateHeader( self.project_info.url, self.project_info.wc_path )
         
        # nothing doing if the wc does not exist
        if self.project_info.need_checkout:
            self.app.trace.info( 'SubversionListHandler.initList no wc_path %s', self.project_info.project_name )
            g = self.list_panel.list_ctrl
            g.InsertStringItem( 0, '' )
            if self.isProjectParent():
                g.SetStringItem( 0, self.columnIndex( self.col_name ), 'Use the Checkout command to fetch files' )
            else:
                g.SetStringItem( 0, self.columnIndex( self.col_name ), 'Use the Update command to fetch files' )
            # set the colour
            item = g.GetItem( 0 )
            item.SetTextColour( wx.RED )
            g.SetItem( item )
            return

        prefix_len = len( self.project_info.wc_path ) + 1
        if len(filter_text) > 0:
            if filter_field == 'Name':
                self.all_files = [f for f in self.project_info.getFilesStatus()
                            if filter_text in f.path[prefix_len:]]
            elif filter_field == 'Author':
                self.all_files = [f for f in self.project_info.getFilesStatus()
                            if f.entry is not None and filter_text in f.entry.commit_author]
        else:
            self.all_files = self.project_info.getFilesStatus()

        self.all_files.sort( SortList( sort_data, self.project_info ) )
    
        self._drawList()

    def sortList( self, sort_data ):
        self.list_panel.list_ctrl.SortItems( SortListCtrl( self.all_files, sort_data, self.project_info ) )

    def columnRequired( self, col ):
        return self.column_info.getInfo( col ).included

    def columnIndex( self, col ):
        return self.column_info.getInfo( col ).column

    def getColumnId( self, col ):
        return self.column_info.getColumnOrder()[col]

    def _drawList( self ):
        prefix_len = len( self.project_info.wc_path ) + 1

        g = self.list_panel.list_ctrl

        for index, file in enumerate( self.all_files ):

            colour = wx.BLACK
            if file.entry is None:
                colour = wx.GREEN
            else:
                if( file.is_locked ):
                    colour = wx.RED
                elif( file.text_status != pysvn.wc_status_kind.normal
                or file.prop_status not in [pysvn.wc_status_kind.normal,pysvn.wc_status_kind.none]
                or file.is_copied or file.is_switched ):
                    colour = wx.BLUE

            for column in self.column_info.getColumnOrder():
                if column == self.col_name:
                    self._draw_NameColumn( index, file, prefix_len )
                elif column == self.col_state:
                    self._draw_StateColumn( index, file )
                elif column == self.col_date:
                    self._draw_DateColumn( index, file )
                elif column == self.col_revision:
                    self._draw_RevisionColumn( index, file )
                elif column == self.col_author:
                    self._draw_AuthorColumn( index, file )
                elif column == self.col_type:
                    self._draw_TypeColumn( index, file )
                elif column == self.col_size:
                    self._draw_SizeColumn( index, file )
                elif column == self.col_mime_type:
                    self._draw_MimeTypeColumn( index, self.project_info.getProperty( file.path, 'svn:mime-type' ) )
                elif column == self.col_eol_style:
                    self._draw_EolStyleColumn( index, self.project_info.getProperty( file.path, 'svn:eol-style' ) )


            # set the colour
            item = g.GetItem( index )
            item.SetTextColour( colour )
            g.SetItem( item )

    def _draw_Column( self, index, column, value ):
        g = self.list_panel.list_ctrl

        max_index = g.GetItemCount()
        if index >= max_index:
            g.InsertStringItem( index, value )
            g.SetItemData( index, index )
        else:
            g.SetStringItem( index, self.columnIndex( column ), value )

    def _draw_NameColumn( self, index, file, prefix_len ):
        self._draw_Column( index, self.col_name, file.path[prefix_len:] )

    def _draw_SizeColumn( self, index, file ):
        self._draw_Column( index, self.col_size, '%d' % 0 )

    def _draw_EolStyleColumn( self, index, value ):
        if value is None:
            value = ''
        self._draw_Column( index, self.col_eol_style, value )

    def _draw_MimeTypeColumn( self, index, value ):
        if value is None:
            value = ''
        self._draw_Column( index, self.col_mime_type, value )

    def _draw_StateColumn( self, index, file ):
        if file.entry is None:
            self._draw_Column( index, self.col_state, '' )
        else:
            self._draw_Column( index, self.col_state, wb_subversion_utils._status_format( file ) )

    def _draw_TypeColumn( self, index, file ):
        if file.entry is None:
            self._draw_Column( index, self.col_type, '' )
        else:
            self._draw_Column( index, self.col_type, str(file.entry.kind) )

    def _draw_AuthorColumn( self, index, file ):
        if file.entry is None or file.entry.commit_author is None:
            self._draw_Column( index, self.col_author, '' )
        else:
            self._draw_Column( index, self.col_author, file.entry.commit_author )

    def _draw_RevisionColumn( self, index, file ):
        if file.entry is None or file.entry.commit_revision.number < 0:
            self._draw_Column( index, self.col_revision, '' )
        else:
            self._draw_Column( index, self.col_revision, str(file.entry.commit_revision.number) )

    def _draw_DateColumn( self, index, file ):
        if file.entry is None or file.entry.commit_time == 0:
            self._draw_Column( index, self.col_date, '' )
        else:
            self._draw_Column( index, self.col_date, wb_subversion_utils.fmtDateTime( file.entry.commit_time ) )

    def getState( self, all_rows ):
        if len(all_rows) == 0:
            return None

        state = wb_list_panel.ListItemState()

        if self.project_info.need_checkout:
            state.ui_project_parent = True
            state.versioned = True
            return state

        state.modified = True
        state.new_versioned = True
        state.versioned = True
        state.unversioned = True
        state.need_checkin = True
        state.conflict = True
        state.file_exists = True

        state.filenames = []

        for row in all_rows:
            filename = self.all_files[ row ].path
            state.filenames.append( filename )

            if not os.path.exists( filename ):
                state.file_exists = False

            if os.path.isdir( filename ):
                state.modified = False
                state.conflict = False
                state.file_exists = False

            text_status = self.all_files[ row ].text_status
            if text_status in [pysvn.wc_status_kind.unversioned]:
                state.versioned = False
            else:
                state.unversioned = False

            state.new_versioned = state.new_versioned and text_status in [pysvn.wc_status_kind.added]

            prop_status = self.all_files[ row ].prop_status
            state.modified = state.modified and (text_status in [pysvn.wc_status_kind.modified,
                                pysvn.wc_status_kind.conflicted]
                        or
                        prop_status in [pysvn.wc_status_kind.modified,
                                pysvn.wc_status_kind.conflicted])
            state.need_checkin = state.need_checkin and (text_status in [pysvn.wc_status_kind.added,
                                    pysvn.wc_status_kind.deleted,
                                    pysvn.wc_status_kind.modified]
                            or
                            prop_status in [pysvn.wc_status_kind.added,
                                    pysvn.wc_status_kind.deleted,
                                    pysvn.wc_status_kind.modified])
            state.conflict = state.conflict and text_status in [pysvn.wc_status_kind.conflicted]

        #state.printState( 'getState' )

        return state

    def getContextMenu( self ):
        if self.project_info.need_checkout:
            menu_template = \
                [('', wb_ids.id_SP_Checkout, 'Checkout' )]
        else:
            menu_template = \
                [('', wb_ids.id_File_Edit, 'Edit' )]
            if sys.platform in ['win32','darwin']:
                menu_template += \
                    [('', wb_ids.id_Shell_Open, 'Open' )]
            menu_template += \
                [('-', 0, 0 )
                ,('', wb_ids.id_SP_DiffWorkBase, 'Diff WC vs. BASE...' )
                ,('', wb_ids.id_SP_DiffWorkHead, 'Diff WC vs. HEAD...' )
                ,('>', wb_ids.id_SP_ConflictMenu, 'Conflict',
                    [('', wb_ids.id_SP_DiffOldMine, 'Diff Conflict Old vs. Mine...' )
                    ,('', wb_ids.id_SP_DiffMineNew, 'Diff Conflict Mine vs. New...' )
                    ,('', wb_ids.id_SP_DiffOldNew, 'Diff Conflict Old vs. New...' )
                    ,('-', 0, 0 )
                    ,('', wb_ids.id_SP_Resolved, 'Resolved Conflict' )
                    ])
                ,('-', 0, 0 )
                ,('', wb_ids.id_SP_Annotate, 'Annotate...' )
                ,('', wb_ids.id_SP_History, 'History...' )
                ,('', wb_ids.id_SP_Info, 'Information...' )
                ,('', wb_ids.id_SP_Properties, 'Properties...' )
                ,('-', 0, 0 )
                ,('', wb_ids.id_SP_Update, 'Update' )
                ,('-', 0, 0 )
                ,('', wb_ids.id_SP_Checkin, 'Checkin...' )
                ,('-', 0, 0 )
                ,('', wb_ids.id_SP_Add, 'Add' )
                ,('', wb_ids.id_SP_Rename, 'Rename' )
                ,('-', 0, 0 )
                ,('', wb_ids.id_SP_Delete, 'Delete' )
                ,('', wb_ids.id_SP_Revert, 'Revert' )
                ,('-', 0, 0 )
                ,('', wb_ids.id_SP_Cleanup, 'Clear locks' )
                ]

        return wb_subversion_utils.populateMenu( wx.Menu(), menu_template )


    def mayEdit( self, row ):
        if self.all_files[ row ].entry is None:
            return not os.path.isdir( self.getFilename() )
        else:
            return self.all_files[ row ].entry.kind != pysvn.node_kind.dir

    def mayOpen( self, row ):
        if self.all_files[ row ].entry is None:
            return not os.path.isdir( self.getFilename( row ) )
        else:
            return self.all_files[ row ].entry.kind == pysvn.node_kind.dir

    def getFilename( self, row ):
        return self.all_files[ row ].path

    def getStatusString( self, row ):
        return wb_subversion_utils._status_format( self.all_files[ row ] )

    def getStatusAndFilenames( self, all_rows ):
        return [(self.getStatusString( row ), self.getFilename( row )) for row in all_rows]

    def isControlled( self, row ):
        return self.all_files[ row ].entry is not None

    def getUrl( self, row ):
        if self.all_files[ row ].entry is None:
            return ''
        else:
            return self.all_files[ row ].entry.url

    def getConflictOld( self, row ):
        if self.all_files[ row ].entry is None:
            return ''
        else:
            return os.path.join( self.project_info.wc_path,
                    self.all_files[row].entry.conflict_old )

    def getConflictNew( self, row ):
        if self.all_files[ row ].entry is None:
            return ''
        else:
            return os.path.join( self.project_info.wc_path,
                    self.all_files[row].entry.conflict_new )

    def getConflictMine( self, row ):
        if self.all_files[ row ].entry is None:
            return ''
        else:
            return os.path.join( self.project_info.wc_path,
                    self.all_files[row].entry.conflict_work )

    #------------------------------------------------------------
    def Cmd_File_Add( self, all_rows ):
        try:
            for row in all_rows:
                filename = self.getFilename( row )
                self.project_info.client_fg.add( filename )
        except pysvn.ClientError, e:
            self.app.log_client_error( e )
        self.app.refreshFrame()

    def Cmd_File_Annotate( self, all_rows ):
        for row in all_rows:
            filename = self.getFilename( row )
            self.app.setProgress( 'Annotating %(count)d', 0 )

            self.app.setAction( 'Annotate %s...' % filename )

            yield self.app.backgroundProcess

            ok = False
            try:
                annotation = self.project_info.client_bg.annotate( filename )
                ok = True
            except pysvn.ClientError, e:
                self.app.log_client_error( e )

            yield self.app.foregroundProcess

            if not ok:
                break

            h_frame = wb_subversion_annotate.AnnotateFrame( self.app, self.project_info, filename, annotation )
            h_frame.Show( True )

        self.app.clearProgress()
        self.app.setAction( 'Ready' )

    def Cmd_File_Checkin( self, all_rows ):
        if len(all_rows) == 0:
            wx.MessageBox( "There are no changes to checkin",
                "Warning", style=wx.OK|wx.ICON_EXCLAMATION )
            return

        all_filenames = [self.getFilename( row ) for row in all_rows]
        message = self.app.getLogMessage( 'Check in', self.getStatusAndFilenames( all_rows ) )
        if not message:
            return

        self.app.setAction( 'Check in %s...' % self.project_info.wc_path )
        self.app.setProgress( 'Sent %(count)d of %(total)d', len( all_rows ) )

        yield self.app.backgroundProcess

        ok = False
        try:
            rev = self.project_info.client_bg.checkin( all_filenames, message )
            ok = True
        except pysvn.ClientError, e:
            self.app.log_client_error( e )

        yield self.app.foregroundProcess

        if ok:
            if rev:
                self.app.log.info( 'Checkin created revision %d' % rev.number )
            else:
                self.app.log.warning( 'No changes to checkin ' )

        self.app.refreshFrame()
        self.app.setAction( 'Ready' )
        self.app.clearProgress()

    def Cmd_File_Cleanup( self, all_rows ):
        try:
            for row in all_rows:
                filename = self.getFilename( row )
                self.project_info.client_fg.cleanup( filename )
        except pysvn.ClientError, e:
            self.app.log_client_error( e )
        self.app.refreshFrame()

    def Cmd_File_Delete( self, all_rows ):
        if self.app.confirmAction( 'Delete', self.getStatusAndFilenames( all_rows ) ):
            for row in all_rows:
                if self.isControlled( row ):
                    try:
                        self.project_info.client_fg.remove( self.getFilename( row ) )
                    except pysvn.ClientError, e:
                        self.app.log_client_error( e )
                else:
                    try:
                        os.remove( self.getFilename( row ) )
                    except (OSError,IOError), e:
                        self.app.log.error( str(e) )

            self.app.refreshFrame()

    def Cmd_File_DiffWorkBase( self, all_rows ):
        for row in all_rows:
            filename = self.getFilename( row )
            self.app.setAction( 'Diff BASE %s...' % filename )
            base_filename = os.path.join(
                self.project_info.wc_path,
                os.path.dirname( filename ),
                '.svn', 'text-base',
                os.path.basename( filename ) + '.svn-base' )

            self.app.Diff( base_filename, filename + '@BASE', filename, filename )

        self.app.setAction( 'Ready' )

    def Cmd_File_DiffWorkHead( self, all_rows ):
        for row in all_rows:
            filename = self.getFilename( row )
            url = self.getUrl( row )

            self.app.setAction( 'Diff HEAD %s...' % filename )

            yield self.app.backgroundProcess

            ok = False
            try:
                head_rev_text = self.project_info.client_bg.cat( filename,
                            revision=pysvn.Revision( pysvn.opt_revision_kind.head ) )
                ok = True
            except pysvn.ClientError, e:
                self.app.log_client_error( e )

            yield self.app.foregroundProcess

            if not ok:
                break

            head_rev_lines = wb_read_file.contentsAsUnicode( head_rev_text ).split('\n')
            if head_rev_lines[-1] == '':
                del head_rev_lines[-1]
            self.app.Diff( head_rev_lines, url + '@HEAD', filename, filename )

        self.app.setAction( 'Ready' )

    def Cmd_File_DiffOldNew( self, all_rows ):
        for row in all_rows:
            old_filename = self.getConflictOld( row )
            new_filename = self.getConflictNew( row )
            self.app.Diff( old_filename, old_filename, new_filename, new_filename )

    def Cmd_File_DiffOldMine( self, all_rows ):
        for row in all_rows:
            old_filename = self.getConflictOld( row )
            mine_filename = self.getConflictMine( row )
            self.app.Diff( old_filename, old_filename, mine_filename, mine_filename )

    def Cmd_File_DiffMineNew( self, all_rows ):
        for row in all_rows:
            mine_filename = self.getConflictMine( row )
            new_filename = self.getConflictNew( row )
            self.app.Diff( mine_filename, mine_filename, new_filename, new_filename )

    def Cmd_File_History( self, rows ):
        for row in rows:
            filename = self.getFilename( row )
            self.app.setAction( 'History %s...' % filename )

            yield self.app.backgroundProcess

            ok = False
            history_entries = []
            try:
                log_entries = self.project_info.client_bg.log( filename, strict_node_history=True )
                for log in log_entries:
                    history_entries.append( (log['revision'].number,
                        log['author'],
                        log['date'],
                        '',
                        log['message'] ) )
                info = self.project_info.client_bg.info( filename )

                tags_url = self.project_info.getTagsUrl( info.url )
                if tags_url:
                    for ls_info in self.project_info.client_bg.ls( tags_url ):
                        history_entries.append( (ls_info['created_rev'].number,
                            ls_info['last_author'],
                            ls_info['time'],
                            'Tag ' + ls_info['name'].split('/')[-1],
                            '' ) )


                branches_url = self.project_info.getBranchesUrl( info.url )
                if branches_url:
                    for ls_info in self.project_info.client_bg.ls( branches_url ):
                        history_entries.append( (ls_info['created_rev'].number,
                            ls_info['last_author'],
                            ls_info['time'],
                            'Branch ' + ls_info['name'].split('/')[-1],
                            '' ) )

                history_entries.sort()

                ok = True
            except pysvn.ClientError, e:
                self.app.log_client_error( e )

            yield self.app.foregroundProcess

            if not ok:
                break

            h_frame = wb_subversion_history.HistoryFileFrame( self.app, self.project_info, filename, info.url, history_entries )
            h_frame.Show( True )

        self.app.setAction( 'Ready' )

    def Cmd_File_Info( self, rows ):
        for row in rows:
            filename = self.getFilename( row )

            try:
                entry = self.project_info.client_fg.info( filename )

                dialog = wb_subversion_info_dialog.InfoDialog( self.app,
                        self.list_panel.list_ctrl,
                        filename,
                        entry )
                dialog.ShowModal()

            except pysvn.ClientError, e:
                self.app.log_client_error( e )

    def Cmd_File_Properties( self, rows ):
        client_fg = self.project_info.client_fg
        for row in rows:
            filename = self.getFilename( row )

            try:
                prop_list = client_fg.proplist( filename,
                        revision=pysvn.Revision( pysvn.opt_revision_kind.working ) )
                if len(prop_list) == 0:
                    prop_dict = {}
                else:
                    _, prop_dict = prop_list[0]
                dialog = wb_subversion_properties_dialog.PropertiesDialog( self.app,
                        self.list_panel.list_ctrl,
                        filename,
                        prop_dict )
                if dialog.ShowModal() == wx.OK:
                    for present, name, value in dialog.getModifiedProperties():
                        if not present:
                            # delete name
                            client_fg.propdel( name, filename )
                        else:
                            # add/update name value
                            client_fg.propset( name, value, filename )

            except pysvn.ClientError, e:
                self.app.log_client_error( e )
                break
            self.app.refreshFrame()

    def Cmd_File_Rename( self, all_rows ):
        for row in all_rows:
            old_filename = self.getFilename( row )
            old_name = os.path.basename( old_filename )

            new_name, force = self.app.renameFile( "Rename", old_name, None )

            if new_name is None:
                break

            if new_name != old_name:
                new_full_filename = os.path.join( os.path.dirname( old_filename ), new_name )
                print 'Rename',old_filename, new_full_filename
                if self.isControlled( row ):
                    try:
                        self.project_info.client_fg.move( old_filename, new_full_filename )
                    except pysvn.ClientError, e:
                        self.app.log_client_error( e )
                else:
                    try:
                        os.rename( old_filename, new_full_filename )
                    except (OSError,IOError), e:
                        self.app.log.error( str(e) )
            else:
                break
        self.app.refreshFrame()

    def Cmd_File_Revert( self, all_rows ):
        if self.app.confirmAction( 'Revert', self.getStatusAndFilenames( all_rows ) ):
            try:
                for row in all_rows:
                    filename = self.getFilename( row )
                    self.project_info.client_fg.revert( filename )
            except pysvn.ClientError, e:
                self.app.log_client_error( e )
            self.app.refreshFrame()

    def Cmd_File_Resolved( self, all_rows ):
        if self.app.confirmAction( 'Resolved', self.getStatusAndFilenames( all_rows ) ):
            try:
                for row in all_rows:
                    filename = self.getFilename( row )
                    self.project_info.client_fg.resolved( filename )
            except pysvn.ClientError, e:
                self.app.log_client_error( e )
            self.app.refreshFrame()

    def Cmd_File_Update( self, all_rows ):
        self.app.setProgress( 'Updated %(count)d', 0 )
        for row in all_rows:
            filename = self.getFilename( row )
            self.app.setAction( 'Update %s...' % filename )

            yield self.app.backgroundProcess

            ok = False
            try:
                rev = self.project_info.client_bg.update( filename, recurse=False )
                ok = True
            except pysvn.ClientError, e:
                self.app.log_client_error( e )

            yield self.app.foregroundProcess

            if not ok:
                break

        if ok:
            if rev.number > 0:
                count = self.app.getProgressValue( 'count' )
                if count == 0:
                    self.app.log.info( 'Updated to revision %d, no new updates' % rev.number )
                elif count == 1:
                    self.app.log.info( 'Updated to revision %d, 1 new update' % rev.number )
                else:
                    self.app.log.info( 'Updated to revision %d, %d new updates' % (rev.number, count) )

        self.app.clearProgress()
        self.app.setAction( 'Ready' )
        self.app.refreshFrame()


class SortList:
    def __init__( self, sort_data, project_info ):
        self.sort_data = sort_data
        self.project_info = project_info

    def __call__( self, a, b ):
        # called to cmp
        a_field = self.getField( a, self.sort_data.getField() )
        b_field = self.getField( b, self.sort_data.getField() )
        ordering = cmp( a_field, b_field )
        if ordering == 0:
            a_path = self.getField( a, SubversionListHandler.col_name )
            b_path = self.getField( b, SubversionListHandler.col_name )
            return cmp( a_path, b_path )
        else:
            return ordering * self.sort_data.getOrder()

    def getField( self, file, field ):
        __pychecker__ = '--no-returnvalues'

        field = self.sort_data.getField()
        if field == SubversionListHandler.col_name:
            return file.path
        if field == SubversionListHandler.col_state:
            # Use positive text_status first
            # then positive prop_status
            # other wise use negative text_status
            text_val = wb_subversion_utils.wc_status_kind_text_sort_map[ file.text_status ]
            if text_val > 0:
                return text_val

            prop_val = wb_subversion_utils.wc_status_kind_text_sort_map[ file.prop_status ]
            if prop_val > 0:
                return prop_val+wb_subversion_utils.prop_sort_offset
                
            if text_val < 0:
                return -text_val

            if self.sort_data.getOrder():
                return 999
            else:
                return 0

        if field == SubversionListHandler.col_revision:
            if file.entry is None or file.entry.commit_revision.number == 0:
                if self.sort_data.getOrder() > 0:
                    return 9999999
                else:
                    return 0
            else:
                return file.entry.commit_revision.number
        if field == SubversionListHandler.col_author:
            if file.entry is None or file.entry.commit_author is None:
                if self.sort_data.getOrder() > 0:
                    return (1, u'')
                else:
                    return (-1, u'')
            else:
                return (0, file.entry.commit_author)

        if field == SubversionListHandler.col_date:
            if file.entry is None:
                return 0
            else:
                return file.entry.commit_time

        if field == SubversionListHandler.col_type:
            if file.entry is None:
                return pysvn.node_kind.none
            else:
                return file.entry.kind


        if field == SubversionListHandler.col_eol_style:
            return self.project_info.getProperty( file.path, 'svn:eol-style' )

        if field == SubversionListHandler.col_mime_type:
            return self.project_info.getProperty( file.path, 'svn:mime-type' )

        raise wb_exceptions.InternalError( 'SortList does not support field %s' % field )

class SortListCtrl(SortList):
    def __init__( self, all_files, sort_data, project_info ):
        SortList.__init__( self, sort_data, project_info )
        self.all_files = all_files

    def getField( self, index, field ):
        return SortList.getField( self, self.all_files[index], field )
