/*
 *  dixchange.c
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "dixchange.h"
#include <gdome.h>

/**
 * Examples
 * <dixchange>
 * <word literal="" phonetic="ʤ" pos="̾" cclass="̵" score="50">
 * <usage category=""/>
 * <usage category="ƻ"/>
 * </word>
 * </dixchange>
 **/

GList *dixchange_parse(const char* filename) {
	/* XXX: names of word_nodelist and word_node must be changed */
    GList *list = NULL;
    GdomeDOMImplementation *domimpl;
    GdomeDocument *document;
    GdomeElement *rootelement;
    GdomeElement *word_element, *usage_element;
    GdomeNode *word_node = NULL, *usage_node = NULL;
    GdomeNodeList *word_nodelist = NULL, *usage_nodelist = NULL;
    GdomeDOMString *tagname1, *tagname2;
    GdomeDOMString *literal_attr, *phonetic_attr, *pos_attr;
    GdomeDOMString *cclass_attr, *score_attr;
    GdomeDOMString *literal, *phonetic, *pos, *cclass, *score;
    GdomeDOMString *usage_category, *usage_score;
    GdomeDOMString *usage_category_attr, *usage_score_attr;
    GdomeDOMString *word_tag, *usage_tag;
    GdomeException exception;
    gint word_nodenum, usage_nodenum, i, j;
    gint word_elementnum = 0;

    /* get DOMImplementation reference*/
    domimpl = gdome_di_mkref();

    /* load DixChange XML file */
    document = gdome_di_createDocFromURI(domimpl, filename, GDOME_LOAD_PARSING, &exception);
    if(document == NULL) {
	return NULL;
    }

    rootelement = gdome_doc_documentElement(document, &exception);
    if(rootelement == NULL) {
	g_printerr("documentElement: NULL Exception #%d\n", exception);
	return NULL;
    }

    word_nodelist = gdome_el_childNodes(rootelement, &exception);
    if(word_nodelist != NULL) {
	word_nodenum = gdome_nl_length(word_nodelist, &exception);
	if(word_nodenum == 0) {
	    if(word_nodelist != NULL) {
		gdome_nl_unref(word_nodelist, &exception);
		return NULL;
	    }
	}
	for(i = 0; i < word_nodenum; i++) {
	    word_node = gdome_nl_item(word_nodelist, i, &exception);
	    word_element = (GdomeElement*)word_node;

	    word_tag = gdome_str_mkref("word");
	    tagname1 = gdome_el_tagName(word_element, &exception);
	    if(gdome_str_equal(tagname1, word_tag)) {
		word_elementnum++;

		literal_attr = gdome_str_mkref("literal");
		phonetic_attr = gdome_str_mkref("phonetic");
		pos_attr = gdome_str_mkref("pos");
		cclass_attr = gdome_str_mkref("cclass");
		score_attr = gdome_str_mkref("score");

		literal = gdome_el_getAttribute(word_element, literal_attr, &exception);
		phonetic = gdome_el_getAttribute(word_element, phonetic_attr, &exception);
		pos = gdome_el_getAttribute(word_element, pos_attr, &exception);
		cclass = gdome_el_getAttribute(word_element, cclass_attr, &exception);
		score = gdome_el_getAttribute(word_element, score_attr, &exception);

		list = dixchange_append_word(list, literal->str, phonetic->str,
					     pos->str, cclass->str, score->str);

		gdome_str_unref(literal_attr);
		gdome_str_unref(phonetic_attr);
		gdome_str_unref(pos_attr);
		gdome_str_unref(cclass_attr);
		gdome_str_unref(score_attr);
		gdome_str_unref(literal);
		gdome_str_unref(phonetic);
		gdome_str_unref(pos);
		gdome_str_unref(cclass);
		gdome_str_unref(score);

		if(gdome_el_hasChildNodes(word_element, &exception)) {
		    usage_nodelist = gdome_el_childNodes(word_element, &exception);
		    if(usage_nodelist != NULL) {
			usage_nodenum = gdome_nl_length(usage_nodelist, &exception);
			if(usage_nodenum != 0) {
			    for(j = 0; j < usage_nodenum; j++) {
				usage_node = gdome_nl_item(usage_nodelist, j, &exception);
				usage_element = (GdomeElement*)usage_node;
				usage_tag = gdome_str_mkref("usage");

				tagname2 = gdome_el_tagName(usage_element, &exception);
				if(gdome_str_isEmpty(tagname2)) {
				    if(gdome_str_equal(tagname2, usage_tag)) {
					usage_category_attr = gdome_str_mkref("category");
					usage_score_attr = gdome_str_mkref("score");

					usage_category = gdome_el_getAttribute(usage_element,
									       usage_category_attr,
									       &exception);
					usage_score = gdome_el_getAttribute(usage_element,
									    usage_score_attr,
									    &exception);
					dixchange_append_usage(g_list_nth(list, word_elementnum - 1),
							       usage_category->str, usage_score->str);

					gdome_str_unref(usage_category_attr);
					gdome_str_unref(usage_score_attr);
					gdome_str_unref(usage_category);
					gdome_str_unref(usage_score);
				    }
				    gdome_str_unref(tagname2);
				}
				gdome_str_unref(usage_tag);
			    }
			}
			gdome_nl_unref(usage_nodelist, &exception);
		    }
		}
	    }
	    gdome_str_unref(tagname1);
	    gdome_str_unref(word_tag);
	}
	gdome_nl_unref(word_nodelist, &exception);
    } else {
	g_printerr("el_childNodes: NULL Exception #%d\n", exception);
	return NULL;
    }
    gdome_doc_unref(document, &exception);
    gdome_di_unref(domimpl, &exception);

    return list;
}

void dixchange_write(GList *list, gchar *filename) {
    GList *pos1, *pos2;
    GdomeDOMImplementation *domimpl;
    GdomeDocument *document;
    GdomeDocumentType *document_type;
    GdomeElement *rootelement;
    GdomeElement *word_element, *usage_element;
    GdomeNode *word_node = NULL, *usage_node = NULL;
    GdomeDOMString *doctype_qualifiedName;
    GdomeDOMString *doctype_publicId, *doctype_systemId;
    GdomeDOMString *version_attr;
    GdomeDOMString *literal_attr, *phonetic_attr, *pos_attr;
    GdomeDOMString *cclass_attr, *score_attr;
    GdomeDOMString *version;
    GdomeDOMString *literal, *phonetic, *pos, *cclass, *score;
    GdomeDOMString *usage_category, *usage_score;
    GdomeDOMString *usage_category_attr, *usage_score_attr;
    GdomeDOMString *dixchange_tag, *word_tag, *usage_tag;
    GdomeException exception;

    dixchange_word *word;
    dixchange_usage *usage;

    /* get DOMImplementation reference*/
    domimpl = gdome_di_mkref();
    /* create DOCTYPE but become NULL in current situation */
    document_type = NULL;
    /*	DOCTYPE *
	doctype_qualifiedName = gdome_str_mkref("dixchange");
	doctype_publicId = gdome_str_mkref("");
	doctype_systemId = gdome_str_mkref(DIXCHANGE_URI);
	document_type = gdome_di_createDocumentType(domimpl, doctype_qualifiedName, doctype_publicId, doctype_systemId, &exception);
	gdome_str_unref(doctype_qualifiedName);
	gdome_str_unref(doctype_publicId);
	gdome_str_unref(doctype_systemId);
    */

    /* create document */
    dixchange_tag = gdome_str_mkref("dixchange");
    document = gdome_di_createDocument(domimpl, NULL, dixchange_tag,
				       document_type, &exception);
    gdome_str_unref(dixchange_tag);

    if(document != NULL) {
	rootelement = gdome_doc_documentElement(document, &exception); // must check NULL
	version_attr = gdome_str_mkref("version");
	version = gdome_str_mkref("0.01");
	gdome_el_setAttribute(rootelement, version_attr, version, &exception);
	gdome_str_unref(version_attr);
	gdome_str_unref(version);

	for(pos1 = g_list_first(list); pos1 != NULL; pos1 = g_list_next(pos1)) {
	    word = pos1->data;

	    word_tag = gdome_str_mkref("word");
	    word_element = gdome_doc_createElement(document, word_tag, &exception); // must check NULL
	    gdome_str_unref(word_tag);


	    if(word->literal[0] != '\0' && word->literal != NULL) {
		literal = gdome_str_mkref(word->literal);
		literal_attr = gdome_str_mkref("literal");
		gdome_el_setAttribute(word_element, literal_attr, literal, &exception);
		gdome_str_unref(literal);
		gdome_str_unref(literal_attr);
	    }

	    if(word->phonetic[0] != '\0' && word->phonetic != NULL) {
		phonetic = gdome_str_mkref(word->phonetic);
		phonetic_attr = gdome_str_mkref("phonetic");
		gdome_el_setAttribute(word_element, phonetic_attr, phonetic, &exception);
		gdome_str_unref(phonetic);
		gdome_str_unref(phonetic_attr);
	    }

	    if(word->pos[0] != '\0' && word->pos != NULL) {
		pos = gdome_str_mkref(word->pos);
		pos_attr = gdome_str_mkref("pos");
		gdome_el_setAttribute(word_element, pos_attr, pos, &exception);
		gdome_str_unref(pos);
		gdome_str_unref(pos_attr);
	    }

	    if(word->cclass[0] != '\0' && word->cclass != NULL) {
		cclass = gdome_str_mkref(word->cclass);
		cclass_attr = gdome_str_mkref("cclass");
		gdome_el_setAttribute(word_element, cclass_attr, cclass, &exception);
		gdome_str_unref(cclass);
		gdome_str_unref(cclass_attr);
	    }

	    if(word->score[0] != '\0' && word->score != NULL) {
		score = gdome_str_mkref(word->score);
		score_attr = gdome_str_mkref("score");
		gdome_el_setAttribute(word_element, score_attr, score, &exception);
		gdome_str_unref(score);
		gdome_str_unref(score_attr);
	    }

	    word_node = gdome_el_appendChild(rootelement, (GdomeNode*)word_element, &exception);
	    if(word_node != (GdomeNode*)word_element) {
		g_printerr("Element.appendChild: failed\n\tException #%d\n", exception);
	    }
	    gdome_n_unref(word_node, &exception);

	    if(word->usage != NULL) {
		for(pos2 = g_list_first(word->usage); pos2 != NULL; pos2 = g_list_next(pos2)) {
		    usage = pos2->data;
		    usage_tag = gdome_str_mkref("usage");
		    usage_element = gdome_doc_createElement(document, usage_tag, &exception); // must check NULL
		    gdome_str_unref(usage_tag);

		    if(usage->category[0] != '\0' && usage->category != NULL) {
			usage_category = gdome_str_mkref(usage->category);
			usage_category_attr = gdome_str_mkref("category");
			gdome_el_setAttribute(usage_element, usage_category_attr,
					      usage_category, &exception);
			gdome_str_unref(usage_category_attr);
			gdome_str_unref(usage_category);
		    }

		    if(usage->category[0] != '\0' && usage->category != NULL) {
			usage_score = gdome_str_mkref(usage->score);
			usage_score_attr = gdome_str_mkref("score");
			gdome_el_setAttribute(usage_element, usage_score_attr,
					      usage_score, &exception);
			gdome_str_unref(usage_score_attr);
			gdome_str_unref(usage_score);
		    }

		    usage_node = gdome_el_appendChild(word_element, (GdomeNode*)usage_element, &exception);
		    if(usage_node != (GdomeNode*)usage_element) {
			g_printerr("Element.appendChild: failed\n\tException #%d\n",
				   exception);
		    }
		    gdome_n_unref(usage_node, &exception);
		}
		gdome_el_unref(usage_element, &exception);
	    }
	    gdome_el_unref(word_element, &exception);
	}

	gdome_di_saveDocToFileEnc(domimpl, document, filename,
				  "UTF-8", GDOME_SAVE_LIBXML_INDENT, &exception);
	gdome_doc_unref(document, &exception);
    } else {
	g_printerr("di_createDocument: NULL Exception #%d\n", exception);
    }

    gdome_di_unref(domimpl, &exception);
}

GList *dixchange_append_word(GList *list, gchar *literal, gchar *phonetic,
			     gchar *pos, gchar *cclass, gchar *score)
{
    dixchange_word *word = NULL;

    word = g_new0(dixchange_word, 1);

    if(literal != NULL)
	word->literal = g_strdup(literal);
    else
	word->literal = g_strdup("");
    if(phonetic != NULL)
	word->phonetic = g_strdup(phonetic);
    else
	word->phonetic = g_strdup("");
    if(pos != NULL)
	word->pos = g_strdup(pos);
    else
	word->pos = g_strdup("");
    if(cclass != NULL)
	word->cclass = g_strdup(cclass);
    else
	word->cclass = g_strdup("");
    if(score != NULL)
	word->score = g_strdup(score);
    else
	word->score = g_strdup("1");
    word->usage = NULL;

    if(word->literal != NULL && word->phonetic != NULL &&
       word->pos != NULL && word->cclass != NULL &&
       word->score != NULL)
	list = g_list_append(list, word);

    return list;
}

void dixchange_append_usage(GList *word, gchar *category, gchar *score)
{
    dixchange_word *data = word->data;
    dixchange_usage *usage = NULL;

    usage = g_new0(dixchange_usage, 1);

    usage->category = g_strdup(category);
    usage->score = g_strdup(score);

    data->usage = g_list_append(data->usage, usage);
}

void dixchange_word_free(GList *list) {
    GList *pos = NULL;
    dixchange_word *data;

    for(pos = g_list_first(list); pos != NULL; pos = g_list_next(pos))
    {
	data = pos->data;
	if(data->literal != NULL)
	    g_free(data->literal);
	if(data->phonetic != NULL)
	    g_free(data->phonetic);
	if(data->pos != NULL)
	    g_free(data->pos);
	if(data->cclass != NULL)
	    g_free(data->cclass);
	if(data->score != NULL)
	    g_free(data->score);

	if(data->usage != NULL)
	    dixchange_usage_free(data->usage);
    }
}

void dixchange_usage_free(GList *list) {
    GList *pos;
    dixchange_usage *data;

    g_return_if_fail(list);

    for(pos = g_list_first(list); pos != NULL; pos = g_list_next(pos)) {
	data = pos->data;
	if(data->category != NULL)
	    g_free(data->category);
	if(data->score != NULL)
	    g_free(data->score);
    }
}

/*
int main(void) {
	char *filename = "./dixchange1.xml";
	char *filename2 = "./dixchangeout.xml";
	GList *list;
	list = dixchange_parse(filename);
	dixchange_write(list, filename2);
	dixchange_word_free(list);
	return 0;
}
*/
