/* ====================================================================
 * Copyright (c) 2008                    Martin Hauner
 *                                       http://subcommander.tigris.org
 *
 * Subcommander is licensed as described in the file doc/COPYING, which
 * you should have received as part of this distribution.
 * ====================================================================
 */

// sc
#include "config.h"
#include "WcViewStatus.h"
#include "StatusId.h"
#include "svn/WcStatus.h"
#include "svn/WcEntry.h"


WcViewStatus::WcViewStatus( svn::WcStatusPtr status )
: _status(status)
{
}

svn::WcStatusPtr WcViewStatus::status() const
{
  return _status;
}

bool WcViewStatus::isDir() const
{
  return _status->isDir();
}

const sc::String& WcViewStatus::path() const
{
  return _status->getName();
}



enum WcViewStatusColumns
{
  ColName,
  ColTextStatus,
  ColPropStatus,
  ColRpLock,
  ColWcLock,
  ColCopied,          ///< add with history
  ColSwitched,
  ColUpdate,
  ColWcRevision,
  ColCommitRevision,
  ColCommitAuthor,
  ColDeepStatus,
  ColEmpty,
  ColMax
};


WcViewStatusData::WcViewStatusData()
{
}

QString WcViewStatusData::data( int column, const WcViewItem* item ) const
{
  svn::WcStatusPtr status = static_cast<const WcViewStatus*>(item)->status();

  switch(column)
  {
  case ColName:
    {
      return QString::fromUtf8( status->getName().getStr() );
    }
  case ColTextStatus:
    {
      return getStatusCode( status->getTextStatus() );
    }
  case ColPropStatus:
    {
      return getStatusCode( status->getPropStatus() );
    }
  case ColRpLock:
    {
      return getLockStatusCode( status->getLockStatus() );
    }
  case ColWcLock:
    {
      return status->isLocked() ?
        getStatusCode(StatusWcLockLocked) : getStatusCode(StatusWcLockNone);
    }
  case ColCopied:
    {
      return status->isCopied() ?
        getStatusCode(StatusHistory) : getStatusCode(StatusHistoryNone);
    }
  case ColSwitched:
    {
      return status->isSwitched() ?
        getStatusCode(StatusSwitched) : getStatusCode(StatusSwitchedNo);
    }
  case ColUpdate:
    {
      bool t = status->getReposTextStatus() != svn::WcStatus_None;
      bool p = status->getReposTextStatus() != svn::WcStatus_None;

      return ( t || p ) ?
        getStatusCode(StatusUpToDateNo) : getStatusCode(StatusUpToDate);
    }
  case ColWcRevision:
    {
      const svn::WcEntry* e = status->getWcEntry();
      if( ! e )
      {
        return "";
      }
      else if( e->getRevnumber() < 0 )
      {
        return "?";
      }
      else if( status->isCopied() )
      {
        return "-";
      }
      else
      {
        return QString().sprintf( "%ld", (unsigned long)e->getRevnumber() );
      }
    }
  case ColCommitRevision:
    {
      const svn::WcEntry* e = status->getWcEntry();
      if( ! e )
      {
        return "";
      }
      else if( e->getCmtRevnumber() < 0 )
      {
        return "?";
      }
      else
      {
        return QString().sprintf( "%ld", (unsigned long)e->getCmtRevnumber() );
      }
    }
  case ColCommitAuthor:
    {
      const svn::WcEntry* e = status->getWcEntry();
      if( ! e )
      {
        return "";
      }
      else if( ! e->getCmtAuthor().getStr() )
      {
        return "?";
      }
      else
      {
        return QString::fromUtf8( e->getCmtAuthor().getStr() );
      }
    }
  case ColDeepStatus:
    {
      // todo
    }
  case ColEmpty:
    return "";
  default:
    return "?";
  }
}

QColor WcViewStatusData::color( int column, const WcViewItem* item ) const
{
  svn::WcStatusPtr status = static_cast<const WcViewStatus*>(item)->status();

  return getStatusColor(status->getTextStatus());
}

QString WcViewStatusData::header( int column ) const
{
  switch(column)
  {
  case ColName: 
    return _q("path/name");
  case ColTextStatus:
    return _q("ts");
  case ColPropStatus:
    return _q("ps");
  case ColRpLock:
    return _q("rl");
  case ColWcLock:
    return _q("wl");
  case ColCopied:
    return _q("ah");
  case ColSwitched:
    return _q("sw");
  case ColUpdate:
    return _q("upd");
  case ColWcRevision:
    return _q("wc rev");
  case ColCommitRevision:
    return _q("cmt rev");
  case ColCommitAuthor:
    return _q("cmt author");
  case ColDeepStatus:
    return _q("ds");
  case ColEmpty:
    return "";
  default:
    return "?";
  }
}

Qt::Alignment WcViewStatusData::alignment( int column ) const
{
  switch(column)
  {
  case ColName:
    return Qt::AlignLeft | Qt::AlignVCenter;
  default:
    return Qt::AlignCenter | Qt::AlignVCenter;
  }
}

int WcViewStatusData::columns() const
{
  return ColMax;
}

bool WcViewStatusData::deep( int column ) const
{
  return column == ColDeepStatus;
}
