/* Copyright (C) 2000-2004  Thomas Bopp, Thorsten Hampel, Ludger Merkens
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 * $Id: cert.pmod,v 1.1.1.1 2006/03/27 12:40:10 exodusd Exp $
 */

constant cvs_version="$Id: cert.pmod,v 1.1.1.1 2006/03/27 12:40:10 exodusd Exp $";

import Standards.ASN1.Types;

#if !constant(PrintableString) 
class PrintableString {
    inherit asn1_printable_string;
}
#endif


/**
 * Read a certificate file 'path' which has to be encoded
 * appropriately (base64).
 *  
 * @param string path - the path of the filename
 * @return mapping of certificate components
 */
mapping read_certificate(string path)
{
    mapping result = ([ ]);
    
    string cert;
    string f = Stdio.read_file(path);
    if ( !stringp(f) ) 
	error("Failed to read certificate file " + path + "\n");
    object msg = Tools.PEM.pem_msg()->init(f);
    object part = msg->parts["CERTIFICATE"] || msg->parts["X509 CERTIFICATE"];
    if ( !objectp(part) )
	error("Failed to parse certificate in file 'steam.cer'.");
    cert = part->decoded_body();
    result->cert = cert;
    
    part = msg->parts["RSA PRIVATE KEY"];
    string key = part->decoded_body();
    if ( !objectp(part) )
	error("Failed to find RSA private key in certificate");
    result->key = key;
    result->rsa = Standards.PKCS.RSA.parse_private_key(key);
#if constant(Crypto.Random)
    result->random = Crypto.Random.random_string;
#else    
    result->random = Crypto.randomness.reasonably_random()->read;
#endif
    return result;

}


string create_cert(mapping vars)
{
#if constant(Crypto.Random)
    function random = Crypto.Random.random_string;
#else    
    function random = Crypto.randomness.reasonably_random()->read;
#endif

#if constant(Crypto.rsa) 
    object rsa = Crypto.rsa()->generate_key(512, random);
#else
    object rsa = Crypto.RSA()->generate_key(512, random);
#endif

    array attr = ({ });
    attr += 
	({ ([ "countryName": PrintableString(vars->country),
	    "organizationName": PrintableString(vars->organization),
	    "organizationUnitName": PrintableString(vars->unit),
	    "localityName": PrintableString(vars->locality),
	    "stateOrProvinceName":PrintableString(vars->province),
	    "commonName": PrintableString(vars->name), ]),
	       });
    string cert = Tools.X509.make_selfsigned_rsa_certificate(
	rsa, 60*60*24*1000, attr);
    string der = MIME.encode_base64(cert);
    string rsa_str = MIME.encode_base64(Standards.PKCS.RSA.private_key(rsa));
    
    der = "-----BEGIN CERTIFICATE-----\n"+der+
	"\n-----END CERTIFICATE-----\n";
    der += "\n-----BEGIN RSA PRIVATE KEY-----\n"+
	rsa_str+"\n-----END RSA PRIVATE KEY\n";
    return der;
}


