inherit "/kernel/module";

import httplib;

#include <database.h>
#include <macros.h>
#include <attributes.h>
#include <classes.h>

class spmAPI {
    array spm_install_package() { return ({ }); }
    void spm_upgrade_package() { } 
    string get_identifier() { return "spmAPI"; }
    void install_package(string source, void|string version) { }
}

static mapping mLogs = ([ ]);

static void install_module() 
{
    // make sure package container is there
    object packages = OBJ("/packages");
    if ( !objectp(packages) ) {
	packages=get_factory(CLASS_CONTAINER)->execute((["name":"packages",]));
	packages->move(OBJ("/"));
    }
    // try to install stylesheets for container !
    object pxsl = OBJ("/stylesheets/spm.xsl");
    if ( objectp(pxsl) ) {
	packages->set_attribute("xsl:content", ([ GROUP("steam"): pxsl, ]));
    }
    else
	FATAL("While installing SPM: failed to find SPM Stylesheet !");
    this()->move(OBJ("/"));
}

static void install_spm_modules(array modules, object dest)
{
    object module, modsrc;
    
    mapping packages = ([ ]);
    object packageSrc = dest->get_object_byname("package");
    if ( !objectp(packageSrc) ) {
	packageSrc = 
	    get_factory(CLASS_CONTAINER)->execute( ([ "name":"package" ]) );
	packageSrc->move(dest);
    }
    
    dest->set_acquire_attribute("xsl:content", 0);
    dest->set_attribute("xsl:content", 
		      ([ GROUP("steam"):OBJ("/stylesheets/content.xsl"), ]) );

    object image;
    foreach(modules, object component ) {
	if ( component->get_object_class() & CLASS_DOCLPC ) {
	    string name = component->get_identifier();
	    
	    object pmod = packageSrc->get_object_byname(name);
	    sscanf(name, "%s.%*s", name);
	    object e = master()->getErrorContainer();
	    master()->inhibit_compile_errors(e);
	    if ( !objectp(pmod) ) {
		module = component->execute( (["name": name, ]) );
		if ( !objectp(module) )
		{
		    master()->inhibit_compile_errors(0);
		    FATAL("failed to compile new instance - throwing");
		    THROW("Failed to load module\n"+e->get()+"\n"+
			  e->get_warnings(), backtrace());
		}
		spm_log(dest, "install.log", 
			"Installing new Module %s", name);
		packages[component] = module->this();
		
	    }
	    else {
		pmod->set_content(component->get_content());
		pmod->upgrade();
		module = pmod->provide_instance();
		module = module->get_object();
		packages[pmod] = module->this();
		spm_log(dest, "install.log", 
			"Updating code of previous module %O", pmod);
	    }
	    master()->inhibit_compile_errors(0);
	    if ( !Program.implements(object_program(module), 
				     object_program(spmAPI())) )
	    {
		spm_log(dest, "install.log", 
			"Module doesnot correctly implement the SPM module API"
			, pmod);
		steam_user_error("The Package component " + 
			  component->get_identifier() +
			  " does not correctly implement the SPM module API ");
	    }
	}
	else if ( component->get_object_class() & CLASS_DOCXSL )
	{
	    switch( component->get_identifier() ) {
	    case "package.xsl":
		// set stylesheet for package
		dest->set_attribute("xsl:content", 
				    ([ GROUP("steam"): component ]));
		spm_log(dest, "install.log", 
			"Setting attribute for package (package.xsl)");
		// todo: check for updating !
		component->move(packageSrc);
		break;
	    }
	}
	else if ( component->get_object_class() & CLASS_DOCUMENT ) {
	    if ( search(component->query_attribute(DOC_MIME_TYPE),"image")>=0)
		image = component;
	    component->move(packageSrc);
	}
    }
    
    foreach( indices(packages), modsrc ) {
	module = packages[modsrc];
	object pmod = dest->get_object_byname(module->get_identifier());
	if ( objectp(pmod) ) {
	    spm_log(dest, "install.log", 
		    "Upgrading installation (spm_upgrade_package())");
	    module->spm_upgrade_package();
	}
	else {
	    spm_log(dest, "install.log", 
		    "New Installation (spm_install_package())");
	    module->spm_install_package();
	    _Database->register_module(module->get_identifier(), module);
	}
	if ( objectp(image) ) {
	    module->set_acquire_attribute(OBJ_ICON, 0);
	    module->set_attribute(OBJ_ICON, image);
	}
	spm_log(dest, "install.log", 
		"Registered Module %s", module->get_identifier());
	MESSAGE("Registered %s Module !", module->get_identifier());
	module->move(dest);
	modsrc->move(packageSrc);
    }
    if ( objectp(image) ) {
	dest->set_acquire_attribute(OBJ_ICON, 0);
	dest->set_attribute(OBJ_ICON, image);
    }
}

static void install_spm_files(array files, object dest, object spm) 
{
    foreach(files, object file) {
	object oldfile = dest->get_object_byname(file->get_identifier());
	//upgrade - previous version!
	if ( objectp(oldfile) ) {
	    spm_log(spm, "install.log",
		    "Upgrading file %s", oldfile->get_identifier());
	    if ( oldfile->get_object_class() & CLASS_DOCUMENT )
		oldfile->set_content(file->get_content());
	    else if ( oldfile->get_object_class() & CLASS_CONTAINER )
		install_spm_files(file->get_inventory(), oldfile, spm);
	}
	else {
	    spm_log(spm, "install.log",
		    "Installing new file %s", file->get_identifier());
	    file->move(dest);
	}
    }
}

static void install_spm_src(array files, object dest, object spm)
{
    install_spm_files(files, dest, spm);
}

static mixed get_package_config(object node, string id) 
{
    object n = node->get_node("/package/"+id);
    if ( !objectp(n) )
	steam_user_error("Configuration failed: missing " + id);
    return n->data;
}

static mapping read_configuration(object configObj)
{
    mapping config = ([ ]);
    werror("Parsing %O\n", configObj->get_content());
    object node = xmlDom.parse(configObj->get_content());
    config->author = get_package_config(node, "author");
    config->description = get_package_config(node, "description");
    config->version = get_package_config(node, "version");
    config->name = get_package_config(node, "name");
    return config;
}

static object create_package_cont(mapping config, object dest)
{
    object package = dest->get_object_byname(config->name);
    if ( objectp(package) )
	return package;
    object factory = get_factory(CLASS_CONTAINER);
    package = factory->execute( ([ "name": config->name, ]) );
    package->set_attribute(PACKAGE_VERSION, config->version);
    package->set_attribute(PACKAGE_AUTHOR, config->author);
    package->set_attribute(OBJ_DESC, config->description);
    return package;
}

object spm_get_logfile(object spm, string logname)
{
    object factory;
    object packageCont;

    if ( spm->get_object_class() & CLASS_CONTAINER )
	packageCont = spm;
    else
	packageCont = spm->get_environment();

    object logs = packageCont->get_object_byname("logs");
    if ( !objectp(logs) ) {
	factory = get_factory(CLASS_CONTAINER);
	logs = factory->execute( ([ "name": "logs", ]) );
	logs->move(packageCont);
    }
    object log = logs->get_object_byname(logname);
    if ( !objectp(log) ) {
	factory = get_factory(CLASS_DOCUMENT);
	log = factory->execute( (["name":logname, "mimetype":"text/html" ]) );
	log->move(logs);
    }
    return log;
}

void spm_log(object spm, string logname, string htmlMessage, mixed ... args) 
{
    if ( mLogs[logname] ) {
	mLogs[logname]->write(sprintf(htmlMessage, @args) + "\n");
	return;
    }
    
    object log = spm_get_logfile(spm, logname);
    if ( objectp(log) ) {
	mLogs[logname] = log->get_content_file("w", ([]));
	mLogs[logname]->write(sprintf(htmlMessage, @args) + "\n");
    }    
}

string spm_read_log(object spm, string logname)
{
    if ( !objectp(spm) )
	return "** package not found (null) **";
    
    if ( objectp(mLogs[logname]) )
	mLogs[logname]->close();

    object log = spm_get_logfile(spm, logname);
    if ( !objectp(log) )
	return "** logfile " + logname + " not found ! ** ";
    object version = log;
    int i = log->query_attribute(DOC_VERSION);
    string html = "";
    while ( objectp(version) && i >= 0 ) {
	html += version->get_content() + "<hr />";
	version = version->query_attribute(OBJ_VERSIONOF);
	i--;
    }
    return html;
}

mixed install_spm(object spm, object dest)
{
    object file;
    object packageCont;
    object packages = OBJ("/packages");

    if ( !objectp(packages) )
	steam_user_error("Failed to find packages - corrupt steam installation!");
    
    
    array(object) files = get_module("tar")->unpack(spm);
    werror("Unpacked: %O\n", files);
    mapping config = ([ ]);
    int foundConfig = 0;
    foreach(files, file) {
	if ( file->get_identifier() == "package.xml" ) {
	    config = spm_check_configuration(file);
	    foundConfig = 1;
	}
    }
    if ( !foundConfig )
	steam_user_error("Configuration file 'package.xml' not found !");

    packageCont = create_package_cont(config, packages);
    packageCont->move(packages);
    object packageFile;
    foreach(files, file) {
	if ( file->get_object_class() & CLASS_CONTAINER ) {
	    switch ( file->get_identifier() ) {
	    case "package":
		packageFile = file;
		break;
	    case "files":
		install_spm_files(file->get_inventory(), OBJ("/"),packageCont);
		break;
	    case "src":
		install_spm_src(file->get_inventory(),packageCont,packageCont);
		break;
	    }
	}
    }
    if ( objectp(packageFile) )
	install_spm_modules(packageFile->get_inventory(), packageCont);
}

mixed execute(mapping vars)
{
    switch ( vars->_action ) {
    case "install":
	object package = find_object((int)vars->install);
	if ( !objectp(package) )
	    return error_page("Cannot install package: not found !");
	install_spm(package, OBJ("/"));
	break;
    case "upload":
	string name = basename(vars["paket.filename"]);
	object pdoc = get_factory(CLASS_DOCUMENT)->execute( (["name":name,]));
	pdoc->set_content(vars->paket);
	pdoc->move(OBJ("/packages"));
	break;
    case "uninstall":
	object cont = find_object((int)vars->id);
	if ( objectp(cont) ) {
	    foreach(cont->get_inventory(), object o) {
		if ( objectp(o) && o->get_object_class() & CLASS_MODULE )
		    o->delete();
	    }
	    cont->move(this_user()->query_attribute(USER_TRASHBIN));
	}
	break;
    case "delete":
	object obj = find_object((int)vars->id);
	if ( objectp(obj) )
	    obj->move(this_user()->query_attribute(USER_TRASHBIN));
	break;
    case "show_install_log":
	object spm = find_object((int)vars->id);
	if ( objectp(spm) ) {
	    return result_page(
		replace(spm_read_log(spm, "install.log"), "\n", "<br />"), 
		"/packages");
	}
	break;
    }
    return redirect("/packages");
}

/**
 * This function takes a container and reads all xml files in it (config files). The
 * files contents is joined into a single xml string.
 *  
 * @param object container - the container with xml configurations in it
 * @return xml of the joined configuration files.
 */
object spm_read_configuration(object container) 
{
    if ( !objectp(container) )
	return 0;
    
    array xmlInv = container->get_inventory_by_class(CLASS_DOCXML);
    array(object) xmlNodes = ({ });
    foreach ( xmlInv, object inv ) {
	object node = xmlDom.parse(inv->get_content());
	werror("Dumping join node: %s\n", node->get_xml());
	xmlNodes += ({ node });
    }
    object config;
    if ( sizeof(xmlNodes) >= 1 ) {
	config = xmlNodes[0];
	for ( int i = 1; i < sizeof(xmlNodes); i++ ) {
	    config->join(xmlNodes[i]);
	}
    }
    return config;
}

int spm_version_value(string version)
{
    int minor, major, release;
    sscanf(version, "%d.%d.%d", release, major, minor);
    return release * 1000000 + major*1000 + minor;
}

int spm_match_versions(string version, string installedVersion)
{
    string   val;
    int iVersion;

    iVersion = spm_version_value(installedVersion);

    if ( sscanf(version, ">=%s", val) ) {
	werror("Iversion: %d, Version=%d\n", iVersion, spm_version_value(val));
	if ( spm_version_value(val) < iVersion )
	    return 1;
    }
    if ( sscanf(version, "==%s", val) ) {
	if ( spm_version_value(val) != iVersion )
	    return 1;
    }
    if ( sscanf(version, "<=%s", val) ) {
	if ( spm_version_value(val) > iVersion )
	    return 1;
    }
    else if ( sscanf(version, ">%s", val) ) {
	if ( spm_version_value(val) <= iVersion )
	    return 1;
    }
    else if ( sscanf(version, "<%s", val) ) {
	if ( spm_version_value(val) >= iVersion )
	    return 1;
    }
    return 0;
}

mapping spm_check_configuration(object config)
{
    mapping cfgMap = ([ ]);
    object node = xmlDom.parse(config->get_content());
    foreach(node->get_children(), object child) {
	if ( child->get_name() == "requires" ) {
	    if ( !spm_match_versions(child->attributes["server"],
				    _Server->get_version()) )
		steam_user_error("Server Version mismatch: need '"+ 
				 child->attributes->server + 
				 "' (Current Server Version is '"+
				 _Server->get_version()+"'");
	    foreach ( child->get_children(), object req ) {
		// check each require
		string modname = req->data;
		object module = get_module(modname);
		if ( !objectp(module) ) 
		    module = get_module("package:"+modname);
		
		if ( !objectp(module) ) 
		    steam_user_error(
			"Steam Packaging Error:\n"+
			"The required package '"+ modname + "' "+
			" is not installed !\n"+
			"Make sure that all required packages are installed " +
			"before installing this package.");
		string versionStr = module->get_version();
		if ( !stringp(req->attributes->version) ) {
		    steam_user_error(
			"Steam Packaging Error:\n"+
			"No Version attribute for " + req->data);
		}
		if ( !spm_match_versions(req->attributes->version,versionStr) )
		{
		    steam_user_error(
			"Steam Packaging Error:\n"+
			"Requirement could not be fullfilled...\n"+
			"Package: '" + modname + " needs " + req->version +
			"(installed " + versionStr + ")");
		}
	    }
	}
	else {
	    cfgMap[child->get_name()] = child->get_data();
	}
    }
    if ( !stringp(cfgMap->name) )
	steam_user_error("Invalid configuration file - missing name !");
    return cfgMap;
}


string spm_get_configuration(object container) 
{
    object node = spm_read_configuration(container);
    return node->get_xml();
}


object get_source_object() { return this(); } // script needs this!
string get_identifier() { return "SPM"; }
int get_object_class() { return ::get_object_class() | CLASS_SCRIPT; }
