/* Copyright (C) 2000-2004  Thomas Bopp, Thorsten Hampel, Ludger Merkens
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 * $Id: ldap.pike,v 1.1.1.1 2005/09/21 14:22:14 exodusd Exp $
 */

constant cvs_version="$Id: ldap.pike,v 1.1.1.1 2005/09/21 14:22:14 exodusd Exp $";

inherit "/kernel/module";

#include <configure.h>
#include <macros.h>
#include <config.h>
#include <attributes.h>
#include <classes.h>
#include <events.h>

//#define LDAP_DEBUG 1

#ifdef LDAP_DEBUG
#define LDAP_LOG(s, args...) werror(s+"\n", args)
#else
#define LDAP_LOG(s, args...)
#endif


//! This module is a ldap client inside sTeam. It reads configuration
//! parameters of sTeam to contact a ldap server.
//! All the user management can be done with ldap this way. Some
//! special scenario might require to modify the modules code to 
//! make it work.
//!
//! The configuration variables used are:
//! server - the server name to connect to (if none is specified, then ldap will not be used)
//! cacheTime - how long (in seconds) shall ldap entries be cached? (to reduce server requests)
//! user   - ldap user for logging in
//! password - the password for the user
//! base_dc - ldap base dc, consult ldap documentation
//! userdn - the dn path where new users are stored
//! groupdn - the dn path where new groups are stored
//! objectName - the ldap object name to be used for the search
//! userAttr - the attribute containing the user's login name
//! passwordAttr - the attribute containing the password
//! emailAttr - the attribute containing the user's email address
//! iconAttr - the attribute containing the user's icon
//! fullnameAttr - the attribute containing the user's surname
//! nameAttr - the attribute containing the user's first name
//! userClass - an attribute value that identifies users
//! userId - an attribute that contains a value that can be used to match users to groups
//! groupAttr - the attribute containing the group's name
//! groupClass - an attribute value that identifies groups
//! groupId - an attribute that contains a value that can be used to match users to groups
//! memberAttr - an attribute that contains a value that can be used to match users to groups
//! descriptionAttr - the attribute that contains a user or group description
//! notfound - defines what should be done if a user or group could not be found in LDAP:
//!            "create" : create and insert a new user in LDAP (at userdn)
//!            "ignore" : do nothing
//! sync - "true"/"false" : sync user or group data
//! authorize - "ldap" : authorize users through LDAP
//! requiredAttr : required attributes for creating new users in LDAP
//! adminAccount - "root" : sTeam account that will receive administrative mails about ldap
//! charset - "utf-8" : charset used in ldap server

static object      oLDAP;
static string sServerURL;
static mapping    config;

static object charset_decoder;
static object charset_encoder;

static object user_cache;
static object group_cache;
static object authorize_cache;
static int cache_time = 0;

static object admin_account = 0;
static array ldap_conflicts = ({ });

private bool bind_root ()
{
  if ( stringp(config["user"]) && sizeof(config["user"])>0 )
    return oLDAP->bind("cn="+config["user"] + 
		       (stringp(config->root_dn) && strlen(config->root_dn)>0?","+config->root_dn:""),
		       config["password"]);
  else return true;  // if no user is set, then we don't need to bind: just return true
}

private static void connect_ldap()
{
  if ( !stringp(config["server"]) )
    return;

   mixed err = catch {
	oLDAP = Protocols.LDAP.client(config["server"]);

	if ( !bind_root() ) {
	  if ( oLDAP->error_number() > 0 )
	    FATAL("Failed to bind ldap: " + oLDAP->error_string());
	  else
	    FATAL("Failed to bind ldap.");
	}
	oLDAP->set_scope(2);
	oLDAP->set_basedn(config["base_dc"]);
    };
    if ( err != 0 )
      FATAL("Failed to build connection to LDAP: %s\n%O", config->server, err);
    if ( objectp(oLDAP) && oLDAP->error_number() > 0 )
      FATAL("Failed to bind ldap: " + oLDAP->error_string());	
}

static void init_module()
{
    config = read_config(Stdio.read_file(CONFIG_DIR+"/ldap.txt"), "ldap");
    if ( !mappingp(config) ) {
	MESSAGE("LDAP Service not started - missing configuration !");
	return; // ldap not started !
    }
    if ( !stringp(config["server"]) ) return;  // ldap deactivated

    //MESSAGE("LDAP configuration is %O", config);
    if ( stringp(config["charset"]) )
      charset_decoder = Locale.Charset.decoder( config["charset"] );
    else charset_decoder = 0;
    charset_encoder = Locale.Charset.encoder( "utf-8" );

    if ( intp(config->cacheTime) )
	cache_time = config->cacheTime;
    else if ( !stringp(config->cacheTime) || (sscanf(config->cacheTime, "%d", cache_time) < 1) )
	cache_time = 0;
    object cache_module = get_module("cache");
    if ( objectp(cache_module) ) {
      user_cache = get_module("cache")->create_cache( "ldap:users", cache_time );
      group_cache = get_module("cache")->create_cache("ldap:groups", cache_time );
      authorize_cache = get_module("cache")->create_cache("ldap:auth", cache_time );
    }

    if ( !config->objectName )
      steam_error("objectName configuration missing !");

    connect_ldap();
/*
    // if our main dc does not exist - create it
    oLDAP->add(config->base_dc, 
	       ([ 
		 "objectclass": ({ "dcObject", "organization" }),
		 "o": "sTeam Authorization Directory",
		 "dc": "steam"
	       ]));
*/
}

void load_module()
{
  add_global_event(EVENT_USER_CHANGE_PW, sync_password, PHASE_NOTIFY);
  add_global_event(EVENT_USER_NEW_TICKET, sync_ticket, PHASE_NOTIFY);
}

static mixed map_results(object results)
{
  array result = ({ });
  
  for ( int i = 1; i <= results->num_entries(); i++ ) {
    mapping            data = ([ ]);
    mapping res = results->fetch(i);
    
    foreach(indices(res), string attr) {
      if ( arrayp(res[attr]) ) {
	if ( sizeof(res[attr]) == 1 )
	  data[attr] = res[attr][0];
	else
	  data[attr] = res[attr];
      }
    }
    if ( results->num_entries() == 1 )
      return data;
    result += ({ data });
  }
  return result;
}

mapping search_user ( string search_str, void|string user, void|string pass )
{
    mapping udata = ([ ]);
    object results;
    
    object ldap;
    if(config->bindUser=="true" && user && pass)
        ldap = bind_user(user, pass);
    if(!objectp(ldap))
        ldap = oLDAP;
    if(!objectp(ldap))
        return 0;

    if ( config->userdn )
    {
      ldap->set_basedn(config->userdn+","+config->base_dc);
      if ( ldap->error_number() )
        MESSAGE("Error: %s", oLDAP->error_string());
    }

    if ( catch(results = 
	       ldap->search(search_str)) )
    {
      // something went wrong. if we were using the global connection
      // try rebuilding it. (so it works for next time?)
      // FIXME: why not also retry the search?
      if(config["bindUser"]!="true")
        connect_ldap(); // try to rebuild connection;
      return 0;
    }
      
    if ( ldap->error_number() )
      werror("Error while searching user: %s", ldap->error_string());
    
    if ( results->num_entries() == 0 ) {
      //MESSAGE("User %s not found in LDAP directory.", uname);
      return 0;
    }
    udata = map_results(results);

    /*
    if ( stringp(udata[config->passwordAttr]) )
      sscanf(udata[config->passwordAttr], 
	     "{crypt}%s", udata[config->passwordAttr]);
    */
    return udata;
}


mixed fix_charset ( string|mapping|array v )
{
  if ( !objectp(charset_encoder) || !objectp(charset_decoder) ) return v;
  if ( stringp(v) ) {
    if ( xml.utf8_check(v) ) return v;  // already utf-8
    string tmp = charset_decoder->feed(v)->drain();
    tmp = charset_encoder->feed(tmp)->drain();
    //LDAP_LOG( "charset conversion: from \"%s\" to \"%s\".", v, tmp );
    return tmp;
  }
  else if ( arrayp(v) ) {
    array tmp = ({ });
    foreach ( v, mixed i )
      tmp += ({ fix_charset(i) });
    return tmp;
  }
  else if ( mappingp(v) ) {
    mapping tmp = ([ ]);
    foreach ( indices(v), mixed i )
      tmp += ([ fix_charset(i) : fix_charset(v[i]) ]);
    return tmp;
  }
}


mapping fetch_user ( string identifier, void|string pass )
{
  if ( !objectp(oLDAP) ) return 0;
  if ( !stringp(identifier) )
    return 0;

  if ( !objectp(user_cache) )
    user_cache = get_module("cache")->create_cache( "ldap:users", cache_time );

  LDAP_LOG("fetch_user(%s)", identifier);
  return user_cache->get( identifier, lambda(){ return fix_charset( search_user( "("+config->userAttr+"="+identifier+")", identifier, pass ) ); } );
}

mapping search_group ( string search_str )
{
  mapping gdata = ([ ]);
  object results;

  if ( !objectp(oLDAP) )
    return 0;
  
  if ( config->groupdn )
    oLDAP->set_basedn(config->groupdn+"," + config->base_dc);

  if ( catch(results = oLDAP->search(search_str)) ) {
    connect_ldap(); // try to rebuild connection;
    return 0;
  }
  
  if ( oLDAP->error_number() )
    FATAL("Error: %s", oLDAP->error_string());
  
  if ( results->num_entries() == 0 ) {
    //MESSAGE("Group %s not found in LDAP directory.", gname);
    return 0;
  }
  gdata = map_results(results);
  return gdata;
}

mapping fetch_group ( string identifier )
{
  if ( !objectp(oLDAP) ) return 0;
  if ( !stringp(identifier) )
    return 0;

  if ( !objectp(group_cache) )
    group_cache = get_module("cache")->create_cache("ldap:groups", cache_time );

  return group_cache->get( identifier, lambda(){ return fix_charset( search_group( "("+config->groupAttr+"="+identifier+")" ) ); } );
}

mapping fetch ( string dn, string pattern )
{
    if ( !objectp(oLDAP) )
	return 0;

  // caller must be module...

  mapping   data;
  object results;

  if ( !_Server->is_module(CALLER) )
    steam_error("Access for non-module denied !");

  if ( stringp(dn) && sizeof(dn)>0 )
    oLDAP->set_basedn(dn+"," + config->base_dc);
  else
    oLDAP->set_basedn(config->base_dc);
  
  if ( catch(results = oLDAP->search(pattern)) )
    {
      connect_ldap(); // try to rebuild connection;
      return 0;
    }
  
  if ( oLDAP->error_number() )
    FATAL("Error: %s", oLDAP->error_string());
  
  if ( results->num_entries() == 0 ) {
    return 0;
  }
  data = map_results(results);
  return data;
}

static bool check_password(string pass, string user_pw)
{
  if ( !stringp(pass) || !stringp(user_pw) )
    return 0;

  MESSAGE("check_password()");
  if ( user_pw[0..4] == "{SHA}" )
    return user_pw[5..] == MIME.encode_base64( sha_hash(pass) );
  if ( user_pw[0..6] == "{crypt}" )
    return crypt(pass, user_pw[7..]);
  return verify_crypt_md5(pass, user_pw);
}

object bind_user(string user, string pass)
{
    MESSAGE("bind_user(%s/%s)", (string)config["server"], user);
    if(!stringp(config["server"]) || config["server"] == "")
      return 0;
    object ldap = Protocols.LDAP.client(config["server"]);
    if(objectp(ldap))
    {
        string userdn="";
        if(config->userdn)
          userdn=config->userdn+",";
        if(ldap->bind(sprintf("%s=%s,%s%s", config->userAttr, user, userdn, 
                                            config->base_dc), pass))
        {
          MESSAGE("LDAP: bind successfull");
          ldap->set_scope(2);
          return ldap;
	}
    }
    else
    {
      MESSAGE("LDAP: connect failed.");
      return 0;
    }
}

bool authorize_ldap(object user, string pass) 
{
  if ( config->authorize != "ldap" )
    return false;
  
  if ( !objectp(user) )
    steam_error("User object expected for authorization !");

  if ( !stringp(pass) || sizeof(pass)<1 )
    return false;

  string uname = user->get_user_name();

  // don't authorize restricted users:
  if ( _Persistence->user_restricted( uname ) ) return false;

  if ( !objectp(authorize_cache) )
    authorize_cache = get_module("cache")->create_cache("ldap:auth", cache_time );

  string cached = authorize_cache->get( uname );
  if (stringp(cached)) {
      if ( check_password( pass, cached ) ) 
      {
        MESSAGE("User %s LDAP cache authorized !", uname);
        return true;
      }
      else 
      {
        MESSAGE("User %s found in LDAP cache - password failed! - %O", uname, cached);
        return false;
      }
  }
  
  mapping udata = fetch_user(uname, pass);
  
  if ( mappingp(udata) ) {
    string dn = udata["dn"];
    if ( !stringp(dn) ) dn = "";

    // check for conflicts (different user in sTeam than in LDAP):
    if (config->checkConflicts=="true" && !stringp(user->query_attribute("ldap:dn")) ) {
      if ( search(ldap_conflicts,uname)<0 ) {
	ldap_conflicts += ({ uname });
        if ( !objectp(admin_account) && stringp(config["adminAccount"]) )
          admin_account = _Persistence->lookup_user(config["adminAccount"]);
	if ( objectp(admin_account) )
	  admin_account->mail(
            "Dear LDAP administrator at "+_Server->get_server_name()
	    +",\n\nthere has been a conflict between LDAP and sTeam:\n"
	    +"User \""+uname+"\" already exists in sTeam, but now "
	    +"there is also an LDAP user with the same name/id.\nYou "
	    +"will need to remove/rename one of them or, if they are "
	    +"the same user, you can overwrite the sTeam data from LDAP "
	    +"by adding a \"dn\" attribute to the sTeam user.",
	    "{LDAP} User conflict: "+uname,
	    "root@"+_Server->get_server_name());
	else
	  werror( "LDAP: user conflict: %s in sTeam vs. %s in LDAP\n", uname, dn );
	return false;
      }
    }
    else if ( search(ldap_conflicts,uname) >= 0 )
      ldap_conflicts -= ({ uname });

    // FIXME: what is this check for? users may well like to use different
    // FIXME: email addresses in different services
    if ( stringp(udata[config->emailAttr]) && 
	 stringp(user->query_attribute(USER_EMAIL)) &&
	 udata[config->emailAttr] != user->query_attribute(USER_EMAIL) )
    {
      MESSAGE("LDAP: user " + uname + " seems to be a different person "+
		"email-ldap(%s), email(%s).", 
	      udata[config->emailAttr], user->query_attribute(USER_EMAIL));
      return false;
    }
    else if ( check_password( pass, udata[config->passwordAttr] ) ) {
      // need to synchronize passwords from ldap if ldap is down ?!
      // this is only done when the ldap password is received
      if ( udata[config->passwordAttr] != user->get_user_password()) {
	//catch(MESSAGE("Sync PW: %s:%s", udata[config->passwordAttr],
	//      user->get_user_password()));
	user->set_user_password(udata[config->passwordAttr], 1);
      }
      authorize_cache->put( uname, udata[config->passwordAttr] );

      MESSAGE("User %s LDAP authorized !", uname);
      return true;
    }
    else {
      MESSAGE("User %s found in LDAP directory - password failed!", uname);
      return false;
    }
  }
  //MESSAGE("User " + uname + " was not found in LDAP directory.");
  // if notfound configuration is set to create, then we should create
  // a user.
  // TODO: No! We only create users in update_user(), not here! Otherwise, if a
  // user mistyped his/her username during login, a user with that name would be
  // created...
  // FIXME? how so? we only get here if the user exists at least in sTeam
  //  if ( config->notfound == "create" )
  //    add_user(uname, pass, user);
  return false;
}

object sync_user(string name)
{
  // don't sync restricted users:
  if ( _Persistence->user_restricted( name ) ) return null;

  mapping udata = fetch_user(name);
  if ( !mappingp(udata) ) return 0;

  //MESSAGE("Sync of ldap user \"%s\": %O", name, udata);
  object user = get_module("users")->get_value(name);
  if ( objectp(user) ) {
    // update user date from LDAP
    if ( ! user->set_attributes( ([
             "pw" : udata[config->passwordAttr],
	     "email" : udata[config->emailAttr],
	     "fullname" : udata[config->fullnameAttr],
	     "firstname" : udata[config->nameAttr],
	     "OBJ_DESC" : udata[config->descriptionAttr],
	   ]) ) )
      werror( "Could not sync user attributes with ldap for \"%s\".\n", name );
  } else {
    // create new user to match LDAP user
    object factory = get_factory(CLASS_USER);
    user = factory->execute( ([
	     "name" : name,
	     "pw" : udata[config->passwordAttr],
	     "email" : udata[config->emailAttr],
	     "fullname" : udata[config->fullnameAttr],
	     "firstname" : udata[config->nameAttr],
	     "OBJ_DESC" : udata[config->descriptionAttr],
	   ]) );
    user->set_user_password(udata[config->passwordAttr], 1);
    user->activate_user(factory->get_activation());
  }
  // sync group membership:
  if ( objectp( user ) ) {
    string primaryGroupId = udata[config->groupId];
    if ( stringp( primaryGroupId ) ) {
      mapping group = search_group("("+config->groupId+"="+primaryGroupId+")");
    }
  }

  return user;
}

object sync_group(string name)
{
  // don't syncronize restricted groups:
  if ( _Persistence->group_restricted( name ) ) return null;

  mapping gdata = fetch_group(name);
  if ( !mappingp(gdata) ) return 0;
  //MESSAGE("Sync of ldap group: %O", gdata);
  //object group = get_module("groups")->lookup(name);
  object group = get_module("groups")->get_value(name);
  if ( objectp(group) ) {
//TODO: check and update group memberships
    // update group date from LDAP
    group->set_attributes( ([
	     "OBJ_DESC" : gdata[config->descriptionAttr],
	   ]) );
  } else {
    // create new group to match LDAP user
    object factory = get_factory(CLASS_GROUP);
    group = factory->execute( ([
	      "name": name,
	      "OBJ_DESC" : gdata[config->descriptionAttr],
	    ]) );
  }
  return group;
}

static void sync_password(int event, object user, object caller)
{
  if ( !mappingp(config) || !stringp(config["server"]) 
       || config["server"] == "" || config->sync != "true" ) 
    return;
  string oldpw = user->get_old_password();
  string crypted = user->get_user_password();
  string name = user->get_user_name();
  // don't sync password for restricted users:
  if ( _Persistence->group_restricted( name ) ) return;
  MESSAGE("LDAP Password sync for " + user->get_user_name());

  object ldap;
  string dn;

  if(config->bindUser=="true" && oldpw && (ldap=bind_user(name, oldpw)))
  {
    if(config->userdn)
      dn = sprintf("%s=%s,%s,%s", config->userAttr, name, config->userdn, config->base_dc);
    else
      dn = sprintf("%s=%s,%s", config->userAttr, name, config->base_dc);
  }
  else
  {
    ldap = oLDAP;
      dn = config->base_dc + " , " + config->userAttr + "=" + name;
  }

  mixed err;
  if(crypted[..2]=="$1$")
    crypted="{crypt}"+crypted;
  err=catch(ldap->modify(dn, ([ config->passwordAttr: ({ 2,crypted }),])));
  authorize_cache->remove( name );
  user_cache->remove( name );
  werror("sync_password(): %s - %s - %O\n", crypted, dn, ldap->error_string());
}

static void sync_ticket(int event, object user, object caller, string ticket)
{
  mixed err;
  if ( !mappingp(config) || config->sync != "true" ) return;
  string name = user->get_user_name();
  string dn = config->base_dc + " , " + config->userAttr + "=" + name;
  err=catch(oLDAP->modify(dn, ([ "userCertificate": ({ 2,ticket }),])));
}


mixed query_attribute(string|int key)
{
    mixed res;
    
    if ( key == OBJ_ICON ) {
	object obj = CALLER->this();
	string uname = obj->get_user_name();
	object results = 
	    oLDAP->search("(objectclass="+config["objectName"]+")");
	for ( int i = 1; i <= results->num_entries(); i++ ) {
	    mapping res = results->fetch(i);
	    
	    if ( res[config["userAttr"]][0] == uname ) 
		return res[config->iconAttr][0];
	}
    }
    return ::query_attribute(key);
}

mixed set_attribute(string|int key, mixed val)
{
    array(string)      keywords;
    object obj = CALLER->this();
 
     
    if ( key == OBJ_ICON ) {
	// set the icon
    }
}

bool is_user(string user)
{
    object results = oLDAP->search("("+config["userAttr"]+"="+user+")");
    return (results->num_entries() > 0);
}

static bool add_user(string name, string password, object user)
{
  // don't add restricted users:
  if ( _Persistence->group_restricted( name ) ) return false;
  
  string fullname = user->get_name();
  string firstname = user->query_attribute(USER_FIRSTNAME);
  string email = user->query_attribute(USER_EMAIL);
  
  mapping attributes = ([
    config["userAttr"]: ({ name }),
    config["fullnameAttr"]: ({ fullname }),
    "objectClass": ({ config["objectName"] }),
    config["passwordAttr"]: ({ make_crypt_md5(password) }),
  ]);
  if ( stringp(firstname) && strlen(firstname) > 0 )
    config["nameAttr"] = ({ firstname });
  if ( stringp(email) && strlen(email) > 0 )
    config["emailAttr"] = ({ email });

  array(string) requiredAttributes =  config["requiredAttr"];
  
  if ( arrayp(requiredAttributes) && sizeof(requiredAttributes) > 0 ) {
    foreach(requiredAttributes, string attr) {
      if ( zero_type(attributes[attr]) )
	attributes[attr] = ({ "-" });
    }
  }
  
  oLDAP->add(config["userAttr"]+"="+name+","+config["base_dc"], attributes);
  int err = oLDAP->error_number();
  if ( err != 0 )
    FATAL("Failed to add user , error number is " + oLDAP->error_string());
  return oLDAP->error_number() == 0;
}

string get_identifier() { return "ldap"; }
