/* Copyright (C) 2004 Christian Schmidt
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include <attributes.h>

/**
 * This module is responsible for sending messages to users/groups/... and e-mails to 
 * remote adresses. It also keeps track of system-wide alias-adresses and of user-forwards.
 */

inherit "/kernel/module";

import Messaging;

#include <database.h>
#include <macros.h>
#include <classes.h>
#include <config.h>
#include <attributes.h>

#define DEBUG_FORWARD

#ifdef DEBUG_FORWARD
#define LOG_FORWARD(s, args...) werror("forward: "+s+"\n", args)
#else
#define LOG_FORWARD
#endif

#if constant(Protocols.SMTP.client) 
#define SMTPCLIENT Protocols.SMTP.client
#else
#define SMTPCLIENT Protocols.SMTP.Client
#endif

//stores aliases & forwards
static mapping(string:array) mAliases, mForwards;

string _mailserver;
int _mailport;

string get_mask_char() { return "/";}

void init_module()
{
    mAliases=([]);
    mForwards=([]);
    add_data_storage(STORE_FORWARD,retrieve_aliases,restore_aliases);
}

void install_module()
{
    _mailserver = _Server->query_config(CFG_MAILSERVER);
    _mailport = (int)_Server->query_config(CFG_MAILPORT);
    LOG_FORWARD("mailserver is: "+_mailserver+":"+_mailport);
}    

string get_identifier() { return "forward"; }

mapping get_aliases()
{
    return mAliases;
}

mapping retrieve_aliases()
{
    if ( CALLER != _Database )
	    THROW("Caller is not database !", E_ACCESS);

    return (["aliases" : mAliases, "forwards" : mForwards]);
}

void restore_aliases(mapping data)
{
    if ( CALLER != _Database )
	    THROW("Caller is not database !", E_ACCESS);

    mAliases=data["aliases"];
    mForwards=data["forwards"];
    
    LOG_FORWARD("loaded "+sizeof(mAliases)+" aliases and "
                +sizeof(mForwards)+" forwards");
}

/**
 * check if an address is valid
 *
 * @author <a href="mailto:sepp@upb.de">Christian Schmidt</a>
 * @param string address - the address to check
 * @return int 1 if valid, 0 if invalid, -1 if access denied
 */
int is_valid(string address)
{
    LOG_FORWARD("checking adress \"%O\"",address);
    LOG_FORWARD("alias...");
    if(arrayp(mAliases[address]))
        return 1; //adress is alias
    LOG_FORWARD("no - trying user...");
    if(objectp(MODULE_USERS->lookup(address)))
        return 1; //address is single user
    LOG_FORWARD("no - trying group...");
    if(objectp(MODULE_GROUPS->lookup(address)))
        return 1; //adress is a group
    LOG_FORWARD("no - trying to replace - with space...");
    if(objectp(MODULE_GROUPS->lookup(replace(address, "-", " "))))
        return 1; //adress is a group
    LOG_FORWARD("no - trying object-id...");
    if(sscanf(address,"%d",int oid))
    {
        LOG_FORWARD("looking for object #%O",oid);
        object tmp=_Database->find_object(oid);
        if(objectp(tmp))
        {
            LOG_FORWARD("checking access on object #%O",oid);
            mixed err = catch { _SECURITY->access_annotate(0, tmp, CALLER, 0); };
            if(err!=0) return -1; //access denied -> invalid target-address
            else return 1; //target is existing object & annotatable
        }
        else LOG_FORWARD("not found");
    }
    LOG_FORWARD("sorry - no valid target found!");
    
    return 0; //no checks succeeded, target address is invalid
}

/**
 * return the remote addresses within an array of mixed addresses
 *
 * @author <a href="mailto:sepp@upb.de">Christian Schmidt</a>
 * @param array(string) names - the addresses to get remotes from
 * @return array(string) the remote adresses
 */
private array(string) get_remote_addresses(array(string) names)
{
    array(string) result=({});
    for(int i=0;i<sizeof(names);i++)
        if(search(names[i],"@")!=-1)
            result+=({names[i]});
    return result;
}

string resolve_name(object grp)
{
  if ( objectp(grp) ) {
    if ( grp->get_object_class() & CLASS_USER )
      return grp->get_user_name();
    return grp->get_identifier();
  }
  return "";
}

/**
 * split targets into remote, groups, users, and objects
 *
 * @author Martin Bhr
 * @param array(string) names - the addresses to work with
 * @return mapping - containing the different recipient types
 */
private mapping resolve_recipients(array(string) names)
{
    array unresolved=({});
    MESSAGE("Resolving: %O", names);
    array(string) resolved=replace_aliases(names);
    MESSAGE("Aliases replaced: %O", resolved);
    resolved=replace_forwards(resolved);
    MESSAGE("Forwards are: %O", resolved);
   
    mapping result =([ "groups":({}), "remote":({}), "users":({}), 
                       "objects":({}) ]);
    object target_obj;
    int oid;
   
    foreach(resolved;; string target)
    {
        if(search(target,"@")!=-1)
            result->remote += ({ target });
        else if ( objectp(target_obj=MODULE_GROUPS->lookup(target)) ) 
            result->groups += ({ target_obj });
        // groupnames may have spaces, but those don't work well with email.
        else if ( objectp(target_obj=MODULE_GROUPS->lookup(replace(target, "-", " "))) ) 
            result->groups += ({ target_obj });
        else if ( objectp(target_obj=MODULE_USERS->lookup(target)) )
            result->users += ({ target_obj });
        else if ( sscanf(target,"%d",oid) 
                  && objectp(target_obj=_Database->find_object(oid)) )
            result->objects += ({ target_obj });
        else
            unresolved += ({ target });
    }
   
    MESSAGE("Remote adresses are: %O", result->remote);
    MESSAGE("Group addresses are: %O", result->groups);
    MESSAGE("User addresses are: %O", result->users);
    MESSAGE("Object addresses are: %O", result->objects);
    if(sizeof(unresolved))
        MESSAGE("Warning! unresolved addresses: %O", unresolved);
    return result;
}

/**
 * within an array of adresses, search for aliases and replace them with 
 * their targets
 *
 * @author <a href="mailto:sepp@upb.de">Christian Schmidt</a>
 * @param array(string) names - the addresses to work with
 * @return array(string) the input-array with aliases replaced by targets
 */
private array(string) replace_aliases(array(string) names)
{
    array(string) result=({});
    foreach(names;; string name)
    {
        if(arrayp(mAliases[name])) //add code for checking aliases on aliases here...
            result+=mAliases[name];
        else 
           result+=({name});
    }
    return Array.uniq(result);
}


/**
 * within an array of adresses, search for forwards and replace them with 
 * their targets
 *
 * @author <a href="mailto:sepp@upb.de">Christian Schmidt</a>
 * @param array(string) names - the addresses to work with
 * @return array(string) the input-array with forwards replaced by targets
 */
private array(string) replace_forwards(array(string) names, void|mapping fwd_done)
{
    array(string) result=({});

    if ( !mappingp(fwd_done) )
      fwd_done = ([ ]);

    for(int i=0;i<sizeof(names);i++)
    {
        if ( fwd_done[names[i]] )
	  continue;
        if(arrayp(mForwards[names[i]]))
        {
            array(string) tmp=mForwards[names[i]];
            for(int j=0;j<sizeof(tmp);j++)
	    {
		if ( !stringp(tmp[j]) )
		    continue;
		
 	        fwd_done[tmp[j]] = 1;
                if(search(tmp[j],"@")!=-1) //remote address
                    result+=({tmp[j]});
                else
                {
                    if(search(tmp[j],"/")!=-1) //local forward-target starts with "/" -> don't forward further
                        result+=({tmp[j]-"/"});
                    else //lookup forward of this forward-target
                    {
                        array(string) tmp2=replace_aliases( ({tmp[j]}) );
                        result+=replace_forwards(tmp2, fwd_done);
                    }
                }
            }
        }
        else result+=({names[i]});
    }
    return result;
}

/**
 * send a message to multiple recipients
 *
 * @author <a href="mailto:sepp@upb.de">Christian Schmidt</a>
 * @param array(string) target - the addresses to send the message to
 * @param Message msg - the message to send (WARNING msg is destructed by sending!)
 * @return int 1 if successful
 */
int send_message(array(string) target, Messaging.Message msg)
{
    int hasLocal=0;
    string rawText=msg->complete_text();
    string sSender=msg->sender();
    array(string) resolved=replace_aliases(target);
    resolved=replace_forwards(resolved);
    array(string) asRemote=get_remote_addresses(resolved);

    array(string) asLocal=resolved-asRemote;
    if(sizeof(asLocal)>0) hasLocal=1;
    if(hasLocal)
        send_local(asLocal,msg);
    else 
	destruct(msg);
    
    LOG_FORWARD("Sending to " + sizeof(asRemote) + " Remote Recipients !");
    
    asRemote = Array.uniq(asRemote);
    send_remote(asRemote, rawText, sSender);
    return 1;
    for(int i=0;i<sizeof(asRemote);i++)
        send_remote(asRemote[i],rawText,sSender);
        
    return 1; //success, add code for failures!
}

/**
 * send a message (rfc2822 raw text) to multiple recipients
 *
 * @author <a href="mailto:sepp@upb.de">Christian Schmidt</a>
 * @param array(string) target - the addresses to send the message to
 * @param string rawText - the text of the message (rfc2822-format!)
 * @param string|void envFrom - the sender-value of the SMTP-envelope (only needed for forwarding, may be left empty)
 * @return int 1 if successful
 */
int send_message_raw(array(string) target, string rawText, string envFrom)
{
    mixed err;
    mapping sendAs=([]);
    sendAs = resolve_recipients(target);
    int res=1; //success
    if(sizeof(sendAs->users))
        err = catch(res=send_users(sendAs->users,Messaging.MIME2Message(rawText)));   
    if ( err ) 
      FATAL("Error while sending message to users: %O", err);
    if (res==0) 
      LOG_FORWARD("Warning! send_message_raw failed on one ore more recipient users!");
    res=1;

    if(sizeof(sendAs->objects))
        err = catch(res=send_objects(sendAs->objects,Messaging.MIME2Message(rawText)));   
    if ( err ) 
      FATAL("Error while sending message to objects: %O", err);
    if (res==0) 
      LOG_FORWARD("Warning! send_message_raw failed on one ore more recipient objects!");

    send_remote(sendAs->remote, rawText, envFrom);
    foreach(sendAs->groups;; object target)
        send_group(target,rawText,envFrom);

    return res;
}

/**
 * send a message to objects
 *
 * @author Martin Bhr
 * @param array(object) targets - the objects to send the message to
 * @param string msg - the message to send
 * @return int 1 if successful
 */
private int send_objects(array(object) targets, Messaging.Message msg)
{
    werror("send_objects(%O, %O)\n", targets, msg->header()->subject);
    if(sizeof(targets)==0 || !arrayp(targets)) return 0;

    int errors;

    foreach(targets; int count; object target)
    {
        Messaging.Message copy;
        if(count<sizeof(targets)-1)
            // duplicate message if more than one recipient
            copy=msg->duplicate();
        else
            // last recipient gets original
            copy=msg;

        mixed err=catch{Messaging.add_message_to_object(copy,target);};
        if(err)
        {
            copy->delete();
            destruct(copy);
            errors++;
            werror("unable to add message to: %s(%d):%O", target->get_identifier(), target->get_object_id(), err);
        }
    }

    if(errors) 
    {
      LOG_FORWARD("Warning!! send_objects encountered errors - some recipients failed!");
      return 0;
    }

    return 1;
}

/**
 * send a message to users
 *
 * @author Martin Bhr
 * @param array(object) users - the users to send the message to
 * @param string msg - the message to send
 * @return int 1 if successful
 */
private int send_users(array(object) targets, Messaging.Message msg)
{
    if(sizeof(targets)==0 || !arrayp(targets)) return 0;

    foreach(targets; int count; object user)
    {
        Messaging.Message copy=msg->duplicate();
        if(count<sizeof(targets)-1)
            // duplicate message if more than one recipient
            copy=msg->duplicate();
        else
            // last recipient gets original
            copy=msg;

        //the recipient gets all rights on his/her copy
        copy->grant_access(user); 

        Messaging.BaseMailBox box = Messaging.get_mailbox(user);
        copy->this()->set_acquire(box->this());
        box->add_message(copy);
    }

    return 1;
}



/**
 * send a message to a group
 *
 * @author Martin Bhr
 * @param object group - the group to send the message to
 * @param string msg - the message to send
 * @return int 1 if successful
 */
private int send_group(object group, string msg, string envFrom)
{
    werror("send_group(%O)\n", group);
    mapping headers = ([]);
    headers["X-sTeam-Group"] = group->get_identifier();
    headers["List-ID"] = replace(group->get_identifier(), " ", "-")+"@"
                         +_Server->query_config("smtp_host");
  
    // TODO: the following might be moved to the place where the message is
    // actually annotated
    object group_workroom = group->query_attribute("GROUP_WORKROOM");
    object modpath=get_module("filepath:tree");
    headers["X-sTeam-path"] = _Server->query_config("web_server")+
          modpath->object_to_filename(group_workroom);
  
    headers["X-sTeam-annotates"] = _Server->query_config("web_server")+":"
                                   +(string)group_workroom->get_object_id();
  
    msg = (((array)headers)[*]*": ")*"\r\n" + "\r\n" + msg;
  
    send_objects( ({ group_workroom }), Messaging.MIME2Message(msg));
  
    //TODO: only send to users that want a copy of group mails
    array(string) members = group->get_members(CLASS_USER)->get_user_name();
    send_message_raw(members, msg, envFrom);  
    
    array subgroups = group->get_sub_groups();
    //get_members_recursive();
    
}

/**
 * send a message to a subgroup
 *
 * @author Martin Bhr
 * @param object group - the group to send the message to
 * @param object parent - the group that the message was initially sent to
 * @param string msg - the message to send
 * @return int 1 if successful
 */
private int send_subgroup(object group, object parent, string msg, string envFrom)
{
  mapping headers = ([]);
  headers["X-sTeam-Subgroup"] = group->get_identifier();
  msg = (((array)headers)[*]*": ")*"\r\n" + "\r\n" + msg;

  //TODO: only send to users that want a copy of group mails
  array(string) members = group->get_members(CLASS_USER)->get_user_name();
  send_message_raw(members, msg, envFrom);  
  
  array subgroups = group->get_sub_groups();
  foreach(subgroups;; object subgroup)
    send_subgroup(subgroup, parent, msg, envFrom);
}


/**
 * send a simple message (subject & text) to multiple recipients
 *
 * @author <a href="mailto:sepp@upb.de">Christian Schmidt</a>
 * @param array(string) target - the addresses to send the message to
 * @param string subject - the subject of the message to send
 * @param string message - the text of the message to send
 * @return int 1 if successful
 */
int send_message_simple(array(string) target, string subject, string message)
{
    int hasLocal=0;
    Messaging.Message msg = Messaging.SimpleMessage(target, subject, message);
    string rawText=msg->complete_text();
    string sSender=msg->sender();
    array(string) resolved=replace_aliases(target);
    resolved=replace_forwards(resolved);
    array(string) asRemote=get_remote_addresses(resolved);
    array(string) asLocal=resolved-asRemote;
    if(sizeof(asLocal)>0) hasLocal=1;
    if(hasLocal)
        send_local(asLocal,msg);
    else destruct(msg);

    asRemote = Array.uniq(asRemote);
    send_remote(asRemote, rawText, sSender);
    return 1;
    for(int i=0;i<sizeof(asRemote);i++)
        send_remote(asRemote[i],rawText,sSender);
        
    return 1; //success, add code for failures!
}

/**
 * replace group-entries with members of group
 * object ids included in input are not changed by this
 *
 * @author <a href="mailto:sepp@upb.de">Christian Schmidt</a>
 * @param array(string) target - the addresses to replace groups in
 * @return array(string) input-array with groups replaced by members of groups
 */
private array(string) expand_local_addresses(array(string) target)
{
    array(string) result=({});
    for(int i=0;i<sizeof(target);i++)
    {
        if(objectp(MODULE_USERS->lookup(target[i])))
        {
            result+=({target[i]});
            continue;
        }
        // FIXME: this is dead code! all groups should have been removed by now.
        object tmp=MODULE_GROUPS->lookup(target[i]);
        if(objectp(tmp))
        {
            result+=tmp->get_members();
            continue;
        }
        if(sscanf(target[i],"%d",int oid))
        {
            result+=({target[i]});
            continue;
        }
        LOG_FORWARD("expand_local_addresses: failed to find \""+target[i]+"\"");
    }
    return result; //now contains only user-names and object-ids
}

private int send_local_single(string recipient, Messaging.Message msg)
{
        int oid;
        if(!sscanf(recipient,"%d",oid)==1)
        {
            object user=MODULE_USERS->lookup(recipient);
            Messaging.BaseMailBox box = Messaging.get_mailbox(user);
            msg->grant_access(user); //the recipient gets all rights on his/her copy
            msg->this()->set_acquire(box->this());
            box->add_message(msg);
        }
        else //store message on object
        {
            object target=_Database->find_object(oid);
            if(!objectp(target)) return 0; //invalid object-id, do nothing
//            msg->grant_access(target->query_attribute(OBJ_OWNER));
            msg->this()->set_acquire(target); 
            Messaging.BaseMailBox box = Messaging.get_mailbox(target);
            if(objectp(box)) //target can be accessed as mailbox
                box->add_message(msg);
            else
                Messaging.add_message_to_object(msg,target);
        }
        return 1;
}
/**
 * send a message to local recipients
 *
 * @author <a href="mailto:sepp@upb.de">Christian Schmidt</a>
 * @param array(string) target - the local addresses to send the message to
 * @param Message msg - the message to send (WARNING msg is destructed by sending!)
 * @return int 1 if successful
 */
private int send_local(array(string) target, Messaging.Message msg)
{
    if(sizeof(target)==0 || !arrayp(target)) return 0;
    
    int result=1; //success

    // FIXME: expand_local_addresses should not be needed anymore:
    array(string) asLocal=expand_local_addresses(target); //resolve aliases & forwards
    LOG_FORWARD("expanded local adresses are:%O",asLocal);

    for(int i=sizeof(asLocal)-1;i>0;i--) //duplicate message if more than one recipient
    {
        Messaging.Message copy=msg->duplicate();
        if(send_local_single(asLocal[i],copy)==0)
        {
            LOG_FORWARD("failed to send message #"+copy->get_object_id()+" to: "+asLocal[i]);
            copy->delete();
            destruct(copy);
            result=0;
        }
    }
    if(send_local_single(asLocal[0],msg)==0) //last recipient gets "original" message
    {
        LOG_FORWARD("failed to send message #"+msg->get_object_id()+" to: "+asLocal[0]);
        msg->delete();
        destruct(msg);
        result=0;
    }

    if(result==0) LOG_FORWARD("Warning!! send_local encountered errors - some recipients failed!");

    return result;
}

/**
 * send a message to a remote address (-> send e-mail)
 *
 * @author <a href="mailto:sepp@upb.de">Christian Schmidt</a>
 * @param string address - the address to send the mail to
 * @param string rawText - the text of the message (rfc2822-format!)
 * @param string|void envFrom - the sender-value of the SMTP-envelope
 * @return int 1 if successful, 0 if not
 */
private int send_remote(string|array address, string rawText, string envFrom)
{
    string fixed;
    if( sscanf(envFrom,"%*s<%s>",fixed) == 0 )
    {
        LOG_FORWARD("send_remote: illegal envFrom! : "+envFrom);
        return 0;
    }
    int l;
    if ( arrayp(address) && (l=sizeof(address)) > 10 ) {
	for ( int i = 0; i < sizeof(address); i+=10 ) {
	    array users;
	    if ( i + 10 >= l )
		users = address[i..];
	    else
		users = address[i..i+9];
	    get_module("smtp")->send_mail_raw(users, rawText, fixed);
	    LOG_FORWARD("Message chunked delivered to "+sprintf("%O", users));
	}
    }
    else {
	get_module("smtp")->send_mail_raw(address, rawText, fixed);
	LOG_FORWARD("Message delivered to " + sprintf("%O", address));
    }
    
    return 1;
}

/**
 * add an alias to system-aliases
 * if an alias with the given name already exists, the alias
 * will point to a list of targets; if you want to replace an alias
 * completely, you'll have to delete it first
 *
 * @author <a href="mailto:sepp@upb.de">Christian Schmidt</a>
 * @param string alias - the name of the alias to add
 * @param string taget - the target the alias should point to
 * @return 1 if successful
 * @see delete_alias
 */
int add_alias(string alias, string target)
{
    mAliases[alias]+=({target});
    require_save();
    return 1;
}

/**
 * remove an alias from the system aliases
 *
 * @author <a href="mailto:sepp@upb.de">Christian Schmidt</a>
 * @param string alias - the alias to delete
 * @return 1 if successful, 0 if alias does not exist
 * @see add_alias
 */
int delete_alias(string alias)
{
    if(arrayp(mAliases[alias]))
    {
        m_delete(mAliases,alias);
        require_save();
        return 1;
    }
    else return 0;
}

/**
 * add a forward for a specific user
 * if user already has a forward, it is extended by the given target
 * target may be a remote e-mail address or a local system address
 * (user, group, object-id, alias)
 *
 * @author <a href="mailto:sepp@upb.de">Christian Schmidt</a>
 * @param object user - the user to add a forward for
 * @param string forward - the target address to add (e-mail or other valid system-address)
 * @param int|void no_more - set to 1, if this forward is "final", so mails get stored at this adress,
 *                           no matter if a forward exists for this target, too
 * @return int 1 if successful, 0 if not
 */
int add_forward(object user, string forward, int|void no_more)
{
    if(intp(no_more) && no_more==1) forward="/"+forward;
    if(user->get_object_class() && CLASS_USER)
    {
        string name=user->get_identifier();
	if ( forward == name )
	  steam_error("add_forward: Unable to resolve forward to itself !");
        mForwards[name]+=({forward});
        require_save();
        return 1;
    }
    else
    {
        LOG_FORWARD("ERROR, add_forward() called for non-user object #"
                     +user->get_object_id());
        return 0;
    }
}

/**
 * remove a user-forward
 *
 * @author <a href="mailto:sepp@upb.de">Christian Schmidt</a>
 * @param object user - the user to remove forwarding for
 * @return int 1 if successful, 0 otherwise
 */
int delete_forward(object user)
{
    if(user->get_object_class() && CLASS_USER)
    {
        if(arrayp(mForwards[user->get_identifier()]))
        {
            m_delete(mForwards,user->get_identifier());
            require_save();
            return 1;
        }
        else return 0; //no forward for this user
    }
    else return 0; //user is no sTeam-user-object
}

/**
 * get the current forward for a user
 *
 * @author <a href="mailto:sepp@upb.de">Christian Schmidt</a>
 * @param object user - the user to get forward for
 * @return array(string) of forwards or 0 if user is not a sTeam-user-object
 */
array(string) get_forward(object user)
{
    if(user->get_object_class() && CLASS_USER)
        return mForwards[user->get_identifier()];
    else return 0;
}

/*
string dump_data()
{
    string res;
    res=sprintf("forwards:%O aliases:%O",mForwards,mAliases);
    LOG_FORWARD("current data of forward-module:\n"+res);
    LOG_FORWARD("mailserver is: "+_mailserver+":"+_mailport);
    return res;
}
*/
