/* Copyright (C) 2000-2004  Thomas Bopp, Thorsten Hampel, Ludger Merkens
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 * $Id: Document.pike,v 1.1.1.1 2005/09/21 14:22:14 exodusd Exp $
 */

constant cvs_version="$Id: Document.pike,v 1.1.1.1 2005/09/21 14:22:14 exodusd Exp $";

//! A Document is an object with content (bytes). The content is stored
//! in the persistency layer of steam. When content is changed the
//! EVENT_UPLOAD and EVENT_DOWNLOAD events are trigered.

inherit "/classes/Object" :  __object;
inherit "/base/content"   : __content;

#include <attributes.h>
#include <classes.h>
#include <macros.h>
#include <events.h>
#include <types.h>
#include <config.h>
#include <database.h>


class Lock {
  object lockingObj; // the object that locks - group or user
  string          lockType; // read, write, shared
  int              expires; // expiring timestamp

  static int check_lock(object user) {
    if ( lockingObj->get_object_class() & CLASS_GROUP )
      return lockingObj->is_member(user);
    return lockingObj == user;
  }

  void create(object group, string type, int|void expire) {
    lockingObj = group;
    lockType = type;
    if ( expire == 0 )
      expires = time() + (60*60*24*7); // one week
  }

  private static object share(object group) {
    return 0; // do not share
  }

  object lock(object group, string type) {
    if ( !check_expire() ) {
      if ( !check_lock(group) )
	steam_error("Unable to re-lock LOCK.");
    }
    lockType = type;
  }

  int check_expire() {
    if ( time() > expires )
      return 1;
    return 0;
  }

  void check(object user, string type) {
    if ( check_expire() ) // do nothing when lock is expired
      return;
    
    if ( lockType!= type )
      return;
    if ( !check_lock(user) )
      steam_error("File is locked "+lockType+" by "+
		  lockingObj->get_identifier());
  }
  
  // no persistency of locks right now
  void load(mapping data) {
  }
  mapping save() {
  }
}

static void init_document() { }

/**
 * Init callback function.
 *  
 */
final static void
init()
{
    __object::init();
    __content::init_content();
    init_document();
}

/**
 * Called after the document was loaded by database.
 *  
 */
static void load_document()
{
}

/**
 * Called after the document was loaded by database.
 *
 */
static void load_object()
{
    load_document();
}

/**
 * Duplicate the Document and its content.
 *  
 * @return the duplicate of this document.
 */
object duplicate(void|mapping params)
{
  // DocumentFactory deals with content_obj variable
  if ( !mappingp(params) )
    params = ([ ]);
  
  params->content_obj = this();
  
  if ( params->content_id ) 
    params->content_obj = 0;
  
  params->mimetype = do_query_attribute(DOC_MIME_TYPE);

  return ::duplicate( params );
}

/**
 * Destructor function of this object removes all references
 * and deletes the content.
 *  
 */
static void 
delete_object()
{
    if ( mappingp(mReferences) ) {
	foreach(indices(mReferences), object o) {
	    if ( !objectp(o) ) continue;
	    
	    o->removed_link();
	}
    }
    // delete all versions
    mapping versions = do_query_attribute(DOC_VERSIONS);
    if ( mappingp(versions) ) {
      foreach(values(versions), object v) {
	if ( objectp(v) ) {
	  if ( v->query_attribute(OBJ_VERSIONOF) != this() ) {
	    steam_error("Version missmatch in object - please report!");
	  }
	  v->delete();
	}
      }
    }
    ::delete_content();
    ::delete_object();
}

/**
 * Adding data storage is redirected to objects functionality.
 *  
 * @param function a - store function
 * @param function b - restore function
 * @return whether adding was successfull.
 */
static bool
add_data_storage(string a,function b, function c, int|void d)
{
    return __object::add_data_storage(a,b,c,d);
}

/**
 * Get the content size of this document.
 *  
 * @return the content size in bytes.
 */
int get_content_size()
{
    return __content::get_content_size();
}

/**
 * Returns the id of the content inside the Database.
 *  
 * @return the content-id inside database
 */
final int get_content_id()
{
  return __content::get_content_id();
}

/**
 * Callback function when a download has finished.
 *  
 */
static void download_finished()
{
    run_event(EVENT_DOWNLOAD_FINISHED, CALLER);
}

/**
 * give status of Document similar to file->stat()
 *
 * @param  none
 * @return ({ \o700, size, atime, mtime, ctime, uid, 0 })
 * @see    file_stat
 * @author Ludger Merkens 
 */
array stat()
{
    int creator_id = get_creator() ? get_creator()->get_object_id() : -1;
    
    return ({
	33279,  // -rwx------
	    get_content_size(),
	    do_query_attribute(OBJ_CREATION_TIME),
	    do_query_attribute(DOC_LAST_MODIFIED),
	    do_query_attribute(DOC_LAST_ACCESSED),
	    creator_id,
	    creator_id,
	    query_attribute(DOC_MIME_TYPE), // aditional, should not be a prob?
	    });
}

int get_object_class() { return ::get_object_class() | CLASS_DOCUMENT; }
final bool is_document() { return true; }

/**
 * content function used for download, this function really resides in
 * base/content and this overridden function just runs the appropriate event
 *  
 * @return the function for downloading (when socket has free space)
 * @see receive_content
 */
function get_content_callback(mapping vars)
{
    object obj = CALLER;

    if ( functionp(obj->get_user_object) && objectp(obj->get_user_object()) )
	obj = obj->get_user_object();

    check_lock("read");
    try_event(EVENT_DOWNLOAD, obj);

    do_set_attribute(DOC_LAST_ACCESSED, time());

    run_event(EVENT_DOWNLOAD, obj);

    return __content::get_content_callback(vars);
}

/**
 * Get the content of this document as a string.
 *  
 * @param int|void len - optional parameter length of content to return.
 * @return the content or the first len bytes of it.
 * @see get_content_callback
 */
string get_content(int|void len)
{
    string      content;
    object obj = CALLER;

    check_lock("read");
    
    try_event(EVENT_DOWNLOAD, obj);
    content = ::get_content(len);

    do_set_attribute(DOC_LAST_ACCESSED, time());

    run_event(EVENT_DOWNLOAD, obj);
    return content;
}

object get_content_file(string mode, mapping vars, string|void client) 
{
  return ((program)"/kernel/DocFile.pike")(this(), mode, vars, client);
}


void check_lock(string type)
{
  object lock = do_query_attribute(OBJ_LOCK);
  if ( objectp(lock) )
    lock->check(this_user(), type);
}


/**
 * Lock the content of this object.
 *  
 * @param object group - the locking group.
 * @param string type - the type of the lock, "read" or "write"
 * @return the content or the first len bytes of it.
 * @see get_content_callback
 */
object lock_content(object group, string type)
{
  object lock = do_query_attribute(OBJ_LOCK);
  set_attribute(OBJ_LOCK, 0); // this checks write permissions
  if ( objectp(lock) ) {
    set_attribute(OBJ_LOCK, lock);
    return lock->lock(group, type);
  }
  lock = Lock(group, type);
  set_attribute(OBJ_LOCK, lock);
  return lock;
}


/**
 * Callback function called when upload is finished.
 *  
 */
static void content_finished()
{
  __content::content_finished();
  run_event(EVENT_UPLOAD, this_user());
}

/**
 * content function used for upload, this function really resides in
 * base/content and this overridden function just runs the appropriate event
 *  
 * @return the function for uploading (called each time a chunk is received)
 * @author Thomas Bopp (astra@upb.de) 
 * @see get_content_callback
 */
function receive_content(int content_size)
{
    object obj = CALLER;
    if ( (obj->get_object_class() & CLASS_USER) && 
	 (functionp(obj->get_user_object) ) &&
	 objectp(obj->get_user_object()) )
	obj = obj->get_user_object();
    
    check_lock("write");

    try_event(EVENT_UPLOAD, obj, content_size);

    int version = do_query_attribute(DOC_VERSION);
    if ( !version )
      version = 1;
    else {
      seteuid(get_creator());
      
      object oldversion = duplicate( ([ "content_id": get_content_id(), ])); 
      oldversion->set_acquire(this());
      oldversion->set_attribute(OBJ_VERSIONOF, this());
      oldversion->set_attribute(DOC_LAST_MODIFIED, do_query_attribute(DOC_LAST_MODIFIED));
      oldversion->set_attribute(DOC_USER_MODIFIED, do_query_attribute(DOC_USER_MODIFIED));
      oldversion->set_attribute(OBJ_CREATION_TIME, do_query_attribute(OBJ_CREATION_TIME));
      mapping versions = do_query_attribute(DOC_VERSIONS);
      if ( !mappingp(versions) )
	versions = ([ ]);
      versions[version] = oldversion;
      
      version++;
      do_set_attribute(DOC_VERSIONS, versions);
    }
    do_set_attribute(DOC_VERSION, version);
    
    set_attribute(DOC_LAST_MODIFIED, time());
    set_attribute(DOC_USER_MODIFIED, this_user());
    return __content::receive_content(content_size);
}

/**
 * See whether the content is locked by someone or not.
 *  
 * @return the locking object.
 */
object is_locked()
{
    return do_query_attribute(OBJ_LOCK);
}

object get_previous_version()
{
    mapping versions = do_query_attribute(DOC_VERSIONS);
    int version = do_query_attribute(DOC_VERSION);
    if ( objectp(versions[version-1]) )
	return versions[version-1];
    while ( version-- > 0 )
	if ( objectp(versions[version]) )
	    return versions[version];
    return 0;
}

string get_etag() 
{
  int lm = do_query_attribute(DOC_LAST_MODIFIED);
  string etag = sprintf("%018x",iObjectID + (lm<<64));
  if ( sizeof(etag) > 18 ) etag = etag[(sizeof(etag)-18)..];
  return etag[0..4]+"-"+etag[5..10]+"-"+etag[11..17];
}


string describe()
{
    return get_identifier()+"(#"+get_object_id()+","+
	master()->describe_program(object_program(this_object()))+","+
	get_object_class()+","+do_query_attribute(DOC_MIME_TYPE)+")";
}
