/* Copyright (C) 2000-2004  Thomas Bopp, Thorsten Hampel, Ludger Merkens
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 * $Id: content.pike,v 1.1.1.1 2005/09/21 14:22:14 exodusd Exp $
 */

constant cvs_version="$Id: content.pike,v 1.1.1.1 2005/09/21 14:22:14 exodusd Exp $";

//! Basic class to support Objects with content (Documents). Content
//! is stored in the persistency layer of sTeam.


inherit Thread.Mutex : w_mutex;


#include <macros.h>
#include <config.h>
#include <assert.h>
#include <database.h>
#include <classes.h>

private static int              iContentID;
private static int       iContentSize = -1;
private static object      oDbUploadHandle;

private static object           oLockWrite;
private static object            oLockRead;
private static int            iDownloads=0;
private static array(string) asUploadCache;

static bool add_data_storage(string s,function a, function b,int|void d);
static void download_finished();
static void require_save(string|void a, string|void b) { _Persistence->require_save(a,b); }

/**
 * This callback function is registered via add_data_storage to provide
 * necessary data for serialisation. The database object calls this function
 * to save the values inside the database.
 *
 * @param  none
 * @return a mixed value containing the new introduced persistent values
 *         for content
 * @see    restore_content_data
 * @see    add_data_storage
 */
mixed retrieve_content_data(string|void index)
{
    if ( CALLER != _Database )
	THROW("Illegal call to retrieve_content_data()", E_ACCESS);
    if (index) {
        switch(index) {
          case "CONTENT_SIZE": return iContentSize;
          case "CONTENT_ID": return iContentID;
        }
    }
    else
        return ([ "CONTENT_SIZE": iContentSize,
                  "CONTENT_ID": iContentID ]);
}

/**
 * This callback function is used to restore data previously read from
 * retrieve_content_data to restore the state of reading
 *
 * @param  a mixed value previously read via retrieve_content_data
 * @return void
 * @see    retrieve_content_data
 * @see    add_data_storage
 */
void restore_content_data(mixed data, string|void index)
{
    if ( CALLER != _Database )
	THROW("Illegal call to restore_content_data()", E_ACCESS);

    if (index) {
        switch (index) {
          case "CONTENT_SIZE" : iContentSize = data; break;
          case "CONTENT_ID": iContentID = data; break;
        }
    }
    else if (arrayp(data)) {
        [ iContentSize, iContentID ] = data;
    }
    else {
        iContentSize = data["CONTENT_SIZE"];
        iContentID = data["CONTENT_ID"];
    }
}


/**
 * Initialize the content. This function only sets the data storage
 * and retrieval functions.
 *  
 */
static void init_content()
{
    add_data_storage(STORE_CONTENT, retrieve_content_data,
                     restore_content_data, 1);
}


class DownloadHandler {
    object odbhandle;
    void create(object oDbHandle) {
        odbhandle = oDbHandle;
    }
    /**
     * This function gets called from the socket object associated with
     * a user downloads a chunk. It cannot be called directly - the
     * function get_content_callback() has to be used instead.
     *
     * @param  int startpos - the position
     * @return a chunk of data | 0 if no more data is present
     * @see    receive_content
     * @see    get_content_callback
     */
    string send_content(int startpos) {
        if ( !objectp(odbhandle) )
            return 0;
        
        string buf = odbhandle->read(DB_CHUNK_SIZE);
        if ( stringp(buf) ) 
	  return buf;

        if (objectp(odbhandle))
        {
            iDownloads--;
            if (iDownloads == 0)           // last download finished?
                destruct(oLockRead);      // release writing lock
        
            destruct(odbhandle);
        }
        // callback to notify about finished downloading
        download_finished();
        
        return 0;
    }

    void destroy() {
        if ( objectp(odbhandle) )
        {
            iDownloads --;
            if (iDownloads ==0)
                destruct(oLockRead);
            destruct(odbhandle);
        }
    }
}


class UploadHandler {
    object odbhandle;

    void create(object oDbHandle) {
        odbhandle= oDbHandle;
    }
    /**
     * save_chunk is passed from receive_content to a data storing process, 
     * to store one chunk of data to the database.
     *
     * @param   string chunk - the data to store
     * @param   int start    - start position of the chunk relative to complete
     *                         data to store.
     * @param   int end      - similar to start
     * @return  void
     * @see     receive_content
     */
    void save_chunk(string chunk) {
        if ( !stringp(chunk) )
        {
            local_content_finished();
            return;
        }
        odbhandle->write(chunk);
    }
    
    /**
     * This function gets called, when an upload is finished. All locks
     * are removed and the object is marked for the database save demon
     * (require_save()).
     *  
     * @see save_chunk
     */
    void
    local_content_finished() {
        int iWrittenSize = odbhandle->sizeof();
        int iWrittenID = odbhandle->dbContID();
        destruct(odbhandle); // will call close() and unlock
	// clean old content?
        iContentID = iWrittenID;
        iContentSize= iWrittenSize;
        require_save(STORE_CONTENT);
        content_finished();
        return ;
    }

    void destroy() {
        if (odbhandle)
            content_finished();
    }
    
}

/**
 * The function returns the function to download the content. The
 * object is configured and locked for the download and the
 * returned function send_content has to be subsequently called
 * in order to get the data. 
 * 
 * @param  none
 * @return function "send_content" a function that returns the content
 *         of a given range.
 * @see    send_content
 */
function get_content_callback(mapping vars)
{
    int t;
    float tf;
    
    if ( iContentID == 0 )
	LOG_DB("get_content_callback: missing ContentID");
    
    if (iDownloads == 0)
    {
	object l = w_mutex::lock();      // wait for content_cleanup
	oLockRead = l;
    }
    iDownloads++;

    object oDbDownloadHandle =
        _Database->new_db_file_handle(iContentID,"r");
    object oDownloadHandler =
        DownloadHandler(oDbDownloadHandle);
    ASSERTINFO(objectp(oDbDownloadHandle), "No File handle found !");
    LOG("db_file_handle() allocated, now sending...\n");
    return oDownloadHandler->send_content;
}


/**
 * Allocate a file handle.
 *  
 */
private final static void allocate_file_handle()
{
    catch(oLockWrite = w_mutex::trylock()); // one upload only
    if (!oLockWrite)
	THROW("no simultanous write access on content", E_ACCESS);
    
    LOG_DB("content.receive_content preparing upload to cID:"+iContentID);
    oDbUploadHandle = _Database->new_db_file_handle(iContentID,"wtc");
    if (!iContentID)
    {
	iContentID = oDbUploadHandle->dbContID();
	LOG_DB("created new cID:"+iContentID);
    }
    if (!iContentID)
	LOG_DB("missing iContentID");
}

/**
 * This function gets called to initialize the download of a content.
 * The returned function has to be called subsequently to write data.
 * After the upload is finished the function has to be called with
 * the parameter 0.
 *
 * @param  int content_size -- the size of the content that will be
 *         passed in chunks to the function returned
 * @return a function, that will be used as call_back by the object
 *         calling receive_content to actually store the data.
 * @see    save_chunk
 * @see    send_content
 * @author Ludger Merkens 
 */
function receive_content(int content_size)
{
    object oHandle = _Database->new_db_file_handle(0,"wtc");
    object oUploadHandler = UploadHandler(oHandle);
    return oUploadHandler->save_chunk;
}

object get_upload_handler(int content_size)
{
    object oHandle = _Database->new_db_file_handle(0,"wtc");
    return UploadHandler(oHandle);
}

/**
 * update_content_size - reread the content size from the database
 * this is a hot-fix function, to allow resyncing with the database tables,
 * this function definitively should be obsolete.
 *
 * @param none
 * @return nothing
 * @author Ludger Merkens
 */
void update_content_size()
{
    object db_handle = _Database->new_db_file_handle(iContentID,"r");
    iContentSize = db_handle->sizeof();
    destruct(db_handle);     // will call close()
    require_save(STORE_CONTENT);
}

/**
 * evaluate the size of this content
 *
 * @param  none
 * @return int - size of content in byte
 * @author Ludger Merkens 
 */
int 
get_content_size()
{
    if (!iContentID) // no or unfinished upload
        return 0;

    if ( iContentSize <= 0 )
    {
        object db_handle = _Database->new_db_file_handle(iContentID,"r");
        iContentSize = db_handle->sizeof();
        destruct(db_handle);     // will call close()
        require_save(STORE_CONTENT);
    }
    return iContentSize;
}


/**
 * Get the ID of the content in the database.
 *  
 * @return the content id
 * @author <a href="mailto:astra@upb.de">Thomas Bopp</a>) 
 */
int get_content_id()
{
    return iContentID;
}

// versioning
void set_content_id(int id)
{
  if ( CALLER != get_factory(CLASS_DOCUMENT) )
      steam_error("Unauthorized call to set_content_id() !");
  iContentID = id;
  require_save(STORE_CONTENT);
}

/**
 * Get the content of the object directly. For large amounts
 * of data the download function should be used. It is possible
 * to pass a len parameter to the function so only the first 'len' bytes
 * are being returned.
 *  
 * @param int|void len
 * @return the content
 * @author <a href="mailto:astra@upb.de">Thomas Bopp</a>) 
 */
static string
_get_content(int|void len)
{
    string buf;
    LOG_DB("content.get_content() of " + iContentID);
    
    if (iDownloads == 0)
    {
        object l;
	catch(l = w_mutex::trylock());
	if ( !objectp(l) )
	    THROW("no simultanous write access on content", E_ACCESS);
	oLockWrite = l;
    }
    iDownloads++;

    mixed cerr = catch
    {
	object db_handle = _Database->new_db_file_handle(iContentID,"r");
	buf = db_handle->read(len);
	destruct(db_handle);
    };

    iDownloads--;
    if (iDownloads == 0)
	destruct(oLockWrite);

    if (cerr)
	throw(cerr);
    
    return buf;
}

string get_content(int|void len)
{
    return _get_content(len);
}

/**
 * set_content, sets the content of this instance.
 *
 * @param  string cont - this will be the new content
 * @return int         - content size (or -1?)
 * @see    receive_content, save_content
 *
 */
int
set_content(string cont)
{
    if ( !stringp(cont) ) 
	error("set_content: no content given - needs string !");
    function save = receive_content(strlen(cont));
    save(cont);  // set the content
    save(0);     // flush the buffers
    return strlen(cont);
}

/**
 * When the object is deleted its content has to be removed too.
 *  
 */
final static void 
delete_content()
{
    if (iContentID)
    {
        object l = w_mutex::lock();
        object h = _Database->new_db_file_handle(iContentID, "wct");
        h->close();
    }
}


static void content_finished() {
    // call for compatibility reasons
}


