/*
* lexer.l
*
* lexical analyzer for wikicode
*/

%{

#include "boolean.h"
#include "debug.h"
#include "error.h"
#include "globals.h"
#include "transforms.h"
#include "yyexternals.h"

extern char* yyinbuffer;

#define YY_INPUT(buf, result, max_size)\
{\
int n, len;\
len = strlen(yyinbuffer);\
result = len < max_size ? len : max_size;\
for ( n = 0; n < result; n++ )\
    buf[n] = yyinbuffer[n];\
yyinbuffer = &yyinbuffer[result];\
}

%}


ALPHANUM    ([a-z]|[A-Z]|[1-9])
LINK_TEXT   [^\]]+
PIKE_CODE   [^\@]+
EMBED_TEXT  [^\}]+
SPACE       (\t|\n|" ")
NOT_SPECIAL [^\n\[=\*#\-]
BLANKLINE   ^(\s*)
BLANK       \r\n\r\n
TABS        (\t+)
ENDDEF      [^:]+

HYPERLINK   "[http://"{LINK_TEXT}"]"
HYPERLINKS  "[https://"{LINK_TEXT}"]"
BARELINK    (("http")|("ftp"))"://"[^[:blank:]\n]+
IMAGE       "[["[iI]"mage:"{LINK_TEXT}"]]"
WIKILINK    "[["{LINK_TEXT}"]]"{ALPHANUM}*
WIKILINKTXT {SPACE}([A-Z]([a-z]+))([A-Z]([a-z]+))+
ANNOTATION  "["{LINK_TEXT}"["{LINK_TEXT}"]]"
EMBED       "{"{EMBED_TEXT}"}"
WIKIPIKE    "@@"{PIKE_CODE}"@@"

BOLD        '''
ITALIC      ''
H1_START    ^=
H2_START    ^==
H3_START    ^===
H4_START    ^====
H1          =
H2          ==
H3          ===
H4          ====
BULLET      ^\*
NUMBER      ^#
INDENT      ^\:
HR          ^"----"-*
LIST        ^[\*#:]+
PRE         ^" "
DEFINITION  ^(\t+){ENDDEF}:\t
TAG         ("<"(.+)">")

%option noyywrap

%%

{LIST}          { make_list(yytext); }
{BOLD}          { bold();            }
{ITALIC}        { italic();          }
{TAG}           { tag(yytext);       }
{PRE}           { preformat(yytext); }
{H4_START}      { heading(4, TRUE);  }
{H3_START}      { heading(3, TRUE);  }
{H2_START}      { heading(2, TRUE);  }
{H1_START}      { heading(1, TRUE);  }
{H4}            { heading(4, FALSE); }
{H3}            { heading(3, FALSE); }
{H2}            { heading(2, FALSE); }
{H1}            { heading(1, FALSE); }
{HR}            { hr();              }
{BARELINK}      { barelink(yytext);  }
{IMAGE}         { image(yytext);     }
{WIKILINK}      { linkInternal(yytext);  }
{WIKILINKTXT}   { linkInternal(yytext);  }
{ANNOTATION}    { annotationInternal(yytext); }
{HYPERLINK}     { hyperlink(yytext); }
{HYPERLINKS}    { hyperlink(yytext); }
{DEFINITION}    { make_def(yytext); }
{EMBED}         { embed(yytext); }
{WIKIPIKE}      { pi_pike(yytext); }
^\n             { blank_line();      }
{BLANKLINE}     { blank_line();      }
{BLANK}         { blank_line();      }
^{NOT_SPECIAL}  { paragraph(yytext); }
{SPACE}         { plaintext(yytext); }
.               { plaintext(yytext); }
