#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>
#include <pthread.h>
#include <signal.h>

#include <simclist.h>

#include "parser.h"

#include "sshguard_log.h"
#include "sshguard_addresskind.h"
#include "sshguard_whitelist.h"
#include "sshguard_procauth.h"
#include "sshguard_fw.h"
#include "sshguard.h"

#define MAX_LOGLINE_LEN     1000

/* switch from 0 (normal) to 1 (suspended) with SIGTSTP and SIGCONT respectively */
int suspended;
/* boolean for debugging enabled/disabled */
int debugging;

list_t limbo;
list_t hell;

/* elements for the list of attackers (both in limbo and hell lists) */
typedef struct {
    char addr[ADDRLEN];
    int addrkind;
    int service;
    time_t whenfirst;
    time_t whenlast;
    unsigned int numhits;
} ipentry_t;

/* dynamic configuration options */
struct {
    time_t pardon_threshold, stale_threshold;
    unsigned int abuse_threshold;
} opts;

/* fill an ipentry_t structure for usage */
static inline void ipentryinit(ipentry_t *ipe, char *ipaddr, int addrkind, int service);
/* dumps usage message to standard error */
static void usage(void);
/* handler for termination-related signals */
void sigfin_handler(int signo);
/* handler for suspension/resume signals */
void sigstpcont_handler(int signo);
/* handle an attack: addr is the author, addrkind its address kind, service the attacked service code */
void report_address(char *addr, int addrkind, int service);
/* cleanup false-alarm attackers from limbo list (ones with too few attacks in too much time) */
void purge_limbo_stale(void);
/* release blocked attackers after their penalty expired */
void *pardonBlocked(void *par);
/* called at exit(): flush blocked addresses and finalize subsystems */
void finishup(void);


int main(int argc, char *argv[]) {
    pthread_t tid;
    int optch;
    int retv;
    char buf[MAX_LOGLINE_LEN];
    
    /* initializations */
    suspended = 0;

    /* pending and blocked address lists */
    list_init(&limbo);
    list_init(&hell);

    /* whitelisting system */
    if (whitelist_init() != 0 || whitelist_conf_init() != 0) {
        fprintf(stderr, "Could not nitialize the whitelist engine.\n");
        exit(1);
    }

    /* process authorization system */
    if (procauth_init() != 0) {
        fprintf(stderr, "Could not initialize the process authorization subsystem.");
        exit(1);
    }

    /* parsing the command line */
    opts.pardon_threshold = DEFAULT_PARDON_THRESHOLD;
    opts.stale_threshold = DEFAULT_STALE_THRESHOLD;
    opts.abuse_threshold = DEFAULT_ABUSE_THRESHOLD;
    debugging = 0;
    while ((optch = getopt(argc, argv, "p:s:a:w:f:dh")) != -1) {
        switch (optch) {
            case 'p':   /* pardon threshold interval */
                opts.pardon_threshold = strtol(optarg, (char **)NULL, 10);
                if (opts.pardon_threshold < 1) {
                    fprintf(stderr, "Doesn't make sense to have a pardon time lower than 1 second. Terminating.");
                    exit(1);
                }
                break;
            case 's':   /* stale threshold interval */
                opts.stale_threshold = strtol(optarg, (char **)NULL, 10);
                if (opts.stale_threshold < 1) {
                    fprintf(stderr, "Doesn't make sense to have a stale threshold lower than 1 second. Terminating.");
                    exit(1);
                }
                break;
            case 'a':   /* abuse threshold count */
                opts.abuse_threshold = strtol(optarg, (char **)NULL, 10);
                if (opts.abuse_threshold < 1) {
                    fprintf(stderr, "Doesn't make sense to have an abuse threshold lower than 1 attemt. Terminating.");
                    exit(1);
                }
                break;
            case 'w':   /* whitelist entries */
                if (optarg[0] == '/' || optarg[0] == '.') {
                    /* add from file */
                    if (whitelist_file(optarg) != 0) {
                        fprintf(stderr, "Could not handle whitelisting for %s.\n", optarg);
                        exit(1);
                    }
                } else {
                    /* add raw content */
                    if (whitelist_add(optarg) != 0) {
                        fprintf(stderr, "Could not handle whitelisting for %s.\n", optarg);
                        exit(1);
                    }
                }
                break;
            case 'f':   /* process pid authorization */
                if (procauth_addprocess(optarg) != 0) {
                    fprintf(stderr, "Could not parse service pid configuration '%s'.\n", optarg);
                    exit(1);
                }
                break;
            case 'd':
                debugging = 1;
                break;
            case 'h':
            default:
                usage();
                exit(1);
        }
    }

    whitelist_conf_fin();

    /* logging system */
    sshguard_log_init();

    /* address blocking system */
    if (fw_init() != FWALL_OK) {
        sshguard_log(LOG_CRIT, "Could not init firewall. Terminating.\n");
        fprintf(stderr, "Could not init firewall. Terminating.\n");
        exit(1);
    }


    /* set finalization routine */
    atexit(finishup);

    /* suspension signals */
    signal(SIGTSTP, sigstpcont_handler);
    signal(SIGCONT, sigstpcont_handler);

    /* termination signals */
    signal(SIGTERM, sigfin_handler);
    signal(SIGHUP, sigfin_handler);
    signal(SIGINT, sigfin_handler);

    /* set debugging value for parser/scanner ... */
    yydebug = debugging;
    yy_flex_debug = debugging;
    
    /* start thread for purging stale blocked addresses */
    if (pthread_create(&tid, NULL, pardonBlocked, NULL) != 0) {
        perror("pthread_create()");
        exit(2);
    }


    /* initialization successful */
    
    sshguard_log(LOG_INFO, "Started successfully [(a,p,s)=(%u, %u, %u)], now ready to scan.", \
            opts.abuse_threshold, (unsigned int)opts.pardon_threshold, (unsigned int)opts.stale_threshold);


    while (fgets(buf, MAX_LOGLINE_LEN, stdin) != NULL) {
        if (suspended) continue;

        retv = parse_line(buf);
        if (retv != 0) {
            /* sshguard_log(LOG_DEBUG, "Skip line '%s'", buf); */
            continue;
        }

        /* extract the IP address */
        sshguard_log(LOG_DEBUG, "Matched IP address %s", attackparser_targetaddr);
       
        /* report IP */
        report_address(attackparser_targetaddr, attackparser_targetaddrkind, attackparser_service);
    }

    /* let exit() call finishup() */
    exit(0);
}


void report_address(char *addr, int addrkind, int service) {
    ipentry_t *tmpent = NULL;
    unsigned int pos;
    int ret;

    /* clean list from stale entries */
    purge_limbo_stale();

    /* protected address? */
    if (whitelist_match(addr, addrkind)) {
        sshguard_log(LOG_INFO, "Pass over address %s because it's been whitelisted.", addr);
        return;
    }
    
    /* search entry in list */
    pos = 0;
    list_iterator_start(&limbo);
    while (list_iterator_hasnext(&limbo)) {
        tmpent = list_iterator_next(&limbo);
        if (strcmp(tmpent->addr, addr) == 0) /* found! */
            break;
        pos++;
    }
    list_iterator_stop(&limbo);
    
    if (pos < list_size(&limbo)) { /* then found */
        /* update last hit */
        tmpent->whenlast = time(NULL);
        tmpent->numhits++;
        if (tmpent->numhits >= opts.abuse_threshold) { /* block IP */
            sshguard_log(LOG_NOTICE, "Blocking %s: %u failures over %u seconds.\n", tmpent->addr, tmpent->numhits, tmpent->whenlast - tmpent->whenfirst);
            ret = fw_block(addr, addrkind, service);
            if (ret != FWALL_OK) sshguard_log(LOG_ERR, "Blocking command failed. Exited: %d", ret);
            /* re-get elem for deleting it from limbo list */
            tmpent = list_extract_at(& limbo, pos);
            list_append(&hell, tmpent);
        }
        return;
    }
    
    /* otherwise: insert the new item */
    tmpent = malloc(sizeof(ipentry_t));
    ipentryinit(tmpent, addr, addrkind, service);
    list_append(&limbo, tmpent);
}

static inline void ipentryinit(ipentry_t *ipe, char *ipaddr, int addrkind, int service) {
    strcpy(ipe->addr, ipaddr);
    ipe->addrkind = addrkind;
    ipe->service = service;
    ipe->whenfirst = ipe->whenlast = time(NULL);
    ipe->numhits = 1;
}

void purge_limbo_stale(void) {
    ipentry_t *tmpent;
    time_t now;
    unsigned int pos = 0;


    now = time(NULL);
    for (pos = 0; pos < list_size(&limbo); pos++) {
        tmpent = list_get_at(&limbo, pos);
        if (now - tmpent->whenfirst > opts.stale_threshold)
            list_delete_at(&limbo, pos);
    }
}

void *pardonBlocked(void *par) {
    time_t now;
    ipentry_t *tmpel;
    int ret;


    srandom(time(NULL));

    while (1) {
        /* wait some time, at most opts.pardon_threshold/3 + 1 sec */
        sleep(1 + random() % (opts.pardon_threshold/2));
        now = time(NULL);
        tmpel = list_get_at(&hell, 0);
        pthread_testcancel();
        pthread_setcancelstate(PTHREAD_CANCEL_DISABLE, &ret);
        while (tmpel != NULL && now - tmpel->whenlast > opts.pardon_threshold) {
            sshguard_log(LOG_INFO, "Releasing %s after %u seconds.\n", tmpel->addr, now - tmpel->whenlast);
            /* fprintf(stderr, "Releasing %s after %u seconds.\n", tmpel->ip, now - tmpel->whenlast); */
            ret = fw_release(tmpel->addr, tmpel->addrkind, tmpel->service);
            if (ret != FWALL_OK) sshguard_log(LOG_ERR, "Release command failed. Exited: %d", ret);
            list_delete_at(&hell, 0);
            free(tmpel);
            tmpel = list_get_at(&hell, 0);
        }
        pthread_setcancelstate(PTHREAD_CANCEL_ENABLE, &ret);
        pthread_testcancel();
    }

    pthread_exit(NULL);
    return NULL;
}

/* finalization routine */
void finishup(void) {
    /* flush blocking rules */
    sshguard_log(LOG_INFO, "Got exit signal, flushing blocked addresses and exiting...");
    fw_flush();
    if (fw_fin() != FWALL_OK) sshguard_log(LOG_ERR, "Cound not finalize firewall.");
    if (whitelist_fin() != 0) sshguard_log(LOG_ERR, "Could not finalize the whitelisting system.");
    if (procauth_fin() != 0) sshguard_log(LOG_ERR, "Could not finalize the process authorization subsystem.");
    sshguard_log_fin();
}

void sigfin_handler(int signo) {
    /* let exit() call finishup() */
    exit(0);
}

void sigstpcont_handler(int signo) {
    /* update "suspended" status */
    switch (signo) {
        case SIGTSTP:
            suspended = 1;
            sshguard_log(LOG_NOTICE, "Got STOP signal, suspending activity.");
            break;
        case SIGCONT:
            suspended = 0;
            sshguard_log(LOG_NOTICE, "Got CONTINUE signal, resuming activity.");
            break;
    }
}

static void usage(void) {
    fprintf(stderr, "Usage:\nsshguard [-a num] [-p sec] [-w <whlst>]{0,n} [-s sec] [-l c] [-f srv:pidfile]{0,n}\n");
    fprintf(stderr, "\t-a\tNumber of hits after which blocking an address (%d)\n", DEFAULT_ABUSE_THRESHOLD);
    fprintf(stderr, "\t-p\tSeconds after which unblocking a blocked address (%d)\n", DEFAULT_PARDON_THRESHOLD);
    fprintf(stderr, "\t-w\tWhitelisting of addr/host/block, or take from file if starts with \"/\" or \".\" (repeatable)\n");
    fprintf(stderr, "\t-s\tSeconds after which forgetting about a cracker candidate (%d)\n", DEFAULT_STALE_THRESHOLD);
    fprintf(stderr, "\t-f\t\"authenticate\" service's logs through its process pid, as in pidfile\n");
    fprintf(stderr, "\t-d\tRun in debug mode, log everything to standard error (not syslog)\n");
}

