-- This file is part of SmartEiffel The GNU Eiffel Compiler Tools and Libraries.
-- See the Copyright notice at the end of this file.
--
deferred class ASSERTION_LIST
	--
	-- To store a list of assertions (see ASSERTION).
	--
	-- Root class of CLASS_INVARIANT, E_ENSURE, LOOP_INVARIANT, CHECK_INVARIANT and REQUIRE_ITEM.
	--

inherit
	VISITABLE
insert
	GLOBALS

feature {ANY}
	start_position: POSITION
			-- The position of the first letter of `name'.

	header_comment: COMMENT

	frozen specialize_in (type: TYPE): like Current is
		require
			type /= Void
		local
			i: INTEGER; a1, a2: ASSERTION; l: like list
		do
			if list = Void then
				Result := Current
			else
				from
					i := list.upper
				until
					i < list.lower or else a1 /= a2
				loop
					a1 := list.item(i)
					a2 := a1.specialize_in(type)
					i := i - 1
				end
				if a1 = a2 then
					Result := Current
				else
					from
						l := list.twin
						Result := twin
						Result.set_list(l)
						l.put(a2, i + 1)
					until
						i < list.lower
					loop
						l.put(list.item(i).specialize_in(type), i)
						i := i - 1
					end
				end
			end
		ensure
			Result.has_been_specialized
		end

	frozen specialize_thru (parent_type: TYPE; parent_edge: PARENT_EDGE; new_type: TYPE): like Current is
		require
			parent_type /= Void
			parent_edge /= Void
			new_type /= Void
			new_type /= parent_type
			has_been_specialized
		local
			i: INTEGER; a1, a2: ASSERTION; l: like list
		do
			if list = Void then
				Result := Current
			else
				from
					i := list.upper
				until
					i < list.lower or else a1 /= a2
				loop
					a1 := list.item(i)
					a2 := a1.specialize_thru(parent_type, parent_edge, new_type)
					i := i - 1
				end
				if a1 = a2 then
					Result := Current
				else
					from
						l := list.twin
						Result := twin
						Result.set_list(l)
						l.put(a2, i + 1)
					until
						i < list.lower
					loop
						l.put(list.item(i).specialize_thru(parent_type, parent_edge, new_type), i)
						i := i - 1
					end
				end
			end
		ensure
			has_been_specialized
			Result.has_been_specialized
		end

	frozen specialize_2 (type: TYPE): like Current is
		require
			has_been_specialized
			not smart_eiffel.status.is_specializing
		local
			i: INTEGER; a1, a2: ASSERTION; l: like list
		do
			if list = Void then
				Result := Current
			else
				from
					i := list.upper
				until
					i < list.lower or else a1 /= a2
				loop
					a1 := list.item(i)
					a2 := a1.specialize_2(type)
					i := i - 1
				end
				if a1 = a2 then
					Result := Current
				else
					from
						l := list.twin
						Result := twin
						Result.set_list(l)
						l.put(a2, i + 1)
					until
						i < list.lower
					loop
						l.put(list.item(i).specialize_2(type), i)
						i := i - 1
					end
				end
			end
		ensure
			has_been_specialized
			Result.has_been_specialized
		end

	frozen has_been_specialized: BOOLEAN is
		local
			i: INTEGER
		do
			Result := True
			if list /= Void then
				from
					i := list.upper
				until
					i < list.lower or else not Result
				loop
					Result := list.item(i).has_been_specialized
					i := i - 1
				end
			end
		ensure
			assertion_check_only: Result
		end

	frozen side_effect_free (type: TYPE): BOOLEAN is
		local
			i: INTEGER
		do
			Result := True
			if list /= Void then
				from
					i := list.upper
				until
					not Result or else i < list.lower
				loop
					Result := list.item(i).side_effect_free(type)
					i := i - 1
				end
			end
		end

	frozen is_always_true: BOOLEAN is
		local
			i: INTEGER; assertion: ASSERTION
		do
			Result := True
			if list /= Void then
				from
					i := list.upper
				until
					not Result or else i < list.lower
				loop
					assertion := list.item(i)
					Result := assertion.is_always_true
					i := i - 1
				end
			end
		end

	frozen collect (type: TYPE) is
		require
			has_been_specialized
			type.feature_collection_done
			smart_eiffel.status.is_collecting
			not smart_eiffel.status.is_specializing
		local
			i: INTEGER
		do
			if list /= Void then
				from
					i := list.upper
				until
					i < list.lower
				loop
					list.item(i).collect(type)
					i := i - 1
				end
			end
		end

	frozen safety_check (type: TYPE) is
		local
			i: INTEGER
		do
			if list /= Void then
				from
					i := list.upper
				until
					i < list.lower
				loop
					list.item(i).safety_check(type)
					i := i - 1
				end
			end
		end

	frozen simplify (type: TYPE): like Current is
			-- This function should never be called (compile_to_* never gets called for assertions that are not
			-- used in the current check mode, and assertions that are usedothers should not get simplified).
		require
			type /= Void
			not smart_eiffel.pretty_flag
			not smart_eiffel.short_or_class_check_flag
			smart_eiffel.status.collecting_done
		do
			check
				simplify_is_only_made_for_boost: False
			end
		ensure
			Result /= Current implies smart_eiffel.magic_count > old smart_eiffel.magic_count
		end

	frozen adapt_for (type: TYPE): like Current is
		require
			type /= Void
		local
			i: INTEGER; a1, a2: ASSERTION; l: like list
		do
			if list = Void then
				Result := Current
			else
				from
					i := list.upper
				until
					i < list.lower or else a1 /= a2
				loop
					a1 := list.item(i)
					a2 := a1.adapt_for(type)
					i := i - 1
				end
				if a1 = a2 then
					Result := Current
				else
					from
						l := list.twin
						Result := twin
						Result.set_list(l)
						l.put(a2, i + 1)
					until
						i < list.lower
					loop
						l.put(list.item(i).adapt_for(type), i)
						i := i - 1
					end
				end
			end
		end

	frozen compile_to_jvm (type: TYPE; last_chance: BOOLEAN) is
			-- If `last_chance' is True, an error message is printed at
			-- run-time.
			-- The final result is always left a top of stack.
		local
			point_true, i: INTEGER; ca: like code_attribute
		do
			ca := code_attribute
			ca.check_opening
			from
				failure.clear_count
				i := list.lower
			until
				i > list.upper
			loop
				list.item(i).compile_to_jvm(type, last_chance)
				failure.add_last(ca.opcode_ifeq)
				i := i + 1
			end
			ca.opcode_iconst_1
			point_true := ca.opcode_goto
			ca.resolve_with(failure)
			ca.opcode_iconst_0
			ca.resolve_u2_branch(point_true)
			ca.check_closing
		end

	frozen use_current (type: TYPE): BOOLEAN is
		local
			i: INTEGER
		do
			if list /= Void then
				from
					i := list.upper
				until
					Result or else i < list.lower
				loop
					Result := list.item(i).use_current(type)
					i := i - 1
				end
			end
		end

	is_empty: BOOLEAN is
		do
			Result := list = Void
		end

	set_header_comment (hc: like header_comment) is
		do
			header_comment := hc
		end

feature {ASSERTION_LIST, E_REQUIRE, E_CHECK, ANONYMOUS_FEATURE, ASSERTION_LIST_VISITOR}
	list: FAST_ARRAY[ASSERTION]

feature {ASSERTION_LIST}
	has_assertion_located_at (sp: POSITION): BOOLEAN is
			-- Is there already the source code of the assertion located at `sp' ?
		require
			not sp.is_unknown
		local
			i: INTEGER
		do
			if list /= Void then
				from
					i := list.upper
				until
					Result or else i < list.lower
				loop
					if list.item(i).start_position = sp then
						Result := True
					end
					i := i - 1
				end
			end
		end

feature {}
	run_feature: RUN_FEATURE
			-- Corresponding one (if any) when runnable.
			-- Note: class invariant are not inside a RUN_FEATURE.
			--|*** TO BE REMOVED ASAP.
			-- No more context dependant information.
			--|*** Dom. jan 25th 2004

	compile_to_c_ (type: TYPE; protect_against_infinite_recursion_flag, inside_feature_flag: BOOLEAN) is
			-- Note: `inside_feature_flag' is meaningful only when `protect_against_infinite_recursion_flag' is
			-- set.
		local
			i: INTEGER; assertion: ASSERTION
		do
			if not is_always_true then
				if protect_against_infinite_recursion_flag then
					cpp.stop_recursive_assertion_opening(inside_feature_flag)
				end
				from
					i := list.lower
				until
					i > list.upper
				loop
					assertion := list.item(i)
					c_compile_assertion(assertion, type)
					i := i + 1
				end
				if protect_against_infinite_recursion_flag then
					cpp.stop_recursive_assertion_closing(inside_feature_flag)
				end
			end
		end

	make (sp: like start_position; hc: like header_comment; l: like list) is
		require
			l /= Void implies not l.is_empty
			hc /= Void or else l /= Void
		do
			start_position := sp
			header_comment := hc
			list := l
		ensure
			start_position = sp
			header_comment = hc
			list = l
		end

feature {E_REQUIRE}
	pretty_print_ (indent_level: INTEGER; name: STRING) is
			-- Where `name' can be "require", "require else", "ensure", "ensure then" or "check" as well.
		local
			index_in_pretty_printer, i: INTEGER
		do
			pretty_printer.set_indent_level(indent_level)
			pretty_printer.put_string(name)
			if header_comment /= Void then
				pretty_printer.set_indent_level(indent_level + 1)
				header_comment.pretty(indent_level)
			end
			if list /= Void then
				from
					i := list.lower
				until
					i > list.upper
				loop
					pretty_printer.set_semi_colon_flag(False)
					index_in_pretty_printer := pretty_printer.index_of_last_character
					list.item(i).pretty(indent_level + 1)
					if i > list.lower and then pretty_printer.is_opening_bracket_after(index_in_pretty_printer) then
						pretty_printer.erase_everything_after(index_in_pretty_printer)
						pretty_printer.set_indent_level(indent_level + 1)
						pretty_printer.put_character(';')
						index_in_pretty_printer := pretty_printer.index_of_last_character					
						pretty_printer.set_semi_colon_flag(False)
						list.item(i).pretty(indent_level + 1)
						pretty_printer.erase_separators_after(index_in_pretty_printer)
					end
					i := i + 1
				end
			end
		end

	c_compile_assertion (assertion: ASSERTION; type: TYPE) is
		do
			if is_always_true then
				-- No code generated.
			else
				cpp.set_check_assertion_mode(check_assertion_mode)
				assertion.compile_to_c(type)
			end
		end

feature {}
	failure: FAST_ARRAY[INTEGER] is
		once
			create Result.with_capacity(12)
		end

	check_assertion_mode: STRING is
		deferred
		ensure
			Result /= Void
		end

feature {ONCE_ROUTINE_POOL}
	clear_run_feature is
		do
			run_feature := Void
		end

feature {ASSERTION_LIST}
	set_list (l: like list) is
		require
			l /= Void
		do
			list := l
		ensure
			list = l
		end

feature {ASSERTION_LIST, ANONYMOUS_FEATURE}
	append (other: ASSERTION_LIST) is
		require
			other /= Void
			other /= Current implies other.list /= list
		do
			if other /= Current then
				list.append_collection(other.list)
			end
		end

invariant
	list /= Void implies not list.is_empty

end -- class ASSERTION_LIST
--
-- ------------------------------------------------------------------------------------------------------------------------------
-- Copyright notice below. Please read.
--
-- SmartEiffel is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License,
-- as published by the Free Software Foundation; either version 2, or (at your option) any later version.
-- SmartEiffel is distributed in the hope that it will be useful but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details. You should have
-- received a copy of the GNU General Public License along with SmartEiffel; see the file COPYING. If not, write to the Free
-- Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
--
-- Copyright(C) 1994-2002: INRIA - LORIA (INRIA Lorraine) - ESIAL U.H.P.       - University of Nancy 1 - FRANCE
-- Copyright(C) 2003-2004: INRIA - LORIA (INRIA Lorraine) - I.U.T. Charlemagne - University of Nancy 2 - FRANCE
--
-- Authors: Dominique COLNET, Philippe RIBET, Cyril ADRIAN, Vincent CROIZIER, Frederic MERIZEN
--
-- http://SmartEiffel.loria.fr - SmartEiffel@loria.fr
-- ------------------------------------------------------------------------------------------------------------------------------
