-- This file is part of SmartEiffel The GNU Eiffel Compiler Tools and Libraries.
-- See the Copyright notice at the end of this file.
--
class INTERNAL_C_LOCAL_LIST
	--
	-- To handle the list of INTERNAL_C_LOCAL actually used by the current pending C function.
	--

insert
	GLOBALS
	
feature {}
	internal_c_local_pool: FAST_ARRAY[INTERNAL_C_LOCAL] is
			-- All allocated INTERNAL_C_LOCAL objects since the beginning. This array is only growing 
			-- and its purpose is to avoid creating many new INTERNAL_C_LOCAL objects for each 
			-- new `pending_c_function' run.
		once
			create Result.with_capacity(32)
		end

	pending_c_function_locals: FAST_ARRAY[INTERNAL_C_LOCAL] is
			-- Local temporary variables of the `pending_c_function' beeing prepared.
			-- The list to be printed at the beginning of the `pending_c_function'.
		once
			create Result.with_capacity(32)
		end

	pending_locked_stack: FAST_ARRAY[INTERNAL_C_LOCAL] is
			-- In order to follow allocation of variable at block/instruction level.
		once
			create Result.with_capacity(32)
		end

	information_declared_item_counter: INTEGER
	
feature {C_PRETTY_PRINTER}
	lock_local (type: TYPE): INTERNAL_C_LOCAL is
			-- Reuse some un-locked variable or add/create a new one.
		require
			type /= Void
		local
			i: INTEGER
		do
			-- First, looking for some un-locked item from the `pending_c_function_locals'. Note that we search
			-- from `lower' to `upper' in order to make the generated C code more stable:
			from
				i := pending_c_function_locals.lower
			until
				(i > pending_c_function_locals.upper) or else (Result /= Void)
			loop
				Result := pending_c_function_locals.item(i)
				if Result.is_locked then
					Result := Void
				elseif Result.has_the_same_c_type_than(type) then
					-- Good luck, we ca reuse this one!
				else
					Result := Void
				end
				i := i + 1
			end
			if Result /= Void then
				Result.lock
			else
				-- Now searching in the `internal_c_local_pool' (again from `lower' to `upper' for C code
				-- stability):
				from
					i := internal_c_local_pool.lower
				until
					(i > internal_c_local_pool.upper) or else (Result /= Void)
				loop
					Result := internal_c_local_pool.item(i)
					if Result.is_member_of_pending_c_function_locals then
						Result := Void
					else
						Result.recycle(type)
						pending_c_function_locals.add_last(Result)
					end
					i := i + 1
				end
				if Result = Void then
					-- We must create a new object:
					create Result.make(type)
					internal_c_local_pool.add_last(Result)
					pending_c_function_locals.add_last(Result)
				end
			end
			pending_locked_stack.add_last(Result)
		ensure
			Result.is_locked
		end

	stack_level: INTEGER is
			-- To be used with `unlock'.
		do
			Result := pending_locked_stack.upper
		end

	unlock (old_stack_level: INTEGER) is
			-- Unlock now unused INTERNAL_C_LOCAL since last `stack_level' call.
		require
			old_stack_level >= stack_level
		do
			from
			until
				stack_level <= old_stack_level
			loop
				pending_locked_stack.last.un_lock				
				pending_locked_stack.remove_last
			end
		ensure
			stack_level = old_stack_level
		end
	
	declaration_dump (out_c: TEXT_FILE_WRITE) is
		local
			i: INTEGER; internal_c_local: INTERNAL_C_LOCAL; buffer: STRING
		do
			if not pending_c_function_locals.is_empty then
				buffer := once "........... unique buffer ..............."
				out_c.put_string(once "/*[INTERNAL_C_LOCAL list*/%N")
				from
					i := pending_c_function_locals.lower
				until
					i > pending_c_function_locals.upper
				loop
					internal_c_local := pending_c_function_locals.item(i)
					buffer.clear_count
					internal_c_local.c_type_in(buffer)
					buffer.extend(' ')
					internal_c_local.append_in(buffer)
					buffer.append(once ";%N")
					out_c.put_string(buffer)
					information_declared_item_counter := information_declared_item_counter + 1
					i := i + 1
				end
				out_c.put_string(once "/*INTERNAL_C_LOCAL list]*/%N")
				-- Now clearing `local_pending_c_function_locals':
				pending_c_function_locals.clear_count
			end
			-- Preparing the next cycle:	
			from
				i := internal_c_local_pool.upper
			until
				i < internal_c_local_pool.lower
			loop
				internal_c_local_pool.item(i).un_lock_after_dump
				i := i - 1
			end
			pending_locked_stack.clear_count
		end
	
	echo_information is
		do
			echo.put_string("Declared INTERNAL_C_LOCAL: ")
			echo.put_integer(information_declared_item_counter)
			echo.put_string(" using range tmp0 .. tmp")
			echo.put_integer(internal_c_local_pool.upper)
			echo.put_string(".%N")
		end
	
end -- class INTERNAL_C_LOCAL_LIST
--
-- ------------------------------------------------------------------------------------------------------------------------------
-- Copyright notice below. Please read.
--
-- SmartEiffel is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License,
-- as published by the Free Software Foundation; either version 2, or (at your option) any later version.
-- SmartEiffel is distributed in the hope that it will be useful but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details. You should have
-- received a copy of the GNU General Public License along with SmartEiffel; see the file COPYING. If not, write to the Free
-- Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
--
-- Copyright(C) 1994-2002: INRIA - LORIA (INRIA Lorraine) - ESIAL U.H.P.       - University of Nancy 1 - FRANCE
-- Copyright(C) 2003-2004: INRIA - LORIA (INRIA Lorraine) - I.U.T. Charlemagne - University of Nancy 2 - FRANCE
--
-- Authors: Dominique COLNET, Philippe RIBET, Cyril ADRIAN, Vincent CROIZIER, Frederic MERIZEN
--
-- http://SmartEiffel.loria.fr - SmartEiffel@loria.fr
-- ------------------------------------------------------------------------------------------------------------------------------

