-- See the Copyright notice at the end of this file.
--
class XML_WORLD_LOADER[E_ -> LOADABLE_GL_OBJECT]

inherit
	WORLD_LOADER[E_]

creation {ANY}
	make

feature {ANY}
	load_from_file (world: WORLD; complete_file_name: STRING) is
		local
			tfr: TEXT_FILE_READ
			tree: XML_TREE
		do
			loaded_file := complete_file_name
			loaded_world := world
			current_light_id := gl.const_light0
			create tfr.connect_to(loaded_file)
			create tree.make(tfr)
			parse(tree.root)
		end

	parse_error is
		do
			std_error.put_string("Error while loading file `")
			std_error.put_string(loaded_file)
			std_error.put_string("'%N")
		end

	find_child (node: XML_NODE; name: STRING): XML_NODE is
		require
			node /= Void
			name /= Void
		local
			i: INTEGER
		do
			from
				i := 1
			until
				Result /= Void or else i > node.children_count
			loop
				if node.child(i).name.is_equal(name) then
					Result := node.child(i)
				end
				i := i + 1
			end
		end

	parse (node: XML_NODE) is
		require
			node /= Void and then node.name.is_equal("world")
		local
			i: INTEGER
		do
			from
				i := 1
			until
				i > node.children_count
			loop
				if node.child(i).name.is_equal("camera") then
					parse_camera(node.child(i))
				elseif node.child(i).name.is_equal("light") then
					parse_light(node.child(i))
				elseif node.child(i).name.is_equal("object") then
					parse_object(node.child(i))
				elseif node.child(i).name.is_equal("fog") then
					parse_fog(node.child(i))
				else
					parse_error
					i := node.children_count
				end
				i := i + 1
			end
		end

	parse_camera (node: XML_NODE) is
		require
			node /= Void and then node.name.is_equal("camera")
		local
			camera: CAMERA
			position, orientation: XML_NODE
			x, y, z: STRING
		do
			position := find_child(node, "position")
			if position = Void then
				parse_error
			else
				x := position.attribute_at("x")
				y := position.attribute_at("y")
				z := position.attribute_at("z")
				if x = Void or else y = Void or else z = Void then
					parse_error
				else
					create camera.make_with_position(x.to_real, y.to_real, z.to_real)
					loaded_world.set_camera(camera)
					orientation := find_child(node, "orientation")
					if orientation  /= Void then
						x := orientation.attribute_at("h")
						y := orientation.attribute_at("p")
						z := orientation.attribute_at("r")
						if x /= Void and then y /= Void and then z /= Void then
							camera.set_orientation(x.to_real, y.to_real, z.to_real)
						end
					end
				end
			end
		end

	parse_light (node: XML_NODE) is
		require
			node /= Void and then node.name.is_equal("light")
		local
			light: LIGHT
			type: STRING
			position, direction, color: XML_NODE
			x, y, z, a: STRING
		do
			create light.make(current_light_id)
			if current_light_id = gl.const_light0 then
				current_light_id := gl.const_light1
			elseif current_light_id = gl.const_light1 then
				current_light_id := gl.const_light2
			elseif current_light_id = gl.const_light2 then
				current_light_id := gl.const_light3
			elseif current_light_id = gl.const_light3 then
				current_light_id := gl.const_light4
			elseif current_light_id = gl.const_light4 then
				current_light_id := gl.const_light5
			elseif current_light_id = gl.const_light5 then
				current_light_id := gl.const_light6
			elseif current_light_id = gl.const_light6 then
				current_light_id := gl.const_light7
			else
				std_error.put_string("Warning: check number of lights in ")
				std_error.put_line(loaded_file)
				current_light_id := current_light_id + 1
			end
			type := node.attribute_at("type")
			if type = Void or else type.is_equal("spot") then
				light.set_spot_position(0, 0, 0)
				position := find_child(node, "position")
				if position /= Void then
					x := position.attribute_at("x")
					y := position.attribute_at("y")
					z := position.attribute_at("z")
					if x /= Void and then y /= Void and then z /= Void then
						light.set_spot_position(x.to_real, y.to_real, z.to_real)
					end
				end
				direction := find_child(node, "direction")
				if direction = Void then
					parse_error
				else
					x := direction.attribute_at("x")
					y := direction.attribute_at("y")
					z := direction.attribute_at("z")
					if x /= Void and then y /= Void and then z /= Void then
						light.set_spot_direction(x.to_real, y.to_real, z.to_real)
					end
				end
			elseif type /= Void and then type.is_equal("directional") then
				direction := find_child(node, "direction")
				if direction = Void then
					parse_error
				else
					x := direction.attribute_at("x")
					y := direction.attribute_at("y")
					z := direction.attribute_at("z")
					if x /= Void and then y /= Void and then z /= Void then
						light.set_directional(x.to_real, y.to_real, z.to_real)
					end
				end
			else
				std_error.put_string("Warning: no light type in ")
				std_error.put_line(loaded_file)
			end
			color := find_child(node, "ambient")
			if color /= Void then
				x := color.attribute_at("r")
				y := color.attribute_at("g")
				z := color.attribute_at("b")
				a := color.attribute_at("a")
				if x /= Void and then y /= Void and then z /= Void and then a /= Void then
					light.set_ambient(create {GL_COLOR}.set_color(x.to_real, y.to_real, z.to_real, a.to_real))
				end
			end
			color := find_child(node, "diffuse")
			if color /= Void then
				x := color.attribute_at("r")
				y := color.attribute_at("g")
				z := color.attribute_at("b")
				a := color.attribute_at("a")
				if x /= Void and then y /= Void and then z /= Void and then a /= Void then
					light.set_diffuse(create {GL_COLOR}.set_color(x.to_real, y.to_real, z.to_real, a.to_real))
				end
			end
			color := find_child(node, "specular")
			if color /= Void then
				x := color.attribute_at("r")
				y := color.attribute_at("g")
				z := color.attribute_at("b")
				a := color.attribute_at("a")
				if x /= Void and then y /= Void and then z /= Void and then a /= Void then
					light.set_specular(create {GL_COLOR}.set_color(x.to_real, y.to_real, z.to_real, a.to_real))
				end
			end
			loaded_world.add_light(light)
		end

	parse_object (node: XML_NODE) is
		require
			node /= Void and then node.name.is_equal("object")
		local
			file: STRING
			object: E_
			position, orientation: XML_NODE
			x, y, z: STRING
		do
			file := node.attribute_at("file")
			if file = Void then
				parse_error
			else
				object := universal_object_loader.load_object(file, universal_texture_loader)
				if object.is_translucent then
					loaded_world.add_translucent_object(object)
				else
					loaded_world.add_opaque_object(object)
				end
				position := find_child(node, "position")
				if position /= Void then
					x := position.attribute_at("x")
					y := position.attribute_at("y")
					z := position.attribute_at("z")
					if x /= Void and then y /= Void and then z /= Void then
						object.set_position(x.to_real, y.to_real, z.to_real)
					end
				end
				orientation := find_child(node, "orientation")
				if orientation /= Void then
					x := position.attribute_at("h")
					y := position.attribute_at("p")
					z := position.attribute_at("r")
					if x /= Void and then y /= Void and then z /= Void then
						object.set_orientation(x.to_real, y.to_real, z.to_real)
					end
				end
			end
		end

	parse_fog (node: XML_NODE) is
		require
			node /= Void and then node.name.is_equal("fog")
		local
			fog: FOG
			type: STRING
			f_start, f_end, f_density: STRING
			color: XML_NODE
			r, g, b, a: STRING
			atmospheric: STRING
		do
			type := node.attribute_at("type")
			inspect
				type
			when "linear" then
				f_start := node.attribute_at("start")
				f_end := node.attribute_at("end")
				if f_start /= Void and then f_end /= Void then
					create fog.make_linear(f_start.to_real, f_end.to_real)
				end
			when "exp" then
				f_start := node.attribute_at("start")
				f_end := node.attribute_at("end")
				f_density := node.attribute_at("density")
				if f_start /= Void and then f_end /= Void and then f_density /= Void then
					create fog.make_exp(f_start.to_real, f_end.to_real, f_density.to_real)
				end
			when "exp2" then
				f_start := node.attribute_at("start")
				f_end := node.attribute_at("end")
				f_density := node.attribute_at("density")
				if f_start /= Void and then f_end /= Void and then f_density /= Void then
					create fog.make_exp2(f_start.to_real, f_end.to_real, f_density.to_real)
				end
			else
				std_error.put_string("Warning: no fog type in ")
				std_error.put_line(loaded_file)
			end
			if fog /= Void then
				color := find_child(node, "color")
				if color /= Void then
					r := color.attribute_at("r")
					g := color.attribute_at("g")
					b := color.attribute_at("b")
					a := color.attribute_at("a")
					if r /= Void and then g /= Void and then b /= Void and then a /= Void then
						fog.set_color(create {GL_COLOR}.set_color(r.to_real, g.to_real, b.to_real, a.to_real))
					end
				end
				atmospheric := node.attribute_at("atmospheric")
				if atmospheric /= Void then
					loaded_world.set_fog(fog, atmospheric.to_boolean)
				else
					loaded_world.set_fog(fog, False)
				end
			end
		end

feature {}
	loaded_file: STRING

	loaded_world: WORLD

	current_light_id: INTEGER

end -- class XML_WORLD_LOADER
--
-- ------------------------------------------------------------------------------------------------------------
-- Copyright notice below. Please read.
--
-- This file is part of the SmartEiffel standard library.
-- Copyright(C) 1994-2002: INRIA - LORIA (INRIA Lorraine) - ESIAL U.H.P.       - University of Nancy 1 - FRANCE
-- Copyright(C) 2003-2006: INRIA - LORIA (INRIA Lorraine) - I.U.T. Charlemagne - University of Nancy 2 - FRANCE
--
-- Authors: Dominique COLNET, Philippe RIBET, Cyril ADRIAN, Vincent CROIZIER, Frederic MERIZEN
--
-- Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated
-- documentation files (the "Software"), to deal in the Software without restriction, including without
-- limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
-- the Software, and to permit persons to whom the Software is furnished to do so, subject to the following
-- conditions:
--
-- The above copyright notice and this permission notice shall be included in all copies or substantial
-- portions of the Software.
--
-- THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT
-- LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO
-- EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
-- AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE
-- OR OTHER DEALINGS IN THE SOFTWARE.
--
-- http://SmartEiffel.loria.fr - SmartEiffel@loria.fr
-- ------------------------------------------------------------------------------------------------------------
