-- See the Copyright notice at the end of this file.
--
expanded class PROTOCOLS

feature {ANY}
	register_protocol_factory (a_protocol_name: STRING; a_factory: FUNCTION[TUPLE, PROTOCOL]) is
		require
			a_protocol_name.count > 0
			not known_protocol(a_protocol_name)
		do
			factories.add(a_factory, a_protocol_name.twin)
		ensure
			known_protocol(a_protocol_name)
		end

	known_protocol (a_protocol_name: STRING): BOOLEAN is
		require
			a_protocol_name.count > 0
		do
			init
			Result := factories.has(a_protocol_name)
		end

	protocol (a_protocol_name: STRING): PROTOCOL is
		require
			a_protocol_name.count > 0
			known_protocol(a_protocol_name)
		do
			init
			Result := unused_protocol(a_protocol_name)
			if Result = Void then
				Result := factories.at(a_protocol_name).item([])
			end
			register(Result)
		end

feature {}
	factories: DICTIONARY[FUNCTION[TUPLE, PROTOCOL], STRING] is
		once
			create {HASHED_DICTIONARY[FUNCTION[TUPLE, PROTOCOL], STRING]} Result.make
		end

	init is
		once
			register_protocol_factory(once "http", agent new_http(once "http", 80))
			register_protocol_factory(once "https", agent new_http(once "https", 443))
			register_protocol_factory(once "file", agent new_file)
		end

	new_http (name: STRING; port: INTEGER): HTTP_PROTOCOL is
		do
			create Result.make(name, port)
		end

	new_file: FILE_PROTOCOL is
		do
			create Result.make
		end

feature {PROTOCOL} -- Protocol registry
	unregister (p: PROTOCOL) is
		require
			not p.is_connected
		local
			i: INTEGER; wr: WEAK_REFERENCE[PROTOCOL]
		do
			from
				i := registered.lower
			until
				i > registered.upper
			loop
				wr := registered.item(i)
				if wr.item = p then
					wr.set_item(Void)
				end
				i := i + 1
			end
			set_unused(p)
		ensure
			not is_registered(p)
		end

	is_registered (p: PROTOCOL): BOOLEAN is
		local
			i: INTEGER
		do
			from
				i := registered.lower
			until
				Result or else i > registered.upper
			loop
				Result := registered.item(i).item = p
				i := i + 1
			end
		end

feature {} -- Protocol registry
	register (p: PROTOCOL) is
		require
			not is_registered(p)
		local
			i: INTEGER; done: BOOLEAN; wr: WEAK_REFERENCE[PROTOCOL]
		do
			from
				i := registered.lower
			until
				done or else i > registered.upper
			loop
				wr := registered.item(i)
				if wr.item = Void then
					wr.set_item(p)
					done := True
				end
				i := i + 1
			end
			if not done then
				create wr.set_item(p)
				registered.add_last(wr)
			end
		ensure
			is_registered(p)
		end

	registered: FAST_ARRAY[WEAK_REFERENCE[PROTOCOL]] is
		once
			create Result.make(0)
		end

feature {} -- Protocol reuse
	unused: HASHED_DICTIONARY[WEAK_ARRAY[PROTOCOL], STRING] is
		once
			create Result.make
		end

	unused_protocols (a_protocol_name: STRING): WEAK_ARRAY[PROTOCOL] is
		do
			Result := unused.reference_at(a_protocol_name)
			if Result = Void then
				create Result.make
				unused.add(Result, a_protocol_name.twin)
			end
		ensure
			Result /= Void
		end

	unused_protocol (a_protocol_name: STRING): PROTOCOL is
		local
			u: WEAK_ARRAY[PROTOCOL]
		do
			u := unused_protocols(a_protocol_name)
			Result := u.item
		ensure
			Result /= Void implies not Result.is_connected
		end

	set_unused (a_protocol: PROTOCOL) is
		require
			not a_protocol.is_connected
		do
			unused_protocols(a_protocol.name).recycle(a_protocol)
		end

end -- class PROTOCOLS
--
-- ------------------------------------------------------------------------------------------------------------
-- Copyright notice below. Please read.
--
-- This file is part of the SmartEiffel standard library.
-- Copyright(C) 1994-2002: INRIA - LORIA (INRIA Lorraine) - ESIAL U.H.P.       - University of Nancy 1 - FRANCE
-- Copyright(C) 2003-2006: INRIA - LORIA (INRIA Lorraine) - I.U.T. Charlemagne - University of Nancy 2 - FRANCE
--
-- Authors: Dominique COLNET, Philippe RIBET, Cyril ADRIAN, Vincent CROIZIER, Frederic MERIZEN
--
-- Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated
-- documentation files (the "Software"), to deal in the Software without restriction, including without
-- limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
-- the Software, and to permit persons to whom the Software is furnished to do so, subject to the following
-- conditions:
--
-- The above copyright notice and this permission notice shall be included in all copies or substantial
-- portions of the Software.
--
-- THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT
-- LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO
-- EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
-- AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE
-- OR OTHER DEALINGS IN THE SOFTWARE.
--
-- http://SmartEiffel.loria.fr - SmartEiffel@loria.fr
-- ------------------------------------------------------------------------------------------------------------
