-- See the Copyright notice at the end of this file.
--
deferred class PATH_NAME
	-- Operating system path name

inherit
	ANY

feature -- Creation/ Initialization

	make_empty is
			-- Make a 'null' path
		deferred
		ensure
			is_empty
		end

	make_root is
			-- Path to root directory (in current drive)
		deferred
		ensure
			is_absolute
		end

	make_current is
			-- Path to current directory (relative). See also `to_absolute'
			-- if you need the absolute current working directory
		deferred
		ensure
			not is_absolute
		end

	make_from_string (s: STRING) is
		deferred
		ensure
			to_string.is_equal (s)
		end

feature -- Constants

	extension_separator: CHARACTER is 
			-- Character used to separate filenames from extensions
		deferred
		end

	directory_separator: CHARACTER is 
			-- Character used to separate directories
			-- This character is forbidden in filenames
		deferred
		end

feature -- Access

	to_string: STRING is
			-- String representation
		deferred
		ensure
			Result /= Void
		end

	drive_specification: STRING is
			-- Drive specified by the current path, Void if none
		deferred
		ensure
			Result /= Void implies to_string.has_prefix (Result)
		end

	count: INTEGER is
			-- Number of elements in_path
		deferred
		ensure
			Result >= 0
		end

	is_empty: BOOLEAN is
			-- Path is null. Note that you can have a null absolute path
			-- (i.e., root) or a null relative path (current directory)
		do
			Result := count = 0
		ensure
			Result = (count = 0)
		end

	last: STRING is
			-- Last component (also known as "basename")
		require
			not is_empty
		deferred
		ensure
			Result /= Void
			not Result.has (directory_separator)
		end

	extension: STRING is
			-- Path extension (may be empty)
		deferred
		ensure
			is_suffix: to_string.has_suffix (Result)
			is_extension: not Result.is_empty implies Result.first = extension_separator
			is_minimal: Result.occurrences (extension_separator) <= 1
			not Result.has (directory_separator)
			is_empty implies Result.is_empty
		end

	is_absolute: BOOLEAN is
			-- absolute path?
		deferred
		end

	as_absolute: like Current is
			-- Equivalent absolute path
		do
			Result := twin
			Result.to_absolute
		ensure
			Result.is_absolute
			Result.is_normalized
		end

	is_normalized: BOOLEAN is
			-- Has no redundant separators, or redundant up-references
		deferred
		end

	is_file: BOOLEAN is
			-- Path points to an existing regular file?
		do
			Result := (create {FILE_TOOLS}).is_file (to_string)
		end

	is_directory: BOOLEAN is
			-- Path points to an existing directory?
		do
			Result := (create {FILE_TOOLS}).is_directory (to_string)
		end

	infix "+" (other: like Current): like Current is
			-- Join with `other' using filesystem semantics
		require
			other /= Void
		do
			Result := twin
			Result.join (other)
		ensure
			-- Result is a fresh instance
		end

	infix "/" (elem: STRING): like Current is
			-- Path with `elem' inside current
		require
			elem /= Void
			not elem.has (directory_separator)
		do
			Result := twin
			Result.add_last (elem)
		ensure
			-- Result is a fresh instance
			Result.is_empty = (is_empty and elem.is_empty)
			Result.last.is_equal (elem)
		end

feature -- Operations

	to_absolute is
			-- Transform into equivalent absolute path
		deferred
		ensure
			is_absolute
			is_normalized
		end

	normalize_case is
			-- Transform into normalized case version (equivalent), with
			-- standard path separators
		deferred
		end

	normalize is
			-- Normalize removing double separators, and up-references
		deferred
		ensure
			is_normalized
			old is_normalized implies (to_string.is_equal (old to_string.twin))
		end

	remove_last is
			-- Remove last component of path (keep the "dirname")
		require
			not is_empty
		deferred
		ensure
			-- (old twin).is_equal (Current / (old last))
			-- assertion above commented because of SE bug
			count = old count - 1 
		end

	add_last (elem: STRING) is
		require
			elem /= Void
			not elem.has (directory_separator)
		deferred
		ensure
			last.is_equal (elem)
			not_reduced: old count <= count
			may_grow_one: count <= old count + 1 
		end

	join (other: like Current) is
			-- Join with `other' using filesystem semantics
		require
			other /= Void
		deferred
		ensure
			-- definition: to_string.is_equal (old (Current+other).to_string)
			-- assertion above is commented out because of SE bug
		end

	expand_user is
			-- Replace an initial "~" or "~user" by user home directory
		deferred
		ensure
			not (old to_string.twin).has_prefix (once "~") implies to_string.is_equal (old to_string.twin)
		end

	expand_variables is
			-- Replace substrings of form $name or ${name} with environment
			-- variable values
		local
			p, len: INTEGER
			new, varname, subst: STRING
			sys: SYSTEM
		do
			new := once ""
			varname := once ""
			subst := once ""
			len := to_string.count
			from
				p := 1
				new.clear_count
				varname.clear_count
			until p > len loop
				if to_string.item (p) = '$' then
					varname.clear_count
					subst.clear_count
					subst.extend ('$')
					p := p + 1
					if p = len then
					elseif to_string.item (p) = '{' then -- ${var}
						subst.extend ('{')
						from 
							p := p + 1
						until p > len or else to_string.item (p) = '}' loop
							varname.extend (to_string.item (p))
							subst.extend (to_string.item (p))
							p := p + 1
						end
						if p <= len then
							p := p + 1 -- skip '}'
							subst.extend ('}')
						else
							varname.clear_count -- invalid variable
						end
					else -- $var
						from until
							p > len or else 
							not (to_string.item (p)).is_letter_or_digit
						loop
							varname.extend (to_string.item (p))
							subst.extend (to_string.item (p))
							p := p + 1
						end
					end
					if
						not varname.is_empty and then
						sys.get_environment_variable (varname) /= Void
					then
						subst.copy (sys.get_environment_variable (varname))
					end
					new.append (subst)
				else
					new.extend (to_string.item (p))
					p := p + 1
				end
			end
			to_string.copy (new)
		ensure
			not (old to_string.twin).has ('$') implies to_string.is_equal (old to_string.twin)
		end

	expand_shellouts is
			-- Replace substrings of form $(command) with execution of
			-- shell commands
		deferred
		ensure
			not to_string.has_substring ("$(") implies to_string.is_equal (old to_string.twin)
		end

end -- class PATH_NAME    
--
-- ------------------------------------------------------------------------------------------------------------
-- Copyright notice below. Please read.
--
-- This file is part of the SmartEiffel standard library.
-- Copyright(C) 1994-2002: INRIA - LORIA (INRIA Lorraine) - ESIAL U.H.P.       - University of Nancy 1 - FRANCE
-- Copyright(C) 2003-2006: INRIA - LORIA (INRIA Lorraine) - I.U.T. Charlemagne - University of Nancy 2 - FRANCE
--
-- Authors: Dominique COLNET, Philippe RIBET, Cyril ADRIAN, Vincent CROIZIER, Frederic MERIZEN
--
-- Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated
-- documentation files (the "Software"), to deal in the Software without restriction, including without
-- limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
-- the Software, and to permit persons to whom the Software is furnished to do so, subject to the following
-- conditions:
--
-- The above copyright notice and this permission notice shall be included in all copies or substantial
-- portions of the Software.
--
-- THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT
-- LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO
-- EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
-- AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE
-- OR OTHER DEALINGS IN THE SOFTWARE.
--
-- http://SmartEiffel.loria.fr - SmartEiffel@loria.fr
-- ------------------------------------------------------------------------------------------------------------
