/*
 * Brian Carrier [carrier@sleuthkit.org]
 * Copyright (c) 2003 Brian Carrier.  All rights reserved
 *
 * SUN - Sun VTOC code
 *
 * This file is part of mmtools
 *
 * mmtools is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * mmtools is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with mactime; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR ANY PARTICULAR PURPOSE.
 *
 * IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, LOSS OF USE, DATA, OR PROFITS OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "mm_tools.h"
#include "mymalloc.h"
#include "error.h"

#include "sun.h"


/*
 * Return a buffer with a description of the partition type
 */
static char *
sun_get_desc(u_int8_t fstype)
{
	char *str = mymalloc(64);
	switch (fstype) {

		case 0:
			strncpy(str, "Unassigned (0x00)", 64);
			break;
		case 1:
			strncpy(str, "boot (0x01)", 64);
			break;
		case 2:
			strncpy(str, "/ (0x02)", 64);
			break;
		case 3:
			strncpy(str, "swap (0x03)", 64);
			break;
		case 4:
			strncpy(str, "/usr/ (0x04)", 64);
			break;
		case 5:
			strncpy(str, "backup (0x05)", 64);
			break;
		case 6:
			strncpy(str, "stand (0x06)", 64);
			break;
		case 7:
			strncpy(str, "/var/ (0x07)", 64);
			break;
		case 8:
			strncpy(str, "/home/ (0x08)", 64);
			break;
		case 9:
			strncpy(str, "alt sector (0x09)", 64);
			break;
		case 10:
			strncpy(str, "cachefs (0x0A)", 64);
			break;
		default:	
			snprintf(str, 64, "Unknown Type (0x%.2x)", fstype);
			break;
	}

	return str;
}


/* 
 * Load an Intel disk label, this is called by sun_load_table 
 */

static void
sun_load_table_i386(MM_INFO *mm, sun_dlabel_i386 *dlabel_x86)
{
	u_int32_t idx = 0;

	if (verbose)
		fprintf(logfp, "load_table_i386: Number of partitions: %d\n",
		  getu16(mm, dlabel_x86->num_parts));

	/* Cycle through the partitions, there are either 8 or 16 */
	for (idx = 0; idx < getu16(mm, dlabel_x86->num_parts); idx++) {

		if (verbose)
			fprintf(logfp,
			  "load_table_i386: %d  Starting Sector: %lu  Size: %lu  Type: %d\n",
			   idx, 
			   (ULONG)getu32(mm, dlabel_x86->part[idx].start_sec),
			   (ULONG)getu32(mm, dlabel_x86->part[idx].size_sec),
			   getu16(mm,dlabel_x86->part[idx].type));

		if (getu32(mm, dlabel_x86->part[idx].size_sec) == 0)
			continue;

		/* Add the partition to the internal sorted list */
		mm_part_add(mm, 
		  getu32(mm, dlabel_x86->part[idx].start_sec),
		  getu32(mm, dlabel_x86->part[idx].size_sec),
		  sun_get_desc(getu16(mm,dlabel_x86->part[idx].type)), -1, idx);
	} 

	return;
}


/* load a sparc disk label, this is called from the general
 * sun_load_table
 */
static void
sun_load_table_sparc(MM_INFO *mm, sun_dlabel_sparc *dlabel_sp)
{
	u_int32_t idx = 0;
	u_int32_t cyl_conv;

	/* The value to convert cylinders to sectors */
	cyl_conv = getu16(mm, dlabel_sp->sec_per_tr) * 
	  getu16(mm, dlabel_sp->num_head);

	if (verbose)
		fprintf(logfp, "load_table_sparc: Number of partitions: %d\n",
		  getu16(mm, dlabel_sp->num_parts));

	/* Cycle through the partitions, there are either 8 or 16 */
	for (idx = 0; idx < getu16(mm, dlabel_sp->num_parts); idx++) {
		u_int32_t part_start = cyl_conv * 
		  getu32(mm, dlabel_sp->part_layout[idx].start_cyl);

		u_int32_t part_size = getu32(mm, 
		  dlabel_sp->part_layout[idx].size_blk);

		if (verbose)
			fprintf(logfp,
			  "load_table_sparc: %d  Starting Sector: %lu  Size: %lu  Type: %d\n",
			   idx, 
			   (ULONG)part_start, 
			   (ULONG)part_size, 
			   getu16(mm,dlabel_sp->part_meta[idx].type));

		if (part_size == 0)
			continue;

		/* Add the partition to the internal sorted list */
		mm_part_add(mm, part_start, part_size, 
		  sun_get_desc(getu16(mm,dlabel_sp->part_meta[idx].type)),
	    	  -1, idx); 
	} 

	return;
}


/* 
 * Process the partition table at the sector address 
 *
 * This method just finds out if it is sparc or Intel and then
 * calls the appropriate method
 */
static void 
sun_load_table(MM_INFO *mm, daddr_t sect)
{
/* this will need to change if any of the disk label structures change */
#define LABEL_BUF_SIZE	512

	sun_dlabel_sparc *dlabel_sp;
	sun_dlabel_i386 *dlabel_x86;
	char buf[LABEL_BUF_SIZE];
	off_t ret;
	off_t addr = (off_t)sect * 512; 


	/* Sanity check in case label sizes change */
	if ((sizeof (*dlabel_sp) > LABEL_BUF_SIZE) || 
	  (sizeof(*dlabel_x86) > LABEL_BUF_SIZE)) {
		error ("sun_load_table: Buffer smaller than label sizes");
	}

	if (verbose)
		fprintf(logfp, "sun_load_table: Trying sector: %lu\n",
		  (ULONG)sect);

	/* Try the given offset */
	if  (addr != (ret = lseek(mm->fd, addr, SEEK_SET))) {
		error ("Error seeking to table sector: %lu  Ret: %lu", 
		  (ULONG)addr, (ULONG)ret);
	}

	if (LABEL_BUF_SIZE != read (mm->fd, (char *)buf, 
	 LABEL_BUF_SIZE)) {
		error ("Error reading SUN Disk Label (Sector: %lu)\n", 
		  (ULONG)sect);
	}

	/* Check the magic value 
	 * Both intel and sparc have the magic value in the same location
	 *
	 * We try both in case someone specifies the exact location of the
	 * intel disk label.
	 * */
	dlabel_sp = (sun_dlabel_sparc *) buf;
	dlabel_x86 = (sun_dlabel_i386 *) buf;
	if (guessu16(mm, dlabel_sp->magic, SUN_MAGIC) == 0) {

		if (getu32(mm, dlabel_sp->sanity) == SUN_SANITY) {
			sun_load_table_sparc(mm, dlabel_sp);
			return;
		}
		else if (getu32(mm, dlabel_x86->sanity) == SUN_SANITY) {
			sun_load_table_i386(mm, dlabel_x86);
			return;
		}
	}


	/* Now try the next sector, which is where the intel 
	 * could be stored */
	if (verbose)
		fprintf(logfp, "sun_load_table: Trying sector: %lu\n",
		  (ULONG)sect + 1);
	if  ((addr + 512) != (ret = lseek(mm->fd, addr + 512, SEEK_SET))) {
		error ("Error seeking to table sector: %lu  Ret: %lu", 
		  (ULONG)addr + 512, (ULONG)ret);
	}

	if (LABEL_BUF_SIZE != read (mm->fd, (char *)buf, LABEL_BUF_SIZE)) {
		error ("Error reading SUN (Intel) Disk Label (Sector: %lu)\n", 
		  (ULONG)sect + 1);
	}

	dlabel_x86 = (sun_dlabel_i386 *) buf;
	if (guessu16(mm, dlabel_x86->magic, SUN_MAGIC)) {
		error ("Invalid SUN (intel) partition table (Sector: %lu) %x\n", 
		  (ULONG)sect, getu16(mm, dlabel_sp->magic));
	}

	if (getu32(mm, dlabel_x86->sanity) != SUN_SANITY) {
		error ("Invalid SUN (intel) sanity value (Sector: %lu) %x\n",
		  (ULONG)sect, getu16(mm, dlabel_sp->magic));
	}

	sun_load_table_i386(mm, dlabel_x86);

	return;
}


/* 
 * Walk the partitions that have already been loaded during _open
 *
 */
void
sun_part_walk(MM_INFO *mm, u_int16_t start, u_int16_t last, int flags, 
  MM_PART_WALK_FN action, char *ptr)
{
	MM_PART *part;
	int cnt = 0;

	if (start < mm->first_part || start > mm->last_part)
		error ("Invalid starting partition: %d", start);

	if (last < mm->first_part || last > mm->last_part)
		error ("Invalid ending partition: %d", last);

	part = mm->part_list;
	while ((part != NULL) && (cnt <= last)) {

		if (cnt >= start)
			action(mm, cnt, part, 0, ptr);

		part = part->next;
		cnt++;
	}

	return;
}


void 
sun_close (MM_INFO *mm)
{
	mm_part_free(mm);
	close(mm->fd);
	free(mm);
}

MM_INFO *
sun_open(const char *path, unsigned char type, daddr_t sect_offset)
{
	char *myname = "sun_open";

	MM_INFO *mm = (MM_INFO *)mymalloc(sizeof(*mm));

	if (type != MM_SUN)
		error ("%s: Invalid media type for Sun Partitions", myname);

	if ((mm->fd = open(path, O_RDONLY)) < 0)
		error("%s: open %s: %m", myname, path);

	/* use the offset provided */
	if (sect_offset)
		mm->sect_offset = sect_offset;
	else
		mm->sect_offset = SUN_SPARC_PART_OFFSET;

	/* inititialize settings */
	mm->mmtype = type;
	mm->str_type = "Sun VTOC";
	mm->part_list = NULL;
	mm->flags = 0;

	/* Assign functions */
	mm->part_walk = sun_part_walk;
	mm->close = sun_close;

	/* Load the partitions into the sorted list */
	sun_load_table(mm, mm->sect_offset);

	/* fill in the sorted list with the 'unknown' values */
	mm_part_unused(mm);

	return mm;
}
