/*
** jcat
** The Sleuth Kit 
**
**
** Brian Carrier [carrier@sleuthkit.org]
** Copyright (c) 2004 Brian Carrier.  All rights reserved
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are
** met:
**
** 1. Redistributions of source code must retain the above copyright notice,
**    this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**    notice, this list of conditions and the following disclaimer in the
**    documentation and/or other materials provided with the distribution.
** 3. The name of the author may not be used to endorse or promote
**    products derived from this software without specific prior written
**    permission.     
**
**
**
** THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
** WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
** MERCHANTABILITY AND FITNESS FOR ANY PARTICULAR PURPOSE.
**
** IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
** INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
** (INCLUDING, BUT NOT LIMITED TO, LOSS OF USE, DATA, OR PROFITS OR
** BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
** WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
** OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
** ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
*/

#include "fs_tools.h"
#include "error.h"

FILE   *logfp;


/* atoinum - convert string to inode number */
INUM_T  
atoinum(const char *str)
{
    char   *cp, *dash;
    INUM_T  inum;

    if (*str == 0)
		return (0);

	/* if we are given the inode in the inode-type-id form, then ignore
	 * the other stuff w/out giving an error 
	 *
	 * This will make scripting easier
	 */
	if ((dash = strchr(str, '-')) != NULL) {
		*dash = '\0';
	}
    inum = STRTOUL(str, &cp, 0);
    if (*cp || cp == str)
		error("bad inode number: %s", str);
    return (inum);
}


/* usage - explain and terminate */

static void usage() {
	printf("usage: %s [-f fstype] [-vV] image [inode] blk\n", progname);
	printf("\tblk: The journal block to view\n");
	printf("\tinode: The file system inode where the journal is located\n");
	printf("\t-v: verbose output to stderr\n");
	printf("\t-V: print version\n");
	printf("\t-f fstype: Image file system type\n");
	printf("Supported file system types:\n");
	fs_print_types();
	exit(1);
}


int
main(int argc, char **argv) 
{
	INUM_T	inum;
	int     ch;
	char   *fstype = DEF_FSTYPE;
	FS_INFO		*fs;
	u_int32_t	blk;


	progname = argv[0];


	while ((ch = getopt(argc, argv, "f:vV")) > 0) {
	switch (ch) {
	default:
		usage();
	case 'f':
		fstype = optarg;
		break;
	case 'v':
		verbose++;
		logfp = stderr;
		break;
	case 'V':
		print_version();
		exit(0);

	}
    }

	if (((optind + 2) != argc) && ((optind + 3) != argc))
		usage();


	/*
	 * Open the file system.
	 */
	fs = fs_open(argv[optind], fstype);
	if ((optind + 3) == argc) {
		inum = atoinum(argv[++optind]);
	}
	else {
		inum = fs->journ_inum;
	}
	blk = atoinum(argv[++optind]);

	if (inum > fs->last_inum) {
		fprintf (stderr, "Inode value is too large for image (%lu)\n", 
		  (ULONG)fs->last_inum);
		return 1;
	}

	if (inum < fs->first_inum) {
		fprintf (stderr, "Inode value is too small for image (%lu)\n", 
		  (ULONG)fs->first_inum);
		return 1;
	}
	if (fs->jopen == NULL) {
		fprintf(stderr, "Journal support does not exist for this file system\n");
		return 1;
	}

	fs->jopen(fs, inum);
	fs->jblk_walk(fs, blk, blk, 0, 0, "");

	fs->close(fs);
	exit(0);
}
