/* 
 * Brian Carrier 
 * carrier@sleuthkit.org
 *
 * Based on code from http://www.win.tue.nl/~aeb/linux/setmax.c
 */

/* setmax.c - aeb, 000326 - use on 2.4.0test9 or newer */
/* IBM part thanks to Matan Ziv-Av <matan@svgalib.org> */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#if defined(LINUX2)

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h> 
#include <fcntl.h>
#include <linux/hdreg.h>

#define READ_NATIVE_MAX_ADDRESS 0xf8
#define LBA	0x40

unsigned int
tolba(unsigned char *args) 
{
	return ((args[6] & 0xf) << 24) + (args[5] << 16) + (args[4] << 8) + args[3];
}

int
get_identity_max(int fd) 
{
	unsigned char args[4+512] = {WIN_IDENTIFY,0,0,1,};
	struct hd_driveid *id = (struct hd_driveid *)&args[4];

	if (ioctl(fd, HDIO_DRIVE_CMD, &args)) {
		perror("HDIO_DRIVE_CMD");
		fprintf(stderr,
			"WIN_IDENTIFY failed - trying WIN_PIDENTIFY\n");
		args[0] = WIN_PIDENTIFY;
		if (ioctl(fd, HDIO_DRIVE_CMD, &args)) {
			perror("HDIO_DRIVE_CMD");
			fprintf(stderr,
			       "WIN_PIDENTIFY also failed - giving up\n");
			exit(1);
		}
	}

	/* this value is capacity, so subtract 1 for max address */
	return id->lba_capacity - 1;
}



/*
 * result: in LBA mode precisely what is expected
 *         in CHS mode the correct H and S, and C mod 65536.
 *
 *         Get the maximum address of the disk
 */
unsigned int
get_native_max(int fd) 
{
	unsigned char args[7];
	int i;

	for (i=0; i<7; i++)
		args[i] = 0;

	args[0] = READ_NATIVE_MAX_ADDRESS;
	args[6] = LBA;
	if (ioctl(fd, HDIO_DRIVE_CMD_AEB, &args)) {
		perror("HDIO_DRIVE_CMD_AEB failed READ_NATIVE_MAX_ADDRESS");
		for (i=0; i<7; i++)
			printf("%d = 0x%x\n", args[i], args[i]);
		exit(1);
	}
	return tolba(args);
}

#endif


void
usage()
{
	fprintf(stderr, "usage: diskstat [-V] DEVICE\n");
	fprintf(stderr, "\t-V: Print version\n");
	return;
}

void
print_version()  
{
	char *str = "The Sleuth Kit";
#ifdef VER
	printf("%s ver %s\n", str, VER);
#else
	printf("%s\n", str);
#endif
	return;
}

#if defined(LINUX2)

int 
main(int argc, char **argv)
{
	int fd, max_ident, max_native;
	char *device = NULL;	/* e.g. "/dev/hda" */
	struct stat devstat;
	int ch;

	while ((ch = getopt(argc, argv, "V")) > 0) {
		switch(ch) {
			case 'V':
				print_version();
				return 0;
			default: 
				usage();
				return 0;
		}
	}

	if (optind < argc)
		device = argv[optind];

	if (!device) {
		fprintf(stderr, "no device specified\n");
		usage();
		exit(1);
	}

	if (0 != stat(device, &devstat)) {
		fprintf(stderr, "Error opening %s\n", device);
		exit(1);
	}

	if ((S_ISCHR(devstat.st_mode) == 0) && 
	  (S_ISBLK(devstat.st_mode) == 0) ) {
		fprintf(stderr, "The file name must correspond to a device\n");
		exit(1);
	}

	fd = open(device, O_RDONLY);
	if (fd == -1) {
		perror("open");
		exit(1);
	}

				
	max_native = get_native_max(fd);
	printf("Maximum Disk Sector: %d\n", max_native);
		
	max_ident = get_identity_max(fd);
	printf("Maximum User Sector: %d\n", max_ident);

	if (max_ident != max_native) {
		printf("\n** HPA Detected (Sectors %d - %d) **\n\n",
			max_ident + 1, max_native);
	}

	close(fd);	
	return 0;

}

#else

int 
main(int argc, char **argv)
{
	int ch;

	while ((ch = getopt(argc, argv, "V")) > 0) {
		switch(ch) {
			case 'V':
				print_version();
				return 0;
			default: 
				usage();
				return 0;
		}
	}

	fprintf(stderr, "This tool works only on Linux systems\n");	
	return 0;
}

#endif
