/*
  Formulas for the SLAT Labeled Transition System.
  Copyright (C) 2005 The MITRE Corporation

  Author: John D. Ramsdell

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
*/

#if !defined FORMULA_H
#define FORMULA_H

#ifdef __cplusplus
extern "C"
{
#endif

/* This defines the C interface to formulas used in SLAT with the
   exception of symbols. */

/* Before any other formula routine is called, a formula stack must be
   provided by calling the following routine.  You might define the
   stack as follows:

   #include <obstack.h>

   #define obstack_chunk_alloc xmalloc
   #define obstack_chunk_free free

   static struct obstack stack[1]; */
struct obstack *formula_stack(struct obstack *);

typedef struct symbol_list *symbol_list_t;

symbol_list_t mk_symbol_list(symbol_t, symbol_list_t);
symbol_t symbol_list_head(symbol_list_t);
symbol_list_t symbol_list_tail(symbol_list_t);
symbol_list_t symbol_list_reverse(symbol_list_t);

typedef enum {TRAN_TRUE, TRAN_FALSE,
	      TRAN_CLASSES, TRAN_PERMISSIONS,
	      TRAN_TYPES, NEXT_TYPES, SAME_TYPES,
	      TRAN_ROLES, NEXT_ROLES, SAME_ROLES,
	      TRAN_USERS, NEXT_USERS, SAME_USERS,
	      TRAN_NOT, TRAN_AND, TRAN_OR,
	      TRAN_IMPLY, TRAN_IFF,
	      TRAN_OKAY, TRAN_AX, TRAN_NEXT, TRAN_UNTIL} tran_type_t;

typedef struct tran *tran_t;

tran_t mk_tran_true(void);
tran_t mk_tran_false(void);
tran_t mk_tran_classes(symbol_list_t);
tran_t mk_tran_permissions(symbol_list_t);
tran_t mk_tran_types(symbol_list_t);
tran_t mk_next_types(symbol_list_t);
tran_t mk_same_types(void);
tran_t mk_tran_roles(symbol_list_t);
tran_t mk_next_roles(symbol_list_t);
tran_t mk_same_roles(void);
tran_t mk_tran_users(symbol_list_t);
tran_t mk_next_users(symbol_list_t);
tran_t mk_same_users(void);
tran_t mk_tran_not(tran_t);
tran_t mk_tran_and(tran_t, tran_t);
tran_t mk_tran_or(tran_t, tran_t);
tran_t mk_tran_imply(tran_t, tran_t);
tran_t mk_tran_iff(tran_t, tran_t);
tran_t mk_tran_okay(void);
tran_t mk_tran_ax(tran_t);
tran_t mk_tran_next(tran_t);
tran_t mk_tran_until(tran_t, tran_t);

tran_type_t tran_type(tran_t);
symbol_list_t tran_list(tran_t);
tran_t tran_arg1(tran_t);
tran_t tran_arg2(tran_t);

typedef struct tran_list *tran_list_t;

tran_list_t mk_tran_list(tran_t, tran_list_t);
tran_t tran_list_head(tran_list_t);
tran_list_t tran_list_tail(tran_list_t);
tran_list_t tran_list_reverse(tran_list_t);

typedef struct arrow *arrow_t;

arrow_t mk_arrow(tran_t, tran_t, int);
tran_t arrow_state(arrow_t);
tran_t arrow_action(arrow_t);
int arrow_more(arrow_t);

typedef struct arrow_list *arrow_list_t;

arrow_list_t mk_arrow_list(arrow_t, arrow_list_t);
arrow_t arrow_list_head(arrow_list_t);
arrow_list_t arrow_list_tail(arrow_list_t);
arrow_list_t arrow_list_reverse(arrow_list_t);

typedef struct diagram *diagram_t;

diagram_t mk_diagram(arrow_list_t, tran_t, tran_t, tran_t);
arrow_list_t diagram_arrows(diagram_t);
tran_t diagram_state(diagram_t);
tran_t diagram_except_state(diagram_t);
tran_t diagram_except_action(diagram_t);

typedef struct diagram_list *diagram_list_t;

diagram_list_t mk_diagram_list(diagram_t, diagram_list_t);
diagram_t diagram_list_head(diagram_list_t);
diagram_list_t diagram_list_tail(diagram_list_t);
diagram_list_t diagram_list_reverse(diagram_list_t);

typedef struct lts *lts_t;

lts_t mk_lts(symbol_list_t types,
	     symbol_list_t roles,
	     symbol_list_t users,
	     symbol_list_t classes,
	     symbol_list_t permissions,
	     tran_t initial,
	     tran_t transition,
	     tran_list_t specifications);
symbol_list_t lts_types(lts_t);
symbol_list_t lts_roles(lts_t);
symbol_list_t lts_users(lts_t);
symbol_list_t lts_classes(lts_t);
symbol_list_t lts_permissions(lts_t);
tran_t lts_initial(lts_t);
tran_t lts_transition(lts_t);
tran_list_t lts_specifications(lts_t);

int is_state(tran_t);

int print_transition(FILE *, tran_t);

void print_lts(FILE *, lts_t);

#ifdef __cplusplus
}
#endif

#endif
