/* $Id: main.c,v 1.21 2003/11/17 20:54:52 kwalker Exp $ */

/*
 * Copyright 2001, 2002, 2003
 * Broadcom Corporation. All rights reserved.
 *
 * This software is furnished under license and may be used and copied only
 * in accordance with the following terms and conditions.  Subject to these
 * conditions, you may download, copy, install, use, modify and distribute
 * modified or unmodified copies of this software in source and/or binary
 * form. No title or ownership is transferred hereby.
 *
 * 1) Any source code used, modified or distributed must reproduce and
 *    retain this copyright notice and list of conditions as they appear in
 *    the source file.
 *
 * 2) No right is granted to use any trade name, trademark, or logo of
 *    Broadcom Corporation.  The "Broadcom Corporation" name may not be
 *    used to endorse or promote products derived from this software
 *    without the prior written permission of Broadcom Corporation.
 *
 * 3) THIS SOFTWARE IS PROVIDED "AS-IS" AND ANY EXPRESS OR IMPLIED
 *    WARRANTIES, INCLUDING BUT NOT LIMITED TO, ANY IMPLIED WARRANTIES OF
 *    MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR
 *    NON-INFRINGEMENT ARE DISCLAIMED. IN NO EVENT SHALL BROADCOM BE LIABLE
 *    FOR ANY DAMAGES WHATSOEVER, AND IN PARTICULAR, BROADCOM SHALL NOT BE
 *    LIABLE FOR DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *    CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *    SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 *    BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 *    WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 *    OR OTHERWISE), EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "libc.h"
#include "cfe_api.h"
#include "elf.h"
#include "netinit.h"
#include "config.h"
#include "getfile.h"
#include "tftp_fileops.h"
#include "parse_config.h"
#include "port.h"

typedef struct {
	unsigned char active;
	unsigned char start_chs[3];
	unsigned char id;
	unsigned char end_chs[3];
	uint32_t start_sector;
	uint32_t size;
} part_entry_t;

/* Don't set this too large; we do have to allocate a buffer this size */
#define MAX_CONFIG_FILE_SIZE (32 * 1024)

#define PAGE_SIZE (4*1024)

static void elf32_ehdr_swap(Elf32_Ehdr *ehdr_ptr);
static int get_config_file(char *loc, char *buf, int32_t buf_size);
static void elf32_phdr_swap(Elf32_Phdr *phdr_ptr);
static void *load_kernel(char *loc);
static int32_t load_ramdisk(char *loc, char *mem_start);
static void zero_bss(void);
extern int _fbss;
extern int _end;
typedef void kernel_entry_fn_t(int, unsigned char **, char **, long *);
kernel_entry_fn_t *kernel_entry;

static int swap;

#ifndef SILENT
#define msg(str) lib_printf str
#else
#define msg(str) do {} while(0)
#endif

#ifdef CONFIG_BLOCKDEV
extern file_ops_t ext2_ops;
#endif


static void init_libs(char *,int);
static void uninit_libs(void);

extern void gets(char *,int,int);

static void fatal_error(char *str);

/* 
 * We end up here after init.S is done setting
 * up some sane environment .  
 * Massive trickery here -- we actually *fall into* this
 * routine, so main.c must be linked immediately after init.S.
 * gross!
 */
void start(long handle, long reserved, long ept, long eptseal)
{
    char *tmp;
    char *config_file;
    char *bootdev_name;
    void *initrd_start;
    int bootdev_type;
    long *linux_args;

    zero_bss();
    cfe_init(handle, ept);
    lib_init(0);
    lib_printf("SiByte Loader, version %i.%i.%i\n", 
	       VERSION_MAJOR, VERSION_MINOR, VERSION_MINUTE);
    lib_printf("Built on " __DATE__"\n");
    if (eptseal != CFE_EPTSEAL)
      lib_printf("WARNING: CFE entrypoint seal doesn't match; continuing anyway.\n");

    config_file = lib_malloc(128);
    bootdev_name = lib_malloc(128);
    if (!config_file || !bootdev_name) {
	fatal_error("Out of memory");
	}

    if (cfe_getenv("BOOT_DEVICE",bootdev_name,128)) {
	lib_strcpy(bootdev_name,"eth0");
	}

    bootdev_type = cfe_getdevinfo(bootdev_name);

    init_libs(bootdev_name,bootdev_type);

    /*
     * Look for a hardwired BOOT_CONFIG environment variable
     * to get our config file.  If it is not present,
     * compose a name from the boot device type
     * and the BOOT_FILE variable which was set by CFE.
     */

    config_file[0] = 0;
    if (cfe_getenv("BOOT_CONFIG", config_file, 128) < 0) {

	char *cptr;
	int clen;

	/* 
	 * Choose a boot method based on the type of the
	 * device that CFE loaded SIBYL from.  Construct
	 * a config file name as follows:
	 *
	 *  network:    tftp:hostname:path/to/config.file
	 *
	 *  disk:	ext2:devicename:partition#:path/to/config
	 */

	if ((bootdev_type & CFE_DEV_MASK) == CFE_DEV_NETWORK) {
	    lib_strcpy(config_file,"tftp:");
	    }
	else {
	    lib_strcpy(config_file,"ext2:");
	    lib_strcat(config_file,bootdev_name);
	    lib_strcat(config_file,":");
	    }

	clen = lib_strlen(config_file);
	cptr = &config_file[0] + clen;

	/* 
	 * No override, try to get boot file name from CFE -
	 * This is the name of our sibyl image usually.
	 */
	if (cfe_getenv("BOOT_FILE",cptr,128-clen-6) == 0) {
	    if (cptr[0] == 0) lib_strcat(cptr,"*:sibyl");
	    lib_strcat(cptr,".conf");
	    }
	else config_file[0] = 0;		/* variable not found */
	}

    /* When all else fails, prompt. */
    if (config_file[0] == 0) {
	lib_printf("BOOT_CONFIG not defined.\n"
		   "Enter config file location: ");
	gets(config_file,128,0);
	}

    config_file[127] = 0;
    tmp = lib_malloc(MAX_CONFIG_FILE_SIZE);
    if (!tmp) {
	fatal_error("Out of memory");
	}
    lib_printf("Getting configuration file %s...\n", config_file);

    for (;;) {
	if (get_config_file(config_file, tmp, MAX_CONFIG_FILE_SIZE) == 0) break;
	lib_printf("Failed to get configuration file.\n"
	    "Enter configuration file location: ");
	gets(config_file,128,0);
	}
    lib_printf("Config file retrieved.\n");
    parse_config_buf(tmp);
    lib_free(config_file);
    lib_free(tmp);

    /* Now, load the kernel */
    initrd_start = load_kernel(boot_config->kernel);

    /* And load the ramdisk, if applicable */
    if (boot_config->initrd) {
	int32_t ramdisk_size;
	char *ptr;
	ramdisk_size = load_ramdisk(boot_config->initrd, initrd_start);
	lib_free(boot_config->initrd);
	boot_config->initrd = lib_malloc(36);
	lib_strcpy(boot_config->initrd, "initrd=");
	ptr = boot_config->initrd + 7;
	ptr += lib_ultostr(ptr, ramdisk_size);
	*ptr = '@';
	ptr++;
	lib_ultostr(ptr, (unsigned long)initrd_start);
	}

    /* 
     * Set up command line arguments as an environment variable in cfe.  The
     * kernel will pick them up from there 
     */
	
    tmp = lib_malloc(lib_strlen(boot_config->root_dev) + 1 +
		     (boot_config->extra_args?(lib_strlen(boot_config->extra_args) + 1):0) +
		     (boot_config->initrd?(lib_strlen(boot_config->initrd) + 1):0));
    if (!tmp) {
	fatal_error("Out of memory\n");
	}

    tmp[0] = 0;
    lib_strcpy(tmp, "root=");
    lib_strcat(tmp, boot_config->root_dev);
    if (boot_config->extra_args) {
	lib_strcat(tmp, " ");
	lib_strcat(tmp, boot_config->extra_args);
	}
    if (boot_config->initrd) {
	lib_strcat(tmp, " ");
	lib_strcat(tmp, boot_config->initrd);
	}
    if (cfe_setenv("LINUX_CMDLINE", tmp) < 0) {
	fatal_error("setenv failed");
	}

    lib_printf("Set up command line arguments to: %s\n", tmp);
    lib_printf("Setting up initial prom_init arguments\n");
    linux_args = lib_malloc(4 * sizeof(long));
    linux_args[0] = handle;
    linux_args[1] = reserved;
    linux_args[2] = ept;
    linux_args[3] = eptseal;
    lib_printf("Cleaning up state...\n");
    uninit_libs();
    lib_printf("Transferring control to the kernel.\n");
    lib_printf("Kernel entry point is at %p\n", kernel_entry);
    kernel_entry(0, 0, 0, linux_args);
}


void fatal_error(char *str)
{
    uninit_libs();
    lib_die(str);
}


static void init_libs(char *bootdev_name,int bootdev_type)
{
#ifdef CONFIG_NETWORK

    char buf[128];
    uint32_t ip_addr = 0;
    uint32_t ip_netmask = 0;
    uint32_t ip_gateway = 0;
    uint32_t ip_ns1 = 0;

    if ((bootdev_type & CFE_DEV_MASK) == CFE_DEV_NETWORK) {
	/*
	 * Do IP addr last.  If we fail anytime before then, it's still
	 * 0, which tells net_init to try dhcp.  
         */
	if (!cfe_getenv("NET_NETMASK", buf, 128)) {
	    buf[127] = 0;
	    ip_netmask = lib_parse_ip(buf);
	    if (!cfe_getenv("NET_GATEWAY", buf, 128)) {
		buf[127] = 0;
		ip_gateway = lib_parse_ip(buf);
		if (!cfe_getenv("NET_NAMESERVER", buf, 128)) {
		    buf[127] = 0;
		    ip_ns1 = lib_parse_ip(buf);
		    if (!cfe_getenv("NET_IPADDR", buf, 128)) {
			buf[127] = 0;
			ip_addr = lib_parse_ip(buf);
			}
		    }
		}
	    }
	if (cfe_getenv("NET_DOMAIN",buf,128)) buf[0] = 0;

	if (!ip_addr) {
	    lib_printf("Environment doesn't specify full IP configuration.  Will try DHCP\n");
	    }
	if (!net_init(bootdev_name,ip_addr, ip_netmask, ip_gateway, ip_ns1, 0, buf)) {
	    lib_printf("Network device '%s' configured\n",bootdev_name);
	    file_handler_add(&tftp_ops);
	    } 
	else {
	    lib_printf("Network booting disabled\n");
	    }
	}
#endif /* CONFIG_NETWORK */

#ifdef CONFIG_BLOCKDEV
    file_handler_add(&ext2_ops);
#endif
}

static int get_config_file(char *loc, char *buf, int32_t buf_size)
{
    file_ops_t *ops;
    void *ops_data;
    int32_t file_len;

    ops = file_handler_find(loc, &ops_data);

    if (!ops) {
	lib_printf("No handler found for this: %s\n", loc);
	return -1;
	}

    file_len = ops->read(ops_data, buf, buf_size);

    if (file_len == MAX_CONFIG_FILE_SIZE) {
	fatal_error("Config file too large\n");
	}

    file_handler_close(ops,ops_data);

    if (file_len < 0) return -1;

    buf[file_len] = 0;

    return 0;
}

static void elf32_ehdr_swap(Elf32_Ehdr *ehdr_ptr)
{
	ehdr_ptr->e_type = port_swap_16(ehdr_ptr->e_type);
	ehdr_ptr->e_machine = port_swap_16(ehdr_ptr->e_machine);
	ehdr_ptr->e_version = port_swap_32(ehdr_ptr->e_version);
	ehdr_ptr->e_entry = port_swap_32(ehdr_ptr->e_entry);
	ehdr_ptr->e_phoff  = port_swap_32(ehdr_ptr->e_phoff);
	ehdr_ptr->e_shoff  = port_swap_32(ehdr_ptr->e_shoff);
	ehdr_ptr->e_flags  = port_swap_32(ehdr_ptr->e_flags);
	ehdr_ptr->e_ehsize  = port_swap_16(ehdr_ptr->e_ehsize);
	ehdr_ptr->e_phentsize  = port_swap_16(ehdr_ptr->e_phentsize);
	ehdr_ptr->e_phnum  = port_swap_16(ehdr_ptr->e_phnum);
	ehdr_ptr->e_shentsize  = port_swap_16(ehdr_ptr->e_shentsize);
	ehdr_ptr->e_shnum  = port_swap_16(ehdr_ptr->e_shnum);
	ehdr_ptr->e_shstrndx  = port_swap_16(ehdr_ptr->e_shstrndx);
}

static void elf32_phdr_swap(Elf32_Phdr *phdr_ptr)
{
	phdr_ptr->p_type = port_swap_32(phdr_ptr->p_type);
	phdr_ptr->p_offset = port_swap_32(phdr_ptr->p_offset);
	phdr_ptr->p_vaddr = port_swap_32(phdr_ptr->p_vaddr);
	phdr_ptr->p_paddr = port_swap_32(phdr_ptr->p_paddr);
	phdr_ptr->p_filesz = port_swap_32(phdr_ptr->p_filesz);
	phdr_ptr->p_memsz = port_swap_32(phdr_ptr->p_memsz);
	phdr_ptr->p_flags = port_swap_32(phdr_ptr->p_flags);
	phdr_ptr->p_align = port_swap_32(phdr_ptr->p_align);
}

static void elf64_ehdr_swap(Elf64_Ehdr *ehdr_ptr)
{
	ehdr_ptr->e_type = port_swap_16(ehdr_ptr->e_type);
	ehdr_ptr->e_machine = port_swap_16(ehdr_ptr->e_machine);
	ehdr_ptr->e_version = port_swap_32(ehdr_ptr->e_version);
	ehdr_ptr->e_entry = port_swap_64(ehdr_ptr->e_entry);
	ehdr_ptr->e_phoff  = port_swap_64(ehdr_ptr->e_phoff);
	ehdr_ptr->e_shoff  = port_swap_64(ehdr_ptr->e_shoff);
	ehdr_ptr->e_flags  = port_swap_32(ehdr_ptr->e_flags);
	ehdr_ptr->e_ehsize  = port_swap_16(ehdr_ptr->e_ehsize);
	ehdr_ptr->e_phentsize  = port_swap_16(ehdr_ptr->e_phentsize);
	ehdr_ptr->e_phnum  = port_swap_16(ehdr_ptr->e_phnum);
	ehdr_ptr->e_shentsize  = port_swap_16(ehdr_ptr->e_shentsize);
	ehdr_ptr->e_shnum  = port_swap_16(ehdr_ptr->e_shnum);
	ehdr_ptr->e_shstrndx  = port_swap_16(ehdr_ptr->e_shstrndx);
}

static void elf64_phdr_swap(Elf64_Phdr *phdr_ptr)
{
	phdr_ptr->p_type = port_swap_32(phdr_ptr->p_type);
	phdr_ptr->p_flags = port_swap_32(phdr_ptr->p_flags);
	phdr_ptr->p_offset = port_swap_64(phdr_ptr->p_offset);
	phdr_ptr->p_vaddr = port_swap_64(phdr_ptr->p_vaddr);
	phdr_ptr->p_paddr = port_swap_64(phdr_ptr->p_paddr);
	phdr_ptr->p_filesz = port_swap_64(phdr_ptr->p_filesz);
	phdr_ptr->p_memsz = port_swap_64(phdr_ptr->p_memsz);
	phdr_ptr->p_align = port_swap_64(phdr_ptr->p_align);
}


static void *load_kernel_elf32(Elf32_Ehdr *ehdr,
			       file_ops_t *ops,
			       void *ops_data)
{
	unsigned long initrd_start_loc = 0;
	Elf32_Phdr *phdrs;
	int32_t readlen;
	int i;

	phdrs = lib_malloc(sizeof(Elf32_Phdr) * ehdr->e_phnum);
	if (!phdrs) {
		fatal_error("Out of memory");
	}
	lib_printf("Loading kernel (ELF32):\n");
	
	ops->seek(ops_data, ehdr->e_phoff, GF_SEEK_SET);
	readlen = ops->read(ops_data, phdrs, sizeof(Elf32_Phdr) * ehdr->e_phnum);
	if (readlen != (ehdr->e_phnum * sizeof(Elf32_Phdr))) {
		fatal_error("Error reading program headers");
	}

	for (i = 0; i < ehdr->e_phnum; i++) {
		if (swap)
			elf32_phdr_swap(&phdrs[i]);
		/* Skip anything that's not a loadable segment */
		if (phdrs[i].p_type == PT_LOAD) {
			if (phdrs[i].p_flags & (PHP_X | PHP_W | PHP_R)) { 
				lib_printf("    %i@0x%X\n", phdrs[i].p_filesz, phdrs[i].p_vaddr);
				ops->seek(ops_data, phdrs[i].p_offset, GF_SEEK_SET);
				ops->read(ops_data, (void *)(
#ifndef __linux__
					(int64_t)
#endif
					phdrs[i].p_vaddr), phdrs[i].p_filesz);
				if (phdrs[i].p_vaddr + phdrs[i].p_memsz > initrd_start_loc) {
					initrd_start_loc = (phdrs[i].p_vaddr + phdrs[i].p_memsz + (PAGE_SIZE - 1)) 
						& ~((unsigned long)(PAGE_SIZE - 1));
				}
				if (phdrs[i].p_filesz < phdrs[i].p_memsz) {
					/* BSS segment.  We need to zero out the data that wasn't actually written by the
					   file image */
					lib_memset((void *)(
#ifndef __linux__
						(int64_t)
#endif
						(phdrs[i].p_vaddr + phdrs[i].p_filesz)), 0, 
						   phdrs[i].p_memsz - phdrs[i].p_filesz);
				}
			}
		}
	}
	lib_printf("done\n");
	file_handler_close(ops,ops_data);
	kernel_entry = (kernel_entry_fn_t *) (
#ifndef __linux__
		(int64_t)
#endif
		(intptr_t) (signed) ehdr->e_entry);
	return (void *)initrd_start_loc;
}

static void *load_kernel_elf64(file_ops_t *ops,
			       void *ops_data)
{
	unsigned long initrd_start_loc = 0;
	Elf64_Ehdr ehdr;
	Elf64_Phdr *phdrs;
	int32_t readlen;
	int i;

	ops->seek(ops_data, 0, GF_SEEK_SET);
	readlen = ops->read(ops_data, &ehdr, sizeof(ehdr));
	if (readlen != sizeof(ehdr)) {
		lib_die("ELF header read failed\n");
	}

	if (swap)
		elf64_ehdr_swap(&ehdr);
	phdrs = lib_malloc(sizeof(Elf64_Phdr) * ehdr.e_phnum);
	if (!phdrs) {
		fatal_error("Out of memory");
	}
	lib_printf("Loading kernel (ELF64):\n");

	ops->seek(ops_data, ehdr.e_phoff, GF_SEEK_SET);
	readlen = ops->read(ops_data, phdrs, sizeof(Elf64_Phdr) * ehdr.e_phnum);
	if (readlen != (ehdr.e_phnum * sizeof(Elf64_Phdr))) {
		fatal_error("Error reading program headers");
	}

	/* XXXKW check for non 32-compat addresses, and > 32-bit offsets */

	for (i = 0; i < ehdr.e_phnum; i++) {
		if (swap)
			elf64_phdr_swap(&phdrs[i]);
		/* Skip anything that's not a loadable segment */
		if (phdrs[i].p_type == PT_LOAD) {
			if (phdrs[i].p_flags & (PHP_X | PHP_W | PHP_R)) { 
				lib_printf("    %i@0x%X\n",
					   (int32_t)phdrs[i].p_filesz,
					   (int32_t)phdrs[i].p_vaddr);
				ops->seek(ops_data, phdrs[i].p_offset, GF_SEEK_SET);
				ops->read(ops_data, (void *)(int32_t)(phdrs[i].p_vaddr),
					  phdrs[i].p_filesz);
				if (phdrs[i].p_vaddr + phdrs[i].p_memsz > initrd_start_loc) {
					initrd_start_loc = (phdrs[i].p_vaddr + phdrs[i].p_memsz + (PAGE_SIZE - 1)) 
						& ~((unsigned long)(PAGE_SIZE - 1));
				}
				if (phdrs[i].p_filesz < phdrs[i].p_memsz) {
					/* BSS segment.  We need to zero out the data that wasn't actually written by the
					   file image */
					lib_memset((void *)(int32_t)((phdrs[i].p_vaddr +
								      phdrs[i].p_filesz)), 0, 
						   phdrs[i].p_memsz - phdrs[i].p_filesz);
				}
			}
		}
	}
	lib_printf("done\n");
	file_handler_close(ops,ops_data);
	kernel_entry = (kernel_entry_fn_t *) (
#ifndef __linux__
		(int64_t)
#endif
		(intptr_t) (signed) ehdr.e_entry);
	return (void *)initrd_start_loc;
}

/*
 * FIXME - Add some sanity checking to the reads and give some
 * useful information in case of failure 
 */
static void *load_kernel(char *loc)
{
	file_ops_t *ops;
	void *ops_data;
	Elf32_Ehdr ehdr;
	int32_t readlen;

	ops = file_handler_find(loc, &ops_data);
	if (!ops) {
		lib_printf("Load failed: %s\n", loc);
		fatal_error("");
	}
	readlen = ops->read(ops_data, &ehdr, sizeof(ehdr));
	if (readlen != sizeof(ehdr)) {
		lib_die("ELF header read failed\n");
	}
	if (lib_memcmp(ehdr.e_ident, ELFMAG, 4)) {
		fatal_error("Elf magic number not present in kernel\n");
	}
#ifdef __MIPSEL__
	swap = (ehdr.e_ident[EI_DATA] == ELFDATA2MSB);
#else
	swap = (ehdr.e_ident[EI_DATA] == ELFDATA2LSB);
#endif
	if (swap)
		elf32_ehdr_swap(&ehdr);
	if (ehdr.e_ident[EI_CLASS] == ELFCLASS32) {
		return load_kernel_elf32(&ehdr, ops, ops_data);
	} else {
//		fatal_error("Elf64 not supported");
		return load_kernel_elf64(ops, ops_data);
	}
	return 0;
}

static int32_t load_ramdisk(char *loc, char *mem_start)
{
	file_ops_t *ops;
	void *ops_data;
	int32_t readlen;
	ops = file_handler_find(loc, &ops_data);
	if (!ops) {
		lib_printf("Load failed: %s\n", loc);
		fatal_error("");
	}
	lib_printf("Loading ramdisk at 0x%X...", (unsigned long)mem_start);
	readlen = ops->read(ops_data, mem_start, 0xffffffff);
	lib_printf("%i bytes loaded\n", readlen);
	return readlen;
}

static void uninit_libs(void)
{
#ifdef CONFIG_NETWORK
	net_uninit();
#endif
}

static void zero_bss(void)
{
    int *ptr,*eptr;

    ptr = &_fbss;
    eptr = &_end;

    while (ptr < eptr) *ptr++ = 0;
}

#ifdef CONFIG_BLOCKDEV
/* libext2fs wants this defined when you compile with -O0.  Must
   be inlined at -O2 or some such */
void com_err(const char *foo, long bar, const char *gla, ...) 
{
}

int sprintf(char *dest,char *arg,...)
{
    lib_strcpy(dest,arg);
    return lib_strlen(dest);
}
#endif


