/* qshutdown, a program to shutdown the shutdown/reboot/suspend/hibernate
 * Copyright (C) 2009 2010  Christian Metscher <hakaishi@web.de>

 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "gui.h"
#include "power.h"
#include "preferences.h"
#include "info.h"

bool timeRunning;

Gui::Gui(QWidget *parent): QWidget(parent){

	setupUi(this);

     finishedExecuted = false;
     timeRunning = false;
     logFileSize = 1.5;

     datetime = QDateTime::currentDateTime();
     elapsedTime.start();

     setWindowFlags(Qt::WindowStaysOnTopHint);

   //Preferences
     pref = new Preferences(this);

   //InfoMessageBox
     infoBox = new Info(this);

   //Changing SizePolicy for Spacers
     horizontalSpacer1->changeSize(40,45,QSizePolicy::Expanding,QSizePolicy::Expanding);
     horizontalSpacer3->changeSize(85,13,QSizePolicy::Maximum,QSizePolicy::Expanding);
     horizontalSpacer4->changeSize(85,13,QSizePolicy::Maximum,QSizePolicy::Expanding);
     horizontalSpacer5->changeSize(85,13,QSizePolicy::Maximum,QSizePolicy::Expanding);

   //TrayIcon
     icon = QIcon(":/pixmap/red_glasses.xpm");
     TIcon = new QSystemTrayIcon(this);
     TIcon->setIcon(icon);
     TIcon->show();

   //TrayIconContextMenu
     menu = new QMenu(this);
     reset_action = new QAction(tr("R&eset \t Ctrl+E"),this);
     quit_action = new QAction(tr("&Quit \t Ctrl+Q"),this);
     info_action = new QAction(tr("&Info \t Ctrl+I"),this);
     log_action = new QAction(tr("&Logfile \t Ctrl+L"),this);
     log_action->setCheckable(true);
     pref_action = new QAction(tr("&Preferences \t Ctrl+P"),this);

     power_actions = new QActionGroup(this);
     shutdown_action = new QAction(tr("&Shutdown \t Ctrl+S"),this);
     shutdown_action->setCheckable(true);
     reboot_action = new QAction(tr("&Reboot \t Ctrl+R"),this);
     reboot_action->setCheckable(true);
     suspend_action = new QAction(tr("S&uspend \t Ctrl+U"),this);
     suspend_action->setCheckable(true);
     hibernate_action = new QAction(tr("&Hibernate \t Ctrl+H"),this);
     hibernate_action->setCheckable(true);
     power_actions->addAction(shutdown_action);
     power_actions->addAction(reboot_action);
     power_actions->addAction(suspend_action);
     power_actions->addAction(hibernate_action);

     menu->addAction(info_action);
     menu->addAction(reset_action);
     menu->addAction(pref_action);
     menu->addAction(log_action);
     menu->addSeparator();
     menu->addAction(shutdown_action);
     menu->addAction(reboot_action);
     menu->addAction(suspend_action);
     menu->addAction(hibernate_action);
     reset_action->setVisible(false);
     menu->addSeparator();
     menu->addAction(quit_action);
     
     TIcon->setContextMenu(menu);  //setting contextmenu for the systray

   //PushButton Minimize
     minim = new QPushButton(this); //to minimize
     minim->setText(tr("&Minimize!"));
     minim->setHidden(true); //not yet visible
     minim->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
     gridLayout5->addWidget(minim,0,1,3,1);

   //LCD-Anzeige
     lcd = new QLCDNumber(4, this); //display for countdown
     lcd->setSegmentStyle(QLCDNumber::Flat);
     lcd->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
     gridLayout4->addWidget(lcd,0,1,1,1);

   //Fonts
     font1 = new QFont;
     font1->setFamily(QString::fromUtf8("Times"));
     font1->setBold(true);

     font2 = new QFont;
     font2->setFamily(QString::fromUtf8("Times"));
     font2->setBold(true);

     font3 = new QFont;
     font3->setFamily(QString::fromUtf8("Times"));

   //Timer
     timer = new QTimer(this); //for updating countdown display and check if time is over
     ti = new QTimer(this); //for displaying the window, till the OK-Button is clicked
     ti->start(30000); //gives every 30 seconds a timeout()-signal

   //Connect signals with slots (actions with funktions)
     connect(radio1, SIGNAL(toggled(bool)), timeEdit, SLOT(setEnabled(bool)));
     connect(radio2, SIGNAL(toggled(bool)), spin, SLOT(setEnabled(bool)));
     connect(comboBox, SIGNAL(currentIndexChanged(int)), this, SLOT(power_action(int)));
     connect(power_actions, SIGNAL(triggered(QAction*)), this, SLOT(power(QAction*)));
     connect(ti, SIGNAL(timeout()), this, SLOT(showW())); //set window state for ti
     connect(ok, SIGNAL(clicked(bool)), this, SLOT(starter())); //starts timer & stopps ti
     connect(timer, SIGNAL(timeout()), this, SLOT(update()));    //runs update() every 10 seconds
     connect(now, SIGNAL(clicked(bool)), this, SLOT(finished()));  //closes the program & starts the shutdown immediately
     connect(minim, SIGNAL(clicked(bool)), this, SLOT(showMinimized())); //minimizes window
     connect(TIcon, SIGNAL(activated(QSystemTrayIcon::ActivationReason)),
       this, SLOT(iconActivated(QSystemTrayIcon::ActivationReason))); //handles systray-symbol
     connect(pref_action, SIGNAL(triggered()), pref, SLOT(show()));
     connect(pref, SIGNAL(starting()), ti, SLOT(stop()));
     connect(pref, SIGNAL(finishing()), ti, SLOT(start())); //start the Qtimer ti if timeRunning == false
     connect(reset_action, SIGNAL(triggered()), this, SLOT(reset()));
     connect(info_action, SIGNAL(triggered()), this, SLOT(setInfoText()));
     connect(info_action, SIGNAL(triggered()), infoBox, SLOT(show()));
     connect(infoBox, SIGNAL(starting()), ti, SLOT(stop()));
     connect(infoBox, SIGNAL(finishing()), ti, SLOT(start()));
     connect(qApp, SIGNAL(aboutToQuit()), this, SLOT(beforeQuit())); //save window size / logs and unregister qshutdown
     connect(quit_action, SIGNAL(triggered()), qApp, SLOT(quit())); //contextmenu "Quit" for the systray quits Programm
}

Gui::~Gui(){ delete font1; delete font2; delete font3; }

void Gui::setInfoText(){ infoBox->setPlainText(informations); }

void Gui::center(){
     QDesktopWidget *desktop = qApp->desktop();
     QPoint pos((desktop->width() - frameGeometry().width()) / 2,
                (desktop->height() - frameGeometry().height()) / 2);
     move(pos);
}

void Gui::iconActivated(QSystemTrayIcon::ActivationReason reason){
     if(reason == QSystemTrayIcon::Trigger){
       if(isHidden()||isMinimized()){  //in case thant the window is minimized or hidden
         center();
         showNormal();
       }
       else
         hide();
     }
}

void Gui::power_action(int action){
     QList<QAction*> actions = power_actions->actions();
     actions[action]->setChecked(true);
     switch(action){
       case 0:
         targetTime->setText(tr("Shutdown-time:"));
         minutes->setText(tr("Minutes till shutdown:"));
         break;
       case 1:
         targetTime->setText(tr("Reboot-time:"));
         minutes->setText(tr("Minutes till reboot:"));
         break;
       case 2:
         targetTime->setText(tr("Suspend-time:"));
         minutes->setText(tr("Minutes till suspend:"));
         break;
       case 3:
         targetTime->setText(tr("Hibernate-time:"));
         minutes->setText(tr("Minutes till hibernate:"));
         break;
       default:;
     }
}

void Gui::power(QAction *action){
     if(action == shutdown_action)
       comboBox->setCurrentIndex(0);
     if(action == reboot_action)
       comboBox->setCurrentIndex(1);
     if(action == suspend_action)
       comboBox->setCurrentIndex(2);
     if(action == hibernate_action)
       comboBox->setCurrentIndex(3);
}

void Gui::showW(){
     if(warnings->isChecked()) //warnings is checked
       showRunningProgram();
}

void Gui::showRunningProgram(){
     center();
     if(isMinimized() || isHidden())
       showNormal();
     else
       activateWindow();
}

void Gui::keyPressEvent(QKeyEvent *kEvent){
     if(kEvent->modifiers() == Qt::ControlModifier){
       if(!timeRunning && !lockAll){
         if(kEvent->key() == Qt::Key_S){
           shutdown_action->setChecked(true);
           comboBox->setCurrentIndex(0);
         }
         if(kEvent->key() == Qt::Key_R){
           reboot_action->setChecked(true);
           comboBox->setCurrentIndex(1);
         }
         if(kEvent->key() == Qt::Key_U){
           suspend_action->setChecked(true);
           comboBox->setCurrentIndex(2);
         }
         if(kEvent->key() == Qt::Key_H){
           hibernate_action->setChecked(true);
           comboBox->setCurrentIndex(3);
         }
       }
       if(!lockAll){
         if(kEvent->key() == Qt::Key_E && timeRunning)
           reset();
         if(kEvent->key() == Qt::Key_L){
           if(log_action->isChecked())
             log_action->setChecked(false);
           else
             log_action->setChecked(true);
         }
         if(kEvent->key() == Qt::Key_P)
           pref->show();
         if(kEvent->key() == Qt::Key_Q)
           qApp->quit();
       }
       if(kEvent->key() == Qt::Key_I){
         setInfoText();
         infoBox->show();
       }
     }
}

void Gui::update(){
     if(finishedExecuted)
       reset();
     Time();  //calculates i or rather the new time every second

     if(warnings->isChecked() && (((i<=100) && (i>95)) || ((i<=40) && (i>35))) && !ti->isActive())
       ti->start(30000);

     QString tip1, tip2, tip3;

     tip1 = ("'qshutdown' - ");
     if(shutdown_action->isChecked())
       tip2 = (tr("shutdown in "));
     if(reboot_action->isChecked())
       tip2 = (tr("reboot in "));
     if(suspend_action->isChecked())
       tip2 = (tr("suspend in "));
     if(hibernate_action->isChecked())
       tip2 = (tr("hibernate in "));

     if((i/60.0)>=60) //Display hours and minutes
       tip3 = (QString::number(i/3600) + tr(" hours + ")
              + QString::number(i/60 - (i/3600)*60) + tr(" minutes"));
     else if(((i/60.0)<60) && (i>=60)) //Display only minutes
       tip3 = (QString::number(i/60) + tr(" minutes"));
     else if(i<60) //Display only seconds
       tip3 = (QString::number(i) + tr(" seconds"));
     setWindowTitle(tip1 + tip2 + tip3);
     TIcon->setToolTip(tip2 + tip3);
     if(i>=60){
       lcdL->setText(tr("minutes"));
       lcd->display(i/60);
     }
     else if(i<60){
       lcdL->setText(tr("seconds"));
       lcd->display(i);
     }

     //this will ensure that the shutdown-type will be executed in case a few seconds were skipped
     if(i<=86390)
       n = 10; //any number to keep i in check
     if((i==0) || ((i>n) && (i>86390)))
       finished(); //execute shutdown-type
}

void Gui::set(){
     timeRunning = true;
     localTime = QTime::currentTime(); //the time now
     if(radio2->isChecked()) //if minute-countdown
       futureTime = localTime.addSecs(spin->value()*60); //add spinbox-value to the current time
     else
       futureTime = timeEdit->time(); //else the future time is the time of the timeEdit
     n = 86400; // any number bigger than i - just for initializing

     if(lockAll)
       warnings->setEnabled(true);

     if(lock->isChecked()){       //when OK-button is clicked and lock is checked
       spin->setDisabled(true);             //deaktivate spinbox
       radio1->setDisabled(true);            //deaktivate radiobutton
       radio2->setDisabled(true);
       lock->setDisabled(true);
       timeEdit->setDisabled(true);
       comboBox->setDisabled(true);
       targetTime->setDisabled(true);
       minutes->setDisabled(true);
       ok->hide();
       minim->show();
       minim->setFocus();
       minim->setAutoDefault(true);         //minimize-button will accept the return key
       power_actions->setDisabled(true);
     }
     showMinimized();
     if(!lockAll)
       reset_action->setVisible(true);
}

void Gui::starter(){ set(); ti->stop(); timer->start(1000); } //stops ti-Qtimer and starts timer-Qtimer

void Gui::Time(){
     localTime = QTime::currentTime();
     i = localTime.secsTo(futureTime); //the difference of the localTime and the future time
     if(i<0)                           //if the date is on the next day
       i = 86400 + localTime.secsTo(futureTime); //add 1 day in seconds
}

void Gui::saveWindowSize(){
   #ifdef Q_OS_WIN32
     QString file(QDesktopServices::storageLocation(QDesktopServices::DataLocation) + "/qshutdown/qshutdown.conf");
     QSettings settings(file, QSettings::IniFormat);
   #else //!Q_OS_WIN32
     QString file(QDir::homePath() + "/.qshutdown/qshutdown.conf");
     QSettings settings(file, QSettings::NativeFormat);
   #endif //Q_OS_WIN32
     settings.setValue("WindowSize/size", size());
}

void Gui::finished(){
     finishedExecuted = true;
     switch(comboBox->currentIndex()){
       case 0:
         saveWindowSize();
         Power::shutdown();
         break;
       case 1:
         saveWindowSize();
         Power::reboot();
         break;
       case 2:
         Power::suspend();
         break;
       case 3:
         Power::hibernate();
         break;
       default:;
     }
}

void Gui::closeEvent(QCloseEvent* window_close){
     hide();
     QWidget::closeEvent(window_close);
}

void Gui::beforeQuit(){
     saveWindowSize();
     if(log_action->isChecked()){ //if logfile is set in the icon contextmenu
     #ifdef Q_OS_WIN32
       QFile logfile(QDesktopServices::storageLocation(QDesktopServices::DataLocation) + "/qshutdown/log.txt");
     #else //!Q_OS_WIN32
       QFile logfile(QDir::homePath() + "/.qshutdown/log.txt");
     #endif //Q_OS_WIN32
       if(!logfile.open(QIODevice::ReadWrite | QIODevice::Text)){
         qCritical("E: Can not open log.txt!");
         return;
       }

/******* writing log to log.txt *******/
       QTextStream out(&logfile);
       while(!logfile.atEnd())
         out.readLine();
       out << "[" << datetime.toString("yyyy.MM.dd hh:mm") << "] "
           << 1/(60000.0/elapsedTime.elapsed()) << " Minutes uptime\n";
       logfile.close();

/******* if log.txt is bigger than set in preferences, delete first line *******/
       bool isBigger;
       if(logfile.size()>logFileSize*1000)
         isBigger = true;
       else
         isBigger = false;
       while(isBigger){
         logfile.open(QIODevice::ReadWrite | QIODevice::Text);
         QTextStream out(&logfile);
         out.readLine();
         QString content = out.readAll();
         logfile.resize(0);
         out << content;
         logfile.close();
         if(logfile.size()<=logFileSize*1000)
           isBigger = false;
       }
     }

  #ifndef Q_OS_WIN32
     QDBusConnection::sessionBus().unregisterObject(OBJECT_NAME, QDBusConnection::UnregisterNode);
     QDBusConnection::sessionBus().unregisterService(SERVICE_NAME);
  #endif //Q_OS_WIN32
}

void Gui::resizeEvent(QResizeEvent* window_resize){
   #ifndef Q_OS_WIN32
     center();
   #endif //Q_OS_WIN32
     QWidget::resizeEvent(window_resize);
}

void Gui::loadSettings(){
/***************** create file and it's entries *****************/
#ifdef Q_OS_WIN32
     QString file(QDesktopServices::storageLocation(QDesktopServices::DataLocation) + "/qshutdown/qshutdown.conf");
     QSettings settings(file, QSettings::IniFormat);
#else //!Q_OS_WIN32
     QString file(QDir::homePath() + "/.qshutdown/qshutdown.conf");
     QSettings settings(file, QSettings::NativeFormat);
#endif //Q_OS_WIN32
     if(!settings.isWritable())
       qCritical() << "W: qshutdown.conf is not writeable!";

     if(!QFile::exists(file) || !settings.contains("Lock_all"))
       settings.setValue("Lock_all", false);
     if(!settings.contains("Time/time_hour"))
       settings.setValue("Time/time_hour", 22);
     if(!settings.contains("Time/time_minute"))
       settings.setValue("Time/time_minute", 00);
     if(!settings.contains("Time/countdown_minutes"))
       settings.setValue("Time/countdown_minutes", 60);
     if(!settings.contains("Time/countdown_at_startup"))
       settings.setValue("Time/countdown_at_startup", false);
     if(!settings.contains("WindowSize/size"))
       settings.setValue("WindowSize/size", QSize(430, 325));
     if(!settings.contains("Fonts/font1"))
       settings.setValue("Fonts/font1", 13);
     if(!settings.contains("Fonts/font2"))
       settings.setValue("Fonts/font2", 18);
     if(!settings.contains("Fonts/font3"))
       settings.setValue("Fonts/font3", 11);
     if(!settings.contains("CheckBoxes/target_time"))
       settings.setValue("CheckBoxes/target_time", false);
     if(!settings.contains("CheckBoxes/countdown"))
       settings.setValue("CheckBoxes/countdown", true);
     if(!settings.contains("heckBoxes/lock"))
       settings.setValue("CheckBoxes/lock", true);
     if(!settings.contains("CheckBoxes/warnings"))
       settings.setValue("CheckBoxes/warnings", true);
     if(!settings.contains("Power/comboBox"))
       settings.setValue("Power/comboBox", 0);
     if(!settings.contains("Logfile/logging"))
       settings.setValue("Logfile/logging", false);
     if(!settings.contains("Logfile/size"))
       settings.setValue("Logfile/size", 1.5);

/***************** read files entries *****************/
     timeEdit->setTime(QTime(settings.value("Time/time_hour").toInt(),settings.value("Time/time_minute").toInt()));
     spin->setValue(settings.value("Time/countdown_minutes").toInt());
     resize(settings.value("WindowSize/size").toSize());
     font1->setPointSize(settings.value("Fonts/font1").toInt());
     font2->setPointSize(settings.value("Fonts/font2").toInt());
     font3->setPointSize(settings.value("Fonts/font3").toInt());
     radio1->setChecked(settings.value("CheckBoxes/target_time").toBool());
     radio2->setChecked(settings.value("CheckBoxes/countdown").toBool());
     lock->setChecked(settings.value("CheckBoxes/lock").toBool());
     comboBox->setCurrentIndex(settings.value("Power/comboBox").toInt());
     warnings->setChecked(settings.value("CheckBoxes/warnings").toBool());
     log_action->setChecked(settings.value("Logfile/logging").toBool());
     logFileSize = settings.value("Logfile/size").toDouble();

     if(settings.value("Lock_all").toBool()){
       lockAll = true;
       reset_action->setDisabled(true);
       pref_action->setDisabled(true);
       lock->setChecked(true);
       lock->setDisabled(true);
       radio1->setDisabled(true);
       radio2->setDisabled(true);
       timeEdit->setDisabled(true);
       spin->setDisabled(true);
       comboBox->setDisabled(true);
       targetTime->setDisabled(true);
       minutes->setDisabled(true);
       quit_action->setDisabled(true);
       log_action->setDisabled(true);
       power_actions->setDisabled(true);
       warnings->setDisabled(true);
     }
     else
       lockAll = false;

     comboBox->setFont(*font1);
     targetTime->setFont(*font1);
     minutes->setFont(*font1);
     spin->setFont(*font1);
     timeEdit->setFont(*font1);
     lcdL->setFont(*font1);
     ok->setFont(*font2);
     minim->setFont(*font2);
     warnings->setFont(*font3);
     lock->setFont(*font3);
     now->setFont(*font3);

     switch(settings.value("Power/comboBox").toInt()){
       case 0:
         targetTime->setText(tr("Shutdown-time:"));
         minutes->setText(tr("Minutes till shutdown:"));
         shutdown_action->setChecked(true);
         break;
       case 1:
         targetTime->setText(tr("Reboot-time:"));
         minutes->setText(tr("Minutes till reboot:"));
         reboot_action->setChecked(true);
         break;
       case 2:
         targetTime->setText(tr("Suspend-time:"));
         minutes->setText(tr("Minutes till suspend:"));
         suspend_action->setChecked(true);
         break;
       case 3:
         targetTime->setText(tr("Hibernate-time:"));
         minutes->setText(tr("Minutes till hibernate:"));
         hibernate_action->setChecked(true);
         break;
       default:;
     }

     if(settings.value("Time/countdown_at_startup").toBool()){
       starter();
       QTimer::singleShot(7000, this, SLOT(close()));
     }
}

void Gui::reset(){
     setWindowTitle("'qshutdown'");
     lcd->display(0);
     TIcon->setToolTip(NULL);
     if(finishedExecuted)
       showNormal();
     timer->stop();
     if(!ti->isActive())
       ti->start(30000);
     timeRunning = false;
     finishedExecuted = false;
     minim->hide();
     ok->show();
     if(!lockAll){
       if(radio1->isChecked())
         timeEdit->setEnabled(true);
       else
         spin->setEnabled(true);
       reset_action->setVisible(false);
       radio1->setEnabled(true);
       radio2->setEnabled(true);
       lock->setEnabled(true);    
       comboBox->setEnabled(true);
       targetTime->setEnabled(true);
       minutes->setEnabled(true);
       power_actions->setEnabled(true);
     }
     else loadSettings();
}
