/***************************************************************************
 *   copyright           : (C) 2002 by Hendrik Sattler                     *
 *   mail                : post@hendrik-sattler.de                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

//own headers
#include <charsets.h>
#include <helper.h>
#include <gtincl.h>

//standard headers
#include <stdio.h>
#include <string.h>
#include <iconv.h>

#include <config.h>
#ifdef WORDS_BIGENDIAN
#  define UCS4_WITH_ENDIANESS "UCS-4BE"
#else
#  define UCS4_WITH_ENDIANESS "UCS-4LE"
#endif

const char* ucs4_get_iconv_charset() {
  static char* ucs4_charset = NULL;
  iconv_t cd;

  if (ucs4_charset == NULL) {
    cd = iconv_open(UCS4_WITH_ENDIANESS,UCS4_WITH_ENDIANESS);
    if (cd == (iconv_t)-1) {
      fprintf(stderr,_("%s: the iconv implementation of this system does not support %s"),_("Error"),UCS4_WITH_ENDIANESS);
      exit(EXIT_FAILURE);
    } else {
      ucs4_charset = UCS4_WITH_ENDIANESS;
    }
    iconv_close(cd);
  }
  return ucs4_charset;
}

size_t ucs4len (const ucs4char_t* input) {
  size_t retval = 0;
  if (input != NULL) while (input[retval] != 0) ++retval;
  return retval;
}

ucs4char_t* ucs4ndup(const ucs4char_t* s, size_t n) {
  ucs4char_t* retval;
  size_t i = 0;

  if (s == NULL) return NULL;
  while (s[i] != 0 && i < n) ++i;
  retval = mem_alloc((i+1)*sizeof(*retval),0);
  (void)memcpy(retval,s,i*sizeof(*s));
  retval[i] = 0;
  return retval;
}
ucs4char_t* ucs4dup(const ucs4char_t* s) {
  return ucs4ndup((ucs4char_t*)s,ucs4len(s)+1);
}

int ucs4ncmp (const ucs4char_t* s1, const ucs4char_t* s2, size_t n) {
  size_t i = 0;

  if (s1 == NULL && s2 == NULL) {
    return 0;
  } else {
    if (s1 == NULL) return -1;
    else if (s2 == NULL) return 1;
    else for (;i < n; ++i) {
      if (s1[i] != s2[i]) {
	if (s1[i] < s2[i]) return -1;
	else return 1;
      }
    }
  }
  return 0;
}
int ucs4cmp (const ucs4char_t* s1, const ucs4char_t* s2) {
  size_t s1len = ucs4len((ucs4char_t*)s1);
  size_t s2len = ucs4len((ucs4char_t*)s2);
  return ucs4ncmp(s1,s2,(s1len >= s2len) ? s1len : s2len);
}

ucs4char_t* ucs4str (const ucs4char_t* haystack,
		     const ucs4char_t* needle)
{
  unsigned int i = 0;
  unsigned int len = ucs4len(needle);

  if (len == 0) return (ucs4char_t*)haystack;
  if (ucs4len(haystack) == 0) return NULL;

  for (; haystack[i] != 0; ++i) {
    if (ucs4ncmp(&(haystack[i]),needle,len) == 0) {
      return (ucs4char_t*)&(haystack[i]);
    }
  }
  return NULL;
}

ucs4char_t* ucs4ncpy (ucs4char_t* dest,
		      const ucs4char_t* src,
		      size_t n) {
  size_t i = 0;
  for (; i<n; ++i) dest[i] = src[i];
  dest[i] = 0;
  return dest;
}
