/***************************************************************************
 *   copyright           : (C) 2004 by Hendrik Sattler                     *
 *   mail                : post@hendrik-sattler.de                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "helper.h"
#include "config.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <strings.h>
#include <ctype.h>
#include <stdlib.h>

#if defined(WINDOWS_API)
#define _WIN32_IE 0x0500 /* IE 5.0 is required, not sure why */
#include <shlobj.h> //link with -lshell32 or -lshfolder
#include <shlwapi.h> //also link with -lshlwapi
#else
#include <pwd.h>
#endif

const char* get_homedir () {
#if defined(WINDOWS_API)
  static char* retval = NULL;
  HRESULT rc;
  
  if (retval == NULL) {
    retval = mem_alloc(MAX_PATH,1);
    rc = SHGetFolderPathA(NULL,CSIDL_APPDATA,NULL,
                          SHGFP_TYPE_CURRENT,retval);
    if (rc == S_FALSE || rc == E_INVALIDARG) {
      mem_realloc(retval,0);
      retval = ".";
    }
  }
  return retval;
#else
  static struct passwd* pwentry = NULL;

  pwentry = getpwuid(getuid());
  if (pwentry != NULL)  return pwentry->pw_dir;
  return getenv("HOME");
#endif
}

char* configfile_get_path (char* filename) {
  char* retval;
  const char* homedir = get_homedir();

  if (str_len(filename) == 0) return NULL;

  if (str_len(homedir) > 0) {
    retval = mem_alloc(strlen(homedir)+2
		       +strlen(PACKAGE_NAME)+1
		       +strlen(filename)+1,1);
#if defined(WINDOWS_API)
    retval = str_dup(homedir);
    retval = mem_realloc(retval,MAX_PATH);
    PathAppendA(retval,PACKAGE_NAME);
    PathAppendA(retval,filename);
#else
    sprintf(retval,"%s%s.%s/%s",
	    homedir,
	    (homedir[strlen(homedir)] == '/') ? "" : "/",
	    PACKAGE_NAME,
	    filename);
#endif
  } else {
    retval = NULL;
  }
  return retval;
}

char** configfile_read (char* filename) {
  char** retval = NULL;
  int entrycount = 2; //not 0 because last entry must be NULL
  int current = 1;
  char* fullpath;
  int status = -1;
  FILE* fd = NULL;
  struct stat fileinfo;
  char* line = NULL;
  char* left = NULL;
  char* right = NULL;
  char* temp = NULL;

  fullpath = configfile_get_path(filename);
  if (fullpath == NULL) return NULL;
  status = open(fullpath,O_RDONLY);
  if (status == -1) return NULL;
  fd = fdopen(status,"r");
  if (fd == NULL ||
      fstat(status,&fileinfo) == -1 ||
      fileinfo.st_size == 0) {
    fclose(fd);
    return NULL;
  }

  retval = mem_realloc(retval,entrycount*sizeof(*retval));
  retval[0] = fullpath;

  line = mem_alloc(fileinfo.st_size+1,1);
  do{
    //read a line from the configuration file
    if (fgets(line,fileinfo.st_size+1,fd) == NULL) break;
    if (*line != '#') {
      //remove end of line characters
      right = strchr(line,'\r');
      if (right != NULL) *right=0;
      right = strchr(line,'\n');
      if (right != NULL) *right=0;

      //skip possible spaces at beginning of line
      left = line;
      while (isspace((int)*left) != 0) ++left;
      if (str_len(left)) {
	++entrycount;
	//find seperator
	right = strchr(left,'=');
	if (right != NULL) {
	  temp = right;
	  //find right part
	  do { ++right; } while (isspace((int)*right) != 0);
	} else {
	  temp = left+strlen(left);
	}
	//find right end of left part
	do { --temp; } while (isspace((int)*temp) != 0);
	*(++temp) = 0;

	retval = mem_realloc(retval,entrycount*sizeof(*retval));
	retval[current] = mem_alloc((2+(temp-left)+1+str_len(right)+1)*sizeof(**retval),1);
	snprintf(retval[current],2+(temp-left)+1+str_len(right)+1,"--%s%s%s",
		 left,(right!=NULL)?"=":"",(right!=NULL)?right:"");
	++current;
      }
    }
  } while (1);
  mem_realloc(line,0);
  retval[current] = NULL;
  fclose(fd);
  return retval;
}
