/***************************************************************************
 *   copyright           : (C) 2002 by Hendrik Sattler                     *
 *   mail                : post@hendrik-sattler.de                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef _GNU_SOURCE
# define _GNU_SOURCE
#endif
#ifndef _REENTRANT
# define _REENTRANT
#endif

#include <charsets.h>
#include <helper.h>
#include <smspdu.h>
#include <timeincl.h>
#include "smscoding.h"
#include "smsudh.h"

#include <stdlib.h>
#include <string.h>
#include <ctype.h>

void sms_pdu_print_item_slot (FILE* fp, struct sms* sms) {
  struct sms_pdu_data* pdu;
  unsigned int i = 0;
  unsigned int p = 1;

  pdu = sms->decoded->pdu;
  if (pdu->parts > 1) p = pdu->parts;
  for (; i < p; ++i) {
    if (sms->encoded[i] != NULL && 
	sms->encoded[i]->slot >= 1) {
      fprintf(fp,"Slot: %d\n", sms->encoded[i]->slot);
    }
  }
}

void sms_pdu_print_item_address (FILE* fp, struct sms* sms) {
  struct sms_pdu_data* pdu;
  char* temp;

  pdu = sms->decoded->pdu;
  if (str_len(pdu->address.digits) > 0 ||
      ucs4len(pdu->address.text) > 0) {
    switch(pdu->options.type) {
    default:
    case SMS_TYPE_SUBMIT: temp = "To"; break;
    case SMS_TYPE_DELIVER: temp =  "From"; break;
    case SMS_TYPE_STATUS_REPORT: temp = "Original To"; break;
    }
    fprintf(fp,"%s: ",temp);
    if (str_len(pdu->address.digits) > 0) {
      fprintf(fp,"%s%s ",
	      (((pdu->address.type&SMS_NUMBER_TYPEMASK) == SMS_NUMTYPE_INTERNAT) ? "+" : ""),
	      pdu->address.digits);
    }
    if (ucs4len(pdu->address.text)) {
      temp = convert_to_system(pdu->address.text,REPMODE_QUESTIONMARK);
      fprintf(fp,"(%s)",temp);
      mem_realloc(temp,0);      
    }
    fprintf(fp,"\n");
  }
}

void sms_pdu_print_item_time (FILE* fp, enum sms_pdu_type type,
			      struct sms_pdu_time* t)
{
  unsigned short weeks;
  unsigned short days;
  unsigned short hours;
  unsigned short minutes;
  unsigned short seconds;
  struct tm sct_tm;
  char* field;
  char temp[81];

  switch (t->format) {
  case SMS_PDU_TIME_NONE:
    break;
  case SMS_PDU_TIME_RELATIVE:
/*     fprintf(fp,"Valid for: %ld seconds\n",t->value); */
    weeks = t->value / (7*24*3600);
    days = (t->value / (24*3600))%7;
    hours = (t->value / 3600)%24;
    minutes = (t->value / 60)%60;
    seconds = t->value%60;
    fprintf(fp,"Valid for:");
    if (weeks) fprintf(fp," %d %s", weeks, (weeks > 1) ? "weeks" : "week");
    if (days) fprintf(fp," %d %s", days, (days > 1) ? "days" : "day");
    if (hours) fprintf(fp," %d %s", hours, (hours > 1) ? "hours" : "hour");
    if (minutes) fprintf(fp," %d %s", minutes, (minutes > 1) ? "minutes" : "minute");
    if (seconds) fprintf(fp," %d %s", seconds, (seconds > 1) ? "seconds" : "second");
    fprintf(fp,"\n");
    break;
  case SMS_PDU_TIME_ABSOLUTE:
    field = NULL;
    memset(&sct_tm,0,sizeof(sct_tm));
    if (localtime_r(&(t->value),&sct_tm) != NULL) {
      if (strftime(temp,sizeof(temp)-1,"%c",&sct_tm) > 0) {
	switch(type) {
	case SMS_TYPE_SUBMIT: field = "Valid until"; break;
	case SMS_TYPE_DELIVER: field =  "Date"; break;
	case SMS_TYPE_STATUS_REPORT: field = "Message status time"; break;
	default: break;
	}
	fprintf(fp,"%s: %s\n",field,temp);
      }
    }
    break;
  }
}

void sms_pdu_print_item_smsc (FILE* fp, struct sms* sms) {
  struct sms_pdu_data* pdu;

  pdu = sms->decoded->pdu;
  if (strlen(sms->decoded->sca.digits)) {
    fprintf(fp,"SMSC number: %s%s\n",
	    (((sms->decoded->sca.type&SMS_NUMBER_TYPEMASK) == SMS_NUMTYPE_INTERNAT) ? "+" : ""),
	    sms->decoded->sca.digits);
  }
}

char* sms_pdu_print_get_type_string (enum sms_pdu_type t) {
  switch (t) {
  case SMS_TYPE_DELIVER: return "SMS-DELIVER";
  case SMS_TYPE_SUBMIT: return "SMS-SUBMIT";
  case SMS_TYPE_STATUS_REPORT: return "SMS-STATUS-REPORT";
  case SMS_TYPE_SUBMIT_REPORT: return "SMS-SUBMIT-REPORT";
  case SMS_TYPE_DELIVER_REPORT: return "SMS-DELIVER-REPORT";
  case SMS_TYPE_COMMAND: return "SMS-COMMAND";
  }
  return "";
}

void sms_pdu_print_item_options (FILE* fp, struct sms* sms) {
  struct sms_pdu_data* pdu;

  pdu = sms->decoded->pdu;
  fprintf(fp,"PDU type: %s\n",sms_pdu_print_get_type_string(pdu->options.type));
  if (pdu->options.mms || pdu->options.rd ||
      pdu->options.rp || pdu->options.sr) {
    fprintf(fp,"PDU flags:");
    if (pdu->options.rp) fprintf(fp," ReplyPath");
    if (pdu->options.sr) fprintf(fp," StatusRequest");
    if (pdu->options.mms) fprintf(fp," MoreMessagesToSend");
    if (pdu->options.rd) fprintf(fp," RejectDuplicate");
    fprintf(fp,"\n");
  }
}

void sms_pdu_print_item_pid (FILE* fp, struct sms* sms) {
  struct sms_pdu_data* pdu;

  pdu = sms->decoded->pdu;
  if (pdu->pid > 0 && (pdu->pid&0xC0) != 0x80) {
    fprintf(fp,"Protocol ID: 0x%02x (",pdu->pid);
    switch (pdu->pid&0xC0) {
    case 0xC0:
      fprintf(fp,"service-center-specific use");
      break;
    case 0x40:
      if (0 < (pdu->pid&0x3F) && (pdu->pid&0x3F) <= 9) {
	fprintf(fp,"replace short message type %d",pdu->pid&0x3F);
      }
      switch (pdu->pid&0x3F) {
      case 0x00: fprintf(fp,"short message type 0"); break;
      case 0x1e: fprintf(fp,"obsolete EMS mark"); break;
      case 0x1f: fprintf(fp,"return call message"); break;
      case 0x3c: fprintf(fp,"ANSI-136 R-DATA"); break;
      case 0x3d: fprintf(fp,"ME data download"); break;
      case 0x3e: fprintf(fp,"ME de-personalization short message"); break;
      case 0x3f: fprintf(fp,"(U)SIM data download"); break;
      default: break;
      }
      break;
    case 0x00:
      if ((pdu->pid&0x20) == 0) {
	fprintf(fp,"SMS-to-SME protocol");
      } else {
	fprintf(fp,"telematic interworking");
      }
      break;
    default:
    case 0x80:
       //reserved value, ignored with if-statement above
      break;
    }
    fprintf(fp,")\n");
  }
}

void sms_pdu_print_item_dcs (FILE* fp, struct sms* sms) {
  struct sms_pdu_data* pdu;

  pdu = sms->decoded->pdu;
  fprintf(fp,"Data coding scheme:");
  if (pdu->scheme.compressed) fprintf(fp," compressed");
  switch (pdu->scheme.encoding) {
  case SMS_CHARSET_GSM: fprintf(fp," 7bit-GSM"); break;
  case SMS_CHARSET_UCS2: fprintf(fp," UCS-2"); break;
  case SMS_CHARSET_8BIT: fprintf(fp," 8bit"); break;
  }
  if (pdu->scheme.options == SMS_DCS_OPT_CLASS &&
      pdu->scheme.class <= 3) {
    fprintf(fp," (class %d)",pdu->scheme.class);
  }
  if (pdu->scheme.autodel) fprintf(fp," marked as auto-delete");
  if (pdu->scheme.options == SMS_DCS_OPT_IND) {
    fprintf(fp,"\nIndication: %s", (pdu->scheme.indsense) ? "new" : "no more");
    switch(pdu->scheme.indtype) {
    case SMS_DCS_IND_VOICE: fprintf(fp," voicemail"); break;
    case SMS_DCS_IND_FAX: fprintf(fp," fax"); break;
    case SMS_DCS_IND_EMAIL: fprintf(fp," e-mail"); break;
    case SMS_DCS_IND_OTHER: fprintf(fp," misc."); break;
    }
    fprintf(fp," message(s) waiting");
  }
  fprintf(fp,"\n");
}

void sms_pdu_print_item_text (FILE* fp, struct sms* sms) {
  struct sms_pdu_data* pdu;
  uint16_t textlen = 0;
  char* temp;
  unsigned int i;

  pdu = sms->decoded->pdu;
  if (pdu->ud == NULL) return;
  for (i = 0; i < pdu->parts; ++i) {
    textlen += ucs4len(pdu->ud[i].text);
  }
  fprintf(fp,"Message length: %d\n\n",textlen);
  if (textlen == 0) return;

  for (i = 0; i < pdu->parts; ++i) {
    if (pdu->ud[i].text == NULL) {
      fprintf(fp,"[...]");
    } else {
      temp = convert_to_system(pdu->ud[i].text,REPMODE_QUESTIONMARK);
      fprintf(fp,"%s",temp);
      mem_realloc(temp,0);
    }
  }
  fprintf(fp,"\n");
}

void sms_pdu_print_item_status (FILE* fp, struct sms_pdu_message_status* m) {
  char rej[] = "service rejected";
  char* temp;
  char* temp2;

  if (m == NULL) return;
  switch (m->s&0xd0) {
  case 0x00:
  case 0x40:
    temp2 = "transaction completed"; break;
  case 0x20:
  case 0x60:
  default:
    temp2 = "SC still trying to transfer short message"; break;
  }
  switch (m->s) {
  case 0x00: temp = "received by SME"; break;
  case 0x01: temp = "forwarded to SME but delivery cannot be confirmed"; break;
  case 0x02: temp = "replaced"; break;
  case 0x20:
  case 0x60: temp = "congestion"; break; 
  case 0x21:
  case 0x61: temp = "SME busy"; break; 
  case 0x22:
  case 0x62: temp = "no response from SME"; break; 
  case 0x23:
  case 0x63: temp = "service rejected"; break; 
  case 0x24: 
  case 0x44:
  case 0x64: temp = "QoS not available"; break; 
  case 0x25:
  case 0x65: temp = "error in SME"; break; 
  case 0x40: temp = "remote procedure error"; break;
  case 0x41: temp = "incompatible destination"; break;
  case 0x42: temp = "connection rejected by SME"; break;
  case 0x43: temp = "not obtainable"; break;
  case 0x45: temp = "no interworking available"; break;
  case 0x46: temp = "validity period expired"; break;
  case 0x47: temp = "deleted by sender"; break;
  case 0x48: temp = "deleted by service center"; break;
  case 0x49: temp = "message does not exist"; break;
  default:
    temp = rej;
  }
  fprintf(fp,"Message status: %s\n",temp2);
  fprintf(fp,"Message status reason: %s\n",temp);
  sms_pdu_print_item_time(fp,SMS_TYPE_STATUS_REPORT,&m->t);
}

void sms_pdu_print_item_mr (FILE* fp, struct sms* sms) {
  struct sms_pdu_data* pdu;

  pdu = sms->decoded->pdu;
  if (pdu->options.type == SMS_TYPE_SUBMIT ||
      pdu->options.type == SMS_TYPE_STATUS_REPORT) {
    fprintf(fp,"Message reference: %d\n",pdu->mref);
    if (pdu->options.type == SMS_TYPE_SUBMIT &&
	pdu->ud != NULL) {
      sms_pdu_print_item_status(fp,pdu->ud[pdu->partnum].ack);
    } else if (pdu->options.type == SMS_TYPE_STATUS_REPORT) {
      sms_pdu_print_item_status(fp,pdu->status);
    }
  }
}

void sms_pdu_print_deliver (FILE* fp, struct sms* sms) {
  struct sms_pdu_data* pdu;

  if (fp == NULL || sms == NULL ||
      sms->decoded == NULL || sms->decoded->pdu == NULL) {
    return;
  }
  pdu = sms->decoded->pdu;
  sms_pdu_print_item_slot(fp,sms);
  sms_pdu_print_item_address(fp,sms);
  sms_pdu_print_item_time(fp,pdu->options.type,&pdu->timedata);
  sms_pdu_print_item_smsc(fp,sms);
  sms_pdu_print_item_mr(fp,sms);
  sms_pdu_print_item_options(fp,sms);
  sms_pdu_print_item_pid(fp,sms);
  sms_pdu_print_item_dcs(fp,sms);
  sms_udh_print(fp,sms->decoded->pdu->ud,
		sms->decoded->pdu->parts);
  sms_pdu_print_item_text(fp,sms);
  fprintf(fp,"\n");
}

void sms_pdu_print_statusreport (FILE* fp, struct sms* sms) {
  struct sms_pdu_data* pdu;

  if (fp == NULL || sms == NULL ||
      sms->decoded == NULL || sms->decoded->pdu == NULL) {
    return;
  }
  pdu = sms->decoded->pdu;
  if (pdu->status != NULL &&
      pdu->status->message_parent == NULL) {
    sms_pdu_print_item_slot(fp,sms);
    sms_pdu_print_item_address(fp,sms);
    sms_pdu_print_item_time(fp,SMS_TYPE_DELIVER,&pdu->timedata);
    sms_pdu_print_item_smsc(fp,sms);
    sms_pdu_print_item_mr(fp,sms);
    sms_pdu_print_item_options(fp,sms);
    sms_pdu_print_item_pid(fp,sms);
    if (pdu->scheme.dcs_present) {
      sms_pdu_print_item_dcs(fp,sms);
    }
    sms_udh_print(fp,sms->decoded->pdu->ud,
		  sms->decoded->pdu->parts);
    sms_pdu_print_item_text(fp,sms);
    fprintf(fp,"\n");
  }
}

void sms_pdu_print (FILE* fp, struct sms* sms) {
  if (fp == NULL || sms == NULL ||
      sms->decoded == NULL || sms->decoded->pdu == NULL) {
    return;
  }
  switch(sms->decoded->pdu->options.type) {
  case SMS_TYPE_DELIVER:
  case SMS_TYPE_SUBMIT:
    sms_pdu_print_deliver(fp,sms);
    break;
  case SMS_TYPE_STATUS_REPORT:
    sms_pdu_print_statusreport(fp,sms);
    break;
  case SMS_TYPE_SUBMIT_REPORT:
  case SMS_TYPE_DELIVER_REPORT:
  case SMS_TYPE_COMMAND:
    fprintf(stderr,"Unsupported pdu type: %s\n\n",
	    sms_pdu_print_get_type_string(sms->decoded->pdu->options.type));
    break;
  }
}
