/*
 * -------------------------------------------------------------------------
 * dwt_int.c -- DWT interface function
 * SWT - Scilab wavelet toolbox
 * Copyright (C) 2005-2006  Roger Liu
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * -------------------------------------------------------------------------
 */

#include "swt_common.h"
#include "dwt.h"
#include "stack-c.h"

/*-------------------------------------------
 * convolution
 *-----------------------------------------*/



int
int_conv (char *fname)
{
  static int l1, m1, n1, l2, m2, n2, l3, m3, n3;
  static int minlhs = 1, maxlhs = 1, minrhs = 2, maxrhs = 2;
  int errCode;
  /* Input Validation */
  CheckRhs (minrhs, maxrhs);
  CheckLhs (minlhs, maxlhs);
  /* Get Input */
  GetRhsVar (1, "d", &m1, &n1, &l1);
  GetRhsVar (2, "d", &m2, &n2, &l2);
  conv_validate (&errCode);
  if (errCode != SUCCESS)
    {
      validate_print (errCode);	/* Indicate error  */
      return 0;			/* Stop when validation fails */
    }
  /* Memory Management */
  m3 = 1;
  n3 = m1 * n1 + m2 * n2 - 1;
  CreateVar (3, "d", &m3, &n3, &l3);
  /* Actual Processing */
  conv (stk (l1), n1 * m1, stk (l3), n3, stk (l2), n2 * m2);
  /* Return Value */
  LhsVar (1) = 3;
  return 0;
}



/*-------------------------------------------
 * wrev
 *-----------------------------------------*/

int
int_wrev (char *fname)
{
  static int l1, m1, n1, l2, m2, n2;
  static int minlhs = 1, maxlhs = 1, minrhs = 1, maxrhs = 1;
  int errCode;

  CheckRhs (minrhs, maxrhs);
  CheckLhs (minlhs, maxlhs);

  wrev_validate (&errCode);
  if (errCode != SUCCESS)
    {
      validate_print (errCode);
      return 0;			
    }
  GetRhsVar (1, "d", &m1, &n1, &l1);

  m2 = m1;
  n2 = n1;
  CreateVar (2, "d", &m2, &n2, &l2);

  wrev (stk (l1), n1 * m1, stk (l2), n1 * m1);

  LhsVar (1) = 2;
  return 0;
}

/*-------------------------------------------
 * qmf
 *-----------------------------------------*/

int
int_qmf (char *fname)
{
  static int l1, m1, n1, l2, m2, n2, l3, m3, n3;
  static int minlhs = 1, maxlhs = 1, minrhs = 1, maxrhs = 2;
  int errCode, flow;
  /* Input Validation  */
  CheckRhs (minrhs, maxrhs);
  CheckLhs (minlhs, maxlhs);
  
  qmf_validate (&flow, &errCode);
  if (errCode != SUCCESS)
    {
      validate_print (errCode);	
      return 0;			
    }

  GetRhsVar (1, "d", &m1, &n1, &l1);

  switch (flow){
  case 1:
    {
      m2 = m1;
      n2 = n1;
      CreateVar (2, "d", &m2, &n2, &l2);
      qmf_even (stk (l1), n1 * m1, stk (l2), n1 * m1);
      LhsVar (1) = 2;
      break;
    }
  case 2:
    {
      GetRhsVar (2, "i", &m2, &n2, &l2);
      m3 = m1;
      n3 = n1;
      CreateVar (3, "d", &m3, &n3, &l3);
      if ((istk(l2)[0] % 2) == 0) /* even */
	qmf_even (stk (l1), n1 * m1, stk (l3), n1 * m1);
      else
	qmf_odd (stk (l1), n1 * m1, stk (l3), n1 * m1);
      LhsVar (1) = 3;
      break;
    }
  default:
    break;
  }
  return 0;
}


/*-------------------------------------------
 * dyaddown
 *-----------------------------------------*/

int
int_dyaddown (char *fname)
{
  static int l1, m1, n1, l2, m2, n2, l3, m3, n3, l4, m4, n4;
  static int minlhs = 1, maxlhs = 1, minrhs = 1, maxrhs = 3;
  int flow, errCode;
  /* Input Validation */
  CheckRhs (minrhs, maxrhs);
  CheckLhs (minlhs, maxlhs);

  flow = 0;
  dyaddown_form_validate (&flow, &errCode);
  if (errCode != SUCCESS)
    {
      validate_print (errCode);
      return 0;			
    }

  switch (flow){
  case 1:
    {
      GetRhsVar (1, "d", &m1, &n1, &l1);
      if (m1 > 1)
	{
	  m2 = m1 / 2;// + (m1%2==0)?0:1;
	  n2 = 1;
	}
      else
	{
	  m2 = 1;
	  n2 = n1 / 2;// + (n1%2==0)?0:1;
	}
      CreateVar (2, "d", &m2, &n2, &l2);
      dyaddown_1D_keep_even (stk(l1), n1 * m1, stk(l2), n2 * m2);
      LhsVar (1) = 2;
      break;
    }
  case 2:
    {
      GetRhsVar (1, "d", &m1, &n1, &l1);
      GetRhsVar (2, "i", &m2, &n2, &l2);
    //  if (m1 > 1)
	//{
	 // m3 = m1 / 2;
	 // n3 = 1;
	//}
    //  else
	//{
	//  m3 = 1;
	//  n3 = n1 / 2;
	//}
    //  CreateVar (3, "d", &m3, &n3, &l3);
      if (istk(l2)[0] % 2 == 0)
	  {
		if (m1 > 1)
		{
		 m3 = m1 / 2;
		 n3 = 1;
		}
		else
		{
		  m3 = 1;
		  n3 = n1 / 2;
		}
		CreateVar (3, "d", &m3, &n3, &l3);
		dyaddown_1D_keep_even (stk(l1), n1 * m1, stk(l3), n3 * m3);
	  }
      else
	  {
		if (m1 > 1)
		{
			m3 = m1 / 2;// + ((m1%2)==0)?0:1;
			n3 = 1;
			if (m1 % 2 != 0)
				m3 += 1;
		}
		else
		{
		  m3 = 1;
		  n3 = n1 / 2;// + ((n1%2)==0)?0:1;
		  if (n1 % 2 != 0)
			  n3 += 1;
		}
		//sprintf("%d\n",m3);
		//sprintf("%d\n",n3);
		CreateVar (3, "d", &m3, &n3, &l3);
		dyaddown_1D_keep_odd (stk(l1), n1 * m1, stk(l3), n3 * m3);
	  }
      LhsVar (1) = 3;
      break;
    }
  case 3:
    {
      GetRhsVar (1, "d", &m1, &n1, &l1);
      //m2 = m1 / 2;
	  m2 = m1;
      n2 = n1 / 2;
      CreateVar (2, "d", &m2, &n2, &l2);
      //dyaddown_2D_keep_even (stk(l1), m1, n1, stk(l2), m2, n2); 
	  dyaddown_2D_keep_even_col(stk(l1), m1, n1, stk(l2), m2, n2);
      LhsVar (1) = 2;
      break;
    }
  case 4:
    {
      GetRhsVar (1, "d", &m1, &n1, &l1);
      GetRhsVar (2, "c", &m2, &n2, &l2);
      //printf("%c\n",cstk(l2)[0]);
      dyaddown_content_validate(cstk(l2), &errCode);
      if (errCode != SUCCESS)
	{
	  validate_print (errCode);
	  return 0;	
	}
      if (strcmp(cstk(l2),"r")==0)
	{
	  m3 = m1 / 2;
	  n3 = n1;
	  CreateVar (3, "d", &m3, &n3, &l3);
	  dyaddown_2D_keep_even_row(stk(l1), m1, n1, 
				    stk(l3), m3, n3); 
	}
      if (strcmp(cstk(l2),"c")==0)
	{
	  m3 = m1;
	  n3 = n1 / 2;
	  CreateVar (3, "d", &m3, &n3, &l3);
	  dyaddown_2D_keep_even_col(stk(l1), m1, n1, 
				    stk(l3), m3, n3); 
	}
      if (strcmp(cstk(l2),"m")==0)
	{
	  m3 = m1 / 2;
	  n3 = n1 / 2;
	  CreateVar (3, "d", &m3, &n3, &l3);
	  dyaddown_2D_keep_even(stk(l1), m1, n1, stk(l3), m3, n3); 
	}
      LhsVar (1) = 3;
      break;
    }
  case 5:
    {
      GetRhsVar (1, "d", &m1, &n1, &l1);
      GetRhsVar (2, "i", &m2, &n2, &l2);
      //m3 = m1 / 2;
	  
      if (istk(l2)[0]%2==0)
	  {
 	    m3 = m1;
        n3 = n1 / 2;
        CreateVar (3, "d", &m3, &n3, &l3);
		dyaddown_2D_keep_even_col(stk(l1), m1, n1, stk(l3), m3, n3); 
	  }
      else
	  {
        m3 = m1;
		n3 = n1 /2;
		if (n1 % 2 != 0)
			n3 += 1;
        CreateVar (3, "d", &m3, &n3, &l3);
	    dyaddown_2D_keep_odd_col(stk(l1), m1, n1, stk(l3), m3, n3); 
	  }
      LhsVar (1) = 3;
      break;
    }
  case 6:
    {
      GetRhsVar (1, "d", &m1, &n1, &l1);
      GetRhsVar (2, "i", &m2, &n2, &l2);
      GetRhsVar (3, "c", &m3, &n3, &l3);
      dyaddown_content_validate(cstk(l3), &errCode);
      if (errCode != SUCCESS)
	{
	  validate_print (errCode);
	  return 0;	
	}
      if ((istk(l2)[0])%2==0)
	{
	  if (strcmp(cstk(l3),"r")==0)
	    {
	      m4 = m1 / 2;
	      n4 = n1;
	      CreateVar (4, "d", &m4, &n4, &l4);
	      dyaddown_2D_keep_even_row(stk(l1), m1, n1, 
					stk(l4), m4, n4); 
	    }
	  if (strcmp(cstk(l3),"c")==0)
	    {
	      m4 = m1;
	      n4 = n1 / 2;
	      CreateVar (4, "d", &m4, &n4, &l4);
	      dyaddown_2D_keep_even_col(stk(l1), m1, n1, 
					stk(l4), m4, n4); 
	    }
	  if (strcmp(cstk(l3),"m")==0)
	    {
	      m4 = m1 / 2;
	      n4 = n1 / 2;
	      CreateVar (4, "d", &m4, &n4, &l4);
	      dyaddown_2D_keep_even(stk(l1), m1, n1, 
				    stk(l4), m4, n4); 
	    }
	}
      else
	{
	  if (strcmp(cstk(l3),"r")==0)
	    {
	      m4 = m1 / 2;
	      n4 = n1;
		  if (m1 % 2 != 0)
			  m4 += 1;
	      CreateVar (4, "d", &m4, &n4, &l4);
	      dyaddown_2D_keep_odd_row(stk(l1), m1, n1, 
				       stk(l4), m4, n4); 
	    }
	  if (strcmp(cstk(l3),"c")==0)
	    {
	      m4 = m1;
	      n4 = n1 / 2;
		  if (n1 % 2 != 0)
			  n4 += 1;
	      CreateVar (4, "d", &m4, &n4, &l4);
	      dyaddown_2D_keep_odd_col(stk(l1), m1, n1, 
				       stk(l4), m4, n4); 
	    }
	  if (strcmp(cstk(l3),"m")==0)
	    {
	      m4 = m1 / 2;
	      n4 = n1 / 2;
		  if (m1 % 2 != 0)
			  m4 += 1;
		  if (n1 % 2 != 0)
			  n4 += 1;
	      CreateVar (4, "d", &m4, &n4, &l4);
	      dyaddown_2D_keep_odd(stk(l1), m1, n1, 
				   stk(l4), m4, n4); 
	    }
	}
      LhsVar (1) = 4;
      break;
    }
  case 7:
    {
      GetRhsVar (1, "d", &m1, &n1, &l1);
      GetRhsVar (2, "c", &m2, &n2, &l2);
      GetRhsVar (3, "i", &m3, &n3, &l3);
      dyaddown_content_validate(cstk(l2), &errCode);
      if (errCode != SUCCESS)
	{
	  validate_print (errCode);
	  return 0;	
	}
      if ((istk(l3)[0])%2==0)
	{
	  if (strcmp(cstk(l2),"r")==0)
	    {
	      m4 = m1 / 2;
	      n4 = n1;
	      CreateVar (4, "d", &m4, &n4, &l4);
	      dyaddown_2D_keep_even_row(stk(l1), m1, n1, 
					stk(l4), m4, n4); 
	    }
	  if (strcmp(cstk(l2),"c")==0)
	    {
	      m4 = m1;
	      n4 = n1 / 2;
	      CreateVar (4, "d", &m4, &n4, &l4);
	      dyaddown_2D_keep_even_col(stk(l1), m1, n1, 
					stk(l4), m4, n4); 
	    }
	  if (strcmp(cstk(l2),"m")==0)
	    {
	      m4 = m1 / 2;
	      n4 = n1 / 2;
	      CreateVar (4, "d", &m4, &n4, &l4);
	      dyaddown_2D_keep_even(stk(l1), m1, n1, 
				    stk(l4), m4, n4); 
	    }
	}
      else
	{
	  if (strcmp(cstk(l2),"r")==0)
	    {
	      m4 = m1 / 2;
	      n4 = n1;
		  if (m1 % 2 != 0)
			  m4 += 1;
	      CreateVar (4, "d", &m4, &n4, &l4);
	      dyaddown_2D_keep_odd_row(stk(l1), m1, n1, 
				       stk(l4), m4, n4); 
	    }
	  if (strcmp(cstk(l2),"c")==0)
	    {
	      m4 = m1;
	      n4 = n1 / 2;
		  if (n1 % 2 != 0)
			  n4 += 1;
	      CreateVar (4, "d", &m4, &n4, &l4);
	      dyaddown_2D_keep_odd_col(stk(l1), m1, n1, 
				       stk(l4), m4, n4); 
	    }
	  if (strcmp(cstk(l2),"m")==0)
	    {
	      m4 = m1 / 2;
	      n4 = n1 / 2;
		  if (m1 % 2 != 0)
			  m4 += 1;
		  if (n1 % 2 != 0)
			  n4 += 1;
	      CreateVar (4, "d", &m4, &n4, &l4);
	      dyaddown_2D_keep_odd(stk(l1), m1, n1, 
				   stk(l4), m4, n4); 
	    }
	}
      LhsVar (1) = 4;
      break;
    }
  default:
    break;
  }
  return 0;
}

/*-------------------------------------------
 * dyadup
 *-----------------------------------------*/

int
int_dyadup (char *fname)
{
  static int l1, m1, n1, l2, m2, n2, l3, m3, n3, l4, m4, n4;
  static int minlhs = 1, maxlhs = 1, minrhs = 1, maxrhs = 3;
  int flow, errCode;
  /* Input Validation */
  CheckRhs (minrhs, maxrhs);
  CheckLhs (minlhs, maxlhs);

  flow = 0;
  dyadup_form_validate (&flow, &errCode);
  if (errCode != SUCCESS)
    {
      validate_print (errCode);
      return 0;			
    }

  switch (flow){
  case 1:
    {
      GetRhsVar (1, "d", &m1, &n1, &l1);
      if (m1 > 1)
	{
	  m2 = m1 * 2 + 1;
	  n2 = 1;
	}
      else
	{
	  m2 = 1;
	  n2 = n1 * 2 + 1;
	}
      CreateVar (2, "d", &m2, &n2, &l2);
      dyadup_1D_feed_even (stk(l1), n1 * m1, stk(l2), n2 * m2);
      LhsVar (1) = 2;
      break;
    }
  case 2:
    {
      GetRhsVar (1, "d", &m1, &n1, &l1);
      GetRhsVar (2, "i", &m2, &n2, &l2);
      if (istk(l2)[0] % 2 == 0)
	{
	  if (m1 > 1)
	    {
	      m3 = m1 * 2 - 1;
	      n3 = 1;
	    }
	  else
	    {
	      m3 = 1;
	      n3 = n1 * 2 - 1;
	    }
	  CreateVar (3, "d", &m3, &n3, &l3);
	  dyadup_1D_feed_odd (stk(l1), n1 * m1, stk(l3), n3 * m3);
	}
      else
	{
	  if (m1 > 1)
	    {
	      m3 = m1 * 2 + 1;
	      n3 = 1;
	    }
	  else
	    {
	      m3 = 1;
	      n3 = n1 * 2 + 1;
	    }
	  CreateVar (3, "d", &m3, &n3, &l3);
	  dyadup_1D_feed_even (stk(l1), n1 * m1, stk(l3), n3 * m3);
	}
      LhsVar (1) = 3;
      break;
    }
  case 3:
    {
      GetRhsVar (1, "d", &m1, &n1, &l1);
      //m2 = m1 * 2 + 1;
      //n2 = n1 * 2 + 1;
	  m2 = m1;
	  n2 = n1 * 2 + 1;
      CreateVar (2, "d", &m2, &n2, &l2);
	  dyadup_2D_feed_even_col (stk(l1), m1, n1, stk(l2), m2, n2);
      //dyadup_2D_feed_even (stk(l1), m1, n1, stk(l2), m2, n2); 
      LhsVar (1) = 2;
      break;
    }
  case 4:
    {
      GetRhsVar (1, "d", &m1, &n1, &l1);
      GetRhsVar (2, "c", &m2, &n2, &l2);
      dyadup_content_validate(cstk(l2), &errCode);
      if (errCode != SUCCESS)
	{
	  validate_print (errCode);
	  return 0;	
	}
      if (strcmp(cstk(l2),"r")==0)
	{
	  m3 = m1 * 2 + 1;
	  n3 = n1;
	  CreateVar (3, "d", &m3, &n3, &l3);
	  dyadup_2D_feed_even_row(stk(l1), m1, n1, 
				  stk(l3), m3, n3); 
	}
      if (strcmp(cstk(l2),"c")==0)
	{
	  m3 = m1;
	  n3 = n1 * 2 + 1;
	  CreateVar (3, "d", &m3, &n3, &l3);
	  dyadup_2D_feed_even_col(stk(l1), m1, n1, 
				    stk(l3), m3, n3); 
	}
      if (strcmp(cstk(l2),"m")==0)
	{
	  m3 = m1 * 2 + 1;
	  n3 = n1 * 2 + 1;
	  CreateVar (3, "d", &m3, &n3, &l3);
	  dyadup_2D_feed_even(stk(l1), m1, n1, stk(l3), m3, n3); 
	}
      LhsVar (1) = 3;
      break;
    }
  case 5:
    {
      GetRhsVar (1, "d", &m1, &n1, &l1);
      GetRhsVar (2, "i", &m2, &n2, &l2);
      if (istk(l2)[0]%2==0)
	{
	  //m3 = m1 * 2 - 1;
	  //n3 = n1 * 2 - 1;
	  m3 = m1;
	  n3 = n1 * 2 - 1;
	  CreateVar (3, "d", &m3, &n3, &l3);
	  //dyadup_2D_feed_odd(stk(l1), m1, n1, stk(l3), m3, n3); 
	  dyadup_2D_feed_odd_col(stk(l1), m1, n1, stk(l3), m3, n3);
	}
      else
	{
	  //m3 = m1 * 2 + 1;
	  //n3 = n1 * 2 + 1;
	  m3 = m1;
	  n3 = n1 * 2 + 1;
	  CreateVar (3, "d", &m3, &n3, &l3);
	  //dyadup_2D_feed_even(stk(l1), m1, n1, stk(l3), m3, n3); 
	  dyadup_2D_feed_even_col(stk(l1), m1, n1, stk(l3), m3, n3);
	}
      LhsVar (1) = 3;
      break;
    }
  case 6:
    {
      GetRhsVar (1, "d", &m1, &n1, &l1);
      GetRhsVar (2, "i", &m2, &n2, &l2);
      GetRhsVar (3, "c", &m3, &n3, &l3);
      dyadup_content_validate(cstk(l3), &errCode);
      if (errCode != SUCCESS)
	{
	  validate_print (errCode);
	  return 0;	
	}
      if ((istk(l2)[0])%2==0)
	{
	  if (strcmp(cstk(l3),"r")==0)
	    {
	      m4 = m1 * 2 - 1;
	      n4 = n1;
	      CreateVar (4, "d", &m4, &n4, &l4);
	      dyadup_2D_feed_odd_row(stk(l1), m1, n1, 
					stk(l4), m4, n4); 
	    }
	  if (strcmp(cstk(l3),"c")==0)
	    {
	      m4 = m1;
	      n4 = n1 * 2 - 1;
	      CreateVar (4, "d", &m4, &n4, &l4);
	      dyadup_2D_feed_odd_col(stk(l1), m1, n1, 
					stk(l4), m4, n4); 
	    }
	  if (strcmp(cstk(l3),"m")==0)
	    {
	      m4 = m1 * 2 - 1;
	      n4 = n1 * 2 - 1;
	      CreateVar (4, "d", &m4, &n4, &l4);
	      dyadup_2D_feed_odd(stk(l1), m1, n1, 
				    stk(l4), m4, n4); 
	    }
	}
      else
	{
	  if (strcmp(cstk(l3),"r")==0)
	    {
	      m4 = m1 * 2 + 1;
	      n4 = n1;
	      CreateVar (4, "d", &m4, &n4, &l4);
	      dyadup_2D_feed_even_row(stk(l1), m1, n1, 
				       stk(l4), m4, n4); 
	    }
	  if (strcmp(cstk(l3),"c")==0)
	    {
	      m4 = m1;
	      n4 = n1 * 2 + 1;
	      CreateVar (4, "d", &m4, &n4, &l4);
	      dyadup_2D_feed_even_col(stk(l1), m1, n1, 
				       stk(l4), m4, n4); 
	    }
	  if (strcmp(cstk(l3),"m")==0)
	    {
	      m4 = m1 * 2 + 1;
	      n4 = n1 * 2 + 1;
	      CreateVar (4, "d", &m4, &n4, &l4);
	      dyadup_2D_feed_even(stk(l1), m1, n1, 
				   stk(l4), m4, n4); 
	    }
	}
      LhsVar (1) = 4;
      break;
    }
  case 7:
    {
      GetRhsVar (1, "d", &m1, &n1, &l1);
      GetRhsVar (2, "c", &m2, &n2, &l2);
      GetRhsVar (3, "i", &m3, &n3, &l3);
      dyadup_content_validate(cstk(l2), &errCode);
      if (errCode != SUCCESS)
	{
	  validate_print (errCode);
	  return 0;	
	}
      if ((istk(l3)[0])%2==0)
	{
	  if (strcmp(cstk(l2),"r")==0)
	    {
	      m4 = m1 * 2 - 1;
	      n4 = n1;
	      CreateVar (4, "d", &m4, &n4, &l4);
	      dyadup_2D_feed_odd_row(stk(l1), m1, n1, 
					stk(l4), m4, n4); 
	    }
	  if (strcmp(cstk(l2),"c")==0)
	    {
	      m4 = m1;
	      n4 = n1 * 2 - 1;
	      CreateVar (4, "d", &m4, &n4, &l4);
	      dyadup_2D_feed_odd_col(stk(l1), m1, n1, 
					stk(l4), m4, n4); 
	    }
	  if (strcmp(cstk(l2),"m")==0)
	    {
	      m4 = m1 * 2 - 1;
	      n4 = n1 * 2 - 1;
	      CreateVar (4, "d", &m4, &n4, &l4);
	      dyadup_2D_feed_odd(stk(l1), m1, n1, 
				    stk(l4), m4, n4); 
	    }
	}
      else
	{
	  if (strcmp(cstk(l2),"r")==0)
	    {
	      m4 = m1 * 2 + 1;
	      n4 = n1;
	      CreateVar (4, "d", &m4, &n4, &l4);
	      dyadup_2D_feed_even_row(stk(l1), m1, n1, 
				       stk(l4), m4, n4); 
	    }
	  if (strcmp(cstk(l2),"c")==0)
	    {
	      m4 = m1;
	      n4 = n1 * 2 + 1;
	      CreateVar (4, "d", &m4, &n4, &l4);
	      dyadup_2D_feed_even_col(stk(l1), m1, n1, 
				       stk(l4), m4, n4); 
	    }
	  if (strcmp(cstk(l2),"m")==0)
	    {
	      m4 = m1 * 2 + 1;
	      n4 = n1 * 2 + 1;
	      CreateVar (4, "d", &m4, &n4, &l4);
	      dyadup_2D_feed_even(stk(l1), m1, n1, 
				   stk(l4), m4, n4); 
	    }
	}
      LhsVar (1) = 4;
      break;
    }
  default:
    break;
  }
  return 0;
}

/*-------------------------------------------
 * wkeep
 *-----------------------------------------*/

int
int_wkeep (char *fname)
{
  static int l1, m1, n1, l2, m2, n2, l3, m3, n3, l4, m4, n4;
  static int minlhs = 1, maxlhs = 1, minrhs = 2, maxrhs = 3;
  int flow, errCode;//, row1, row2, col1, col2;

  CheckRhs (minrhs, maxrhs);
  CheckLhs (minlhs, maxlhs);

  flow = 0;
  wkeep_form_validate (&flow, &errCode);
  if (errCode != SUCCESS)
    {
      validate_print (errCode);
      return 0;			
    }
  l1 = 0;
  l2 = 0;
  l3 = 0;
  switch (flow){
  case 1:
    {
      GetRhsVar (1, "d", &m1, &n1, &l1);
      GetRhsVar (2, "i", &m2, &n2, &l2);
      wkeep_content_validate(flow, &errCode, l1, l2, l3);
      if (errCode != SUCCESS)
	{
	  validate_print (errCode);
	  return 0;	
	}
      if (n1 > 1)
	{
	  m3 = 1;
	  n3 = istk (l2)[0];
	}
      else
	{
	  m3 = istk (l2)[0];
	  n3 = 1;
	}
      CreateVar (3, "d", &m3, &n3, &l3);
      wkeep_1D_center(stk(l1), n1 * m1, stk(l3), n3 * m3);
      LhsVar (1) = 3;
      break;
    }
  case 2:
    {
      GetRhsVar (1, "d", &m1, &n1, &l1);
      GetRhsVar (2, "i", &m2, &n2, &l2);
      wkeep_content_validate(flow, &errCode, l1, l2, l3);
      if (errCode != SUCCESS)
	{
	  validate_print (errCode);
	  return 0;	
	}            
      m3 = istk(l2)[0];
      n3 = istk(l2)[1];
      CreateVar (3, "d", &m3, &n3, &l3);
      wkeep_2D_center(stk(l1), m1, n1, stk(l3), m3, n3);
      LhsVar (1) = 3;
      break;
    }
  case 3:
    {
      GetRhsVar (1, "d", &m1, &n1, &l1);
      GetRhsVar (2, "i", &m2, &n2, &l2);
      GetRhsVar (3, "c", &m3, &n3, &l3);
      wkeep_content_validate(flow, &errCode, l1, l2, l3);
      if (errCode != SUCCESS)
	{
	  validate_print (errCode);
	  return 0;	
	}
      if (n1 > 1)
	{
	  m4 = 1;
	  n4 = istk (l2)[0];
	}
      else
	{
	  m4 = istk (l2)[0];
	  n4 = 1;
	}
      CreateVar (4, "d", &m4, &n4, &l4);
      if (((cstk(l3)[0] - 'l') == 0)||((cstk(l3)[0] - 'L') == 0))
	wkeep_1D_left(stk(l1), n1 * m1, stk(l4), n4 * m4);
      else if (((cstk(l3)[0] - 'c') == 0)||((cstk(l3)[0] - 'C') == 0))
	wkeep_1D_center(stk(l1), n1 * m1, stk(l4), n4 * m4);
      else if (((cstk(l3)[0] - 'r') == 0)||((cstk(l3)[0] - 'R') == 0))
	wkeep_1D_right(stk(l1), n1 * m1, stk(l4), n4 * m4);
      LhsVar (1) = 4;
      break;
    }
  case 4:
    {
      GetRhsVar (1, "d", &m1, &n1, &l1);
      GetRhsVar (2, "i", &m2, &n2, &l2);
      GetRhsVar (3, "i", &m3, &n3, &l3);
      wkeep_content_validate(flow, &errCode, l1, l2, l3);
      if (errCode != SUCCESS)
	{
	  validate_print (errCode);
	  return 0;	
	}
      if (n1 > 1)
	{
	  m4 = 1;
	  n4 = istk (l2)[0];
	}
      else
	{
	  m4 = istk (l2)[0];
	  n4 = 1;
	}
      CreateVar (4, "d", &m4, &n4, &l4);
      wkeep_1D_index(stk(l1),n1 * m1,stk(l4),n4 * m4,istk(l3)[0]);
      LhsVar (1) = 4;
      break;
    }
  case 5:
    {
      GetRhsVar (1, "d", &m1, &n1, &l1);
      GetRhsVar (2, "i", &m2, &n2, &l2);
      GetRhsVar (3, "i", &m3, &n3, &l3);
      wkeep_content_validate(flow, &errCode, l1, l2, l3);
      if (errCode != SUCCESS)
	{
	  validate_print (errCode);
	  return 0;	
	}
      m4 = istk(l2)[0];
      n4 = istk(l2)[1];
      CreateVar (4, "d", &m4, &n4, &l4);
      wkeep_2D_index(stk(l1), m1, n1, stk(l4), 
		     m4, n4, istk(l3)[0], istk(l3)[1]);
      LhsVar (1) = 4;
      break;
    }
  default:
    break;
  }
  return 0;
}

/*-------------------------------------------
 * wextend
 *-----------------------------------------*/

int
int_wextend (char *fname)
{

  static int l1, m1, n1, l2, m2, n2, m3, n3, l3;
  static int l4, m4, n4, l5, m5, n5, l6, m6, n6;
  static int minlhs = 1, maxlhs = 1, minrhs = 4, maxrhs = 5;
  int flow, errCode;// adr, po, row;//col;
  char c = 'b';
  char cr[2] = "b";
  char **str;
  extend_method extMethod;

  CheckRhs (minrhs, maxrhs);
  CheckLhs (minlhs, maxlhs);

  if (GetType(1) == sci_matrix)
    GetRhsVar (1, "i", &m1, &n1, &l1);
  if (GetType(1) == sci_strings)
    GetRhsVar (1, "c", &m1, &n1, &l1);

  flow = 0;
  //sciprint("before form validate\n");
  wextend_form_validate (&flow, &errCode, l1);
  //sciprint("flow=%d\n",flow);
  if (errCode != SUCCESS)
    {
	  //sciprint("form validate error!\n");
      validate_print (errCode);
      return 0;			
    }
  l2 = 0;
  l3 = 0;
  l4 = 0;
  l5 = 0;
  
  
  switch(flow) {
  case 1:
    {
      GetRhsVar (2, "c", &m2, &n2, &l2);
      GetRhsVar (3, "d", &m3, &n3, &l3);
      GetRhsVar (4, "i", &m4, &n4, &l4);
      GetRhsVar (5, "c", &m5, &n5, &l5);
      wextend_content_validate (flow, &errCode, l2, l3, l4, l5, str);
      if (errCode != SUCCESS)
	{
	  validate_print (errCode);
	  return 0;			
	}
      extend_method_parse (cstk(l2), &extMethod);
      if ((extMethod == PER) && ((m3*n3)%2 != 0))
	{
	  if (((cstk(l5)[0]-'l') == 0) || ((cstk(l5)[0]-'r') == 0))
	    {
	      if (m3 > 1)
		{
		  m6 = m3 + istk(l4)[0] + 1;
		  n6 = 1;
		}
	      else
		{
		  m6 = 1;
		  n6 = n3 + istk(l4)[0] + 1;
		}
	    }
	  if ((cstk(l5)[0]-'b') == 0)
	    {
	      if (m3 > 1)
		{
		  m6 = m3 + 2*istk(l4)[0] + 1;
		  n6 = 1;
		}
	      else
		{
		  m6 = 1;
		  n6 = n3 + 2*istk(l4)[0] + 1;
		}
	    }
	}
      else
	{
	  if (((cstk(l5)[0]-'l') == 0) || ((cstk(l5)[0]-'r') == 0))
	    {
	      if (m3 > 1)
		{
		  m6 = m3 + istk(l4)[0];
		  n6 = 1;
		}
	      else
		{
		  m6 = 1;
		  n6 = n3 + istk(l4)[0];
		}
	    }
	  if ((cstk(l5)[0]-'b') == 0)
	    {
	      if (m3 > 1)
		{
		  m6 = m3 + 2*istk(l4)[0];
		  n6 = 1;
		}
	      else
		{
		  m6 = 1;
		  n6 = n3 + 2*istk(l4)[0];
		}
	    }
	}
      CreateVar (6, "d", &m6, &n6, &l6);
      if (strcmp(cstk(l5),"l") == 0)
	wextend_1D_left(stk(l3), m3*n3, stk(l6), m6*n6, extMethod);
      else if (strcmp(cstk(l5),"r") == 0)
	wextend_1D_right(stk(l3), m3*n3, stk(l6), m6*n6, extMethod);
      else
	wextend_1D_center(stk(l3), m3*n3, stk(l6), m6*n6, extMethod);
      LhsVar (1) = 6;
      break;
    }
  case 2:
    {
      GetRhsVar (2, "c", &m2, &n2, &l2);
      GetRhsVar (3, "d", &m3, &n3, &l3);
      GetRhsVar (4, "i", &m4, &n4, &l4);
      wextend_content_validate (flow, &errCode, l2, l3, l4, l5, str);
      if (errCode != SUCCESS)
	{
	  validate_print (errCode);
	  return 0;			
	}
      extend_method_parse (cstk(l2), &extMethod);
      if ((extMethod == PER) && ((m3*n3)%2 != 0))
	{
	  if (m3 > 1)
	    {
	      m5 = m3 + 2*istk(l4)[0] + 1;
	      n5 = 1;
	    }
	  else
	    {
	      m5 = 1;
	      n5 = n3 + 2*istk(l4)[0] + 1;
	    }
	}
      else
	{
	  if (m3 > 1)
	    {
	      m5 = m3 + 2*istk(l4)[0];
	      n5 = 1;
	    }
	  else
	    {
	      m5 = 1;
	      n5 = n3 + 2*istk(l4)[0];
	    }
	}
      CreateVar (5, "d", &m5, &n5, &l5);
      wextend_1D_center(stk(l3), m3*n3, stk(l5), m5*n5, extMethod);
      LhsVar (1) = 5;
      break;
    }
  case 3:
    {
      GetRhsVar (2, "c", &m2, &n2, &l2);
      GetRhsVar (3, "d", &m3, &n3, &l3);
      GetRhsVar (4, "i", &m4, &n4, &l4);
      GetRhsVar (5, "S", &m5, &n5, &str);
      wextend_content_validate (flow, &errCode, l2, l3, l4, l5, str);
      if (errCode != SUCCESS)
	{
	  validate_print (errCode);
	  return 0;			
	}
      extend_method_parse (cstk(l2), &extMethod);
      if ((extMethod == PER) && (m3%2 != 0))
	{
	  if (((str[0][0]-'l') == 0) || ((str[0][0]-'r') == 0))
	    m6 = m3 + istk(l4)[0] + 1;
	  if ((str[0][0]-'b') == 0)
	    m6 = m3 + 2*istk(l4)[0] + 1;
	}
      else
	{
	  if (((str[0][0]-'l') == 0) || ((str[0][0]-'r') == 0))
	    m6 = m3 + istk(l4)[0];
	  if ((str[0][0]-'b') == 0)
	    m6 = m3 + 2*istk(l4)[0];
	}
      if ((extMethod == PER) && (n3%2 != 0))
	{
	  if (((str[1][0]-'l') == 0) || ((str[1][0]-'r') == 0))
	    n6 = n3 + istk(l4)[1] + 1;
	  if ((str[1][0]-'b') == 0)
	    n6 = n3 + 2*istk(l4)[1] + 1;
	}
      else
	{
	  if (((str[1][0]-'l') == 0) || ((str[1][0]-'r') == 0))
	    n6 = n3 + istk(l4)[1];
	  if ((str[1][0]-'b') == 0)
	    n6 = n3 + 2*istk(l4)[1];
	}
      CreateVar (6, "d", &m6, &n6, &l6);
      //wextend_2D (stk(l3), m3, n3, stk(l6), m6, n6, 
		//  extMethod, &str[0][0], &str[1][0]);
	  wextend_2D (stk(l3), m3, n3, stk(l6), m6, n6, 
		  extMethod, &str[1][0], &str[0][0]);
      LhsVar (1) = 6;
      FreeRhsSVar(str);
      break;
    }
  case 4:
    {
      GetRhsVar (2, "c", &m2, &n2, &l2);
      GetRhsVar (3, "d", &m3, &n3, &l3);
      GetRhsVar (4, "i", &m4, &n4, &l4);
      wextend_content_validate (flow, &errCode, l2, l3, l4, l5, str);
      if (errCode != SUCCESS)
	{
	  validate_print (errCode);
	  return 0;			
	}
      extend_method_parse (cstk(l2), &extMethod);
      if ((extMethod == PER) && (m3%2 != 0))
	m5 = m3 + 2*istk(l4)[0] + 1;
      else
	m5 = m3 + 2*istk(l4)[0];
      if ((extMethod == PER) && (n3%2 != 0))
	n5 = n3 + 2*istk(l4)[0] + 1;
      else
	n5 = n3 + 2*istk(l4)[0];
      CreateVar (5, "d", &m5, &n5, &l5);
      wextend_2D (stk(l3), m3, n3, stk(l5), m5, n5, 
		  extMethod, &c, &c);
      LhsVar (1) = 5;
      break;
    }
  case 5:
    {
      GetRhsVar (2, "c", &m2, &n2, &l2);
      GetRhsVar (3, "d", &m3, &n3, &l3);
      GetRhsVar (4, "i", &m4, &n4, &l4);
      wextend_content_validate (flow, &errCode, l2, l3, l4, l5, str);
      if (errCode != SUCCESS)
	{
	  validate_print (errCode);
	  return 0;			
	}
      extend_method_parse (cstk(l2), &extMethod);
      if ((extMethod == PER) && (m3%2 != 0))
	m5 = m3 + 2*istk(l4)[0] + 1;
      else
	m5 = m3 + 2*istk(l4)[0];
      if ((extMethod == PER) && (n3%2 != 0))
	n5 = n3 + 2*istk(l4)[1] + 1;
      else
	n5 = n3 + 2*istk(l4)[1];
      CreateVar (5, "d", &m5, &n5, &l5);
      wextend_2D (stk(l3), m3, n3, stk(l5), m5, n5, 
		  extMethod, &c, &c);
      LhsVar (1) = 5;
      break;
    }
  case 6:
    {
      GetRhsVar (2, "c", &m2, &n2, &l2);
      GetRhsVar (3, "d", &m3, &n3, &l3);
      GetRhsVar (4, "i", &m4, &n4, &l4);
      GetRhsVar (5, "c", &m5, &n5, &l5);
	  //sciprint("before wextend content validate!\n");
      wextend_content_validate (flow, &errCode, l2, l3, l4, l5, str);
	  //sciprint("after wextend content validate!\n");
      if (errCode != SUCCESS)
	{
	  validate_print (errCode);
	  return 0;			
	}
      extend_method_parse (cstk(l2), &extMethod);
      if ((extMethod == PER) && (m3%2 != 0))
	{
	  if ((strcmp(cstk(l5),"l")==0) || (strcmp(cstk(l5),"r")==0))
	    m6 = m3 + istk(l4)[0] + 1;
	  if ((strcmp(cstk(l5),"b") == 0))
	    m6 = m3 + 2*istk(l4)[0] + 1;
	}
      else
	{
	  if ((strcmp(cstk(l5),"l")==0) || (strcmp(cstk(l5),"r")==0))
	    m6 = m3 + istk(l4)[0];
	  if ((strcmp(cstk(l5),"b") == 0))
	    m6 = m3 + 2*istk(l4)[0];
	}
      n6 = n3;
      CreateVar (6, "d", &m6, &n6, &l6);
	  //sciprint("after creation!flow=6\n");
      wextend_2D_row (stk(l3), m3, n3, stk(l6), m6, n6, 
		  extMethod, cstk(l5));
      LhsVar (1) = 6;
      break;
    }
  case 7:
    {
      GetRhsVar (2, "c", &m2, &n2, &l2);
      GetRhsVar (3, "d", &m3, &n3, &l3);
      GetRhsVar (4, "i", &m4, &n4, &l4);
      wextend_content_validate (flow, &errCode, l2, l3, l4, l5, str);
      if (errCode != SUCCESS)
	{
	  validate_print (errCode);
	  return 0;			
	}
      extend_method_parse (cstk(l2), &extMethod);
      if ((extMethod == PER) && (m3%2 != 0))
	m5 = m3 + 2*istk(l4)[0] + 1;
      else
	m5 = m3 + 2*istk(l4)[0];
      n5 = n3;
      CreateVar (5, "d", &m5, &n5, &l5);
      wextend_2D_row (stk(l3), m3, n3, stk(l5), m5, n5, 
		  extMethod, &cr);
      LhsVar (1) = 5;
      break;
    }
  case 8:
    {
      GetRhsVar (2, "c", &m2, &n2, &l2);
      GetRhsVar (3, "d", &m3, &n3, &l3);
      GetRhsVar (4, "i", &m4, &n4, &l4);
      GetRhsVar (5, "c", &m5, &n5, &l5);
      wextend_content_validate (flow, &errCode, l2, l3, l4, l5, str);
      if (errCode != SUCCESS)
	{
	  validate_print (errCode);
	  return 0;			
	}
      extend_method_parse (cstk(l2), &extMethod);
      if ((extMethod == PER) && (n3%2 != 0))
	{
	  if ((strcmp(cstk(l5),"l")==0) || (strcmp(cstk(l5),"r")==0))
	    n6 = n3 + istk(l4)[0] + 1;
	  if ((strcmp(cstk(l5),"b") == 0))
	    n6 = n3 + 2*istk(l4)[0] + 1;
	}
      else
	{
	  if ((strcmp(cstk(l5),"l")==0) || (strcmp(cstk(l5),"r")==0))
	    n6 = n3 + istk(l4)[0];
	  if ((strcmp(cstk(l5),"b") == 0))
	    n6 = n3 + 2*istk(l4)[0];
	}
      m6 = m3;
      CreateVar (6, "d", &m6, &n6, &l6);
      wextend_2D_col (stk(l3), m3, n3, stk(l6), m6, n6, 
		  extMethod, cstk(l5));
      LhsVar (1) = 6;
      break;
    }
  case 9:
    {
      GetRhsVar (2, "c", &m2, &n2, &l2);
      GetRhsVar (3, "d", &m3, &n3, &l3);
      GetRhsVar (4, "i", &m4, &n4, &l4);
      wextend_content_validate (flow, &errCode, l2, l3, l4, l5, str);
      if (errCode != SUCCESS)
	{
	  validate_print (errCode);
	  return 0;			
	}
      extend_method_parse (cstk(l2), &extMethod);
      if ((extMethod == PER) && (n3%2 != 0))
	n5 = n3 + 2*istk(l4)[0] + 1;
      else
	n5 = n3 + 2*istk(l4)[0];
      m5 = m3;
      CreateVar (5, "d", &m5, &n5, &l5);
      wextend_2D_col (stk(l3), m3, n3, stk(l5), m5, n5, 
		  extMethod, &cr);
      LhsVar (1) = 5;
      break;
    }
  case 10:
    {
      GetRhsVar (2, "c", &m2, &n2, &l2);
      GetRhsVar (3, "d", &m3, &n3, &l3);
      GetRhsVar (4, "i", &m4, &n4, &l4);
      GetRhsVar (5, "c", &m5, &n5, &l5);
      wextend_content_validate (flow, &errCode, l2, l3, l4, l5, str);
      if (errCode != SUCCESS)
	{
	  validate_print (errCode);
	  return 0;			
	}
      extend_method_parse (cstk(l2), &extMethod);
      if ((extMethod == PER) && (m3%2 != 0))
	{
	  if ((cstk(l5)[0]=='l') || (cstk(l5)[0]=='r'))
	    m6 = m3 + istk(l4)[0] + 1;
	  if (cstk(l5)[0]=='b')
	    m6 = m3 + 2*istk(l4)[0] + 1;
	}
      else
	{
	  if ((cstk(l5)[0]=='l') || (cstk(l5)[0]=='r'))
	    m6 = m3 + istk(l4)[0];
	  if (cstk(l5)[0]=='b')
	    m6 = m3 + 2*istk(l4)[0];
	}
      if ((extMethod == PER) && (n3%2 != 0))
	{
	  if ((cstk(l5)[1]=='l') || (cstk(l5)[1]=='r'))
	    n6 = n3 + istk(l4)[1] + 1;
	  if (cstk(l5)[1]=='b')
	    n6 = n3 + 2*istk(l4)[1] + 1;
	}
      else
	{
	  if ((cstk(l5)[1]=='l') || (cstk(l5)[1]=='r'))
	    n6 = n3 + istk(l4)[1];
	  if (cstk(l5)[1]=='b')
	    n6 = n3 + 2*istk(l4)[1];
	}

	
      CreateVar (6, "d", &m6, &n6, &l6);
      wextend_2D (stk(l3), m3, n3, stk(l6), m6, n6, 
		  extMethod, &cstk(l5)[1], &cstk(l5)[0]);
	  //wextend_2D (stk(l3), m3, n3, stk(l6), m6, n6, 
		//  extMethod, &cstk(l5)[0], &cstk(l5)[1]);
      LhsVar (1) = 6;
      break;
    }
  default:
    break;
  }
  //sciprint("flow=%d\n",flow);
  return 0;
}
