/* HP Scanjet 3900 series - SANE Backend controller

   Copyright (C) 2005 Jonathan Bravo Lopez <jkdsoft@gmail.com>

   This file is part of the SANE package.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

   As a special exception, the authors of SANE give permission for
   additional uses of the libraries contained in this release of SANE.

   The exception is that, if you link a SANE library with other files
   to produce an executable, this does not by itself cause the
   resulting executable to be covered by the GNU General Public
   License.  Your use of that executable is in no way restricted on
   account of linking the SANE library code into it.

   This exception does not, however, invalidate any other reasons why
   the executable file might be covered by the GNU General Public
   License.

   If you submit changes to SANE to the maintainers to be included in
   a subsequent release, you agree by submitting the changes that
   those changes may be distributed with this exception intact.

   If you write modifications of your own for SANE, it is your choice
   whether to permit this exception to apply to your modifications.
   If you do not wish that, delete this exception notice.
*/

/* Backend Code for SANE*/
#define HP3900_CONFIG_FILE "hp3900.conf"

#define HP3970 0
#define HP4070 1
#define HP4370 2

#include "../include/sane/config.h"
#include "../include/sane/sane.h"
#include "../include/sane/sanei.h"
#include "../include/sane/sanei_backend.h"
#include "../include/sane/sanei_config.h"
#include "../include/sane/saneopts.h"
#include "../include/sane/sanei_usb.h"
#include "../include/sane/sanei_debug.h"

#include "hp3900_rts8822.c"

/* This structure is temporal until debugging entire windows driver */
struct st_gamma
{
  SANE_Int depth;
  SANE_Int *table[3];
};

/* options enumerator */
typedef enum
{
  optCount = 0,
  optTLX, optTLY, optBRX, optBRY,
  optDPI,
  optGammaTableRed,		/* Gamma Tables */
  optGammaTableGreen,
  optGammaTableBlue,
  optType,
  optColorMode,
  optDepth,
  optThreshold,
  optLast
} EOptionIndex;

/* linked list of SANE_Device structures */
typedef struct TDevListEntry
{
  struct TDevListEntry *pNext;
  SANE_Device dev;
  char *devname;
}
TDevListEntry;

typedef struct
{
  char *pszVendor;
  char *pszName;
}
TScannerModel;

typedef union
{
  SANE_Word w;
  SANE_Word *wa;		/* word array */
  SANE_String s;
}
TOptionValue;

typedef struct
{
  SANE_Option_Descriptor aOptions[optLast];
  TOptionValue aValues[optLast];
  struct params ScanParams;

  SANE_Word *aGammaTableR;	/* a 16-to-16 bit color lookup table */
  SANE_Word *aGammaTableG;	/* a 16-to-16 bit color lookup table */
  SANE_Word *aGammaTableB;	/* a 16-to-16 bit color lookup table */

  SANE_Int fScanning;		/* TRUE if actively scanning */
  SANE_Int fCanceled;
}
TScanner;

static TScannerModel Model_HP39xx =
  { "Hewlett-Packard", "RTS8822 chipset based" };

static TDevListEntry *_pFirstSaneDev = 0;
static SANE_Int iNumSaneDev = 0;
static const SANE_Device **_pSaneDevList = 0;

/* Option constrains */
static const SANE_Int setResolutions[] =
  { 8, 75, 100, 150, 200, 300, 600, 1200, 2400 };
static const SANE_Int depth_list[] = { 2, 8, 16 };	/*{4, 1, 8, 12, 16}; */
static const SANE_Range rangeThreshold = { 0, 255, 0 };
static const SANE_Range rangeGammaTable = { 0, 65535, 0 };
static SANE_Range rangeXmm = { 0, 220, 1 };
static SANE_Range rangeYmm = { 0, 300, 1 };
static const SANE_Range rangeXoffset = { 0, 20, 1 };
static const SANE_Range rangeYoffset = { 0, 70, 1 };
static const SANE_Int offsetX = 0;
static const SANE_Int offsetY = 0;
static SANE_String_Const source_list[] = {
  SANE_I18N ("Flatbed"),
  SANE_I18N ("Slide"),
  SANE_I18N ("Negative"),
  0
};
static SANE_String_Const colormode_list[] = {
  SANE_I18N ("Color"),
  SANE_I18N ("Gray"),
  SANE_I18N ("Lineart"),
  0
};
static SANE_Byte *Image = NULL;
static SANE_Byte *Rest = NULL;
static SANE_Int Rest_amount = 0;
static SANE_Int mylin = 0;
static struct st_gamma myGamma;
static SANE_Int con_colormode = -1;
static SANE_Int con_threshold = 40;
static SANE_Int Scanner_Model = -1;

/* Own functions */

static void
Silent_Compile (void)
{
  /*
     There are some functions in hp3900_rts8822.c that aren't used yet.
     To avoid compilation warnings we will use them here
   */

  SANE_Byte a = 1;

  if (a == 0)
    {
      Buttons_Status ();
      Chipset_Name (NULL, 0);
    }
}

static void
Range_GetXY (SANE_Int source, SANE_Int type)
{
  struct st_coords *coords = Constrains_Get (source);

  if (coords != NULL)
    {
      switch (type)
	{
	case 1:		/* Y */
	  rangeYmm.max = coords->height;
	  break;
	default:		/* X */
	  rangeXmm.max = coords->width;
	  break;
	}
    }
}

static SANE_Int
Set_Constrains (SANE_Int Scanner_Model)
{
  /*
   * Values are set in milimiters
   */

  struct st_coords coords;

  switch (Scanner_Model)
    {
    case HP4070:
      /* Reflective */
      coords.left = 0;
      coords.top = 0;
      coords.width = 220;
      coords.height = 300;
      Constrains_Set (ST_NORMAL, &coords);
      /* Negative & Slide */
      coords.left = 58;
      coords.top = 0;
      coords.width = 99;
      coords.height = 197;
      Constrains_Set (ST_TA, &coords);
      Constrains_Set (ST_NEG, &coords);
      break;
    default:			/* HP3970 and others */
      /* Reflective */
      coords.left = 0;
      coords.top = 0;
      coords.width = 220;
      coords.height = 300;
      Constrains_Set (ST_NORMAL, &coords);
      /* Negative & Slide */
      coords.left = 88;
      coords.top = 0;
      coords.width = 42;
      coords.height = 83;
      Constrains_Set (ST_TA, &coords);
      Constrains_Set (ST_NEG, &coords);
      break;
    }

  return OK;
}

static SANE_Int
get_ScannerModel (SANE_Int product, SANE_Int vendor)
{
  SANE_Int rst = -1;

  if (vendor == 0x3f0)
    {
      switch (product)
	{
	case 0x2305:		/* HP Scanjet 3970c */
	  rst = HP3970;
	  break;
	case 0x2405:		/* HP Scanjet 4070 Photosmart */
	  rst = HP4070;
	  break;
	case 0x4105:		/* HP Scanjet 4370 */
	  rst = HP4370;
	  break;
	}
    }

  return rst;
}

static void
Set_Coordinates (SANE_Int scantype, SANE_Int resolution,
		 struct st_coords *coords)
{
  struct st_coords *limits = Constrains_Get (scantype);

  if (coords->left == -1)
    coords->left = 0;

  if (coords->width == -1)
    coords->width = limits->width;

  if (coords->top == -1)
    coords->top = 0;

  if (coords->height == -1)
    coords->height = limits->height;

  coords->left = MM_TO_PIXEL (coords->left, resolution);
  coords->width = MM_TO_PIXEL (coords->width, resolution);
  coords->top = MM_TO_PIXEL (coords->top, resolution);
  coords->height = MM_TO_PIXEL (coords->height, resolution);

  Constrains_Check (resolution, scantype, coords);
}


static SANE_Int
Color_to_Lineart (SANE_Byte * buffer, SANE_Int size, SANE_Int depth,
		  SANE_Int threshold)
{
  if (buffer != NULL)
    {
      SANE_Int c;
      SANE_Int dot_size = 3 * ((depth > 8) ? 2 : 1);
      SANE_Int colour;
      SANE_Byte *pColor = buffer;
      USHORT *sColor = (void *) buffer;

      if ((threshold < 0) || (threshold > 255))
	threshold = 0x80;

      threshold = (((1 << depth) - 1) * threshold) / 255;

      for (c = 0; c < size / dot_size; c++)
	{
	  if (depth > 8)
	    {
	      colour =
		((*sColor + *(sColor + 1) + *(sColor + 2)) / 3) & 0xffff;
	      colour = (colour >= threshold) ? 65535 : 0;

	      *sColor = colour;
	      *(sColor + 1) = colour;
	      *(sColor + 2) = colour;
	      sColor += 3;
	    }
	  else
	    {
	      colour = ((*pColor + *(pColor + 1) + *(pColor + 2)) / 3) & 0xff;
	      colour = (colour >= threshold) ? 255 : 0;
	      *pColor = colour;
	      *(pColor + 1) = colour;
	      *(pColor + 2) = colour;
	      pColor += 3;
	    }
	}
    }

  return OK;
}

static SANE_Int
Color_to_Gray (SANE_Byte * buffer, SANE_Int size, SANE_Int depth)
{
  if (buffer != NULL)
    {
      SANE_Int c;
      SANE_Int dot_size = 3 * ((depth > 8) ? 2 : 1);
      SANE_Int colour;
      SANE_Byte *pColor = buffer;
      USHORT *sColor = (void *) buffer;

      for (c = 0; c < size / dot_size; c++)
	{
	  if (depth > 8)
	    {
	      colour =
		((*sColor + *(sColor + 1) + *(sColor + 2)) / 3) & 0xffff;
	      *sColor = colour;
	      *(sColor + 1) = colour;
	      *(sColor + 2) = colour;
	      sColor += 3;
	    }
	  else
	    {
	      colour = ((*pColor + *(pColor + 1) + *(pColor + 2)) / 3) & 0xff;
	      *pColor = colour;
	      *(pColor + 1) = colour;
	      *(pColor + 2) = colour;
	      pColor += 3;
	    }
	}
    }

  return OK;
}

static void
Gamma_Free (void)
{
  SANE_Int a;

  DBG (2, "> Gamma_Free()\n");

  /* Destroy gamma tables */
  for (a = 0; a < 3; a++)
    {
      if (myGamma.table[a] != NULL)
	{
	  free (myGamma.table[a]);
	  myGamma.table[a] = NULL;
	}
    }
}

static SANE_Int
Gamma_Create8bit (TScanner * s)
{
  SANE_Int a, b, c;

  DBG (2, "> Gamma_Create8bit(*s)\n");

  /* Delete previous gamma tables if exist */
  Gamma_Free ();

  /* Create gamma tables */
  for (a = 0; a < 3; a++)
    myGamma.table[a] = (SANE_Word *) malloc (256 * sizeof (SANE_Word));


  if ((myGamma.table[0] != NULL) && (myGamma.table[1] != NULL)
      && (myGamma.table[2] != NULL))
    {
      c = 0;
      for (b = 0; b < 256; b++)
	{
	  *(myGamma.table[0] + b) = (s->aGammaTableR[c] >> 8) & 0xff;
	  *(myGamma.table[1] + b) = (s->aGammaTableG[c] >> 8) & 0xff;
	  *(myGamma.table[2] + b) = (s->aGammaTableB[c] >> 8) & 0xff;
	  c += 256;
	}
    }
  else
    Gamma_Free ();

  return OK;
}

static void
Gamma_Apply (TScanner * s, SANE_Byte * buffer, SANE_Int size, SANE_Int depth)
{
  if (buffer != NULL)
    {
      SANE_Int c;
      SANE_Int dot_size = 3 * ((depth > 8) ? 2 : 1);
      SANE_Byte *pColor = buffer;
      USHORT *sColor = (void *) buffer;

      for (c = 0; c < size / dot_size; c++)
	{
	  if (depth > 8)
	    {
	      *sColor =
		(s->aGammaTableR !=
		 NULL) ? s->aGammaTableR[*sColor] : *sColor;
	      *(sColor + 1) =
		(s->aGammaTableG !=
		 NULL) ? s->aGammaTableG[*(sColor + 1)] : *(sColor + 1);
	      *(sColor + 2) =
		(s->aGammaTableB !=
		 NULL) ? s->aGammaTableB[*(sColor + 2)] : *(sColor + 2);
	      sColor += 3;
	    }
	  else
	    {
	      *pColor =
		(myGamma.table[CL_RED] !=
		 NULL) ? myGamma.table[CL_RED][*pColor] : *pColor;
	      *(pColor + 1) =
		(myGamma.table[CL_GREEN] !=
		 NULL) ? myGamma.table[CL_GREEN][*(pColor + 1)] : *(pColor +
								    1);
	      *(pColor + 2) =
		(myGamma.table[CL_BLUE] !=
		 NULL) ? myGamma.table[CL_BLUE][*(pColor + 2)] : *(pColor +
								   2);
	      pColor += 3;
	    }
	}
    }
}

static SANE_Int
Get_Source (SANE_String source)
{
  SANE_Int rst;

  if (strcmp (source, source_list[0]) == 0)
    rst = ST_NORMAL;
  else if (strcmp (source, source_list[1]) == 0)
    rst = ST_TA;
  else if (strcmp (source, source_list[2]) == 0)
    rst = ST_NEG;
  else
    rst = ST_NORMAL;

  return rst;
}

static SANE_Int
Get_Colormode (SANE_String colormode)
{
  SANE_Int rst;

  if (strcmp (colormode, colormode_list[0]) == 0)
    rst = 0;
  else if (strcmp (colormode, colormode_list[1]) == 0)
    rst = CM_GRAY;
  else if (strcmp (colormode, colormode_list[2]) == 0)
    rst = CM_LINEART;
  else
    rst = 0;

  return rst;
}

static SANE_Status
Translate_coords (struct st_coords *coords)
{
  SANE_Int data;

  DBG (2, "> translate_coords(*coords)\n");

  if ((coords->left < 0) || (coords->top < 0) ||
      (coords->width < 0) || (coords->height < 0))
    return SANE_STATUS_INVAL;

  if (coords->width < coords->left)
    {
      data = coords->left;
      coords->left = coords->width;
      coords->width = data;
    }

  if (coords->height < coords->top)
    {
      data = coords->top;
      coords->top = coords->height;
      coords->height = data;
    }

  coords->width -= coords->left;
  coords->height -= coords->top;

  if (coords->width == 0)
    coords->width++;
  if (coords->height == 0)
    coords->height++;

  return SANE_STATUS_GOOD;
}

static SANE_Int
dot_size (SANE_Handle h)
{
  TScanner *s = (TScanner *) h;
  SANE_Int rst = 0;
  SANE_Int channels, bytes_per_channel;

  if (s != NULL)
    {
      channels = (s->ScanParams.colormode == CM_LINEART) ? 1 : 3;
      bytes_per_channel = (s->ScanParams.depth > 8) ? 2 : 1;
      rst = bytes_per_channel * channels;
    }

  return rst;
}

static size_t
max_string_size (const SANE_String_Const strings[])
{
  size_t size, max_size = 0;
  SANE_Int i;

  DBG (2, "> max_string_size:\n");

  for (i = 0; strings[i]; ++i)
    {
      size = strlen (strings[i]) + 1;
      if (size > max_size)
	max_size = size;
    }

  return max_size;
}

static void
_InitOptions (TScanner * s)
{
  SANE_Int i;
  SANE_Option_Descriptor *pDesc;
  TOptionValue *pVal;

  DBG (2, "> _InitOptions:\n");

  /* set a neutral gamma */

  if (s->aGammaTableR == NULL)
    {
      SANE_Int j;
      double value, c;

      s->aGammaTableR = malloc (65535 * sizeof (SANE_Word));
      s->aGammaTableG = malloc (65535 * sizeof (SANE_Word));
      s->aGammaTableB = malloc (65535 * sizeof (SANE_Word));

      for (j = 0; j < 65535; j++)
	{
	  value = (j / (65535. - 1));
	  value = pow (value, (1. / 2.2));
	  value = value * (65535. - 1);
	  c = (SANE_Int) value;
	  if (c > (65535 - 1))
	    c = (65535 - 1);
	  else if (c < 0)
	    c = 0;

	  s->aGammaTableR[j] = c;
	  s->aGammaTableG[j] = c;
	  s->aGammaTableB[j] = c;
	}
    }

  /* By default preview scan */
  s->ScanParams.scantype = ST_NORMAL;
  s->ScanParams.colormode = 0;
  s->ScanParams.resolution_x = 75;
  s->ScanParams.resolution_y = 75;
  s->ScanParams.coords.left = 0;
  s->ScanParams.coords.top = 0;
  s->ScanParams.coords.width = 220;
  s->ScanParams.coords.height = 300;
  s->ScanParams.depth = 8;
  s->ScanParams.channel = 0;

  for (i = optCount; i < optLast; i++)
    {

      pDesc = &s->aOptions[i];
      pVal = &s->aValues[i];

      /* defaults */
      pDesc->name = "";
      pDesc->title = "";
      pDesc->desc = "";
      pDesc->type = SANE_TYPE_INT;
      pDesc->unit = SANE_UNIT_NONE;
      pDesc->size = sizeof (SANE_Word);
      pDesc->constraint_type = SANE_CONSTRAINT_NONE;
      pDesc->cap = 0;

      switch (i)
	{

	case optCount:
	  pDesc->title = SANE_TITLE_NUM_OPTIONS;
	  pDesc->desc = SANE_DESC_NUM_OPTIONS;
	  pDesc->cap = SANE_CAP_SOFT_DETECT;
	  pVal->w = (SANE_Word) optLast;
	  break;

	case optTLX:
	  pDesc->name = SANE_NAME_SCAN_TL_X;
	  pDesc->title = SANE_TITLE_SCAN_TL_X;
	  pDesc->desc = SANE_DESC_SCAN_TL_X;
	  pDesc->unit = SANE_UNIT_MM;
	  pDesc->constraint_type = SANE_CONSTRAINT_RANGE;
	  pDesc->constraint.range = &rangeXmm;
	  pDesc->cap = SANE_CAP_SOFT_SELECT | SANE_CAP_SOFT_DETECT;
	  pVal->w = rangeXmm.max + offsetX;
	  break;

	case optTLY:
	  pDesc->name = SANE_NAME_SCAN_TL_Y;
	  pDesc->title = SANE_TITLE_SCAN_TL_Y;
	  pDesc->desc = SANE_DESC_SCAN_TL_Y;
	  pDesc->unit = SANE_UNIT_MM;
	  pDesc->constraint_type = SANE_CONSTRAINT_RANGE;
	  pDesc->constraint.range = &rangeYmm;
	  pDesc->cap = SANE_CAP_SOFT_SELECT | SANE_CAP_SOFT_DETECT;
	  pVal->w = rangeYmm.max + offsetY;
	  break;

	case optBRX:
	  pDesc->name = SANE_NAME_SCAN_BR_X;
	  pDesc->title = SANE_TITLE_SCAN_BR_X;
	  pDesc->desc = SANE_DESC_SCAN_BR_X;
	  pDesc->unit = SANE_UNIT_MM;
	  pDesc->constraint_type = SANE_CONSTRAINT_RANGE;
	  pDesc->constraint.range = &rangeXmm;
	  pDesc->cap = SANE_CAP_SOFT_SELECT | SANE_CAP_SOFT_DETECT;
	  pVal->w = rangeXmm.max + offsetX;
	  break;

	case optBRY:
	  pDesc->name = SANE_NAME_SCAN_BR_Y;
	  pDesc->title = SANE_TITLE_SCAN_BR_Y;
	  pDesc->desc = SANE_DESC_SCAN_BR_Y;
	  pDesc->unit = SANE_UNIT_MM;
	  pDesc->constraint_type = SANE_CONSTRAINT_RANGE;
	  pDesc->constraint.range = &rangeYmm;
	  pDesc->cap = SANE_CAP_SOFT_SELECT | SANE_CAP_SOFT_DETECT;
	  pVal->w = rangeYmm.max + offsetY;
	  break;

	case optDPI:
	  pDesc->name = SANE_NAME_SCAN_RESOLUTION;
	  pDesc->title = SANE_TITLE_SCAN_RESOLUTION;
	  pDesc->desc = SANE_DESC_SCAN_RESOLUTION;
	  pDesc->unit = SANE_UNIT_DPI;
	  pDesc->constraint_type = SANE_CONSTRAINT_WORD_LIST;
	  pDesc->constraint.word_list = setResolutions;
	  pDesc->cap = SANE_CAP_SOFT_SELECT | SANE_CAP_SOFT_DETECT;
	  pVal->w = setResolutions[1];
	  break;

	case optGammaTableRed:
	  pDesc->name = SANE_NAME_GAMMA_VECTOR_R;
	  pDesc->title = SANE_TITLE_GAMMA_VECTOR_R;
	  pDesc->desc = SANE_DESC_GAMMA_VECTOR_R;
	  pDesc->size = 65536 * sizeof (SANE_Word);
	  pDesc->unit = SANE_UNIT_NONE;
	  pDesc->constraint_type = SANE_CONSTRAINT_RANGE;
	  pDesc->constraint.range = &rangeGammaTable;
	  pDesc->cap = SANE_CAP_SOFT_SELECT | SANE_CAP_SOFT_DETECT;
	  pVal->wa = s->aGammaTableR;
	  break;

	case optGammaTableGreen:
	  pDesc->name = SANE_NAME_GAMMA_VECTOR_G;
	  pDesc->title = SANE_TITLE_GAMMA_VECTOR_G;
	  pDesc->desc = SANE_DESC_GAMMA_VECTOR_G;
	  pDesc->size = 65536 * sizeof (SANE_Word);
	  pDesc->unit = SANE_UNIT_NONE;
	  pDesc->constraint_type = SANE_CONSTRAINT_RANGE;
	  pDesc->constraint.range = &rangeGammaTable;
	  pDesc->cap = SANE_CAP_SOFT_SELECT | SANE_CAP_SOFT_DETECT;
	  pVal->wa = s->aGammaTableG;
	  break;

	case optGammaTableBlue:
	  pDesc->name = SANE_NAME_GAMMA_VECTOR_B;
	  pDesc->title = SANE_TITLE_GAMMA_VECTOR_B;
	  pDesc->desc = SANE_DESC_GAMMA_VECTOR_B;
	  pDesc->size = 65536 * sizeof (SANE_Word);
	  pDesc->unit = SANE_UNIT_NONE;
	  pDesc->constraint_type = SANE_CONSTRAINT_RANGE;
	  pDesc->constraint.range = &rangeGammaTable;
	  pDesc->cap = SANE_CAP_SOFT_SELECT | SANE_CAP_SOFT_DETECT;
	  pVal->wa = s->aGammaTableB;
	  break;

	case optType:
	  pDesc->name = SANE_NAME_SCAN_SOURCE;
	  pDesc->title = SANE_TITLE_SCAN_SOURCE;
	  pDesc->desc = SANE_DESC_SCAN_SOURCE;
	  pDesc->type = SANE_TYPE_STRING;
	  pDesc->size = max_string_size (source_list);
	  pDesc->constraint_type = SANE_CONSTRAINT_STRING_LIST;
	  pDesc->constraint.string_list = source_list;
	  pDesc->cap = SANE_CAP_SOFT_SELECT | SANE_CAP_SOFT_DETECT;
	  pVal->s = strdup (source_list[0]);
	  break;

	case optColorMode:
	  pDesc->name = SANE_NAME_SCAN_MODE;
	  pDesc->title = SANE_TITLE_SCAN_MODE;
	  pDesc->desc = SANE_DESC_SCAN_MODE;
	  pDesc->type = SANE_TYPE_STRING;
	  pDesc->size = max_string_size (colormode_list);
	  pDesc->constraint_type = SANE_CONSTRAINT_STRING_LIST;
	  pDesc->constraint.string_list = colormode_list;
	  pDesc->cap = SANE_CAP_SOFT_SELECT | SANE_CAP_SOFT_DETECT;
	  pVal->s = strdup (colormode_list[0]);
	  break;

	case optDepth:
	  pDesc->name = SANE_NAME_BIT_DEPTH;
	  pDesc->title = SANE_TITLE_BIT_DEPTH;
	  pDesc->desc = SANE_DESC_BIT_DEPTH;
	  pDesc->type = SANE_TYPE_INT;
	  pDesc->unit = SANE_UNIT_BIT;
	  pDesc->constraint_type = SANE_CONSTRAINT_WORD_LIST;
	  pDesc->constraint.word_list = depth_list;
	  pDesc->cap = SANE_CAP_SOFT_SELECT | SANE_CAP_SOFT_DETECT;
	  pVal->w = depth_list[1];
	  break;

	case optThreshold:
	  pDesc->name = SANE_NAME_THRESHOLD;
	  pDesc->title = SANE_TITLE_THRESHOLD;
	  pDesc->desc = SANE_DESC_THRESHOLD;
	  pDesc->type = SANE_TYPE_INT;
	  pDesc->unit = SANE_UNIT_NONE;
	  pDesc->constraint_type = SANE_CONSTRAINT_RANGE;
	  pDesc->constraint.range = &rangeThreshold;
	  pDesc->cap |=
	    SANE_CAP_SOFT_SELECT | SANE_CAP_SOFT_DETECT | SANE_CAP_INACTIVE;
	  pVal->w = 0x80;
	  break;
	}
    }
}

static SANE_Int
_ReportDevice (TScannerModel * pModel, const char *pszDeviceName)
{
  TDevListEntry *pNew, *pDev;

  DBG (2, "> _ReportDevice:\n");

  pNew = malloc (sizeof (TDevListEntry));
  if (!pNew)
    return -1;

  /* add new element to the end of the list */
  if (_pFirstSaneDev == NULL)
    {
      _pFirstSaneDev = pNew;
    }
  else
    {
      /* Add at the end of existing list */
      for (pDev = _pFirstSaneDev; pDev->pNext; pDev = pDev->pNext)
	{
	  ;
	}
      pDev->pNext = pNew;
    }

  /* fill in new element */
  pNew->pNext = 0;
  /* we use devname to avoid having to free a const
   * pointer */
  pNew->devname = (char *) strdup (pszDeviceName);
  pNew->dev.name = pNew->devname;
  (char *) strdup (pszDeviceName);
  pNew->dev.vendor = pModel->pszVendor;
  pNew->dev.model = pModel->pszName;
  pNew->dev.type = "flatbed scanner";

  iNumSaneDev++;

  return 0;
}

static SANE_Status
attach_one_device (SANE_String_Const devname)
{
  DBG (2, "> attach_one_device\n");

  _ReportDevice (&Model_HP39xx, devname);

  return SANE_STATUS_GOOD;
}

/* Sane default functions */

SANE_Status
sane_init (SANE_Int * version_code, SANE_Auth_Callback authorize)
{
  FILE *conf_fp;		/* Config file stream  */
  SANE_Char line[PATH_MAX];
  SANE_Char *str = NULL;
  SANE_String_Const proper_str;
  SANE_Int nline = 0;

  /* Initialize debug */
  DBG_INIT ();

  DBG (2, "> sane_init\n");

  authorize = authorize;	/* silence gcc */

  /* Initialize usb */
  sanei_usb_init ();

  /* Parse config file */
  conf_fp = sanei_config_open (HP3900_CONFIG_FILE);
  if (conf_fp)
    {
      while (sanei_config_read (line, sizeof (line), conf_fp))
	{
	  nline++;
	  if (str)
	    free (str);

	  proper_str = sanei_config_get_string (line, &str);

	  /* Discards white lines and comments */
	  if ((str != NULL) && (proper_str != line) && (str[0] != '#'))
	    {
	      /* If line's not blank or a comment, then it's the device
	       * filename or a usb directive. */
	      sanei_usb_attach_matching_devices (line, attach_one_device);
	    }
	}
      fclose (conf_fp);
    }
  else
    {
      /* By default add hp3970c scanner */
      sanei_usb_attach_matching_devices ("usb 0x03f0 0x2305",
					 attach_one_device);
    }

  /* Return backend version */
  if (version_code != NULL)
    *version_code = SANE_VERSION_CODE (V_MAJOR, V_MINOR, 0);

  return SANE_STATUS_GOOD;
}

SANE_Status
sane_get_devices (const SANE_Device *** device_list, SANE_Bool local_only)
{
  TDevListEntry *pDev;
  SANE_Int i;

  DBG (2, "> sane_get_devices\n");

  local_only = local_only;

  if (_pSaneDevList)
    free (_pSaneDevList);

  _pSaneDevList = malloc (sizeof (*_pSaneDevList) * (iNumSaneDev + 1));
  if (!_pSaneDevList)
    return SANE_STATUS_NO_MEM;

  i = 0;
  for (pDev = _pFirstSaneDev; pDev; pDev = pDev->pNext)
    _pSaneDevList[i++] = &pDev->dev;

  _pSaneDevList[i++] = 0;	/* last entry is 0 */
  *device_list = _pSaneDevList;

  return SANE_STATUS_GOOD;
}

SANE_Status
sane_open (SANE_String_Const name, SANE_Handle * h)
{
  TScanner *s;
  SANE_Status result;
  SANE_Word vendor, product;

  DBG (2, "> sane_open\n");

  /* check the name */
  if (strlen (name) == 0)
    {
      /* default to first available device */
      name = _pFirstSaneDev->dev.name;
    }

  result = sanei_usb_open (name, &usbdn);
  if (result != SANE_STATUS_GOOD)
    return result;

  if (sanei_usb_get_vendor_product (usbdn, &vendor, &product) ==
      SANE_STATUS_GOOD)
    Scanner_Model = get_ScannerModel (product, vendor);
  else
    Scanner_Model = HP3970;

  s = malloc (sizeof (TScanner));
  if (!s)
    return SANE_STATUS_NO_MEM;

  memset (s, 0, sizeof (TScanner));	/* Clear everything to zero */

  /* Initializing RTS */
  Init_Vars ();

  if (HP_OP_INIT_SCANNER () != OK)
    {
      free ((void *) s);
      return SANE_STATUS_INVAL;
    }

  Set_Constrains (Scanner_Model);

  /* Silencing unused functions */
  Silent_Compile ();

  _InitOptions (s);
  *h = s;

  return SANE_STATUS_GOOD;
}

void
sane_close (SANE_Handle h)
{
  DBG (2, "- sane_close...\n");

  h = h;			/* silence gcc */
  if (Image != NULL)
    {
      free (Image);
      Image = NULL;
    }

  if (Rest != NULL)
    {
      free (Rest);
      Rest = NULL;
    }

  HP_OP_END_SCANNER ();
  sanei_usb_close (usbdn);
}

void
sane_exit (void)
{
  TDevListEntry *pDev, *pNext;

  /*printf("- sane_exit...\n"); */

  /* free device list memory */
  if (_pSaneDevList)
    {
      for (pDev = _pFirstSaneDev; pDev; pDev = pNext)
	{
	  pNext = pDev->pNext;
	  free (pDev->devname);
	  /* pDev->dev.name is the same pointer that pDev->devname */
	  free (pDev);
	}
      _pFirstSaneDev = 0;
      free (_pSaneDevList);
      _pSaneDevList = 0;
    }

}

const SANE_Option_Descriptor *
sane_get_option_descriptor (SANE_Handle h, SANE_Int n)
{
  TScanner *s;

  DBG (2, "- SANE_Option_Descriptor...\n");

  if ((n < optCount) || (n >= optLast))
    return NULL;

  s = (TScanner *) h;
  return &s->aOptions[n];
}

SANE_Status
sane_control_option (SANE_Handle h, SANE_Int n, SANE_Action Action,
		     void *pVal, SANE_Int * pInfo)
{
  TScanner *s;
  SANE_Int info;

  DBG (2, "> sane_control_option\n");

  s = (TScanner *) h;
  info = 0;

  switch (Action)
    {
    case SANE_ACTION_GET_VALUE:
      switch (n)
	{
	  /* Get options of type SANE_Word */
	case optBRX:
	case optTLX:
	case optBRY:
	case optTLY:
	  *(SANE_Word *) pVal = s->aValues[n].w;
	  break;
	case optCount:
	case optDPI:
	  *(SANE_Word *) pVal = s->aValues[n].w;
	  break;
	  /* Get options of type SANE_Word array */
	case optGammaTableRed:
	case optGammaTableGreen:
	case optGammaTableBlue:
	  memcpy (pVal, s->aValues[n].wa, s->aOptions[n].size);
	  break;
	case optType:
	  strcpy (pVal, s->aValues[n].s);
	  break;
	case optColorMode:
	  strcpy (pVal, s->aValues[n].s);
	  break;
	case optDepth:
	  *(SANE_Int *) pVal = s->aValues[n].w;
	  break;
	case optThreshold:
	  *(SANE_Int *) pVal = s->aValues[n].w;
	  break;
	}
      break;
    case SANE_ACTION_SET_VALUE:
      if (s->fScanning)
	return SANE_STATUS_INVAL;
      switch (n)
	{
	case optCount:
	  return SANE_STATUS_INVAL;
	  break;
	case optBRX:
	case optTLX:
	case optBRY:
	case optTLY:
	  info |= SANE_INFO_RELOAD_PARAMS;
	  s->aValues[n].w = *(SANE_Word *) pVal;
	  break;
	case optDPI:
	  info |= SANE_INFO_RELOAD_PARAMS;
	  s->aValues[n].w = *(SANE_Word *) pVal;
	  break;
	case optGammaTableRed:
	case optGammaTableGreen:
	case optGammaTableBlue:
	  memcpy (s->aValues[n].wa, pVal, s->aOptions[n].size);
	  break;
	case optType:
	  if (strcmp (s->aValues[n].s, pVal) != 0)
	    {
	      struct st_coords *coords;
	      SANE_Int source;

	      if (s->aValues[n].s)
		free (s->aValues[n].s);
	      s->aValues[n].s = strdup (pVal);

	      source = Get_Source (s->aValues[optType].s);
	      coords = Constrains_Get (source);
	      if (coords != NULL)
		{
		  Range_GetXY (source, 0);
		  Range_GetXY (source, 1);
		  s->aValues[optTLX].w = 0;
		  s->aValues[optTLY].w = 0;
		  s->aValues[optBRX].w = coords->width;
		  s->aValues[optBRY].w = coords->height;
		}

	      info |= SANE_INFO_RELOAD_PARAMS | SANE_INFO_RELOAD_OPTIONS;
	    }
	  break;
	case optColorMode:
	  if (strcmp (s->aValues[n].s, pVal) != 0)
	    {
	      if (s->aValues[n].s)
		free (s->aValues[n].s);
	      s->aValues[n].s = strdup (pVal);
	      if (Get_Colormode (s->aValues[n].s) == CM_LINEART)
		s->aOptions[optThreshold].cap &= ~SANE_CAP_INACTIVE;
	      else
		s->aOptions[optThreshold].cap |= SANE_CAP_INACTIVE;
	      info |= SANE_INFO_RELOAD_PARAMS | SANE_INFO_RELOAD_OPTIONS;
	    }
	  break;
	case optDepth:
	  info |= SANE_INFO_RELOAD_PARAMS;
	  s->aValues[n].w = *(SANE_Word *) pVal;
	  break;
	case optThreshold:
	  info |= SANE_INFO_RELOAD_PARAMS;
	  s->aValues[n].w = *(SANE_Int *) pVal;
	  break;
	}
      if (pInfo != NULL)
	*pInfo = info;
      break;
    case SANE_ACTION_SET_AUTO:
      return SANE_STATUS_UNSUPPORTED;
    default:
      return SANE_STATUS_INVAL;
    }

  return SANE_STATUS_GOOD;
}

SANE_Status
sane_get_parameters (SANE_Handle h, SANE_Parameters * p)
{
  TScanner *s;
  struct st_coords coords;
  SANE_Int res, source, bitmode;

  DBG (2, "> sane_get_parameters\n");

  s = (TScanner *) h;

  /* first do some checks */

  /* Get Scan type */
  source = Get_Source (s->aValues[optType].s);

  /* Get depth */
  bitmode = s->aValues[optDepth].w;

  /* Get resolution */
  res = s->aValues[optDPI].w;

  /* Get image coordinates in milimeters */
  coords.left = s->aValues[optTLX].w;
  coords.top = s->aValues[optTLY].w;
  coords.width = s->aValues[optBRX].w;
  coords.height = s->aValues[optBRY].w;

  /* validate coords */
  if (Translate_coords (&coords) != SANE_STATUS_GOOD)
    return SANE_STATUS_INVAL;

  Set_Coordinates (source, res, &coords);

  /* return the data */
  p->format = SANE_FRAME_RGB;
  p->last_frame = SANE_TRUE;

  p->depth = bitmode;

  p->lines = coords.height;
  p->pixels_per_line = coords.width;
  p->bytes_per_line = coords.width * dot_size (h);

  return SANE_STATUS_GOOD;
}

SANE_Status
sane_start (SANE_Handle h)
{
  SANE_Status rst = SANE_STATUS_GOOD;
  TScanner *s;

  struct st_coords coords;
  SANE_Int res, source, colormode, bitmode;

  DBG (2, "> sane_start\n");

  s = (TScanner *) h;

  /* first do some checks */

  /* Get Scan type */
  source = Get_Source (s->aValues[optType].s);

  /* Get color mode */
  colormode = Get_Colormode (s->aValues[optColorMode].s);

  /* This will be removed after debugging driver at all */
  con_colormode = colormode;
  colormode = 0;

  /* Get threshold */
  con_threshold = s->aValues[optThreshold].w;

  /* Get depth */
  bitmode = s->aValues[optDepth].w;

  /* Get resolution */
  res = s->aValues[optDPI].w;

  /* Get image coordinates in milimeters */
  coords.left = s->aValues[optTLX].w;
  coords.top = s->aValues[optTLY].w;
  coords.width = s->aValues[optBRX].w;
  coords.height = s->aValues[optBRY].w;

  /* Validate coords */
  rst = Translate_coords (&coords);

  if (rst == SANE_STATUS_GOOD)
    {
      rst = SANE_STATUS_INVAL;

      /* Generate fake gamma tables if depth == 8 */
      Gamma_Create8bit (s);

      /* Stop previusly started scan */
      StopScan ();

      s->ScanParams.scantype = source;
      s->ScanParams.colormode = colormode;
      s->ScanParams.resolution_x = res;
      s->ScanParams.resolution_y = res;
      memcpy (&s->ScanParams.coords, &coords, sizeof (struct st_coords));
      s->ScanParams.depth = bitmode;
      s->ScanParams.channel = 0;

      Set_Coordinates (source, res, &s->ScanParams.coords);

      if (RT_OP_SET_PARAMETER (&s->ScanParams) == OK)
	{
	  /* Start scanning proccess */
	  if (RT_OP_START_SCAN (0, 0) == OK)
	    {
	      /* Allocate buffer to read one line */
	      mylin = 0;
	      if (Image != NULL)
		{
		  free (Image);
		  Image = NULL;
		}

	      if (Rest != NULL)
		{
		  free (Rest);
		  Rest = NULL;
		  Rest_amount = 0;
		}

	      Image =
		(SANE_Byte *) malloc (bytesperline * sizeof (SANE_Byte));
	      if (Image != NULL)
		{
		  Rest =
		    (SANE_Byte *) malloc (bytesperline * sizeof (SANE_Byte));
		  if (Rest == NULL)
		    {
		      free (Image);
		      Image = NULL;
		    }
		  else
		    rst = SANE_STATUS_GOOD;
		}
	    }
	}
    }

  if (rst == SANE_STATUS_INVAL)
    DBG (2, "> sane_start: SANE_STATUS_INVAL\n");

  return rst;
}

SANE_Status
sane_read (SANE_Handle h, SANE_Byte * buf, SANE_Int maxlen, SANE_Int * len)
{
  SANE_Byte *buffer = (SANE_Byte *) buf;
  TScanner *s;
  SANE_Int thwidth;
  SANE_Int transferred;
  SANE_Int bufflength;

  DBG (2, "> sane_read\n");

  s = (TScanner *) h;

  /* nothing has been read at the moment */
  *len = 0;

  /* if we read all the lines return EOF */
  if ((mylin == s->ScanParams.coords.height) || (Reading.Cancel == TRUE))
    {
      StopScan ();
      if (Image != NULL)
	{
	  free (Image);
	  free (Rest);
	  Image = NULL;
	  Rest = NULL;
	}
      DBG (2, "> sane_read: All lines read\n");
      return SANE_STATUS_EOF;
    }

  /* read as many lines the buffer may contain and while there are lines to be read */
  thwidth = s->ScanParams.coords.width * dot_size (h);
  while ((*len < maxlen) && (mylin < s->ScanParams.coords.height))
    {
      /* Is there any data waiting for being passed ? */
      if (Rest_amount != 0)
	{
	  bufflength = min (maxlen - *len, Rest_amount);
	  memcpy (buffer, Rest, bufflength);
	  *len += bufflength;
	  buffer += bufflength;
	  Rest_amount -= bufflength;
	  if (Rest_amount == 0)
	    mylin++;
	}
      else
	{
	  /* Read from scanner */
	  if (Read_Image (bytesperline, Image, &transferred) != OK)
	    {
	      DBG (2, "> sane_read: Read_Image returned ERROR\n");
	      return SANE_STATUS_EOF;
	    }

	  if (transferred != 0)
	    {
	      bufflength = min (maxlen - *len, thwidth);
	      memcpy (buffer, Image, bufflength);
	      *len += bufflength;
	      buffer += bufflength;

	      if (bufflength < thwidth)
		{
		  Rest_amount = thwidth - bufflength;
		  memcpy (Rest, Image + bufflength, Rest_amount);
		}
	      else
		mylin++;
	    }
	  else
	    break;
	}
    }

  if (*len > 0)
    {
      Gamma_Apply (s, buf, *len, s->ScanParams.depth);

      if (con_colormode == CM_GRAY)
	Color_to_Gray (buf, *len, s->ScanParams.depth);
      else if (con_colormode == CM_LINEART)
	Color_to_Lineart (buf, *len, s->ScanParams.depth, con_threshold);
    }

  return SANE_STATUS_GOOD;
}

void
sane_cancel (SANE_Handle h)
{
  DBG (2, "> sane_cancel\n");

  Reading.Cancel = TRUE;

  /* Change this in the future */
  h = h;			/* silence gcc */
  /*StopScan(); */
  /*if (Image != NULL)
     {
     free(Image);
     Image = NULL;
     } */
}

SANE_Status
sane_set_io_mode (SANE_Handle handle, SANE_Bool non_blocking)
{
  DBG (2, "> sane_set_io_mode\n");

  handle = handle;		/* silence gcc */
  non_blocking = non_blocking;	/* silence gcc */

  return SANE_STATUS_UNSUPPORTED;
}

SANE_Status
sane_get_select_fd (SANE_Handle handle, SANE_Int * fd)
{
  DBG (2, "> sane_get_select_fd\n");

  handle = handle;		/* silence gcc */
  fd = fd;			/* silence gcc */

  return SANE_STATUS_UNSUPPORTED;
}
