 /* sane - Scanner Access Now Easy.
   Copyright (C) 2003 Johannes Hub (JohannesHub@t-online.de)

   This file was initially copied from the hp3300 backend.
   This file is part of the SANE package.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

   As a special exception, the authors of SANE give permission for
   additional uses of the libraries contained in this release of SANE.

   The exception is that, if you link a SANE library with other files
   to produce an exutable, this does not by itself cause the
   resulting executable to be covered by the GNU General Public
   License.  Your use of that executable is in no way restricted on
   account of linking the SANE library code into it.

   This exception does not, however, invalidate any other reasons why
   the executable file might be covered by the GNU General Public
   License.

   If you submit changes to SANE to the maintainers to be included in
   a subsequent release, you agree by submitting the changes that
   those changes may be distributed with this exception intact.

   If you write modifications of your own for SANE, it is your choice
   whether to permit this exception to apply to your modifications.
   If you do not wish that, delete this exception notice.
*/

/*
    Concept for a backend for scanners based on the RTS88xx chipset,
    such as HP4400C, HP4470C.
    Parts of this source were inspired by other backends.

		History:

		Version 0.18  21.11.04 13.alpha,
				- source sorted,
				- now only SANEI_USB_SUPPORT for a better overview(xfermodules removed)
				- read and verify the MainBoardID 
		Version 0.17p 02.11.04 12.alpha, source sorted, little fixes, SANEI_USB_SUPPORT implemented
		Version 0.17p 02.11.04 12.alpha, source sourted, little fixes, SANEI_USB_SUPPORT implemented
		Version 0.17b 30.03.04 10.alpha, little fixes and libusb implemented
		Version 0.17  09.03.04 9. alpha, HP3500 included
		Version 0.16  06.02.04 8. alpha, wait counting on LCD
		Version 0.15a 29.01.04 7. alpha, CCD switch moved to config file
		Version 0.15  11.01.04 6. alpha, a second CCD implemented
		Version 0.13a 21.11.04 4. alpha, an little fix included
		Version 0.12  22.10.03 third alpha, Backend name changed to HP_RTS88xx
		Version 0.11  30.08.03 second alpha
		Version 0.10  19.07.03 first alpha
*/

/*
		Provides a simple interface to read and write data from the scanner,
		without any knowledge whether it's a parallel or USB scanner

		enable DEDUG output:
		use a nomal shell (konsole) and type
		export SANE_DEBUG_HP_RTS88XX=32
		export XSANE_DEBUG=100
		xsane
*/


#include <unistd.h>    /* read, open, write */
#include <fcntl.h>     /* open */
#include <stdio.h>     /* printf */
#include <errno.h>     /* better error reports */
#include <string.h>    /* better error reports */
#include <stdlib.h>    /* malloc(), free() on FreeBSD */
#include "../include/sane/sanei_usb.h"

#include "hp_rts_xfer.h"

#ifdef STANDALONE
	#include "../include/sane/config.h"
	#include "hp_rts_35x0c.h"
	#include "hp_rts_44x0c.h"
#endif


static SANE_Int _fDeviceFound;
static TScannerModel *_pModel;
static TDevListEntry *_pFirstSaneDev = 0;
static SANE_Int iNumSaneDev = 0;
static const SANE_Device **_pSaneDevList = 0;

/* list of supported models */
TScannerModel ScannerModels[] = {
	{"Hewlett-Packard",	"UnknownModel",		0x3F0,	0x0000, eUnknownModel},
	{"Hewlett-Packard",	"ScanJet 3500C",	0x3F0,	0x2205, eHp3500c}, /*RTS8801C2*/
#ifdef DEBUG_HP3500
	{"Hewlett-Packard",	"ScanJet 3530C",	0x3F0,	0x0805, eHp3530c}, /*RTS8801C2*/
#else
	{"Hewlett-Packard",	"ScanJet 3530C",	0x3F0,	0x2005, eHp3530c}, /*RTS8801C2*/
#endif
	{"Hewlett-Packard",	"ScanJet 3570C",	0x3F0,	0x2005, eHp3570c}, /*RTS8801C2 is the same as 3500 */
	{"Hewlett-Packard",	"ScanJet 4400C",	0x3F0,	0x0705, eHp4400c}, /*RTS8891*/
	{"Hewlett-Packard",	"ScanJet 4470C",	0x3F0,	0x0805, eHp4470c}, /*RTS8891*/
#ifdef ENABLE_VI8920
	{"Visioneer",		"Onetouch 8920 USB",	0x0461,	0x0371, eVi8920}, /*RTS8801C2-004*/
#endif
#if 0
	{"Visioneer",		"Onetouch 8900 USB",	0x0461,	0x0371, eVi8900}, /*RTS8801C2-004*/
	{"Visioneer",		"Onetouch 8700 USB",	0x0461,	0x0371, eVi8700}, /*RTS8801C2-004*/
	{"Visioneer",		"OneTouch 5800 USB",	0x04a7,	0x0226, eVi5800}, /*RTS8801C2-004*/
	{"Visioneer",		"OneTouch 5300 USB",	0x04a7,	0x0226, eVi5300}, /*RTS8801C2-004*/
	{"Visioneer",		"OneTouch 4800 USB",	0x04a7,	0x0224, eVi4800}, /*RTS8801B-0001*/
	{"Vantas",			"3000",								0x04a7,	0x0224, eVa3000}, /*RTS8801B-0001*/
	{"Microtek",		"Scanport 3000",			0x04a7,	0x0224, eMi3000}, /*RTS8801B-0001*/
	{"Plustek",			"OpticPro S6",				0x0???,	0x????, ePlS6}, 	/*RTS8801D*/
	{"Prolink",			"Winscan Pro 2448U",	0x06dc,	0x0014, ePrlS6}, 	/*RTS8801C2-006*/
#endif
/* last entry all zeros */
  {0, 0, 0, 0, 0}
};


/****************************************************************************/
/* utility function to show a hexdump of a buffer */
void Hp_rts_DumpHex(SANE_Byte *pabData, SANE_Int iLen, SANE_Int iWidth, SANE_Int withLf)
/****************************************************************************/
{
  SANE_Int i;

  printf("    ");
  for (i = 0; i < iWidth ; i++) {
    printf(" %02X", i);
  }
  for (i = 0; i < iLen; i++) {
    if ((i % iWidth) == 0) {
      if (withLf) printf("\n");
      printf("%04X", i);
    }
    printf(" %02X", pabData[i]);
  }
  if (withLf) printf("\n");
}

/****************************************************************************/
/* utility function to show a bit dump of a byte */
void Hp_rts_DumpBits(SANE_Byte reg, SANE_Byte pabData)
/****************************************************************************/
{
	SANE_Int i = 0;

	printf("RegisterDump : Register %x will be set to the bits ", reg);
	for (i = 0; i < 8; i++) {
		if (pabData & 0x80)
			printf("1");
		else
			printf("0");
		if (i==3) printf(" ");
		pabData = pabData << 1;
	}
	printf("\n");
}

/****************************************************************************
  Hp_rts_MatchUsbDevice
  ==============
    Matches a given USB vendor and product id against a list of
    supported scanners.

  IN  iVendor   USB vendor ID
      iProduct  USB product ID
  OUT *ppModel  Pointer to TScannerModel structure

  Returns SANE_TRUE if a matching USB scanner was found
*/
SANE_Bool Hp_rts_MatchUsbDevice(SANE_Int iVendor, SANE_Int iProduct, TScannerModel **ppModel)
/****************************************************************************/
{
	TScannerModel *pModels = ScannerModels;

	DBG(DBG_MSG,"Hp_rts_MatchUsbDevice :Matching USB device 0x%04X-0x%04X ... \n", iVendor, iProduct);
	while (pModels->pszName != NULL)
	{
		if ((pModels->iVendor == iVendor) && (pModels->iProduct == iProduct))
		{
			DBG(DBG_MSG,"Hp_rts_MatchUsbDevice : found %s %s\n", pModels->pszVendor, pModels->pszName);
			*ppModel = pModels;
			return SANE_TRUE;
		}
		/* next model to match */
		pModels++;
	}
	DBG(DBG_MSG,"Hp_rts_MatchUsbDevice : nothing found\n");
	return SANE_FALSE;
}



/************************************************************************
  Public functions
************************************************************************/


/*static TFnReportDevice *_pfnReportDevice;*/
static TScannerModel *_pModel;

/****************************************************************************/
static int
_ReportDevice (TScannerModel * pModel, SANE_Char *pszDeviceName)
/****************************************************************************/
{
  TDevListEntry *pNew, *pDev;
#if 0
  DBG (DBG_MSG, "_ReportDevice '%s'", pszDeviceName);
#endif 
  pNew = malloc (sizeof (TDevListEntry));
  if (!pNew)
    {
      DBG (DBG_ERR, "_ReportDevice: no mem\n");
      return -1;
    }

  /* add new element to the end of the list */
  if (_pFirstSaneDev == 0)
    {
      _pFirstSaneDev = pNew;
    }
  else
    {
      for (pDev = _pFirstSaneDev; pDev->pNext; pDev = pDev->pNext)
	{
	  ;
	}
      pDev->pNext = pNew;
    }

  /* fill in new element */
  pNew->pNext = 0;
  pNew->dev.name = strdup (pszDeviceName);
  pNew->dev.vendor = pModel->pszVendor;
  pNew->dev.model = pModel->pszName;
  pNew->dev.type = "flatbed scanner";

  iNumSaneDev++;

  return 0;
}

/* callback for sanei_usb_attach_matching_devices */
/****************************************************************************/
static SANE_Status
_AttachUsb (SANE_String_Const devname)
/****************************************************************************/
{
  DBG (DBG_MSG, "_AttachUsb: found %s", devname);

	_fDeviceFound ++;
  _ReportDevice (_pModel, (SANE_Char *) devname);
/*  _pfnReportDevice (_pModel, (SANE_Char *) devname);*/

  return SANE_STATUS_GOOD;
}

/****************************************************************************
  Hp_rts_XferInit
  ===============
    Probes all registered modules to see if it can be used to access a
    compatible scanner. The first module that supports the scanner is
    used.

  Returns the handle to a TXferDev structure which was used to find the
  scanner, returns 0 or negative otherwise.
*/
SANE_Int Hp_rts_XferInit(/*EScannerModel *peModel*/void)
/****************************************************************************/
{
	TScannerModel *pModels = ScannerModels;

	DBG (DBG_MSG,"Hp_rts_XferInit ....\n");

  sanei_usb_init ();
	_fDeviceFound = 0;
/*	_pfnReportDevice = _ReportDevice;*/

	/* loop over all scanner models */
	while (pModels->pszName != NULL)
	{
		DBG (DBG_MSG, "Looking for %s...", pModels->pszName);
		_pModel = pModels;
		if (sanei_usb_find_devices ((SANE_Int) pModels->iVendor,
					(SANE_Int) pModels->iProduct,
					_AttachUsb) != SANE_STATUS_GOOD)
		{
			DBG (DBG_ERR, "Error invoking sanei_usb_find_devices\n");
			return SANE_FALSE;
		}
#if 0
		DBG(DBG_MSG, "Hp_rts_XferInit: found device ");
		if (pModels->eModel != eUnknownModel){
			_AttachUsb("libusb:001:002*");
		}
#endif
		DBG (DBG_MSG, "\n");
		pModels++;
	}

	if (_fDeviceFound)
		return SANE_TRUE;
	else
		return SANE_FALSE;
}


/****************************************************************************/
SANE_Int Hp_rts_XferOpen (SANE_String_Const devname,EScannerModel *peModel)
/****************************************************************************/
{
	SANE_Status status;
	SANE_Word vendor, product;
	int fd;
	TScannerModel *pModel;

	DBG (DBG_MSG, "Hp_rts_XferOpen, trying to open %s...\n", devname);

	status = sanei_usb_open (devname, &fd);
	if (status != SANE_STATUS_GOOD)
	{
		DBG (DBG_ERR, "Hp_rts_XferOpen: sanei_usb_open failed %s\n", devname);
		return -1;
	}
#if 0
	DBG (DBG_MSG, "Hp_rts_XferOpen, trying to get sanei_usb_get_vendor_product for fd %d\n", (int) fd);
#endif
	status = sanei_usb_get_vendor_product (fd, &vendor, &product);
	if (status == SANE_STATUS_GOOD)
	{
#if 0
		DBG (DBG_MSG, "Hp_rts_XferOpen, trying Hp_rts_MatchUsbDevice\n");
#endif		
		if (Hp_rts_MatchUsbDevice (vendor, product, &pModel)== SANE_TRUE)
		{
#if 0
			DBG (DBG_MSG, "pModel->eModel = %d\n",pModel->eModel);
#endif			
			*peModel = pModel->eModel;
		}else
		{
			DBG (DBG_ERR, "Hp_rts_XferOpen: Hp_rts_MatchUsbDevice failed %s\n", devname);
			sanei_usb_close (fd);
			return -1;
		}
	}else
	{
		DBG (DBG_ERR, "Hp_rts_XferOpen: sanei_usb_get_vendor_product failed %s\n", devname);
		sanei_usb_close (fd);
		return -1;
	}
#if 0
	DBG (DBG_MSG, "handle = %d\n", (int) fd);
#endif
	return fd;
}

/****************************************************************************/
SANE_Int Hp_rts_XferExit(SANE_Int iHandle)
/****************************************************************************/
{

	if (iHandle < 0) {
		return(-1);
	}
	/* close usb device */
	sanei_usb_close (iHandle);
	return SANE_TRUE;
}


/****************************************************************************/
SANE_Int Hp_rts_BulkWrite (SANE_Int iHandle, SANE_Byte reg, SANE_Byte *pabBuf,
                       SANE_Int iSize, SANE_Int wHeader)
/****************************************************************************/
{
	static SANE_Byte init[]={0x88,0,0x00,0x01};
	SANE_Char *request;     /* Compose a request string. */
	SANE_Int size;

#ifdef USBDEBUG
	DBG(DBG_USB,"UsbWriteBulk ...\n");
#endif
	if (iHandle == -1) {
		return(SANE_FALSE);
	}

	if (wHeader){
		request=malloc(iSize + 4);     /* Compose a request string. */
		init[1]=(SANE_Byte)reg;
		init[2]=(SANE_Byte)(iSize>>8); /* split count over these bytes.*/
		init[3]=(SANE_Byte)iSize;
		memcpy(request,init,4);        /* stick them together */
		memcpy(request+4,pabBuf,iSize);
		size = iSize + 4;
#ifdef USBDEBUG
		SANE_Byte *tmpw=request;
		SANE_Int tmp_size=size;
		fprintf(stderr,"WB1 transfer type=bulk size=%d dir=OUT\n",tmp_size);
		while(tmp_size--) fprintf(stderr,"%02hhx ",*tmpw++);
		fprintf(stderr,"\n");
#endif
		if (sanei_usb_write_bulk(iHandle, request,&size) != SANE_STATUS_GOOD) {
			DBG(DBG_ERR, "UsbWriteBulk: ERROR: Bulk write failed\n");
			return SANE_FALSE;
		}
		free(request);                  /* clean up */
	}else
	{
#ifdef USBDEBUG
		SANE_Byte *tmpw=pabBuf;
		SANE_Int tmp_size=iSize;
		fprintf(stderr,"WB2 transfer type=bulk size=%d dir=OUT\n",tmp_size);
		while(tmp_size--) fprintf(stderr,"%02hhx ",*tmpw++);
		fprintf(stderr,"\n");
#endif
    if (sanei_usb_write_bulk(iHandle,pabBuf,&iSize) != SANE_STATUS_GOOD) {
			DBG(DBG_ERR, "UsbWriteBulk: ERROR: Bulk write failed\n");
			return SANE_FALSE;
    }
  }
  return SANE_TRUE;
}


/****************************************************************************/
SANE_Int Hp_rts_BulkRead(SANE_Int iHandle, SANE_Byte reg, SANE_Byte *pabBuf,
                     SANE_Int iSize, SANE_Int wWrite)
/****************************************************************************/
{
	static SANE_Byte init[]={0x80,0,0x00,0x01};
	size_t size;

#ifdef USBDEBUG
	DBG(DBG_USB,"UsbReadBulk ...\n");
#endif
	if (iHandle == -1) {
		return SANE_FALSE;
	}

	if (wWrite){
#ifdef USBDEBUG
		DBG(DBG_USB,"UsbReadBulk with write\n");
#endif
		init[1] = reg;
		init[2] = (iSize>>8) & 0xFF; /*split count over these bytes.*/
		init[3] = (iSize) & 0xFF;
		size = 4;
#ifdef USBDEBUG
		SANE_Byte *tmpw=init;
		SANE_Int tmp_size=size;
		fprintf(stderr,"RR transfer type=bulk size=%d dir=OUT\n",tmp_size);
		while(tmp_size--) fprintf(stderr,"%02hhx ",*tmpw++);
		fprintf(stderr,"\n");
#endif
		if (sanei_usb_write_bulk(iHandle,(unsigned char *) init,&size) != SANE_STATUS_GOOD) {
			DBG(DBG_ERR,"UsbReadBulk: ERROR: Bulk write failed\n");
			return SANE_FALSE;
		}
	}
#ifdef USBDEBUG
	DBG(DBG_USB,"UsbReadBulk, now read ...\n");
#endif
	size = iSize;
	if (sanei_usb_read_bulk(iHandle, pabBuf,&size) != SANE_STATUS_GOOD) {
			DBG(DBG_ERR,"UsbReadBulk: ERROR: Bulk read failed\n");
			return SANE_FALSE;
	}
#ifdef USBDEBUG
	SANE_Byte *tmpr=pabBuf;
	SANE_Int tmp_size=size;
	fprintf(stderr,"RR transfer type=bulk size=%d dir=IN\n",tmp_size);
	while(tmp_size--) fprintf(stderr,"%02hhx ",*tmpr++);
	fprintf(stderr,"\n");
#endif
	return SANE_TRUE;
}


/****************************************************************************/
SANE_Int Hp_rts_RegWrite(SANE_Int iHandle, SANE_Byte bReg, SANE_Byte bData)
/****************************************************************************/
{
#ifdef USBDEBUG
	DBG(DBG_USB, "SaneiUsbWriteReg...\n");
#endif
	if (Hp_rts_BulkWrite(iHandle,bReg,&bData,1, SANE_TRUE))
		return(SANE_TRUE);
	else
		return(SANE_FALSE);
}


/****************************************************************************/
SANE_Int Hp_rts_RegRead(SANE_Int iHandle, SANE_Byte bReg, SANE_Byte *pbData)
/****************************************************************************/
{
#ifdef USBDEBUG
	DBG(DBG_USB, "SaneiUsbReadReg...\n");
#endif
	if (Hp_rts_BulkRead(iHandle,bReg,pbData,1,SANE_TRUE))
		return(SANE_TRUE);
	else
		return(SANE_FALSE);
}


/****************************************************************************/
SANE_Int Hp_rts_BulkReadall(SANE_Int iHandle, SANE_Byte *pabData)
/****************************************************************************/
{
	SANE_Byte init[]={0x80,0,0x00,0xf4};

#ifdef USBDEBUG
	DBG(DBG_USB, "SaneiUsbReadBulk_all_regs...\n");
#endif
	if (iHandle < 0) {
		return -1;
	}

	if (Hp_rts_BulkWrite(iHandle,1,(char *)init,4, SANE_FALSE))
	{
		if (Hp_rts_BulkRead(iHandle,0x81,pabData,192,SANE_FALSE))
		{
			if (Hp_rts_BulkRead(iHandle,0x81,pabData+192,52,SANE_FALSE))
				return(SANE_TRUE);

		} else return(SANE_FALSE);
	} else return(SANE_FALSE);
	return SANE_TRUE;
}

/*************************************************************************
  Returns SANE_TRUE if a known chipset was found. */
SANE_Bool Hp_rts_ProbeRegisters(THWParams *pHWParams)
/*************************************************************************/
{
	SANE_Byte bData1,r;
	SANE_Int iHandle;

	iHandle = pHWParams->iXferHandle;
	bData1 = 0;
	DBG(DBG_MSG, "Hp_rts_ProbeRegisters: iHandle = %d, Probing scanner...\n",(SANE_Int)iHandle);
	Hp_rts_RegRead( iHandle, 0x00, &bData1); /* read register 0x00 */
	if ((bData1 == 0xE5) || (bData1 == 0xF5)) {
		switch (pHWParams->ScannerModel) {
			case eHp3500c: case eHp3530c: case eHp3570c:
#ifdef ENABLE_VI8920
			case eVi8920:
#endif
				return SANE_STATUS_GOOD;
				break;
			case eHp4400c: case eHp4470c:
				Hp_rts_RegRead(iHandle,MAINBOARD_ID, &r);
				if (r == 0)
					return SANE_STATUS_GOOD;
				DBG(DBG_ERR, "Hp_rts_ProbeRegisters: Wrong MainBoardID = %d\n",r);
				DBG(DBG_ERR, "Hp_rts_ProbeRegisters: No valid scanner with RTS88xx chipset found!\n");
				Hp_rts_XferExit(iHandle);
				return SANE_STATUS_IO_ERROR ;
				break;
			default:
				DBG(DBG_ERR, "Hp_rts_ProbeRegisters: ERROR: unknown model! (%d) %s\n",
					(SANE_Int)pHWParams->ScannerModel, ScannerModels[(SANE_Int)pHWParams->ScannerModel].pszName);
			return SANE_STATUS_IO_ERROR;
		}
	}
	else
	{
		DBG(DBG_ERR, "Hp_rts_ProbeRegisters: No valid scanner with RTS88xx chipset found!\n");
		Hp_rts_XferExit(iHandle);
		return SANE_STATUS_IO_ERROR ;
	}
}

/****************************************************************************/
SANE_Int Hp_rts_Set_double_reg(SANE_Int iHandle, SANE_Byte reg,
                           SANE_Byte c1, SANE_Byte c2 )
/****************************************************************************/
{
  SANE_Byte regs[5];

#ifdef USBDEBUG
	DBG(DBG_SCAN, "Hp_rts_Set_double_reg....\n");
#endif
  regs[0] = c1;
  regs[1] = c2;
  return(Hp_rts_BulkWrite (iHandle, reg, regs, 2,SANE_TRUE));
}

/****************************************************************************/
SANE_Int Hp_rts_Read_double_reg(SANE_Int iHandle, SANE_Byte reg, SANE_Byte *pbData )
/****************************************************************************/
{
#ifdef USBDEBUG
	DBG(DBG_SCAN, "Hp_rts_Read_double_reg....\n");
#endif
	return(Hp_rts_BulkRead(iHandle,reg,pbData,2,SANE_TRUE));
}

/****************************************************************************/
/*Return the number of bytes ready for collection                           */
SANE_Word Hp_rts_data_ready(SANE_Int iHandle, SANE_Word *size){
/****************************************************************************/
	static SANE_Byte command[]={0x90,0x00,0x00,0x03};
	SANE_Byte data[5];

#if 0
	DBG(DBG_SCAN, "Hp_rts_data_ready....\n");
#endif
	/* check if the scanner is moving*/
	if (SANE_TRUE /*read_reg(iHandle,MOVE_START_STOP) & 0x08*/)
	{
		Hp_rts_BulkWrite(iHandle,0x00,command,4,SANE_FALSE);
		Hp_rts_BulkRead(iHandle,0x00,data,3,SANE_FALSE);
		*size = data[0];
		*size = *size + (data[1]<<8);
		*size = *size + (data[2]<<16);
		if(*size > 0xe500) *size=0xe500;
		DBG(DBG_SCAN, "     Hp_rts_data_ready....data %x\n",*size);
		if (*size > 0){
			return(SANE_TRUE); }
		else
			return(SANE_FALSE);
	}else return(SANE_FALSE);
}


/****************************************************************************/
SANE_Int Hp_rts_read_data(SANE_Int iHandle, SANE_Word size,SANE_Byte *data) {
/****************************************************************************/
  static SANE_Byte write_command[]={0x91,0x00,0x00,0x00};

  write_command[3]=size & 0xff;
  write_command[2]=(size>>8) & 0xff;
  write_command[1]=(size>>16) & 0xff;

#ifdef USBDEBUG
	DBG(DBG_SCAN, "Hp_rts_read_data..%x\n",size);
#endif
  Hp_rts_BulkWrite(iHandle,0x00,write_command,4,SANE_FALSE);
  Hp_rts_BulkRead(iHandle,0x00,data,size,SANE_FALSE);
  return(SANE_TRUE);
}

/****************************************************************************/
SANE_Bool
Hp_rts_is_moving( SANE_Int iHandle )
/****************************************************************************/
{
	SANE_Byte	r;

	Hp_rts_RegRead(iHandle,REG_MOVE_CONTROL_TEST, &r);
/*	if ( Hp_rts_RegRead(iHandle,REG_MOVE_CONTROL_TEST, &r) < 0)
		return -1;*/
	if (r == 0x08)
		return SANE_TRUE;
	return SANE_FALSE;
}

/****************************************************************************/
SANE_Int
Hp_rts_start_moving( SANE_Int iHandle )
/****************************************************************************/
{
	DBG(DBG_MSG,"Hp_rts_start_moving...");
	if (Hp_rts_RegWrite(iHandle,REG_MOVE_CONTROL_TEST, 2) &&
	    Hp_rts_RegWrite(iHandle,REG_MOVE_CONTROL_TEST, 2) &&
	    Hp_rts_RegWrite(iHandle,REG_MOVE_CONTROL_TEST, 0) &&
	    Hp_rts_RegWrite(iHandle,REG_MOVE_CONTROL_TEST, 0) &&
	    Hp_rts_RegWrite(iHandle,REG_MOVE_CONTROL_TEST, 8) &&
	    Hp_rts_RegWrite(iHandle,REG_MOVE_CONTROL_TEST, 8))
		{
			DBG(DBG_MSG,"Hp_rts_start_moving..TRUE\n");
			return SANE_TRUE;
		}
		else
		{
			DBG(DBG_MSG,"Hp_rts_start_moving..FALSE\n");
			return SANE_FALSE;
		}
}

/****************************************************************************/
SANE_Int
Hp_rts_stop_moving( SANE_Int iHandle )
/****************************************************************************/
{
	DBG(DBG_MSG,"Hp_rts_stop_moving...");
	if (Hp_rts_RegWrite(iHandle,REG_MOVE_CONTROL_TEST, 2) &&
	    Hp_rts_RegWrite(iHandle,REG_MOVE_CONTROL_TEST, 2) &&
	    Hp_rts_RegWrite(iHandle,REG_MOVE_CONTROL_TEST, 0) &&
	    Hp_rts_RegWrite(iHandle,REG_MOVE_CONTROL_TEST, 0))
		{
			DBG(DBG_MSG,"Hp_rts_stop_moving..TRUE\n");
			return SANE_TRUE;
		}
		else
		{
			DBG(DBG_MSG,"Hp_rts_stop_moving..FALSE\n");
			return SANE_FALSE;
		}
}


/****************************************************************************/
/* Write a gamma table */
void
Hp_rts_WriteGammaCalibTable (SANE_Int iHandle,
			const SANE_Int *pabGammaR, const SANE_Int *pabGammaG,  const SANE_Int *pabGammaB)
/****************************************************************************/
{
  SANE_Byte cmd[3];
  SANE_Byte *buffer;
  SANE_Int i, j;

  iHandle = iHandle;

  /* Setup dummy gamma correction table */
  buffer = malloc (2 * 65536);

  cmd[0] = 2;
  cmd[1] = 0;
  cmd[2] = 0;

  for (i = 0; i < 3; i++)
    {
      const SANE_Int *ptr = (i == 0) ? pabGammaR :
	(i == 1) ? pabGammaG : pabGammaB;

      for (j = 0; j < 65536; j++)	/* Truncate integers to shorts */
	buffer[j] = ptr[j];

/*      hp5400_bulk_command_write (iHandle, 0x2A01 + i, cmd, 3, 2 * 65536,
				 65536, (void *) buffer);*/
    }
  free (buffer);

  return;
}

/****************************************************************************
void
Hp_rts_SetDefaultGamma (SANE_Int iHandle)
****************************************************************************
{
  SANE_Int *buffer = malloc (sizeof (SANE_Int) * 65536);
  SANE_Int i;

  for (i = 0; i < 65336; i++)
    buffer[i] = i;

  Hp_rts_WriteGammaCalibTable (iHandle, buffer, buffer, buffer);
}*/

/****************************************************************************/
SANE_Bool
Hp_rts_Check_Moving (SANE_Int iHandle)
/****************************************************************************/
{
	SANE_Int i;
#if 0
	SANE_Word size;
#endif

	DBG(DBG_MSG,"Hp_rts_Check_Moving: wait for moving ...\n");
	i = 0;
	while( !(Hp_rts_is_moving(iHandle)) && ( i < 25 )){
		usleep(10000);
		i++;
		};
	if ( i >= 25 ){
		DBG(DBG_MSG,"Hp_rts_Check_Moving: Time over, no move : %d!, will stop the movement\n",i);
		/*stop it and make sure it stopped!!! */
		Hp_rts_stop_moving(iHandle);
		return(SANE_FALSE);
	}
#if 0
	/* the scanner is now moving. Check it */
	DBG(DBG_MSG,"Hp_rts_Check_Moving: the scanner is now scanning. Check it\n");
	i = 0;
	while( (Hp_rts_data_ready(iHandle,&size) == SANE_FALSE) && ( i < 25 )){
		usleep(10000);
		i++;
	}
	if ( i >= 25 ){
		DBG(DBG_MSG,"Hp_rts_Check_Scanning: Time over, scanner no data : %d, will stop the scanner!\n",i);
		/*stop it and make sure it stopped!!! */
		Hp_rts_stop_moving(iHandle);
		return(SANE_FALSE);
	}
#endif
	return(SANE_TRUE);
}

/****************************************************************************/
SANE_Int
Hp_rts_Read_Sram (SANE_Int iHandle, SANE_Byte *pabData,
                            SANE_Int iSize)
/****************************************************************************/
{
	SANE_Byte init[]={0x81,0x00,0x08,0x18};

	if (iHandle < 0) {
		return(-1);
	}
#ifdef USBDEBUG
	DBG(DBG_MSG,"Hp_rts_Read_Sram...\n");
#endif
	init[2]=(SANE_Byte)(iSize>>8); /*split count over these bytes.*/
	init[3]=(SANE_Byte)iSize;

	if (Hp_rts_BulkWrite(iHandle,0x00,/*(SANE_Byte *)*/init,4,SANE_FALSE)){
#if 0
		DBG(DBG_MSG,"Read SRAM, READ NOW %d bytes\n",iSize);
		if (iSize >= 0x800){
			DBG(DBG_MSG,"Read SRAM, READ the first %d bytes\n",2048);
			if (Hp_rts_BulkRead(iHandle,0x00,(SANE_Byte *)pabData, 2048,SANE_FALSE) == 0){
				DBG(DBG_MSG,"Read SRAM, READ the left %d bytes\n",iSize-2048);
				return (Hp_rts_BulkRead(iHandle,0x00,(SANE_Byte *)pabData, iSize-2048,SANE_FALSE));
			}
		}
		else{
#endif
			return (Hp_rts_BulkRead(iHandle,0x00,(SANE_Byte *)pabData, iSize,SANE_FALSE));
#if 0
		}
#endif
	}
	return(SANE_FALSE);
}

/****************************************************************************/
SANE_Int
Hp_rts_set_value_lsbfirst(	SANE_Byte	*regs,
			SANE_Int		firstreg,
			SANE_Int		totalregs,
			SANE_Word	value)
/****************************************************************************/
{
	while (totalregs--)
	{
		regs[firstreg++] = value & 0xff;
		value >>= 8;
	}
	return 0;
}

/****************************************************************************/
SANE_Int
Hp_rts_set_noscan_distance(			SANE_Byte	*regs,
					SANE_Word	value)
/****************************************************************************/
{
	return Hp_rts_set_value_lsbfirst(regs, 0x60, 2, value);
}

/****************************************************************************/
SANE_Int
Hp_rts_set_total_distance(			SANE_Byte	*regs,
					SANE_Word	value)
/****************************************************************************/
{
	return Hp_rts_set_value_lsbfirst(regs, 0x62, 2, value);
}

/****************************************************************************/
SANE_Int
Hp_rts_set_scanline_end(			SANE_Byte	*regs,
					SANE_Word	value)
/****************************************************************************/
{
	return Hp_rts_set_value_lsbfirst(regs, 0x6c, 2, value);
}

/****************************************************************************/
SANE_Int
Hp_rts_set_scanline_start(			SANE_Byte	*regs,
					SANE_Word	value)
/****************************************************************************/
{
	return Hp_rts_set_value_lsbfirst	(regs, 0x66, 2, value);
}


/****************************************************************************/
SANE_Status Hp_rts_CircBufferInit(SANE_Handle h, TDataPipe *p, SANE_Int iBytesPerLine,
													SANE_Int iMisAlignment, SANE_Bool mode)
/****************************************************************************/
{
	TScanner  *s;
	SANE_Int iHandle;
	SANE_Int bufsize;

	/* prevent compiler from complaining about unused parameters */
	mode = mode;

	s = (TScanner *)h;
	iHandle = s->HWParams.iXferHandle;
	p->iBytesPerLine = iBytesPerLine;
	bufsize = p->iBytesPerLine * 3;
	p->iLinesPerCircBuf = 1;
	p->iMisAlignment = iMisAlignment;

	DBG(DBG_SCAN, "Hp_rts_CircBufferInit: _iBytesPerLine = %d\n", p->iBytesPerLine);
	DBG(DBG_SCAN, "Hp_rts_CircBufferInit: _iLinesPerCircBuf = %d\n", p->iLinesPerCircBuf);

	p->pabCircBuf = (SANE_Byte *)malloc(bufsize+(3*p->iBytesPerLine));
	if (p->pabCircBuf == NULL) {
		DBG(DBG_ERR, "Hp_rts_CircBufferInit: Unable to allocate %d bytes for circular buffer (%x - %x)\n",
								(SANE_Int)bufsize,(unsigned int)p->pabCircBuf,(unsigned int)p->pabCircBuf+bufsize);
		return SANE_STATUS_NO_MEM;
	}
	DBG(DBG_SCAN,"Hp_rts_CircBufferInit: Allocate %d bytes for circular buffer (%x - %x)\n",
								(SANE_Int)bufsize,(unsigned int)p->pabCircBuf,(unsigned int)p->pabCircBuf+bufsize);
	/* init transfer buffer */
	return Hp_rts_XferBufferInit(h, p);
}


/****************************************************************************/
void Hp_rts_CircBufferExit(SANE_Handle h)
/****************************************************************************/
{
	TScanner  *s;

	s = (TScanner *)h;
	if (s->DataPipe.pabCircBuf != NULL) {
		free(s->DataPipe.pabCircBuf);
		s->DataPipe.pabCircBuf = NULL;
	}
	Hp_rts_XferBufferExit( h );
	DBG(DBG_SCAN,"Hp_rts_CircBufferExit\n");
}


/****************************************************************************/
/* gets an line from the circular buffer. */
SANE_Bool Hp_rts_CircBufferGetLine(SANE_Handle h, TDataPipe *p, SANE_Byte *pabLine,
												SANE_Bool mode){
/****************************************************************************/

	SANE_Int line_lenght,copy_lines;
	#ifdef DEBUG_FILE
	SANE_Word size_t,data_count;
	#endif

	TScanner  *s;

	s = (TScanner *)h;

	DBG(DBG_SCAN,"   Hp_rts_CircBufferGetLine starts here\n");
	mode = mode;
	line_lenght = p->iBytesPerLine;
	copy_lines = 0;

	if (pabLine != NULL) {
		if (Hp_rts_XferBufferGetLine(h, p, &p->pabCircBuf[0])){
			memcpy(pabLine, &p->pabCircBuf[0], p->iBytesPerLine);
			#ifdef DEBUG_FILE
				data_count = 0;
				size_t = p->iBytesPerLine;
				while(size_t--) fprintf(s->ScanParams.FD_r,"%c",p->pabCircBuf[data_count++]);
			#endif
			return SANE_TRUE;
		}
	}
	return SANE_FALSE;
}


/****************************************************************************/
SANE_Bool Hp_rts_XferBufferGetLine(SANE_Handle h, TDataPipe *p, SANE_Byte *pabLine)
/****************************************************************************/
{
	TScanner  *s;
	SANE_Word size;
	SANE_Word size_t;
	SANE_Word Lines, Skip, lBytes;
	SANE_Int iHandle,i,wt;
	SANE_Byte *buf;

	s = (TScanner *)h;
	iHandle = s->HWParams.iXferHandle;

	DBG(DBG_SCAN,"     Hp_rts_XferBufferGetLine in: = iBytesPerLine=%3d iLinesLeft=%3d\n",
								(SANE_Int)p->iBytesPerLine,(SANE_Int)s->ScanParams.iLinesLeft);

	switch (s->ScanParams.iDpi) {
		case 150:
		case 200:
		case 300:
		case 600:
			if (s->ScanParams.mode == COLOR)
				Skip = 1;
			else
				Skip = 2;
		break;
		default:
			Skip = 1;
		break;
	}
	lBytes = p->iBytesPerLine;

	/* Step 1, check if we have to receive more data from the scanner */
	if (p->iCurLine == 0){
		i = 0;
		while( i < 10 ){
			Hp_rts_data_ready(iHandle,&size);
			if (size > lBytes)
				i = 9;
			usleep(10);
			i++;
		}
		#if 0
			DBG(DBG_SCAN,"size %d  ",(SANE_Int) size);
		#endif
		Hp_rts_data_ready(iHandle,&size);
		Lines = (size / lBytes) / Skip; /* make shure we can div. it */
		size_t = (Lines * lBytes) * Skip;
		#ifdef DEBUG_SCAN
			DBG(DBG_SCAN,"     Hp_rts_XferBufferGetLine: size %d lBytes %d size_t %d\n",size,lBytes,size_t);
			DBG(DBG_SCAN,"     Hp_rts_XferBufferGetLine: write to buffer at 0x%x %d Lines\n",
									(unsigned int) p->pabXferBuf,Lines);
		#endif
		p->iLastLine = Lines * Skip;
		if ( size_t > 0 ){
			Hp_rts_read_data(iHandle,size_t,p->pabXferBuf);
		}else
		{ /* fix: the scanner is stoped.
					we have to read all bytes to
					restart him                  */
			#ifdef DEBUG_SCAN
				DBG(DBG_SCAN,"     Hp_rts_XferBufferGetLine: restart %d\n",size_t);
			#endif
			if ( size > 0 ){
				Hp_rts_read_data(iHandle,size,p->pabXferBuf);
				size_t = lBytes - size;
				i = 0;
				while( (Hp_rts_data_ready(iHandle,&size) == SANE_FALSE) && ( i < 3 )){
					sleep(1);
					i++;
				}
				#ifdef DEBUG_SCAN
					DBG(DBG_SCAN,"     Hp_rts_XferBufferGetLine: restart read %d\n",size_t);
				#endif
				Hp_rts_read_data(iHandle,size,p->pabXferBuf);
			}
			else{ /* no more data aviable */
				p->iLastLine = 0;
				return(SANE_FALSE);
			}
		}
		wt = (10 * (XFER_BUF_SIZE - size))/2;
		if (wt < 1) wt = 1;
		if (wt > 65000) wt = 65000;
		#ifdef DEBUG_SCAN
			DBG(DBG_SCAN,"     Wait %d s, size %d, iLastLine %d\n",wt, size,(SANE_Int)p->iLastLine);
		#endif
		usleep(wt);
	}

	/* Step 2, copy one line */
	if (pabLine != NULL) {
		buf = p->pabXferBuf + (p->iCurLine * lBytes);
		#ifdef DEBUG_SCAN
			DBG(DBG_SCAN,"     Hp_rts_XferBufferGetLine: read from buffer at 0x%x (%d), iLinesLeft %d, iScanned %d\n",
					(unsigned int) buf,p->iCurLine,(SANE_Int)s->ScanParams.iLinesLeft,(SANE_Int)p->iScanned);
		#endif
		memcpy(pabLine, buf, lBytes);
	}

	/* Step 3, set the advance pointer */
	p->iCurLine = p->iCurLine + Skip;
	p->iScanned = p->iScanned + 1;

	if ( p->iCurLine >= p->iLastLine  ){
		p->iCurLine = 0;
		#ifdef DEBUG_SCAN
			DBG(DBG_SCAN,"     Hp_rts_XferBufferGetLine: set p->iCurLine = 0\n");
		#endif
	}
	#ifdef DEBUG_SCAN
		DBG(DBG_SCAN,"     Hp_rts_XferBufferGetLine: exit\n");
	#endif
	return(SANE_TRUE);
}


/****************************************************************************/
SANE_Status Hp_rts_XferBufferInit(SANE_Handle h, TDataPipe *p)
/****************************************************************************/
{
	TScanner  *s;
	#ifdef DEBUG_FILE
	SANE_Word x1,x2,length ;
	#endif

	SANE_Int bufsize;

	s = (TScanner *)h;
	bufsize = XFER_BUF_SIZE/**10*/;

	p->iLinesPerXferBuf = bufsize / (p->iBytesPerLine);
	p->iLastLine = 0;
	DBG(DBG_SCAN,"Hp_rts_XferBufferInit: iLinesPerXferBuf = %d\n", p->iLinesPerXferBuf);
	DBG(DBG_SCAN,"Hp_rts_XferBufferInit: Xfer block size = 0x%x\n", p->iLinesPerXferBuf *
																													p->iBytesPerLine);

	#ifdef DEBUG_FILE
		/* Opening the testfile for debuging */
		if ( !s->ScanParams.DebugOpen ){
			s->ScanParams.DebugOpen = SANE_TRUE;
			s->ScanParams.FD_r = fopen("view_r.pnm","w");
			x1 = s->ScanParams.iX;
			x2 = s->ScanParams.iWidth ;
			length = s->ScanParams.iLenght - 1;
			DBG(DBG_SCAN,"TempFile, length:%d, x1:%d, x2:%d \n",length,x1,x2);
			switch (s->ScanParams.mode) {
				case BLACK_WHITE:
				case GRAY:
					fprintf(s->ScanParams.FD_r,"P5\n%d %d\n255\n",x2,length+100);
				break;
				case COLOR:
					fprintf(s->ScanParams.FD_r,"P6\n%d %d\n255\n",x2,length+100);
				break;
			}
		}
	#endif
	p->iCurLine = 0;

	p->pabXferBuf = (SANE_Byte *)malloc(bufsize);
	if (p->pabXferBuf == 0){
		DBG(DBG_ERR,"Hp_rts_XferBufferInit: MEM alloc is not not possible !\n");
		return SANE_STATUS_NO_MEM;
	} else{
		DBG(DBG_SCAN,"Hp_rts_XferBufferInit alloc: p->pabXferBuf = 0x%x bytes at 0x%x\n",
								bufsize,(unsigned int) p->pabXferBuf);
		return SANE_STATUS_GOOD;
	}
}


/****************************************************************************/
void Hp_rts_XferBufferExit( SANE_Handle h )
/****************************************************************************/
{
	TScanner *s;

	s = (TScanner *)h;
	if (s->DataPipe.pabXferBuf != NULL) {
		DBG(DBG_SCAN,"Hp_rts_XferBufferExit free: p->pabXferBuf at 0x%x\n",
				(unsigned int) s->DataPipe.pabXferBuf);
		free(s->DataPipe.pabXferBuf);
		s->DataPipe.pabXferBuf = NULL;
	}
	else { ;
		DBG(DBG_ERR, "Hp_rts_XferBufExit: Xfer buffer not initialised!\n");
	}
}


