 /* sane - Scanner Access Now Easy.
   Copyright (C) 2003 Johannes Hub (JohannesHub@foni.net)

   This file was initially copied from the hp3300 backend.
   This file is part of the SANE package.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

   As a special exception, the authors of SANE give permission for
   additional uses of the libraries contained in this release of SANE.

   The exception is that, if you link a SANE library with other files
   to produce an exutable, this does not by itself cause the
   resulting executable to be covered by the GNU General Public
   License.  Your use of that executable is in no way restricted on
   account of linking the SANE library code into it.

   This exception does not, however, invalidate any other reasons why
   the executable file might be covered by the GNU General Public
   License.

   If you submit changes to SANE to the maintainers to be included in
   a subsequent release, you agree by submitting the changes that
   those changes may be distributed with this exception intact.

   If you write modifications of your own for SANE, it is your choice
   whether to permit this exception to apply to your modifications.
   If you do not wish that, delete this exception notice.
*/

/*
    Concept for a backend for scanners based on the RTS88xx chipset,
    such as HP4400C, HP4470C.
    Parts of this source were inspired by other backends.

    History:

    Version 0.17g 18.04.04 10.alpha, little fixes
    Version 0.17e 07.04.04 10.alpha, little fixes
    Version 0.17d 05.04.04 10.alpha, little fixes
    Version 0.17b 30.03.04 10.alpha, little fixes and libusb implemented
    Version 0.17a 11.03.04 10.alpha, little fixes
    Version 0.17  09.03.04 9. alpha, HP3500 included
    Version 0.16  06.02.04 8. alpha, wait counting on LCD
    Version 0.15a 29.01.04 7. alpha, CCD switch moved to config file
    Version 0.15  11.01.04 6. alpha, a second CCD implemented
    Version 0.14  21/11/22 5. alpha
    Version 0.13a 21.11.04 4. alpha, an little fix included
    Version 0.13  13.11.04 4. alpha
    Version 0.12  22.10.03 third alpha, Backend name changed to HP_RTS88xx
    Version 0.11  30.08.03 second alpha
    Version 0.10  19.07.03 first alpha
*/

/*
    Core HP44x0c functions.
*/

#include <stdio.h>  /* fopen, fread, fwrite, fclose etc */
#include <stdarg.h> /* va_list for vfprintf */
#include <string.h> /* memcpy, memset */
#include <unistd.h> /* unlink */
#include <stdlib.h> /* malloc, free */
#include <math.h>   /* exp, pow */

#include "hp_rts_xfer.h"
#include "hp_rts_44x0c.h"


/****************************************************************************/
void Hp44x0_Wakeup(THWParams *pHWParams,TScanParams *pParams)
/****************************************************************************/
{
	SANE_Byte Reg_10, Reg_11;
	SANE_Int iHandle;

	/* prevent compiler from complaining about unused parameters */
	pParams->mode = pParams->mode;
	iHandle = pHWParams->iXferHandle;

	if (iHandle <= 0) {
		return;
	}
	/* write magic startup sequence */
	Hp_rts_RegWrite(iHandle,0xb3,0x00);
	Hp_rts_RegWrite(iHandle,0xb3,0x00);
	/*  test_reg( 0xb2,0x00,20 );*/
	Hp_rts_RegWrite(iHandle,0xb2,0x02);
	Hp_rts_RegWrite(iHandle,0xb1,0x00);
	if (pParams->optXPA)
		Reg_11 = 0x08;
	else
		Reg_11 = 0x28;
	Reg_10 = 0x28;
	Hp_rts_Set_double_reg(iHandle, LEDSTATUS, Reg_10, Reg_11 );
	usleep(1000);
	/*  test_double_reg( 0x20,0x3a,0xf2,21 );*/
	Hp_rts_RegWrite(iHandle,0xb3,0x01);
	Hp_rts_RegWrite(iHandle,0xb3,0x01);
	Hp_rts_RegWrite(iHandle,0xb3,0x00);
	Hp_rts_RegWrite(iHandle,0xb3,0x00);
	Hp_rts_Set_double_reg(iHandle,0x12,0xff,0x20);
	Hp_rts_Set_double_reg(iHandle,0x14,0xf8,0x28);
	Hp_rts_RegWrite(iHandle,0xb3,0x00);
	Hp_rts_RegWrite(iHandle,0xb3,0x00);
	Hp_rts_Set_double_reg(iHandle, LEDSTATUS, Reg_10, Reg_11 );
	usleep(1000);
	Hp_rts_RegWrite(iHandle,0xd9,0x80);
	memcpy(pHWParams->regs,Hp44x0_switch_on_regs,254);
	pHWParams->regs[0x10] = Reg_10;
	pHWParams->regs[0x11] = Reg_11;
	Hp_rts_BulkWrite (iHandle, 0x00,pHWParams->regs, 0xb3,SANE_TRUE);
	Hp_rts_BulkWrite (iHandle, 0xb4,pHWParams->regs+0xb3, 0x3f,SANE_TRUE);

	/*  need we a new callibriation ? */
}


/****************************************************************************/
void Hp44x0_Down(SANE_Int iHandle)
/****************************************************************************/
{
	if (iHandle <= 0) {
		return;
	}
	/* switch off the little green LEDs for color selector */
	Hp_rts_Set_double_reg(iHandle,LEDSTATUS,0x08,0x28);
	usleep(10000);
	/*test_reg( 0xb2, 0x02,13 ); */
	/* switch off the green power LED */
	Hp_rts_RegWrite(iHandle,0xb2,0x00);
	/* ?? */
	Hp_rts_RegWrite(iHandle,0xb1,0x00);
	/* switch off lamp, LCD and power LED */
	Hp_rts_RegWrite(iHandle,0xb3,0x04);
	Hp_rts_RegWrite(iHandle,0xb3,0x04);
}

/****************************************************************************/
void patch_and_send_values(SANE_Int iHandle, SANE_Byte *my_values){
/****************************************************************************/
	/* read registers, patch and write out them */
	SANE_Byte v1,v2;

	Hp_rts_BulkReadall(iHandle,registers);
	v1 = *my_values++;
	v2 = *my_values++;
	registers[v1] = v2;
	while(*my_values){
		v1 = *my_values++;
		v2 = *my_values++;
		registers[v1] = v2;
	}
	Hp_rts_BulkWrite(iHandle,0x00,registers,0xb3,SANE_TRUE);
	Hp_rts_BulkWrite(iHandle,0xb4,registers+0xb3,0x3f,SANE_TRUE);
}

/****************************************************************************/
void patch_regs(THWParams *pHWParams,SANE_Byte *my_values){
/****************************************************************************/
	/* read registers, patch and write out them */
	SANE_Byte v1,v2;

	v1 = *my_values++;
	v2 = *my_values++;
	pHWParams->regs[v1] = v2;
	while(*my_values){
		v1 = *my_values++;
		v2 = *my_values++;
		pHWParams->regs[v1] = v2;
	}
}

/****************************************************************************/
void Hp44x0_set_display( SANE_Int iHandle, SANE_Byte val )
/****************************************************************************/
{
	static SANE_Word LcdValues_h []=
	/* left char       0,      1,     2,     3 */
								{ 0xa18f2,0x0890,0x10b3,0x18b1};
	static SANE_Word LcdValues_l []=
	/* right char       0,      1,     2,     3      4 */
								{ 0xa68c,0x2280,0xe484,0x6684,0x6288,
	/*                  5,      6,     7,     8      9 */
								  0x468c,0xc68c,0x2284,0xe68c,0x668c};
	SANE_Byte reg_20,reg_21;

	if (val < 40){
		reg_20 = (LcdValues_l [val%10]  >> 8) & 0x00ff;
		reg_21 =  LcdValues_l [val%10] & 0x00ff;
		reg_20 = reg_20 | (LcdValues_h [val/10]  >> 8) & 0x00ff;
		reg_21 = reg_21 |  LcdValues_h [val/10] & 0x00ff;
		Hp_rts_RegWrite(iHandle,MOVE_START_STOP,0x00);
		Hp_rts_Set_double_reg(iHandle, LCD_CONTROL1, reg_20, reg_21 );
	};
}

/****************************************************************************/
void write_cal_bytes( THWParams *pHWParams, SANE_Byte b ) {
/****************************************************************************/
  SANE_Word i,n;
  SANE_Int iHandle;

  iHandle = pHWParams->iXferHandle;
  callibration_buffer[0]=0x89;
  callibration_buffer[1]=0x00;
  callibration_buffer[2]=0x00;
  callibration_buffer[3]=0x20;
  callibration_buffer[4]=0;
  n = 2;
  for(i=5;i<36;i++) {
     callibration_buffer[i]   = b;
     callibration_buffer[i+1] = n;
     i++;
     n = n + 2;
  }
  Hp_rts_BulkWrite(iHandle,0x00,callibration_buffer,36,SANE_FALSE);
}


/****************************************************************************/
SANE_Bool Hp44x0_cal_scanner(THWParams *pHWParams, TScanParams *pParams) {
/****************************************************************************/
  SANE_Word i,n;
  SANE_Int iHandle;
  SANE_Byte Reg_10, Reg_11;

  iHandle = pHWParams->iXferHandle;
  Hp_rts_RegWrite(iHandle,MOVE_START_STOP,0x00);
  Hp_rts_RegWrite(iHandle,MOVE_START_STOP,0x00);
  memcpy(registers,Hp44x0_init_1,0xf4);
  if (pParams->mode == COLOR){
     Reg_10 = 0x28;
  } else {
     Reg_10 = 0x18;}
  if (pParams->optXPA){
     Reg_11 = 0x1b;
  } else {
     Reg_11 = 0x3b;}
  registers[0x10] = Reg_10;
  registers[0x11] = Reg_11;
  Hp_rts_BulkWrite(iHandle,0x00,registers,0xb3,SANE_TRUE);
  Hp_rts_BulkWrite(iHandle,0xb4,registers+0xb3,0x3f,SANE_TRUE);
  patch_and_send_values(iHandle,Hp44x0_patch_2);
  patch_and_send_values(iHandle,Hp44x0_patch_3);

/*  if (pParams->mode == COLOR){*/
     Hp_rts_Set_double_reg(iHandle, LEDSTATUS, 0x20, 0x28);
/*     usleep(10000);*/
     Hp_rts_Set_double_reg(iHandle, LEDSTATUS, 0x28, 0x28);
/*
  } else {
     Hp_rts_Set_double_reg(iHandle, LEDSTATUS, 0x10, 0x28);
     usleep(10000);
     Hp_rts_Set_double_reg(iHandle, LEDSTATUS, 0x18, 0x28);
  }
*/
  usleep(10000);
  Hp_rts_RegWrite(iHandle,MOVE_START_STOP,0x02);
  Hp_rts_RegWrite(iHandle,MOVE_START_STOP,0x02);
  Hp_rts_RegWrite(iHandle,MOVE_START_STOP,0x00);
  Hp_rts_RegWrite(iHandle,MOVE_START_STOP,0x00);
  Hp_rts_RegWrite(iHandle,MOVE_START_STOP,0x00);
  Hp_rts_RegWrite(iHandle,MOVE_START_STOP,0x00);
  patch_and_send_values(iHandle,Hp44x0_patch_4); /* REG 10 Handler! */

  for (n=0;n<3;n++){
    switch (n){
      case 0: Hp_rts_RegWrite(iHandle,0x93,0x06); break;
      case 1: Hp_rts_RegWrite(iHandle,0x93,0x02); break;
      case 2: Hp_rts_RegWrite(iHandle,0x93,0x01); break;
      default:;
    }
    Hp_rts_Set_double_reg(iHandle,0x91, 0x81, 0x00);
    callibration_buffer[0]=0x89;
    callibration_buffer[1]=0x00;
    callibration_buffer[2]=0x08;
    callibration_buffer[3]=0x18;
    callibration_buffer[4]=0;
    for(i=5;i<0x818;i++) {
       callibration_buffer[i]=callibration_buffer[i-1]+1;
       if(callibration_buffer[i]==0x61) callibration_buffer[i]=0;
    };
    Hp_rts_BulkWrite(iHandle,0x00,callibration_buffer,0x818+4,SANE_FALSE);
    Hp35x0c_set_sram_page(iHandle,0x81);
/*    Hp_rts_Set_double_reg(iHandle,0x91, 0x81, 0x00);*/
    Hp_rts_Read_Sram (iHandle, callibration_buffer,0x0818);
#if 0
    read_bulk_callibriation_data(callibration_buffer,0x0818);
/*    read_bulk_callibriation_data(buffer2,0x0818); and verify it*/
#endif
	};
  memcpy(registers,Hp44x0_init_5,0xf4);
/*
  if (pParams->mode == COLOR){
     registers[LEDSTATUS] = 0x28;
  } else {
     registers[LEDSTATUS] = 0x18;
  }*/
  Hp_rts_BulkWrite(iHandle,0x00,registers,0xb3,SANE_TRUE);
  Hp_rts_BulkWrite(iHandle,0xb4,registers+0xb3,0x3f,SANE_TRUE);

  for(i=0;i<5;i++) {
    write_cal_bytes( pHWParams, i );
    Hp_rts_Set_double_reg(iHandle, 0x91, 0x00, 0x10 );
/*    read_bulk_callibriation_data(buffer2,0x20); and verify it*/
  };
  Hp_rts_RegWrite(iHandle,LAMPREG,0x80);
  Hp_rts_RegWrite(iHandle,LAMPREG,0xad);
  Hp_rts_Set_double_reg(iHandle, 0x14, 0x78, 0x28 );
  Hp_rts_RegWrite(iHandle,LAMP_INTENSITY,0xa0);
  Hp_rts_RegWrite(iHandle,LAMP_INTENSITY,0xa7);
  Hp_rts_RegWrite(iHandle,LAMPREG,0x8d);
  Hp_rts_RegWrite(iHandle,LAMPREG,0xad);
  Hp_rts_RegWrite(iHandle,BUTTON_1,0x00);
  Hp_rts_RegWrite(iHandle,BUTTON_2,0x00);
  Hp_rts_RegWrite(iHandle,BUTTON_1,0x00);
  Hp_rts_RegWrite(iHandle,BUTTON_2,0x00);
  Hp_rts_RegWrite(iHandle,LAMP_INTENSITY,0xa0);
  Hp_rts_RegWrite(iHandle,LAMP_INTENSITY,0xa7);
  Hp_rts_RegWrite(iHandle,MOVE_START_STOP,0x00);
  Hp_rts_RegWrite(iHandle,MOVE_START_STOP,0x00);

  if (pParams->mode == COLOR){
     Hp_rts_Set_double_reg(iHandle, 0x10, 0x28, 0x3f);
  } else {
     Hp_rts_Set_double_reg(iHandle, 0x10, 0x18, 0x3f);
  }
  usleep(10000);
  Hp_rts_RegWrite(iHandle,LAMPREG,0xad);
  usleep(10000);
/*  if (pParams->mode == COLOR){*/
     Hp_rts_Set_double_reg(iHandle, LEDSTATUS, 0x20, 0x3f);
/*  } else {
     Hp_rts_Set_double_reg(iHandle, LEDSTATUS, 0x10, 0x3f);
  }*/
  usleep(10000);
  return(SANE_TRUE);
}


/****************************************************************************/
SANE_Int Hp44x0_park_to_home(THWParams *pHWParams){
/* Moves scanner to home position */
/****************************************************************************/
	SANE_Int n,iHandle;
	SANE_Byte	buffer[0xffc0];

	iHandle = pHWParams->iXferHandle;

	DBG(DBG_MSG,"park to home...,");
	/* make shure the scaner is stoped */
	Hp_rts_stop_moving(iHandle);

  if (Hp35x0c_is_rewound(iHandle ))
		DBG(DBG_MSG,"i'm home\n");
	if(read_reg(iHandle,MICROSWITCH) & 0x02) {
		DBG(DBG_MSG,"i'm home\n");
		return(0);
	} else {
		DBG(DBG_MSG,"i must moving\n");
		memcpy(pHWParams->regs,Hp44x0_move_back,255);
/*
		pHWParams->regs[0x32] = 0x80;
		pHWParams->regs[0x33] = 0x81;
		pHWParams->regs[0x34] = 0x10;
		pHWParams->regs[0x35] = 0x10;
		pHWParams->regs[0x36] = 0x21; *0x24;*
		pHWParams->regs[0x39] = 0x02; * ??*
		pHWParams->regs[0x3A] = 0x0e;
		pHWParams->regs[0x60] = 0x40;
		pHWParams->regs[0x61] = 0x1f;
		pHWParams->regs[0x62] = 0x41;
		pHWParams->regs[0x63] = 0x1f;
		pHWParams->regs[0xb2] = 0x16;
		pHWParams->regs[0xe2-1] = 0x17; * StepSize */
		Hp_rts_BulkWrite(iHandle,0x00,pHWParams->regs,0xb3,SANE_TRUE);
		Hp_rts_BulkWrite(iHandle,0xb4,pHWParams->regs+0xb3,0x3f,SANE_TRUE);
		Hp_rts_RegWrite(iHandle,0xd3,0x00);
		Hp_rts_start_moving(iHandle);
		usleep(1000);
		while (!Hp35x0c_is_rewound(iHandle)  &&
			(Hp_rts_data_ready(iHandle,&n) ||
			 Hp_rts_is_moving(iHandle) > 0))
		{
			if (n)
			{
				if (n > (SANE_Word)sizeof(buffer))
					n = sizeof(buffer);
				Hp_rts_read_data(iHandle, n, buffer);
			}
			else
			{
				usleep(10000);
			}
		}
		return(Hp_rts_stop_moving(iHandle));
	}
}

/****************************************************************************/
SANE_Int Hp44x0_move_to_pos(THWParams *pHWParams, TScanParams *pParams){
/* Moves scanner to position */
/****************************************************************************/
	SANE_Int iHandle;
	SANE_Word pos;

	iHandle = pHWParams->iXferHandle;

	pos = (pParams->iY) + 90;
	DBG(DBG_MSG,"move_to_pos: move to %d (pixel), iTopLeftY = %d iY = %d\n",
			(int)pos,(int)pHWParams->iTopLeftY,(int)pParams->iY);

	pHWParams->regs[0x33] = 0x81; /* move fast */
	pHWParams->regs[0x36] = 0x29; /*2C; * 2C slow forward, 24 reverse */
	pHWParams->regs[0x3A] = 0x1b;
	pHWParams->regs[0x60] = 0x01;
	pHWParams->regs[0x61] = 0x00;
	pHWParams->regs[0x62] = (pos+1) & 0xff;
	pHWParams->regs[0x63] = ((pos+1) >> 8) & 0xff;
	pHWParams->regs[0x65] = 0x00;  /* switch to move only (no scan)*/
	pHWParams->regs[0xE2-1] = 0x03/*08*/ ; /* Stepsize */
	if (pParams->optXPA)
		pHWParams->regs[0x11] = pHWParams->regs[0x11] & 0xdf;
	else
		pHWParams->regs[0x11] = pHWParams->regs[0x11] | 0x02;

	Hp_rts_Set_double_reg(iHandle,0x10,pHWParams->regs[0x10],pHWParams->regs[0x11]);
	usleep(10000);
	Hp_rts_BulkWrite(iHandle,0x00,pHWParams->regs,0xb3,SANE_TRUE);
	Hp_rts_BulkWrite(iHandle,0xb4,pHWParams->regs+0xb3,0x3f,SANE_TRUE);
	Hp_rts_RegWrite(iHandle,0xd3,0x03);

	Hp_rts_start_moving(iHandle);
	usleep(1000);
	while(  (read_reg(iHandle,MOVE_START_STOP) & 0x08)/* &&
		!(read_reg(iHandle,MICROSWITCH) & 0x02) */) {
		usleep(10000);
	};
	return(SANE_TRUE);
}


/****************************************************************************/
SANE_Int Hp44x0_wait_for_WarmUp(SANE_Int iHandle,THWParams *pHWParams){
/****************************************************************************/
  /* Reads 11 bytes and check the last one */
SANE_Char i;
#ifdef DEBUG
SANE_Char c;
#endif
  /* we must here implement a wait loop for the lamp */
  Hp_rts_BulkRead(iHandle,0x84,read_buffer,0x0b,SANE_TRUE);
#ifdef DEBUG
  c = read_buffer[0xa];
  DBG(DBG_MSG,"wait_for_WarmUp: read %x\n",c);
#endif
  i = 15;
  if ( pHWParams->lamp_warm == SANE_FALSE ){
    while (i--){
      Hp44x0_set_display( iHandle, i );
      sleep(1);
    }
    pHWParams->lamp_warm = SANE_TRUE;
  }
  return( /*(read_buffer[0xa] & 0x60) == 0x60*/ SANE_TRUE );
}

/****************************************************************************/
SANE_Bool Hp44x0_start_scan(THWParams *pHWParams, TScanParams *pParams,
								TDataPipe *pDataPipe){
/****************************************************************************/
#ifdef DEBUG
	SANE_Word lengtha1,lengtha2;
#endif
	SANE_Int iHandle;
	SANE_Int	ires;
	SANE_Word x1,x2,length;

	iHandle = pHWParams->iXferHandle;

	DBG(DBG_MSG,"start_scan: check resolution %d\n",pParams->iDpi);
	for (ires = 0; Hp44x0c_resparms[ires].resolution &&
								 Hp44x0c_resparms[ires].resolution != pParams->iDpi; ++ires);
	if (Hp44x0c_resparms[ires].resolution == 0){
		DBG(DBG_MSG,"start_scan: this resolution %d not found\n",pParams->iDpi);
		return SANE_FALSE;
	}

	memcpy(pHWParams->regs,Hp44x0_myinit33,254);  /* CCD_Type 0*/
	Hp44x0_move_to_pos(pHWParams,pParams);

	x1 = pParams->iX;
	x2 = pParams->iWidth + pParams->iX;
	length = pParams->iLenght * 2;
	pDataPipe->iScanned = pParams->iLenght;
	DBG(DBG_MSG,"start_scan: calculate scandata x1 = %d; x2 = %d; lenght = %d\n",x1,x2,length);

	switch (pParams->mode) {
		case BLACK_WHITE:
			DBG(DBG_MSG,"Copy 300 DPI BW Regs\n");
			memcpy(pHWParams->regs,Hp44x0_300_BW_1,254);  /* CCD_Type 0*/
/*			memcpy(pHWParams->regs,Hp44x0_myinit33,254);  / CCD_Type 0*/
			pHWParams->regs[0xD2 ]=  0x03;
			pHWParams->regs[0xD3 ]=  0x05;
			pHWParams->regs[0xE4 ]=  0x1c;
			pHWParams->regs[0xE5 ]=  0x10;
			pHWParams->regs[0xd5 ]= 0xab;/* ab gray */
			break;
		case GRAY:
			DBG(DBG_MSG,"Copy 300 DPI Regs\n");
			if (pParams->oCCD_Type)
				memcpy(pHWParams->regs,Hp44x0_300_true_1_3,254); /* CCD_Type 1*/
			else
				memcpy(pHWParams->regs,Hp44x0_myinit33,254);  /* CCD_Type 0*/
			if (pParams->iDpi == 600){
				DBG(DBG_MSG,"Patch to 600 DPI\n");
				patch_regs(pHWParams,Hp44x0_patch_600);
			}
			pHWParams->regs[0xd5 ]= 0xab;/* ab gray */
			break;
		case COLOR:
			DBG(DBG_MSG,"Copy 300 DPI Regs\n");
			if (pParams->oCCD_Type)
				memcpy(pHWParams->regs,Hp44x0_300_true_1_3,254); /* CCD_Type 1*/
			else
				memcpy(pHWParams->regs,Hp44x0_myinit33,254);  /* CCD_Type 0*/
			if (pParams->iDpi == 600){
				DBG(DBG_MSG,"Patch to 600 DPI\n");
				patch_regs(pHWParams,Hp44x0_patch_600);
			}
/*			pHWParams->regs[0x33 ]= 0x82;* for test*/
			pHWParams->regs[0xd5 ]= 0x0f;/* 0f color*/
			break;

		default:
			DBG(DBG_ERR, "Hp44x0_init_scan: Invalid parameter\n");
			return SANE_FALSE;
	}/* switch */

	/* horizon resolution */
	pHWParams->regs[0x10 ]=  Hp44x0c_resparms[ires].reg_10_value;
	pHWParams->regs[0x36 ]=  Hp44x0c_resparms[ires].reg_36_value;
/*	pHWParams->regs[0xE4 ]=  Hp44x0c_resparms[ires].reg_E4_value;
	pHWParams->regs[0xE5 ]=  Hp44x0c_resparms[ires].reg_E5_value;*/
	pHWParams->regs[0xE2-1]= Hp44x0c_resparms[ires].step_size;


#if 0

	/* color releated */
	pHWParams->regs[0x72 ]=  Hp44x0c_resparms[ires].reg_72_value;
	pHWParams->regs[0x73 ]=  Hp44x0c_resparms[ires].reg_73_value;
	pHWParams->regs[0x74 ]=  Hp44x0c_resparms[ires].reg_74_value;
#endif
#if 0
	pHWParams->regs[0x11 ]=  Hp44x0c_resparms[ires].reg_11_value;
	pHWParams->regs[0xBF ]=  Hp44x0c_resparms[ires].reg_BF_value;
	pHWParams->regs[0xC0 ]=  Hp44x0c_resparms[ires].reg_C0_value;
	pHWParams->regs[0xCB ]=  Hp44x0c_resparms[ires].reg_CB_value;
	pHWParams->regs[0xCC ]=  Hp44x0c_resparms[ires].reg_CC_value;
	pHWParams->regs[0xCD ]=  Hp44x0c_resparms[ires].reg_CD_value;
	pHWParams->regs[0xCE ]=  Hp44x0c_resparms[ires].reg_CE_value;
	pHWParams->regs[0xCF ]=  Hp44x0c_resparms[ires].reg_CF_value;
	pHWParams->regs[0xd0 ]=  Hp44x0c_resparms[ires].reg_D0_value;
	pHWParams->regs[0xd1 ]=  Hp44x0c_resparms[ires].reg_D1_value;
	pHWParams->regs[0xd2 ]=  Hp44x0c_resparms[ires].reg_D2_value;
	pHWParams->regs[0xd3 ]=  Hp44x0c_resparms[ires].reg_D3_value;
	pHWParams->regs[0xd7 ]=  Hp44x0c_resparms[ires].reg_D7_value;
	pHWParams->regs[0xC4 ]=  Hp44x0c_resparms[ires].reg_C4_value; /* colors ? */
	pHWParams->regs[0xC5 ]=  Hp44x0c_resparms[ires].reg_C5_value; /* colors ? */
	pHWParams->regs[0xd5 ]=  Hp44x0c_resparms[ires].reg_D5_value; 

	pHWParams->regs[0x35 ]=  Hp44x0c_resparms[ires].reg_35_value;
	pHWParams->regs[0x3A ]=  Hp44x0c_resparms[ires].reg_3A_value;
Hp44x0_300_BW_1
#endif
	/* vertical resolution */
#if 0
	pHWParams->regs[0x3A ]=  Hp44x0c_resparms[ires].reg_3A_value;
#endif

#if 1
	if (pParams->optXPA)
		pHWParams->regs[0x11] = pHWParams->regs[0x11] & 0xdf;
	else
		pHWParams->regs[0x11] = pHWParams->regs[0x11] | 0x02;
#endif
#if 0
	if (pParams->mode == COLOR)
		pHWParams->regs[0xd5 ]= 0x0f;/* 0f color*/
	else
		pHWParams->regs[0xd5 ]= 0xab;/* ab gray */
#endif

	Hp_rts_set_noscan_distance(pHWParams->regs, 0x01 );
/*	pHWParams->regs[0x60]= 0x01;
	pHWParams->regs[0x61]= 0x00;*/

	/* modify reg 62 + 63 */
	Hp_rts_set_total_distance(pHWParams->regs, length+1 );
/*	pHWParams->regs[0x62]=(length+1) & 0xff;
	pHWParams->regs[0x63]=((length+1) >> 8) & 0xff;*/
  
	pHWParams->regs[0x65]= 0x20; /* Schan mode on */

	/* modify reg 66 + 67 */
	Hp_rts_set_scanline_start(pHWParams->regs, x1 );
/*	pHWParams->regs[0x66]= x1 & 0xff;
	pHWParams->regs[0x67]= (x1 >> 8) & 0xff;*/

	/* modify reg 6C + 6D */
	Hp_rts_set_scanline_end(pHWParams->regs, x2 );
/*  pHWParams->regs[0x6c]= x2 & 0xff;
	pHWParams->regs[0x6d]= (x2 >> 8) & 0xff;*/

	#ifdef DEBUG
		lengtha1 = 0;
		lengtha2 = 0;
		lengtha1 = pHWParams->regs[0x60];
		lengtha1 = lengtha1 + ((pHWParams->regs[0x61] <<8) & 0xFF00);
		lengtha2 = pHWParams->regs[0x62];
		lengtha2 = lengtha2 + ((pHWParams->regs[0x63] <<8) & 0xFF00);
		DBG(DBG_MSG,"start_scan: Init scan_region: dpi=%d iLenght=%d, iScanned=%d lenght=%d mode=%x xPDA=%x\n",
		pParams->iDpi,pParams->iLenght,pDataPipe->iScanned,length,pParams->mode,pParams->optXPA);
		#endif

#if 0
	pHWParams->regs[0xDA-1]= pParams->intensity2; /* Lamp density */
#endif
#if 0
	if (pParams->mode == COLOR){
/*
		pHWParams->regs[0x05] = 0x20 - pParams->GainR;
		pHWParams->regs[0x06] = 0x20 - pParams->GainG;
		pHWParams->regs[0x07] = 0x20 - pParams->GainB;
/
		pHWParams->regs[0x08] = 0x20 - pParams->GainR;
		pHWParams->regs[0x09] = 0x20 - pParams->GainG;
		pHWParams->regs[0x0a] = 0x20 - pParams->GainB;*/
	} else {
	}
#endif
	pHWParams->regs[0x08] = /*0x20 -*/ pParams->brightness;
	pHWParams->regs[0x09] = /*0x20 -*/ pParams->brightness;
	pHWParams->regs[0x0a] = /*0x20 -*/ pParams->brightness;

	#if 1
		printf("start_scan: start the scanner!\n");
		DumpHex(pHWParams->regs,255,16,SANE_TRUE);
	#endif

	Hp_rts_Set_double_reg(iHandle,0x10,pHWParams->regs[0x10],pHWParams->regs[0x11]);
	usleep(10000);

	Hp_rts_BulkWrite(iHandle,0x00,pHWParams->regs,0xb3,SANE_TRUE);
	Hp_rts_BulkWrite(iHandle,0xb4,pHWParams->regs+0xb3,0x3f,SANE_TRUE);
#if 0
	if (pParams->mode == BLACK_WHITE)
		Hp_rts_RegWrite(iHandle,0xd3,0x17);
	else
		Hp_rts_RegWrite(iHandle,0xd3,0x03);
#endif
	Hp_rts_RegWrite(iHandle,0xd3,pHWParams->regs[0xd3-1]);

	Hp_rts_start_moving(iHandle);
	usleep(10000);
	/* wait for moving and check it */
	return(Hp_rts_Check_Scanning (iHandle));
}


/****************************************************************************/
SANE_Bool Hp44x0_init_scan(THWParams *pHWParams, TScanParams *pParams,
													TDataPipe *pDataPipe){
/****************************************************************************/

	SANE_Int iHandle;
	SANE_Int	ires;

	iHandle = pHWParams->iXferHandle;
	DBG(DBG_MSG,"Hp44x0_init_scan: check resolution %d\n",pParams->iDpi);
	for (ires = 0; Hp44x0c_resparms[ires].resolution &&
								 Hp44x0c_resparms[ires].resolution != pParams->iDpi; ++ires);
	if (Hp44x0c_resparms[ires].resolution == 0){
		DBG(DBG_MSG,"Hp44x0_init_scan: this resolution %d not found\n",pParams->iDpi);
		return SANE_FALSE;
	}

	Hp_rts_RegWrite(iHandle,LAMP_INTENSITY,0xa7);
	Hp_rts_RegWrite(iHandle,LAMP_INTENSITY,0xa0);
	Hp_rts_RegWrite(iHandle,LAMP_INTENSITY,0xa7);
	Hp_rts_RegWrite(iHandle,0xb3,0x01);
	Hp_rts_RegWrite(iHandle,0xb3,0x01);
	Hp_rts_RegWrite(iHandle,0xb3,0x00);
	Hp_rts_RegWrite(iHandle,0xb3,0x00);
	Hp_rts_Set_double_reg(iHandle, 0x12, 0xff, 0x20 );
	Hp_rts_Set_double_reg(iHandle, 0x14, 0xf8, 0x28 );
	Hp_rts_RegWrite(iHandle,0xb3,0x00);
	Hp_rts_RegWrite(iHandle,0xb3,0x00);

	memcpy(pHWParams->regs,Hp44x0_init_1,0xf4);
	pHWParams->regs[0x10 ]=  Hp44x0c_resparms[ires].reg_10_value;
	pHWParams->regs[0x11 ]=  Hp44x0c_resparms[ires].reg_11_value;

	if (pParams->optXPA)
		pHWParams->regs[0x11] = pHWParams->regs[0x11] & 0xdf;
	else
		pHWParams->regs[0x11] = pHWParams->regs[0x11] | 0x02;

	Hp_rts_Set_double_reg(iHandle,0x10,pHWParams->regs[0x10],
																		pHWParams->regs[0x11]);
	usleep(10000);
	Hp_rts_RegWrite(iHandle,0xd9,0x80);

	#ifndef WITH_TSTBACKEND
	Hp44x0_wait_for_WarmUp(iHandle,pHWParams);
	#endif

	Hp_rts_BulkWrite(iHandle,0x00,pHWParams->regs,0xb3,SANE_TRUE);
	Hp_rts_BulkWrite(iHandle,0xb4,pHWParams->regs+0xb3,0x3f,SANE_TRUE);

	Hp_rts_RegWrite(iHandle,LAMP_INTENSITY,0xa7);
	Hp_rts_RegWrite(iHandle,0xd9,0x8d);
	Hp_rts_RegWrite(iHandle,0xd9,0xad);
	Hp_rts_RegWrite(iHandle,0xb3,0x00);
	Hp_rts_RegWrite(iHandle,0xb3,0x00);
	/*  test_double_reg( LEDSTATUS, 0x28, 0x3f, 7 );  */
	usleep(10000);
	Hp_rts_Set_double_reg(iHandle,0x10,pHWParams->regs[0x10],
																		pHWParams->regs[0x11]);

	/*  test_double_reg( LEDSTATUS, 0x28, 0x3f, 8 );  */
	Hp_rts_RegWrite(iHandle,0xd9,0xad);

	return(Hp44x0_start_scan(pHWParams,pParams,pDataPipe));
}



/****************************************************************************/
SANE_Bool XferBufferGetLine(SANE_Handle h, TDataPipe *p, SANE_Byte *pabLine)
/****************************************************************************/
{
	TScanner  *s;
	SANE_Word size;
	SANE_Word size_t;
	SANE_Word Lines, Skip, lBytes;
	SANE_Int iHandle,i,wt;
	SANE_Byte *buf;

	s = (TScanner *)h;
	iHandle = s->HWParams.iXferHandle;

	DBG(DBG_SCAN,"   XferBufferGetLine in: = p->iBytesPerLine=%3d p->iCurLine=%3d \n",
								(int)p->iBytesPerLine,(int)p->iCurLine);

	switch (s->ScanParams.iDpi) {
		case 150:
		case 200:
		case 300:
		case 600:
			if (s->ScanParams.mode == COLOR)
				Skip = 1;
			else
				Skip = 2;
		break;
		default:
			Skip = 1;
		break;
	}

	lBytes = p->iBytesPerLine;
	/* Step 1, check if we have to receive more data from the scanner */
	if (p->iCurLine == 0){
		i = 0;
		while( i < 10 ){
			Hp_rts_data_ready(iHandle,&size);
			if (size > lBytes)
				i = 9;
			usleep(10);
			i++;
		}
		#ifdef DEBUG_SCAN
			DBG(DBG_SCAN,"size %d  ",(int) size);
		#endif
		Hp_rts_data_ready(iHandle,&size);
		Lines = (size / lBytes) / Skip; /* make shure we can div. it */
		size_t = (Lines * lBytes) * Skip;
		#ifdef DEBUG_SCAN
			DBG(DBG_SCAN,"   XferBufferGetLine: size %d lBytes %d size_t %d\n",size,lBytes,size_t);
			DBG(DBG_SCAN,"   XferBufferGetLine: write to buffer at 0x%x %d Lines\n",
									(unsigned int) p->pabXferBuf,Lines);
		#endif
		p->iLastLine = Lines * Skip;
		if ( size_t > 0 ){
			Hp_rts_read_data(iHandle,size_t,p->pabXferBuf);
		}else
		{ /* fix: the scanner is stoped.
					we have to read all bytes to
					restart him                  */
			#ifdef DEBUG_SCAN
				DBG(DBG_SCAN,"   XferBufferGetLine: restart %d\n",size_t);
			#endif
			if ( size > 0 ){
				Hp_rts_read_data(iHandle,size,p->pabXferBuf);
				size_t = lBytes - size;
				i = 0;
				while( (Hp_rts_data_ready(iHandle,&size) == SANE_FALSE) && ( i < 3 )){
					sleep(1);
					i++;
				}
				#ifdef DEBUG_SCAN
					DBG(DBG_SCAN,"   XferBufferGetLine: restart read %d\n",size_t);
				#endif
				Hp_rts_read_data(iHandle,size_t,p->pabXferBuf);
			}
			else{ /* no more data aviable */
				p->iLastLine = 0;
				return(SANE_FALSE);
			}
		}
		wt = (10 * (XFER_BUF_SIZE - size))/2;
		if (wt < 1) wt = 1;
		if (wt > 65000) wt = 65000;
		#ifdef DEBUG_SCAN
			DBG(DBG_SCAN,"   Wait %d s, size %d, iLastLine %d\n",wt, size,(int)p->iLastLine);
		#endif
		usleep(wt);
	}

	/* Step 2, copy one line */
	if (pabLine != NULL) {
		buf = p->pabXferBuf + (p->iCurLine * lBytes);
		#ifdef DEBUG_SCAN
			DBG(DBG_SCAN,"   XferBufferGetLine: read from buffer at 0x%x (%d), iLinesLeft %d, iScanned %d\n",
					(unsigned int) buf,p->iCurLine,(int)s->ScanParams.iLinesLeft,(int)p->iScanned);
		#endif
		memcpy(pabLine, buf, lBytes);
	}

	/* Step 3, ste the advance pointer */
	p->iCurLine = p->iCurLine + Skip;
	p->iScanned = p->iScanned - 1;

	if ( p->iCurLine >= p->iLastLine  ){
		p->iCurLine = 0;
		#ifdef DEBUG_SCAN
			DBG(DBG_SCAN,"   XferBufferGetLine: set p->iCurLine = 0\n");
		#endif
	}
	#ifdef DEBUG_SCAN
		DBG(DBG_SCAN,"XferBufferGetLine: exit\n");
	#endif
	return(SANE_TRUE);
}


/****************************************************************************/
SANE_Status XferBufferInit(SANE_Handle h, TDataPipe *p)
/****************************************************************************/
{
	TScanner  *s;
	#ifdef DEBUG_FILE
	SANE_Word x1,x2,length ;
	#endif

	SANE_Int bufsize;

	s = (TScanner *)h;
	bufsize = XFER_BUF_SIZE/**10*/;

	p->iLinesPerXferBuf = bufsize / (p->iBytesPerLine);
	p->iLastLine = 0;
	DBG(DBG_SCAN,"XferBufferInit: iLinesPerXferBuf = %d\n", p->iLinesPerXferBuf);
	DBG(DBG_SCAN,"XferBufferInit: Xfer block size = 0x%x\n", p->iLinesPerXferBuf *
																													p->iBytesPerLine);

	#ifdef DEBUG_FILE
		/* Opening the testfile for debuging */
		if ( !s->ScanParams.DebugOpen ){
			s->ScanParams.DebugOpen = SANE_TRUE;
			s->ScanParams.FD_r = fopen("view_r.pnm","w");
			#if 0
				s->ScanParams.FD_g = fopen("view_g.pnm","w");
				s->ScanParams.FD_b = fopen("view_b.pnm","w");
			#endif
			x1 = s->ScanParams.iX;
			x2 = s->ScanParams.iWidth ;
			length = s->ScanParams.iLenght - 1;
			DBG(DBG_SCAN,"TempFile, length:%d, x1:%d, x2:%d \n",length,x1,x2);
			switch (s->ScanParams.mode) {
				case BLACK_WHITE:
				case GRAY:
					DBG(DBG_SCAN,s->ScanParams.FD_r,"P5\n%d %d\n255\n",x2,length+100);
				break;
				case COLOR:
					DBG(DBG_SCAN,s->ScanParams.FD_r,"P6\n%d %d\n255\n",x2,length+100);
				break;
			}
			#if 0
				DBG(DBG_SCAN,s->ScanParams.FD_g,"P6\n%d %d\n255\n",x2,length+100);
				DBG(DBG_SCAN,s->ScanParams.FD_b,"P6\n%d %d\n255\n",x2,length+100);
			#endif
		}
	#endif
	p->iCurLine = 0;

	p->pabXferBuf = (SANE_Byte *)malloc(bufsize);
	if (p->pabXferBuf == 0){
		DBG(DBG_ERR,"XferBufferInit: MEM alloc is not not possible !\n");
		return SANE_STATUS_NO_MEM;
	} else{
		DBG(DBG_SCAN,"XferBufferInit alloc: p->pabXferBuf = 0x%x bytes at 0x%x\n",
								bufsize,(unsigned int) p->pabXferBuf);
		return SANE_STATUS_GOOD;
	}
}


/****************************************************************************/
void XferBufferExit( SANE_Handle h )
/****************************************************************************/
{
	TScanner *s;

	s = (TScanner *)h;
	if (s->DataPipe.pabXferBuf != NULL) {
		DBG(DBG_SCAN,"XferBufferExit free: p->pabXferBuf at 0x%x\n",
				(unsigned int) s->DataPipe.pabXferBuf);
		free(s->DataPipe.pabXferBuf);
		s->DataPipe.pabXferBuf = NULL;
	}
	else { ;
		DBG(DBG_ERR, "XferBufExit: Xfer buffer not initialised!\n");
	}
}

/****************************************************************************/
/* gets an line from the circular buffer. */
void CircBufferGetLine(SANE_Handle h, TDataPipe *p, SANE_Byte *pabLine,
												SANE_Bool mode){
/****************************************************************************/

	SANE_Int line_lenght,copy_lines;
	#ifdef DEBUG_FILE
	SANE_Word size_t,data_count;
	#endif

	TScanner  *s;

	s = (TScanner *)h;

	DBG(DBG_SCAN,"  CircBufferGetLine starts here\n");
	mode = mode;
	line_lenght = p->iBytesPerLine;
	copy_lines = 0;

	if (pabLine != NULL) {
		XferBufferGetLine(h, p, &p->pabCircBuf[0]);
		memcpy(pabLine, &p->pabCircBuf[0], p->iBytesPerLine);
		#ifdef DEBUG_FILE
			data_count = 0;
			size_t = p->iBytesPerLine;
			/*DBG(DBG_SCAN,"  CircBufferGetLine: write into file size_t = %d\n",size_t);*/
			while(size_t--) DBG(DBG_SCAN,s->ScanParams.FD_r,"%c",p->pabCircBuf[data_count++]);
			#if 0
			if (p->iPass == 1)
				while(size_t--) DBG(DBG_SCAN,s->ScanParams.FD_r,"%c",p->pabCircBuf[data_count++]);
			if (p->iPass == 2)
				while(size_t--) DBG(DBG_SCAN,s->ScanParams.FD_g,"%c",p->pabCircBuf[data_count++]);
			if (p->iPass == 3)
				while(size_t--) DBG(DBG_SCAN,s->ScanParams.FD_b,"%c",p->pabCircBuf[data_count++]);
			#endif
		#endif
	}
}


/****************************************************************************/
SANE_Status CircBufferInit(SANE_Handle h, TDataPipe *p, SANE_Int iBytesPerLine,
													SANE_Int iMisAlignment, SANE_Bool mode)
/****************************************************************************/
{
	TScanner  *s;
	SANE_Int iHandle;
	SANE_Int bufsize;

	/* prevent compiler from complaining about unused parameters */
	mode = mode;

	s = (TScanner *)h;
	iHandle = s->HWParams.iXferHandle;
	p->iBytesPerLine = iBytesPerLine;
	bufsize = p->iBytesPerLine * 3;
	p->iLinesPerCircBuf = 1;
	p->iMisAlignment = iMisAlignment;

	DBG(DBG_SCAN, "CircBufferInit: _iBytesPerLine = %d\n", p->iBytesPerLine);
	DBG(DBG_SCAN, "CircBufferInit: _iLinesPerCircBuf = %d\n", p->iLinesPerCircBuf);

	p->pabCircBuf = (SANE_Byte *)malloc(bufsize+(3*p->iBytesPerLine));
	if (p->pabCircBuf == NULL) {
		DBG(DBG_ERR, "CircBufferInit: Unable to allocate %d bytes for circular buffer (%x - %x)\n",
								(int)bufsize,(unsigned int)p->pabCircBuf,(unsigned int)p->pabCircBuf+bufsize);
		return SANE_STATUS_NO_MEM;
	}
	DBG(DBG_SCAN,"CircBufferInit: Allocate %d bytes for circular buffer (%x - %x)\n",
								(int)bufsize,(unsigned int)p->pabCircBuf,(unsigned int)p->pabCircBuf+bufsize);
	/* init transfer buffer */
	return XferBufferInit(h, p);
}


/****************************************************************************/
void CircBufferExit(SANE_Handle h)
/****************************************************************************/
{
	TScanner  *s;

	s = (TScanner *)h;
	if (s->DataPipe.pabCircBuf != NULL) {
		free(s->DataPipe.pabCircBuf);
		s->DataPipe.pabCircBuf = NULL;
	}
	XferBufferExit( h );
	DBG(DBG_SCAN,"CircBufferExit\n");
}


/****************************************************************************/
/*  Lamp control functions                                                  */
SANE_Bool Hp44x0_GetLamp(SANE_Int iHandle, SANE_Bool *pfLampIsOn)
/****************************************************************************/
{
	SANE_Byte bData;

	Hp_rts_RegRead(iHandle, LAMPREG, &bData);
	*pfLampIsOn = ((bData & 0x80) != 0);
	return SANE_TRUE;
}


/****************************************************************************/
SANE_Bool Hp44x0_SetLamp(SANE_Int iHandle, SANE_Bool fLampOn)
/****************************************************************************/
{
	SANE_Byte bData;

	Hp_rts_RegRead(iHandle, LAMPREG, &bData);
	if (fLampOn) {
		Hp_rts_RegWrite(iHandle, LAMPREG, bData | 0x80);
	}
	else {
		Hp_rts_RegWrite(iHandle, LAMPREG, bData & ~0x80);
	}
	return SANE_TRUE;
}


/****************************************************************************/
SANE_Int
Hp44x0c_nvram_enable_controller(SANE_Int iHandle,SANE_Int enable)
/****************************************************************************/
{
	SANE_Byte r;

	if (Hp_rts_RegRead(iHandle, 0x1d, &r) < 0)
		return -1;
	if (enable)
		r |= 1;
	else
		r &= ~1;
	return Hp_rts_RegWrite(iHandle,0x1d, r);

}

