# frozen_string_literal: true

module Gitlab
  class Experiment
    module RSpecHelpers
      def stub_experiments(experiments)
        experiments.each do |name, variant|
          variant = :control if variant == false
          raise ArgumentError, 'variant must be a symbol or false' unless variant.is_a?(Symbol)

          klass = Gitlab::Experiment.send(:constantize, name) # rubocop:disable GitlabSecurity/PublicSend

          # We have to use this high level any_instance behavior as there's
          # not an alternative that allows multiple wrappings of `new`.
          allow_any_instance_of(klass).to receive(:enabled?).and_return(true)
          allow_any_instance_of(klass).to receive(:resolve_variant_name).and_return(variant.to_s)
        end
      end

      def wrapped_experiment(experiment, shallow: false, failure: nil, &block)
        if shallow
          yield experiment if block.present?
          return experiment
        end

        receive_wrapped_new = receive(:new).and_wrap_original do |new, *new_args, &new_block|
          instance = new.call(*new_args)
          instance.tap(&block) if block.present?
          instance.tap(&new_block) if new_block.present?
          instance
        end

        klass = experiment.class == Class ? experiment : experiment.class
        if failure
          expect(klass).to receive_wrapped_new, failure
        else
          allow(klass).to receive_wrapped_new
        end
      end
    end

    module RSpecMatchers
      extend RSpec::Matchers::DSL

      def require_experiment(experiment, matcher_name, classes: false)
        klass = experiment.class == Class ? experiment : experiment.class
        unless klass <= Gitlab::Experiment
          raise(
            ArgumentError,
            "#{matcher_name} matcher is limited to experiment instances#{classes ? ' and classes' : ''}"
          )
        end

        if experiment == klass && !classes
          raise ArgumentError, "#{matcher_name} matcher requires an instance of an experiment"
        end

        experiment != klass
      end

      matcher :exclude do |context|
        ivar = :'@excluded'

        match do |experiment|
          require_experiment(experiment, 'exclude')
          experiment.context(context)

          experiment.instance_variable_set(ivar, nil)
          !experiment.run_callbacks(:exclusion_check) { :not_excluded }
        end

        failure_message do
          %(expected #{context} to be excluded)
        end

        failure_message_when_negated do
          %(expected #{context} not to be excluded)
        end
      end

      matcher :segment do |context|
        ivar = :'@variant_name'

        match do |experiment|
          require_experiment(experiment, 'segment')
          experiment.context(context)

          experiment.instance_variable_set(ivar, nil)
          experiment.run_callbacks(:segmented_run)

          @actual = experiment.instance_variable_get(ivar)
          @expected ? @actual.to_s == @expected.to_s : @actual.present?
        end

        chain :into do |expected|
          raise ArgumentError, 'variant name must be provided' if expected.blank?

          @expected = expected.to_s
        end

        failure_message do
          %(expected #{context} to be segmented#{message_details})
        end

        failure_message_when_negated do
          %(expected #{context} not to be segmented#{message_details})
        end

        def message_details
          message = ''
          message += %( into variant\n    expected variant: #{@expected}) if @expected
          message += %(\n      actual variant: #{@actual}) if @actual
          message
        end
      end

      matcher :track do |event, *event_args|
        match do |experiment|
          expect_tracking_on(experiment, false, event, *event_args)
        end

        match_when_negated do |experiment|
          expect_tracking_on(experiment, true, event, *event_args)
        end

        chain :for do |expected_variant|
          raise ArgumentError, 'variant name must be provided' if expected.blank?

          @expected_variant = expected_variant.to_s
        end

        chain(:with_context) { |expected_context| @expected_context = expected_context }
        chain(:on_any_instance) { @on_self = false }

        def expect_tracking_on(experiment, negated, event, *event_args)
          @experiment = experiment
          @on_self = true if require_experiment(experiment, 'track', classes: !@on_self) && @on_self.nil?
          wrapped_experiment(experiment, shallow: @on_self, failure: failure_message(:no_new, event)) do |instance|
            @experiment = instance
            allow(@experiment).to receive(:track)

            if negated
              expect(@experiment).not_to receive_tracking_call_for(event, *event_args)
            else
              expect(@experiment).to receive_tracking_call_for(event, *event_args)
            end
          end
        end

        def receive_tracking_call_for(event, *event_args)
          receive(:track).with(*[event, *event_args]) do # rubocop:disable CodeReuse/ActiveRecord
            if @expected_variant
              expect(@experiment.variant.name).to eq(@expected_variant), failure_message(:variant, event)
            end

            if @expected_context
              expect(@experiment.context.value).to include(@expected_context), failure_message(:context, event)
            end
          end
        end

        def failure_message(failure_type, event)
          case failure_type
          when :variant
            <<~MESSAGE.strip
              expected #{@experiment.inspect} to have tracked #{event.inspect} for variant
                  expected variant: #{@expected_variant}
                    actual variant: #{@experiment.variant.name}
            MESSAGE
          when :context
            <<~MESSAGE.strip
              expected #{@experiment.inspect} to have tracked #{event.inspect} with context
                  expected context: #{@expected_context}
                    actual context: #{@experiment.context.value}
            MESSAGE
          when :no_new
            %(expected #{@experiment.inspect} to have tracked #{event.inspect}, but no new instances were created)
          end
        end
      end
    end
  end
end

RSpec.configure do |config|
  config.include Gitlab::Experiment::RSpecHelpers
  config.include Gitlab::Experiment::Dsl

  config.include Gitlab::Experiment::RSpecMatchers, :experiment
  config.define_derived_metadata(file_path: Regexp.new('/spec/experiments/')) do |metadata|
    metadata[:type] = :experiment
  end
end
