/*
 * Entry point and user interface for the applet.
 *
 * Rhythmbox Applet
 * Copyright (C) 2004 Paul Kuliniewicz <kuliniew@purdue.edu>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#include "rb-proxy.h"

#include <panel-applet.h>
#include <libgnomeui/gnome-about.h>
#include <gtk/gtkbutton.h>
#include <gtk/gtkhbox.h>
#include <gtk/gtkiconfactory.h>
#include <gtk/gtkimage.h>
#include <gtk/gtklabel.h>
#include <gtk/gtktooltips.h>
#include <gtk/gtkvbox.h>

#include <string.h>

/* Avoid compiler warnings in BONOBO_UI_VERB and friends. */
#define MY_BONOBO_UI_UNSAFE_VERB(name,cb) \
	{ (name), ((BonoboUIVerbFn)(cb)), NULL, NULL }
#define MY_BONOBO_UI_VERB_END             \
	{ NULL, NULL, NULL, NULL }

#define RB_STOCK_PLAY		"rhythmbox-play"
#define RB_STOCK_PAUSE		"rhythmbox-pause"
#define RB_STOCK_STOP		"rhythmbox-stop"
#define RB_STOCK_PREVIOUS	"rhythmbox-previous"
#define RB_STOCK_NEXT		"rhythmbox-next"

typedef struct {
	GtkWidget *applet;

	RbProxy *proxy;
	gchar *song_tip;

	GtkIconFactory *factory;
	GtkTooltips *tooltips;

	GtkWidget *main_box;
	GtkWidget *button_box;
	GtkWidget *time;
	GtkWidget *time_event;
	GtkWidget *previous;
	GtkWidget *play_pause_stop;
	GtkWidget *next;

	GtkWidget *play_image;
	GtkWidget *pause_image;
	GtkWidget *stop_image;
	GtkWidget *previous_image;
	GtkWidget *next_image;
} AppletData;

static gboolean rb_applet_fill (PanelApplet *applet,
			    const gchar *iid,
			    gpointer unused);

static void set_play_pause_stop_appearance (AppletData *data,
					    GtkWidget *image,
					    const gchar *tip);
static void set_initial_state (AppletData *data);

static GtkIconFactory *create_icon_factory (void);
static void destroy_icon_factory (GtkIconFactory *factory);

static void about_cb (BonoboUIComponent *uic,
		      AppletData *data,
		      const gchar *verbname);
static void play_pause_stop_cb (GtkButton *button,
				AppletData *data);
static void previous_cb (GtkButton *button,
			 AppletData *data);
static void next_cb (GtkButton *button,
		     AppletData *data);

static gboolean button_press_cb (GtkWidget *widget,
				 GdkEventButton *event,
				 AppletData *data);
static void change_orient_cb (PanelApplet *applet,
			      PanelAppletOrient orient,
			      AppletData *data);
static void destroy_cb (GtkWidget *widget,
			AppletData *data);

static void time_updated_cb (RbProxy *proxy,
			     glong play_time,
			     AppletData *data);
static void song_changed_cb (RbProxy *proxy,
			     const RbProxySongInfo *song,
			     AppletData *data);
static void playing_changed_cb (RbProxy *proxy,
				gboolean playing,
				AppletData *data);
static void connection_changed_cb (RbProxy *proxy,
				   gboolean connected,
				   AppletData *data);

static const BonoboUIVerb rb_applet_menu_verbs[] = {
	MY_BONOBO_UI_UNSAFE_VERB ("About", about_cb),
	MY_BONOBO_UI_VERB_END
};

/* The applet itself. */

static gboolean
rb_applet_fill (PanelApplet *applet,
		const gchar *iid,
		gpointer unused)
{
	AppletData *data;
	GtkWidget *label;

	if (strcmp (iid, "OAFIID:GNOME_Rhythmbox_Applet") != 0)
		return FALSE;

	data = g_new0 (AppletData, 1);
	data->applet = GTK_WIDGET (applet);
	panel_applet_set_flags (applet, PANEL_APPLET_EXPAND_MINOR);

	data->proxy = rb_proxy_new ();
	data->factory = create_icon_factory ();
	data->tooltips = gtk_tooltips_new ();
	data->song_tip = NULL;
	data->main_box = NULL;

	/* Images */

	data->play_image = gtk_image_new_from_stock
		(RB_STOCK_PLAY, GTK_ICON_SIZE_SMALL_TOOLBAR);
	g_object_ref (data->play_image);
	data->pause_image = gtk_image_new_from_stock
		(RB_STOCK_PAUSE, GTK_ICON_SIZE_SMALL_TOOLBAR);
	g_object_ref (data->pause_image);
	data->stop_image = gtk_image_new_from_stock
		(RB_STOCK_STOP, GTK_ICON_SIZE_SMALL_TOOLBAR);
	g_object_ref (data->stop_image);
	data->previous_image = gtk_image_new_from_stock
		(RB_STOCK_PREVIOUS, GTK_ICON_SIZE_SMALL_TOOLBAR);
	g_object_ref (data->previous_image);
	data->next_image = gtk_image_new_from_stock
		(RB_STOCK_NEXT, GTK_ICON_SIZE_SMALL_TOOLBAR);
	g_object_ref (data->next_image);

	/* Applet widgets */

	data->time_event = gtk_event_box_new ();
	data->time = gtk_label_new ("--:--");
	gtk_container_add (GTK_CONTAINER (data->time_event), data->time);

	data->button_box = gtk_hbox_new (FALSE, 0);

	data->previous = gtk_button_new ();
	gtk_container_add
		(GTK_CONTAINER (data->previous), data->previous_image);
	gtk_tooltips_set_tip (data->tooltips, data->previous,
			      "Play previous song", NULL);
	g_signal_connect (data->previous, "clicked",
			  G_CALLBACK (previous_cb), data);
	g_signal_connect (data->previous, "button_press_event",
			  G_CALLBACK (button_press_cb), data);
	gtk_box_pack_start_defaults
		(GTK_BOX (data->button_box), data->previous);

	data->play_pause_stop = gtk_button_new ();
	gtk_container_add
		(GTK_CONTAINER (data->play_pause_stop), data->play_image);
	gtk_tooltips_set_tip (data->tooltips, data->play_pause_stop,
			      "Start playing", NULL);
	g_signal_connect (data->play_pause_stop, "clicked",
			  G_CALLBACK (play_pause_stop_cb), data);
	g_signal_connect (data->play_pause_stop, "button_press_event",
			  G_CALLBACK (button_press_cb), data);
	gtk_box_pack_start_defaults
		(GTK_BOX (data->button_box), data->play_pause_stop);

	data->next = gtk_button_new ();
	gtk_container_add
		(GTK_CONTAINER (data->next), data->next_image);
	gtk_tooltips_set_tip (data->tooltips, data->next,
			      "Play next song", NULL);
	g_signal_connect (data->next, "clicked",
			  G_CALLBACK (next_cb), data);
	g_signal_connect (data->next, "button_press_event",
			  G_CALLBACK (button_press_cb), data);
	gtk_box_pack_start_defaults
		(GTK_BOX (data->button_box), data->next);

	panel_applet_setup_menu_from_file (applet,
					   UI_DIR,
					   "GNOME_Rhythmbox_Applet.xml",
					   NULL,
					   rb_applet_menu_verbs,
					   data);

	g_signal_connect (data->proxy, "song_changed",
			  G_CALLBACK (song_changed_cb), data);

	g_signal_connect (data->proxy, "playing_changed",
			  G_CALLBACK (playing_changed_cb), data);

	g_signal_connect (data->proxy, "connection_changed",
			  G_CALLBACK (connection_changed_cb), data);

	g_signal_connect (data->proxy, "time_updated",
			  G_CALLBACK (time_updated_cb), data);

	g_signal_connect (applet, "change_orient",
			  G_CALLBACK (change_orient_cb), data);

	g_signal_connect (applet, "destroy",
			  G_CALLBACK (destroy_cb), data);

	/* Initialize with current state. */

	set_initial_state (data);

	/* Finish up. */

	gtk_widget_show_all (GTK_WIDGET (applet));

	return TRUE;
}

static void
set_initial_state (AppletData *data)
{
	/* eew, hackish */

	RbProxySongInfo *song = rb_proxy_get_song_info (data->proxy);
	gboolean connected = rb_proxy_is_connected (data->proxy);
	gboolean playing = rb_proxy_is_playing (data->proxy);
	PanelAppletOrient orient = panel_applet_get_orient (
		PANEL_APPLET (data->applet));

	change_orient_cb (PANEL_APPLET (data->applet), orient, data);
	connection_changed_cb (data->proxy, connected, data);
	song_changed_cb (data->proxy, song, data);
	playing_changed_cb (data->proxy, playing, data);

	rb_proxy_song_info_free (song);
}

static void
set_play_pause_stop_appearance (AppletData *data,
				GtkWidget *image,
				const gchar *tip)
{
	gtk_container_remove (
		GTK_CONTAINER (data->play_pause_stop),
		gtk_bin_get_child (GTK_BIN (data->play_pause_stop)));
	gtk_container_add (GTK_CONTAINER (data->play_pause_stop), image);
	gtk_widget_show (image);
	gtk_tooltips_set_tip (data->tooltips, data->play_pause_stop,
			      tip, NULL);
}

/* Applet support. */

static GtkIconFactory *
create_icon_factory (void)
{
	/*
	 * Most of this was taken from the Rhythmbox source, from
	 * lib/rb-stock-icons.c.
	 */

	GtkIconFactory *factory;
	int i;

	static const char *items[] = {
		RB_STOCK_PLAY,
		RB_STOCK_PAUSE,
		RB_STOCK_STOP,
		RB_STOCK_PREVIOUS,
		RB_STOCK_NEXT,
	};
	
	factory = gtk_icon_factory_new ();
	gtk_icon_factory_add_default (factory);

	for (i = 0; i < (int) G_N_ELEMENTS (items); i++)
	{
		GtkIconSet *icon_set;
		GdkPixbuf *pixbuf;
		gchar *file;

		file = g_strconcat (SHARE_DIR, items[i], ".png", NULL);
		pixbuf = gdk_pixbuf_new_from_file (file, NULL);
		g_free (file);
		icon_set = gtk_icon_set_new_from_pixbuf (pixbuf);
		gtk_icon_factory_add (factory, items[i], icon_set);
		gtk_icon_set_unref (icon_set);
		g_object_unref (G_OBJECT (pixbuf));
	}

	return factory;
}

static void
destroy_icon_factory (GtkIconFactory *factory)
{
	g_return_if_fail (factory != NULL);

	gtk_icon_factory_remove_default (factory);
	g_object_unref (G_OBJECT (factory));
}

/* Callbacks for user actions. */

static void
play_pause_stop_cb (GtkButton *button, AppletData *data)
{
	rb_proxy_play_pause_stop (data->proxy);
}

static void
previous_cb (GtkButton *button, AppletData *data)
{
	rb_proxy_previous (data->proxy);
}

static void
next_cb (GtkButton *button, AppletData *data)
{
	rb_proxy_next (data->proxy);
}

static void
about_cb (BonoboUIComponent *uic,
	  AppletData *data,
	  const gchar *verbname)
{
	static const gchar *authors[] = {
		"Paul Kuliniewicz <kuliniew@purdue.edu>",
		NULL
	};

	static const gchar *documenters[] = {
		NULL
	};

	GtkWidget *about_dialog;

	about_dialog = gnome_about_new (
		"Rhythmbox Applet", PACKAGE_VERSION,
		"(C) 2004 Paul Kuliniewicz",
		"This applet lets you control Rhythmbox from a panel.",
		authors,
		documenters,
		NULL,
		NULL);

	g_signal_connect (about_dialog, "destroy",
			  G_CALLBACK (gtk_widget_destroyed), &about_dialog);

	gtk_widget_show (about_dialog);
}

/* Callbacks for other user interface events. */

static gboolean
button_press_cb (GtkWidget *widget, GdkEventButton *event, AppletData *data)
{
	/* Make sure non-left-clicks propagate up to the applet. */
	if (event->button != 1)
		g_signal_stop_emission_by_name (widget, "button_press_event");
	return FALSE;
}

static void
change_orient_cb (PanelApplet *applet,
		  PanelAppletOrient orient,
		  AppletData *data)
{
	if (data->main_box != NULL)
	{
		g_object_ref (G_OBJECT (data->time_event));
		gtk_container_remove (GTK_CONTAINER (data->main_box),
				      data->time_event);
		g_object_ref (G_OBJECT (data->button_box));
		gtk_container_remove (GTK_CONTAINER (data->main_box),
				      data->button_box);
		gtk_container_remove (GTK_CONTAINER (applet),
				      data->main_box);
		gtk_widget_destroy (data->main_box);
	}

	switch (orient)
	{
	case PANEL_APPLET_ORIENT_UP:
	case PANEL_APPLET_ORIENT_DOWN:
		data->main_box = gtk_hbox_new (FALSE, 5);
		break;
	case PANEL_APPLET_ORIENT_LEFT:
	case PANEL_APPLET_ORIENT_RIGHT:
		data->main_box = gtk_vbox_new (FALSE, 0);
		break;
	default:
		g_return_if_reached ();
	}

	gtk_box_pack_start_defaults (GTK_BOX (data->main_box),
				     data->time_event);
	gtk_box_pack_start_defaults (GTK_BOX (data->main_box),
				     data->button_box);
	gtk_container_add (GTK_CONTAINER (applet), data->main_box);
	gtk_widget_show_all (data->main_box);
}

static void
destroy_cb (GtkWidget *widget, AppletData *data)
{
	g_object_unref (data->play_image);
	g_object_unref (data->pause_image);
	g_object_unref (data->stop_image);
	g_object_unref (data->previous_image);
	g_object_unref (data->next_image);

	gtk_object_destroy (GTK_OBJECT (data->tooltips));
	destroy_icon_factory (data->factory);

	g_object_unref (G_OBJECT (data->proxy));

	g_free (data);
}

/* Callbacks for Rhythmbox events. */

static void
time_updated_cb (RbProxy *proxy, glong play_time, AppletData *data)
{
	if (play_time >= 0)
	{
		glong minutes = play_time / 60;
		glong seconds = play_time % 60;
		gchar *text = g_strdup_printf ("%ld:%.2ld", minutes, seconds);
		gtk_label_set_text (GTK_LABEL (data->time), text);
		g_free (text);
		text = g_strdup_printf (data->song_tip, minutes, seconds);
		gtk_tooltips_set_tip (data->tooltips, data->time_event,
				      text, NULL);
		g_free (text);
	}
	else
	{
		gtk_label_set_text (GTK_LABEL (data->time), "--:--");
		gtk_tooltips_set_tip (data->tooltips, data->time_event,
				      "Not playing", NULL);
	}
}

static void
song_changed_cb (RbProxy *proxy, const RbProxySongInfo *song, AppletData *data)
{
	g_free (data->song_tip);

	if (song != NULL)
	{
		if (song->duration > 0)
		{
			/* playing local file */
			data->song_tip = g_strdup_printf (
				"%s\nfrom %s by %s\n%%ld:%%.2ld of %d:%.2d",
				song->title,
				song->album,
				song->artist,
				song->duration / 60,
				song->duration % 60);
		}
		else
		{
			/* playing Internet radio */
			data->song_tip = g_strdup_printf (
				"%s\n%%ld:%%.2ld",
				song->title);
		}
	}
	else
		data->song_tip = g_strdup ("Not playing");

	time_updated_cb (proxy, rb_proxy_get_play_time (proxy), data);
}

static void
playing_changed_cb (RbProxy *proxy, gboolean playing, AppletData *data)
{
	GtkWidget *image;
	const gchar *tip;

	if (!playing)
	{
		image = data->play_image;
		tip = "Start playing";
	}
	else if (rb_proxy_is_pausable (proxy))
	{
		image = data->pause_image;
		tip = "Pause playback";
	}
	else
	{
		image = data->stop_image;
		tip = "Stop playback";
	}

	set_play_pause_stop_appearance (data, image, tip);
}

static void
connection_changed_cb (RbProxy *proxy, gboolean connected, AppletData *data)
{
	gtk_widget_set_sensitive (data->previous, connected);
	gtk_widget_set_sensitive (data->next, connected);
}

/* Entry point. */

PANEL_APPLET_BONOBO_FACTORY ("OAFIID:GNOME_Rhythmbox_Applet_Factory",
			     PANEL_TYPE_APPLET,
			     "Rhythmbox Applet Factory",
			     "0",
			     rb_applet_fill,
			     NULL);
