/* imv.c - Main routines for imv.
 *
 * Copyright (C) 2001  Oskar Liljeblad
 *
 * This file is part of the file renaming utilities (renameutils).
 *
 * This software is copyrighted work licensed under the terms of the
 * GNU General Public License. Please consult the file `COPYING' for
 * details.
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif
/* POSIX */
#if HAVE_UNISTD_H
#include <unistd.h>
#endif
/* C89 */
#include <stdlib.h>
#include <signal.h>
#include <stdio.h>
#include <errno.h>
#include <locale.h>
/* POSIX/gnulib */
#include <stdbool.h>
/* gnulib */
#include <progname.h>
/* Gettext */
#include <gettext.h> /* will include <libintl.h> if ENABLE_NLS */
#define _(String) gettext(String)
/* Readline */
#include <readline/readline.h>
#include <readline/history.h>
/* common */
#include "common/memory.h"
#include "common/error.h"
#include "common/io-utils.h"
#include "common/string-utils.h"
#include "common/common.h"

#define PROGRAM "imv"

static char *sourcefile = NULL;
static char *first_text = NULL;

static int
insert_first_text(void)
{
	return rl_insert_text(first_text);
}

static void
int_signal_handler(int signal)
{
	puts("");
	warn(_("no changes made"));
	exit(0);
}

static void
display_imv_help(void)
{
	printf(_("Usage: %s [OPTION]... FILE\n\
Rename a file by editing the destination name using GNU readline.\n\
All options except the options listed below are passed to mv(1).\n\
\n\
      --help                 display this help and exit\n\
      --version              output version information and exit\n\
\n\
Report bugs to <%s>.\n"), program_name, PACKAGE_BUGREPORT);
}

static void
display_imv_version(void)
{
	printf(_("%s (%s) %s\n\
Written by Oskar Liljeblad.\n\
\n\
Copyright (C) 2001-2004 Oskar Liljeblad.\n\
This is free software; see the source for copying conditions.  There is NO\n\
warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n"),
		PROGRAM, PACKAGE, VERSION);
}

int
main(int argc, char **argv)
{
	struct sigaction action;
	char *newname;
	bool gotend = false;
	int c;
	int len;

    	set_program_name(argv[0]);

	if (setlocale (LC_ALL, "") == NULL)
    	    die(_("invalid locale"));
	if (bindtextdomain (PACKAGE, LOCALEDIR) == NULL)
    	    die_errno(NULL);
	if (textdomain (PACKAGE) == NULL)
    	    die_errno(NULL);

	if (argc == 2 && strcmp(argv[1], "--help") == 0) {
		display_imv_help();
		exit(0);
	}
	if (argc == 2 && strcmp(argv[1], "--version") == 0) {
		display_imv_version();
		exit(0);
	}

	memclear(&action, sizeof(sigaction));
	action.sa_handler = int_signal_handler;
	action.sa_flags = SA_RESTART;
	if (sigaction(SIGINT, &action, NULL) < 0)
		die_errno(NULL);
	action.sa_handler = SIG_IGN;
	if (sigaction(SIGQUIT, &action, NULL) < 0)
		die_errno(NULL);

	for (c = 1; c < argc; c++) {
		if (!gotend && strcmp(argv[c], "--") == 0) {
			gotend = true;
		} else if (gotend || argv[c][0] != '-') {
			if (sourcefile != NULL)
				die(_("too many arguments"));
			sourcefile = argv[c];
		}
	}

	if (sourcefile == NULL)
		die(_("missing file argument"));

	len = strlen(sourcefile);
	if (sourcefile[len-1] == '/')
		sourcefile[len-1] = '\0';
	first_text = xstrdup(sourcefile);

	do {
		add_history(sourcefile);
		rl_startup_hook = insert_first_text;
		newname = readline("> ");

		if (newname != NULL
				&& strcmp(newname, "") != 0
				&& strcmp(newname, sourcefile) != 0) {
			char **args;

			if (file_exists(newname)) {
				char *line;
				printf(_("%s: overwrite `%s'? "), program_name, newname);
				fflush(stdout);
				line = read_line(stdin);
				if (line == NULL) {
					free(newname);
					newname = NULL;
					break;
				}
				chomp(line);
				if (strcmp(line, "y") != 0) {
					free(line);
					free(first_text);
					first_text = newname;
					continue;
				}
			}

			args = xmalloc(argc + (gotend ? 2 : 3));
			memcpy(args, argv, argc * sizeof(char *));
			if (gotend) {
				args[argc] = newname;
				args[argc+1] = NULL;
			} else {
				args[argc] = "--";
				args[argc+1] = newname;
				args[argc+2] = NULL;
			}
			args[0] = "mv";
			execvp("mv", args);
			die(_("cannot execute mv"));
		}
	} while (false);

	if (newname == NULL)
		puts("");

	warn(_("no changes made"));
	exit(0);
}
