// $Id$
//
//  Copyright (C) 2013 Paolo Tosco
//
//  Copyright (C) 2004-2006 Rational Discovery LLC
//
//   @@ All Rights Reserved @@
//  This file is part of the RDKit.
//  The contents are covered by the terms of the BSD license
//  which is included in the file license.txt, found at the root
//  of the RDKit source tree.
//
#ifdef WIN32
#define _USE_MATH_DEFINES
#endif
#include <cmath>
#include "Params.h"

#include <iostream>
#include <sstream>
#include <RDGeneral/StreamOps.h>
#include <boost/lexical_cast.hpp>
#include <boost/tokenizer.hpp>
#include <Geometry/point.h>

typedef boost::tokenizer<boost::char_separator<char> > tokenizer;

namespace ForceFields {
namespace MMFF {

class std::unique_ptr<MMFFAromCollection> MMFFAromCollection::ds_instance = nullptr;

extern const std::uint8_t defaultMMFFArom[];

MMFFAromCollection *MMFFAromCollection::getMMFFArom(
    const std::uint8_t *mmffArom) {
  if (!ds_instance || mmffArom) {
    ds_instance.reset(new MMFFAromCollection(mmffArom));
  }
  return ds_instance.get();
}

MMFFAromCollection::MMFFAromCollection(const std::uint8_t *mmffArom) {
  if (!mmffArom) {
    mmffArom = defaultMMFFArom;
  }
  for (unsigned int i = 0; i < sizeof(mmffArom) / sizeof(std::uint8_t); ++i) {
    d_params.push_back(mmffArom[i]);
  }
}

const std::uint8_t defaultMMFFArom[] = {37, 38, 39, 44, 58, 59, 63, 64, 65,
                                          66, 69, 76, 78, 79, 80, 81, 82};

class std::unique_ptr<MMFFDefCollection> MMFFDefCollection::ds_instance = nullptr;

extern const std::string defaultMMFFDef;

MMFFDefCollection *MMFFDefCollection::getMMFFDef(const std::string &mmffDef) {
  if (!ds_instance || !mmffDef.empty()) {
    ds_instance.reset(new MMFFDefCollection(mmffDef));
  }
  return ds_instance.get();
}

MMFFDefCollection::MMFFDefCollection(std::string mmffDef) {
  if (mmffDef.empty()) {
    mmffDef = defaultMMFFDef;
  }
  std::istringstream inStream(mmffDef);
  std::string inLine = RDKit::getLine(inStream);
  unsigned int oldAtomType = 0;
  unsigned int atomType;
  while (!(inStream.eof())) {
    if (inLine[0] != '*') {
      MMFFDef mmffDefObj;
      boost::char_separator<char> tabSep("\t");
      tokenizer tokens(inLine, tabSep);
      tokenizer::iterator token = tokens.begin();

      // skip first token
      ++token;
#ifdef RDKIT_MMFF_PARAMS_USE_STD_MAP
      unsigned int atomType = boost::lexical_cast<unsigned int>(*token);
#else
      atomType = (std::uint8_t)(boost::lexical_cast<unsigned int>(*token));
#endif
      ++token;
      // Level 2 (currently = Level 1, see MMFF.I page 513)
      mmffDefObj.eqLevel[0] =
          (std::uint8_t)(boost::lexical_cast<unsigned int>(*token));
      ++token;
      // Level 3
      mmffDefObj.eqLevel[1] =
          (std::uint8_t)(boost::lexical_cast<unsigned int>(*token));
      ++token;
      // Level 4
      mmffDefObj.eqLevel[2] =
          (std::uint8_t)(boost::lexical_cast<unsigned int>(*token));
      ++token;
      // Level 5
      mmffDefObj.eqLevel[3] =
          (std::uint8_t)(boost::lexical_cast<unsigned int>(*token));
      ++token;
      if (atomType != oldAtomType) {
#ifdef RDKIT_MMFF_PARAMS_USE_STD_MAP
        d_params[atomType] = mmffDefObj;
#else
        d_params.push_back(mmffDefObj);
#endif
        oldAtomType = atomType;
      }
    }
    inLine = RDKit::getLine(inStream);
  }
}
const std::string defaultMMFFDef =
    "*\n"
    "*          Copyright (c) Merck and Co., Inc., 1994, 1995, 1996\n"
    "*                         All Rights Reserved\n"
    "*\n"
    "*      PRIMARY\n"
    "* MMFF   MMFF                                 \n"
    "*SYMBOL  TYPE   DEFAULT TYPES       DEFINITION           \n"
    "*\n"
    "CR	1	1	1	1	0	ALKYL	CARBON	\n"
    "C=C	2	2	2	1	0	VINYLIC	\n"
    "*	CSP2	2	2	2	1	0	GENERIC	CSP2	\n"
    "*	CGD	2	2	2	1	0	GUANIDINE	CARBON	"
    "\n"
    "C=O	3	3	3	1	0	GENERAL	CARBONYL	"
    "C	\n"
    "*	C=N	3	3	3	1	0	C=N	\n"
    "*	C=OR	3	3	3	1	0	KETONE	OR	"
    "ALDEHYDE	CO	\n"
    "*	C=ON	3	3	3	1	0	AMIDE	CARBONYL	"
    "\n"
    "*	COO	3	3	3	1	0	CARBOXYLIC	ACID	"
    "OF	EST	\n"
    "*	COON	3	3	3	1	0	CARBAMATE	"
    "CARBONYL	\n"
    "*	COOO	3	3	3	1	0	CARBONIC	ACID	"
    "OR	ESTER	\n"
    "*	C=OS	3	3	3	1	0	THIOESTER,	C=O	"
    "\n"
    "*	C=S	3	3	3	1	0	THIOESTER,	C=S	"
    "\n"
    "*	C=SN	3	3	3	1	0	THIOAMIDE	\n"
    "*	CSO2	3	3	3	1	0	CARBON	IN	>C=SO2	"
    "\n"
    "CSP	4	4	4	1	0	ACETYLENIC	C	"
    "\n"
    "*	=C=	4	4	4	1	0	ALLENIC	C	\n"
    "HC	5	5	5	5	0	H-C	\n"
    "*	HSI	5	5	5	5	0	H-SI	\n"
    "*	HP	5	5	5	5	0	H-P	\n"
    "OR	6	6	6	6	0	O-CSP3	\n"
    "*	OH2	6	6	6	6	0	OXYGEN	IN	H2O	"
    "\n"
    "*	OC=O	6	6	6	6	0	ESTER	OR	ACID	"
    "-O-	\n"
    "*	OC=C	6	6	6	6	0	ENOL	OR	"
    "PHEMOLIC	O	\n"
    "*	OC=N	6	6	6	6	0	OXYGEN	IN	-O-C=N	"
    "MOIETY\n"
    "*	OSO3	6	6	6	6	0	DIVALENT	O	"
    "IN	SULFATE\n"
    "*	OSO2	6	6	6	6	0	DIVALENT	O	"
    "IN	SULFITE\n"
    "*	OSO	6	6	6	6	0	PAIR	OF	"
    "DIVALENT	O	ON	S\n"
    "*	-OS	6	6	6	6	0	OTHER	DIVALENT	"
    "O	ON	S\n"
    "*	OPO3	6	6	6	6	0	DIVALENT	O	"
    "IN	PHOSPHATE\n"
    "*	OPO2	6	6	6	6	0	DIVALENT	O	"
    "IN	PHOSPHITE\n"
    "*	OPO	6	6	6	6	0	PAIR	OF	"
    "DIVALENT	O	ON	P\n"
    "*	-OP	6	6	6	6	0	OTHER	DIVALENT	"
    "O	ON	P\n"
    "*	-O-	6	6	6	6	0	GENERAL	DIVALENT	"
    "OX	\n"
    "O=C	7	7	7	6	0	O=C,	GENERIC	\n"
    "*	O=CN	7	7	7	6	0	O=C	IN	AMIDES	"
    "\n"
    "*	O=CR	7	7	7	6	0	O=C	IN	KET,	"
    "ALD	\n"
    "*	O=CO	7	7	7	6	0	O=C	IN	ACIDS,	"
    "ESTERS	\n"
    "*	O=S	7	7	7	6	0	TERMINAL	O	"
    "SULFOXIDES	\n"
    "*	O=N	7	7	7	6	0	NITROSO-GROUP	OXYGEN	"
    "\n"
    "NR	8	8	8	8	0	AMINE	N	\n"
    "N=C	9	9	9	8	0	N=C,	IMINES	\n"
    "*	N=N	9	9	9	8	0	N=N,	AZO	"
    "COMPOUNDS	\n"
    "NC=O	10	10	10	8	0	N-C=O,	AMIDES	\n"
    "*	NC=S	10	10	10	8	0	N-C=S	(DELOC	LP)	"
    "\n"
    "*	NN=C	10	10	10	8	0	N-N=C	(DELOC	LP)	"
    "\n"
    "*	NN=N	10	10	10	8	0	N-N=N	(DELOC	LP)	"
    "\n"
    "F	11	11	11	11	0	FLUORINE	\n"
    "CL	12	12	12	12	0	CHLORINE	\n"
    "BR	13	13	13	13	0	BROMINE	\n"
    "I	14	14	14	14	0	IODINE	\n"
    "S	15	15	15	15	0	THIOL,	SULFIDE	\n"
    "S=C	16	16	16	15	0	S	DOUBLY	BONDED	"
    "TO	C	\n"
    "S=O	17	17	17	15	0	SULFOXIDE	S	"
    "\n"
    "SO2	18	18	18	15	0	SULFONE	S	\n"
    "*	SO2N	18	18	18	15	0	SULFONAMIDE	S	"
    "\n"
    "*	SO3	18	18	18	15	0	SULFONATE	S	"
    "\n"
    "*	=SO2	18	18	18	15	0	OXYGENATED	SULFONE	"
    "S	\n"
    "*	SNO	18	18	18	15	0	NITROGEN	ANALOG	"
    "OF	SO2	\n"
    "SI	19	19	19	19	0	SILICON	\n"
    "CR4R	20	20	1	1	0	C	IN	"
    "CYCLOBUTYL	\n"
    "HOR	21	21	21	5	0	H-O,	ALCOHOLS	"
    "\n"
    "*	HO	21	21	21	5	0	GENERAL	H	ON	"
    "O	\n"
    "CR3R	22	22	22	1	0	C	IN	"
    "CYCLOPROPLY	\n"
    "HNR	23	23	23	5	0	H-N,	AMINES	\n"
    "*	H3N	23	23	23	5	0	H,	AMMONIA	\n"
    "*	HPYL	23	23	23	5	0	H-N	IN	PYRROLE	"
    "\n"
    "*	HN	23	23	23	5	0	GENERAL	H-N	\n"
    "HOCO	24	24	24	5	0	H-O,	ACIDS	\n"
    "*	HOP	24	21	21	5	0	H-O-P,	PHOS	ACIDS	"
    "\n"
    "PO4	25	25	25	25	0	PHOSPHODIESTER	\n"
    "*	PO3	25	25	25	25	0	TETRACRD	P,	"
    "3	OXYGENS	\n"
    "*	PO2	25	25	25	25	0	TETRACRD	P,	"
    "2	OXYGENS	\n"
    "*	PO	25	25	25	25	0	TETRACRD	P,	"
    "2	OXYGENS	\n"
    "*	PTET	25	25	25	25	0	GENERAL	TETRACRD	"
    "P	\n"
    "P	26	26	26	25	0	TRICOORDINATE	P	\n"
    "HN=C	27	27	28	5	0	IMINE	N-H	\n"
    "*	HN=N	27	27	28	5	0	AZO	N-H	\n"
    "HNCO	28	28	28	5	0	H-N,	AMIDES	\n"
    "*	HNCC	28	28	28	5	0	H-N,	ENAMINES	"
    "\n"
    "*	HNCS	28	28	28	5	0	H-N,	THIOAMIDES	"
    "\n"
    "*	HNCN	28	28	28	5	0	H-N,	HN-C=N	\n"
    "*	HNNC	28	28	28	5	0	H-N,	HN-N=C	\n"
    "*	HNNN	28	28	28	5	0	H-N,	HN-N=N	\n"
    "*	HSP2	28	28	28	5	0	GENERAL	H	ON	"
    "SP2	N	\n"
    "HOCC	29	29	29	5	0	H-O,	ENOLS,	PHENOLS	"
    "\n"
    "*	HOCN	29	29	29	5	0	H-O	IN	HO-C=N	"
    "\n"
    "CE4R	30	30	2	1	0	C=C	IN	4-RING	"
    "\n"
    "HOH	31	31	31	31	0	H-OH	\n"
    "O2CM	32	32	7	6	0	O,	CARBOXYLATE	"
    "ANION	\n"
    "*	OXN	32	32	7	6	0	OXIDE	ON	"
    "NITROHGEN	\n"
    "*	O2N	32	32	7	6	0	NITRO-GROUP	OXYGEN	"
    "\n"
    "*	O2NO	32	32	7	6	0	NITRO-GROUP	IN	"
    "NITRATE	\n"
    "*	O3N	32	32	7	6	0	NITRATE	ANION	OXYGEN	"
    "\n"
    "*	O-S	32	32	7	6	0	SINGLE	TERM	O	"
    "ON	TET	S	\n"
    "*	O2S	32	32	7	6	0	SULFONES,	"
    "SULFONAMIDES	\n"
    "*	O3S	32	32	7	6	0	SULFONATES,	TERM	"
    "OX	\n"
    "*	O4S	32	32	7	6	0	SO4(3-)	\n"
    "*	OSMS	32	32	7	6	0	THIOSULFINATE	O	"
    "(-1/2)	\n"
    "*	OP	32	32	7	6	0	TERMINAL	O,	"
    "O-P	\n"
    "*	O2P	32	32	7	6	0	TERMINAL	O,	"
    "O2P	GROUP	\n"
    "*	O3P	32	32	7	6	0	TERMINAL	O,	"
    "O3P	GROUP	\n"
    "*	O4P	32	32	7	6	0	TERMINAL	O,	"
    "PO4(-3)	\n"
    "*	O4CL	32	32	7	6	0	TERMINAL	O	"
    "IN	CLO4(-)	\n"
    "HOS	33	33	21	5	0	H-O-S,	SULF	ACIDS	"
    "\n"
    "NR+	34	34	8	8	0	N+,	QUATERNARY	"
    "N	\n"
    "OM	35	35	6	6	0	OXIDE	OXYGEN	ON	"
    "SP3	C	\n"
    "*	OM2	35	35	6	6	0	OXIDE	OXYGEN	ON	"
    "SP2	C	\n"
    "HNR+	36	36	36	5	0	H-N+	\n"
    "*	HNN+	36	36	36	5	0	H	ON	"
    "IMIDAZOLIUM	N	\n"
    "*	HNC+	36	36	36	5	0	H	ON	"
    "PROTONATED	N+=C-N	\n"
    "*	HGD+	36	36	36	5	0	H	ON	"
    "GUANIDINIUM	N	\n"
    "CB	37	37	2	1	0	AROMATIC	C	\n"
    "NPYD	38	38	9	8	0	AROMATIC	N,	"
    "PYRIDINE	\n"
    "NPYL	39	39	10	8	0	AROMATIC	N,	"
    "PYRROLE	\n"
    "NC=C	40	40	10	8	0	N-C=C	(DELOC	LP)	"
    "\n"
    "*	NC=N	40	40	10	8	0	N-C=N	(DELOC	LP)	"
    "\n"
    "CO2M	41	41	3	1	0	C	IN	CO2-	"
    "ANION	\n"
    "*	CS2M	41	41	3	1	0	THIOCARBOXYLATE	C	"
    "\n"
    "NSP	42	42	42	8	0	N	TRIPLE	BONDED	"
    "\n"
    "NSO2	43	43	10	8	0	N,	SULFONAMIDES	"
    "\n"
    "STHI	44	44	16	15	0	S	IN	"
    "THIOPHENE	\n"
    "NO2	45	45	10	8	0	NITRO	GROUP	N	"
    "\n"
    "*	NO3	45	45	10	8	0	NITRATE	GROUP	N	"
    "\n"
    "N=O	46	46	9	8	0	NITROSO	GROUP	N	"
    "\n"
    "NAZT	47	47	42	8	0	TERMINAL	N,	"
    "AZIDE	\n"
    "NSO	48	48	9	8	0	DIVAL.	N	IN	"
    "S(N)(O)	GP	\n"
    "O+	49	49	6	6	0	OXONIUM	(TRICOORD)	"
    "O	\n"
    "HO+	50	50	21	5	0	H	ON	OXONIUM	"
    "OXYGEN	\n"
    "O=+	51	51	7	6	0	OXENIUM	OXYGEN+	\n"
    "HO=+	52	52	21	5	0	H	ON	OXENIUM	"
    "O+	\n"
    "=N=	53	53	42	8	0	N	TWICE	DOUBLE	"
    "BONDED	\n"
    "N+=C	54	54	9	8	0	IMINIUM	NITROGEN	"
    "\n"
    "*	N+=N	54	54	9	8	0	AZONIUM	NITROGEN	"
    "\n"
    "NCN+	55	55	10	8	0	N	IN	+N=C-N:	"
    ";	Q=1/2	\n"
    "NGD+	56	56	10	8	0	GUANIDINIUM	N;	"
    "Q=1/3	\n"
    "CGD+	57	57	2	1	0	GUANIDINIUM	CARBON	"
    "\n"
    "*	CNN+	57	57	2	1	0	C	IN	+N=C-N	"
    "RESONANCE	\n"
    "NPD+	58	58	10	8	0	N	PYRIDINIUM	"
    "ION	\n"
    "OFUR	59	59	6	6	0	AROMATIC	O,	"
    "FURAN	\n"
    "C%	60	60	4	1	0	ISONITRILE	CARBON	\n"
    "NR%	61	61	42	8	0	ISONITRILE	N	"
    "\n"
    "NM	62	62	10	8	0	SULFONAMIDE	N-	\n"
    "C5A	63	63	2	1	0	ALPHA	AROM	5-RING	"
    "C	\n"
    "C5B	64	64	2	1	0	BETA	AROM	5-RING	"
    "C	\n"
    "N5A	65	65	9	8	0	ALPHA	AROM	5-RING	"
    "N	\n"
    "N5B	66	66	9	8	0	ALPHA	AROM	5-RING	"
    "N	\n"
    "N2OX	67	67	9	8	0	NITROGEN	IN	"
    "N-OXIDE	\n"
    "N3OX	68	68	8	8	0	NITROGEN	IN	"
    "N-OXIDE	\n"
    "NPOX	69	69	9	8	0	NITROGEN	IN	"
    "N-OXIDE	\n"
    "OH2	70	70	70	70	70	OXYGEN	IN	WATER	"
    "\n"
    "HS	71	71	5	5	0	H-S	\n"
    "S2CM	72	72	16	15	0	THIOCARBOXYLATE	S	"
    "\n"
    "*	S-P	72	72	16	15	0	TERMINAL	SULFUR	"
    "ON	P	\n"
    "*	SM	72	72	16	15	0	TERMINAL	SULFUR	"
    "ON	C	\n"
    "*	SSMO	72	72	16	15	0	TERM	S,	"
    "THIOSULFINATE	\n"
    "SO2M	73	73	18	15	0	SULFUR	IN	"
    "SULFINATE	\n"
    "*	SSOM	73	73	18	15	0	SULFUR,	THIOSULFINATE	"
    "\n"
    "=S=O	74	74	17	15	0	SULFINYL	SULFUR,	"
    "C=S=O	\n"
    "-P=C	75	75	26	25	0	P	DOUBLY	BONDED	"
    "TO	C\n"
    "N5M	76	76	9	8	0	NEG	N	IN	"
    "TETRAZOLE	AN	\n"
    "CLO4	77	77	12	12	0	CHLORINE	IN	"
    "CLO4(-)	\n"
    "C5	78	78	2	1	0	GENERAL	AROM	5-RING	"
    "C	\n"
    "N5	79	79	9	8	0	GENERAL	AROM	5-RING	"
    "N	\n"
    "CIM+	80	80	2	1	0	C	IN	N-C-N,	"
    "IM+	ION	\n"
    "NIM+	81	81	10	8	0	N	IN	N-C-N,	"
    "IM+	ION	\n"
    "N5AX	82	82	9	8	0	5R	NITROGEN	"
    "IN	N-OXIDE	\n"
    "*	N5BX	82	82	9	8	0	5R	NITROGEN	"
    "IN	N-OXIDE	\n"
    "*	N5OX	82	82	9	8	0	5R	NITROGEN	"
    "IN	N-OXIDE	\n"
    "FE+2	87	87	87	87	87	IRON	+2	"
    "CATION\n"
    "FE+3	88	88	88	88	88	IRON	+3	"
    "CATION\n"
    "F-	89	89	89	89	89	FLUORIDE	ANION\n"
    "CL-	90	90	90	90	90	CHLORIDE	ANION\n"
    "BR-	91	91	91	91	91	BROMIDE	ANION\n"
    "LI+	92	92	92	92	92	LITHIUM	CATION	\n"
    "NA+	93	93	93	93	93	SODIUM	CATION	\n"
    "K+	94	94	94	94	94	POTASSIUM	CATION	\n"
    "ZN+2	95	95	95	95	95	DIPOSITIVE	ZINC	"
    "CATION	\n"
    "*	ZINC	95	95	95	95	95	DIPOSITIVE	ZINC	"
    "CATION	\n"
    "CA+2	96	96	96	96	96	DIPOSITIVE	CALCIUM	"
    "CATION\n"
    "CU+1	97	97	97	97	97	MONOPOSITIVE	COPPER	"
    "CATION\n"
    "CU+2	98	98	98	98	98	DIPOSITIVE	COPPER	"
    "CATION\n"
    "MG+2	99	99	99	99	99	DIPOSITIVE	"
    "MAGNESIUM	CATION\n";

class std::unique_ptr<MMFFPropCollection> MMFFPropCollection::ds_instance = nullptr;

extern const std::string defaultMMFFProp;

MMFFPropCollection *MMFFPropCollection::getMMFFProp(
    const std::string &mmffProp) {
  if (!ds_instance || !mmffProp.empty()) {
    ds_instance.reset(new MMFFPropCollection(mmffProp));
  }
  return ds_instance.get();
}

MMFFPropCollection::MMFFPropCollection(std::string mmffProp) {
  if (mmffProp.empty()) {
    mmffProp = defaultMMFFProp;
  }
  std::istringstream inStream(mmffProp);
  std::string inLine = RDKit::getLine(inStream);
  while (!(inStream.eof())) {
    if (inLine[0] != '*') {
      MMFFProp mmffPropObj;
      boost::char_separator<char> tabSep("\t");
      tokenizer tokens(inLine, tabSep);
      tokenizer::iterator token = tokens.begin();

#ifdef RDKIT_MMFF_PARAMS_USE_STD_MAP
      unsigned int atomType = boost::lexical_cast<unsigned int>(*token);
#else
      d_iAtomType.push_back(
          (std::uint8_t)boost::lexical_cast<unsigned int>(*token));
#endif
      ++token;
      mmffPropObj.atno =
          (std::uint8_t)(boost::lexical_cast<unsigned int>(*token));
      ++token;
      mmffPropObj.crd =
          (std::uint8_t)(boost::lexical_cast<unsigned int>(*token));
      ++token;
      mmffPropObj.val =
          (std::uint8_t)(boost::lexical_cast<unsigned int>(*token));
      ++token;
      mmffPropObj.pilp =
          (std::uint8_t)(boost::lexical_cast<unsigned int>(*token));
      ++token;
      mmffPropObj.mltb =
          (std::uint8_t)(boost::lexical_cast<unsigned int>(*token));
      ++token;
      mmffPropObj.arom =
          (std::uint8_t)(boost::lexical_cast<unsigned int>(*token));
      ++token;
      mmffPropObj.linh =
          (std::uint8_t)(boost::lexical_cast<unsigned int>(*token));
      ++token;
      mmffPropObj.sbmb =
          (std::uint8_t)(boost::lexical_cast<unsigned int>(*token));
      ++token;
#ifdef RDKIT_MMFF_PARAMS_USE_STD_MAP
      d_params[atomType] = mmffPropObj;
#else
      d_params.push_back(mmffPropObj);
#endif
    }
    inLine = RDKit::getLine(inStream);
  }
}
const std::string defaultMMFFProp =
    "*\n"
    "*          Copyright (c) Merck and Co., Inc., 1994, 1995, 1996\n"
    "*                         All Rights Reserved\n"
    "*\n"
    "* MMFFPROP - MMFF atom-type properties\n"
    "*\n"
    "* atype aspec crd val  pilp mltb arom lin sbmb\n"
    "1	6	4	4	0	0	0	0	0\n"
    "2	6	3	4	0	2	0	0	1\n"
    "3	6	3	4	0	2	0	0	1\n"
    "4	6	2	4	0	3	0	1	1\n"
    "5	1	1	1	0	0	0	0	0\n"
    "6	8	2	2	1	0	0	0	0\n"
    "7	8	1	2	0	2	0	0	0\n"
    "8	7	3	3	1	0	0	0	0\n"
    "9	7	2	3	0	2	0	0	1\n"
    "10	7	3	3	1	1	0	0	0\n"
    "11	9	1	1	1	0	0	0	0\n"
    "12	17	1	1	1	0	0	0	0\n"
    "13	35	1	1	1	0	0	0	0\n"
    "14	53	1	1	1	0	0	0	0\n"
    "15	16	2	2	1	0	0	0	0\n"
    "16	16	1	2	0	2	0	0	0\n"
    "17	16	3	4	0	2	0	0	0\n"
    "18	16	4	4	0	0	0	0	0\n"
    "19	14	4	4	0	0	0	0	0\n"
    "20	6	4	4	0	0	0	0	0\n"
    "21	1	1	1	0	0	0	0	0\n"
    "22	6	4	4	0	0	0	0	0\n"
    "23	1	1	1	0	0	0	0	0\n"
    "24	1	1	1	0	0	0	0	0\n"
    "25	15	4	4	0	0	0	0	0\n"
    "26	15	3	3	1	0	0	0	0\n"
    "27	1	1	1	0	0	0	0	0\n"
    "28	1	1	1	0	0	0	0	0\n"
    "29	1	1	1	0	0	0	0	0\n"
    "30	6	3	4	0	2	0	0	1\n"
    "31	1	1	1	0	0	0	0	0\n"
    "32	8	1	12	1	1	0	0	0\n"
    "33	1	1	1	0	0	0	0	0\n"
    "34	7	4	4	0	0	0	0	0\n"
    "35	8	1	1	1	1	0	0	0\n"
    "36	1	1	1	0	0	0	0	0\n"
    "37	6	3	4	0	2	1	0	1\n"
    "38	7	2	3	0	2	1	0	0\n"
    "39	7	3	3	1	1	1	0	1\n"
    "40	7	3	3	1	0	0	0	0\n"
    "41	6	3	4	0	1	0	0	0\n"
    "42	7	1	3	0	3	0	0	0\n"
    "43	7	3	3	1	0	0	0	0\n"
    "44	16	2	2	1	1	1	0	0\n"
    "45	7	3	4	0	2	0	0	0\n"
    "46	7	2	3	0	2	0	0	0\n"
    "47	7	1	2	0	2	0	0	0\n"
    "48	7	2	2	0	0	0	0	0\n"
    "49	8	3	3	0	0	0	0	0\n"
    "50	1	1	1	0	0	0	0	0\n"
    "51	8	2	3	0	2	0	0	0\n"
    "52	1	1	1	0	0	0	0	0\n"
    "53	7	2	4	0	2	0	1	0\n"
    "54	7	3	4	0	2	0	0	1\n"
    "55	7	3	34	0	1	0	0	0\n"
    "56	7	3	34	0	1	0	0	0\n"
    "57	6	3	4	0	2	0	0	1\n"
    "58	7	3	4	0	1	1	0	1\n"
    "59	8	2	2	1	1	1	0	0\n"
    "60	6	1	3	0	3	0	0	0\n"
    "61	7	2	4	0	3	0	1	0\n"
    "62	7	2	2	1	0	0	0	0\n"
    "63	6	3	4	0	2	1	0	1\n"
    "64	6	3	4	0	2	1	0	1\n"
    "65	7	2	3	0	2	1	0	0\n"
    "66	7	2	3	0	2	1	0	0\n"
    "67	7	3	4	0	2	0	0	1\n"
    "68	7	4	4	0	0	0	0	0\n"
    "69	7	3	4	0	1	1	0	0\n"
    "70	8	2	2	1	0	0	0	0\n"
    "71	1	1	1	0	0	0	0	0\n"
    "72	16	1	1	1	1	0	0	0\n"
    "73	16	3	3	0	0	0	0	0\n"
    "74	16	2	4	0	2	0	0	0\n"
    "75	15	2	3	0	2	0	0	1\n"
    "76	7	2	2	1	0	0	0	0\n"
    "77	17	4	4	0	0	0	0	0\n"
    "78	6	3	4	0	2	1	0	1\n"
    "79	7	2	3	0	2	1	0	0\n"
    "80	6	3	4	0	2	0	0	1\n"
    "81	7	3	4	0	1	1	0	1\n"
    "82	7	3	4	0	1	1	0	0\n"
    "87	26	0	0	0	0	0	0	0\n"
    "88	26	0	0	0	0	0	0	0\n"
    "89	9	0	0	0	0	0	0	0\n"
    "90	17	0	0	0	0	0	0	0\n"
    "91	35	0	0	0	0	0	0	0\n"
    "92	3	0	0	0	0	0	0	0\n"
    "93	11	0	0	0	0	0	0	0\n"
    "94	19	0	0	0	0	0	0	0\n"
    "95	30	0	0	0	0	0	0	0\n"
    "96	20	0	0	0	0	0	0	0\n"
    "97	29	0	0	0	0	0	0	0\n"
    "98	29	0	0	0	0	0	0	0\n"
    "99	12	0	0	0	0	0	0	0\n";

class std::unique_ptr<MMFFPBCICollection> MMFFPBCICollection::ds_instance = nullptr;

extern const std::string defaultMMFFPBCI;

MMFFPBCICollection *MMFFPBCICollection::getMMFFPBCI(
    const std::string &mmffPBCI) {
  if (!ds_instance || !mmffPBCI.empty()) {
    ds_instance.reset(new MMFFPBCICollection(mmffPBCI));
  }
  return ds_instance.get();
}

MMFFPBCICollection::MMFFPBCICollection(std::string mmffPBCI) {
  if (mmffPBCI.empty()) {
    mmffPBCI = defaultMMFFPBCI;
  }
  std::istringstream inStream(mmffPBCI);
  std::string inLine = RDKit::getLine(inStream);
  while (!(inStream.eof())) {
    if (inLine[0] != '*') {
      MMFFPBCI mmffPBCIObj;
      boost::char_separator<char> tabSep("\t");
      tokenizer tokens(inLine, tabSep);
      tokenizer::iterator token = tokens.begin();

#ifdef RDKIT_MMFF_PARAMS_USE_STD_MAP
      // IMPORTANT: skip the first field
      ++token;
      unsigned int atomType = boost::lexical_cast<unsigned int>(*token);
#else
      // IMPORTANT: skip the first two fields
      ++token;
#endif
      ++token;
      mmffPBCIObj.pbci = boost::lexical_cast<double>(*token);
      ++token;
      mmffPBCIObj.fcadj = boost::lexical_cast<double>(*token);
      ++token;
#ifdef RDKIT_MMFF_PARAMS_USE_STD_MAP
      d_params[atomType] = mmffPBCIObj;
#else
      d_params.push_back(mmffPBCIObj);
#endif
    }
    inLine = RDKit::getLine(inStream);
  }
}
const std::string defaultMMFFPBCI =
    "*\n"
    "*          Copyright (c) Merck and Co., Inc., 1994, 1995, 1996\n"
    "*                         All Rights Reserved\n"
    "*\n"
    "* MMFF Partial Bond Charge Incs and Formal-Charge Adj. Factors: "
    "19-MAY-1994\n"
    "*\n"
    "* type    pbci      fcadj   Origin/Comment\n"
    "0	1	0.000	0.000	E94\n"
    "0	2	-0.135	0.000	E94\n"
    "0	3	-0.095	0.000	E94\n"
    "0	4	-0.200	0.000	E94\n"
    "0	5	-0.023	0.000	E94\n"
    "0	6	-0.243	0.000	E94\n"
    "0	7	-0.687	0.000	E94\n"
    "0	8	-0.253	0.000	E94\n"
    "0	9	-0.306	0.000	E94\n"
    "0	10	-0.244	0.000	E94\n"
    "0	11	-0.317	0.000	E94\n"
    "0	12	-0.304	0.000	E94\n"
    "0	13	-0.238	0.000	E94\n"
    "0	14	-0.208	0.000	E94\n"
    "0	15	-0.236	0.000	E94\n"
    "0	16	-0.475	0.000	E94\n"
    "0	17	-0.191	0.000	E94\n"
    "0	18	-0.118	0.000	E94\n"
    "0	19	0.094	0.000	E94\n"
    "0	20	-0.019	0.000	E94\n"
    "0	21	0.157	0.000	E94\n"
    "0	22	-0.095	0.000	E94\n"
    "0	23	0.193	0.000	E94\n"
    "0	24	0.257	0.000	E94\n"
    "0	25	0.012	0.000	E94\n"
    "0	26	-0.142	0.000	E94\n"
    "0	27	0.094	0.000	E94\n"
    "0	28	0.058	0.000	E94\n"
    "0	29	0.207	0.000	E94\n"
    "0	30	-0.166	0.000	E94\n"
    "0	31	0.161	0.000	E94\n"
    "0	32	-0.732	0.500	E94\n"
    "0	33	0.257	0.000	E94\n"
    "0	34	-0.491	0.000	E94\n"
    "0	35	-0.456	0.500	E94\n"
    "0	36	-0.031	0.000	E94\n"
    "0	37	-0.127	0.000	E94\n"
    "0	38	-0.437	0.000	E94\n"
    "0	39	-0.104	0.000	E94\n"
    "0	40	-0.264	0.000	E94\n"
    "0	41	0.052	0.000	E94\n"
    "0	42	-0.757	0.000	E94\n"
    "0	43	-0.326	0.000	E94\n"
    "0	44	-0.237	0.000	E94\n"
    "0	45	-0.260	0.000	E94\n"
    "0	46	-0.429	0.000	E94\n"
    "0	47	-0.418	0.000	E94\n"
    "0	48	-0.525	0.000	E94\n"
    "0	49	-0.283	0.000	E94\n"
    "0	50	0.284	0.000	E94\n"
    "0	51	-1.046	0.000	E94\n"
    "0	52	-0.546	0.000	E94\n"
    "0	53	-0.048	0.000	E94\n"
    "0	54	-0.424	0.000	E94\n"
    "0	55	-0.476	0.000	E94\n"
    "0	56	-0.438	0.000	E94\n"
    "0	57	-0.105	0.000	E94\n"
    "0	58	-0.488	0.000	E94\n"
    "0	59	-0.337	0.000	E94\n"
    "0	60	-0.635	0.000	E94\n"
    "0	61	-0.265	0.000	E94\n"
    "0	62	-0.125	0.250	E94\n"
    "0	63	-0.180	0.000	E94\n"
    "0	64	-0.181	0.000	E94\n"
    "0	65	-0.475	0.000	E94\n"
    "0	66	-0.467	0.000	E94\n"
    "0	67	-0.099	0.000	==	69\n"
    "0	68	-0.135	0.000	E94\n"
    "0	69	-0.099	0.000	E94\n"
    "0	70	-0.269	0.000	E94\n"
    "0	71	-0.071	0.000	E94\n"
    "0	72	-0.580	0.500	E94\n"
    "0	73	-0.200	0.000	E94\n"
    "0	74	-0.301	0.000	E94\n"
    "0	75	-0.255	0.000	E94\n"
    "0	76	-0.568	0.250	E94\n"
    "0	77	-0.282	0.000	E94\n"
    "0	78	-0.168	0.000	E94\n"
    "0	79	-0.471	0.000	==	(65+66)/2\n"
    "0	80	-0.144	0.000	E94\n"
    "0	81	-0.514	0.000	E94\n"
    "0	82	-0.099	0.000	==	69\n"
    "0	83	0.000	0.000	Unused\n"
    "0	84	0.000	0.000	Unused\n"
    "0	85	0.000	0.000	Unused\n"
    "0	86	0.000	0.000	Unused\n"
    "0	87	2.000	0.000	Ionic	charge\n"
    "0	88	3.000	0.000	Ionic	charge\n"
    "0	89	-1.000	0.000	Ionic	charge\n"
    "0	90	-1.000	0.000	Ionic	charge\n"
    "0	91	-1.000	0.000	Ionic	charge\n"
    "0	92	1.000	0.000	Ionic	charge\n"
    "0	93	1.000	0.000	Ionic	charge\n"
    "0	94	1.000	0.000	Ionic	charge\n"
    "0	95	2.000	0.000	Ionic	charge\n"
    "0	96	2.000	0.000	Ionic	charge\n"
    "0	97	1.000	0.000	Ionic	charge\n"
    "0	98	2.000	0.000	Ionic	charge\n"
    "0	99	2.000	0.000	Ionic	charge\n";

class std::unique_ptr<MMFFChgCollection> MMFFChgCollection::ds_instance = nullptr;

extern const std::string defaultMMFFChg;

MMFFChgCollection *MMFFChgCollection::getMMFFChg(const std::string &mmffChg) {
  if (!ds_instance || !mmffChg.empty()) {
    ds_instance.reset(new MMFFChgCollection(mmffChg));
  }
  return ds_instance.get();
}

MMFFChgCollection::MMFFChgCollection(std::string mmffChg) {
  if (mmffChg.empty()) {
    mmffChg = defaultMMFFChg;
  }
  std::istringstream inStream(mmffChg);
  std::string inLine = RDKit::getLine(inStream);
  while (!(inStream.eof())) {
    if (inLine[0] != '*') {
      MMFFChg mmffChgObj;
      boost::char_separator<char> tabSep("\t");
      tokenizer tokens(inLine, tabSep);
      tokenizer::iterator token = tokens.begin();

#ifdef RDKIT_MMFF_PARAMS_USE_STD_MAP
      unsigned int bondType = boost::lexical_cast<unsigned int>(*token);
#else
      d_bondType.push_back(
          (std::uint8_t)(boost::lexical_cast<unsigned int>(*token)));
#endif
      ++token;
#ifdef RDKIT_MMFF_PARAMS_USE_STD_MAP
      unsigned int iAtomType = boost::lexical_cast<unsigned int>(*token);
#else
      d_iAtomType.push_back(
          (std::uint8_t)(boost::lexical_cast<unsigned int>(*token)));
#endif
      ++token;
#ifdef RDKIT_MMFF_PARAMS_USE_STD_MAP
      unsigned int jAtomType = boost::lexical_cast<unsigned int>(*token);
#else
      d_jAtomType.push_back(
          (std::uint8_t)(boost::lexical_cast<unsigned int>(*token)));
#endif
      ++token;
      mmffChgObj.bci = boost::lexical_cast<double>(*token);
      ++token;
#ifdef RDKIT_MMFF_PARAMS_USE_STD_MAP
      d_params[bondType][iAtomType][jAtomType] = mmffChgObj;
#else
      d_params.push_back(mmffChgObj);
#endif
    }
    inLine = RDKit::getLine(inStream);
  }
}
const std::string defaultMMFFChg =
    "*\n"
    "*          Copyright (c) Merck and Co., Inc., 1994, 1995, 1996\n"
    "*                         All Rights Reserved\n"
    "*\n"
    "* MMFF BOND-CHARGE INCREMENTS - Rev: 26-OCT-94  Source: MMFF\n"
    "* C94  - CORE MMFF parameter, obtained from fit to dipole moments\n"
    "* #C94 - CORE MMFF parameter, either fixed or adjusted to fit \n"
    "*        HF/6-31G* interaction energies\n"
    "* X94  - EXTD MMFF parameter, obtained from fit to dipole moments\n"
    "* #X94 - EXTD MMFF parameter, fixed in the fit to dipole moments\n"
    "* E94  - derived from partial bond charge increments (empirical rule)\n"
    "*\n"
    "*  types       bci     Source\n"
    "0	1	1	0.0000	#C94\n"
    "0	1	2	-0.1382	C94\n"
    "0	1	3	-0.0610	#C94\n"
    "0	1	4	-0.2000	#X94\n"
    "0	1	5	0.0000	#C94\n"
    "0	1	6	-0.2800	#C94\n"
    "0	1	8	-0.2700	#C94\n"
    "0	1	9	-0.2460	#C94\n"
    "0	1	10	-0.3001	C94\n"
    "0	1	11	-0.3400	#C94\n"
    "0	1	12	-0.2900	#C94\n"
    "0	1	13	-0.2300	#X94\n"
    "0	1	14	-0.1900	#X94\n"
    "0	1	15	-0.2300	#C94\n"
    "0	1	17	-0.1935	X94\n"
    "0	1	18	-0.1052	X94\n"
    "0	1	19	0.0805	X94\n"
    "0	1	20	0.0000	#C94\n"
    "0	1	22	-0.0950	E94\n"
    "0	1	25	0.0000	#X94\n"
    "0	1	26	-0.1669	X94\n"
    "0	1	34	-0.5030	C94\n"
    "0	1	35	-0.4274	X94\n"
    "0	1	37	-0.1435	C94\n"
    "0	1	39	-0.2556	C94\n"
    "0	1	40	-0.3691	C94\n"
    "0	1	41	0.1060	#C94\n"
    "0	1	43	-0.3557	X94\n"
    "0	1	45	-0.2402	X94\n"
    "0	1	46	-0.3332	X94\n"
    "0	1	54	-0.3461	C94\n"
    "0	1	55	-0.4895	C94\n"
    "0	1	56	-0.3276	C94\n"
    "0	1	57	-0.1050	E94\n"
    "0	1	58	-0.4880	E94\n"
    "0	1	61	-0.2657	X94\n"
    "0	1	62	-0.2000	#X94\n"
    "0	1	63	-0.1800	E94\n"
    "0	1	64	-0.1810	E94\n"
    "0	1	67	-0.0990	E94\n"
    "0	1	68	-0.2560	#C94\n"
    "0	1	72	-0.5500	#X94\n"
    "0	1	73	-0.0877	X94\n"
    "0	1	75	-0.2550	E94\n"
    "0	1	78	-0.1680	E94\n"
    "0	1	80	-0.1440	E94\n"
    "0	1	81	-0.5140	E94\n"
    "0	2	2	0.0000	#C94\n"
    "1	2	2	0.0000	#C94\n"
    "1	2	3	-0.0144	C94\n"
    "0	2	4	-0.0650	E94\n"
    "1	2	4	-0.0650	E94\n"
    "0	2	5	0.1500	#C94\n"
    "0	2	6	-0.0767	C94\n"
    "1	2	9	-0.1710	E94\n"
    "0	2	10	-0.1090	E94\n"
    "0	2	11	-0.1495	X94\n"
    "0	2	12	-0.1400	#X94\n"
    "0	2	13	-0.1100	#X94\n"
    "0	2	14	-0.0900	#X94\n"
    "0	2	15	-0.1010	E94\n"
    "0	2	17	-0.0560	E94\n"
    "0	2	18	0.0170	E94\n"
    "0	2	19	0.2290	E94\n"
    "0	2	20	0.1160	E94\n"
    "0	2	22	0.0400	E94\n"
    "0	2	25	0.1470	E94\n"
    "0	2	30	-0.0310	E94\n"
    "0	2	34	-0.3560	E94\n"
    "0	2	35	-0.3500	#X94\n"
    "1	2	37	0.0284	C94\n"
    "1	2	39	0.0310	E94\n"
    "0	2	40	-0.1000	#C94\n"
    "0	2	41	0.2500	#C94\n"
    "0	2	43	-0.1910	E94\n"
    "0	2	45	-0.2044	X94\n"
    "0	2	46	-0.2940	E94\n"
    "0	2	55	-0.3410	E94\n"
    "0	2	56	-0.3030	E94\n"
    "0	2	62	-0.0500	#X94\n"
    "1	2	63	-0.0450	E94\n"
    "1	2	64	-0.0460	E94\n"
    "1	2	67	0.0360	E94\n"
    "0	2	72	-0.4500	#X94\n"
    "1	2	81	-0.3790	E94\n"
    "1	3	3	0.0000	#C94\n"
    "1	3	4	-0.1050	E94\n"
    "0	3	5	0.0600	#C94\n"
    "0	3	6	-0.1500	#C94\n"
    "0	3	7	-0.5700	#C94\n"
    "0	3	9	-0.4500	#C94\n"
    "1	3	9	-0.2110	E94\n"
    "0	3	10	-0.0600	C94\n"
    "0	3	11	-0.2220	E94\n"
    "0	3	12	-0.2090	E94\n"
    "0	3	15	-0.1410	E94\n"
    "0	3	16	-0.3800	#X94\n"
    "0	3	17	-0.0960	E94\n"
    "0	3	18	-0.0230	E94\n"
    "0	3	20	0.0530	#C94\n"
    "0	3	22	0.0000	E94\n"
    "0	3	25	0.1070	E94\n"
    "1	3	30	-0.0710	E94\n"
    "0	3	35	-0.3610	E94\n"
    "1	3	37	0.0862	C94\n"
    "1	3	39	-0.0090	E94\n"
    "0	3	40	-0.0500	#C94\n"
    "0	3	41	0.1470	E94\n"
    "0	3	43	-0.2363	X94\n"
    "0	3	45	-0.1650	E94\n"
    "0	3	48	-0.4300	E94\n"
    "0	3	51	-0.9500	#X94\n"
    "0	3	53	-0.0134	X94\n"
    "0	3	54	-0.4000	#C94\n"
    "1	3	54	-0.3290	E94\n"
    "0	3	55	-0.3810	E94\n"
    "0	3	56	-0.3430	E94\n"
    "1	3	57	-0.0100	E94\n"
    "1	3	58	-0.3930	E94\n"
    "0	3	62	-0.0300	E94\n"
    "1	3	63	-0.0850	E94\n"
    "1	3	64	-0.0860	E94\n"
    "0	3	67	-0.0040	E94\n"
    "0	3	74	-0.3190	X94\n"
    "0	3	75	-0.2474	X94\n"
    "1	3	78	-0.0730	E94\n"
    "1	3	80	-0.0490	E94\n"
    "0	4	5	0.1770	E94\n"
    "0	4	6	-0.0430	E94\n"
    "0	4	7	-0.4870	E94\n"
    "0	4	9	-0.3000	E94\n"
    "1	4	9	-0.1060	E94\n"
    "0	4	10	-0.0440	E94\n"
    "0	4	15	-0.0360	E94\n"
    "0	4	20	0.1810	E94\n"
    "0	4	22	0.1050	E94\n"
    "0	4	30	0.0340	E94\n"
    "1	4	37	0.0730	E94\n"
    "0	4	40	-0.0640	E94\n"
    "0	4	42	-0.5571	X94\n"
    "0	4	43	-0.1260	E94\n"
    "1	4	63	0.0200	E94\n"
    "1	4	64	0.0190	E94\n"
    "0	5	19	0.2000	#X94\n"
    "0	5	20	0.0000	#C94\n"
    "0	5	22	-0.1000	#C94\n"
    "0	5	30	-0.1500	#C94\n"
    "0	5	37	-0.1500	#C94\n"
    "0	5	41	0.2203	C94\n"
    "0	5	57	-0.1500	#C94\n"
    "0	5	63	-0.1500	#C94\n"
    "0	5	64	-0.1500	#C94\n"
    "0	5	78	-0.1500	#C94\n"
    "0	5	80	-0.1500	#C94\n"
    "0	6	6	0.0000	#C94\n"
    "0	6	8	-0.1000	#C94\n"
    "0	6	9	-0.0630	E94\n"
    "0	6	10	0.0355	C94\n"
    "0	6	15	0.0070	E94\n"
    "0	6	17	0.0520	E94\n"
    "0	6	18	0.1837	X94\n"
    "0	6	19	0.2974	X94\n"
    "0	6	20	0.2579	C94\n"
    "0	6	21	0.4000	#C94\n"
    "0	6	22	0.1480	E94\n"
    "0	6	24	0.5000	#C94\n"
    "0	6	25	0.2712	X94\n"
    "0	6	26	0.1010	E94\n"
    "0	6	29	0.4500	#C94\n"
    "0	6	30	0.0770	E94\n"
    "0	6	33	0.5000	#X94\n"
    "0	6	37	0.0825	C94\n"
    "0	6	39	0.1390	E94\n"
    "0	6	40	-0.0210	E94\n"
    "0	6	41	0.2950	E94\n"
    "0	6	43	-0.0830	E94\n"
    "0	6	45	-0.0090	X94\n"
    "0	6	54	-0.1810	E94\n"
    "0	6	55	-0.2330	E94\n"
    "0	6	57	0.1380	E94\n"
    "0	6	58	-0.2450	E94\n"
    "0	6	63	0.0630	E94\n"
    "0	6	64	0.0620	E94\n"
    "0	7	17	0.5000	#X94\n"
    "0	7	46	0.1618	X94\n"
    "0	7	74	0.5000	#X94\n"
    "0	8	8	0.0000	#C94\n"
    "0	8	9	-0.0530	E94\n"
    "0	8	10	0.0090	E94\n"
    "0	8	12	-0.0510	E94\n"
    "0	8	15	0.0170	E94\n"
    "0	8	17	0.0620	E94\n"
    "0	8	19	0.3470	E94\n"
    "0	8	20	0.2096	C94\n"
    "0	8	22	0.1580	E94\n"
    "0	8	23	0.3600	#C94\n"
    "0	8	25	0.2679	X94\n"
    "0	8	26	0.1110	E94\n"
    "0	8	34	-0.2380	E94\n"
    "0	8	39	0.1490	E94\n"
    "0	8	40	-0.0110	E94\n"
    "0	8	43	-0.0730	E94\n"
    "0	8	45	-0.0070	E94\n"
    "0	8	46	-0.1760	E94\n"
    "0	8	55	-0.2230	E94\n"
    "0	8	56	-0.1850	E94\n"
    "0	9	9	0.0000	#C94\n"
    "0	9	10	0.0620	E94\n"
    "0	9	12	0.0020	E94\n"
    "0	9	15	0.0700	E94\n"
    "0	9	18	0.1880	E94\n"
    "0	9	19	0.4000	E94\n"
    "0	9	20	0.2870	E94\n"
    "0	9	25	0.3180	E94\n"
    "0	9	27	0.4000	#C94\n"
    "0	9	34	-0.1850	E94\n"
    "0	9	35	-0.1500	E94\n"
    "1	9	37	0.1790	E94\n"
    "1	9	39	0.2020	E94\n"
    "0	9	40	0.0420	E94\n"
    "0	9	41	0.3580	E94\n"
    "0	9	45	0.0460	E94\n"
    "0	9	53	0.3179	X94\n"
    "0	9	54	-0.1180	E94\n"
    "0	9	55	-0.1700	E94\n"
    "0	9	56	-0.1320	E94\n"
    "1	9	57	0.2010	E94\n"
    "0	9	62	0.1810	E94\n"
    "1	9	63	0.1260	E94\n"
    "1	9	64	0.1250	E94\n"
    "0	9	67	0.2070	E94\n"
    "1	9	78	0.1380	E94\n"
    "1	9	81	-0.2080	E94\n"
    "0	10	10	0.0000	#C94\n"
    "0	10	13	0.0060	E94\n"
    "0	10	14	0.0360	E94\n"
    "0	10	15	0.0080	E94\n"
    "0	10	17	0.0530	E94\n"
    "0	10	20	0.2250	E94\n"
    "0	10	22	0.1490	E94\n"
    "0	10	25	0.2560	E94\n"
    "0	10	26	0.1020	E94\n"
    "0	10	28	0.3700	#C94\n"
    "0	10	34	-0.2470	E94\n"
    "0	10	35	-0.2120	E94\n"
    "0	10	37	0.1170	E94\n"
    "0	10	39	0.1400	E94\n"
    "0	10	40	-0.0200	E94\n"
    "0	10	41	0.2960	E94\n"
    "0	10	45	-0.0160	E94\n"
    "0	10	63	0.0640	E94\n"
    "0	10	64	0.0630	E94\n"
    "0	11	20	0.2980	E94\n"
    "0	11	22	0.2317	X94\n"
    "0	11	25	0.3290	E94\n"
    "0	11	26	0.1750	E94\n"
    "0	11	37	0.1900	E94\n"
    "0	11	40	0.0530	E94\n"
    "0	12	15	0.0680	E94\n"
    "0	12	18	0.1860	E94\n"
    "0	12	19	0.3701	X94\n"
    "0	12	20	0.2900	#C94\n"
    "0	12	22	0.2273	X94\n"
    "0	12	25	0.3160	E94\n"
    "0	12	26	0.2112	X94\n"
    "0	12	37	0.1770	E94\n"
    "0	12	40	0.0400	E94\n"
    "0	12	57	0.1990	E94\n"
    "0	12	63	0.1240	E94\n"
    "0	12	64	0.1230	E94\n"
    "0	13	20	0.2190	E94\n"
    "0	13	22	0.1430	E94\n"
    "0	13	37	0.1110	E94\n"
    "0	13	64	0.0570	E94\n"
    "0	14	20	0.1890	E94\n"
    "0	14	37	0.0810	E94\n"
    "0	15	15	0.0000	#C94\n"
    "0	15	18	0.1180	E94\n"
    "0	15	19	0.3300	E94\n"
    "0	15	20	0.2170	E94\n"
    "0	15	22	0.1410	E94\n"
    "0	15	25	0.2480	E94\n"
    "0	15	26	0.0940	E94\n"
    "0	15	30	0.0700	E94\n"
    "0	15	37	0.1015	C94\n"
    "0	15	40	-0.0280	E94\n"
    "0	15	43	-0.0900	E94\n"
    "0	15	57	0.1310	E94\n"
    "0	15	63	0.0560	E94\n"
    "0	15	64	0.0550	E94\n"
    "0	15	71	0.1800	#C94\n"
    "0	16	16	0.0000	#C94\n"
    "0	17	17	0.0000	#X94\n"
    "0	17	20	0.1720	E94\n"
    "0	17	22	0.0960	E94\n"
    "0	17	37	0.0640	E94\n"
    "0	17	43	-0.1350	E94\n"
    "0	18	18	0.0000	#X94\n"
    "0	18	20	0.0990	E94\n"
    "0	18	22	0.0230	E94\n"
    "0	18	32	-0.6500	#X94\n"
    "0	18	37	-0.0090	E94\n"
    "0	18	39	0.0140	E94\n"
    "0	18	43	-0.1380	X94\n"
    "0	18	48	-0.5895	X94\n"
    "0	18	55	-0.3580	E94\n"
    "0	18	58	-0.3700	E94\n"
    "0	18	62	0.2099	X94\n"
    "0	18	63	-0.0620	E94\n"
    "0	18	64	-0.0630	E94\n"
    "0	18	80	-0.0260	E94\n"
    "0	19	19	0.0000	#X94\n"
    "0	19	20	-0.1130	E94\n"
    "0	19	37	-0.2210	E94\n"
    "0	19	40	-0.3580	E94\n"
    "0	19	63	-0.2740	E94\n"
    "0	19	75	-0.3490	E94\n"
    "0	20	20	0.0000	#C94\n"
    "0	20	22	-0.0760	E94\n"
    "0	20	25	0.0310	E94\n"
    "0	20	26	-0.1230	E94\n"
    "0	20	30	-0.1380	#C94\n"
    "0	20	34	-0.4720	E94\n"
    "0	20	37	-0.1080	E94\n"
    "0	20	40	-0.2450	E94\n"
    "0	20	41	0.0710	E94\n"
    "0	20	43	-0.3070	E94\n"
    "0	20	45	-0.2410	E94\n"
    "0	22	22	0.0000	#C94\n"
    "0	22	30	-0.0710	E94\n"
    "0	22	34	-0.3960	E94\n"
    "0	22	37	-0.0320	E94\n"
    "0	22	40	-0.1690	E94\n"
    "0	22	41	0.1470	E94\n"
    "0	22	43	-0.2310	E94\n"
    "0	22	45	-0.1650	E94\n"
    "0	23	39	-0.2700	#C94\n"
    "0	23	62	-0.4000	#X94\n"
    "0	23	67	-0.2920	E94\n"
    "0	23	68	-0.3600	#C94\n"
    "0	25	25	0.0000	#X94\n"
    "0	25	32	-0.7000	#X94\n"
    "0	25	37	-0.1390	E94\n"
    "0	25	39	-0.1160	E94\n"
    "0	25	40	-0.2760	E94\n"
    "0	25	43	-0.3380	E94\n"
    "0	25	57	-0.1170	E94\n"
    "0	25	63	-0.1920	E94\n"
    "0	25	71	-0.0362	X94\n"
    "0	25	72	-0.6773	X94\n"
    "0	26	26	0.0000	#X94\n"
    "0	26	34	-0.3490	E94\n"
    "0	26	37	0.0150	E94\n"
    "0	26	40	-0.1220	E94\n"
    "0	26	71	0.0960	X94\n"
    "0	28	40	-0.4000	#C94\n"
    "0	28	43	-0.4200	#X94\n"
    "0	28	48	-0.4000	#X94\n"
    "0	30	30	0.0000	#C94\n"
    "0	30	40	-0.0980	E94\n"
    "1	30	67	0.0670	E94\n"
    "0	31	70	-0.4300	#C94\n"
    "0	32	41	0.6500	#C94\n"
    "0	32	45	0.5200	#X94\n"
    "0	32	67	0.6330	E94\n"
    "0	32	68	0.7500	#C94\n"
    "0	32	69	0.7500	#C94\n"
    "0	32	73	0.3500	#X94\n"
    "0	32	77	0.4500	#X94\n"
    "0	32	82	0.6330	E94\n"
    "0	34	36	0.4500	#C94\n"
    "0	34	37	0.3640	E94\n"
    "0	34	43	0.1650	E94\n"
    "0	35	37	0.3290	E94\n"
    "0	35	63	0.2760	E94\n"
    "0	36	54	-0.4000	#C94\n"
    "0	36	55	-0.4500	#C94\n"
    "0	36	56	-0.4500	#C94\n"
    "0	36	58	-0.4570	E94\n"
    "4	36	58	-0.4500	#C94\n"
    "0	36	81	-0.4500	#C94\n"
    "0	37	37	0.0000	#C94\n"
    "1	37	37	0.0000	#C94\n"
    "0	37	38	-0.3100	#C94\n"
    "0	37	39	0.0230	E94\n"
    "1	37	39	0.0230	E94\n"
    "0	37	40	-0.1000	#C94\n"
    "0	37	41	0.1790	E94\n"
    "0	37	43	-0.1990	E94\n"
    "0	37	45	-0.1330	E94\n"
    "0	37	46	-0.3020	E94\n"
    "0	37	55	-0.3490	E94\n"
    "0	37	56	-0.3110	E94\n"
    "1	37	57	0.0220	E94\n"
    "0	37	58	-0.3610	E94\n"
    "1	37	58	-0.3610	E94\n"
    "4	37	58	-0.3500	#C94\n"
    "0	37	61	-0.1380	E94\n"
    "0	37	62	0.0020	E94\n"
    "0	37	63	0.0000	#C94\n"
    "1	37	63	-0.0530	E94\n"
    "0	37	64	0.0000	#C94\n"
    "1	37	64	-0.0540	E94\n"
    "1	37	67	0.0280	E94\n"
    "0	37	69	-0.0895	C94\n"
    "0	37	78	-0.0410	E94\n"
    "0	37	81	-0.3870	E94\n"
    "1	37	81	-0.3870	E94\n"
    "0	38	38	0.0000	#C94\n"
    "0	38	63	0.2570	E94\n"
    "0	38	64	0.2560	E94\n"
    "0	38	69	0.3380	E94\n"
    "0	38	78	0.2690	E94\n"
    "1	39	39	0.0000	#C94\n"
    "0	39	40	-0.1600	E94\n"
    "0	39	45	-0.1560	E94\n"
    "0	39	63	-0.1516	C94\n"
    "1	39	63	-0.0760	E94\n"
    "0	39	64	-0.0770	E94\n"
    "1	39	64	-0.0770	E94\n"
    "0	39	65	-0.4180	C94\n"
    "0	39	78	-0.0640	E94\n"
    "0	40	40	0.0000	#C94\n"
    "0	40	45	0.0040	E94\n"
    "0	40	46	-0.1650	E94\n"
    "0	40	54	-0.1600	E94\n"
    "0	40	63	0.0840	E94\n"
    "0	40	64	0.0830	E94\n"
    "0	40	78	0.0960	E94\n"
    "0	41	41	0.0000	#C94\n"
    "0	41	55	-0.5280	E94\n"
    "0	41	62	-0.1770	E94\n"
    "0	41	72	-0.5000	#X94\n"
    "0	41	80	-0.1960	E94\n"
    "0	42	61	0.4920	E94\n"
    "0	43	43	0.0000	#X94\n"
    "0	43	45	0.0660	E94\n"
    "0	43	64	0.1450	E94\n"
    "0	44	63	0.0400	#C94\n"
    "0	44	65	-0.2207	C94\n"
    "0	44	78	0.0690	E94\n"
    "0	44	80	0.0930	E94\n"
    "0	45	63	0.0800	E94\n"
    "0	45	64	0.0790	E94\n"
    "0	45	78	0.0920	E94\n"
    "0	47	53	0.3700	#X94\n"
    "0	49	50	0.5673	C94\n"
    "0	51	52	0.5000	#X94\n"
    "0	55	57	0.3544	C94\n"
    "0	55	62	0.3510	E94\n"
    "0	55	64	0.2950	E94\n"
    "0	55	80	0.3320	E94\n"
    "0	56	57	0.4000	#C94\n"
    "0	56	63	0.2580	E94\n"
    "0	56	80	0.2700	E94\n"
    "4	57	58	-0.4000	#C94\n"
    "1	57	63	-0.0750	E94\n"
    "1	57	64	-0.0760	E94\n"
    "0	58	63	0.3080	E94\n"
    "0	58	64	0.3070	E94\n"
    "0	59	63	0.1400	#C94\n"
    "0	59	65	-0.1209	C94\n"
    "0	59	78	0.1690	E94\n"
    "0	59	80	0.1930	E94\n"
    "0	59	82	0.2380	E94\n"
    "0	60	61	0.3700	#X94\n"
    "0	62	63	-0.0550	E94\n"
    "0	62	64	-0.0560	E94\n"
    "0	63	63	0.0000	#C94\n"
    "1	63	63	0.0000	#C94\n"
    "0	63	64	0.0000	#C94\n"
    "0	63	66	-0.3381	C94\n"
    "0	63	72	-0.4000	E94\n"
    "0	63	78	0.0120	E94\n"
    "0	63	81	-0.3340	E94\n"
    "0	64	64	0.0000	#C94\n"
    "0	64	65	-0.2888	C94\n"
    "0	64	66	-0.2272	C94\n"
    "0	64	78	0.0130	E94\n"
    "0	64	81	-0.3330	E94\n"
    "0	64	82	0.0820	E94\n"
    "0	65	66	0.0000	#C94\n"
    "0	65	78	0.3070	E94\n"
    "0	65	81	-0.0390	E94\n"
    "0	65	82	0.3760	E94\n"
    "0	66	66	0.0000	#C94\n"
    "0	66	78	0.2990	E94\n"
    "0	66	81	-0.0470	E94\n"
    "0	71	75	-0.0958	X94\n"
    "0	72	73	0.4500	#X94\n"
    "0	76	76	0.0000	#X94\n"
    "0	76	78	0.4000	#X94\n"
    "0	78	78	0.0000	#C94\n"
    "1	78	78	0.0000	#C94\n"
    "0	78	79	-0.3030	E94\n"
    "0	78	81	-0.3500	#C94\n"
    "0	79	81	-0.0430	E94\n"
    "0	80	81	-0.4000	#C94\n";

class std::unique_ptr<MMFFBondCollection> MMFFBondCollection::ds_instance = nullptr;

extern const std::string defaultMMFFBond;

MMFFBondCollection *MMFFBondCollection::getMMFFBond(
    const std::string &mmffBond) {
  if (!ds_instance || !mmffBond.empty()) {
    ds_instance.reset(new MMFFBondCollection(mmffBond));
  }
  return ds_instance.get();
}

MMFFBondCollection::MMFFBondCollection(std::string mmffBond) {
  if (mmffBond.empty()) {
    mmffBond = defaultMMFFBond;
  }
  std::istringstream inStream(mmffBond);
  std::string inLine = RDKit::getLine(inStream);
  while (!(inStream.eof())) {
    if (inLine[0] != '*') {
      MMFFBond mmffBondObj;
      boost::char_separator<char> tabSep("\t");
      tokenizer tokens(inLine, tabSep);
      tokenizer::iterator token = tokens.begin();

#ifdef RDKIT_MMFF_PARAMS_USE_STD_MAP
      unsigned int bondType = boost::lexical_cast<unsigned int>(*token);
#else
      d_bondType.push_back(
          (std::uint8_t)(boost::lexical_cast<unsigned int>(*token)));
#endif
      ++token;
#ifdef RDKIT_MMFF_PARAMS_USE_STD_MAP
      unsigned int atomType = boost::lexical_cast<unsigned int>(*token);
#else
      d_iAtomType.push_back(
          (std::uint8_t)(boost::lexical_cast<unsigned int>(*token)));
#endif
      ++token;
#ifdef RDKIT_MMFF_PARAMS_USE_STD_MAP
      unsigned int jAtomType = boost::lexical_cast<unsigned int>(*token);
#else
      d_jAtomType.push_back(
          (std::uint8_t)(boost::lexical_cast<unsigned int>(*token)));
#endif
      ++token;
      mmffBondObj.kb = boost::lexical_cast<double>(*token);
      ++token;
      mmffBondObj.r0 = boost::lexical_cast<double>(*token);
      ++token;
#ifdef RDKIT_MMFF_PARAMS_USE_STD_MAP
      d_params[bondType][atomType][jAtomType] = mmffBondObj;
#else
      d_params.push_back(mmffBondObj);
#endif
    }
    inLine = RDKit::getLine(inStream);
  }
}

const std::string defaultMMFFBond =
    "*\n"
    "*           Copyright (c) Merck and Co., Inc., 1994, 1995, 1996\n"
    "*                           All Rights Reserved\n"
    "*\n"
    "* MMFF BOND PARAMETERS- Rev: 26-OCT-94  Source: MMFF\n"
    "* C94  = CORE MMFF parameter - obtained from ab initio data\n"
    "* X94  = EXTD MMFF parameter - fit to additional ab initio data\n"
    "* E94  = r0 from fit to X-ray data, kb from empirical rule\n"
    "* #C94 = r0 lies between C94 and E94 values, kb from empirical rule\n"
    "* #X94 = r0 lies between X94 and E94 values, kb from empirical rule\n"
    "* #E94  = r0 and k both from empirical rules\n"
    "*\n"
    "*	types	kb	r0	Source\n"
    "0	1	1	4.258	1.508	C94\n"
    "0	1	2	4.539	1.482	C94\n"
    "0	1	3	4.190	1.492	C94\n"
    "0	1	4	4.707	1.459	X94\n"
    "0	1	5	4.766	1.093	C94\n"
    "0	1	6	5.047	1.418	C94\n"
    "0	1	8	5.084	1.451	C94\n"
    "0	1	9	4.763	1.458	C94\n"
    "0	1	10	4.664	1.436	C94\n"
    "0	1	11	6.011	1.360	#C94\n"
    "0	1	12	2.974	1.773	C94\n"
    "0	1	13	2.529	1.949	E94\n"
    "0	1	14	1.706	2.090	E94\n"
    "0	1	15	2.893	1.805	C94\n"
    "0	1	17	2.841	1.813	X94\n"
    "0	1	18	3.258	1.772	X94\n"
    "0	1	19	2.866	1.830	#X94\n"
    "0	1	20	4.650	1.504	C94\n"
    "0	1	22	4.286	1.482	E94\n"
    "0	1	25	2.980	1.810	#X94\n"
    "0	1	26	2.790	1.830	#X94\n"
    "0	1	34	3.844	1.480	#C94\n"
    "0	1	35	7.915	1.307	X94\n"
    "0	1	37	4.957	1.486	C94\n"
    "0	1	39	6.114	1.445	C94\n"
    "0	1	40	4.922	1.446	C94\n"
    "0	1	41	3.830	1.510	#C94\n"
    "0	1	43	3.971	1.472	X94\n"
    "0	1	45	3.844	1.480	X94\n"
    "0	1	46	3.813	1.482	X94\n"
    "0	1	54	4.267	1.461	C94\n"
    "0	1	55	4.646	1.454	C94\n"
    "0	1	56	4.166	1.453	C94\n"
    "0	1	57	4.669	1.461	E94\n"
    "0	1	58	4.329	1.451	E94\n"
    "0	1	61	4.845	1.424	X94\n"
    "0	1	62	4.456	1.444	X94\n"
    "0	1	63	4.481	1.471	E94\n"
    "0	1	64	4.518	1.469	E94\n"
    "0	1	67	4.188	1.459	E94\n"
    "0	1	68	4.217	1.479	C94\n"
    "0	1	72	2.956	1.801	X94\n"
    "0	1	73	2.608	1.839	X94\n"
    "0	1	75	2.547	1.858	E94\n"
    "0	1	78	4.593	1.465	E94\n"
    "0	1	80	4.373	1.477	E94\n"
    "0	1	81	4.512	1.441	E94\n"
    "0	2	2	9.505	1.333	C94\n"
    "1	2	2	5.310	1.430	#C94\n"
    "1	2	3	4.565	1.468	C94\n"
    "0	2	4	9.538	1.297	E94\n"
    "1	2	4	5.657	1.415	E94\n"
    "0	2	5	5.170	1.083	C94\n"
    "0	2	6	5.520	1.373	C94\n"
    "1	2	9	6.385	1.360	E94\n"
    "0	2	10	6.329	1.362	E94\n"
    "0	2	11	6.283	1.350	#X94\n"
    "0	2	12	3.390	1.720	#X94\n"
    "0	2	13	3.413	1.854	E94\n"
    "0	2	14	2.062	2.025	E94\n"
    "0	2	15	3.896	1.720	E94\n"
    "0	2	17	3.247	1.773	E94\n"
    "0	2	18	3.789	1.728	E94\n"
    "0	2	19	3.052	1.811	E94\n"
    "0	2	20	4.593	1.465	E94\n"
    "0	2	22	4.926	1.448	E94\n"
    "0	2	25	3.750	1.742	E94\n"
    "0	2	30	8.166	1.331	E94\n"
    "0	2	34	5.207	1.407	E94\n"
    "0	2	35	10.343	1.250	#X94\n"
    "1	2	37	5.007	1.449	C94\n"
    "1	2	39	6.164	1.368	E94\n"
    "0	2	40	6.110	1.370	#C94\n"
    "0	2	41	3.746	1.505	C94\n"
    "0	2	43	4.928	1.420	E94\n"
    "0	2	45	4.725	1.430	#X94\n"
    "0	2	46	7.466	1.325	E94\n"
    "0	2	55	6.164	1.368	E94\n"
    "0	2	56	6.246	1.365	E94\n"
    "0	2	62	7.105	1.336	X94\n"
    "1	2	63	6.030	1.400	E94\n"
    "1	2	64	5.754	1.411	E94\n"
    "1	2	67	4.685	1.432	E94\n"
    "0	2	72	4.179	1.700	#X94\n"
    "1	2	81	6.357	1.361	E94\n"
    "1	3	3	4.418	1.489	C94\n"
    "1	3	4	5.135	1.438	E94\n"
    "0	3	5	4.650	1.101	C94\n"
    "0	3	6	5.801	1.355	C94\n"
    "0	3	7	12.950	1.222	C94\n"
    "0	3	9	10.077	1.290	C94\n"
    "1	3	9	6.273	1.364	E94\n"
    "0	3	10	5.829	1.369	C94\n"
    "0	3	11	6.570	1.340	E94\n"
    "0	3	12	3.449	1.715	E94\n"
    "0	3	15	3.536	1.748	E94\n"
    "0	3	16	4.735	1.665	E94\n"
    "0	3	17	2.888	1.808	E94\n"
    "0	3	18	3.394	1.760	E94\n"
    "0	3	20	3.298	1.530	C94\n"
    "0	3	22	4.593	1.465	E94\n"
    "0	3	25	3.164	1.792	E94\n"
    "1	3	30	4.481	1.471	E94\n"
    "0	3	35	11.012	1.237	E94\n"
    "1	3	37	4.488	1.457	C94\n"
    "1	3	39	5.978	1.375	E94\n"
    "0	3	40	6.110	1.370	#C94\n"
    "0	3	41	4.286	1.482	E94\n"
    "0	3	43	4.928	1.420	X94\n"
    "0	3	45	4.531	1.440	E94\n"
    "0	3	48	5.412	1.398	E94\n"
    "0	3	51	8.562	1.290	#X94\n"
    "0	3	53	7.637	1.320	#X94\n"
    "0	3	54	10.333	1.280	C94\n"
    "1	3	54	2.771	1.563	E94\n"
    "0	3	55	4.886	1.422	E94\n"
    "0	3	56	4.907	1.421	E94\n"
    "1	3	57	5.492	1.422	E94\n"
    "1	3	58	5.163	1.409	E94\n"
    "0	3	62	7.568	1.322	E94\n"
    "1	3	63	5.468	1.423	E94\n"
    "1	3	64	5.288	1.431	E94\n"
    "0	3	67	8.217	1.304	E94\n"
    "0	3	74	5.204	1.639	X94\n"
    "0	3	75	4.191	1.710	#X94\n"
    "1	3	78	5.705	1.413	E94\n"
    "1	3	80	6.719	1.375	E94\n"
    "0	4	4	15.206	1.200	#X94\n"
    "0	4	5	5.726	1.065	X94\n"
    "0	4	6	7.193	1.328	E94\n"
    "0	4	7	14.916	1.176	E94\n"
    "0	4	9	15.589	1.172	E94\n"
    "1	4	9	7.041	1.338	E94\n"
    "0	4	10	6.824	1.345	E94\n"
    "0	4	15	4.330	1.690	E94\n"
    "0	4	20	5.178	1.436	E94\n"
    "0	4	22	5.400	1.426	E94\n"
    "0	4	30	10.227	1.282	E94\n"
    "1	4	37	5.445	1.424	E94\n"
    "0	4	42	16.582	1.160	#X94\n"
    "0	4	43	6.947	1.341	E94\n"
    "1	4	63	5.633	1.416	E94\n"
    "1	4	64	5.492	1.422	E94\n"
    "0	5	19	2.254	1.485	X94\n"
    "0	5	20	4.852	1.093	C94\n"
    "0	5	22	5.191	1.082	C94\n"
    "0	5	30	5.176	1.086	C94\n"
    "0	5	37	5.306	1.084	C94\n"
    "0	5	41	3.256	1.144	C94\n"
    "0	5	57	5.633	1.076	C94\n"
    "0	5	63	5.531	1.080	C94\n"
    "0	5	64	5.506	1.080	C94\n"
    "0	5	78	5.506	1.080	C94\n"
    "0	5	80	5.633	1.076	C94\n"
    "0	6	6	4.088	1.449	E94\n"
    "0	6	8	5.059	1.450	C94\n"
    "0	6	9	4.491	1.395	E94\n"
    "0	6	10	5.982	1.410	C94\n"
    "0	6	15	4.757	1.661	E94\n"
    "0	6	17	5.779	1.608	E94\n"
    "0	6	18	5.326	1.630	#X94\n"
    "0	6	19	4.661	1.660	#X94\n"
    "0	6	20	5.623	1.433	C94\n"
    "0	6	21	7.794	0.972	C94\n"
    "0	6	22	4.556	1.433	E94\n"
    "0	6	24	7.403	0.981	C94\n"
    "0	6	25	5.243	1.630	#X94\n"
    "0	6	26	5.481	1.618	E94\n"
    "0	6	29	7.839	0.973	C94\n"
    "0	6	30	9.359	1.271	E94\n"
    "0	6	33	7.143	0.986	X94\n"
    "0	6	37	5.614	1.376	C94\n"
    "0	6	39	4.629	1.388	E94\n"
    "0	6	40	4.609	1.389	E94\n"
    "0	6	41	6.754	1.342	E94\n"
    "0	6	43	3.937	1.426	E94\n"
    "0	6	45	4.321	1.404	X94\n"
    "0	6	54	5.117	1.365	E94\n"
    "0	6	55	4.772	1.381	E94\n"
    "0	6	57	7.128	1.330	E94\n"
    "0	6	58	4.792	1.380	E94\n"
    "0	6	63	7.324	1.324	E94\n"
    "0	6	64	6.664	1.345	E94\n"
    "0	7	17	8.770	1.500	#X94\n"
    "0	7	46	9.329	1.235	X94\n"
    "0	7	74	9.129	1.490	#X94\n"
    "0	8	8	3.264	1.420	E94\n"
    "0	8	9	4.581	1.342	E94\n"
    "0	8	10	3.909	1.378	E94\n"
    "0	8	12	3.371	1.761	E94\n"
    "0	8	15	4.060	1.652	E94\n"
    "0	8	17	3.901	1.663	E94\n"
    "0	8	19	4.254	1.700	E94\n"
    "0	8	20	5.107	1.456	C94\n"
    "0	8	22	4.223	1.457	E94\n"
    "0	8	23	6.490	1.019	C94\n"
    "0	8	25	4.629	1.660	E94\n"
    "0	8	26	4.027	1.699	E94\n"
    "0	8	34	3.775	1.386	E94\n"
    "0	8	39	3.435	1.408	E94\n"
    "0	8	40	3.710	1.390	E94\n"
    "0	8	43	3.977	1.374	E94\n"
    "0	8	45	4.267	1.358	E94\n"
    "0	8	46	5.519	1.301	E94\n"
    "0	8	55	4.229	1.360	E94\n"
    "0	8	56	3.995	1.373	E94\n"
    "0	9	9	7.256	1.243	E94\n"
    "1	9	9	3.808	1.384	E94\n"
    "0	9	10	4.480	1.347	E94\n"
    "0	9	12	3.635	1.739	E94\n"
    "0	9	15	3.791	1.671	E94\n"
    "0	9	18	4.465	1.626	E94\n"
    "0	9	19	3.687	1.741	E94\n"
    "0	9	20	4.401	1.447	E94\n"
    "0	9	25	5.379	1.619	E94\n"
    "0	9	27	6.230	1.026	C94\n"
    "0	9	34	3.223	1.423	E94\n"
    "0	9	35	5.095	1.366	E94\n"
    "1	9	37	5.529	1.393	E94\n"
    "1	9	39	4.685	1.337	E94\n"
    "0	9	40	4.382	1.352	E94\n"
    "0	9	41	5.650	1.388	E94\n"
    "0	9	45	4.857	1.329	E94\n"
    "0	9	53	7.291	1.242	X94\n"
    "0	9	54	4.991	1.323	E94\n"
    "0	9	55	3.825	1.383	E94\n"
    "0	9	56	4.602	1.341	E94\n"
    "1	9	57	6.824	1.345	E94\n"
    "0	9	62	4.749	1.334	E94\n"
    "1	9	63	6.824	1.345	E94\n"
    "1	9	64	5.458	1.396	E94\n"
    "0	9	67	6.752	1.258	E94\n"
    "1	9	78	6.644	1.351	E94\n"
    "1	9	81	3.909	1.378	E94\n"
    "0	10	10	3.977	1.374	E94\n"
    "0	10	13	3.110	1.878	E94\n"
    "0	10	14	1.967	2.029	E94\n"
    "0	10	15	3.593	1.686	E94\n"
    "0	10	17	3.930	1.661	E94\n"
    "0	10	20	4.240	1.456	E94\n"
    "0	10	22	4.970	1.418	E94\n"
    "0	10	25	3.820	1.714	E94\n"
    "0	10	26	3.651	1.727	E94\n"
    "0	10	28	6.663	1.015	C94\n"
    "0	10	34	3.960	1.375	E94\n"
    "0	10	35	4.898	1.375	E94\n"
    "0	10	37	5.482	1.395	E94\n"
    "0	10	39	4.382	1.352	E94\n"
    "0	10	40	3.841	1.382	E94\n"
    "0	10	41	7.466	1.325	E94\n"
    "0	10	45	3.524	1.402	E94\n"
    "0	10	63	6.137	1.369	E94\n"
    "0	10	64	5.952	1.376	E94\n"
    "0	11	20	6.339	1.348	E94\n"
    "0	11	22	5.296	1.389	X94\n"
    "0	11	25	6.019	1.583	E94\n"
    "0	11	26	6.204	1.575	E94\n"
    "0	11	37	6.511	1.342	E94\n"
    "0	11	40	4.187	1.440	E94\n"
    "0	12	15	2.978	2.031	E94\n"
    "0	12	18	2.808	2.051	E94\n"
    "0	12	19	2.838	2.050	#X94\n"
    "0	12	20	2.859	1.751	C94\n"
    "0	12	22	3.056	1.750	#X94\n"
    "0	12	25	3.063	2.023	E94\n"
    "0	12	26	2.448	2.100	#X94\n"
    "0	12	37	3.378	1.721	E94\n"
    "0	12	40	3.737	1.731	E94\n"
    "0	12	57	3.714	1.694	E94\n"
    "0	12	63	3.413	1.718	E94\n"
    "0	12	64	3.649	1.699	E94\n"
    "0	13	20	2.767	1.920	E94\n"
    "0	13	22	2.928	1.902	E94\n"
    "0	13	37	3.031	1.891	E94\n"
    "0	13	64	3.031	1.891	E94\n"
    "0	14	20	0.884	2.332	E94\n"
    "0	14	37	1.781	2.075	E94\n"
    "0	15	15	2.531	2.050	C94\n"
    "0	15	18	2.214	2.094	E94\n"
    "0	15	19	2.022	2.146	E94\n"
    "0	15	20	2.757	1.822	E94\n"
    "0	15	22	3.802	1.727	E94\n"
    "0	15	25	2.319	2.112	E94\n"
    "0	15	26	2.359	2.106	E94\n"
    "0	15	30	3.750	1.731	E94\n"
    "0	15	37	3.565	1.765	C94\n"
    "0	15	40	3.859	1.666	E94\n"
    "0	15	43	3.221	1.717	E94\n"
    "0	15	57	3.993	1.713	E94\n"
    "0	15	63	3.724	1.733	E94\n"
    "0	15	64	3.548	1.747	E94\n"
    "0	15	71	4.014	1.341	C94\n"
    "0	17	20	2.397	1.865	E94\n"
    "0	17	22	2.566	1.844	E94\n"
    "0	17	37	3.098	1.787	E94\n"
    "0	17	43	4.900	1.601	E94\n"
    "0	18	20	3.172	1.780	E94\n"
    "0	18	22	2.757	1.822	E94\n"
    "0	18	32	10.748	1.450	#X94\n"
    "0	18	37	3.281	1.770	X94\n"
    "0	18	39	3.504	1.693	X94\n"
    "0	18	43	3.301	1.710	#X94\n"
    "0	18	48	6.186	1.540	X94\n"
    "0	18	55	4.432	1.628	E94\n"
    "0	18	58	2.568	1.783	E94\n"
    "0	18	62	5.510	1.570	#X94\n"
    "0	18	63	3.524	1.749	E94\n"
    "0	18	64	3.856	1.723	E94\n"
    "0	18	80	4.150	1.702	E94\n"
    "0	19	20	2.288	1.900	E94\n"
    "0	19	37	3.072	1.809	E94\n"
    "0	19	40	4.470	1.686	E94\n"
    "0	19	63	3.219	1.795	E94\n"
    "0	19	75	1.600	2.226	E94\n"
    "0	20	20	3.663	1.526	C94\n"
    "0	20	22	4.251	1.484	E94\n"
    "0	20	25	2.718	1.838	E94\n"
    "0	20	26	2.588	1.853	E94\n"
    "0	20	30	3.977	1.507	C94\n"
    "0	20	34	4.171	1.460	E94\n"
    "0	20	37	3.740	1.516	E94\n"
    "0	20	40	4.784	1.427	E94\n"
    "0	20	41	4.286	1.482	E94\n"
    "0	20	43	3.737	1.487	E94\n"
    "0	20	45	3.844	1.480	E94\n"
    "0	22	22	3.969	1.499	C94\n"
    "0	22	30	3.785	1.513	E94\n"
    "0	22	34	4.103	1.464	E94\n"
    "0	22	37	4.481	1.471	E94\n"
    "0	22	40	4.188	1.459	E94\n"
    "0	22	41	5.071	1.441	E94\n"
    "0	22	43	4.070	1.466	E94\n"
    "0	22	45	4.311	1.452	E94\n"
    "0	23	39	7.112	1.012	C94\n"
    "0	23	62	6.339	1.026	X94\n"
    "0	23	67	6.610	1.019	#E94\n"
    "0	23	68	5.899	1.038	C94\n"
    "0	25	25	1.514	2.253	E94\n"
    "0	25	32	8.296	1.510	#X94\n"
    "0	25	37	3.586	1.755	E94\n"
    "0	25	39	4.370	1.676	E94\n"
    "0	25	40	4.629	1.660	E94\n"
    "0	25	43	3.237	1.762	X94\n"
    "0	25	57	4.356	1.699	E94\n"
    "0	25	63	3.711	1.745	E94\n"
    "0	25	71	3.001	1.411	X94\n"
    "0	25	72	3.744	1.950	#X94\n"
    "0	26	26	1.414	2.279	E94\n"
    "0	26	34	3.395	1.748	E94\n"
    "0	26	37	3.207	1.788	E94\n"
    "0	26	40	4.870	1.646	E94\n"
    "0	26	71	2.959	1.415	C94\n"
    "0	28	40	6.576	1.018	C94\n"
    "0	28	43	6.265	1.028	X94\n"
    "0	28	48	6.413	1.024	X94\n"
    "0	30	30	9.579	1.343	C94\n"
    "1	30	30	5.355	1.428	E94\n"
    "0	30	40	8.447	1.298	E94\n"
    "1	30	67	5.274	1.404	E94\n"
    "0	31	70	7.880	0.969	C94\n"
    "0	32	41	9.756	1.261	C94\n"
    "0	32	45	9.420	1.233	X94\n"
    "0	32	67	7.926	1.269	E94\n"
    "0	32	68	4.398	1.348	C94\n"
    "0	32	69	6.098	1.261	C94\n"
    "0	32	73	8.427	1.510	#X94\n"
    "0	32	77	10.648	1.450	#X94\n"
    "0	32	82	8.594	1.252	E94\n"
    "0	34	36	6.163	1.028	C94\n"
    "0	34	37	4.347	1.450	E94\n"
    "0	34	43	4.401	1.351	E94\n"
    "0	35	37	9.767	1.262	E94\n"
    "0	35	63	12.760	1.207	E94\n"
    "0	36	54	6.529	1.022	C94\n"
    "0	36	55	6.744	1.014	C94\n"
    "0	36	56	6.490	1.017	C94\n"
    "0	36	58	6.610	1.019	#E94\n"
    "0	36	81	6.980	1.016	C94\n"
    "0	37	37	5.573	1.374	C94\n"
    "1	37	37	5.178	1.436	E94\n"
    "0	37	38	5.737	1.333	C94\n"
    "0	37	39	5.978	1.375	E94\n"
    "1	37	39	5.650	1.388	E94\n"
    "0	37	40	6.168	1.398	C94\n"
    "0	37	41	4.537	1.468	E94\n"
    "0	37	43	4.764	1.428	X94\n"
    "0	37	45	4.705	1.431	E94\n"
    "0	37	46	6.191	1.367	E94\n"
    "0	37	55	6.615	1.352	E94\n"
    "0	37	56	5.055	1.414	E94\n"
    "1	37	57	5.092	1.440	E94\n"
    "0	37	58	7.432	1.326	E94\n"
    "1	37	58	5.055	1.414	E94\n"
    "0	37	61	5.724	1.385	E94\n"
    "0	37	62	7.137	1.335	E94\n"
    "0	37	63	6.095	1.372	C94\n"
    "1	37	63	5.178	1.436	E94\n"
    "0	37	64	6.161	1.379	C94\n"
    "1	37	64	5.265	1.432	E94\n"
    "1	37	67	4.725	1.430	E94\n"
    "0	37	69	5.396	1.352	C94\n"
    "0	37	78	6.719	1.375	E94\n"
    "0	37	81	3.987	1.471	E94\n"
    "1	37	81	4.531	1.440	E94\n"
    "0	38	38	5.002	1.246	C94\n"
    "0	38	63	7.299	1.330	E94\n"
    "0	38	64	6.978	1.340	E94\n"
    "0	38	69	5.036	1.321	E94\n"
    "0	38	78	6.218	1.366	E94\n"
    "0	39	40	4.101	1.367	E94\n"
    "0	39	45	3.524	1.402	E94\n"
    "0	39	63	6.301	1.364	C94\n"
    "1	39	63	6.137	1.369	E94\n"
    "0	39	64	6.357	1.361	E94\n"
    "1	39	64	5.482	1.395	E94\n"
    "0	39	65	5.513	1.339	C94\n"
    "0	39	78	6.137	1.369	E94\n"
    "0	40	40	4.248	1.359	E94\n"
    "0	40	45	4.305	1.356	E94\n"
    "0	40	46	4.727	1.335	E94\n"
    "0	40	54	6.817	1.256	E94\n"
    "0	40	63	6.733	1.348	E94\n"
    "0	40	64	6.644	1.351	E94\n"
    "0	40	78	5.900	1.378	E94\n"
    "0	41	41	5.029	1.443	E94\n"
    "0	41	55	5.577	1.391	E94\n"
    "0	41	62	7.137	1.335	E94\n"
    "0	41	72	4.519	1.678	X94\n"
    "0	41	80	5.222	1.434	E94\n"
    "0	42	61	16.223	1.087	E94\n"
    "0	43	43	4.211	1.361	E94\n"
    "0	43	45	3.710	1.390	E94\n"
    "0	43	64	5.389	1.399	E94\n"
    "0	44	63	3.589	1.717	C94\n"
    "0	44	65	3.374	1.684	C94\n"
    "0	44	78	3.711	1.734	E94\n"
    "0	44	80	3.910	1.719	E94\n"
    "0	45	63	5.119	1.411	E94\n"
    "0	45	64	5.076	1.413	E94\n"
    "0	45	78	5.724	1.385	E94\n"
    "0	47	53	12.192	1.140	#X94\n"
    "0	49	50	6.812	0.991	C94\n"
    "0	51	52	7.100	0.987	X94\n"
    "0	55	57	7.227	1.319	C94\n"
    "0	55	62	3.977	1.374	E94\n"
    "0	55	64	5.529	1.393	E94\n"
    "0	55	80	7.500	1.324	E94\n"
    "0	56	57	4.137	1.383	C94\n"
    "0	56	63	5.900	1.378	E94\n"
    "0	56	80	6.470	1.357	E94\n"
    "1	57	63	5.400	1.426	E94\n"
    "1	57	64	5.135	1.438	E94\n"
    "0	58	63	6.794	1.346	E94\n"
    "0	58	64	6.164	1.368	E94\n"
    "0	59	63	5.787	1.360	C94\n"
    "0	59	65	4.756	1.388	C94\n"
    "0	59	78	6.127	1.364	E94\n"
    "0	59	80	7.064	1.332	E94\n"
    "0	59	82	3.855	1.431	E94\n"
    "0	60	61	15.749	1.170	#X94\n"
    "0	62	63	6.947	1.341	E94\n"
    "0	62	64	6.273	1.364	E94\n"
    "1	63	63	5.729	1.412	E94\n"
    "0	63	64	7.118	1.377	C94\n"
    "0	63	66	8.326	1.313	C94\n"
    "0	63	72	4.503	1.679	E94\n"
    "0	63	78	7.434	1.352	E94\n"
    "0	63	81	7.778	1.316	E94\n"
    "0	64	64	4.313	1.418	C94\n"
    "1	64	64	4.926	1.448	E94\n"
    "0	64	65	8.258	1.335	C94\n"
    "0	64	66	4.456	1.369	C94\n"
    "0	64	78	5.492	1.422	E94\n"
    "0	64	81	5.824	1.381	E94\n"
    "0	64	82	6.794	1.346	E94\n"
    "0	65	66	7.243	1.323	C94\n"
    "0	65	78	8.447	1.298	E94\n"
    "0	65	81	5.223	1.313	E94\n"
    "0	65	82	5.622	1.297	E94\n"
    "0	66	66	3.874	1.368	C94\n"
    "0	66	78	6.385	1.360	E94\n"
    "0	66	81	3.960	1.375	E94\n"
    "0	67	67	6.085	1.280	E94\n"
    "0	71	75	2.852	1.423	X94\n"
    "0	72	73	2.628	2.035	X94\n"
    "0	76	76	4.286	1.357	X94\n"
    "0	76	78	6.824	1.345	X94\n"
    "0	78	78	5.573	1.374	C94\n"
    "0	78	79	8.890	1.287	E94\n"
    "0	78	81	5.046	1.381	C94\n"
    "0	79	79	6.408	1.269	E94\n"
    "0	79	81	4.305	1.356	E94\n"
    "0	80	81	8.237	1.335	C94\n";

class std::unique_ptr<MMFFBndkCollection> MMFFBndkCollection::ds_instance = nullptr;

extern const std::string defaultMMFFBndk;

MMFFBndkCollection *MMFFBndkCollection::getMMFFBndk(
    const std::string &mmffBndk) {
  if (!ds_instance || !mmffBndk.empty()) {
    ds_instance.reset(new MMFFBndkCollection(mmffBndk));
  }
  return ds_instance.get();
}

MMFFBndkCollection::MMFFBndkCollection(std::string mmffBndk) {
  if (mmffBndk.empty()) {
    mmffBndk = defaultMMFFBndk;
  }
  std::istringstream inStream(mmffBndk);
  std::string inLine = RDKit::getLine(inStream);
  while (!(inStream.eof())) {
    if (inLine[0] != '*') {
      MMFFBond mmffBondObj;
      boost::char_separator<char> tabSep("\t");
      tokenizer tokens(inLine, tabSep);
      tokenizer::iterator token = tokens.begin();

#ifdef RDKIT_MMFF_PARAMS_USE_STD_MAP
      unsigned int iAtomicNum = boost::lexical_cast<unsigned int>(*token);
#else
      d_iAtomicNum.push_back(
          (std::uint8_t)(boost::lexical_cast<unsigned int>(*token)));
#endif
      ++token;
#ifdef RDKIT_MMFF_PARAMS_USE_STD_MAP
      unsigned int jAtomicNum = boost::lexical_cast<unsigned int>(*token);
#else
      d_jAtomicNum.push_back(
          (std::uint8_t)boost::lexical_cast<unsigned int>(*token));
#endif
      ++token;
      mmffBondObj.r0 = boost::lexical_cast<double>(*token);
      ++token;
      mmffBondObj.kb = boost::lexical_cast<double>(*token);
      ++token;
#ifdef RDKIT_MMFF_PARAMS_USE_STD_MAP
      d_params[iAtomicNum][jAtomicNum] = mmffBondObj;
#else
      d_params.push_back(mmffBondObj);
#endif
    }
    inLine = RDKit::getLine(inStream);
  }
}

const std::string defaultMMFFBndk =
    "*\n"
    "*          Copyright (c) Merck and Co., Inc., 1994, 1995, 1996\n"
    "*                         All Rights Reserved\n"
    "*\n"
    "* MMFF BOND LENGTH, FORCE CONSTANT DEFAULT-RULE PARAMETERS\n"
    "* C94 =  Fitted to ab-initio derived core MMFF force constants\n"
    "* E94 =  Based on Herschberg/Laurie parameterization of Badger's rule\n"
    "*\n"
    "*	species	r0(ref)	kb(ref)	Source\n"
    "1	6	1.084	5.15	C94\n"
    "1	7	1.001	7.35	C94\n"
    "1	8	0.947	9.10	C94\n"
    "1	9	0.92	10.6	E94\n"
    "1	14	1.48	2.3	E94\n"
    "1	15	1.415	2.95	C94\n"
    "1	16	1.326	4.30	C94\n"
    "1	17	1.28	4.3	E94\n"
    "1	35	1.41	4.2	E94\n"
    "1	53	1.60	2.7	E94\n"
    "6	6	1.512	3.80	C94\n"
    "6	7	1.439	4.55	C94\n"
    "6	8	1.393	5.40	C94\n"
    "6	9	1.353	6.20	C94\n"
    "6	14	1.86	2.6	E94\n"
    "6	15	1.84	2.7	E94\n"
    "6	16	1.812	2.85	C94\n"
    "6	17	1.781	2.75	C94\n"
    "6	35	1.94	2.6	E94\n"
    "6	53	2.16	1.4	E94\n"
    "7	7	1.283	6.00	C94\n"
    "7	8	1.333	5.90	C94\n"
    "7	9	1.36	5.9	E94\n"
    "7	14	1.74	3.7	E94\n"
    "7	15	1.65	4.8	E94\n"
    "7	16	1.674	3.75	C94\n"
    "7	17	1.75	3.5	E94\n"
    "7	35	1.90	2.9	E94\n"
    "7	53	2.10	1.6	E94\n"
    "8	8	1.48	3.6	E94\n"
    "8	9	1.42	4.6	E94\n"
    "8	14	1.63	5.2	E94\n"
    "8	15	1.66	4.7	E94\n"
    "8	16	1.470	9.90	C94\n"
    "8	17	1.70	4.1	E94\n"
    "8	35	1.85	3.4	E94\n"
    "8	53	2.05	1.6	E94\n"
    "9	14	1.57	6.4	E94\n"
    "9	15	1.54	7.1	E94\n"
    "9	16	1.55	6.9	E94\n"
    "14	14	2.32	1.3	E94\n"
    "14	15	2.25	1.5	E94\n"
    "14	16	2.15	2.0	E94\n"
    "14	17	2.02	3.1	E94\n"
    "14	35	2.19	2.1	E94\n"
    "14	53	2.44	1.5	E94\n"
    "15	15	2.21	1.7	E94\n"
    "15	16	2.10	2.4	E94\n"
    "15	17	2.03	3.0	E94\n"
    "15	35	2.21	2.0	E94\n"
    "15	53	2.47	1.4	E94\n"
    "16	16	2.052	2.50	C94\n"
    "16	17	2.04	2.9	E94\n"
    "16	35	2.24	1.9	E94\n"
    "16	53	2.40	1.7	E94\n"
    "17	17	1.99	3.5	E94\n"
    "35	35	2.28	2.4	E94\n"
    "53	53	2.67	1.6	E94\n";

class std::unique_ptr<MMFFHerschbachLaurieCollection>
    MMFFHerschbachLaurieCollection::ds_instance = nullptr;

extern const std::string defaultMMFFHerschbachLaurie;

MMFFHerschbachLaurieCollection *
MMFFHerschbachLaurieCollection::getMMFFHerschbachLaurie(
    const std::string &mmffHerschbachLaurie) {
  if (!ds_instance || !mmffHerschbachLaurie.empty()) {
    ds_instance.reset(new MMFFHerschbachLaurieCollection(mmffHerschbachLaurie));
  }
  return ds_instance.get();
}

MMFFHerschbachLaurieCollection::MMFFHerschbachLaurieCollection(
    std::string mmffHerschbachLaurie) {
  if (mmffHerschbachLaurie.empty()) {
    mmffHerschbachLaurie = defaultMMFFHerschbachLaurie;
  }
  std::istringstream inStream(mmffHerschbachLaurie);
  std::string inLine = RDKit::getLine(inStream);
  while (!(inStream.eof())) {
    if (inLine[0] != '*') {
      MMFFHerschbachLaurie mmffHerschbachLaurieObj;
      boost::char_separator<char> tabSep("\t");
      tokenizer tokens(inLine, tabSep);
      tokenizer::iterator token = tokens.begin();

#ifdef RDKIT_MMFF_PARAMS_USE_STD_MAP
      unsigned int iRow = boost::lexical_cast<unsigned int>(*token);
#else
      d_iRow.push_back(
          (std::uint8_t)(boost::lexical_cast<unsigned int>(*token)));
#endif
      ++token;
#ifdef RDKIT_MMFF_PARAMS_USE_STD_MAP
      unsigned int jRow = boost::lexical_cast<unsigned int>(*token);
#else
      d_jRow.push_back(
          (std::uint8_t)boost::lexical_cast<unsigned int>(*token));
#endif
      ++token;
      mmffHerschbachLaurieObj.a_ij = boost::lexical_cast<double>(*token);
      ++token;
      mmffHerschbachLaurieObj.d_ij = boost::lexical_cast<double>(*token);
      ++token;
      mmffHerschbachLaurieObj.dp_ij = boost::lexical_cast<double>(*token);
      ++token;
#ifdef RDKIT_MMFF_PARAMS_USE_STD_MAP
      d_params[iRow][jRow] = mmffHerschbachLaurieObj;
#else
      d_params.push_back(mmffHerschbachLaurieObj);
#endif
    }
    inLine = RDKit::getLine(inStream);
  }
}

const std::string defaultMMFFHerschbachLaurie =
    "*Table I. Parameters for Badger's Rule.\n"
    "*\n"
    "*	i j a_ij d_ij dp_ij\n"
    "0	0	1.26	0.025	0.025\n"
    "0	1	1.66	0.30	0.36\n"
    "0	2	1.84	0.38	0.58\n"
    "0	3	1.98	0.49	0.65\n"
    "0	4	2.03	0.51	0.80\n"
    "0	5	2.03	0.25	0.81\n"
    "0	30	1.85	0.15	0.53\n"
    "0	40	1.84	0.61	0.61\n"
    "0	50	1.78	0.97	0.62\n"
    "1	1	1.91	0.68	0.68\n"
    "1	2	2.28	0.74	0.92\n"
    "1	3	2.35	0.85	1.02\n"
    "1	4	2.33	0.68	1.12\n"
    "1	5	2.50	0.97	1.22\n"
    "1	30	2.08	1.14	0.97\n"
    "1	40	2.34	1.17	1.08\n"
    "2	2	2.41	1.18	1.18\n"
    "2	3	2.52	1.02	1.28\n"
    "2	4	2.61	1.28	1.40\n"
    "2	5	2.60	0.84	1.24\n"
    "3	3	2.58	1.41	1.35\n"
    "3	4	2.66	0.86	1.48\n"
    "3	5	2.75	1.14	1.55\n"
    "4	4	2.85	1.62	1.62\n"
    "4	5	2.76	1.25	1.51\n";

class std::unique_ptr<MMFFCovRadPauEleCollection> MMFFCovRadPauEleCollection::ds_instance =
    nullptr;

extern const std::string defaultMMFFCovRadPauEle;

MMFFCovRadPauEleCollection *MMFFCovRadPauEleCollection::getMMFFCovRadPauEle(
    const std::string &mmffCovRadPauEle) {
  if (!ds_instance || !mmffCovRadPauEle.empty()) {
    ds_instance.reset(new MMFFCovRadPauEleCollection(mmffCovRadPauEle));
  }
  return ds_instance.get();
}

MMFFCovRadPauEleCollection::MMFFCovRadPauEleCollection(
    std::string mmffCovRadPauEle) {
  if (mmffCovRadPauEle.empty()) {
    mmffCovRadPauEle = defaultMMFFCovRadPauEle;
  }
  std::istringstream inStream(mmffCovRadPauEle);
  std::string inLine = RDKit::getLine(inStream);
  while (!(inStream.eof())) {
    if (inLine[0] != '*') {
      MMFFCovRadPauEle mmffCovRadPauEleObj;
      boost::char_separator<char> tabSep("\t");
      tokenizer tokens(inLine, tabSep);
      tokenizer::iterator token = tokens.begin();

#ifdef RDKIT_MMFF_PARAMS_USE_STD_MAP
      unsigned int atomicNum = boost::lexical_cast<unsigned int>(*token);
#else
      d_atomicNum.push_back(
          (std::uint8_t)(boost::lexical_cast<unsigned int>(*token)));
#endif
      ++token;
      mmffCovRadPauEleObj.r0 = boost::lexical_cast<double>(*token);
      ++token;
      mmffCovRadPauEleObj.chi = boost::lexical_cast<double>(*token);
      ++token;
#ifdef RDKIT_MMFF_PARAMS_USE_STD_MAP
      d_params[atomicNum] = mmffCovRadPauEleObj;
#else
      d_params.push_back(mmffCovRadPauEleObj);
#endif
    }
    inLine = RDKit::getLine(inStream);
  }
}

const std::string defaultMMFFCovRadPauEle =
    "*	atomicNum	covRad	pauEle\n"
    "1	0.33	2.20\n"
    "3	1.34	0.97\n"
    "6	0.77	2.50\n"
    "7	0.73	3.07\n"
    "8	0.72	3.50\n"
    "9	0.74	4.12\n"
    "11	1.54	1.01\n"
    "12	1.30	1.23\n"
    "14	1.15	1.74\n"
    "15	1.09	2.06\n"
    "16	1.03	2.44\n"
    "17	1.01	2.83\n"
    "19	1.96	0.91\n"
    "20	1.74	1.04\n"
    "29	1.38	1.75\n"
    "30	1.31	1.66\n"
    "35	1.15	2.74\n"
    "53	1.33	2.21\n";

class std::unique_ptr<MMFFAngleCollection> MMFFAngleCollection::ds_instance = nullptr;

extern const std::string defaultMMFFAngleData[];

MMFFAngleCollection *MMFFAngleCollection::getMMFFAngle(
    const std::string &mmffAngle) {
  if (!ds_instance || !mmffAngle.empty()) {
    ds_instance.reset(new MMFFAngleCollection(mmffAngle));
  }
  return ds_instance.get();
}

MMFFAngleCollection::MMFFAngleCollection(std::string mmffAngle) {
  if (mmffAngle.empty()) {
    unsigned int i = 0;
    while (defaultMMFFAngleData[i] != "EOS") {
      mmffAngle += defaultMMFFAngleData[i];
      ++i;
    }
  }
  std::istringstream inStream(mmffAngle);

  std::string inLine = RDKit::getLine(inStream);
  while (!(inStream.eof())) {
    if (inLine[0] != '*') {
      MMFFAngle mmffAngleObj;
      boost::char_separator<char> tabSep("\t");
      tokenizer tokens(inLine, tabSep);
      tokenizer::iterator token = tokens.begin();

#ifdef RDKIT_MMFF_PARAMS_USE_STD_MAP
      unsigned int angleType = boost::lexical_cast<unsigned int>(*token);
#else
      d_angleType.push_back(
          (std::uint8_t)(boost::lexical_cast<unsigned int>(*token)));
#endif
      ++token;
#ifdef RDKIT_MMFF_PARAMS_USE_STD_MAP
      unsigned int iAtomType = boost::lexical_cast<unsigned int>(*token);
#else
      d_iAtomType.push_back(
          (std::uint8_t)(boost::lexical_cast<unsigned int>(*token)));
#endif
      ++token;
#ifdef RDKIT_MMFF_PARAMS_USE_STD_MAP
      unsigned int jAtomType = boost::lexical_cast<unsigned int>(*token);
#else
      d_jAtomType.push_back(
          (std::uint8_t)(boost::lexical_cast<unsigned int>(*token)));
#endif
      ++token;
#ifdef RDKIT_MMFF_PARAMS_USE_STD_MAP
      unsigned int kAtomType = boost::lexical_cast<unsigned int>(*token);
#else
      d_kAtomType.push_back(
          (std::uint8_t)(boost::lexical_cast<unsigned int>(*token)));
#endif
      ++token;
      mmffAngleObj.ka = boost::lexical_cast<double>(*token);
      ++token;
      mmffAngleObj.theta0 = boost::lexical_cast<double>(*token);
      ++token;
#ifdef RDKIT_MMFF_PARAMS_USE_STD_MAP
      d_params[angleType][iAtomType][jAtomType][kAtomType] = mmffAngleObj;
#else
      d_params.push_back(mmffAngleObj);
#endif
    }
    inLine = RDKit::getLine(inStream);
  }
}

// another joy of VC++ "compiler limit: string exceeds 65535 bytes in length"
// compels us to
// break this into pieces
const std::string defaultMMFFAngleData[] = {
    "*\n"
    "*           Copyright (c) Merck and Co., Inc., 1994, 1995, 1996\n"
    "*                           All Rights Reserved\n"
    "*\n"
    "* MMFF ANGLE PARAMETERS- Rev:  26-Oct-94  Source: MMFF\n"
    "* C94  = CORE MMFF parameter - obtained from ab initio data\n"
    "* X94  = EXTD MMFF parameter - fit to additional ab initio data\n"
    "* E94  = theta0 from fit to X-ray data, ka from empirical rule\n"
    "* #E94 = theta0 and ka both from empirical rules\n"
    "*\n"
    "*    atom types      ka     theta0    Comment/origin\n"
    "0	0	1	0	0.000	108.900	0:*-1-*	MMFF	DEF\n"
    "0	1	1	1	0.851	109.608	C94\n"
    "0	1	1	2	0.736	109.445	C94\n"
    "0	1	1	3	0.777	107.517	C94\n"
    "0	1	1	4	1.006	110.265	E94\n"
    "0	1	1	5	0.636	110.549	C94\n"
    "0	1	1	6	0.992	108.133	C94\n"
    "0	1	1	8	0.777	108.290	C94\n"
    "0	1	1	9	1.136	108.194	E94\n"
    "0	1	1	10	1.050	109.960	C94\n"
    "0	1	1	11	1.225	108.313	C94\n"
    "0	1	1	12	1.056	108.679	C94\n"
    "0	1	1	13	1.078	106.820	E94\n"
    "0	1	1	14	0.980	109.945	E94\n"
    "0	1	1	15	0.743	107.397	C94\n"
    "0	1	1	17	1.089	108.578	E94\n"
    "0	1	1	18	1.093	109.315	E94\n"
    "0	1	1	19	0.755	115.436	E94\n"
    "0	1	1	20	1.021	108.659	E94\n"
    "0	1	1	22	1.001	110.125	E94\n"
    "0	1	1	25	0.803	112.356	X94\n"
    "0	1	1	26	0.833	109.879	E94\n"
    "0	1	1	34	1.179	106.493	C94\n"
    "0	1	1	37	0.756	108.617	C94\n"
    "0	1	1	39	0.927	109.170	C94\n"
    "0	1	1	40	1.130	108.678	E94\n"
    "0	1	1	41	0.330	98.422	C94\n"
    "0	1	1	43	1.135	108.019	E94\n"
    "0	1	1	45	1.197	105.028	E94\n"
    "0	1	1	54	1.173	106.424	E94\n"
    "0	1	1	55	1.150	107.604	E94\n"
    "0	1	1	56	1.199	110.371	C94\n"
    "0	1	1	57	1.012	109.900	E94\n"
    "0	1	1	58	1.179	106.327	E94\n"
    "0	1	1	61	1.125	109.311	E94\n"
    "0	1	1	63	1.006	110.058	E94\n"
    "0	1	1	64	0.988	111.064	E94\n"
    "0	1	1	67	1.216	104.557	E94\n"
    "0	1	1	68	1.018	107.195	C94\n"
    "0	1	1	73	1.160	104.658	E94\n",
    "0	1	1	78	1.012	109.850	E94\n"
    "0	1	1	80	0.947	113.327	E94\n"
    "0	1	1	81	1.108	109.837	E94\n"
    "0	2	1	2	1.113	111.453	C94\n"
    "0	2	1	3	0.667	104.829	C94\n"
    "0	2	1	4	1.022	109.873	E94\n"
    "0	2	1	5	0.632	110.292	C94\n"
    "0	2	1	6	1.074	108.699	C94\n"
    "0	2	1	8	0.884	111.553	C94\n"
    "0	2	1	9	1.118	109.577	E94\n"
    "0	2	1	10	1.160	107.963	E94\n"
    "0	2	1	11	1.192	110.419	E94\n"
    "0	2	1	12	1.070	109.410	E94\n"
    "0	2	1	15	1.078	109.560	E94\n"
    "0	2	1	17	1.077	109.434	E94\n"
    "0	2	1	18	1.188	105.110	E94\n"
    "0	2	1	20	1.053	107.448	E94\n"
    "0	2	1	22	0.942	114.020	E94\n"
    "0	2	1	25	0.893	106.815	E94\n"
    "0	2	1	26	1.029	99.065	E94\n"
    "0	2	1	34	1.066	111.817	E94\n"
    "0	2	1	37	0.985	111.446	E94\n"
    "0	2	1	39	1.124	109.513	E94\n"
    "0	2	1	40	1.149	108.270	E94\n"
    "0	2	1	45	1.232	103.978	E94\n"
    "0	2	1	63	0.935	114.692	E94\n"
    "0	2	1	67	1.224	104.687	E94\n"
    "0	3	1	3	0.974	111.746	E94\n"
    "0	3	1	4	1.019	109.850	E94\n"
    "0	3	1	5	0.650	108.385	C94\n"
    "0	3	1	6	0.528	104.112	C94\n"
    "0	3	1	8	1.197	105.837	E94\n"
    "0	3	1	9	1.201	105.535	E94\n"
    "0	3	1	10	0.634	102.655	C94\n"
    "0	3	1	11	1.189	110.328	E94\n"
    "0	3	1	12	1.136	106.064	E94\n"
    "0	3	1	13	1.147	103.645	E94\n"
    "0	3	1	14	1.048	106.404	E94\n"
    "0	3	1	15	1.125	107.192	E94\n"
    "0	3	1	17	1.092	108.602	E94\n"
    "0	3	1	18	1.120	108.119	E94\n"
    "0	3	1	20	0.969	111.830	E94\n"
    "0	3	1	22	0.999	110.522	E94\n"
    "0	3	1	26	0.742	116.555	E94\n"
    "0	3	1	34	1.141	107.871	E94\n"
    "0	3	1	37	1.011	109.833	E94\n"
    "0	3	1	39	1.136	108.751	E94\n"
    "0	3	1	40	1.174	106.941	E94\n"
    "0	3	1	41	1.033	108.216	E94\n"
    "0	3	1	45	1.221	104.281	E94\n"
    "0	3	1	63	1.069	107.077	E94\n",
    "0	3	1	64	1.028	109.186	E94\n"
    "0	3	1	81	1.167	107.327	E94\n"
    "0	4	1	4	0.954	114.186	E94\n"
    "0	4	1	5	0.615	111.417	X94\n"
    "0	4	1	6	1.273	109.977	E94\n"
    "0	4	1	8	1.099	111.063	E94\n"
    "0	4	1	9	1.187	106.750	E94\n"
    "0	4	1	10	1.117	110.488	E94\n"
    "0	4	1	13	1.021	110.047	E94\n"
    "0	4	1	15	1.028	112.432	E94\n"
    "0	4	1	18	1.187	105.351	E94\n"
    "0	4	1	22	1.174	102.556	E94\n"
    "0	4	1	26	0.853	108.999	E94\n"
    "0	4	1	34	1.148	108.160	E94\n"
    "0	4	1	37	0.993	111.424	E94\n"
    "0	5	1	5	0.516	108.836	C94\n"
    "0	5	1	6	0.781	108.577	C94\n"
    "0	5	1	8	0.653	110.297	C94\n"
    "0	5	1	9	0.733	109.894	C94\n"
    "0	5	1	10	0.740	107.646	C94\n"
    "0	5	1	11	0.875	107.897	C94\n"
    "0	5	1	12	0.698	108.162	C94\n"
    "0	5	1	13	0.613	106.049	E94\n"
    "0	5	1	14	0.508	113.019	E94\n"
    "0	5	1	15	0.576	109.609	C94\n"
    "0	5	1	17	0.634	107.944	X94\n"
    "0	5	1	18	0.663	106.855	X94\n"
    "0	5	1	19	0.450	113.195	X94\n"
    "0	5	1	20	0.706	111.000	C94\n"
    "0	5	1	22	0.618	110.380	E94\n"
    "0	5	1	25	0.487	109.486	X94\n"
    "0	5	1	26	0.466	111.172	X94\n"
    "0	5	1	34	0.872	106.224	C94\n"
    "0	5	1	35	0.644	125.663	X94\n"
    "0	5	1	37	0.627	109.491	C94\n"
    "0	5	1	39	0.811	106.299	C94\n"
    "0	5	1	40	0.719	109.870	C94\n"
    "0	5	1	41	0.525	108.904	C94\n"
    "0	5	1	43	0.692	109.083	X94\n"
    "0	5	1	45	0.741	105.197	X94\n"
    "0	5	1	46	0.719	106.735	X94\n"
    "0	5	1	54	0.874	106.973	C94\n"
    "0	5	1	55	0.861	108.507	C94\n"
    "0	5	1	56	0.814	108.223	C94\n"
    "0	5	1	57	0.626	110.420	E94\n"
    "0	5	1	58	0.750	105.481	E94\n"
    "0	5	1	61	0.710	109.227	X94\n"
    "0	5	1	62	0.655	113.035	X94\n"
    "0	5	1	63	0.621	110.467	E94\n"
    "0	5	1	64	0.622	110.457	E94\n"
    "0	5	1	67	0.732	106.474	E94\n",
    "0	5	1	68	0.748	103.817	C94\n"
    "0	5	1	72	0.547	116.576	X94\n"
    "0	5	1	73	0.633	107.153	X94\n"
    "0	5	1	78	0.640	109.078	E94\n"
    "0	5	1	80	0.684	105.144	E94\n"
    "0	5	1	81	0.721	107.870	E94\n"
    "0	6	1	6	1.156	111.368	C94\n"
    "0	6	1	8	1.333	112.223	E94\n"
    "0	6	1	9	1.224	116.950	E94\n"
    "0	6	1	10	1.432	108.568	E94\n"
    "0	6	1	11	1.593	106.900	E94\n"
    "0	6	1	15	1.273	112.012	E94\n"
    "0	6	1	17	1.348	108.655	E94\n"
    "0	6	1	19	0.906	117.214	E94\n"
    "0	6	1	20	1.293	108.202	E94\n"
    "0	6	1	22	1.287	108.913	E94\n"
    "0	6	1	25	1.171	103.598	E94\n"
    "0	6	1	26	0.888	118.433	E94\n"
    "0	6	1	34	1.257	114.975	E94\n"
    "0	6	1	37	0.878	107.978	C94\n"
    "0	6	1	39	1.485	106.464	E94\n"
    "0	6	1	40	1.371	110.779	E94\n"
    "0	6	1	41	1.333	106.467	E94\n"
    "0	6	1	45	1.523	104.438	E94\n"
    "0	6	1	57	1.308	108.467	E94\n"
    "0	6	1	63	1.351	106.535	E94\n"
    "0	6	1	64	1.238	111.308	E94\n"
    "0	8	1	8	1.203	110.856	E94\n"
    "0	8	1	9	1.133	114.080	E94\n"
    "0	8	1	10	1.258	108.683	E94\n"
    "0	8	1	12	1.217	107.251	E94\n"
    "0	8	1	15	1.120	112.356	E94\n"
    "0	8	1	20	1.116	109.353	E94\n"
    "0	8	1	25	1.143	98.698	E94\n"
    "0	8	1	34	1.138	113.412	E94\n"
    "0	8	1	37	1.090	110.992	E94\n"
    "0	8	1	39	1.364	104.193	E94\n"
    "0	8	1	40	0.964	123.962	E94\n"
    "0	8	1	41	1.234	103.868	E94\n"
    "0	8	1	43	1.137	113.596	E94\n"
    "0	8	1	45	1.583	96.139	E94\n"
    "0	8	1	57	1.038	114.266	E94\n"
    "0	8	1	63	1.104	110.598	E94\n"
    "0	8	1	64	1.156	108.127	E94\n"
    "0	9	1	10	1.209	110.720	E94\n"
    "0	9	1	12	1.173	109.152	E94\n"
    "0	9	1	15	1.024	117.465	E94\n"
    "0	9	1	25	1.060	102.432	E94\n"
    "0	9	1	37	1.077	111.565	E94\n"
    "0	9	1	40	1.084	116.728	E94\n"
    "0	9	1	80	1.163	107.509	E94\n",
    "0	10	1	10	1.191	111.995	E94\n"
    "0	10	1	15	1.161	110.502	E94\n"
    "0	10	1	17	1.269	105.509	E94\n"
    "0	10	1	20	1.220	104.838	E94\n"
    "0	10	1	22	1.132	109.262	E94\n"
    "0	10	1	25	1.015	104.822	E94\n"
    "0	10	1	37	1.107	110.423	E94\n"
    "0	10	1	40	1.264	108.536	E94\n"
    "0	10	1	41	1.087	110.961	E94\n"
    "0	10	1	57	1.268	103.622	E94\n"
    "0	11	1	11	1.638	106.081	C94\n"
    "0	11	1	15	1.254	109.517	E94\n"
    "0	11	1	20	1.243	107.637	E94\n"
    "0	11	1	25	1.244	97.532	E94\n"
    "0	11	1	34	1.338	108.669	E94\n"
    "0	11	1	35	1.556	110.367	E94\n"
    "0	11	1	37	1.151	112.278	E94\n"
    "0	11	1	41	1.301	105.053	E94\n"
    "0	11	1	45	1.550	100.991	E94\n"
    "0	11	1	73	1.303	106.569	E94\n"
    "0	11	1	75	0.884	114.378	E94\n"
    "0	12	1	12	1.096	110.422	C94\n"
    "0	12	1	15	1.146	111.064	E94\n"
    "0	12	1	18	1.299	104.827	E94\n"
    "0	12	1	19	0.932	108.971	E94\n"
    "0	12	1	20	1.081	108.605	E94\n"
    "0	12	1	22	1.097	108.028	E94\n"
    "0	12	1	25	0.989	106.118	E94\n"
    "0	12	1	37	1.076	109.030	E94\n"
    "0	12	1	39	1.150	110.359	E94\n"
    "0	12	1	45	1.353	101.430	E94\n"
    "0	12	1	63	1.071	109.474	E94\n"
    "0	12	1	64	1.093	108.338	E94\n"
    "0	13	1	13	1.093	111.645	E94\n"
    "0	13	1	20	1.084	106.534	E94\n"
    "0	13	1	22	1.068	107.469	E94\n"
    "0	13	1	45	1.305	101.383	E94\n"
    "0	14	1	20	1.021	107.718	E94\n"
    "0	15	1	15	1.147	111.896	E94\n"
    "0	15	1	25	1.059	103.308	E94\n"
    "0	15	1	34	1.222	107.318	E94\n"
    "0	15	1	37	1.051	110.959	E94\n"
    "0	15	1	40	1.149	111.005	E94\n"
    "0	15	1	41	1.263	100.981	E94\n"
    "0	15	1	63	1.060	110.596	E94\n"
    "0	15	1	64	1.059	110.703	E94\n"
    "0	15	1	73	1.289	105.029	E94\n"
    "0	17	1	37	1.065	110.049	E94\n"
    "0	18	1	20	1.121	107.960	E94\n"
    "0	18	1	22	1.283	101.125	E94\n"
    "0	18	1	37	1.203	104.390	E94\n",
    "0	18	1	45	1.287	105.273	E94\n"
    "0	18	1	64	1.093	109.683	E94\n"
    "0	19	1	54	0.772	119.506	E94\n"
    "0	20	1	20	1.229	99.084	E94\n"
    "0	20	1	37	1.052	107.428	E94\n"
    "0	20	1	45	1.169	106.335	E94\n"
    "0	22	1	22	0.990	111.226	E94\n"
    "0	22	1	25	0.885	107.293	E94\n"
    "0	22	1	34	1.045	112.940	E94\n"
    "0	22	1	37	1.037	108.586	E94\n"
    "0	22	1	45	1.182	106.181	E94\n"
    "0	25	1	25	0.551	127.138	E94\n"
    "0	25	1	34	0.779	119.271	E94\n"
    "0	25	1	37	0.784	113.945	E94\n"
    "0	25	1	40	1.062	102.417	E94\n"
    "0	25	1	58	0.916	110.234	E94\n"
    "0	26	1	26	0.625	118.700	E94\n"
    "0	34	1	34	1.216	109.167	E94\n"
    "0	34	1	37	1.075	111.275	E94\n"
    "0	34	1	41	1.048	112.238	E94\n"
    "0	34	1	63	1.077	111.412	E94\n"
    "0	34	1	73	1.142	110.240	E94\n"
    "0	37	1	37	0.986	111.315	E94\n"
    "0	37	1	40	1.129	109.188	E94\n"
    "0	37	1	43	1.074	111.478	E94\n"
    "0	37	1	45	1.259	102.800	E94\n"
    "0	37	1	57	0.981	112.047	E94\n"
    "0	37	1	64	1.175	102.239	E94\n"
    "0	37	1	68	1.100	109.983	E94\n"
    "0	37	1	78	1.005	110.638	E94\n"
    "0	37	1	81	1.176	107.040	E94\n"
    "0	39	1	39	1.260	108.547	E94\n"
    "0	40	1	40	1.182	112.005	E94\n"
    "0	40	1	55	1.322	105.786	E94\n"
    "0	40	1	63	1.032	114.505	E94\n"
    "0	40	1	64	1.000	116.376	E94\n"
    "0	41	1	41	1.082	105.400	E94\n"
    "0	41	1	63	1.061	107.112	E94\n"
    "0	41	1	81	1.093	110.553	E94\n"
    "0	45	1	45	1.391	102.088	E94\n"
    "0	0	2	0	0.000	119.400	0:*-2-*	MMFF	DEF\n"
    "1	0	2	0	0.000	118.200	0:*-2-*	MMFF	DEF\n"
    "2	0	2	0	0.000	120.800	2::*-2-*	MMFF	DEF\n"
    "3	0	2	0	0.000	62.600	3::*-2-*	MMFF	DEF\n"
    "6	0	2	0	0.000	60.500	6:*-2-*	MMFF	DEF\n"
    "0	1	2	1	0.752	118.043	C94\n"
    "0	1	2	2	0.672	122.141	C94\n"
    "1	1	2	2	0.684	116.929	C94\n"
    "1	1	2	3	0.698	116.104	C94\n"
    "0	1	2	4	0.828	125.045	E94\n"
    "1	1	2	4	0.846	121.613	E94\n",
    "0	1	2	5	0.446	120.108	C94\n"
    "0	1	2	6	1.160	115.518	E94\n"
    "0	1	2	10	1.015	116.707	E94\n"
    "0	1	2	12	0.983	115.343	E94\n"
    "0	1	2	13	0.964	115.395	E94\n"
    "0	1	2	15	0.939	119.465	E94\n"
    "0	1	2	17	0.883	121.868	E94\n"
    "0	1	2	18	0.961	117.918	E94\n"
    "0	1	2	20	0.880	118.310	E94\n"
    "0	1	2	22	0.873	119.114	E94\n"
    "0	1	2	30	0.826	124.605	E94\n"
    "1	1	2	37	0.721	116.064	C94\n"
    "0	1	2	40	0.982	118.515	E94\n"
    "0	1	2	45	1.121	109.921	E94\n"
    "0	1	2	56	1.006	117.192	E94\n"
    "1	1	2	63	0.768	127.945	E94\n"
    "1	1	2	64	0.966	113.884	E94\n"
    "1	1	2	67	1.115	110.185	E94\n"
    "1	2	2	2	0.747	121.550	C94\n"
    "2	2	2	2	0.796	126.284	E94\n"
    "6	2	2	2	0.173	60.549	E94\n"
    "1	2	2	3	0.545	111.297	C94\n"
    "2	2	2	3	0.893	118.456	E94\n"
    "5	2	2	3	0.184	59.145	E94\n"
    "1	2	2	4	0.902	121.053	E94\n"
    "2	2	2	4	0.889	119.794	E94\n"
    "0	2	2	5	0.535	121.004	C94\n"
    "1	2	2	5	0.463	118.442	C94\n"
    "0	2	2	6	1.117	121.267	C94\n"
    "1	2	2	6	1.204	114.538	E94\n"
    "1	2	2	9	0.960	123.536	E94\n"
    "2	2	2	9	1.045	116.273	E94\n"
    "0	2	2	10	1.003	120.828	E94\n"
    "1	2	2	10	1.026	117.324	E94\n"
    "0	2	2	11	1.089	119.100	X94\n"
    "1	2	2	11	1.090	116.828	E94\n"
    "0	2	2	12	0.931	120.132	X94\n"
    "1	2	2	12	0.957	117.526	E94\n"
    "0	2	2	13	0.867	122.717	E94\n"
    "0	2	2	14	0.818	122.584	E94\n"
    "1	2	2	14	0.819	122.344	E94\n"
    "0	2	2	15	0.931	121.553	E94\n"
    "1	2	2	15	0.949	119.466	E94\n"
    "0	2	2	17	0.977	117.167	E94\n"
    "0	2	2	18	1.044	114.561	E94\n"
    "0	2	2	19	0.668	124.721	E94\n"
    "0	2	2	20	0.931	117.784	E94\n"
    "0	2	2	22	0.809	126.820	E94\n"
    "3	2	2	22	0.149	66.165	E94\n"
    "0	2	2	25	0.700	123.830	E94\n"
    "0	2	2	34	1.066	116.151	E94\n",
    "0	2	2	35	0.911	137.103	X94\n"
    "1	2	2	37	0.598	117.508	C94\n"
    "2	2	2	37	0.817	124.229	E94\n"
    "1	2	2	39	0.976	122.360	E94\n"
    "0	2	2	40	0.773	126.830	C94\n"
    "1	2	2	40	0.976	120.132	E94\n"
    "0	2	2	41	0.432	110.442	C94\n"
    "0	2	2	43	1.144	111.808	E94\n"
    "0	2	2	45	1.194	109.231	X94\n"
    "1	2	2	45	1.062	113.984	E94\n"
    "0	2	2	46	1.005	121.534	E94\n"
    "0	2	2	55	0.995	121.154	E94\n"
    "0	2	2	56	1.234	108.879	E94\n"
    "0	2	2	62	0.808	135.269	X94\n"
    "1	2	2	63	0.948	118.277	E94\n"
    "1	2	2	64	0.866	123.528	E94\n"
    "2	2	2	64	0.859	121.998	E94\n"
    "1	2	2	67	1.132	112.136	E94\n"
    "0	2	2	72	0.770	134.269	X94\n"
    "1	2	2	81	1.078	116.541	E94\n"
    "2	3	2	3	0.853	120.370	E94\n"
    "2	3	2	4	0.878	119.739	E94\n"
    "1	3	2	5	0.487	117.291	C94\n"
    "1	3	2	6	1.142	116.738	E94\n"
    "2	3	2	9	1.005	117.648	E94\n"
    "1	3	2	10	1.039	115.698	E94\n"
    "1	3	2	11	1.150	112.876	E94\n"
    "1	3	2	12	0.997	114.732	E94\n"
    "1	3	2	13	0.946	116.643	E94\n"
    "1	3	2	14	0.891	117.111	E94\n"
    "1	3	2	15	1.023	114.635	E94\n"
    "1	3	2	20	0.870	119.265	E94\n"
    "1	3	2	22	0.816	123.510	E94\n"
    "1	3	2	30	1.025	112.209	E94\n"
    "1	3	2	34	1.099	111.723	E94\n"
    "1	3	2	35	1.141	118.767	E94\n"
    "2	3	2	37	0.868	119.758	E94\n"
    "1	3	2	40	1.024	116.408	E94\n"
    "1	3	2	41	0.855	119.505	E94\n"
    "1	3	2	43	1.046	114.257	E94\n"
    "1	3	2	45	1.077	112.401	E94\n"
    "1	3	2	46	1.066	114.841	E94\n"
    "2	4	2	4	0.832	124.158	E94\n"
    "0	4	2	5	0.573	121.000	#E94\n"
    "1	4	2	5	0.545	120.000	#E94\n"
    "2	4	2	9	0.973	120.845	E94\n"
    "1	4	2	15	0.906	122.447	E94\n"
    "1	4	2	18	0.947	119.537	E94\n"
    "1	4	2	30	0.819	126.938	E94\n"
    "2	4	2	37	0.864	121.093	E94\n"
    "1	4	2	40	1.083	114.355	E94\n",
    "1	4	2	45	1.158	109.426	E94\n"
    "2	4	2	63	0.860	122.442	E94\n"
    "0	5	2	5	0.365	119.523	C94\n"
    "0	5	2	6	0.589	108.757	C94\n"
    "1	5	2	9	0.643	117.000	#E94\n"
    "0	5	2	10	0.667	114.859	E94\n"
    "0	5	2	11	0.795	108.186	X94\n"
    "0	5	2	12	0.622	110.650	X94\n"
    "0	5	2	13	0.566	113.513	E94\n"
    "0	5	2	15	0.546	119.562	E94\n"
    "0	5	2	17	0.492	124.000	#E94\n"
    "0	5	2	18	0.548	119.053	E94\n"
    "0	5	2	22	0.534	120.000	#E94\n"
    "0	5	2	25	0.395	124.000	#E94\n"
    "0	5	2	30	0.572	120.000	#E94\n"
    "0	5	2	35	0.682	124.164	X94\n"
    "1	5	2	37	0.491	117.423	C94\n"
    "1	5	2	39	0.655	115.724	E94\n"
    "0	5	2	40	0.568	112.322	C94\n"
    "0	5	2	41	0.294	123.706	C94\n"
    "0	5	2	45	0.728	107.774	X94\n"
    "0	5	2	55	0.651	116.000	#E94\n"
    "0	5	2	62	0.568	125.344	X94\n"
    "1	5	2	63	0.550	120.000	#E94\n"
    "1	5	2	64	0.546	120.000	#E94\n"
    "0	5	2	72	0.531	122.009	X94\n"
    "1	5	2	81	0.665	115.000	#E94\n"
    "1	6	2	9	1.214	120.520	E94\n"
    "0	6	2	10	1.311	115.921	E94\n"
    "0	6	2	22	1.080	120.560	E94\n"
    "0	6	2	35	1.172	132.391	E94\n"
    "1	6	2	37	1.198	114.441	E94\n"
    "0	6	2	40	1.239	119.073	E94\n"
    "0	6	2	45	1.637	102.438	E94\n"
    "1	9	2	10	1.098	119.802	E94\n"
    "1	9	2	15	0.915	127.574	E94\n"
    "2	9	2	37	0.981	119.536	E94\n"
    "1	9	2	40	0.922	130.521	E94\n"
    "0	10	2	12	1.144	112.723	E94\n"
    "0	10	2	15	1.078	117.519	E94\n"
    "0	10	2	25	1.144	100.818	E94\n"
    "1	10	2	37	1.021	117.139	E94\n"
    "0	10	2	40	0.988	126.034	E94\n"
    "0	10	2	41	0.951	120.000	E94\n"
    "0	12	2	12	1.012	119.105	E94\n"
    "0	12	2	17	1.110	114.206	E94\n"
    "0	12	2	18	1.201	110.553	E94\n"
    "0	12	2	19	0.704	126.646	E94\n"
    "0	12	2	20	0.903	120.563	E94\n"
    "0	12	2	30	0.892	122.753	E94\n"
    "1	12	2	37	0.976	116.136	E94\n",
    "0	12	2	45	1.076	115.543	E94\n"
    "0	13	2	18	1.132	113.616	E94\n"
    "0	15	2	15	0.996	123.027	E94\n"
    "0	15	2	35	0.950	133.654	E94\n"
    "1	15	2	37	1.007	115.757	E94\n"
    "0	15	2	40	0.895	128.924	E94\n"
    "0	17	2	17	1.051	117.955	E94\n"
    "1	18	2	37	1.183	106.608	E94\n"
    "0	22	2	22	0.841	122.108	E94\n"
    "3	22	2	22	0.180	58.963	E94\n"
    "1	22	2	37	0.806	124.693	E94\n"
    "1	30	2	37	0.849	123.816	E94\n"
    "1	35	2	37	0.991	128.032	E94\n"
    "0	40	2	40	0.949	128.436	E94\n"
    "0	40	2	56	1.072	120.987	E94\n"
    "1	40	2	63	0.922	124.268	E94\n"
    "1	40	2	64	0.955	121.881	E94\n"
    "0	40	2	72	0.820	135.317	E94\n"
    "0	45	2	45	1.284	108.095	E94\n"
    "2	64	2	64	0.888	120.342	E94\n"
    "0	0	3	0	0.000	117.300	0:*-3-*	MMFF	DEF\n"
    "1	0	3	0	0.000	115.800	1:*-3-*	MMFF	DEF\n"
    "4	0	3	0	0.000	90.800	4:*-3-*	MMFF	DEF\n"
    "7	0	3	0	0.000	91.100	7:*-3-*	MMFF	DEF\n"
    "8	0	3	0	0.000	88.900	7:*-3-*	MMFF	DEF\n"
    "0	1	3	1	1.151	118.016	C94\n"
    "1	1	3	2	1.106	116.853	C94\n"
    "1	1	3	3	1.214	114.612	C94\n"
    "0	1	3	5	0.808	117.280	C94\n"
    "0	1	3	6	1.043	109.716	C94\n"
    "0	1	3	7	0.938	124.410	C94\n"
    "0	1	3	9	0.978	119.788	E94\n"
    "1	1	3	9	1.038	115.132	E94\n"
    "0	1	3	10	0.984	112.735	C94\n"
    "0	1	3	12	1.007	113.972	E94\n"
    "0	1	3	15	1.024	113.612	E94\n"
    "0	1	3	16	0.949	119.986	E94\n"
    "0	1	3	18	0.732	134.097	E94\n"
    "0	1	3	20	0.830	120.312	E94\n"
    "0	1	3	22	0.928	115.001	E94\n"
    "0	1	3	35	1.058	122.808	E94\n"
    "1	1	3	37	1.051	115.191	C94\n"
    "1	1	3	39	1.178	107.895	E94\n"
    "0	1	3	40	0.979	118.457	E94\n"
    "0	1	3	41	0.897	116.681	E94\n"
    "0	1	3	43	1.046	113.731	X94\n"
    "0	1	3	45	1.132	109.019	E94\n"
    "0	1	3	51	1.160	116.573	X94\n"
    "0	1	3	53	1.052	115.065	X94\n"
    "0	1	3	54	1.135	111.322	E94\n"
    "1	1	3	58	1.162	108.129	E94\n",
    "0	1	3	62	1.119	111.523	E94\n"
    "1	1	3	63	0.909	117.001	E94\n"
    "1	1	3	64	0.887	118.253	E94\n"
    "0	1	3	67	1.142	110.666	E94\n"
    "0	1	3	74	1.010	116.851	X94\n"
    "0	1	3	75	0.646	128.037	X94\n"
    "2	2	3	2	0.976	112.562	E94\n"
    "6	2	3	2	0.157	62.792	E94\n"
    "2	2	3	3	0.957	113.239	E94\n"
    "1	2	3	5	0.901	115.350	C94\n"
    "1	2	3	6	0.932	106.510	C94\n"
    "1	2	3	7	0.936	122.623	C94\n"
    "1	2	3	9	0.831	122.253	C94\n"
    "2	2	3	9	1.120	111.408	E94\n"
    "1	2	3	10	1.042	111.721	C94\n"
    "1	2	3	12	0.901	120.769	E94\n"
    "1	2	3	15	1.057	112.105	E94\n"
    "1	2	3	16	0.881	124.850	E94\n"
    "1	2	3	22	0.969	113.027	E94\n"
    "1	2	3	25	0.853	109.794	E94\n"
    "2	2	3	37	0.973	112.935	E94\n"
    "2	2	3	39	1.197	107.592	E94\n"
    "1	2	3	40	0.910	123.437	E94\n"
    "1	2	3	43	1.105	111.169	E94\n"
    "1	2	3	53	1.082	114.032	E94\n"
    "1	2	3	54	1.012	118.588	E94\n"
    "1	2	3	55	1.186	107.278	E94\n"
    "1	2	3	56	1.151	108.909	E94\n"
    "2	2	3	63	0.918	116.947	E94\n"
    "2	2	3	64	1.033	110.084	E94\n"
    "1	2	3	67	1.022	117.597	E94\n"
    "2	3	3	3	0.822	121.775	E94\n"
    "8	3	3	3	1.280	89.965	E94\n"
    "1	3	3	5	0.943	113.762	C94\n"
    "1	3	3	6	0.935	103.030	C94\n"
    "1	3	3	7	0.919	117.024	C94\n"
    "1	3	3	9	1.050	115.704	E94\n"
    "1	3	3	10	1.129	110.421	E94\n"
    "1	3	3	12	1.053	111.492	E94\n"
    "1	3	3	15	1.390	97.562	E94\n"
    "1	3	3	16	1.092	111.888	E94\n"
    "1	3	3	20	0.977	110.910	E94\n"
    "1	3	3	22	1.010	110.295	E94\n"
    "8	3	3	30	1.353	87.789	E94\n"
    "2	3	3	37	0.932	114.949	E94\n"
    "2	3	3	39	1.237	105.384	E94\n"
    "1	3	3	40	1.003	117.124	E94\n"
    "1	3	3	41	0.790	124.361	E94\n"
    "1	3	3	45	0.919	121.023	E94\n"
    "1	3	3	53	1.170	109.169	E94\n"
    "2	3	3	63	0.981	112.685	E94\n",
    "2	3	3	64	0.880	118.840	E94\n"
    "1	3	3	67	1.119	111.860	E94\n"
    "1	4	3	6	1.269	111.750	E94\n"
    "1	4	3	7	1.126	120.852	E94\n"
    "1	4	3	9	1.192	109.833	E94\n"
    "2	4	3	37	0.964	114.081	E94\n"
    "0	5	3	5	0.594	116.699	C94\n"
    "0	5	3	6	0.819	108.253	C94\n"
    "0	5	3	7	0.670	123.439	C94\n"
    "0	5	3	9	0.623	119.491	C94\n"
    "1	5	3	9	0.638	117.168	E94\n"
    "0	5	3	10	0.874	111.761	C94\n"
    "0	5	3	16	0.522	124.405	E94\n"
    "1	5	3	37	0.564	116.400	E94\n"
    "0	5	3	40	0.959	111.684	C94\n"
    "0	5	3	53	0.644	118.000	#E94\n"
    "0	5	3	54	0.816	115.471	C94\n"
    "1	5	3	63	0.559	118.000	#E94\n"
    "1	5	3	64	0.566	117.000	#E94\n"
    "0	5	3	67	0.700	113.698	E94\n"
    "0	6	3	6	1.678	109.094	E94\n"
    "0	6	3	7	1.155	124.425	C94\n"
    "0	6	3	9	1.275	119.478	E94\n"
    "1	6	3	9	1.416	111.868	E94\n"
    "0	6	3	10	1.405	112.187	E94\n"
    "0	6	3	16	1.269	116.317	E94\n"
    "0	6	3	20	1.182	113.581	E94\n"
    "4	6	3	20	1.495	93.130	E94\n"
    "0	6	3	22	1.276	110.826	E94\n"
    "7	6	3	30	1.530	93.191	E94\n"
    "1	6	3	37	0.808	102.881	C94\n"
    "1	6	3	39	1.611	104.655	E94\n"
    "0	6	3	40	1.371	113.565	E94\n"
    "0	6	3	41	1.477	102.658	E94\n"
    "0	6	3	43	1.330	114.183	E94\n"
    "0	6	3	48	1.315	115.328	E94\n"
    "0	6	3	51	1.409	120.427	E94\n"
    "0	6	3	54	1.495	110.510	E94\n"
    "0	6	3	62	1.421	112.542	E94\n"
    "1	6	3	63	1.339	109.082	E94\n"
    "1	6	3	64	1.267	111.993	E94\n"
    "1	6	3	80	1.256	113.698	E94\n"
    "1	7	3	9	1.147	127.084	E94\n"
    "0	7	3	10	0.907	127.152	C94\n"
    "0	7	3	12	0.984	130.049	E94\n"
    "0	7	3	15	1.101	123.313	E94\n"
    "0	7	3	20	0.713	129.492	C94\n"
    "0	7	3	22	1.093	121.851	E94\n"
    "1	7	3	30	0.972	129.010	E94\n"
    "1	7	3	37	0.734	119.968	C94\n"
    "1	7	3	39	1.352	116.727	E94\n",
    "0	7	3	41	1.281	112.087	E94\n"
    "0	7	3	43	1.163	124.549	X94\n"
    "0	7	3	48	1.114	127.879	E94\n"
    "1	7	3	54	1.288	114.184	E94\n"
    "0	7	3	55	1.258	120.056	E94\n"
    "0	7	3	56	1.175	123.854	E94\n"
    "1	7	3	58	1.323	117.081	E94\n"
    "0	7	3	62	1.129	129.349	E94\n"
    "1	7	3	63	1.036	126.456	E94\n"
    "1	7	3	64	1.071	124.133	E94\n"
    "1	7	3	78	0.955	132.047	E94\n"
    "1	9	3	9	1.119	120.094	E94\n"
    "2	9	3	9	1.021	124.131	E94\n"
    "0	9	3	10	1.105	120.697	E94\n"
    "1	9	3	10	1.154	116.608	E94\n"
    "0	9	3	12	1.056	118.046	E94\n"
    "0	9	3	15	1.036	119.679	E94\n"
    "1	9	3	15	1.042	118.787	E94\n"
    "1	9	3	16	0.936	127.665	E94\n"
    "0	9	3	17	1.035	117.902	E94\n"
    "0	9	3	18	1.121	114.698	E94\n"
    "0	9	3	20	0.951	120.437	E94\n"
    "0	9	3	22	1.040	116.861	E94\n"
    "0	9	3	25	0.955	109.442	E94\n"
    "0	9	3	35	1.054	134.470	E94\n"
    "1	9	3	37	0.997	119.569	E94\n"
    "2	9	3	37	1.060	114.740	E94\n"
    "0	9	3	40	0.844	128.078	C94\n"
    "1	9	3	40	1.018	124.152	E94\n"
    "0	9	3	41	1.114	112.513	E94\n"
    "0	9	3	45	1.497	102.140	E94\n"
    "2	9	3	54	1.244	108.056	E94\n"
    "1	9	3	57	1.038	118.096	E94\n"
    "1	9	3	63	1.004	120.054	E94\n"
    "1	9	3	64	1.053	117.060	E94\n"
    "1	9	3	80	0.959	124.150	E94\n"
    "0	10	3	10	1.612	114.923	C94\n"
    "0	10	3	15	1.167	112.206	E94\n"
    "0	10	3	16	1.005	123.150	E94\n"
    "0	10	3	18	1.299	106.052	E94\n"
    "0	10	3	20	1.019	115.213	E94\n"
    "4	10	3	20	1.338	92.724	E94\n"
    "0	10	3	22	1.076	113.651	E94\n"
    "7	10	3	30	1.438	90.508	E94\n"
    "0	10	3	35	1.223	122.649	E94\n"
    "1	10	3	37	1.101	112.495	E94\n"
    "1	10	3	39	1.434	104.419	E94\n"
    "0	10	3	40	1.093	119.697	E94\n"
    "0	10	3	43	1.144	115.929	E94\n"
    "0	10	3	51	1.375	114.685	E94\n"
    "0	10	3	55	1.286	109.590	E94\n",
    "0	10	3	56	1.200	113.168	E94\n"
    "1	10	3	63	1.075	114.623	E94\n"
    "1	10	3	64	1.098	113.233	E94\n"
    "1	10	3	78	1.182	109.543	E94\n"
    "0	11	3	40	1.296	113.244	E94\n"
    "0	11	3	75	0.850	120.964	E94\n"
    "0	12	3	40	1.095	115.284	E94\n"
    "1	12	3	63	0.965	117.217	E94\n"
    "0	12	3	74	1.110	116.502	E94\n"
    "0	15	3	15	1.109	115.620	E94\n"
    "0	15	3	16	0.981	124.329	E94\n"
    "0	15	3	17	1.191	110.607	E94\n"
    "0	15	3	18	1.061	118.034	E94\n"
    "4	15	3	20	1.345	91.041	E94\n"
    "1	15	3	30	1.026	113.753	E94\n"
    "1	15	3	37	1.037	113.305	E94\n"
    "0	15	3	40	1.066	117.388	E94\n"
    "1	15	3	57	0.896	122.260	E94\n"
    "0	15	3	67	1.407	102.583	E94\n"
    "0	15	3	74	1.076	119.117	E94\n"
    "1	16	3	30	0.991	117.695	E94\n"
    "0	16	3	35	1.030	130.230	E94\n"
    "1	16	3	37	0.934	121.415	E94\n"
    "1	16	3	39	1.004	123.196	E94\n"
    "0	16	3	62	0.963	126.347	E94\n"
    "1	16	3	63	1.006	117.454	E94\n"
    "1	16	3	64	1.064	114.110	E94\n"
    "0	17	3	17	0.939	123.528	E94\n"
    "1	18	3	37	0.948	118.188	E94\n"
    "4	20	3	20	1.495	94.800	C94\n"
    "4	20	3	22	1.286	89.459	E94\n"
    "7	20	3	37	1.282	89.733	E94\n"
    "4	20	3	43	1.384	90.526	E94\n"
    "0	22	3	22	0.932	115.334	E94\n"
    "4	22	3	22	1.496	83.915	E94\n"
    "1	22	3	37	0.940	114.995	E94\n"
    "1	25	3	37	0.677	123.404	E94\n"
    "0	25	3	67	0.661	131.520	E94\n"
    "2	37	3	37	0.933	115.566	E94\n"
    "1	37	3	40	0.987	118.790	E94\n"
    "1	37	3	41	0.864	119.565	E94\n"
    "1	37	3	43	1.125	110.383	X94\n"
    "1	37	3	45	1.120	110.268	E94\n"
    "1	37	3	54	1.033	117.645	E94\n"
    "1	37	3	62	1.085	114.132	E94\n"
    "2	37	3	63	0.934	116.163	E94\n"
    "2	37	3	64	0.955	114.701	E94\n"
    "1	37	3	67	1.084	114.460	E94\n"
    "2	39	3	39	1.231	112.582	E94\n"
    "0	40	3	40	1.146	117.002	C94\n"
    "1	40	3	63	0.888	126.089	E94\n",
    "1	40	3	64	1.145	110.889	E94\n"
    "0	40	3	75	0.790	122.163	E94\n"
    "0	45	3	53	1.382	105.849	E94\n"
    "1	55	3	64	1.267	104.747	E94\n"
    "2	64	3	64	0.989	113.280	E94\n"
    "0	0	4	0	0.000	180.000	0:*-4-*	MMFF	DEF\n"
    "1	0	4	0	0.000	180.000	1:*-4-*	MMFF	DEF\n"
    "0	1	4	4	0.423	180.000	E94\n"
    "0	1	4	42	0.463	180.000	E94\n"
    "0	2	4	2	0.442	180.000	E94\n"
    "1	2	4	4	0.432	180.000	E94\n"
    "0	2	4	30	0.444	180.000	E94\n"
    "1	2	4	42	0.474	180.000	E94\n"
    "1	3	4	4	0.427	180.000	E94\n"
    "1	3	4	42	0.469	180.000	E94\n"
    "0	4	4	5	0.281	180.000	E94\n"
    "0	4	4	6	0.551	180.000	E94\n"
    "0	4	4	10	0.486	180.000	E94\n"
    "1	4	4	37	0.430	180.000	E94\n"
    "0	7	4	9	0.648	180.000	E94\n"
    "1	9	4	42	0.537	180.000	E94\n"
    "0	15	4	42	0.487	180.000	E94\n"
    "0	20	4	42	0.469	180.000	E94\n"
    "0	22	4	42	0.472	180.000	E94\n"
    "1	37	4	42	0.472	180.000	E94\n"
    "0	42	4	43	0.541	180.000	E94\n"
    "1	42	4	63	0.474	180.000	E94\n"
    "1	42	4	64	0.473	180.000	E94\n"
    "0	0	6	0	0.000	110.400	0:*-6-*	MMFF	DEF\n"
    "3	0	6	0	0.000	57.900	3::*-6-*	MMFF	DEF\n"
    "4	0	6	0	0.000	90.200	4:*-6-*	MMFF	DEF\n"
    "0	1	6	1	1.197	106.926	C94\n"
    "0	1	6	2	0.967	103.614	C94\n"
    "0	1	6	3	0.923	108.055	C94\n"
    "0	1	6	6	1.884	103.905	E94\n"
    "0	1	6	8	1.629	105.422	E94\n"
    "0	1	6	9	1.628	106.496	E94\n"
    "0	1	6	10	1.656	105.317	E94\n"
    "0	1	6	15	1.480	111.230	E94\n"
    "0	1	6	17	1.493	111.951	E94\n"
    "0	1	6	18	1.370	116.346	E94\n"
    "0	1	6	19	1.093	114.943	E94\n"
    "0	1	6	20	1.316	112.833	E94\n"
    "0	1	6	21	0.793	106.503	C94\n"
    "0	1	6	22	1.391	109.759	E94\n"
    "0	1	6	25	1.095	115.581	X94\n"
    "0	1	6	26	1.170	112.081	E94\n"
    "0	1	6	37	1.075	102.846	C94\n"
    "0	1	6	40	1.719	103.733	E94\n"
    "0	1	6	41	1.454	109.046	E94\n"
    "0	1	6	43	1.642	105.462	E94\n",
    "0	1	6	45	1.642	105.875	X94\n"
    "0	1	6	63	1.449	109.545	E94\n"
    "0	1	6	64	1.512	106.848	E94\n"
    "0	2	6	2	1.354	113.339	E94\n"
    "0	2	6	3	0.671	98.438	C94\n"
    "0	2	6	18	1.365	117.169	E94\n"
    "0	2	6	25	1.025	120.078	E94\n"
    "0	2	6	29	0.816	105.727	C94\n"
    "0	2	6	37	1.418	110.694	E94\n"
    "0	2	6	57	1.341	114.785	E94\n"
    "0	3	6	3	1.455	110.067	E94\n"
    "0	3	6	4	1.409	112.404	E94\n"
    "0	3	6	8	1.648	105.872	E94\n"
    "0	3	6	10	1.596	108.437	E94\n"
    "0	3	6	18	1.274	121.468	E94\n"
    "0	3	6	19	1.019	119.840	E94\n"
    "0	3	6	20	1.379	111.381	E94\n"
    "4	3	6	20	1.748	91.216	E94\n"
    "0	3	6	22	1.328	113.491	E94\n"
    "0	3	6	24	0.583	111.948	C94\n"
    "0	3	6	25	1.006	121.410	E94\n"
    "0	3	6	29	0.876	111.417	E94\n"
    "0	3	6	37	0.614	95.300	C94\n"
    "0	3	6	64	1.424	111.483	E94\n"
    "0	4	6	18	1.423	115.233	E94\n"
    "0	6	6	21	1.362	95.697	E94\n"
    "0	8	6	21	0.832	99.409	C94\n"
    "0	9	6	21	1.115	101.592	E94\n"
    "0	10	6	21	0.923	99.688	C94\n"
    "0	18	6	18	1.334	125.242	E94\n"
    "0	18	6	33	0.812	115.364	X94\n"
    "0	18	6	37	1.429	114.473	E94\n"
    "0	18	6	39	1.558	114.152	E94\n"
    "0	18	6	43	1.710	108.479	E94\n"
    "0	19	6	19	0.642	141.096	E94\n"
    "0	19	6	21	0.597	118.204	X94\n"
    "0	19	6	37	0.941	124.421	E94\n"
    "4	20	6	20	1.339	89.100	C94\n"
    "0	20	6	21	0.944	104.587	E94\n"
    "0	20	6	37	1.394	110.394	E94\n"
    "0	21	6	40	1.124	101.417	E94\n"
    "0	21	6	43	1.058	103.253	E94\n"
    "0	21	6	54	1.175	100.000	#E94\n"
    "0	21	6	55	1.139	101.000	#E94\n"
    "3	22	6	22	0.242	58.680	E94\n"
    "3	22	6	43	0.279	57.087	E94\n"
    "0	24	6	25	0.607	118.533	X94\n"
    "0	25	6	25	0.777	129.375	E94\n"
    "0	25	6	37	1.099	115.923	E94\n"
    "0	26	6	37	1.090	116.692	E94\n"
    "0	29	6	30	0.986	108.000	#E94\n",
    "0	29	6	37	0.726	105.409	C94\n"
    "0	29	6	64	0.923	108.922	E94\n"
    "0	37	6	37	1.462	108.967	E94\n"
    "0	37	6	58	1.607	108.274	E94\n"
    "0	0	8	0	0.000	110.400	0:*-8-*	MMFF	DEF\n"
    "3	0	8	0	0.000	58.500	3::*-8-*	MMFF	DEF\n"
    "4	0	8	0	0.000	95.000	4:*-8-*	MMFF	DEF\n"
    "0	1	8	1	1.090	107.018	C94\n"
    "0	1	8	6	1.297	102.829	C94\n"
    "0	1	8	8	1.347	105.708	E94\n"
    "0	1	8	9	1.182	114.240	E94\n"
    "0	1	8	10	1.307	108.079	E94\n"
    "0	1	8	15	1.085	118.283	E94\n"
    "0	1	8	17	1.096	117.478	E94\n"
    "0	1	8	19	0.779	122.759	E94\n"
    "0	1	8	20	1.221	105.873	E94\n"
    "0	1	8	22	1.147	109.200	E94\n"
    "0	1	8	23	0.763	109.062	C94\n"
    "0	1	8	25	0.865	117.482	E94\n"
    "0	1	8	26	0.926	112.630	E94\n"
    "0	1	8	40	1.363	105.609	E94\n"
    "0	1	8	45	1.266	110.149	E94\n"
    "0	1	8	46	1.265	111.092	E94\n"
    "0	6	8	6	1.776	107.296	E94\n"
    "0	6	8	17	1.664	105.334	E94\n"
    "0	6	8	22	1.456	107.100	E94\n"
    "0	6	8	23	0.861	100.510	C94\n"
    "3	8	8	8	0.230	60.000	E94\n"
    "0	8	8	23	0.792	108.917	E94\n"
    "0	8	8	25	1.068	110.595	E94\n"
    "0	8	8	26	1.047	110.816	E94\n"
    "0	9	8	23	0.832	108.864	E94\n"
    "4	10	8	20	1.805	84.690	E94\n"
    "0	10	8	23	0.846	106.788	E94\n"
    "0	12	8	22	1.227	107.439	E94\n"
    "0	15	8	19	0.845	125.674	E94\n"
    "4	17	8	17	1.198	110.056	E94\n"
    "0	17	8	23	0.647	116.842	E94\n"
    "0	19	8	23	0.542	112.000	#E94\n"
    "4	20	8	20	1.103	90.370	C94\n"
    "0	20	8	23	0.684	113.359	C94\n"
    "3	22	8	22	0.209	57.087	E94\n"
    "0	22	8	23	0.697	110.033	E94\n"
    "0	22	8	25	0.896	115.361	E94\n"
    "0	23	8	23	0.595	105.998	C94\n"
    "0	23	8	25	0.510	117.000	#E94\n"
    "0	23	8	26	0.553	110.959	E94\n"
    "0	23	8	34	0.808	109.000	#E94\n"
    "0	23	8	39	0.757	111.820	E94\n"
    "0	23	8	40	0.819	108.120	E94\n"
    "0	23	8	43	0.857	106.222	E94\n",
    "0	23	8	55	0.868	106.000	#E94\n"
    "0	23	8	56	0.876	105.092	E94\n"
    "0	0	9	0	0.000	111.500	0:*-9-*	MMFF	DEF\n"
    "1	0	9	0	0.000	109.100	1:*-9-*	MMFF	DEF\n"
    "0	1	9	3	0.878	106.409	C94\n"
    "0	1	9	9	1.306	110.005	E94\n"
    "0	1	9	53	1.216	113.995	X94\n"
    "0	1	9	67	1.391	106.413	E94\n"
    "1	2	9	3	1.242	109.856	E94\n"
    "1	2	9	9	1.306	112.528	E94\n"
    "1	3	9	3	1.204	111.488	E94\n"
    "1	3	9	4	1.194	113.272	E94\n"
    "0	3	9	6	1.579	106.872	E94\n"
    "0	3	9	8	1.386	108.822	E94\n"
    "1	3	9	9	1.390	108.355	E94\n"
    "0	3	9	10	1.365	109.548	E94\n"
    "0	3	9	12	1.373	103.303	E94\n"
    "0	3	9	15	1.265	110.780	E94\n"
    "0	3	9	18	1.205	114.743	E94\n"
    "0	3	9	20	1.198	109.751	E94\n"
    "0	3	9	25	0.873	119.927	E94\n"
    "0	3	9	27	0.818	108.779	C94\n"
    "0	3	9	34	1.355	108.199	E94\n"
    "0	3	9	35	1.511	109.907	E94\n"
    "1	3	9	37	1.185	111.663	E94\n"
    "1	3	9	39	1.396	108.538	E94\n"
    "0	3	9	40	1.365	109.440	E94\n"
    "0	3	9	41	1.169	112.551	E94\n"
    "0	3	9	45	1.369	109.796	E94\n"
    "1	3	9	53	1.351	110.578	E94\n"
    "1	3	9	54	1.643	98.943	E94\n"
    "0	3	9	55	1.431	106.195	E94\n"
    "0	3	9	56	1.375	109.289	E94\n"
    "1	3	9	57	1.125	115.780	E94\n"
    "1	3	9	63	1.247	109.989	E94\n"
    "1	3	9	64	1.302	106.461	E94\n"
    "1	3	9	78	1.323	106.641	E94\n"
    "1	3	9	81	1.567	101.581	E94\n"
    "0	4	9	19	0.456	161.741	E94\n"
    "1	4	9	67	1.402	108.868	E94\n"
    "0	6	9	67	1.794	105.043	E94\n"
    "0	9	9	10	1.518	109.154	E94\n"
    "1	9	9	37	1.397	108.014	E94\n"
    "0	9	9	40	1.594	106.413	E94\n"
    "0	9	9	62	1.390	114.417	E94\n"
    "1	9	9	63	1.320	112.325	E94\n"
    "1	9	9	64	1.352	109.711	E94\n"
    "1	37	9	53	1.343	110.162	E94\n"
    "1	37	9	67	1.296	111.871	E94\n"
    "0	40	9	67	1.538	108.056	E94\n"
    "1	53	9	64	1.318	111.149	E94\n",
    "0	0	10	0	0.000	117.500	0:*-10-*	MMFF	DEF\n"
    "3	0	10	0	0.000	58.900	3::*-10-*	MMFF	DEF\n"
    "4	0	10	0	0.000	92.900	4:*-10-*	MMFF	DEF\n"
    "0	1	10	1	1.117	117.909	C94\n"
    "0	1	10	2	1.004	118.916	E94\n"
    "0	1	10	3	0.821	119.600	C94\n"
    "0	1	10	6	1.179	108.865	C94\n"
    "0	1	10	8	1.137	116.189	E94\n"
    "0	1	10	9	1.132	117.005	E94\n"
    "0	1	10	10	1.247	111.009	E94\n"
    "0	1	10	17	1.014	122.388	E94\n"
    "0	1	10	20	0.960	119.679	E94\n"
    "0	1	10	25	0.745	125.390	E94\n"
    "0	1	10	28	0.552	120.066	C94\n"
    "0	1	10	37	1.038	116.332	E94\n"
    "0	1	10	39	1.060	120.838	E94\n"
    "0	1	10	40	1.194	113.314	E94\n"
    "0	1	10	41	1.031	118.033	E94\n"
    "0	1	10	45	1.268	109.599	E94\n"
    "0	1	10	63	0.949	122.185	E94\n"
    "0	1	10	64	0.960	121.315	E94\n"
    "0	2	10	2	1.146	112.878	E94\n"
    "0	2	10	3	1.000	120.703	E94\n"
    "0	2	10	6	1.405	111.609	E94\n"
    "0	2	10	20	1.132	111.544	E94\n"
    "0	2	10	28	0.638	118.553	E94\n"
    "0	2	10	37	0.977	121.506	E94\n"
    "0	3	10	3	0.709	120.274	C94\n"
    "0	3	10	4	0.864	130.236	E94\n"
    "0	3	10	6	0.960	110.133	C94\n"
    "0	3	10	8	1.168	116.075	E94\n"
    "4	3	10	8	1.527	93.608	E94\n"
    "0	3	10	9	1.174	116.443	E94\n"
    "0	3	10	10	1.184	115.377	E94\n"
    "0	3	10	13	0.998	118.867	E94\n"
    "0	3	10	14	0.871	124.162	E94\n"
    "0	3	10	15	1.076	118.969	E94\n"
    "0	3	10	17	1.132	116.612	E94\n"
    "0	3	10	20	0.936	122.540	E94\n"
    "4	3	10	20	1.371	93.349	E94\n"
    "0	3	10	22	0.975	120.929	E94\n"
    "0	3	10	25	0.794	122.157	E94\n"
    "0	3	10	26	0.848	117.912	E94\n"
    "0	3	10	28	0.575	120.277	C94\n"
    "0	3	10	34	1.251	112.201	E94\n"
    "0	3	10	35	1.395	112.633	E94\n"
    "0	3	10	37	1.023	118.596	E94\n"
    "0	3	10	40	1.216	113.680	E94\n"
    "0	3	10	41	1.098	115.913	E94\n"
    "0	3	10	45	1.212	113.447	E94\n"
    "0	3	10	63	1.091	115.381	E94\n",
    "0	3	10	64	1.048	117.574	E94\n"
    "0	4	10	20	0.816	131.702	E94\n"
    "0	6	10	28	0.829	113.214	E94\n"
    "0	6	10	37	1.393	111.476	E94\n"
    "0	8	10	28	0.703	117.160	E94\n"
    "0	8	10	37	1.167	115.599	E94\n"
    "0	9	10	26	0.847	123.206	E94\n"
    "0	9	10	28	0.751	114.501	E94\n"
    "0	9	10	37	1.222	113.553	E94\n"
    "0	9	10	39	1.310	115.309	E94\n"
    "0	10	10	28	0.735	114.715	E94\n"
    "0	10	10	41	1.237	113.743	E94\n"
    "0	15	10	28	0.614	119.033	E94\n"
    "4	20	10	20	1.381	91.694	E94\n"
    "0	20	10	28	0.555	123.394	E94\n"
    "0	20	10	37	1.006	117.703	E94\n"
    "3	22	10	22	0.202	58.894	E94\n"
    "0	22	10	28	0.605	119.583	E94\n"
    "0	25	10	28	0.447	122.785	E94\n"
    "0	28	10	28	0.435	115.630	C94\n"
    "0	28	10	34	0.757	113.000	#E94\n"
    "0	28	10	35	0.836	114.000	#E94\n"
    "0	28	10	37	0.628	118.227	E94\n"
    "0	28	10	40	0.754	113.000	#E94\n"
    "0	28	10	41	0.560	128.067	E94\n"
    "0	28	10	63	0.640	118.099	E94\n"
    "0	28	10	64	0.643	117.575	E94\n"
    "0	37	10	40	1.232	112.412	E94\n"
    "0	0	15	0	0.000	97.900	0:*-15-*	MMFF	DEF\n"
    "4	0	15	0	0.000	80.200	4:*-15-*	MMFF	DEF\n"
    "0	1	15	1	1.654	97.335	C94\n"
    "0	1	15	2	1.321	97.853	E94\n"
    "0	1	15	3	1.325	97.326	E94\n"
    "0	1	15	4	1.344	97.370	E94\n"
    "0	1	15	9	1.725	89.814	E94\n"
    "0	1	15	15	1.377	100.316	C94\n"
    "0	1	15	18	1.309	101.641	E94\n"
    "0	1	15	19	1.007	102.069	E94\n"
    "0	1	15	20	1.366	94.913	E94\n"
    "0	1	15	22	1.268	99.768	E94\n"
    "0	1	15	25	0.967	104.732	E94\n"
    "0	1	15	30	1.379	95.613	E94\n"
    "0	1	15	37	1.439	97.111	C94\n"
    "0	1	15	40	1.555	94.643	E94\n"
    "0	1	15	57	1.301	98.686	E94\n"
    "0	1	15	63	1.304	98.330	E94\n"
    "0	1	15	64	1.306	98.066	E94\n"
    "0	1	15	71	0.931	96.494	C94\n"
    "0	2	15	2	1.434	95.108	E94\n"
    "0	2	15	3	1.318	98.813	E94\n"
    "0	2	15	4	1.426	95.780	E94\n",
    "0	2	15	15	1.457	97.789	E94\n"
    "0	2	15	37	1.362	96.942	E94\n"
    "0	2	15	43	1.709	90.872	E94\n"
    "0	3	15	3	1.402	95.424	E94\n"
    "0	3	15	6	1.804	94.075	E94\n"
    "0	3	15	15	1.403	99.399	E94\n"
    "4	3	15	20	1.666	79.842	E94\n"
    "0	3	15	37	1.308	98.541	E94\n"
    "0	3	15	63	1.390	96.051	E94\n"
    "0	3	15	71	0.830	97.000	#E94\n"
    "0	6	15	37	1.679	97.231	E94\n"
    "0	8	15	8	1.444	105.143	E94\n"
    "0	8	15	37	1.446	98.976	E94\n"
    "0	9	15	9	1.626	98.524	E94\n"
    "0	9	15	64	1.504	97.105	E94\n"
    "0	10	15	15	1.415	103.715	E94\n"
    "0	12	15	37	1.428	97.534	E94\n"
    "0	15	15	15	1.413	104.893	E94\n"
    "0	15	15	18	1.563	99.173	E94\n"
    "0	15	15	37	1.361	100.790	E94\n"
    "0	15	15	64	1.332	102.040	E94\n"
    "0	15	15	71	0.787	99.239	C94\n"
    "4	20	15	30	1.978	73.428	E94\n"
    "0	20	15	37	1.361	95.589	E94\n"
    "0	25	15	25	0.947	99.505	E94\n"
    "4	25	15	25	1.030	87.982	E94\n"
    "0	25	15	26	1.002	96.851	E94\n"
    "0	25	15	37	1.172	95.428	E94\n"
    "0	26	15	37	1.144	96.710	E94\n"
    "4	30	15	30	1.732	79.546	E94\n"
    "0	37	15	37	1.295	98.802	E94\n"
    "0	37	15	63	1.379	96.197	E94\n"
    "0	37	15	64	1.286	99.423	E94\n"
    "0	37	15	71	0.813	96.222	C94\n"
    "0	71	15	71	0.734	93.377	C94\n"
    "0	0	17	0	0.000	99.400	0:*-17-*	MMFF	DEF\n"
    "4	0	17	0	0.000	78.400	4:*-17-*	MMFF	DEF\n"
    "0	1	17	1	1.415	93.266	X94\n"
    "0	1	17	2	1.387	94.732	E94\n"
    "0	1	17	3	1.430	92.852	E94\n"
    "0	1	17	6	1.863	92.132	E94\n"
    "0	1	17	7	1.408	107.104	X94\n"
    "0	1	17	8	1.661	91.498	E94\n"
    "0	1	17	10	1.547	94.839	E94\n"
    "0	1	17	20	1.453	91.368	E94\n"
    "0	1	17	22	1.423	92.591	E94\n"
    "0	1	17	37	1.376	94.911	E94\n"
    "0	2	17	2	1.313	97.901	E94\n"
    "0	2	17	7	1.478	105.412	E94\n"
    "0	2	17	43	1.207	108.882	E94\n"
    "0	3	17	7	1.513	103.431	E94\n",
    "0	6	17	6	2.164	97.766	E94\n"
    "0	6	17	7	1.850	107.431	E94\n"
    "0	7	17	8	1.438	113.808	E94\n"
    "0	7	17	10	1.525	110.549	E94\n"
    "0	7	17	20	1.442	104.737	E94\n"
    "0	7	17	22	1.449	104.928	E94\n"
    "0	7	17	37	1.500	104.313	E94\n"
    "4	8	17	20	1.891	78.354	E94\n"
    "0	8	17	37	1.687	91.169	E94\n"
    "0	37	17	37	1.487	91.633	E94\n"
    "0	0	18	0	0.000	104.600	0:*-18-*	MMFF	DEF\n"
    "4	0	18	0	0.000	80.300	4:*-18-*	MMFF	DEF\n"
    "0	1	18	1	1.230	101.166	X94\n"
    "0	1	18	2	1.264	100.420	E94\n"
    "0	1	18	3	1.242	100.883	E94\n"
    "0	1	18	6	1.744	95.671	X94\n"
    "0	1	18	9	1.438	99.465	E94\n"
    "0	1	18	20	1.224	101.315	E94\n"
    "0	1	18	22	1.207	101.417	E94\n"
    "0	1	18	32	1.446	107.066	X94\n"
    "0	1	18	37	1.234	101.070	E94\n"
    "0	1	18	43	1.449	98.014	X94\n"
    "0	1	18	48	1.277	106.586	X94\n"
    "0	1	18	62	1.374	102.402	X94\n"
    "0	2	18	2	1.254	101.492	E94\n"
    "0	2	18	6	1.664	98.668	E94\n"
    "0	2	18	9	1.539	96.849	E94\n"
    "0	2	18	32	1.422	108.979	E94\n"
    "0	2	18	37	1.263	100.489	E94\n"
    "0	2	18	48	1.083	116.668	E94\n"
    "0	3	18	9	1.418	100.361	E94\n"
    "0	3	18	32	1.557	103.453	E94\n"
    "0	3	18	43	1.350	101.747	E94\n"
    "0	6	18	6	1.922	103.052	X94\n"
    "0	6	18	9	1.916	97.446	E94\n"
    "0	6	18	32	1.837	108.063	X94\n"
    "0	6	18	37	1.528	102.229	E94\n"
    "0	6	18	43	1.644	103.815	E94\n"
    "0	9	18	12	1.464	101.180	E94\n"
    "0	9	18	32	1.583	109.945	E94\n"
    "0	9	18	37	1.358	102.378	E94\n"
    "0	9	18	43	1.323	109.227	E94\n"
    "0	12	18	32	1.584	103.959	E94\n"
    "0	12	18	37	1.376	98.976	E94\n"
    "0	15	18	32	1.497	107.170	E94\n"
    "0	15	18	37	1.324	101.399	E94\n"
    "0	20	18	32	1.383	109.292	E94\n"
    "0	20	18	37	1.108	106.508	E94\n"
    "4	20	18	43	1.831	80.297	E94\n"
    "0	22	18	32	1.465	105.247	E94\n"
    "0	32	18	32	1.569	120.924	X94\n",
    "0	32	18	37	1.497	105.280	X94\n"
    "0	32	18	39	1.804	101.600	X94\n"
    "0	32	18	43	1.569	108.548	X94\n"
    "0	32	18	48	1.229	126.841	X94\n"
    "0	32	18	55	1.509	112.548	E94\n"
    "0	32	18	58	1.592	106.139	E94\n"
    "0	32	18	62	1.326	121.426	X94\n"
    "0	32	18	63	1.571	103.212	E94\n"
    "0	32	18	64	1.634	101.771	E94\n"
    "0	32	18	80	1.400	110.401	E94\n"
    "0	37	18	37	1.157	104.380	E94\n"
    "0	37	18	39	1.404	99.854	X94\n"
    "0	37	18	43	1.416	99.200	X94\n"
    "0	37	18	48	1.330	104.466	E94\n"
    "0	37	18	55	1.397	100.926	E94\n"
    "0	37	18	62	1.178	110.665	E94\n"
    "0	37	18	63	1.202	102.735	E94\n"
    "0	43	18	43	1.545	99.905	X94\n"
    "0	43	18	64	1.285	104.868	E94\n"
    "0	0	19	0	0.000	108.700	0:*-19-*	MMFF	DEF\n"
    "4	0	19	0	0.000	89.900	4:*-19-*	MMFF	DEF\n"
    "0	1	19	1	0.616	113.339	E94\n"
    "0	1	19	5	0.390	110.795	X94\n"
    "0	1	19	6	0.777	113.958	X94\n"
    "0	1	19	8	0.716	111.521	E94\n"
    "0	1	19	9	0.779	106.380	E94\n"
    "0	1	19	12	0.729	108.947	X94\n"
    "0	1	19	20	0.656	108.828	E94\n"
    "0	1	19	40	0.754	108.858	E94\n"
    "0	1	19	63	0.699	106.924	E94\n"
    "0	1	19	75	0.530	111.633	E94\n"
    "0	2	19	12	0.819	102.981	E94\n"
    "0	5	19	5	0.258	108.699	X94\n"
    "0	5	19	6	0.520	109.677	X94\n"
    "0	5	19	8	0.461	109.070	E94\n"
    "0	5	19	12	0.446	106.756	X94\n"
    "0	6	19	6	1.051	111.280	E94\n"
    "0	6	19	12	0.968	106.022	E94\n"
    "0	6	19	37	0.870	108.096	E94\n"
    "0	8	19	8	0.862	108.099	E94\n"
    "0	8	19	12	0.786	110.683	E94\n"
    "0	12	19	12	0.879	104.597	E94\n"
    "0	15	19	15	0.816	108.681	E94\n"
    "4	20	19	20	0.802	89.931	E94\n"
    "0	37	19	37	0.726	105.045	E94\n"
    "0	0	20	0	0.000	113.200	0:*-20-*	MMFF	DEF\n"
    "4	0	20	0	0.000	88.800	4:*-20-*	MMFF	DEF\n"
    "0	1	20	1	0.943	113.131	E94\n"
    "0	1	20	3	0.906	114.940	E94\n"
    "0	1	20	5	0.417	114.057	C94\n"
    "0	1	20	6	1.231	110.677	E94\n",
    "0	1	20	8	1.080	111.090	E94\n"
    "0	1	20	10	1.100	110.057	E94\n"
    "0	1	20	11	1.173	110.993	E94\n"
    "0	1	20	12	0.976	114.773	E94\n"
    "0	1	20	15	1.035	111.226	E94\n"
    "0	1	20	18	0.978	115.383	E94\n"
    "0	1	20	20	0.502	113.313	C94\n"
    "0	1	20	22	0.915	115.201	E94\n"
    "0	1	20	25	0.744	116.096	E94\n"
    "0	1	20	26	0.721	117.611	E94\n"
    "0	1	20	30	0.908	115.220	E94\n"
    "0	1	20	34	1.090	110.505	E94\n"
    "0	1	20	37	0.947	112.650	E94\n"
    "0	1	20	41	0.973	111.787	E94\n"
    "0	1	20	43	1.087	110.187	E94\n"
    "0	1	20	45	1.132	108.074	E94\n"
    "0	2	20	3	0.982	111.060	E94\n"
    "0	2	20	5	0.596	113.035	E94\n"
    "0	2	20	6	1.139	115.851	E94\n"
    "0	2	20	12	0.951	116.750	E94\n"
    "0	2	20	20	0.931	114.138	E94\n"
    "0	3	20	3	0.982	109.919	E94\n"
    "0	3	20	5	0.624	112.989	C94\n"
    "0	3	20	6	1.157	113.611	E94\n"
    "4	3	20	8	1.473	87.271	E94\n"
    "0	3	20	10	1.016	113.988	E94\n"
    "0	3	20	11	1.184	109.849	E94\n"
    "0	3	20	12	0.969	114.891	E94\n"
    "0	3	20	13	1.008	110.951	E94\n"
    "0	3	20	20	0.849	118.273	E94\n"
    "4	3	20	20	1.524	88.961	C94\n"
    "0	3	20	34	1.137	107.667	E94\n"
    "4	3	20	37	1.382	85.619	E94\n"
    "0	3	20	43	0.960	116.707	E94\n"
    "0	4	20	5	0.584	115.078	E94\n"
    "0	4	20	20	0.920	115.312	E94\n"
    "0	5	20	5	0.439	109.107	C94\n"
    "0	5	20	6	0.818	111.352	C94\n"
    "0	5	20	8	0.728	114.011	C94\n"
    "0	5	20	9	0.657	112.826	E94\n"
    "0	5	20	10	0.663	112.010	E94\n"
    "0	5	20	12	0.339	114.117	C94\n"
    "0	5	20	15	0.562	114.339	E94\n"
    "0	5	20	17	0.561	113.000	#E94\n"
    "0	5	20	18	0.605	111.570	E94\n"
    "0	5	20	20	0.564	113.940	C94\n"
    "0	5	20	26	0.472	109.722	E94\n"
    "0	5	20	30	0.688	116.038	C94\n"
    "0	5	20	34	0.661	112.000	#E94\n"
    "0	5	20	37	0.552	115.670	E94\n"
    "0	5	20	40	0.682	111.331	E94\n",
    "0	5	20	43	0.655	111.686	E94\n"
    "0	6	20	6	1.443	114.408	E94\n"
    "0	6	20	10	1.225	116.666	E94\n"
    "0	6	20	13	1.162	114.868	E94\n"
    "0	6	20	20	1.109	116.117	E94\n"
    "4	6	20	20	1.433	93.413	C94\n"
    "0	6	20	22	1.106	117.205	E94\n"
    "0	6	20	30	1.144	114.705	E94\n"
    "4	6	20	30	1.658	87.873	E94\n"
    "0	8	20	20	1.185	105.606	E94\n"
    "4	8	20	20	1.486	91.244	C94\n"
    "0	8	20	26	0.874	111.782	E94\n"
    "0	9	20	20	1.103	109.640	E94\n"
    "0	10	20	15	1.170	109.525	E94\n"
    "0	10	20	17	1.127	110.564	E94\n"
    "0	10	20	18	1.404	100.845	E94\n"
    "0	10	20	20	1.032	113.170	E94\n"
    "4	10	20	20	1.468	87.497	E94\n"
    "4	10	20	30	1.507	86.657	E94\n"
    "0	10	20	37	0.963	117.360	E94\n"
    "0	11	20	11	1.504	108.020	E94\n"
    "0	11	20	17	1.221	109.460	E94\n"
    "0	11	20	20	1.051	116.673	E94\n"
    "0	11	20	30	0.997	120.309	E94\n"
    "0	12	20	12	1.020	117.603	E94\n"
    "0	12	20	19	0.973	105.821	E94\n"
    "0	12	20	20	0.866	118.108	C94\n"
    "0	12	20	30	0.887	120.399	E94\n"
    "0	13	20	13	1.077	113.361	E94\n"
    "0	13	20	20	0.938	115.037	E94\n"
    "0	14	20	20	0.837	112.888	E94\n"
    "0	15	20	15	1.094	114.048	E94\n"
    "0	15	20	20	1.058	109.793	E94\n"
    "4	15	20	20	1.324	90.483	E94\n"
    "0	15	20	30	0.960	115.468	E94\n"
    "4	15	20	30	1.447	86.726	E94\n"
    "4	17	20	17	1.309	94.977	E94\n"
    "0	17	20	20	0.930	116.108	E94\n"
    "0	18	20	20	1.007	113.480	E94\n"
    "4	18	20	20	1.355	90.185	E94\n"
    "0	18	20	41	1.241	102.656	E94\n"
    "0	19	20	19	0.567	122.298	E94\n"
    "4	19	20	19	0.921	88.477	E94\n"
    "0	20	20	20	1.008	108.644	E94\n"
    "4	20	20	20	1.149	90.294	C94\n"
    "0	20	20	22	0.840	119.817	E94\n"
    "4	20	20	22	1.364	86.669	E94\n"
    "4	20	20	25	1.181	84.818	E94\n"
    "0	20	20	30	0.994	109.745	E94\n"
    "4	20	20	30	1.399	85.303	C94\n"
    "0	20	20	34	1.069	111.143	E94\n",
    "4	20	20	34	1.382	90.128	E94\n"
    "0	20	20	37	0.833	119.709	E94\n"
    "4	20	20	37	1.346	86.810	E94\n"
    "0	20	20	40	1.097	110.254	E94\n"
    "0	20	20	41	0.922	114.408	E94\n"
    "0	20	20	43	0.964	116.540	E94\n"
    "4	20	20	43	1.290	92.879	E94\n"
    "0	20	20	45	1.083	110.090	E94\n"
    "0	22	20	22	0.866	118.829	E94\n"
    "4	22	20	22	1.649	79.399	E94\n"
    "4	26	20	26	0.789	96.811	E94\n"
    "0	26	20	34	0.843	113.805	E94\n"
    "0	34	20	41	1.070	111.943	E94\n"
    "0	37	20	43	0.954	117.365	E94\n"
    "0	0	22	0	0.000	116.100	0:*-22-*	MMFF	DEF\n"
    "3	0	22	0	0.000	59.400	3::*-22-*	MMFF	DEF\n"
    "4	0	22	0	0.000	91.600	4:*-22-*	MMFF	DEF\n"
    "0	1	22	1	0.903	116.483	E94\n"
    "0	1	22	2	0.884	118.360	E94\n"
    "0	1	22	3	0.836	121.424	E94\n"
    "0	1	22	4	0.900	117.720	E94\n"
    "0	1	22	5	0.604	111.788	E94\n"
    "0	1	22	6	1.179	113.545	E94\n"
    "0	1	22	8	0.973	117.469	E94\n"
    "0	1	22	17	1.070	109.087	E94\n"
    "0	1	22	18	1.097	108.265	E94\n"
    "0	1	22	22	0.871	118.246	E94\n"
    "0	1	22	37	0.882	118.041	E94\n"
    "0	1	22	43	1.014	114.899	E94\n"
    "3	2	22	2	0.263	48.820	E94\n"
    "0	2	22	3	0.956	114.147	E94\n"
    "0	2	22	4	0.784	126.957	E94\n"
    "0	2	22	5	0.573	115.869	E94\n"
    "0	2	22	6	1.012	123.319	E94\n"
    "0	2	22	22	0.880	118.260	E94\n"
    "3	2	22	22	0.166	60.845	E94\n"
    "0	2	22	45	1.009	116.146	E94\n"
    "0	3	22	3	0.819	122.977	E94\n"
    "0	3	22	4	0.876	119.718	E94\n"
    "0	3	22	5	0.559	116.738	E94\n"
    "0	3	22	6	1.184	113.646	E94\n"
    "0	3	22	8	1.072	112.261	E94\n"
    "0	3	22	10	0.987	117.750	E94\n"
    "0	3	22	12	0.930	118.047	E94\n"
    "4	3	22	20	1.267	90.869	E94\n"
    "0	3	22	22	0.861	119.252	E94\n"
    "4	3	22	22	1.196	93.287	E94\n"
    "4	3	22	30	1.301	89.217	E94\n"
    "0	3	22	37	0.852	120.464	E94\n"
    "0	3	22	40	1.033	114.288	E94\n"
    "0	3	22	43	1.124	109.441	E94\n",
    "0	3	22	45	1.117	110.033	E94\n"
    "0	4	22	5	0.560	118.000	#E94\n"
    "0	4	22	6	1.200	113.650	E94\n"
    "0	4	22	8	0.966	119.034	E94\n"
    "0	4	22	15	0.931	120.455	E94\n"
    "0	4	22	22	0.877	118.890	E94\n"
    "0	4	22	45	1.089	112.227	E94\n"
    "0	5	22	5	0.242	114.938	C94\n"
    "0	5	22	6	0.683	117.836	E94\n"
    "0	5	22	8	0.621	115.758	E94\n"
    "0	5	22	10	0.658	113.806	E94\n"
    "0	5	22	11	0.776	108.296	X94\n"
    "0	5	22	12	0.620	109.865	X94\n"
    "0	5	22	20	0.623	110.000	#E94\n"
    "0	5	22	22	0.583	117.875	C94\n"
    "0	5	22	37	0.532	119.438	E94\n"
    "0	5	22	40	0.653	112.855	E94\n"
    "0	5	22	41	0.519	122.000	#E94\n"
    "0	5	22	43	0.658	112.128	E94\n"
    "0	5	22	45	0.665	112.000	#E94\n"
    "0	6	22	12	1.136	118.409	E94\n"
    "0	6	22	17	1.328	108.583	E94\n"
    "0	6	22	18	1.381	107.009	E94\n"
    "0	6	22	22	1.124	115.942	E94\n"
    "3	6	22	22	0.205	60.711	E94\n"
    "0	6	22	37	1.093	118.170	E94\n"
    "3	6	22	43	0.179	68.138	E94\n"
    "0	6	22	45	1.422	108.368	E94\n"
    "0	8	22	22	0.925	120.144	E94\n"
    "3	8	22	22	0.176	61.507	E94\n"
    "0	10	22	22	0.916	121.411	E94\n"
    "3	10	22	22	0.184	60.603	E94\n"
    "0	11	22	11	1.610	102.859	E94\n"
    "0	11	22	22	1.062	116.086	X94\n"
    "0	12	22	12	1.067	114.988	E94\n"
    "0	12	22	22	0.925	117.971	X94\n"
    "0	13	22	13	1.085	113.473	E94\n"
    "0	13	22	22	0.908	117.606	E94\n"
    "0	15	22	22	0.918	120.404	E94\n"
    "0	17	22	22	1.029	111.106	E94\n"
    "0	18	22	22	1.078	109.054	E94\n"
    "0	20	22	22	0.812	122.430	E94\n"
    "4	20	22	22	1.198	92.930	E94\n"
    "0	22	22	22	0.787	124.070	E94\n"
    "3	22	22	22	0.171	60.000	C94\n"
    "4	22	22	22	1.225	91.653	E94\n"
    "0	22	22	30	0.777	124.514	E94\n"
    "0	22	22	34	0.983	116.415	E94\n"
    "0	22	22	37	0.847	120.135	E94\n"
    "3	22	22	40	0.178	61.163	E94\n"
    "0	22	22	41	0.886	118.045	E94\n",
    "3	22	22	43	0.176	61.536	E94\n"
    "0	22	22	45	1.022	114.380	E94\n"
    "0	34	22	41	1.008	116.095	E94\n"
    "0	37	22	37	0.846	120.774	E94\n"
    "3	37	22	37	0.237	51.029	E94\n"
    "0	37	22	43	0.936	119.789	E94\n"
    "0	0	25	0	0.000	106.500	0:*-25-*	MMFF	DEF\n"
    "4	0	25	0	0.000	89.100	4:*-25-*	MMFF	DEF\n"
    "0	1	25	1	1.072	99.158	X94\n"
    "0	1	25	3	1.268	91.423	E94\n"
    "0	1	25	6	1.394	98.288	X94\n"
    "0	1	25	8	1.150	101.775	E94\n"
    "0	1	25	12	1.180	98.890	E94\n"
    "0	1	25	15	1.074	103.431	E94\n"
    "0	1	25	25	0.852	100.707	E94\n"
    "0	1	25	32	1.186	107.891	X94\n"
    "0	1	25	37	0.972	104.924	E94\n"
    "0	1	25	40	1.358	93.644	E94\n"
    "0	1	25	43	1.190	98.760	X94\n"
    "0	1	25	71	0.537	109.363	E94\n"
    "0	1	25	72	0.976	111.306	X94\n"
    "0	2	25	6	1.302	102.892	E94\n"
    "0	2	25	8	1.022	109.148	E94\n"
    "0	2	25	10	1.629	85.839	E94\n"
    "0	2	25	32	0.983	120.127	E94\n"
    "0	2	25	72	0.863	119.249	E94\n"
    "0	3	25	6	1.277	103.026	E94\n"
    "0	3	25	32	1.164	109.307	E94\n"
    "0	6	25	6	1.769	99.311	X94\n"
    "0	6	25	8	1.419	104.161	E94\n"
    "0	6	25	9	1.403	105.407	E94\n"
    "0	6	25	10	1.448	102.194	E94\n"
    "0	6	25	11	1.680	99.260	E94\n"
    "0	6	25	12	1.489	98.818	E94\n"
    "0	6	25	32	1.501	109.688	X94\n"
    "0	6	25	37	1.312	102.280	E94\n"
    "0	6	25	39	1.617	97.314	E94\n"
    "0	6	25	40	1.380	105.601	E94\n"
    "0	6	25	71	0.844	100.242	E94\n"
    "0	6	25	72	1.219	112.058	E94\n"
    "0	8	25	8	1.224	105.341	E94\n"
    "0	8	25	10	1.214	104.893	E94\n"
    "0	8	25	11	1.411	101.655	E94\n"
    "0	8	25	20	1.010	108.094	E94\n"
    "0	8	25	32	1.217	114.325	E94\n"
    "0	8	25	37	1.106	104.742	E94\n"
    "0	8	25	40	1.265	103.617	E94\n"
    "0	8	25	72	0.977	117.767	E94\n"
    "0	9	25	32	1.232	114.493	E94\n"
    "0	10	25	10	1.346	98.856	E94\n"
    "0	10	25	32	1.273	110.640	E94\n",
    "0	10	25	72	1.021	114.624	E94\n"
    "0	11	25	32	1.528	106.045	E94\n"
    "0	12	25	12	1.303	99.224	E94\n"
    "0	12	25	32	1.305	106.320	E94\n"
    "0	15	25	15	1.113	107.673	E94\n"
    "4	15	25	15	1.264	93.138	E94\n"
    "0	15	25	32	1.248	107.964	E94\n"
    "0	15	25	72	0.933	119.729	E94\n"
    "4	20	25	20	1.220	85.039	E94\n"
    "0	20	25	72	0.965	111.595	E94\n"
    "0	25	25	72	0.890	106.612	E94\n"
    "0	32	25	32	1.248	122.857	X94\n"
    "0	32	25	37	1.097	113.430	E94\n"
    "0	32	25	39	1.605	99.255	E94\n"
    "0	32	25	40	1.122	119.057	E94\n"
    "0	32	25	43	1.257	110.308	X94\n"
    "0	32	25	57	1.219	108.740	E94\n"
    "0	32	25	63	1.211	108.168	E94\n"
    "0	32	25	71	0.642	117.733	X94\n"
    "0	32	25	72	1.050	121.823	E94\n"
    "0	37	25	37	0.947	107.124	E94\n"
    "0	37	25	40	0.965	112.107	E94\n"
    "0	37	25	72	0.868	118.776	E94\n"
    "0	40	25	40	1.496	95.270	E94\n"
    "0	40	25	72	1.035	114.441	E94\n"
    "0	57	25	57	1.059	102.995	E94\n"
    "0	63	25	63	1.032	102.950	E94\n"
    "0	71	25	71	0.419	100.483	X94\n"
    "0	0	26	0	0.000	98.100	0:*-26-*	MMFF	DEF\n"
    "4	0	26	0	0.000	83.600	4:*-26-*	MMFF	DEF\n"
    "0	1	26	1	1.085	98.054	E94\n"
    "0	1	26	8	1.263	96.331	E94\n"
    "0	1	26	10	1.115	102.175	E94\n"
    "0	1	26	12	1.147	98.926	X94\n"
    "0	1	26	15	1.141	100.260	E94\n"
    "0	1	26	20	1.075	98.171	E94\n"
    "0	1	26	26	0.997	92.571	E94\n"
    "0	1	26	37	1.081	98.754	E94\n"
    "0	1	26	71	0.672	97.353	X94\n"
    "0	6	26	6	1.833	97.935	E94\n"
    "0	6	26	11	1.663	100.061	E94\n"
    "0	6	26	12	1.442	99.021	E94\n"
    "0	8	26	8	1.189	105.662	E94\n"
    "0	8	26	12	1.028	110.069	E94\n"
    "0	8	26	34	1.509	93.096	E94\n"
    "0	11	26	11	1.757	94.795	E94\n"
    "0	12	26	15	1.271	99.730	E94\n"
    "0	12	26	34	1.508	90.565	E94\n"
    "0	12	26	40	1.165	103.783	E94\n"
    "0	12	26	71	0.704	96.577	X94\n"
    "0	15	26	26	1.047	96.592	E94\n",
    "0	15	26	40	1.543	91.164	E94\n"
    "4	20	26	20	1.252	83.624	E94\n"
    "0	71	26	71	0.473	94.470	X94\n"
    "0	0	30	0	0.000	134.200	0:*-30-*	MMFF	DEF\n"
    "1	0	30	0	0.000	131.800	1:*-30-*	MMFF	DEF\n"
    "4	0	30	0	0.000	97.700	4:*-30-*	MMFF	DEF\n"
    "7	0	30	0	0.000	92.300	7:*-30-*	MMFF	DEF\n"
    "1	2	30	3	0.778	128.756	E94\n"
    "0	2	30	15	0.805	130.439	E94\n"
    "0	2	30	20	0.727	132.187	E94\n"
    "0	2	30	22	0.737	131.100	E94\n"
    "1	2	30	30	0.751	132.225	E94\n"
    "1	3	30	4	0.721	134.566	E94\n"
    "1	3	30	5	0.410	135.975	E94\n"
    "1	3	30	6	0.845	137.596	E94\n"
    "1	3	30	20	0.714	130.677	E94\n"
    "7	3	30	20	1.280	89.957	E94\n"
    "1	3	30	30	0.857	122.418	E94\n"
    "7	3	30	30	1.260	93.102	E94\n"
    "0	4	30	20	0.690	136.444	E94\n"
    "0	5	30	20	0.390	131.835	C94\n"
    "0	5	30	30	0.364	132.652	C94\n"
    "0	6	30	30	0.876	139.045	E94\n"
    "0	15	30	15	0.876	130.718	E94\n"
    "4	15	30	15	1.239	101.359	E94\n"
    "0	15	30	30	0.782	132.228	E94\n"
    "4	15	30	30	1.141	100.902	E94\n"
    "4	20	30	30	1.117	95.513	C94\n"
    "7	20	30	30	1.191	93.909	E94\n"
    "0	20	30	40	0.769	134.526	E94\n"
    "1	20	30	67	0.704	138.631	E94\n"
    "4	22	30	22	1.179	93.007	E94\n"
    "8	30	30	30	1.230	93.732	E94\n"
    "0	30	30	40	0.706	145.470	E94\n"
    "1	30	30	67	0.907	125.792	E94\n"
    "0	0	34	0	0.000	109.400	0:*-34-*	MMFF	DEF\n"
    "4	0	34	0	0.000	89.400	4:*-34-*	MMFF	DEF\n"
    "0	1	34	1	0.862	112.251	C94\n"
    "0	1	34	2	1.154	109.212	E94\n"
    "0	1	34	8	1.330	106.399	E94\n"
    "0	1	34	9	1.166	112.989	E94\n"
    "0	1	34	10	1.388	104.291	E94\n"
    "0	1	34	20	1.201	106.135	E94\n"
    "0	1	34	26	0.913	112.004	E94\n"
    "0	1	34	36	0.576	111.206	C94\n"
    "0	1	34	37	1.141	109.045	E94\n"
    "0	2	34	36	0.694	112.000	#E94\n"
    "0	8	34	36	0.796	109.753	E94\n"
    "0	9	34	36	0.793	108.649	E94\n"
    "0	10	34	36	0.828	108.000	#E94\n"
    "4	20	34	20	1.448	89.411	E94\n",
    "0	20	34	36	0.665	112.526	E94\n"
    "0	22	34	36	0.694	110.000	#E94\n"
    "0	36	34	36	0.578	107.787	C94\n"
    "0	36	34	37	0.717	108.668	E94\n"
    "0	36	34	43	0.840	108.000	#E94\n"
    "0	0	37	0	0.000	118.800	0:*-37-*	MMFF	DEF\n"
    "1	0	37	0	0.000	115.900	1:*-37-*	MMFF	DEF\n"
    "3	0	37	0	0.000	64.700	3::*-37-*	MMFF	DEF\n"
    "4	0	37	0	0.000	91.800	4:*-37-*	MMFF	DEF\n"
    "0	1	37	37	0.803	120.419	C94\n"
    "0	1	37	38	0.992	118.432	E94\n"
    "0	1	37	58	1.027	116.528	E94\n"
    "0	1	37	63	0.837	123.024	E94\n"
    "0	1	37	64	0.821	124.073	E94\n"
    "0	1	37	69	1.038	115.506	E94\n"
    "1	2	37	37	0.712	119.695	C94\n"
    "1	2	37	38	1.029	117.220	E94\n"
    "1	3	37	37	0.798	114.475	C94\n"
    "7	3	37	37	1.320	90.784	E94\n"
    "1	3	37	38	1.109	112.724	E94\n"
    "1	3	37	58	1.134	111.566	E94\n"
    "1	3	37	69	1.119	111.916	E94\n"
    "1	4	37	37	0.906	119.614	E94\n"
    "1	4	37	38	1.087	114.623	E94\n"
    "0	5	37	37	0.563	120.571	C94\n"
    "0	5	37	38	0.693	115.588	C94\n"
    "0	5	37	58	0.699	113.316	E94\n"
    "0	5	37	63	0.702	121.238	C94\n"
    "0	5	37	64	0.523	121.446	C94\n"
    "0	5	37	69	0.794	111.638	C94\n"
    "0	5	37	78	0.563	119.432	E94\n"
    "0	6	37	37	0.968	116.495	C94\n"
    "0	6	37	38	1.324	115.886	E94\n"
    "0	6	37	64	1.139	118.868	E94\n"
    "1	9	37	37	0.974	121.003	E94\n"
    "1	9	37	38	1.137	117.591	E94\n"
    "0	10	37	37	1.025	117.918	E94\n"
    "0	10	37	38	1.088	120.135	E94\n"
    "0	10	37	58	1.077	120.925	E94\n"
    "0	11	37	37	1.094	118.065	E94\n"
    "0	11	37	38	1.223	117.328	E94\n"
    "0	12	37	37	0.950	118.495	E94\n"
    "0	12	37	38	1.126	113.859	E94\n"
    "0	12	37	64	1.076	111.320	E94\n"
    "0	13	37	37	0.917	118.117	E94\n"
    "0	14	37	37	0.861	118.045	E94\n"
    "0	15	37	37	0.755	121.037	C94\n"
    "0	15	37	38	1.027	119.421	E94\n"
    "0	15	37	64	0.976	117.125	E94\n"
    "0	17	37	37	0.930	119.408	E94\n"
    "0	17	37	38	1.179	110.828	E94\n",
    "0	17	37	64	0.946	118.357	E94\n"
    "0	18	37	37	1.029	113.991	X94\n"
    "0	18	37	38	1.278	106.908	E94\n"
    "0	18	37	64	0.975	117.029	E94\n"
    "0	19	37	37	0.660	125.278	E94\n"
    "0	20	37	37	0.744	129.614	E94\n"
    "4	20	37	37	1.217	93.425	E94\n"
    "0	22	37	37	0.805	125.777	E94\n"
    "3	22	37	37	0.152	64.704	E94\n"
    "0	22	37	38	0.904	124.494	E94\n"
    "0	25	37	37	0.718	121.600	E94\n"
    "0	26	37	37	0.691	122.967	E94\n"
    "0	34	37	37	1.030	116.423	E94\n"
    "0	34	37	64	1.074	113.905	E94\n"
    "0	35	37	37	0.964	131.858	E94\n"
    "0	35	37	38	1.187	124.980	E94\n"
    "0	37	37	37	0.669	119.977	C94\n"
    "1	37	37	37	0.864	122.227	E94\n"
    "4	37	37	37	1.380	90.193	E94\n"
    "0	37	37	38	0.596	126.139	C94\n"
    "1	37	37	38	1.033	117.271	E94\n"
    "0	37	37	39	1.038	117.619	E94\n"
    "1	37	37	39	1.078	114.622	E94\n"
    "0	37	37	40	1.045	121.633	C94\n"
    "0	37	37	41	0.892	119.572	E94\n"
    "0	37	37	43	1.013	117.860	X94\n"
    "0	37	37	45	1.114	112.337	E94\n"
    "0	37	37	46	0.999	120.038	E94\n"
    "0	37	37	55	1.002	120.163	E94\n"
    "0	37	37	56	1.020	117.801	E94\n"
    "1	37	37	57	0.881	120.932	E94\n"
    "0	37	37	58	1.014	120.052	E94\n"
    "1	37	37	58	1.127	112.251	E94\n"
    "0	37	37	61	1.072	115.515	E94\n"
    "0	37	37	62	0.941	124.384	E94\n"
    "0	37	37	63	0.478	111.243	C94\n"
    "1	37	37	63	0.894	120.190	E94\n"
    "0	37	37	64	0.423	112.567	C94\n"
    "1	37	37	64	0.912	118.973	E94\n"
    "1	37	37	67	1.064	114.980	E94\n"
    "0	37	37	69	0.872	116.778	C94\n"
    "1	37	37	69	1.042	116.438	E94\n"
    "0	37	37	78	0.974	116.439	E94\n"
    "0	37	37	81	1.034	115.664	E94\n"
    "1	37	37	81	1.104	111.759	E94\n"
    "0	38	37	38	0.725	128.938	C94\n"
    "0	38	37	40	1.024	123.755	E94\n"
    "0	38	37	43	1.165	115.355	E94\n"
    "0	38	37	58	0.979	128.362	E94\n"
    "1	38	37	58	1.257	111.356	E94\n"
    "0	38	37	62	1.148	118.349	E94\n",
    "0	38	37	63	1.095	115.386	E94\n"
    "1	38	37	63	1.076	114.910	E94\n"
    "0	38	37	64	1.070	116.605	E94\n"
    "1	38	37	67	1.289	109.610	E94\n"
    "0	40	37	58	1.103	119.417	E94\n"
    "0	40	37	63	0.943	122.904	E94\n"
    "0	40	37	64	0.931	123.541	E94\n"
    "0	40	37	78	0.931	123.604	E94\n"
    "0	41	37	58	0.967	120.535	E94\n"
    "0	45	37	63	1.031	116.781	E94\n"
    "0	45	37	64	1.156	110.199	E94\n"
    "0	45	37	69	1.248	111.041	E94\n"
    "0	58	37	62	1.016	125.987	E94\n"
    "0	58	37	63	1.152	112.628	E94\n"
    "0	58	37	64	1.291	106.250	E94\n"
    "1	58	37	64	1.108	113.166	E94\n"
    "0	58	37	78	1.188	110.842	E94\n"
    "0	0	38	0	0.000	113.800	0:*-38-*	MMFF	DEF\n"
    "0	37	38	37	1.085	115.406	C94\n"
    "0	37	38	38	1.289	112.016	C94\n"
    "0	37	38	63	1.230	110.181	E94\n"
    "0	37	38	64	1.207	111.032	E94\n"
    "0	37	38	69	1.238	114.692	E94\n"
    "0	37	38	78	1.118	114.813	E94\n"
    "0	38	38	38	1.343	118.516	E94\n"
    "0	0	39	0	0.000	120.700	0:*-39-*	MMFF	DEF\n"
    "1	0	39	0	0.000	125.400	1:*-39-*	MMFF	DEF\n"
    "0	1	39	63	0.854	123.380	C94\n"
    "0	1	39	65	1.111	118.049	E94\n"
    "1	2	39	63	0.858	130.275	E94\n"
    "1	2	39	65	0.900	133.220	E94\n"
    "1	3	39	63	0.900	127.045	E94\n"
    "1	3	39	65	1.126	118.909	E94\n"
    "0	6	39	63	1.166	122.985	E94\n"
    "0	6	39	65	1.396	117.707	E94\n"
    "0	8	39	63	1.000	124.868	E94\n"
    "0	8	39	65	1.057	127.145	E94\n"
    "1	9	39	63	0.981	127.725	E94\n"
    "1	9	39	65	1.170	122.487	E94\n"
    "0	10	39	63	1.109	119.788	E94\n"
    "0	10	39	65	1.118	124.961	E94\n"
    "0	18	39	63	1.108	117.061	X94\n"
    "0	23	39	63	0.551	127.770	C94\n"
    "0	23	39	65	0.752	118.352	C94\n"
    "0	23	39	78	0.581	124.000	#E94\n"
    "0	25	39	63	0.667	134.561	E94\n"
    "0	25	39	65	0.944	118.135	E94\n"
    "0	37	39	63	0.900	127.009	E94\n"
    "1	37	39	63	0.922	125.312	E94\n"
    "1	37	39	65	1.080	121.090	E94\n"
    "0	40	39	63	0.984	126.832	E94\n",
    "0	45	39	63	1.056	121.641	E94\n"
    "0	45	39	65	1.354	112.464	E94\n"
    "0	63	39	63	1.152	109.599	C94\n"
    "1	63	39	63	0.887	128.078	E94\n"
    "0	63	39	64	1.004	120.577	E94\n"
    "1	63	39	64	0.899	126.936	E94\n"
    "0	63	39	65	1.284	112.087	C94\n"
    "1	63	39	65	1.146	117.990	E94\n"
    "0	63	39	78	1.300	105.800	E94\n"
    "0	64	39	65	1.007	126.117	E94\n"
    "0	65	39	65	1.462	116.898	C94\n"
    "0	0	40	0	0.000	115.000	0:*-40-*	MMFF	DEF\n"
    "3	0	40	0	0.000	57.800	3::*-40-*	MMFF	DEF\n"
    "0	1	40	1	1.064	113.703	E94\n"
    "0	1	40	2	0.998	118.873	E94\n"
    "0	1	40	3	1.007	118.319	E94\n"
    "0	1	40	6	1.421	109.742	E94\n"
    "0	1	40	9	1.203	113.198	E94\n"
    "0	1	40	10	1.232	111.320	E94\n"
    "0	1	40	11	1.436	104.665	E94\n"
    "0	1	40	12	1.202	109.320	E94\n"
    "0	1	40	20	1.047	114.970	E94\n"
    "0	1	40	25	0.912	114.483	E94\n"
    "0	1	40	28	0.689	112.374	C94\n"
    "0	1	40	30	1.024	118.604	E94\n"
    "0	1	40	37	0.835	107.349	C94\n"
    "0	1	40	39	1.254	110.622	E94\n"
    "0	1	40	40	1.183	114.011	E94\n"
    "0	1	40	45	1.223	112.226	E94\n"
    "0	1	40	46	1.025	122.982	E94\n"
    "0	1	40	63	1.084	114.473	E94\n"
    "0	1	40	64	1.064	115.483	E94\n"
    "0	2	40	2	0.997	120.651	E94\n"
    "0	2	40	3	0.981	121.660	E94\n"
    "0	2	40	6	1.316	115.626	E94\n"
    "0	2	40	9	1.118	119.196	E94\n"
    "0	2	40	10	1.142	117.260	E94\n"
    "0	2	40	19	0.732	128.087	E94\n"
    "0	2	40	28	0.767	111.053	C94\n"
    "0	2	40	37	1.049	117.022	E94\n"
    "0	2	40	39	1.192	115.106	E94\n"
    "0	2	40	40	1.060	122.253	E94\n"
    "0	2	40	63	1.008	120.447	E94\n"
    "0	3	40	3	0.883	128.240	E94\n"
    "0	3	40	8	1.259	111.557	E94\n"
    "0	3	40	9	1.106	119.822	E94\n"
    "0	3	40	10	1.269	111.261	E94\n"
    "0	3	40	12	1.146	112.718	E94\n"
    "0	3	40	15	1.105	117.871	E94\n"
    "0	3	40	20	1.130	112.139	E94\n"
    "0	3	40	22	1.072	114.420	E94\n",
    "0	3	40	25	0.820	121.724	E94\n"
    "0	3	40	28	0.700	114.808	C94\n"
    "0	3	40	37	1.056	116.655	E94\n"
    "0	3	40	40	1.147	117.511	E94\n"
    "0	3	40	64	1.132	113.602	E94\n"
    "0	6	40	28	0.889	110.000	#E94\n"
    "0	8	40	28	0.764	111.915	E94\n"
    "0	8	40	37	1.216	112.920	E94\n"
    "0	8	40	63	1.351	108.085	E94\n"
    "0	9	40	28	0.774	112.549	E94\n"
    "0	9	40	37	1.236	112.751	E94\n"
    "0	10	40	28	0.799	109.725	E94\n"
    "0	10	40	37	1.316	108.686	E94\n"
    "0	11	40	37	1.546	101.687	E94\n"
    "0	15	40	15	1.154	121.497	E94\n"
    "3	22	40	22	0.204	57.777	E94\n"
    "0	22	40	37	1.066	114.220	E94\n"
    "0	22	40	63	1.126	112.006	E94\n"
    "0	25	40	28	0.485	120.000	#E94\n"
    "0	25	40	37	0.868	117.977	E94\n"
    "0	26	40	28	0.506	118.000	#E94\n"
    "0	26	40	37	0.812	122.336	E94\n"
    "0	28	40	28	0.560	109.160	C94\n"
    "0	28	40	30	0.656	119.230	E94\n"
    "0	28	40	37	0.662	110.288	C94\n"
    "0	28	40	39	0.789	110.951	E94\n"
    "0	28	40	40	0.782	111.731	E94\n"
    "0	28	40	45	0.674	120.000	#E94\n"
    "0	28	40	54	0.738	118.714	E94\n"
    "0	28	40	63	0.670	116.188	E94\n"
    "0	28	40	64	0.659	117.057	E94\n"
    "0	28	40	78	0.618	119.829	E94\n"
    "0	37	40	37	1.004	119.018	E94\n"
    "0	37	40	45	1.376	106.579	E94\n"
    "0	37	40	54	1.394	107.777	E94\n"
    "0	37	40	63	1.060	116.867	E94\n"
    "0	45	40	64	1.283	111.332	E94\n"
    "0	45	40	78	1.410	105.678	E94\n"
    "0	46	40	64	1.189	116.345	E94\n"
    "0	0	41	0	0.000	118.300	0:*-41-*	MMFF	DEF\n"
    "0	1	41	32	1.209	114.689	C94\n"
    "0	1	41	72	1.024	114.936	X94\n"
    "0	2	41	32	1.309	115.461	C94\n"
    "0	3	41	32	1.210	114.810	E94\n"
    "0	5	41	32	0.912	113.960	C94\n"
    "0	6	41	72	1.319	113.899	E94\n"
    "0	9	41	72	1.089	117.795	E94\n"
    "0	10	41	72	1.039	121.240	E94\n"
    "0	20	41	32	1.090	120.965	E94\n"
    "0	22	41	32	1.079	122.748	E94\n"
    "0	32	41	32	1.181	130.600	C94\n",
    "0	32	41	37	1.136	118.871	E94\n"
    "0	32	41	41	1.401	107.694	E94\n"
    "0	37	41	72	1.035	114.919	E94\n"
    "0	55	41	72	0.982	123.972	E94\n"
    "0	62	41	72	1.052	120.425	E94\n"
    "0	72	41	72	0.912	130.128	X94\n"
    "0	72	41	80	1.094	112.175	E94\n"
    "0	0	43	0	0.000	113.300	0:*-43-*	MMFF	DEF\n"
    "0	1	43	1	1.109	110.353	E94\n"
    "0	1	43	2	1.052	114.321	E94\n"
    "0	1	43	3	0.938	121.050	E94\n"
    "0	1	43	4	0.927	123.204	E94\n"
    "0	1	43	18	1.116	115.011	X94\n"
    "0	1	43	25	0.853	115.637	X94\n"
    "0	1	43	28	0.646	113.739	X94\n"
    "0	1	43	37	1.083	112.511	E94\n"
    "0	1	43	45	1.140	115.034	E94\n"
    "0	1	43	64	1.025	116.188	E94\n"
    "0	2	43	18	1.227	110.268	E94\n"
    "0	3	43	18	1.011	121.488	X94\n"
    "0	3	43	20	1.053	113.913	E94\n"
    "4	3	43	20	1.327	93.575	E94\n"
    "0	3	43	28	0.626	117.464	X94\n"
    "0	4	43	28	0.616	122.000	E94\n"
    "0	4	43	45	1.253	112.373	E94\n"
    "0	6	43	18	1.673	104.311	E94\n"
    "3	6	43	22	0.279	54.827	E94\n"
    "0	6	43	28	0.868	110.000	#E94\n"
    "0	6	43	37	1.519	105.833	E94\n"
    "0	6	43	43	1.603	108.652	E94\n"
    "0	8	43	18	1.511	104.036	E94\n"
    "0	8	43	28	0.794	110.320	E94\n"
    "0	15	43	15	1.558	103.008	E94\n"
    "0	15	43	18	1.409	108.458	E94\n"
    "0	17	43	18	1.367	111.904	E94\n"
    "0	18	43	18	1.144	120.463	E94\n"
    "0	18	43	20	0.961	123.768	E94\n"
    "4	18	43	20	1.451	92.867	E94\n"
    "0	18	43	22	1.171	112.379	E94\n"
    "0	18	43	28	0.628	116.881	X94\n"
    "0	18	43	34	1.324	111.347	E94\n"
    "0	18	43	37	1.185	112.132	X94\n"
    "0	18	43	43	1.379	109.036	E94\n"
    "0	18	43	64	1.108	116.279	E94\n"
    "0	20	43	28	0.626	115.000	#E94\n"
    "3	22	43	22	0.209	57.032	E94\n"
    "0	25	43	28	0.468	118.274	X94\n"
    "0	28	43	28	0.477	112.596	X94\n"
    "0	28	43	34	0.810	110.000	#E94\n"
    "0	28	43	37	0.669	113.350	X94\n"
    "0	28	43	64	0.658	115.293	E94\n",
    "0	0	44	0	0.000	91.600	0:*-44-*	MMFF	DEF\n"
    "0	63	44	63	1.962	88.495	C94\n"
    "0	63	44	65	2.261	94.137	C94\n"
    "0	63	44	78	1.738	86.270	E94\n"
    "0	63	44	80	1.748	86.194	E94\n"
    "0	65	44	65	1.530	101.147	E94\n"
    "0	65	44	80	1.629	93.534	E94\n"
    "0	78	44	78	0.903	119.401	E94\n"
    "0	0	45	0	0.000	116.700	0:*-45-*	MMFF	DEF\n"
    "0	1	45	32	1.260	118.182	X94\n"
    "0	2	45	32	1.294	118.082	X94\n"
    "0	3	45	32	1.343	115.589	E94\n"
    "0	6	45	32	1.787	111.682	X94\n"
    "0	8	45	32	1.515	115.695	E94\n"
    "0	9	45	32	1.339	123.850	E94\n"
    "0	10	45	32	1.578	112.194	E94\n"
    "0	20	45	32	1.245	118.893	E94\n"
    "0	22	45	32	1.293	117.503	E94\n"
    "0	32	45	32	1.467	128.036	X94\n"
    "0	32	45	37	1.298	117.857	E94\n"
    "0	32	45	39	1.715	107.633	E94\n"
    "0	32	45	40	1.497	116.432	E94\n"
    "0	32	45	43	1.545	113.711	E94\n"
    "0	32	45	63	1.335	116.765	E94\n"
    "0	32	45	64	1.330	116.908	E94\n"
    "0	32	45	78	1.394	114.962	E94\n"
    "0	0	46	0	0.000	111.000	0:*-46-*	MMFF	DEF\n"
    "0	1	46	7	1.440	110.492	X94\n"
    "0	2	46	7	1.489	112.709	E94\n"
    "0	7	46	8	1.724	109.817	E94\n"
    "0	7	46	37	1.519	110.569	E94\n"
    "0	7	46	40	1.650	111.405	E94\n"
    "0	0	48	0	0.000	118.400	0:*-48-*	MMFF	DEF\n"
    "0	3	48	18	1.065	122.928	E94\n"
    "0	18	48	28	0.736	113.969	X94\n"
    "0	0	49	0	0.000	111.400	0:*-49-*	MMFF	DEF\n"
    "0	50	49	50	0.522	111.433	C94\n"
    "0	0	51	0	0.000	111.400	0:*-51-*	MMFF	DEF\n"
    "0	3	51	52	0.913	111.360	X94\n"
    "0	0	53	0	0.000	180.000	0:*-53-*	MMFF	DEF\n"
    "0	3	53	47	0.574	180.000	E94\n"
    "0	9	53	47	0.649	180.000	E94\n"
    "0	0	54	0	0.000	119.500	0:*-54-*	MMFF	DEF\n"
    "1	0	54	0	0.000	115.700	1:*-54-*	MMFF	DEF\n"
    "0	1	54	1	0.923	121.439	E94\n"
    "0	1	54	3	0.707	124.083	C94\n"
    "0	1	54	36	0.294	122.881	C94\n"
    "0	3	54	6	1.376	115.398	E94\n"
    "1	3	54	9	1.128	114.457	E94\n"
    "0	3	54	36	0.685	119.698	C94\n"
    "1	3	54	40	1.105	116.439	E94\n",
    "0	6	54	36	0.826	115.000	#E94\n"
    "0	9	54	40	1.195	123.403	E94\n"
    "0	36	54	36	0.300	113.943	C94\n"
    "0	0	55	0	0.000	120.800	0:*-55-*	MMFF	DEF\n"
    "0	1	55	1	0.951	119.946	E94\n"
    "0	1	55	36	0.307	126.448	C94\n"
    "0	1	55	37	1.032	117.035	E94\n"
    "0	1	55	57	0.751	120.606	C94\n"
    "0	1	55	80	0.972	121.082	E94\n"
    "0	2	55	3	1.041	116.994	E94\n"
    "0	2	55	36	0.621	120.000	#E94\n"
    "0	2	55	57	1.047	118.847	E94\n"
    "0	3	55	9	1.053	121.298	E94\n"
    "0	3	55	36	0.567	124.000	#E94\n"
    "0	3	55	57	0.953	123.573	E94\n"
    "0	3	55	62	1.041	122.163	E94\n"
    "0	6	55	36	0.833	114.000	#E94\n"
    "0	6	55	57	1.408	112.958	E94\n"
    "0	8	55	36	0.656	122.000	#E94\n"
    "0	8	55	57	1.259	113.209	E94\n"
    "0	9	55	57	1.001	126.373	E94\n"
    "0	18	55	36	0.578	125.000	#E94\n"
    "0	18	55	57	1.054	122.320	E94\n"
    "0	36	55	36	0.355	117.729	C94\n"
    "0	36	55	37	0.623	120.405	E94\n"
    "0	36	55	41	0.485	134.689	E94\n"
    "0	36	55	57	0.663	119.499	C94\n"
    "0	36	55	64	0.632	118.000	#E94\n"
    "0	36	55	80	0.684	115.880	E94\n"
    "0	37	55	57	1.110	115.816	E94\n"
    "0	41	55	57	0.911	126.801	E94\n"
    "0	57	55	62	1.054	123.366	E94\n"
    "0	57	55	64	1.026	119.465	E94\n"
    "0	0	56	0	0.000	119.100	0:*-56-*	MMFF	DEF\n"
    "0	1	56	36	0.472	123.585	C94\n"
    "0	1	56	57	0.774	119.267	C94\n"
    "0	2	56	9	1.181	116.311	E94\n"
    "0	2	56	36	0.582	124.037	E94\n"
    "0	2	56	57	1.029	118.607	E94\n"
    "0	3	56	36	0.585	121.521	E94\n"
    "0	3	56	57	0.885	126.567	E94\n"
    "0	8	56	36	0.785	111.009	E94\n"
    "0	8	56	57	1.288	110.357	E94\n"
    "0	9	56	36	0.683	120.258	E94\n"
    "0	9	56	57	1.186	115.661	E94\n"
    "0	36	56	36	0.450	117.534	C94\n"
    "0	36	56	37	0.602	120.000	#E94\n"
    "0	36	56	57	0.646	120.649	C94\n"
    "0	36	56	63	0.579	123.766	E94\n"
    "0	36	56	80	0.625	120.000	#E94\n"
    "0	37	56	57	1.058	115.912	E94\n",
    "0	57	56	63	1.019	118.915	E94\n"
    "0	0	57	0	0.000	120.900	0:*-57-*	MMFF	DEF\n"
    "1	0	57	0	0.000	118.100	1:*-57-*	MMFF	DEF\n"
    "0	1	57	55	1.017	117.865	E94\n"
    "1	3	57	55	1.085	115.034	E94\n"
    "0	5	57	55	0.674	116.747	C94\n"
    "0	6	57	55	1.279	119.257	E94\n"
    "1	9	57	55	0.980	128.143	E94\n"
    "0	12	57	55	1.058	118.327	E94\n"
    "0	15	57	55	0.983	123.646	E94\n"
    "0	25	57	55	0.790	122.889	E94\n"
    "1	37	57	55	0.967	121.379	E94\n"
    "0	55	57	55	0.855	126.476	C94\n"
    "1	55	57	63	1.016	118.800	E94\n"
    "1	55	57	64	1.039	117.166	E94\n"
    "0	56	57	56	1.342	120.010	C94\n"
    "0	0	58	0	0.000	119.000	0:*-58-*	MMFF	DEF\n"
    "1	0	58	0	0.000	119.900	1:*-58-*	MMFF	DEF\n"
    "0	1	58	37	1.003	119.236	E94\n"
    "0	1	58	64	0.961	121.070	E94\n"
    "1	3	58	37	0.983	121.506	E94\n"
    "0	6	58	37	1.371	114.370	E94\n"
    "0	18	58	37	1.005	120.665	E94\n"
    "0	36	58	37	0.650	118.713	E94\n"
    "0	36	58	63	0.650	118.000	#E94\n"
    "0	36	58	64	0.620	120.051	E94\n"
    "0	37	58	37	0.996	122.710	E94\n"
    "1	37	58	37	1.036	118.260	E94\n"
    "0	37	58	63	1.087	116.989	E94\n"
    "0	37	58	64	1.061	117.942	E94\n"
    "0	0	59	0	0.000	105.600	0:*-59-*	MMFF	DEF\n"
    "0	63	59	63	1.273	106.313	C94\n"
    "0	63	59	65	1.750	107.755	C94\n"
    "0	63	59	78	1.713	101.179	E94\n"
    "0	63	59	80	1.599	105.341	E94\n"
    "0	65	59	65	1.754	107.683	E94\n"
    "0	65	59	78	1.644	107.142	E94\n"
    "0	65	59	82	1.864	103.624	E94\n"
    "0	0	61	0	0.000	180.000	0:*-61-*	MMFF	DEF\n"
    "0	1	61	60	0.475	180.000	E94\n"
    "0	37	61	42	0.536	180.000	E94\n"
    "0	37	61	60	0.484	180.000	E94\n"
    "0	0	62	0	0.000	108.300	0:*-62-*	MMFF	DEF\n"
    "0	1	62	18	1.316	109.273	X94\n"
    "0	2	62	23	0.817	105.542	X94\n"
    "0	3	62	3	1.318	106.821	E94\n"
    "0	3	62	18	1.311	111.144	E94\n"
    "0	3	62	55	1.528	102.414	E94\n"
    "0	9	62	18	1.515	107.660	E94\n"
    "0	18	62	37	1.229	114.618	E94\n"
    "0	18	62	41	1.366	108.722	E94\n",
    "0	18	62	63	1.427	106.284	E94\n"
    "0	18	62	64	1.317	110.366	E94\n"
    "0	0	63	0	0.000	123.300	0:*-63-*	MMFF	DEF\n"
    "1	0	63	0	0.000	124.300	1:*-63-*	MMFF	DEF\n"
    "0	1	63	39	0.935	121.832	E94\n"
    "0	1	63	44	0.902	122.101	E94\n"
    "0	1	63	59	1.175	115.253	E94\n"
    "0	1	63	64	0.737	131.378	E94\n"
    "0	1	63	66	0.865	127.610	E94\n"
    "1	2	63	39	1.027	117.864	E94\n"
    "1	2	63	59	0.987	127.524	E94\n"
    "1	2	63	64	0.730	133.818	E94\n"
    "1	2	63	66	0.828	132.383	E94\n"
    "1	3	63	39	0.900	125.395	E94\n"
    "1	3	63	44	0.935	120.481	E94\n"
    "1	3	63	59	1.158	117.219	E94\n"
    "1	3	63	64	0.766	130.065	E94\n"
    "1	3	63	66	0.950	123.049	E94\n"
    "1	4	63	44	0.848	126.602	E94\n"
    "1	4	63	59	1.211	114.804	E94\n"
    "1	4	63	64	0.795	127.817	E94\n"
    "0	5	63	39	0.617	121.127	C94\n"
    "0	5	63	44	0.393	126.141	C94\n"
    "0	5	63	59	0.784	114.076	C94\n"
    "0	5	63	64	0.577	131.721	C94\n"
    "0	5	63	66	0.643	125.134	C94\n"
    "0	5	63	78	0.482	130.000	#E94\n"
    "0	5	63	81	0.588	124.000	#E94\n"
    "0	6	63	39	1.234	120.509	E94\n"
    "0	6	63	59	1.564	113.514	E94\n"
    "0	6	63	64	0.951	131.301	E94\n"
    "1	9	63	39	1.068	121.741	E94\n"
    "1	9	63	44	0.963	124.598	E94\n"
    "1	9	63	64	0.804	134.237	E94\n"
    "1	9	63	66	0.912	133.020	E94\n"
    "0	10	63	39	1.084	120.356	E94\n"
    "0	10	63	44	1.112	115.732	E94\n"
    "0	10	63	59	1.307	116.218	E94\n"
    "0	10	63	64	0.867	128.750	E94\n"
    "0	10	63	66	0.981	127.617	E94\n"
    "0	12	63	39	1.111	114.439	E94\n"
    "0	12	63	44	1.035	119.321	E94\n"
    "0	12	63	64	0.838	126.226	E94\n"
    "0	12	63	66	0.980	122.280	E94\n"
    "0	15	63	39	1.064	117.958	E94\n"
    "0	15	63	44	0.952	125.654	E94\n"
    "0	15	63	64	0.813	129.284	E94\n"
    "0	15	63	66	0.962	124.490	E94\n"
    "0	18	63	44	1.110	116.077	E94\n"
    "0	18	63	64	0.740	135.028	E94\n"
    "0	19	63	39	0.647	132.369	E94\n",
    "0	19	63	64	0.517	141.986	E94\n"
    "0	25	63	39	0.597	139.439	E94\n"
    "0	25	63	66	0.776	122.699	E94\n"
    "0	35	63	59	1.351	124.475	E94\n"
    "0	35	63	64	0.808	145.098	E94\n"
    "0	37	63	39	1.011	132.046	C94\n"
    "1	37	63	39	0.934	123.481	E94\n"
    "0	37	63	44	0.764	133.930	E94\n"
    "1	37	63	44	0.915	121.637	E94\n"
    "0	37	63	59	1.041	124.836	E94\n"
    "1	37	63	59	1.214	114.211	E94\n"
    "0	37	63	64	0.679	122.881	C94\n"
    "1	37	63	64	0.742	131.784	E94\n"
    "0	37	63	66	0.742	140.668	E94\n"
    "1	37	63	66	0.871	128.130	E94\n"
    "0	38	63	39	1.022	124.814	E94\n"
    "0	38	63	64	0.910	126.513	E94\n"
    "0	39	63	39	0.910	131.461	E94\n"
    "1	39	63	39	1.105	119.174	E94\n"
    "0	39	63	40	1.112	119.261	E94\n"
    "1	39	63	44	1.144	114.126	E94\n"
    "0	39	63	45	1.166	115.115	E94\n"
    "1	39	63	57	0.931	123.222	E94\n"
    "0	39	63	58	1.042	123.231	E94\n"
    "1	39	63	63	0.949	122.353	E94\n"
    "0	39	63	64	0.813	107.255	C94\n"
    "1	39	63	64	0.943	123.441	E94\n"
    "0	39	63	66	1.012	110.865	C94\n"
    "1	39	63	66	1.095	120.834	E94\n"
    "0	40	63	44	0.943	125.881	E94\n"
    "0	40	63	59	1.298	117.078	E94\n"
    "0	40	63	64	0.845	130.865	E94\n"
    "0	40	63	66	0.940	130.926	E94\n"
    "0	44	63	45	1.125	114.633	E94\n"
    "0	44	63	56	1.030	120.178	E94\n"
    "0	44	63	62	0.991	122.899	E94\n"
    "1	44	63	63	0.894	123.341	E94\n"
    "0	44	63	64	0.853	108.480	C94\n"
    "0	44	63	66	0.854	114.516	C94\n"
    "0	44	63	72	0.915	129.129	E94\n"
    "0	44	63	78	1.217	106.254	E94\n"
    "0	44	63	81	1.278	108.400	E94\n"
    "0	45	63	59	1.467	108.824	E94\n"
    "0	45	63	64	0.940	122.725	E94\n"
    "0	45	63	66	1.164	116.157	E94\n"
    "0	56	63	66	0.875	134.888	E94\n"
    "1	57	63	66	0.945	123.246	E94\n"
    "0	58	63	64	0.965	122.522	E94\n"
    "0	59	63	64	1.035	110.108	C94\n"
    "0	59	63	66	1.181	115.592	C94\n"
    "0	62	63	66	0.976	128.662	E94\n",
    "1	63	63	64	0.776	129.499	E94\n"
    "1	63	63	66	0.929	124.689	E94\n"
    "0	66	63	72	0.911	129.610	E94\n"
    "0	0	64	0	0.000	121.400	0:*-64-*	MMFF	DEF\n"
    "1	0	64	0	0.000	121.700	1:*-64-*	MMFF	DEF\n"
    "0	1	64	63	0.776	128.041	E94\n"
    "0	1	64	64	0.766	128.061	E94\n"
    "0	1	64	65	0.963	120.640	E94\n"
    "0	1	64	66	0.952	120.685	E94\n"
    "0	1	64	81	1.050	114.735	E94\n"
    "0	1	64	82	1.013	117.414	E94\n"
    "1	2	64	63	0.861	122.947	E94\n"
    "1	2	64	64	0.816	125.433	E94\n"
    "1	2	64	65	0.907	125.781	E94\n"
    "1	2	64	66	1.010	118.540	E94\n"
    "1	2	64	82	0.923	124.473	E94\n"
    "1	3	64	63	0.828	124.890	E94\n"
    "1	3	64	64	0.774	128.286	E94\n"
    "1	3	64	65	0.973	120.954	E94\n"
    "1	3	64	66	0.949	121.821	E94\n"
    "1	3	64	81	0.995	118.754	E94\n"
    "1	4	64	63	0.845	123.889	E94\n"
    "1	4	64	64	0.804	126.131	E94\n"
    "1	4	64	65	1.036	117.401	E94\n"
    "1	4	64	66	1.010	118.254	E94\n"
    "0	5	64	63	0.501	126.170	C94\n"
    "0	5	64	64	0.546	127.405	C94\n"
    "0	5	64	65	0.664	118.412	C94\n"
    "0	5	64	66	0.699	120.478	C94\n"
    "0	5	64	78	0.482	127.331	E94\n"
    "0	5	64	81	0.605	120.000	#E94\n"
    "0	5	64	82	0.597	122.000	#E94\n"
    "0	6	64	63	1.112	120.985	E94\n"
    "0	6	64	64	1.043	123.922	E94\n"
    "0	6	64	65	1.348	115.506	E94\n"
    "0	6	64	66	1.156	123.890	E94\n"
    "1	9	64	64	0.959	120.924	E94\n"
    "1	9	64	65	1.098	119.529	E94\n"
    "1	9	64	66	1.013	123.743	E94\n"
    "0	10	64	63	0.937	123.695	E94\n"
    "0	10	64	64	0.893	125.735	E94\n"
    "0	10	64	65	1.016	124.788	E94\n"
    "0	10	64	66	1.065	121.125	E94\n"
    "0	12	64	63	0.845	126.259	E94\n"
    "0	12	64	64	0.869	124.058	E94\n"
    "0	12	64	65	1.020	120.198	E94\n"
    "0	12	64	66	0.971	122.900	E94\n"
    "0	13	64	63	0.845	123.004	E94\n"
    "0	13	64	64	0.883	120.111	E94\n"
    "0	15	64	63	0.870	124.581	E94\n"
    "0	15	64	64	0.882	123.309	E94\n",
    "0	15	64	65	1.008	121.049	E94\n"
    "0	15	64	66	0.990	121.826	E94\n"
    "0	18	64	65	1.065	118.404	E94\n"
    "0	18	64	66	1.067	118.002	E94\n"
    "0	37	64	63	0.906	117.966	C94\n"
    "0	37	64	64	0.854	136.087	C94\n"
    "1	37	64	64	0.772	128.673	E94\n"
    "0	37	64	65	0.799	134.844	E94\n"
    "1	37	64	65	0.942	122.866	E94\n"
    "0	37	64	66	0.845	130.337	E94\n"
    "0	37	64	78	0.706	135.432	E94\n"
    "0	37	64	81	0.917	124.856	E94\n"
    "0	37	64	82	0.946	123.684	E94\n"
    "1	37	64	82	1.000	119.086	E94\n"
    "0	38	64	63	0.988	121.242	E94\n"
    "0	38	64	64	0.858	129.014	E94\n"
    "0	38	64	65	0.989	127.335	E94\n"
    "0	38	64	66	1.022	124.454	E94\n"
    "0	39	64	64	1.086	114.312	E94\n"
    "0	39	64	65	1.060	122.481	E94\n"
    "1	39	64	65	1.204	114.188	E94\n"
    "1	39	64	66	1.170	115.157	E94\n"
    "0	40	64	63	0.948	123.538	E94\n"
    "0	40	64	64	0.928	123.853	E94\n"
    "0	40	64	65	0.958	129.125	E94\n"
    "0	40	64	81	1.035	123.154	E94\n"
    "0	40	64	82	1.183	115.934	E94\n"
    "0	43	64	63	0.885	126.749	E94\n"
    "0	43	64	64	0.898	124.876	E94\n"
    "0	43	64	65	1.024	123.706	E94\n"
    "0	43	64	66	1.017	123.409	E94\n"
    "0	45	64	63	0.981	120.063	E94\n"
    "0	45	64	64	0.921	123.014	E94\n"
    "0	45	64	65	1.276	110.521	E94\n"
    "0	45	64	66	1.199	113.371	E94\n"
    "0	55	64	64	0.907	124.405	E94\n"
    "0	55	64	65	1.002	125.220	E94\n"
    "1	57	64	65	1.020	117.950	E94\n"
    "1	57	64	66	0.959	121.017	E94\n"
    "0	58	64	63	1.075	115.646	E94\n"
    "0	58	64	64	0.815	131.812	E94\n"
    "0	58	64	66	0.978	126.562	E94\n"
    "0	62	64	64	0.885	126.560	E94\n"
    "0	62	64	65	1.073	121.703	E94\n"
    "0	63	64	64	0.866	108.239	C94\n"
    "1	63	64	64	0.827	124.584	E94\n"
    "0	63	64	66	1.038	111.621	C94\n"
    "0	63	64	78	1.172	105.176	E94\n"
    "0	63	64	81	1.164	110.895	E94\n"
    "0	63	64	82	1.395	101.902	E94\n"
    "0	64	64	64	0.967	115.037	E94\n"
    "0	64	64	65	0.916	113.570	C94\n"
    "1	64	64	66	1.003	118.067	E94\n",
    "0	64	64	78	1.194	103.479	E94\n"
    "0	64	64	82	1.210	108.553	E94\n"
    "0	65	64	66	1.055	115.369	C94\n"
    "0	65	64	81	1.168	116.240	E94\n"
    "0	66	64	66	0.932	129.624	E94\n"
    "0	0	65	0	0.000	104.500	0:*-65-*	MMFF	DEF\n"
    "0	39	65	64	1.738	101.550	C94\n"
    "0	39	65	66	1.589	106.360	C94\n"
    "0	39	65	82	1.740	101.208	E94\n"
    "0	44	65	64	1.430	103.829	C94\n"
    "0	44	65	66	1.366	110.552	E94\n"
    "0	44	65	78	1.419	104.213	E94\n"
    "0	59	65	64	1.788	103.452	C94\n"
    "0	59	65	81	1.774	104.872	E94\n"
    "0	0	66	0	0.000	106.900	0:*-66-*	MMFF	DEF\n"
    "0	63	66	64	1.206	103.779	C94\n"
    "0	63	66	66	1.406	106.735	C94\n"
    "0	63	66	78	1.339	105.365	E94\n"
    "0	63	66	81	1.408	106.806	E94\n"
    "0	64	66	65	1.709	107.658	C94\n"
    "0	65	66	66	1.932	111.306	C94\n"
    "0	0	67	0	0.000	119.900	0:*-67-*	MMFF	DEF\n"
    "1	0	67	0	0.000	116.600	1:*-67-*	MMFF	DEF\n"
    "0	1	67	3	0.982	120.683	E94\n"
    "0	1	67	9	1.178	115.581	E94\n"
    "0	1	67	32	1.233	119.589	E94\n"
    "0	1	67	67	1.257	111.574	E94\n"
    "1	2	67	32	1.118	126.320	E94\n"
    "1	2	67	67	1.231	113.438	E94\n"
    "0	3	67	23	0.567	128.000	#E94\n"
    "0	3	67	32	1.290	120.945	E94\n"
    "1	3	67	37	1.122	113.631	E94\n"
    "1	9	67	30	1.142	118.899	E94\n"
    "0	9	67	32	1.325	125.531	E94\n"
    "1	9	67	37	1.186	115.979	E94\n"
    "0	23	67	32	0.805	120.000	#E94\n"
    "1	30	67	32	1.370	114.854	E94\n"
    "1	32	67	37	1.240	120.019	E94\n"
    "0	32	67	67	1.504	117.327	E94\n"
    "1	37	67	67	1.310	110.017	E94\n"
    "0	0	68	0	0.000	108.800	0:*-68-*	MMFF	DEF\n"
    "0	1	68	1	1.159	108.238	C94\n"
    "0	1	68	23	0.772	107.200	C94\n"
    "0	1	68	32	0.958	110.757	C94\n"
    "0	23	68	23	0.650	104.892	C94\n"
    "0	23	68	32	0.659	112.977	C94\n"
    "0	0	69	0	0.000	120.300	0:*-69-*	MMFF	DEF\n"
    "0	32	69	37	1.123	121.777	C94\n"
    "0	32	69	38	1.486	117.217	E94\n"
    "0	37	69	37	1.223	116.447	C94\n"
    "0	38	69	38	1.122	125.930	E94\n"
    "0	31	70	31	0.658	103.978	C94\n"
    "0	0	73	0	0.000	106.600	0:*-73-*	MMFF	DEF\n"
    "0	1	73	32	1.590	100.180	X94\n"
    "0	1	73	72	1.481	96.166	X94\n"
    "0	32	73	32	1.665	115.012	X94\n"
    "0	32	73	72	1.326	115.134	X94\n"
    "0	0	74	0	0.000	113.000	0:*-74-*	MMFF	DEF\n"
    "0	3	74	7	1.357	113.010	X94\n"
    "0	0	75	0	0.000	94.900	0:*-75-*	MMFF	DEF\n"
    "0	1	75	3	1.138	96.779	E94\n"
    "0	3	75	19	1.044	91.970	E94\n"
    "0	3	75	71	0.729	95.899	X94\n"
    "0	0	76	0	0.000	107.600	0:*-76-*	MMFF	DEF\n"
    "0	76	76	76	1.434	109.889	X94\n"
    "0	76	76	78	1.493	103.519	X94\n"
    "0	78	76	78	1.235	109.421	E94\n"
    "0	0	77	0	0.000	109.500	0:*-77-*	MMFF	DEF\n"
    "0	32	77	32	1.652	109.472	X94\n"
    "0	0	78	0	0.000	121.900	0:*-78-*	MMFF	DEF\n"
    "1	0	78	0	0.000	126.100	1:*-78-*	MMFF	DEF\n"
    "0	1	78	78	0.744	130.960	E94\n"
    "0	1	78	81	0.938	121.477	E94\n"
    "1	3	78	78	0.827	125.468	E94\n"
    "1	3	78	81	0.922	123.748	E94\n"
    "0	5	78	76	0.584	123.407	X94\n"
    "0	5	78	78	0.546	128.000	C94\n"
    "0	5	78	79	0.617	122.000	#E94\n"
    "0	5	78	81	0.542	109.881	C94\n"
    "1	9	78	78	0.863	129.501	E94\n"
    "1	9	78	81	0.991	125.857	E94\n"
    "0	37	78	76	0.770	137.282	E94\n"
    "0	37	78	78	0.803	128.249	E94\n"
    "0	37	78	81	0.864	128.714	E94\n"
    "0	38	78	78	0.844	130.617	E94\n"
    "0	38	78	81	1.023	123.532	E94\n"
    "0	39	78	64	0.734	138.714	E94\n"
    "0	39	78	78	1.202	109.426	E94\n"
    "0	40	78	76	0.930	130.150	E94\n"
    "0	40	78	78	0.778	135.746	E94\n"
    "0	40	78	81	1.058	121.251	E94\n"
    "0	44	78	63	0.677	141.902	E94\n"
    "0	44	78	64	0.663	142.589	E94\n"
    "0	44	78	66	0.816	134.701	E94\n"
    "0	44	78	78	1.089	111.702	E94\n"
    "0	45	78	76	1.199	114.467	E94\n"
    "0	45	78	78	0.915	125.050	E94\n"
    "0	45	78	81	1.216	112.926	E94\n"
    "0	59	78	64	0.963	128.471	E94\n"
    "0	59	78	65	1.097	128.375	E94\n"
    "0	59	78	78	1.443	105.916	E94\n"
    "0	63	78	64	0.942	117.779	E94\n"
    "0	64	78	65	0.835	131.530	E94\n"
    "0	64	78	78	1.038	111.834	E94\n"
    "0	66	78	78	1.030	118.376	E94\n"
    "0	76	78	76	1.245	113.179	X94\n"
    "0	76	78	78	1.159	111.900	E94\n"
    "0	78	78	78	1.336	99.459	E94\n"
    "0	78	78	81	1.302	105.130	C94\n"
    "0	79	78	81	1.217	114.792	E94\n"
    "0	0	79	0	0.000	103.400	0:*-79-*	MMFF	DEF\n"
    "0	78	79	81	1.569	102.043	E94\n"
    "0	79	79	81	1.625	104.857	E94\n"
    "0	0	80	0	0.000	121.900	0:*-80-*	MMFF	DEF\n"
    "1	0	80	0	0.000	128.200	1:*-80-*	MMFF	DEF\n"
    "0	1	80	81	0.864	127.147	E94\n"
    "1	3	80	81	0.886	128.181	E94\n"
    "0	5	80	81	0.651	125.682	C94\n"
    "0	18	80	81	1.032	120.869	E94\n"
    "0	41	80	81	0.909	125.057	E94\n"
    "0	44	80	55	0.918	127.755	E94\n"
    "0	44	80	81	1.184	112.411	E94\n"
    "0	55	80	59	1.254	120.263	E94\n"
    "0	55	80	81	0.991	127.612	E94\n"
    "0	56	80	81	1.003	126.038	E94\n"
    "0	59	80	81	1.439	112.030	E94\n"
    "0	81	80	81	1.205	108.609	C94\n"
    "0	0	81	0	0.000	119.500	0:*-81-*	MMFF	DEF\n"
    "0	1	81	63	0.996	120.129	E94\n"
    "0	1	81	64	0.978	119.970	E94\n"
    "0	1	81	78	0.879	126.535	E94\n"
    "0	1	81	79	1.144	116.113	E94\n"
    "0	1	81	80	0.895	126.324	E94\n"
    "1	2	81	78	0.927	125.080	E94\n"
    "1	2	81	80	0.895	128.399	E94\n"
    "1	9	81	78	1.015	124.270	E94\n"
    "1	9	81	80	1.106	120.028	E94\n"
    "0	36	81	64	0.522	130.295	E94\n"
    "0	36	81	66	0.583	128.738	E94\n"
    "0	36	81	78	0.578	124.658	C94\n"
    "0	36	81	80	0.575	124.787	C94\n"
    "0	37	81	64	0.929	122.408	E94\n"
    "1	37	81	64	0.983	119.722	E94\n"
    "0	37	81	65	1.184	114.158	E94\n"
    "1	37	81	65	1.281	110.477	E94\n"
    "1	37	81	78	0.884	126.208	E94\n"
    "1	37	81	80	0.940	123.333	E94\n"
    "0	63	81	64	1.115	114.945	E94\n"
    "0	64	81	65	1.075	122.099	E94\n"
    "0	64	81	80	1.143	113.176	E94\n"
    "0	66	81	80	1.067	122.250	E94\n"
    "0	78	81	80	0.957	110.556	C94\n"
    "0	79	81	80	1.379	107.936	E94\n"
    "0	32	82	59	1.666	114.660	E94\n"
    "0	32	82	64	1.075	131.706	E94\n"
    "0	32	82	65	1.238	129.293	E94\n"
    "0	59	82	64	1.563	105.660	E94\n"
    "0	64	82	65	1.281	112.955	E94\n",
    "EOS"};

class std::unique_ptr<MMFFStbnCollection> MMFFStbnCollection::ds_instance = nullptr;

extern const std::string defaultMMFFStbn;

MMFFStbnCollection *MMFFStbnCollection::getMMFFStbn(
    const std::string &mmffStbn) {
  if (!ds_instance || !mmffStbn.empty()) {
    ds_instance.reset(new MMFFStbnCollection(mmffStbn));
  }
  return ds_instance.get();
}

MMFFStbnCollection::MMFFStbnCollection(std::string mmffStbn) {
  if (mmffStbn.empty()) {
    mmffStbn = defaultMMFFStbn;
  }
  std::istringstream inStream(mmffStbn);
  std::string inLine = RDKit::getLine(inStream);
  while (!(inStream.eof())) {
    if (inLine[0] != '*') {
      MMFFStbn mmffStbnObj;
      boost::char_separator<char> tabSep("\t");
      tokenizer tokens(inLine, tabSep);
      tokenizer::iterator token = tokens.begin();

#ifdef RDKIT_MMFF_PARAMS_USE_STD_MAP
      unsigned int stretchBendType = boost::lexical_cast<unsigned int>(*token);
#else
      d_stretchBendType.push_back(
          (std::uint8_t)(boost::lexical_cast<unsigned int>(*token)));
#endif
      ++token;
#ifdef RDKIT_MMFF_PARAMS_USE_STD_MAP
      unsigned int iAtomType = boost::lexical_cast<unsigned int>(*token);
#else
      d_iAtomType.push_back(
          (std::uint8_t)(boost::lexical_cast<unsigned int>(*token)));
#endif
      ++token;
#ifdef RDKIT_MMFF_PARAMS_USE_STD_MAP
      unsigned int jAtomType = boost::lexical_cast<unsigned int>(*token);
#else
      d_jAtomType.push_back(
          (std::uint8_t)(boost::lexical_cast<unsigned int>(*token)));
#endif
      ++token;
#ifdef RDKIT_MMFF_PARAMS_USE_STD_MAP
      unsigned int kAtomType = boost::lexical_cast<unsigned int>(*token);
#else
      d_kAtomType.push_back(
          (std::uint8_t)(boost::lexical_cast<unsigned int>(*token)));
#endif
      ++token;
      mmffStbnObj.kbaIJK = boost::lexical_cast<double>(*token);
      ++token;
      mmffStbnObj.kbaKJI = boost::lexical_cast<double>(*token);
      ++token;
#ifdef RDKIT_MMFF_PARAMS_USE_STD_MAP
      d_params[stretchBendType][iAtomType][jAtomType][kAtomType] = mmffStbnObj;
#else
      d_params.push_back(mmffStbnObj);
#endif
    }
    inLine = RDKit::getLine(inStream);
  }
}

const std::string defaultMMFFStbn =
    "*\n"
    "*          Copyright (c) Merck and Co., Inc., 1994, 1995, 1996\n"
    "*                         All Rights Reserved\n"
    "*\n"
    "* MMFF STRETCH-BEND PARAMETERS- Rev: 23-FEB-93  Source: MMFF\n"
    "* C94 - CORE MMFF parameter, from fits to HF/6-31G* 2nd D's\n"
    "* X94 - EXTD MMFF parameter, also from fits to HF/6-31G* 2nd D's\n"
    "*\n"
    "* types I, J, K    kbaIJK    kbaKJI   Source\n"
    "0	1	1	1	0.206	0.206	C94\n"
    "0	1	1	2	0.136	0.197	C94\n"
    "0	1	1	3	0.211	0.092	C94\n"
    "0	1	1	5	0.227	0.070	C94\n"
    "0	1	1	6	0.173	0.417	C94\n"
    "0	1	1	8	0.136	0.282	C94\n"
    "0	1	1	10	0.187	0.338	C94\n"
    "0	1	1	11	0.209	0.633	C94\n"
    "0	1	1	12	0.176	0.386	C94\n"
    "0	1	1	15	0.139	0.217	C94\n"
    "0	1	1	34	0.236	0.436	C94\n"
    "0	1	1	37	0.152	0.260	C94\n"
    "0	1	1	39	0.144	0.595	C94\n"
    "0	1	1	41	0.122	0.051	C94\n"
    "0	1	1	56	0.262	0.451	C94\n"
    "0	1	1	68	0.186	0.125	C94\n"
    "0	2	1	2	0.282	0.282	C94\n"
    "0	2	1	3	0.206	0.022	C94\n"
    "0	2	1	5	0.234	0.088	C94\n"
    "0	2	1	6	0.183	0.387	C94\n"
    "0	2	1	8	0.214	0.363	C94\n"
    "0	3	1	5	0.157	0.115	C94\n"
    "0	3	1	6	-0.036	0.456	C94\n"
    "0	3	1	10	0.038	0.195	C94\n"
    "0	5	1	5	0.115	0.115	C94\n"
    "0	5	1	6	0.013	0.436	C94\n"
    "0	5	1	8	0.027	0.358	C94\n"
    "0	5	1	9	0.040	0.418	C94\n"
    "0	5	1	10	0.043	0.261	C94\n"
    "0	5	1	11	0.003	0.452	C94\n"
    "0	5	1	12	-0.018	0.380	C94\n"
    "0	5	1	15	0.018	0.255	C94\n"
    "0	5	1	18	0.121	0.218	X94\n"
    "0	5	1	20	0.069	0.327	C94\n"
    "0	5	1	22	0.055	0.267	X94\n"
    "0	5	1	34	-0.003	0.342	C94\n"
    "0	5	1	37	0.074	0.287	C94\n"
    "0	5	1	39	0.092	0.607	C94\n"
    "0	5	1	40	0.023	0.335	C94\n"
    "0	5	1	41	0.093	0.118	C94\n"
    "0	5	1	54	0.016	0.343	C94\n"
    "0	5	1	55	0.030	0.397	C94\n"
    "0	5	1	56	0.031	0.384	C94\n"
    "0	5	1	68	0.041	0.216	C94\n"
    "0	6	1	6	0.320	0.320	C94\n"
    "0	6	1	37	0.310	0.160	C94\n"
    "0	11	1	11	0.586	0.586	C94\n"
    "0	12	1	12	0.508	0.508	C94\n"
    "0	1	2	1	0.250	0.250	C94\n"
    "0	1	2	2	0.203	0.207	C94\n"
    "2	1	2	2	0.222	0.269	C94\n"
    "2	1	2	3	0.244	0.292	C94\n"
    "0	1	2	5	0.215	0.128	C94\n"
    "2	1	2	37	0.246	0.260	C94\n"
    "1	2	2	2	0.250	0.219	C94\n"
    "2	2	2	3	0.155	0.112	C94\n"
    "0	2	2	5	0.207	0.157	C94\n"
    "1	2	2	5	0.267	0.159	C94\n"
    "0	2	2	6	0.118	0.576	C94\n"
    "2	2	2	37	0.143	0.172	C94\n"
    "0	2	2	40	0.289	0.390	C94\n"
    "0	2	2	41	0.191	-0.047	C94\n"
    "1	3	2	5	0.264	0.156	C94\n"
    "0	5	2	5	0.140	0.140	C94\n"
    "0	5	2	6	0.213	0.502	C94\n"
    "2	5	2	37	0.153	0.288	C94\n"
    "0	5	2	40	0.070	0.463	C94\n"
    "0	5	2	41	0.191	0.005	C94\n"
    "0	1	3	1	0.358	0.358	C94\n"
    "2	1	3	2	0.246	0.409	C94\n"
    "2	1	3	3	0.303	0.145	C94\n"
    "0	1	3	5	0.321	0.183	C94\n"
    "0	1	3	6	0.338	0.732	C94\n"
    "0	1	3	7	0.154	0.856	C94\n"
    "0	1	3	10	0.223	0.732	C94\n"
    "2	1	3	37	0.217	0.207	C94\n"
    "1	2	3	5	0.407	0.159	C94\n"
    "1	2	3	6	0.429	0.473	C94\n"
    "1	2	3	7	0.214	0.794	C94\n"
    "1	2	3	9	0.227	0.610	C94\n"
    "1	2	3	10	0.298	0.600	C94\n"
    "1	3	3	5	0.251	0.133	C94\n"
    "1	3	3	6	0.066	0.668	C94\n"
    "1	3	3	7	-0.093	0.866	C94\n"
    "0	5	3	5	0.126	0.126	C94\n"
    "0	5	3	6	0.174	0.734	C94\n"
    "0	5	3	7	0.032	0.805	C94\n"
    "0	5	3	9	0.037	0.669	C94\n"
    "0	5	3	10	0.169	0.619	C94\n"
    "0	5	3	40	0.087	0.685	C94\n"
    "0	5	3	54	0.098	0.210	C94\n"
    "0	6	3	7	0.494	0.578	C94\n"
    "4	6	3	20	1.179	0.752	X94\n"
    "2	6	3	37	0.350	0.175	C94\n"
    "0	7	3	10	0.771	0.353	C94\n"
    "0	7	3	20	0.865	-0.181	C94\n"
    "2	7	3	37	0.707	0.007	C94\n"
    "0	9	3	40	0.680	0.260	C94\n"
    "0	10	3	10	1.050	1.050	C94\n"
    "4	20	3	20	0.536	0.536	C94\n"
    "0	40	3	40	0.482	0.482	C94\n"
    "0	1	6	1	0.309	0.309	C94\n"
    "0	1	6	2	0.157	0.375	C94\n"
    "0	1	6	3	-0.153	0.252	C94\n"
    "0	1	6	21	0.256	0.143	C94\n"
    "0	1	6	37	0.163	0.375	C94\n"
    "0	2	6	3	-0.228	0.052	C94\n"
    "0	2	6	29	0.259	0.163	C94\n"
    "4	3	6	20	0.456	0.379	X94\n"
    "0	3	6	24	0.215	0.064	C94\n"
    "0	3	6	37	-0.225	-0.320	C94\n"
    "0	8	6	21	0.304	0.055	C94\n"
    "0	10	6	21	0.419	0.158	C94\n"
    "0	18	6	33	0.309	0.120	X94\n"
    "4	20	6	20	0.739	0.739	C94\n"
    "0	29	6	37	0.130	0.241	C94\n"
    "0	31	6	31	0.227	0.227	X94\n"
    "0	1	8	1	0.312	0.312	C94\n"
    "0	1	8	6	0.212	0.354	C94\n"
    "0	1	8	23	0.309	0.135	C94\n"
    "0	6	8	23	0.418	0.020	C94\n"
    "4	20	8	20	0.653	0.653	C94\n"
    "0	20	8	23	0.128	0.122	C94\n"
    "0	23	8	23	0.190	0.190	C94\n"
    "0	1	9	3	0.326	0.580	C94\n"
    "0	3	9	27	0.464	0.222	C94\n"
    "0	1	10	1	0.063	0.063	C94\n"
    "0	1	10	3	-0.021	0.340	C94\n"
    "0	1	10	6	-0.024	0.374	C94\n"
    "0	1	10	28	0.155	-0.051	C94\n"
    "0	3	10	3	-0.219	-0.219	C94\n"
    "0	3	10	6	0.497	0.513	C94\n"
    "0	3	10	28	0.137	0.066	C94\n"
    "0	28	10	28	0.081	0.081	C94\n"
    "0	1	15	1	0.125	0.125	C94\n"
    "0	1	15	15	0.012	0.238	C94\n"
    "0	1	15	37	0.048	0.229	C94\n"
    "0	1	15	71	0.080	-0.012	C94\n"
    "0	15	15	71	0.172	-0.068	C94\n"
    "0	37	15	71	0.187	-0.027	C94\n"
    "0	71	15	71	0.045	0.045	C94\n"
    "0	1	18	1	0.023	0.023	X94\n"
    "0	1	18	6	0.003	0.213	X94\n"
    "0	1	18	32	-0.091	0.390	X94\n"
    "0	1	18	43	-0.008	0.607	X94\n"
    "0	6	18	6	0.088	0.088	X94\n"
    "0	6	18	32	0.123	0.369	X94\n"
    "0	32	18	32	0.404	0.404	X94\n"
    "0	32	18	43	0.384	0.281	X94\n"
    "0	43	18	43	0.428	0.428	X94\n"
    "0	1	20	5	0.290	0.098	C94\n"
    "0	1	20	20	0.179	0.004	C94\n"
    "0	3	20	5	-0.049	0.171	C94\n"
    "4	3	20	20	0.607	0.437	C94\n"
    "0	5	20	5	0.182	0.182	C94\n"
    "0	5	20	6	0.051	0.312	C94\n"
    "0	5	20	8	0.072	0.226	C94\n"
    "0	5	20	12	0.014	0.597	C94\n"
    "0	5	20	20	0.101	0.079	C94\n"
    "0	5	20	30	0.108	0.123	C94\n"
    "4	6	20	20	0.823	0.396	C94\n"
    "4	8	20	20	0.701	0.369	C94\n"
    "0	12	20	20	0.310	0.000	C94\n"
    "4	20	20	20	0.283	0.283	C94\n"
    "4	20	20	30	0.340	0.529	C94\n"
    "0	1	22	5	0.067	0.174	X94\n"
    "0	1	22	22	0.199	0.039	X94\n"
    "0	5	22	5	0.254	0.254	C94\n"
    "0	5	22	22	0.181	0.108	C94\n"
    "5	22	22	22	0.000	0.000	C94\n"
    "0	5	26	5	-0.121	-0.121	X94\n"
    "0	5	30	20	0.251	0.007	C94\n"
    "0	5	30	30	0.267	0.054	C94\n"
    "4	20	30	30	0.413	0.705	C94\n"
    "0	1	34	1	0.202	0.202	C94\n"
    "0	1	34	36	0.160	-0.009	C94\n"
    "0	36	34	36	0.087	0.087	C94\n"
    "0	1	37	37	0.485	0.311	C94\n"
    "1	2	37	37	0.321	0.235	C94\n"
    "1	3	37	37	0.179	0.217	C94\n"
    "0	5	37	37	0.279	0.250	C94\n"
    "0	5	37	38	0.267	0.389	C94\n"
    "0	5	37	63	0.216	0.434	C94\n"
    "0	5	37	64	0.167	0.364	C94\n"
    "0	5	37	69	0.273	0.391	C94\n"
    "0	6	37	37	0.830	0.339	C94\n"
    "0	15	37	37	0.650	0.259	C94\n"
    "0	37	37	37	-0.411	-0.411	C94\n"
    "0	37	37	38	-0.424	-0.466	C94\n"
    "0	37	37	40	0.429	0.901	C94\n"
    "0	37	37	63	-0.173	-0.215	C94\n"
    "0	37	37	64	-0.229	-0.229	C94\n"
    "0	37	37	69	-0.244	-0.555	C94\n"
    "0	38	37	38	-0.516	-0.516	C94\n"
    "0	37	38	37	-0.342	-0.342	C94\n"
    "0	37	38	38	-0.164	-1.130	C94\n"
    "0	1	39	63	0.313	0.500	C94\n"
    "0	23	39	63	-0.131	0.422	C94\n"
    "0	23	39	65	-0.122	0.281	C94\n"
    "0	63	39	63	0.469	0.469	C94\n"
    "0	63	39	65	0.741	0.506	C94\n"
    "0	65	39	65	0.706	0.706	C94\n"
    "0	1	40	28	0.238	0.091	C94\n"
    "0	1	40	37	0.153	0.590	C94\n"
    "0	2	40	28	0.342	0.156	C94\n"
    "0	3	40	28	0.228	0.104	C94\n"
    "0	28	40	28	0.094	0.094	C94\n"
    "0	28	40	37	0.186	0.423	C94\n"
    "0	1	41	32	0.503	0.943	C94\n"
    "0	2	41	32	0.594	0.969	C94\n"
    "0	5	41	32	0.276	0.852	C94\n"
    "0	32	41	32	0.652	0.652	C94\n"
    "0	18	43	23	0.377	0.057	X94\n"
    "0	23	43	23	0.082	0.082	X94\n"
    "0	63	44	63	0.591	0.591	C94\n"
    "0	63	44	65	0.857	0.978	C94\n"
    "0	50	49	50	0.072	0.072	C94\n"
    "0	1	54	3	0.192	-0.051	C94\n"
    "0	1	54	36	0.240	0.079	C94\n"
    "0	3	54	36	0.005	0.127	C94\n"
    "0	36	54	36	0.148	0.148	C94\n"
    "0	1	55	36	0.189	0.033	C94\n"
    "0	1	55	57	0.166	0.211	C94\n"
    "0	36	55	36	0.106	0.106	C94\n"
    "0	36	55	57	0.093	0.080	C94\n"
    "0	1	56	36	0.211	-0.040	C94\n"
    "0	1	56	57	0.026	0.386	C94\n"
    "0	36	56	36	0.101	0.101	C94\n"
    "0	36	56	57	0.108	0.068	C94\n"
    "0	5	57	55	0.043	0.420	C94\n"
    "0	55	57	55	0.125	0.125	C94\n"
    "0	56	57	56	0.431	0.431	C94\n"
    "0	58	57	58	0.732	0.732	C94\n"
    "0	63	59	63	0.497	0.497	C94\n"
    "0	63	59	65	0.723	0.874	C94\n"
    "0	5	63	39	0.009	0.654	C94\n"
    "0	5	63	44	-0.015	0.446	C94\n"
    "0	5	63	59	0.067	0.588	C94\n"
    "0	5	63	64	0.055	0.370	C94\n"
    "0	5	63	66	0.110	0.464	C94\n"
    "0	37	63	39	0.178	0.523	C94\n"
    "0	37	63	64	-0.045	0.497	C94\n"
    "0	39	63	64	0.422	0.409	C94\n"
    "0	39	63	66	0.436	0.525	C94\n"
    "0	44	63	64	0.581	0.426	C94\n"
    "0	44	63	66	0.542	0.365	C94\n"
    "0	59	63	64	0.852	0.332	C94\n"
    "0	59	63	66	0.775	0.300	C94\n"
    "0	5	64	63	0.086	0.345	C94\n"
    "0	5	64	64	0.085	0.369	C94\n"
    "0	5	64	65	0.051	0.436	C94\n"
    "0	5	64	66	0.113	0.452	C94\n"
    "0	37	64	63	0.059	0.299	C94\n"
    "0	37	64	64	0.277	0.377	C94\n"
    "0	63	64	64	0.206	0.030	C94\n"
    "0	63	64	66	0.171	0.078	C94\n"
    "0	64	64	65	0.079	0.403	C94\n"
    "0	65	64	66	0.406	0.066	C94\n"
    "0	39	65	64	0.528	0.644	C94\n"
    "0	39	65	66	0.397	0.258	C94\n"
    "0	44	65	64	0.816	0.543	C94\n"
    "0	59	65	64	1.177	0.594	C94\n"
    "0	63	66	64	0.213	-0.173	C94\n"
    "0	63	66	66	0.234	0.077	C94\n"
    "0	64	66	65	-0.149	0.383	C94\n"
    "0	65	66	66	0.199	0.101	C94\n"
    "0	1	68	1	0.217	0.217	C94\n"
    "0	1	68	23	0.285	0.050	C94\n"
    "0	1	68	32	-0.047	0.503	C94\n"
    "0	23	68	23	0.145	0.145	C94\n"
    "0	23	68	32	-0.182	0.504	C94\n"
    "0	32	69	37	1.018	0.418	C94\n"
    "0	37	69	37	-0.169	-0.169	C94\n"
    "0	31	70	31	0.210	0.210	C94\n"
    "0	5	78	78	0.279	0.250	C94\n"
    "0	5	78	81	0.083	0.250	C94\n"
    "0	78	78	81	-0.398	0.314	C94\n"
    "0	5	80	81	-0.101	0.691	C94\n"
    "0	81	80	81	0.732	0.732	C94\n"
    "0	36	81	78	0.021	0.368	C94\n"
    "0	36	81	80	0.018	0.422	C94\n"
    "0	78	81	80	0.366	0.419	C94\n";

class std::unique_ptr<MMFFDfsbCollection> MMFFDfsbCollection::ds_instance = nullptr;

extern const std::string defaultMMFFDfsb;

MMFFDfsbCollection *MMFFDfsbCollection::getMMFFDfsb(
    const std::string &mmffDfsb) {
  if (!ds_instance || !mmffDfsb.empty()) {
    ds_instance.reset(new MMFFDfsbCollection(mmffDfsb));
  }
  return ds_instance.get();
}

MMFFDfsbCollection::MMFFDfsbCollection(std::string mmffDfsb) {
  if (mmffDfsb.empty()) {
    mmffDfsb = defaultMMFFDfsb;
  }
  std::istringstream inStream(mmffDfsb);
  std::string inLine = RDKit::getLine(inStream);
  while (!(inStream.eof())) {
    if (inLine[0] != '*') {
      MMFFStbn mmffStbnObj;
      boost::char_separator<char> tabSep("\t");
      tokenizer tokens(inLine, tabSep);
      tokenizer::iterator token = tokens.begin();

      unsigned int iAtomicNum = boost::lexical_cast<unsigned int>(*token);
      ++token;
      unsigned int jAtomicNum = boost::lexical_cast<unsigned int>(*token);
      ++token;
      unsigned int kAtomicNum = boost::lexical_cast<unsigned int>(*token);
      ++token;
      mmffStbnObj.kbaIJK = boost::lexical_cast<double>(*token);
      ++token;
      mmffStbnObj.kbaKJI = boost::lexical_cast<double>(*token);
      ++token;
      d_params[iAtomicNum][jAtomicNum][kAtomicNum] = mmffStbnObj;
    }
    inLine = RDKit::getLine(inStream);
  }
}
const std::string defaultMMFFDfsb =
    "*\n"
    "*          Copyright (c) Merck and Co., Inc., 1994, 1995, 1996\n"
    "*                         All Rights Reserved\n"
    "*\n"
    "* MMFF Default Stretch-Bend Parameters\n"
    "*     Row in \n"
    "*  Periodic Table         \n"
    "*   IR   JR   KR   F(I_J,K)  F(K_J,I)\n"
    "0	1	0	0.15	0.15\n"
    "0	1	1	0.10	0.30\n"
    "0	1	2	0.05	0.35\n"
    "0	1	3	0.05	0.35\n"
    "0	1	4	0.05	0.35\n"
    "0	2	0	0.00	0.00\n"
    "0	2	1	0.00	0.15\n"
    "0	2	2	0.00	0.15\n"
    "0	2	3	0.00	0.15\n"
    "0	2	4	0.00	0.15\n"
    "1	1	1	0.30	0.30\n"
    "1	1	2	0.30	0.50\n"
    "1	1	3	0.30	0.50\n"
    "1	1	4	0.30	0.50\n"
    "2	1	2	0.50	0.50\n"
    "2	1	3	0.50	0.50\n"
    "2	1	4	0.50	0.50\n"
    "3	1	3	0.50	0.50\n"
    "3	1	4	0.50	0.50\n"
    "4	1	4	0.50	0.50\n"
    "1	2	1	0.30	0.30\n"
    "1	2	2	0.25	0.25\n"
    "1	2	3	0.25	0.25\n"
    "1	2	4	0.25	0.25\n"
    "2	2	2	0.25	0.25\n"
    "2	2	3	0.25	0.25\n"
    "2	2	4	0.25	0.25\n"
    "3	2	3	0.25	0.25\n"
    "3	2	4	0.25	0.25\n"
    "4	2	4	0.25	0.25\n";

class std::unique_ptr<MMFFOopCollection> MMFFOopCollection::ds_instance[2] = {nullptr, nullptr};

extern const std::string defaultMMFFOop;
extern const std::string defaultMMFFsOop;

MMFFOopCollection *MMFFOopCollection::getMMFFOop(const bool isMMFFs,
                                                 const std::string &mmffOop) {
  unsigned int i = (isMMFFs ? 1 : 0);
  if (!ds_instance[i] || !mmffOop.empty()) {
    ds_instance[i] = std::unique_ptr<MMFFOopCollection>(new MMFFOopCollection(isMMFFs, mmffOop));
  }
  return ds_instance[i].get();
}

MMFFOopCollection::MMFFOopCollection(const bool isMMFFs, std::string mmffOop) {
  if (mmffOop.empty()) {
    mmffOop = (isMMFFs ? defaultMMFFsOop : defaultMMFFOop);
  }
  std::istringstream inStream(mmffOop);
  std::string inLine = RDKit::getLine(inStream);
  while (!(inStream.eof())) {
    if (inLine[0] != '*') {
      MMFFOop mmffOopObj;
      boost::char_separator<char> tabSep("\t");
      tokenizer tokens(inLine, tabSep);
      tokenizer::iterator token = tokens.begin();

#ifdef RDKIT_MMFF_PARAMS_USE_STD_MAP
      unsigned int iAtomType = boost::lexical_cast<unsigned int>(*token);
#else
      d_iAtomType.push_back(
          (std::uint8_t)(boost::lexical_cast<unsigned int>(*token)));
#endif
      ++token;
#ifdef RDKIT_MMFF_PARAMS_USE_STD_MAP
      unsigned int jAtomType = boost::lexical_cast<unsigned int>(*token);
#else
      d_jAtomType.push_back(
          (std::uint8_t)(boost::lexical_cast<unsigned int>(*token)));
#endif
      ++token;
#ifdef RDKIT_MMFF_PARAMS_USE_STD_MAP
      unsigned int kAtomType = boost::lexical_cast<unsigned int>(*token);
#else
      d_kAtomType.push_back(
          (std::uint8_t)(boost::lexical_cast<unsigned int>(*token)));
#endif
      ++token;
#ifdef RDKIT_MMFF_PARAMS_USE_STD_MAP
      unsigned int lAtomType = boost::lexical_cast<unsigned int>(*token);
#else
      d_lAtomType.push_back(
          (std::uint8_t)(boost::lexical_cast<unsigned int>(*token)));
#endif
      ++token;
      mmffOopObj.koop = boost::lexical_cast<double>(*token);
      ++token;
#ifdef RDKIT_MMFF_PARAMS_USE_STD_MAP
      d_params[iAtomType][jAtomType][kAtomType][lAtomType] = mmffOopObj;
#else
      d_params.push_back(mmffOopObj);
#endif
    }
    inLine = RDKit::getLine(inStream);
  }
}

const std::string defaultMMFFOop =
    "*\n"
    "*          Copyright (c) Merck and Co., Inc., 1994, 1995, 1996\n"
    "*                         All Rights Reserved\n"
    "*\n"
    "* MMFF OUT-OF-PLANE PARAMETERS- Rev: 15-OCT-93  Source: MMFF\n"
    "* C94  - CORE MMFF parameter, from fits to HF/6-31G* 2nd D's\n"
    "* #C94 - Value adjusted from CORE MMFF93 value\n"
    "*\n"
    "*   MMFF  atom types     koop    Source\n"
    "0	2	0	0	0.020	*-2-*-*	C94	DEF\n"
    "1	2	1	2	0.030	C94\n"
    "1	2	2	2	0.027	C94\n"
    "1	2	2	3	0.026	C94\n"
    "1	2	2	5	0.013	C94\n"
    "1	2	2	37	0.032	C94\n"
    "2	2	2	5	0.013	C94\n"
    "2	2	3	5	0.012	C94\n"
    "2	2	5	5	0.006	C94\n"
    "2	2	5	6	0.027	C94\n"
    "2	2	5	37	0.017	C94\n"
    "2	2	5	40	0.012	C94\n"
    "2	2	5	41	0.008	C94\n"
    "0	3	0	0	0.130	*-3-*-*	C94	DEF\n"
    "1	3	1	7	0.146	C94\n"
    "1	3	2	7	0.138	C94\n"
    "1	3	3	7	0.134	C94\n"
    "1	3	5	7	0.122	C94\n"
    "1	3	6	7	0.141	C94\n"
    "1	3	7	10	0.129	C94\n"
    "1	3	7	37	0.138	C94\n"
    "2	3	5	7	0.113	C94\n"
    "2	3	5	9	0.081	C94\n"
    "2	3	6	7	0.127	C94\n"
    "2	3	7	10	0.116	C94\n"
    "3	3	5	7	0.113	C94\n"
    "3	3	6	7	0.127	C94\n"
    "5	3	5	7	0.103	C94\n"
    "5	3	5	9	0.074	C94\n"
    "5	3	5	54	0.078	C94\n"
    "5	3	6	7	0.119	C94\n"
    "5	3	7	10	0.102	C94\n"
    "5	3	9	40	0.067	C94\n"
    "6	3	7	37	0.127	C94\n"
    "7	3	10	10	0.113	C94\n"
    "7	3	20	20	0.151	C94\n"
    "9	3	40	40	0.057	C94\n"
    "0	8	0	0	0.000	*-8-*-*	C94	DEF\n"
    "0	10	0	0	-0.020	*-10-*-*	C94	DEF\n"
    "1	10	1	3	-0.02	#C94	(C93=-0.034)\n"
    "1	10	3	6	-0.033	C94\n"
    "1	10	3	28	-0.02	#C94	(C93=-0.030)\n"
    "3	10	3	28	-0.030	C94\n"
    "3	10	28	28	-0.019	C94\n"
    "0	17	0	0	0.000	*-17-*-*	E94	DEF\n"
    "0	26	0	0	0.000	*-26-*-*	CE4	DEF\n"
    "0	30	0	0	0.010	*-30-*-*	C94	DEF\n"
    "5	30	20	30	0.008	C94\n"
    "0	37	0	0	0.035	*-37-*-*	C94	DEF\n"
    "1	37	37	37	0.040	C94\n"
    "2	37	37	37	0.031	C94\n"
    "3	37	37	37	0.027	C94\n"
    "5	37	37	37	0.015	C94\n"
    "5	37	37	38	0.046	C94\n"
    "5	37	37	63	0.008	C94\n"
    "5	37	37	64	0.012	C94\n"
    "5	37	37	69	0.016	C94\n"
    "5	37	38	38	0.084	C94\n"
    "6	37	37	37	0.048	C94\n"
    "15	37	37	37	0.025	C94\n"
    "37	37	37	40	0.046	C94\n"
    "0	39	0	0	0.020	*-39-*-*	C94	DEF\n"
    "1	39	63	63	0.012	C94\n"
    "23	39	63	63	-0.014	C94\n"
    "23	39	63	65	0.021	C94\n"
    "23	39	65	65	0.062	C94\n"
    "0	40	0	0	-0.005	*-40-*-*	C94	DEF\n"
    "1	40	28	37	-0.006	C94\n"
    "2	40	28	28	-0.007	C94\n"
    "3	40	28	28	-0.007	C94\n"
    "28	40	28	37	0.004	C94\n"
    "0	41	0	0	0.180	*-41-*-*	C94	DEF\n"
    "1	41	32	32	0.178	C94\n"
    "2	41	32	32	0.161	C94\n"
    "5	41	32	32	0.158	C94\n"
    "0	43	0	0	0.000	*-43-*-*	E94	DEF\n"
    "0	45	0	0	0.150	*-45-*-*	E94	DEF\n"
    "0	49	0	0	0.000	*-49-*-*	E94	DEF\n"
    "50	49	50	50	0.000	#C94\n"
    "0	54	0	0	0.020	*-54-*-*	C94	DEF\n"
    "1	54	3	36	0.016	C94\n"
    "3	54	36	36	0.018	C94\n"
    "0	55	0	0	0.020	*-55-*-*	C94	DEF\n"
    "1	55	36	57	0.020	#C94\n"
    "36	55	36	57	0.020	#C94\n"
    "0	56	0	0	0.020	*-56-*-*	C94	DEF\n"
    "1	56	36	57	0.020	#C94\n"
    "36	56	36	57	0.020	#C94\n"
    "0	57	0	0	0.080	*-57-*-*	C94	DEF\n"
    "5	57	55	55	0.038	C94\n"
    "56	57	56	56	0.158	C94\n"
    "0	58	0	0	0.025	*-58-*-*	E94	DEF\n"
    "0	63	0	0	0.050	*-63-*-*	C94	DEF\n"
    "5	63	39	64	0.019	C94\n"
    "5	63	39	66	0.068	C94\n"
    "5	63	44	64	0.014	C94\n"
    "5	63	44	66	0.055	C94\n"
    "5	63	59	64	0.033	C94\n"
    "5	63	59	66	0.085	C94\n"
    "37	63	39	64	0.010	C94\n"
    "0	64	0	0	0.040	*-64-*-*	C94	DEF\n"
    "5	64	63	64	0.006	C94\n"
    "5	64	63	66	0.043	C94\n"
    "5	64	64	65	0.052	C94\n"
    "5	64	65	66	0.094	C94\n"
    "37	64	63	64	-0.011	C94\n"
    "0	67	0	0	0.070	*-67-*-*	E94	DEF\n"
    "0	69	0	0	0.070	*-69-*-*	C94	DEF\n"
    "32	69	37	37	0.067	C94\n"
    "0	73	0	0	0.000	*-73-*-*	E94	DEF\n"
    "0	78	0	0	0.045	*-78-*-*	C94	DEF\n"
    "5	78	78	81	0.046	C94\n"
    "0	80	0	0	0.080	*-80-*-*	C94	DEF\n"
    "5	80	81	81	0.057	C94\n"
    "0	81	0	0	0.025	*-81-*-*	C94	DEF\n"
    "36	81	78	80	0.016	C94\n"
    "0	82	0	0	0.000	*-82-*-*	E94	DEF\n";
const std::string defaultMMFFsOop =
    "*\n"
    "*   Copyright (c) Merck and Co., Inc., 1994,1995,1996,1997,1998,1999\n"
    "*                         All Rights Reserved\n"
    "*\n"
    "* MMFF94s OUT-OF-PLANE PARAMETERS\n"
    "* C94  - CORE MMFF94 parameter, from fits to HF/6-31G* 2nd D's\n"
    "* C94S - CORE MMFF94s parameter,adjusted to promote planarity\n"
    "* #C94 - Value adjusted from CORE MMFF94 value\n"
    "*\n"
    "*   MMFF  atom types     koop    Source\n"
    "0	2	0	0	0.020	*-2-*-*	C94	DEF\n"
    "1	2	1	2	0.030	C94\n"
    "1	2	2	2	0.027	C94\n"
    "1	2	2	3	0.026	C94\n"
    "1	2	2	5	0.013	C94\n"
    "1	2	2	37	0.032	C94\n"
    "2	2	2	5	0.013	C94\n"
    "2	2	3	5	0.012	C94\n"
    "2	2	5	5	0.006	C94\n"
    "2	2	5	6	0.027	C94\n"
    "2	2	5	37	0.017	C94\n"
    "2	2	5	40	0.012	C94\n"
    "2	2	5	41	0.008	C94\n"
    "0	3	0	0	0.130	*-3-*-*	C94	DEF\n"
    "1	3	1	7	0.146	C94\n"
    "1	3	2	7	0.138	C94\n"
    "1	3	3	7	0.134	C94\n"
    "1	3	5	7	0.122	C94\n"
    "1	3	6	7	0.141	C94\n"
    "1	3	7	10	0.129	C94\n"
    "1	3	7	37	0.138	C94\n"
    "2	3	5	7	0.113	C94\n"
    "2	3	5	9	0.081	C94\n"
    "2	3	6	7	0.127	C94\n"
    "2	3	7	10	0.116	C94\n"
    "3	3	5	7	0.113	C94\n"
    "3	3	6	7	0.127	C94\n"
    "5	3	5	7	0.103	C94\n"
    "5	3	5	9	0.074	C94\n"
    "5	3	5	54	0.078	C94\n"
    "5	3	6	7	0.119	C94\n"
    "5	3	7	10	0.102	C94\n"
    "5	3	9	40	0.067	C94\n"
    "6	3	7	37	0.127	C94\n"
    "7	3	10	10	0.113	C94\n"
    "7	3	20	20	0.151	C94\n"
    "9	3	40	40	0.057	C94\n"
    "0	8	0	0	0.000	*-8-*-*	C94	DEF\n"
    "0	10	0	0	0.015	*-10-*-*	C94S	DEF\n"
    "1	10	1	3	0.015	C94S\n"
    "1	10	3	6	0.015	C94\n"
    "1	10	3	28	0.015	C94S\n"
    "3	10	3	28	0.015	C94S\n"
    "3	10	28	28	0.015	C94S\n"
    "0	17	0	0	0.000	*-17-*-*	E94	DEF\n"
    "0	26	0	0	0.000	*-26-*-*	CE4	DEF\n"
    "0	30	0	0	0.010	*-30-*-*	C94	DEF\n"
    "5	30	20	30	0.008	C94\n"
    "0	37	0	0	0.035	*-37-*-*	C94	DEF\n"
    "1	37	37	37	0.040	C94\n"
    "2	37	37	37	0.031	C94\n"
    "3	37	37	37	0.027	C94\n"
    "5	37	37	37	0.015	C94\n"
    "5	37	37	38	0.046	C94\n"
    "5	37	37	63	0.008	C94\n"
    "5	37	37	64	0.012	C94\n"
    "5	37	37	69	0.016	C94\n"
    "5	37	38	38	0.084	C94\n"
    "6	37	37	37	0.048	C94\n"
    "15	37	37	37	0.025	C94\n"
    "37	37	37	40	0.046	C94\n"
    "0	39	0	0	0.020	*-39-*-*	C94	DEF\n"
    "1	39	63	63	0.012	C94\n"
    "23	39	63	63	-0.014	C94\n"
    "23	39	63	65	0.021	C94\n"
    "23	39	65	65	0.062	C94\n"
    "0	40	0	0	0.030	*-40-*-*	C94S	DEF\n"
    "1	40	28	37	0.030	C94S\n"
    "2	40	28	28	0.030	C94S\n"
    "3	40	28	28	0.030	C94S\n"
    "28	40	28	37	0.030	C94S\n"
    "0	41	0	0	0.180	*-41-*-*	C94	DEF\n"
    "1	41	32	32	0.178	C94\n"
    "2	41	32	32	0.161	C94\n"
    "5	41	32	32	0.158	C94\n"
    "0	43	0	0	0.000	*-43-*-*	E94	DEF\n"
    "0	45	0	0	0.150	*-45-*-*	E94	DEF\n"
    "0	49	0	0	0.000	*-49-*-*	E94	DEF\n"
    "50	49	50	50	0.000	#C94\n"
    "0	54	0	0	0.020	*-54-*-*	C94	DEF\n"
    "1	54	3	36	0.016	C94\n"
    "3	54	36	36	0.018	C94\n"
    "0	55	0	0	0.020	*-55-*-*	C94	DEF\n"
    "1	55	36	57	0.020	#C94\n"
    "36	55	36	57	0.020	#C94\n"
    "0	56	0	0	0.020	*-56-*-*	C94	DEF\n"
    "1	56	36	57	0.020	#C94\n"
    "36	56	36	57	0.020	#C94\n"
    "0	57	0	0	0.080	*-57-*-*	C94	DEF\n"
    "5	57	55	55	0.038	C94\n"
    "56	57	56	56	0.158	C94\n"
    "0	58	0	0	0.025	*-58-*-*	E94	DEF\n"
    "0	63	0	0	0.050	*-63-*-*	C94	DEF\n"
    "5	63	39	64	0.019	C94\n"
    "5	63	39	66	0.068	C94\n"
    "5	63	44	64	0.014	C94\n"
    "5	63	44	66	0.055	C94\n"
    "5	63	59	64	0.033	C94\n"
    "5	63	59	66	0.085	C94\n"
    "37	63	39	64	0.010	C94\n"
    "0	64	0	0	0.040	*-64-*-*	C94	DEF\n"
    "5	64	63	64	0.006	C94\n"
    "5	64	63	66	0.043	C94\n"
    "5	64	64	65	0.052	C94\n"
    "5	64	65	66	0.094	C94\n"
    "37	64	63	64	-0.011	C94\n"
    "0	67	0	0	0.070	*-67-*-*	E94	DEF\n"
    "0	69	0	0	0.070	*-69-*-*	C94	DEF\n"
    "32	69	37	37	0.067	C94\n"
    "0	73	0	0	0.000	*-73-*-*	E94	DEF\n"
    "0	78	0	0	0.045	*-78-*-*	C94	DEF\n"
    "5	78	78	81	0.046	C94\n"
    "0	80	0	0	0.080	*-80-*-*	C94	DEF\n"
    "5	80	81	81	0.057	C94\n"
    "0	81	0	0	0.025	*-81-*-*	C94	DEF\n"
    "36	81	78	80	0.016	C94\n"
    "0	82	0	0	0.000	*-82-*-*	E94	DEF\n";

class std::unique_ptr<MMFFTorCollection> MMFFTorCollection::ds_instance[2] = {nullptr, nullptr};

extern const std::string defaultMMFFTor;
extern const std::string defaultMMFFsTor;

MMFFTorCollection *MMFFTorCollection::getMMFFTor(const bool isMMFFs,
                                                 const std::string &mmffTor) {
  unsigned int i = (isMMFFs ? 1 : 0);
  if (!ds_instance[i] || !mmffTor.empty()) {
    ds_instance[i] = std::unique_ptr<MMFFTorCollection>(new MMFFTorCollection(isMMFFs, mmffTor));
  }
  return ds_instance[i].get();
}

MMFFTorCollection::MMFFTorCollection(const bool isMMFFs, std::string mmffTor) {
  if (mmffTor.empty()) {
    mmffTor = (isMMFFs ? defaultMMFFsTor : defaultMMFFTor);
  }
  std::istringstream inStream(mmffTor);
  std::string inLine = RDKit::getLine(inStream);
  while (!(inStream.eof())) {
    if (inLine[0] != '*') {
      MMFFTor mmffTorObj;
      boost::char_separator<char> tabSep("\t");
      tokenizer tokens(inLine, tabSep);
      tokenizer::iterator token = tokens.begin();

#ifdef RDKIT_MMFF_PARAMS_USE_STD_MAP
      unsigned int torType = boost::lexical_cast<unsigned int>(*token);
#else
      d_torType.push_back(
          (std::uint8_t)(boost::lexical_cast<unsigned int>(*token)));
#endif
      ++token;
#ifdef RDKIT_MMFF_PARAMS_USE_STD_MAP
      unsigned int iAtomType = boost::lexical_cast<unsigned int>(*token);
#else
      d_iAtomType.push_back(
          (std::uint8_t)(boost::lexical_cast<unsigned int>(*token)));
#endif
      ++token;
#ifdef RDKIT_MMFF_PARAMS_USE_STD_MAP
      unsigned int jAtomType = boost::lexical_cast<unsigned int>(*token);
#else
      d_jAtomType.push_back(
          (std::uint8_t)(boost::lexical_cast<unsigned int>(*token)));
#endif
      ++token;
#ifdef RDKIT_MMFF_PARAMS_USE_STD_MAP
      unsigned int kAtomType = boost::lexical_cast<unsigned int>(*token);
#else
      d_kAtomType.push_back(
          (std::uint8_t)(boost::lexical_cast<unsigned int>(*token)));
#endif
      ++token;
#ifdef RDKIT_MMFF_PARAMS_USE_STD_MAP
      unsigned int lAtomType = boost::lexical_cast<unsigned int>(*token);
#else
      d_lAtomType.push_back(
          (std::uint8_t)(boost::lexical_cast<unsigned int>(*token)));
#endif
      ++token;
      mmffTorObj.V1 = boost::lexical_cast<double>(*token);
      ++token;
      mmffTorObj.V2 = boost::lexical_cast<double>(*token);
      ++token;
      mmffTorObj.V3 = boost::lexical_cast<double>(*token);
      ++token;
#ifdef RDKIT_MMFF_PARAMS_USE_STD_MAP
      d_params[torType][iAtomType][jAtomType][kAtomType][lAtomType] =
          mmffTorObj;
#else
      d_params.push_back(mmffTorObj);
#endif
    }
    inLine = RDKit::getLine(inStream);
  }
}

const std::string defaultMMFFTor =
    "*\n"
    "*           Copyright (c) Merck and Co., Inc., 1994, 1995, 1996\n"
    "*                           All Rights Reserved\n"
    "*\n"
    "* MMFF TORSION PARAMETERS- Rev:  26-OCT-94  Source: MMFF\n"
    "* C94  - CORE MMFF parameter - from fits to conformational energies\n"
    "* X94  - EXTD MMFF parameter - also from fits to conformational E's\n"
    "* E94  - EXTD MMFF parameter - from empirical rule\n"
    "* #E94 - Adjusted from empirical rule value\n"
    "*\n"
    "*      atom types       V1      V2      V3     Source\n"
    "0	0	1	1	0	0.000	0.000	0.300	C94	"
    "0:*-1-1-*	Def\n"
    "5	0	1	1	0	0.200	-0.800	1.500	C94	"
    "5:*-1-1-*	Def\n"
    "0	1	1	1	1	0.103	0.681	0.332	C94\n"
    "5	1	1	1	1	0.144	-0.547	1.126	C94\n"
    "0	1	1	1	2	-0.295	0.438	0.584	C94\n"
    "0	1	1	1	3	0.066	-0.156	0.143	C94\n"
    "0	1	1	1	5	0.639	-0.630	0.264	C94\n"
    "0	1	1	1	6	-0.688	1.757	0.477	C94\n"
    "5	1	1	1	6	0.000	0.000	0.054	C94\n"
    "0	1	1	1	8	-1.420	-0.092	1.101	C94\n"
    "5	1	1	1	8	0.000	-0.158	0.323	C94\n"
    "0	1	1	1	11	0.593	0.662	1.120	C94\n"
    "0	1	1	1	12	-0.678	0.417	0.624	C94\n"
    "0	1	1	1	15	-0.714	0.698	0.000	C94\n"
    "0	1	1	1	34	-0.647	0.550	0.590	C94\n"
    "0	2	1	1	5	0.321	-0.411	0.144	C94\n"
    "0	3	1	1	3	0.443	0.000	-1.140	C94\n"
    "0	3	1	1	5	-0.256	0.058	0.000	C94\n"
    "0	3	1	1	6	-0.679	-0.029	0.000	C94\n"
    "0	5	1	1	5	0.284	-1.386	0.314	C94\n"
    "0	5	1	1	6	-0.654	1.072	0.279	C94\n"
    "0	5	1	1	8	-0.744	-1.235	0.337	C94\n"
    "0	5	1	1	10	0.000	0.000	0.427	C94\n"
    "0	5	1	1	11	0.000	0.516	0.291	C94\n"
    "0	5	1	1	12	0.678	-0.602	0.398	C94\n"
    "0	5	1	1	15	1.142	-0.644	0.367	C94\n"
    "0	5	1	1	25	0.000	0.000	0.295	X94\n"
    "0	5	1	1	34	0.692	-0.530	0.278	C94\n"
    "0	5	1	1	37	0.000	0.000	0.389	C94\n"
    "0	5	1	1	39	0.000	0.000	0.278	C94\n"
    "0	5	1	1	41	0.000	0.000	-0.141	C94\n"
    "0	5	1	1	56	0.000	0.000	0.324	C94\n"
    "0	5	1	1	68	0.000	0.000	0.136	C94\n"
    "0	6	1	1	6	0.408	1.397	0.961	C94\n"
    "5	6	1	1	6	0.313	-1.035	1.631	C94\n"
    "0	8	1	1	8	1.055	0.834	0.000	C94\n"
    "0	11	1	1	11	-0.387	-0.543	1.405	C94\n"
    "0	12	1	1	12	0.000	0.000	0.893	C94\n"
    "0	15	1	1	15	-0.177	0.000	0.049	C94\n"
    "0	0	1	2	0	0.000	0.000	0.000	C94	"
    "0:*-1-2-*	Def\n"
    "2	0	1	2	0	0.000	0.000	0.000	E94	"
    "2:*1-2-*	Def\n"
    "5	0	1	2	0	0.000	0.000	0.000	C94	"
    "5:*-1-2-*	Def\n"
    "0	0	1	2	2	0.000	0.000	-0.650	C94	"
    "0:*-1-2=2	Def\n"
    "5	0	1	2	2	0.000	0.000	-0.650	C94	"
    "5:*-1-2=2	Def\n"
    "0	1	1	2	1	0.419	0.296	0.282	C94\n"
    "0	1	1	2	2	-0.494	0.274	-0.630	C94\n"
    "0	1	1	2	5	0.075	0.000	0.358	C94\n"
    "0	2	1	2	2	-0.293	0.115	-0.508	C94\n"
    "0	2	1	2	5	0.301	0.104	0.507	C94\n"
    "0	3	1	2	1	0.565	-0.554	0.234	C94\n"
    "0	3	1	2	2	-0.577	-0.482	-0.427	C94\n"
    "0	3	1	2	5	0.082	0.000	0.123	C94\n"
    "0	5	1	2	1	0.000	-0.184	0.220	C94\n"
    "0	5	1	2	2	0.501	-0.410	-0.535	C94\n"
    "2	5	1	2	2	0.000	0.000	0.055	C94\n"
    "2	5	1	2	3	0.000	0.000	-0.108	C94\n"
    "0	5	1	2	5	-0.523	-0.228	0.208	C94\n"
    "2	5	1	2	37	0.000	0.000	0.000	C94\n"
    "0	6	1	2	1	-0.467	0.000	0.490	C94\n"
    "0	6	1	2	2	0.425	0.168	-0.875	C94\n"
    "0	6	1	2	5	0.000	0.136	0.396	C94\n"
    "0	8	1	2	1	-0.504	0.371	0.557	C94\n"
    "0	8	1	2	2	0.541	0.539	-1.009	C94\n"
    "0	8	1	2	5	0.000	0.204	0.464	C94\n"
    "0	0	1	3	0	0.000	0.400	0.300	C94	"
    "0:*-1-3-*	Def\n"
    "2	0	1	3	0	0.000	0.500	0.350	C94	"
    "2:*-1-3-*	Def\n"
    "5	0	1	3	0	0.000	0.000	0.000	E94	"
    "5:*1-3-*	Def\n"
    "0	0	1	3	1	0.000	0.000	0.550	C94	"
    "0:*-1-3-1	Def\n"
    "0	0	1	3	5	0.000	0.200	0.700	C94	"
    "0:*-1-3-5	Def\n"
    "0	0	1	3	7	0.000	0.400	0.400	C94	"
    "0:*-1-3-7	Def\n"
    "0	1	1	3	1	0.103	0.177	0.545	C94\n"
    "0	1	1	3	5	-0.072	0.316	0.674	C94\n"
    "0	1	1	3	6	-0.117	-0.333	0.202	C94\n"
    "0	1	1	3	7	0.825	0.139	0.325	C94\n"
    "0	1	1	3	10	-0.927	1.112	1.388	C94\n"
    "0	2	1	3	5	0.663	-0.167	0.426	C94\n"
    "0	2	1	3	7	-0.758	0.112	0.563	C94\n"
    "0	5	1	3	1	-0.073	0.085	0.531	C94\n"
    "2	5	1	3	2	0.000	0.000	0.115	C94\n"
    "2	5	1	3	3	0.000	0.000	0.446	C94\n"
    "0	5	1	3	5	-0.822	0.501	1.008	C94\n"
    "0	5	1	3	6	0.000	-0.624	0.330	C94\n"
    "0	5	1	3	7	0.659	-1.407	0.308	C94\n"
    "0	5	1	3	10	-0.412	0.693	0.087	C94\n"
    "2	5	1	3	37	0.000	0.000	0.056	C94\n"
    "0	5	1	3	43	0.000	1.027	0.360	X94\n"
    "0	5	1	3	51	0.000	1.543	0.350	X94\n"
    "0	5	1	3	53	0.000	0.501	0.000	X94\n"
    "0	5	1	3	74	0.000	0.513	-0.344	X94\n"
    "0	5	1	3	75	0.000	0.511	-0.186	X94\n"
    "0	6	1	3	6	0.447	0.652	0.318	C94\n"
    "0	6	1	3	7	-0.395	0.730	-0.139	C94\n"
    "0	10	1	3	7	0.338	2.772	2.145	C94\n"
    "0	10	1	3	10	0.548	0.000	1.795	C94\n"
    "0	0	1	4	0	0.000	0.000	0.000	C94	"
    "0:*-1-4-*	Def\n"
    "0	0	1	6	0	0.000	0.000	0.200	C94	"
    "0:*-1-6-*	Def\n"
    "5	0	1	6	0	0.000	-0.200	0.400	C94	"
    "5:*-1-6-*	Def\n"
    "0	1	1	6	1	-0.681	0.755	0.755	C94\n"
    "5	1	1	6	1	0.000	0.243	-0.596	C94\n"
    "0	1	1	6	3	-0.547	0.000	0.320	C94\n"
    "0	1	1	6	21	0.000	0.270	0.237	C94\n"
    "0	2	1	6	21	0.102	0.460	-0.128	C94\n"
    "0	3	1	6	21	-1.652	-1.660	0.283	C94\n"
    "0	5	1	6	1	0.571	0.319	0.570	C94\n"
    "0	5	1	6	2	0.000	0.000	0.306	C94\n"
    "0	5	1	6	3	0.572	0.000	-0.304	C94\n"
    "0	5	1	6	21	0.596	-0.276	0.346	C94\n"
    "0	5	1	6	25	0.000	0.000	0.061	X94\n"
    "0	5	1	6	37	0.000	0.000	0.106	C94\n"
    "0	5	1	6	45	0.000	0.000	-0.174	X94\n"
    "0	6	1	6	1	0.229	-0.710	0.722	C94\n"
    "5	6	1	6	1	0.000	0.000	0.040	C94\n"
    "0	6	1	6	21	1.488	-3.401	-0.320	C94\n"
    "0	37	1	6	21	0.712	1.320	-0.507	C94\n"
    "0	0	1	8	0	0.000	-0.300	0.500	C94	"
    "0:*-1-8-*	Def\n"
    "5	0	1	8	0	0.000	0.000	0.297	E94	"
    "5:*1-8-*	Def\n"
    "0	1	1	8	1	-0.439	0.786	0.272	C94\n"
    "5	1	1	8	1	0.115	-0.390	0.658	C94\n"
    "0	1	1	8	6	-0.608	0.339	1.496	C94\n"
    "0	1	1	8	23	-0.428	0.323	0.280	C94\n"
    "0	2	1	8	23	0.594	-0.409	0.155	C94\n"
    "0	5	1	8	1	0.393	-0.385	0.562	C94\n"
    "0	5	1	8	6	0.598	-0.158	0.399	C94\n"
    "0	5	1	8	23	-0.152	-0.440	0.357	C94\n"
    "0	0	1	9	0	0.000	0.000	0.000	C94	"
    "0:*-1-9-*	Def\n"
    "5	0	1	9	0	0.000	0.000	0.000	E94	"
    "5:*1-9-*	Def\n"
    "0	5	1	9	3	0.204	-0.335	-0.352	C94\n"
    "0	5	1	9	53	0.000	0.000	0.097	X94\n"
    "0	0	1	10	0	0.000	0.000	0.300	C94	"
    "0:*-1-10-*	Def\n"
    "5	0	1	10	0	0.000	0.000	0.000	E94	"
    "5:*1-10-*	Def\n"
    "0	0	1	10	3	0.000	0.000	1.000	C94	"
    "0:*-1-10-3	Def\n"
    "0	1	1	10	3	-1.027	0.694	0.948	C94\n"
    "0	1	1	10	6	0.159	-0.552	0.198	C94\n"
    "0	1	1	10	28	0.552	-0.380	0.326	C94\n"
    "0	3	1	10	3	3.100	-2.529	1.494	C94\n"
    "0	3	1	10	28	0.079	0.280	0.402	C94\n"
    "0	5	1	10	1	0.000	0.000	0.779	C94\n"
    "0	5	1	10	3	-2.099	1.363	0.021	C94\n"
    "0	5	1	10	6	-0.162	0.832	0.552	C94\n"
    "0	5	1	10	28	-0.616	0.000	0.274	C94\n"
    "0	0	1	15	0	0.000	0.000	0.400	C94	"
    "0:*-1-15-*	Def\n"
    "5	0	1	15	0	0.000	0.000	0.336	E94	"
    "5:*1-15-*	Def\n"
    "0	1	1	15	1	-1.047	0.170	0.398	C94\n"
    "0	1	1	15	15	-1.438	0.263	0.501	C94\n"
    "0	1	1	15	71	-0.376	-0.133	0.288	C94\n"
    "0	5	1	15	1	1.143	-0.231	0.447	C94\n"
    "0	5	1	15	15	1.555	-0.323	0.456	C94\n"
    "0	5	1	15	37	0.000	0.000	0.459	C94\n"
    "0	5	1	15	71	0.229	0.203	0.440	C94\n"
    "0	0	1	17	0	0.000	0.000	0.350	C94	"
    "0:*-1-17-*	Def\n"
    "5	0	1	17	0	0.000	0.000	0.000	E94	"
    "5:*1-17-*	Def\n"
    "0	5	1	17	1	0.000	0.000	0.536	X94\n"
    "0	5	1	17	7	0.000	0.000	0.212	X94\n"
    "0	0	1	18	0	0.000	0.000	0.100	C94	"
    "0:*-1-18-*	Def\n"
    "5	0	1	18	0	0.000	0.000	0.112	E94	"
    "5:*1-18-*	Def\n"
    "0	5	1	18	1	0.000	0.000	0.000	X94\n"
    "0	5	1	18	6	0.000	0.000	0.099	X94\n"
    "0	5	1	18	32	0.000	0.585	0.388	X94\n"
    "0	5	1	18	43	0.000	-0.412	0.121	X94\n"
    "0	5	1	18	48	0.000	0.000	0.195	X94\n"
    "0	5	1	18	62	0.000	0.000	-0.088	X94\n"
    "0	0	1	19	0	0.000	0.000	0.150	C94	"
    "0:*-1-19-*	Def\n"
    "5	0	1	19	0	0.000	0.000	0.179	E94	"
    "5:*1-19-*	Def\n"
    "0	5	1	19	5	0.000	0.000	0.196	X94\n"
    "0	5	1	19	6	0.000	0.000	0.176	X94\n"
    "0	5	1	19	12	0.000	0.000	0.152	X94\n"
    "0	0	1	20	0	0.000	0.000	0.350	C94	"
    "0:*-1-20-*	Def\n"
    "5	0	1	20	0	0.000	0.000	0.350	C94	"
    "5:*-1-20-*	Def\n"
    "0	5	1	20	5	0.000	0.000	0.344	C94\n"
    "0	5	1	20	20	0.000	0.000	0.361	C94\n"
    "0	0	1	22	0	0.000	0.000	0.236	E94	"
    "0:*1-22-*	Def\n"
    "5	0	1	22	0	0.000	0.000	0.236	E94	"
    "5:*1-22-*	Def\n"
    "0	0	1	25	0	0.000	0.000	0.300	C94	"
    "0:*-1-25-*	Def\n"
    "5	0	1	25	0	0.000	0.000	0.251	E94	"
    "5:*1-25-*	Def\n"
    "0	1	1	25	1	0.000	-0.207	0.232	X94\n"
    "0	1	1	25	32	0.000	0.288	0.218	X94\n"
    "0	5	1	25	1	0.000	0.152	0.235	X94\n"
    "0	5	1	25	6	0.000	0.000	0.495	X94\n"
    "0	5	1	25	32	0.000	-0.130	0.214	X94\n"
    "0	5	1	25	43	0.000	0.000	0.466	X94\n"
    "0	5	1	25	72	0.000	0.000	0.243	X94\n"
    "0	0	1	26	0	0.000	0.000	0.450	C94	"
    "0:*-1-26-*	Def\n"
    "5	0	1	26	0	0.000	0.000	0.376	E94	"
    "5:*1-26-*	Def\n"
    "0	5	1	26	12	0.000	0.000	0.439	X94\n"
    "0	5	1	26	71	0.000	0.000	0.472	X94\n"
    "0	0	1	34	0	0.000	0.000	0.250	C94	"
    "0:*-1-34-*	Def\n"
    "5	0	1	34	0	0.000	0.000	0.198	E94	"
    "5:*1-34-*	Def\n"
    "0	1	1	34	36	0.000	0.000	0.187	C94\n"
    "0	5	1	34	1	0.000	0.000	0.247	C94\n"
    "0	5	1	34	36	0.000	0.000	0.259	C94\n"
    "0	0	1	37	0	0.000	0.000	0.200	C94	"
    "0:*-1-37-*	Def\n"
    "5	0	1	37	0	0.000	0.000	0.000	E94	"
    "5:*1-37-*	Def\n"
    "0	1	1	37	37	0.000	0.449	0.000	C94\n"
    "0	5	1	37	37	0.000	-0.420	0.391	C94\n"
    "0	6	1	37	37	0.000	0.000	0.150	C94\n"
    "0	0	1	39	0	0.000	0.000	0.000	C94	"
    "0:*-1-39-*	Def\n"
    "5	0	1	39	0	0.000	0.000	0.000	E94	"
    "5:*1-39-*	Def\n"
    "0	1	1	39	63	0.000	-0.080	-0.056	C94\n"
    "0	5	1	39	63	0.000	0.000	-0.113	C94\n"
    "0	0	1	40	0	0.000	0.000	0.250	C94	"
    "0:*-1-40-*	Def\n"
    "5	0	1	40	0	0.000	0.000	0.297	E94	"
    "5:*1-40-*	Def\n"
    "0	5	1	40	28	0.000	-0.097	0.203	C94\n"
    "0	5	1	40	37	0.000	0.000	0.329	C94\n"
    "0	0	1	41	0	0.000	0.600	0.000	C94	"
    "0:*-1-41-*	Def\n"
    "0	1	1	41	32	0.000	1.263	0.000	C94\n"
    "0	5	1	41	32	0.000	0.000	-0.106	C94\n"
    "0	5	1	41	72	0.000	0.632	0.000	X94\n"
    "0	0	1	43	0	0.000	0.000	0.150	C94	"
    "0:*-1-43-*	Def\n"
    "5	0	1	43	0	0.000	0.000	0.297	E94	"
    "5:*1-43-*	Def\n"
    "0	5	1	43	18	0.357	-0.918	0.000	X94\n"
    "0	5	1	43	25	0.000	0.000	0.061	X94\n"
    "0	5	1	43	28	-0.249	0.382	0.343	X94\n"
    "0	0	1	45	0	0.000	0.000	0.100	C94	"
    "0:*-1-45-*	Def\n"
    "0	5	1	45	32	0.000	0.000	0.125	X94\n"
    "0	0	1	46	0	0.000	0.000	-0.500	C94	"
    "0:*-1-46-*	Def\n"
    "0	5	1	46	7	0.000	0.000	-0.540	X94\n"
    "0	0	1	54	0	0.000	0.000	0.000	C94	"
    "0:*-1-54-*	Def\n"
    "2	0	1	54	0	0.000	0.000	0.000	E94	"
    "2:*1-54-*	Def\n"
    "5	0	1	54	0	0.000	0.000	0.000	E94	"
    "5:*1-54-*	Def\n"
    "0	5	1	54	3	0.000	0.000	-0.315	C94\n"
    "0	5	1	54	36	0.000	0.000	0.315	C94\n"
    "0	0	1	55	0	0.000	0.000	0.000	C94	"
    "0:*-1-55-*	Def\n"
    "5	0	1	55	0	0.000	0.000	0.000	E94	"
    "5:*1-55-*	Def\n"
    "0	5	1	55	36	0.000	-0.058	0.084	C94\n"
    "0	5	1	55	57	0.000	-0.058	-0.092	C94\n"
    "0	0	1	56	0	0.000	0.000	-0.300	C94	"
    "0:*-1-56-*	Def\n"
    "0	1	1	56	36	0.875	0.668	-0.015	C94\n"
    "0	1	1	56	57	-0.870	0.775	-0.406	C94\n"
    "0	5	1	56	36	-0.958	-0.629	-0.372	C94\n"
    "0	5	1	56	57	0.952	-0.715	-0.483	C94\n"
    "0	0	1	57	0	0.000	0.000	0.000	E94	"
    "0:*1-57-*	Def\n"
    "5	0	1	57	0	0.000	0.000	0.000	E94	"
    "5:*1-57-*	Def\n"
    "0	0	1	58	0	0.000	0.000	0.000	E94	"
    "0:*1-58-*	Def\n"
    "0	0	1	62	0	0.000	0.000	0.250	C94	"
    "0:*-1-62-*	Def\n"
    "0	5	1	62	18	0.000	0.000	0.270	X94\n"
    "0	0	1	63	0	0.000	0.000	0.000	E94	"
    "0:*1-63-*	Def\n"
    "5	0	1	63	0	0.000	0.000	0.000	E94	"
    "5:*1-63-*	Def\n"
    "0	0	1	64	0	0.000	0.000	0.000	E94	"
    "0:*1-64-*	Def\n"
    "5	0	1	64	0	0.000	0.000	0.000	E94	"
    "5:*1-64-*	Def\n"
    "0	0	1	67	0	0.000	0.000	0.000	E94	"
    "0:*1-67-*	Def\n"
    "5	0	1	67	0	0.000	0.000	0.000	E94	"
    "5:*1-67-*	Def\n"
    "0	0	1	68	0	0.000	0.000	0.400	C94	"
    "0:*-1-68-*	Def\n"
    "0	1	1	68	1	-0.117	0.090	0.751	C94\n"
    "0	1	1	68	23	0.373	0.153	0.635	C94\n"
    "0	1	1	68	32	-0.090	-0.169	0.075	C94\n"
    "0	5	1	68	1	0.134	-0.112	0.329	C94\n"
    "0	5	1	68	23	-0.361	-0.202	0.560	C94\n"
    "0	5	1	68	32	0.072	0.218	0.093	C94\n"
    "0	0	1	73	0	0.000	0.000	0.500	C94	"
    "0:*-1-73-*	Def\n"
    "0	5	1	73	32	0.000	0.000	0.509	X94\n"
    "0	5	1	73	72	0.000	0.000	0.443	X94\n"
    "0	0	1	75	0	0.000	0.000	0.000	E94	"
    "0:*1-75-*	Def\n"
    "0	0	1	78	0	0.000	0.000	0.000	E94	"
    "0:*1-78-*	Def\n"
    "0	0	1	80	0	0.000	0.000	0.000	E94	"
    "0:*1-80-*	Def\n"
    "0	0	1	81	0	0.000	0.000	0.000	E94	"
    "0:*1-81-*	Def\n"
    "0	0	2	2	0	0.000	12.000	0.000	C94	"
    "0:*-2=2-*	Def\n"
    "1	0	2	2	0	0.000	1.800	0.000	C94	"
    "1:*=2-2=*	Def\n"
    "5	0	2	2	0	0.000	12.000	0.000	C94	"
    "5:*-2=2-*	Def\n"
    "0	1	2	2	1	-0.403	12.000	0.000	C94\n"
    "0	1	2	2	2	0.000	12.000	0.000	C94\n"
    "1	1	2	2	2	-0.418	2.089	-0.310	C94\n"
    "0	1	2	2	5	0.000	12.000	0.000	C94\n"
    "1	1	2	2	5	0.412	2.120	0.269	C94\n"
    "1	2	2	2	2	0.094	1.621	0.877	C94\n"
    "0	2	2	2	5	0.000	12.000	0.000	C94\n"
    "1	2	2	2	5	0.317	1.421	-0.870	C94\n"
    "0	3	2	2	5	0.000	12.000	0.000	C94\n"
    "0	5	2	2	5	0.000	12.000	0.000	C94\n"
    "1	5	2	2	5	-0.406	1.767	0.000	C94\n"
    "0	5	2	2	6	0.000	12.000	0.000	C94\n"
    "0	5	2	2	37	0.000	12.000	0.000	C94\n"
    "0	5	2	2	40	0.000	12.000	0.000	C94\n"
    "0	5	2	2	41	0.000	12.000	0.000	C94\n"
    "0	5	2	2	45	0.000	12.000	0.000	X94\n"
    "0	5	2	2	62	0.000	12.000	0.000	X94\n"
    "1	0	2	3	0	0.000	2.500	0.000	#E94	"
    "0:*-2-3-*	Def\n"
    "1	1	2	3	1	0.136	1.798	0.630	C94\n"
    "1	1	2	3	5	0.497	2.405	0.357	C94\n"
    "1	1	2	3	6	-0.211	1.925	-0.131	C94\n"
    "1	1	2	3	7	-0.401	2.028	-0.318	C94\n"
    "1	1	2	3	10	-0.084	2.214	-0.610	C94\n"
    "1	2	2	3	1	-0.325	1.553	-0.487	C94\n"
    "1	2	2	3	5	-0.295	2.024	-0.590	C94\n"
    "1	2	2	3	6	-0.143	1.466	0.000	C94\n"
    "1	2	2	3	7	0.362	1.978	0.000	C94\n"
    "1	2	2	3	9	0.296	1.514	0.481	C94\n"
    "1	2	2	3	10	0.095	1.583	0.380	C94\n"
    "1	5	2	3	1	0.213	1.728	-0.042	C94\n"
    "1	5	2	3	5	-0.208	1.622	0.223	C94\n"
    "1	5	2	3	6	0.359	1.539	0.194	C94\n"
    "1	5	2	3	7	0.000	2.046	0.000	C94\n"
    "1	5	2	3	9	-0.290	1.519	-0.470	C94\n"
    "1	5	2	3	10	0.000	1.395	0.227	C94\n"
    "1	0	2	4	0	0.000	0.000	0.000	C94	"
    "0:*-2-4-*	Def\n"
    "0	0	2	6	0	0.000	3.100	0.000	C94	"
    "0:*-2-6-*	Def\n"
    "2	0	2	6	0	0.000	3.600	0.000	E94	"
    "2:*-2-6-*	Def\n"
    "5	0	2	6	0	0.000	3.600	0.000	E94	"
    "5:*-2-6-*	Def\n"
    "0	2	2	6	1	-1.953	3.953	-1.055	C94\n"
    "0	2	2	6	3	-1.712	2.596	-0.330	C94\n"
    "0	2	2	6	29	-0.215	2.810	-0.456	C94\n"
    "0	5	2	6	1	1.951	3.936	1.130	C94\n"
    "0	5	2	6	3	1.719	2.628	0.360	C94\n"
    "0	5	2	6	29	0.216	2.808	0.456	C94\n"
    "1	0	2	9	0	0.000	1.800	0.000	E94	"
    "1:*-2-9-*	Def\n"
    "0	0	2	10	0	0.000	6.000	0.000	E94	"
    "0:*-2-10-*	Def\n"
    "2	0	2	10	0	0.000	6.000	0.000	E94	"
    "2:*-2-10-*	Def\n"
    "5	0	2	10	0	0.000	6.000	0.000	E94	"
    "5:*-2-10-*	Def\n"
    "0	0	2	15	0	0.000	1.423	0.000	E94	"
    "0:*-2-15-*	Def\n"
    "2	0	2	15	0	0.000	1.423	0.000	E94	"
    "2:*-2-15-*	Def\n"
    "5	0	2	15	0	0.000	1.423	0.000	E94	"
    "5:*-2-15-*	Def\n"
    "0	0	2	17	0	0.000	1.423	0.000	E94	"
    "0:*-2-17-*	Def\n"
    "0	0	2	18	0	0.000	0.000	0.000	E94	"
    "0:*-2-18-*	Def\n"
    "2	0	2	18	0	0.000	0.000	0.000	E94	"
    "2:*-2-18-*	Def\n"
    "5	0	2	18	0	0.000	0.000	0.000	E94	"
    "5:*-2-18-*	Def\n"
    "0	0	2	19	0	0.000	0.000	0.000	E94	"
    "0:*-2-19-*	Def\n"
    "0	0	2	20	0	0.000	0.000	0.000	E94	"
    "0:*-2-20-*	Def\n"
    "2	0	2	20	0	0.000	0.000	0.000	E94	"
    "2:*-2-20-*	Def\n"
    "0	0	2	22	0	0.000	0.000	0.000	E94	"
    "0:*-2-22-*	Def\n"
    "2	0	2	22	0	0.000	0.000	0.000	E94	"
    "2:*-2-22-*	Def\n"
    "5	0	2	22	0	0.000	0.000	0.000	E94	"
    "5:*-2-22-*	Def\n"
    "0	0	2	25	0	0.000	0.000	0.000	E94	"
    "0:*-2-25-*	Def\n"
    "0	0	2	30	0	0.000	12.000	0.000	E94	"
    "0:*-2-30-*	Def\n"
    "0	0	2	34	0	0.000	0.000	0.000	E94	"
    "0:*-2-34-*	Def\n"
    "2	0	2	34	0	0.000	0.000	0.000	E94	"
    "2:*-2-34-*	Def\n"
    "1	0	2	37	0	0.000	2.000	0.000	C94	"
    "1:*-2-37-*	Def\n"
    "1	1	2	37	37	0.000	2.952	-0.079	C94\n"
    "1	2	2	37	37	0.000	1.542	0.434	C94\n"
    "1	5	2	37	37	0.000	1.308	-0.357	C94\n"
    "1	0	2	39	0	0.000	6.000	0.000	E94	"
    "1:*-2-39-*	Def\n"
    "0	0	2	40	0	0.000	3.700	0.000	C94	"
    "0:*-2-40-*	Def\n"
    "2	0	2	40	0	0.000	3.600	0.000	E94	"
    "2:*-2-40-*	Def\n"
    "5	0	2	40	0	0.000	3.600	0.000	E94	"
    "5:*-2-40-*	Def\n"
    "0	2	2	40	28	0.000	3.756	-0.530	C94\n"
    "0	5	2	40	28	0.073	3.698	0.291	C94\n"
    "0	0	2	41	0	0.000	1.200	0.000	C94	"
    "0:*-2-41-*	Def\n"
    "2	0	2	41	0	0.000	1.800	0.000	E94	"
    "2:*-2-41-*	Def\n"
    "0	2	2	41	32	0.000	1.235	0.000	C94\n"
    "0	5	2	41	32	0.000	1.231	0.000	C94\n"
    "0	0	2	43	0	0.000	3.600	0.000	E94	"
    "0:*-2-43-*	Def\n"
    "2	0	2	43	0	0.000	3.600	0.000	E94	"
    "2:*-2-43-*	Def\n"
    "0	0	2	45	0	0.000	2.200	0.000	C94	"
    "0:*-2-45-*	Def\n"
    "2	0	2	45	0	0.000	1.800	0.000	E94	"
    "2:*-2-45-*	Def\n"
    "0	2	2	45	32	0.000	2.212	0.000	X94\n"
    "0	5	2	45	32	0.000	2.225	0.000	X94\n"
    "0	0	2	46	0	0.000	1.800	0.000	E94	"
    "0:*-2-46-*	Def\n"
    "2	0	2	46	0	0.000	1.800	0.000	E94	"
    "2:*-2-46-*	Def\n"
    "0	0	2	55	0	0.000	4.800	0.000	E94	"
    "0:*-2-55-*	Def\n"
    "0	0	2	56	0	0.000	4.800	0.000	E94	"
    "0:*-2-56-*	Def\n"
    "0	0	2	62	0	0.000	8.000	0.000	C94	"
    "0:*-2-62-*	Def\n"
    "0	2	2	62	23	1.693	7.903	0.532	X94\n"
    "0	5	2	62	23	-1.696	7.897	-0.482	X94\n"
    "1	0	2	63	0	0.000	1.800	0.000	E94	"
    "1:*-2-63-*	Def\n"
    "1	0	2	64	0	0.000	1.800	0.000	E94	"
    "1:*-2-64-*	Def\n"
    "1	0	2	67	0	0.000	1.800	0.000	E94	"
    "1:*-2-67-*	Def\n"
    "1	0	2	81	0	0.000	4.800	0.000	E94	"
    "1:*-2-81-*	Def\n"
    "1	0	3	3	0	0.000	0.600	0.000	C94	"
    "0:*-3-3-*	Def\n"
    "4	0	3	3	0	0.000	1.800	0.000	E94	"
    "4:*-3-3-*	Def\n"
    "1	1	3	3	1	-0.486	0.714	0.000	C94\n"
    "1	1	3	3	6	-0.081	-0.125	0.132	C94\n"
    "1	1	3	3	7	1.053	1.327	0.000	C94\n"
    "1	5	3	3	6	0.000	0.188	0.436	C94\n"
    "1	5	3	3	7	0.000	0.177	-0.412	C94\n"
    "1	6	3	3	6	0.269	0.437	0.000	C94\n"
    "1	6	3	3	7	-0.495	0.793	-0.318	C94\n"
    "1	7	3	3	7	-0.260	1.084	0.193	C94\n"
    "0	0	3	6	0	0.000	5.500	0.000	C94	"
    "0:*-3-6-*	Def\n"
    "2	0	3	6	0	0.000	5.500	0.000	C94	"
    "2:*-3-6-*	Def\n"
    "4	0	3	6	0	0.000	3.600	0.000	E94	"
    "4:*-3-6-*	Def\n"
    "5	0	3	6	0	0.000	3.600	0.000	E94	"
    "5:*-3-6-*	Def\n"
    "0	1	3	6	1	-1.244	5.482	0.365	C94\n"
    "0	1	3	6	24	-1.166	5.078	-0.545	C94\n"
    "0	1	3	6	37	-0.677	5.854	0.521	C94\n"
    "2	2	3	6	24	0.256	4.519	0.258	C94\n"
    "2	3	3	6	24	1.663	4.073	0.094	C94\n"
    "0	5	3	6	1	0.526	5.631	0.691	C94\n"
    "0	5	3	6	2	0.159	6.586	0.216	C94\n"
    "0	5	3	6	24	-2.285	4.737	0.468	C94\n"
    "0	7	3	6	0	0.700	6.500	-0.400	C94	"
    "0:7-3-6-*	Def\n"
    "0	7	3	6	1	0.682	7.184	-0.935	C94\n"
    "0	7	3	6	2	-0.168	6.572	-0.151	C94\n"
    "0	7	3	6	24	1.662	6.152	-0.058	C94\n"
    "0	7	3	6	37	0.635	5.890	-0.446	C94\n"
    "2	37	3	6	24	0.000	3.892	-0.094	C94\n"
    "0	0	3	9	0	0.000	16.000	0.000	C94	"
    "0:*-3=9-*	Def\n"
    "1	0	3	9	0	0.000	1.800	0.000	E94	"
    "1:*-3-9-*	Def\n"
    "5	0	3	9	0	0.000	12.000	0.000	E94	"
    "5:*-3-9-*	Def\n"
    "0	2	3	9	27	0.000	16.000	0.000	C94\n"
    "0	5	3	9	1	0.687	16.152	0.894	C94\n"
    "0	5	3	9	27	0.000	16.000	0.000	C94\n"
    "0	40	3	9	1	-0.758	18.216	-0.188	C94\n"
    "0	40	3	9	27	0.000	16.000	0.000	C94\n"
    "0	0	3	10	0	0.000	6.000	0.000	C94	"
    "0:*-3-10-*	Def\n"
    "2	0	3	10	0	0.000	6.000	0.000	C94	"
    "2:*-3-10-*	Def\n"
    "4	0	3	10	0	0.000	6.000	0.000	C94	"
    "4:*-3-10-*	Def\n"
    "5	0	3	10	0	0.000	6.000	0.000	E94	"
    "5:*-3-10-*	Def\n"
    "0	1	3	10	1	0.647	6.159	0.507	C94\n"
    "0	1	3	10	6	-1.035	8.791	1.464	C94\n"
    "0	1	3	10	28	-0.294	5.805	1.342	C94\n"
    "2	2	3	10	28	-0.287	7.142	0.120	C94\n"
    "0	5	3	10	1	-0.183	6.314	1.753	C94\n"
    "0	5	3	10	3	-0.751	5.348	0.209	C94\n"
    "0	5	3	10	28	-0.388	5.972	0.459	C94\n"
    "0	7	3	10	1	-0.319	6.294	-0.147	C94\n"
    "0	7	3	10	3	0.776	-0.585	-0.145	C94\n"
    "0	7	3	10	6	1.107	8.631	-0.452	C94\n"
    "0	7	3	10	28	1.435	4.975	-0.454	C94\n"
    "0	10	3	10	28	0.000	3.495	1.291	C94\n"
    "0	0	3	15	0	0.000	1.423	0.000	E94	"
    "0:*-3-15-*	Def\n"
    "2	0	3	15	0	0.000	1.423	0.000	E94	"
    "2:*-3-15-*	Def\n"
    "4	0	3	15	0	0.000	1.423	0.000	E94	"
    "4:*-3-15-*	Def\n"
    "5	0	3	15	0	0.000	1.423	0.000	E94	"
    "5:*-3-15-*	Def\n"
    "0	0	3	17	0	0.000	1.423	0.000	E94	"
    "0:*-3-17-*	Def\n"
    "5	0	3	17	0	0.000	1.423	0.000	E94	"
    "5:*-3-17-*	Def\n"
    "0	0	3	18	0	0.000	0.000	0.000	E94	"
    "0:*-3-18-*	Def\n"
    "2	0	3	18	0	0.000	0.000	0.000	E94	"
    "2:*-3-18-*	Def\n"
    "0	0	3	20	0	0.000	0.000	-0.300	C94	"
    "0:*-3-20-*	Def\n"
    "2	0	3	20	0	0.000	0.000	0.000	E94	"
    "2:*-3-20-*	Def\n"
    "4	0	3	20	0	0.000	0.000	-0.300	C94	"
    "4:*-3-20-*	Def\n"
    "5	0	3	20	0	0.000	0.000	0.000	E94	"
    "5:*-3-20-*	Def\n"
    "0	7	3	20	0	0.000	0.400	0.400	C94	"
    "0:7-3-20-*	Def\n"
    "0	7	3	20	5	0.000	0.000	-0.131	C94\n"
    "0	7	3	20	20	0.000	0.000	0.000	C94\n"
    "0	20	3	20	5	0.000	0.000	0.085	C94\n"
    "0	20	3	20	20	0.000	0.000	0.000	C94\n"
    "0	0	3	22	0	0.000	0.000	0.000	E94	"
    "0:*-3-22-*	Def\n"
    "2	0	3	22	0	0.000	0.000	0.000	E94	"
    "2:*-3-22-*	Def\n"
    "4	0	3	22	0	0.000	0.000	0.000	E94	"
    "4:*-3-22-*	Def\n"
    "5	0	3	22	0	0.000	0.000	0.000	E94	"
    "5:*-3-22-*	Def\n"
    "0	7	3	22	0	0.000	0.400	0.400	C94	"
    "0:7-3-22-*	Def\n"
    "0	0	3	25	0	0.000	0.000	0.000	E94	"
    "0:*-3-25-*	Def\n"
    "2	0	3	25	0	0.000	0.000	0.000	E94	"
    "2:*-3-25-*	Def\n"
    "1	0	3	30	0	0.000	1.800	0.000	E94	"
    "1:*-3-30-*	Def\n"
    "4	0	3	30	0	0.000	1.800	0.000	E94	"
    "4:*-3-30-*	Def\n"
    "1	0	3	37	0	0.000	2.500	0.000	#E94	"
    "1:*-3-37-*	Def\n"
    "4	0	3	37	0	0.000	1.800	0.000	E94	"
    "4:*-3-37-*	Def\n"
    "1	1	3	37	37	0.000	2.428	0.000	C94\n"
    "1	6	3	37	37	0.000	1.743	0.000	C94\n"
    "1	7	3	37	37	0.000	2.256	0.000	C94\n"
    "1	43	3	37	37	-0.241	3.385	-0.838	X94\n"
    "1	0	3	39	0	0.000	5.500	0.000	#E94	"
    "1:*-3-39-*	Def\n"
    "0	0	3	40	0	0.000	3.900	0.000	C94	"
    "0:*-3-40-*	Def\n"
    "2	0	3	40	0	0.000	3.600	0.000	E94	"
    "2:*-3-40-*	Def\n"
    "5	0	3	40	0	0.000	3.600	0.000	E94	"
    "5:*-3-40-*	Def\n"
    "0	5	3	40	28	-1.477	4.362	0.902	C94\n"
    "0	9	3	40	28	1.496	4.369	-0.417	C94\n"
    "0	40	3	40	28	0.178	3.149	0.778	C94\n"
    "0	0	3	41	0	0.000	1.800	0.000	E94	"
    "0:*-3-41-*	Def\n"
    "2	0	3	41	0	0.000	1.800	0.000	E94	"
    "2:*-3-41-*	Def\n"
    "0	0	3	43	0	0.000	4.500	0.000	C94	"
    "0:*-3-43-*	Def\n"
    "2	0	3	43	0	0.000	3.600	0.000	E94	"
    "2:*-3-43-*	Def\n"
    "4	0	3	43	0	0.000	3.600	0.000	E94	"
    "4:*-3-43-*	Def\n"
    "5	0	3	43	0	0.000	3.600	0.000	E94	"
    "5:*-3-43-*	Def\n"
    "0	1	3	43	18	1.712	3.309	0.233	X94\n"
    "0	1	3	43	28	-0.414	4.168	-0.875	X94\n"
    "0	7	3	43	18	-0.880	5.091	-0.129	X94\n"
    "0	7	3	43	28	0.536	5.276	-0.556	X94\n"
    "2	37	3	43	18	-0.701	4.871	1.225	X94\n"
    "2	37	3	43	28	-0.086	5.073	0.878	X94\n"
    "0	0	3	45	0	0.000	1.800	0.000	E94	"
    "0:*-3-45-*	Def\n"
    "2	0	3	45	0	0.000	1.800	0.000	E94	"
    "2:*-3-45-*	Def\n"
    "0	0	3	48	0	0.000	0.000	0.892	E94	"
    "0:*-3-48-*	Def\n"
    "0	0	3	51	0	0.000	13.500	0.000	C94	"
    "0:*-3-51-*	Def\n"
    "0	1	3	51	52	0.000	13.549	0.000	X94\n"
    "0	0	3	54	0	0.000	8.000	0.000	C94	"
    "0:*-3-54-*	Def\n"
    "1	0	3	54	0	0.000	2.500	0.000	#E94	"
    "1:*-3-54-*	Def\n"
    "5	0	3	54	0	0.000	12.000	0.000	E94	"
    "5:*-3-54-*	Def\n"
    "0	5	3	54	1	0.000	8.000	0.000	C94\n"
    "0	5	3	54	36	0.000	8.000	0.000	C94\n"
    "0	0	3	55	0	0.000	4.800	0.000	E94	"
    "0:*-3-55-*	Def\n"
    "2	0	3	55	0	0.000	4.800	0.000	E94	"
    "2:*-3-55-*	Def\n"
    "0	0	3	56	0	0.000	4.800	0.000	E94	"
    "0:*-3-56-*	Def\n"
    "2	0	3	56	0	0.000	4.800	0.000	E94	"
    "2:*-3-56-*	Def\n"
    "1	0	3	57	0	0.000	2.500	0.000	#E94	"
    "1:*-3-57-*	Def\n"
    "1	0	3	58	0	0.000	4.800	0.000	E94	"
    "1:*-3-58-*	Def\n"
    "0	0	3	62	0	0.000	3.600	0.000	E94	"
    "0:*-3-62-*	Def\n"
    "2	0	3	62	0	0.000	3.600	0.000	E94	"
    "2:*-3-62-*	Def\n"
    "5	0	3	62	0	0.000	3.600	0.000	E94	"
    "5:*-3-62-*	Def\n"
    "1	0	3	63	0	0.000	2.500	0.000	#E94	"
    "1:*-3-63-*	Def\n"
    "1	0	3	64	0	0.000	2.500	0.000	#E94	"
    "1:*-3-64-*	Def\n"
    "0	0	3	67	0	0.000	12.000	0.000	E94	"
    "0:*-3-67-*	Def\n"
    "0	0	3	74	0	0.000	19.000	0.000	C94	"
    "0:*-3-74-*	Def\n"
    "0	1	3	74	7	0.000	19.349	0.000	X94\n"
    "0	0	3	75	0	0.000	19.000	0.000	C94	"
    "0:*-3-75-*	Def\n"
    "0	1	3	75	71	0.000	18.751	0.000	X94\n"
    "1	0	3	78	0	0.000	2.500	0.000	#E94	"
    "1:*-3-78-*	Def\n"
    "1	0	3	80	0	0.000	2.500	0.000	#E94	"
    "1:*-3-80-*	Def\n"
    "0	0	6	6	0	0.000	-2.000	0.000	E94	"
    "0:*-6-6-*	Def\n"
    "5	0	6	6	0	0.000	-2.000	0.000	E94	"
    "5:*-6-6-*	Def\n"
    "0	0	6	8	0	0.900	-1.100	-0.500	C94	"
    "0:*-6-8-*	Def\n"
    "5	0	6	8	0	0.000	0.000	0.274	E94	"
    "5:*-6-8-*	Def\n"
    "0	21	6	8	1	0.261	-0.330	-0.542	C94\n"
    "0	21	6	8	23	1.503	-1.853	-0.476	C94\n"
    "0	0	6	9	0	0.000	3.600	0.000	E94	"
    "0:*-6-9-*	Def\n"
    "5	0	6	9	0	0.000	3.600	0.000	E94	"
    "5:*-6-9-*	Def\n"
    "0	0	6	10	0	1.200	0.500	-1.000	C94	"
    "0:*-6-10-*	Def\n"
    "0	21	6	10	1	0.875	0.180	-0.733	C94\n"
    "0	21	6	10	3	0.529	0.000	-1.163	C94\n"
    "0	0	6	15	0	0.000	-4.000	0.000	E94	"
    "0:*-6-15-*	Def\n"
    "0	0	6	17	0	0.000	1.423	0.000	E94	"
    "0:*-6-17-*	Def\n"
    "5	0	6	17	0	0.000	1.423	0.000	E94	"
    "5:*-6-17-*	Def\n"
    "0	0	6	18	0	0.000	0.000	0.100	C94	"
    "0:*-6-18-*	Def\n"
    "5	0	6	18	0	0.000	0.000	0.103	E94	"
    "5:*-6-18-*	Def\n"
    "0	33	6	18	1	-0.520	-0.471	-0.267	X94\n"
    "0	33	6	18	6	-1.623	0.204	0.438	X94\n"
    "0	33	6	18	32	1.616	0.425	0.191	X94\n"
    "0	0	6	19	0	0.000	0.000	0.150	C94	"
    "0:*-6-19-*	Def\n"
    "5	0	6	19	0	0.000	0.000	0.165	E94	"
    "5:*-6-19-*	Def\n"
    "0	21	6	19	1	-0.620	-0.329	0.303	X94\n"
    "0	21	6	19	5	0.683	0.220	0.000	X94\n"
    "0	0	6	20	0	0.000	0.000	0.400	C94	"
    "0:*-6-20-*	Def\n"
    "4	0	6	20	0	0.000	0.000	0.217	E94	"
    "4:*-6-20-*	Def\n"
    "5	0	6	20	0	0.000	0.000	0.217	E94	"
    "5:*-6-20-*	Def\n"
    "0	20	6	20	5	0.000	0.000	-0.079	C94\n"
    "4	20	6	20	20	0.000	0.000	0.000	C94\n"
    "0	0	6	22	0	0.000	0.000	0.217	E94	"
    "0:*-6-22-*	Def\n"
    "0	0	6	25	0	0.000	0.000	0.650	C94	"
    "0:*-6-25-*	Def\n"
    "5	0	6	25	0	0.000	0.000	0.231	E94	"
    "5:*-6-25-*	Def\n"
    "0	1	6	25	1	-1.704	-0.452	0.556	X94\n"
    "0	1	6	25	6	0.000	0.000	0.777	X94\n"
    "0	1	6	25	32	1.205	0.914	0.612	X94\n"
    "0	24	6	25	6	-3.209	-7.622	1.065	X94\n"
    "0	24	6	25	32	-5.891	-3.332	0.290	X94\n"
    "0	0	6	26	0	0.000	0.000	0.346	E94	"
    "0:*-6-26-*	Def\n"
    "0	0	6	30	0	0.000	3.600	0.000	E94	"
    "0:*-6-30-*	Def\n"
    "2	0	6	30	0	0.000	3.600	0.000	E94	"
    "2:*-6-30-*	Def\n"
    "0	0	6	37	0	0.000	3.200	0.000	C94	"
    "0:*-6-37-*	Def\n"
    "5	0	6	37	0	0.000	3.600	0.000	E94	"
    "5:*-6-37-*	Def\n"
    "0	1	6	37	37	0.000	4.382	0.000	C94\n"
    "0	3	6	37	37	0.000	2.576	0.000	C94\n"
    "0	29	6	37	37	0.000	2.801	0.000	C94\n"
    "0	0	6	39	0	0.000	0.000	0.000	E94	"
    "0:*-6-39-*	Def\n"
    "0	0	6	40	0	0.000	0.000	0.274	E94	"
    "0:*-6-40-*	Def\n"
    "0	0	6	41	0	0.000	3.600	0.000	E94	"
    "0:*-6-41-*	Def\n"
    "0	0	6	43	0	0.000	0.000	0.274	E94	"
    "0:*-6-43-*	Def\n"
    "0	0	6	45	0	0.000	6.000	0.000	C94	"
    "0:*-6-45-*	Def\n"
    "0	1	6	45	32	0.000	6.208	0.000	X94\n"
    "0	0	6	54	0	0.000	3.600	0.000	E94	"
    "0:*-6-54-*	Def\n"
    "0	0	6	55	0	0.000	3.600	0.000	E94	"
    "0:*-6-55-*	Def\n"
    "0	0	6	57	0	0.000	3.600	0.000	E94	"
    "0:*-6-57-*	Def\n"
    "0	0	6	58	0	0.000	3.600	0.000	E94	"
    "0:*-6-58-*	Def\n"
    "0	0	6	63	0	0.000	3.600	0.000	E94	"
    "0:*-6-63-*	Def\n"
    "0	0	6	64	0	0.000	3.600	0.000	E94	"
    "0:*-6-64-*	Def\n"
    "0	0	8	8	0	0.000	0.000	0.375	E94	"
    "0:*-8-8-*	Def\n"
    "5	0	8	8	0	0.000	0.000	0.375	E94	"
    "5:*-8-8-*	Def\n"
    "0	0	8	9	0	0.000	3.600	0.000	E94	"
    "0:*-8-9-*	Def\n"
    "5	0	8	9	0	0.000	3.600	0.000	E94	"
    "5:*-8-9-*	Def\n"
    "0	0	8	10	0	0.000	0.000	0.000	E94	"
    "0:*-8-10-*	Def\n"
    "4	0	8	10	0	0.000	0.000	0.000	E94	"
    "4:*-8-10-*	Def\n"
    "0	0	8	15	0	0.000	0.000	0.424	E94	"
    "0:*-8-15-*	Def\n"
    "0	0	8	17	0	0.000	1.423	0.000	E94	"
    "0:*-8-17-*	Def\n"
    "4	0	8	17	0	0.000	1.423	0.000	E94	"
    "4:*-8-17-*	Def\n"
    "5	0	8	17	0	0.000	1.423	0.000	E94	"
    "5:*-8-17-*	Def\n"
    "0	0	8	19	0	0.000	0.000	0.225	E94	"
    "0:*-8-19-*	Def\n"
    "0	0	8	20	0	0.000	0.000	0.350	C94	"
    "0:*-8-20-*	Def\n"
    "4	0	8	20	0	0.000	0.000	0.300	C94	"
    "4:*-8-20-*	Def\n"
    "5	0	8	20	0	0.000	0.000	0.297	E94	"
    "5:*-8-20-*	Def\n"
    "0	20	8	20	5	0.000	0.120	0.472	C94\n"
    "4	20	8	20	20	0.000	-0.097	0.200	C94\n"
    "0	23	8	20	5	-0.101	-0.324	0.371	C94\n"
    "0	23	8	20	20	0.107	0.253	0.151	C94\n"
    "0	0	8	22	0	0.000	0.000	0.297	E94	"
    "0:*-8-22-*	Def\n"
    "0	0	8	25	0	0.000	0.000	0.316	E94	"
    "0:*-8-25-*	Def\n"
    "5	0	8	25	0	0.000	0.000	0.316	E94	"
    "5:*-8-25-*	Def\n"
    "0	0	8	26	0	0.000	0.000	0.474	E94	"
    "0:*-8-26-*	Def\n"
    "5	0	8	26	0	0.000	0.000	0.474	E94	"
    "5:*-8-26-*	Def\n"
    "0	0	8	34	0	0.000	0.000	0.250	E94	"
    "0:*-8-34-*	Def\n"
    "0	0	8	39	0	0.000	0.000	0.000	E94	"
    "0:*-8-39-*	Def\n"
    "0	0	8	40	0	0.000	0.000	0.375	E94	"
    "0:*-8-40-*	Def\n"
    "0	0	8	43	0	0.000	0.000	0.375	E94	"
    "0:*-8-43-*	Def\n"
    "0	0	8	45	0	0.000	3.600	0.000	E94	"
    "0:*-8-45-*	Def\n"
    "0	0	8	46	0	0.000	3.600	0.000	E94	"
    "0:*-8-46-*	Def\n"
    "0	0	8	55	0	0.000	3.600	0.000	E94	"
    "0:*-8-55-*	Def\n"
    "0	0	8	56	0	0.000	3.600	0.000	E94	"
    "0:*-8-56-*	Def\n"
    "0	0	9	9	0	0.000	12.000	0.000	E94	"
    "0:*-9-9-*	Def\n"
    "1	0	9	9	0	0.000	1.800	0.000	E94	"
    "1:*-9-9-*	Def\n"
    "5	0	9	9	0	0.000	12.000	0.000	E94	"
    "5:*-9-9-*	Def\n"
    "0	0	9	10	0	0.000	6.000	0.000	E94	"
    "0:*-9-10-*	Def\n"
    "5	0	9	10	0	0.000	6.000	0.000	E94	"
    "5:*-9-10-*	Def\n"
    "0	0	9	15	0	0.000	1.423	0.000	E94	"
    "0:*-9-15-*	Def\n"
    "0	0	9	18	0	0.000	0.000	0.000	E94	"
    "0:*-9-18-*	Def\n"
    "0	0	9	19	0	0.000	0.000	0.000	E94	"
    "0:*-9-19-*	Def\n"
    "0	0	9	20	0	0.000	0.000	0.000	E94	"
    "0:*-9-20-*	Def\n"
    "0	0	9	25	0	0.000	0.000	0.000	E94	"
    "0:*-9-25-*	Def\n"
    "0	0	9	34	0	0.000	0.000	0.000	E94	"
    "0:*-9-34-*	Def\n"
    "5	0	9	34	0	0.000	0.000	0.000	E94	"
    "5:*-9-34-*	Def\n"
    "1	0	9	37	0	0.000	1.800	0.000	E94	"
    "1:*-9-37-*	Def\n"
    "1	0	9	39	0	0.000	6.000	0.000	E94	"
    "1:*-9-39-*	Def\n"
    "0	0	9	40	0	0.000	3.600	0.000	E94	"
    "0:*-9-40-*	Def\n"
    "0	0	9	41	0	0.000	4.800	0.000	E94	"
    "0:*-9-41-*	Def\n"
    "0	0	9	45	0	0.000	1.800	0.000	E94	"
    "0:*-9-45-*	Def\n"
    "0	0	9	54	0	0.000	12.000	0.000	E94	"
    "0:*-9-54-*	Def\n"
    "0	0	9	55	0	0.000	4.800	0.000	E94	"
    "0:*-9-55-*	Def\n"
    "0	0	9	56	0	0.000	4.800	0.000	E94	"
    "0:*-9-56-*	Def\n"
    "1	0	9	57	0	0.000	1.800	0.000	E94	"
    "1:*-9-57-*	Def\n"
    "0	0	9	62	0	0.000	3.600	0.000	E94	"
    "0:*-9-62-*	Def\n"
    "1	0	9	63	0	0.000	1.800	0.000	E94	"
    "1:*-9-63-*	Def\n"
    "1	0	9	64	0	0.000	1.800	0.000	E94	"
    "1:*-9-64-*	Def\n"
    "0	0	9	67	0	0.000	12.000	0.000	E94	"
    "0:*-9-67-*	Def\n"
    "1	0	9	78	0	0.000	1.800	0.000	E94	"
    "1:*-9-78-*	Def\n"
    "1	0	9	81	0	0.000	4.800	0.000	E94	"
    "1:*-9-81-*	Def\n"
    "0	0	10	10	0	0.000	0.000	0.000	E94	"
    "0:*-10-10-*	Def\n"
    "5	0	10	10	0	0.000	0.000	0.000	E94	"
    "5:*-10-10-*	Def\n"
    "0	0	10	15	0	0.000	0.000	0.000	E94	"
    "0:*-10-15-*	Def\n"
    "0	0	10	17	0	0.000	4.743	0.000	E94	"
    "0:*-10-17-*	Def\n"
    "0	0	10	20	0	0.000	0.000	0.000	E94	"
    "0:*-10-20-*	Def\n"
    "4	0	10	20	0	0.000	0.000	0.000	E94	"
    "4:*-10-20-*	Def\n"
    "5	0	10	20	0	0.000	0.000	0.000	E94	"
    "5:*-10-20-*	Def\n"
    "0	0	10	22	0	0.000	0.000	0.000	E94	"
    "0:*-10-22-*	Def\n"
    "0	0	10	25	0	0.000	0.000	0.000	E94	"
    "0:*-10-25-*	Def\n"
    "0	0	10	26	0	0.000	0.000	0.000	E94	"
    "0:*-10-26-*	Def\n"
    "5	0	10	26	0	0.000	0.000	0.000	E94	"
    "5:*-10-26-*	Def\n"
    "0	0	10	34	0	0.000	0.000	0.000	E94	"
    "0:*-10-34-*	Def\n"
    "0	0	10	37	0	0.000	6.000	0.000	E94	"
    "0:*-10-37-*	Def\n"
    "0	0	10	39	0	0.000	0.000	0.000	E94	"
    "0:*-10-39-*	Def\n"
    "0	0	10	40	0	0.000	0.000	0.000	E94	"
    "0:*-10-40-*	Def\n"
    "5	0	10	40	0	0.000	0.000	0.000	E94	"
    "5:*-10-40-*	Def\n"
    "0	0	10	41	0	0.000	6.000	0.000	E94	"
    "0:*-10-41-*	Def\n"
    "0	0	10	45	0	0.000	6.000	0.000	E94	"
    "0:*-10-45-*	Def\n"
    "0	0	10	63	0	0.000	6.000	0.000	E94	"
    "0:*-10-63-*	Def\n"
    "0	0	10	64	0	0.000	6.000	0.000	E94	"
    "0:*-10-64-*	Def\n"
    "0	0	15	15	0	-1.400	-8.300	1.000	C94	"
    "0:*-15-15-*	Def\n"
    "5	0	15	15	0	0.000	-8.000	0.000	E94	"
    "5:*-15-15-*	Def\n"
    "0	1	15	15	1	-1.663	-8.408	1.433	C94\n"
    "0	1	15	15	71	-1.088	-8.245	0.411	C94\n"
    "0	0	15	18	0	0.000	0.000	0.160	E94	"
    "0:*-15-18-*	Def\n"
    "0	0	15	19	0	0.000	0.000	0.255	E94	"
    "0:*-15-19-*	Def\n"
    "5	0	15	19	0	0.000	0.000	0.255	E94	"
    "5:*-15-19-*	Def\n"
    "0	0	15	20	0	0.000	0.000	0.336	E94	"
    "0:*-15-20-*	Def\n"
    "4	0	15	20	0	0.000	0.000	0.336	E94	"
    "4:*-15-20-*	Def\n"
    "0	0	15	22	0	0.000	0.000	0.336	E94	"
    "0:*-15-22-*	Def\n"
    "0	0	15	25	0	0.000	0.000	0.358	E94	"
    "0:*-15-25-*	Def\n"
    "4	0	15	25	0	0.000	0.000	0.358	E94	"
    "4:*-15-25-*	Def\n"
    "0	0	15	26	0	0.000	0.000	0.537	E94	"
    "0:*-15-26-*	Def\n"
    "0	0	15	30	0	0.000	1.423	0.000	E94	"
    "0:*-15-30-*	Def\n"
    "4	0	15	30	0	0.000	1.423	0.000	E94	"
    "4:*-15-30-*	Def\n"
    "0	0	15	37	0	0.000	1.300	0.000	C94	"
    "0:*-15-37-*	Def\n"
    "5	0	15	37	0	0.000	1.423	0.000	E94	"
    "5:*-15-37-*	Def\n"
    "0	1	15	37	37	0.000	2.177	0.000	C94\n"
    "0	71	15	37	37	0.000	0.505	0.333	C94\n"
    "0	0	15	40	0	0.000	0.000	0.424	E94	"
    "0:*-15-40-*	Def\n"
    "0	0	15	43	0	0.000	0.000	0.424	E94	"
    "0:*-15-43-*	Def\n"
    "0	0	15	57	0	0.000	1.423	0.000	E94	"
    "0:*-15-57-*	Def\n"
    "0	0	15	63	0	0.000	1.423	0.000	E94	"
    "0:*-15-63-*	Def\n"
    "0	0	15	64	0	0.000	1.423	0.000	E94	"
    "0:*-15-64-*	Def\n"
    "0	0	17	20	0	0.000	0.000	0.000	E94	"
    "0:*-17-20-*	Def\n"
    "4	0	17	20	0	0.000	0.000	0.000	E94	"
    "4:*-17-20-*	Def\n"
    "5	0	17	20	0	0.000	0.000	0.000	E94	"
    "5:*-17-20-*	Def\n"
    "0	0	17	22	0	0.000	0.000	0.000	E94	"
    "0:*-17-22-*	Def\n"
    "0	0	17	37	0	0.000	1.423	0.000	E94	"
    "0:*-17-37-*	Def\n"
    "0	0	17	43	0	0.000	3.795	0.000	E94	"
    "0:*-17-43-*	Def\n"
    "0	0	18	20	0	0.000	0.000	0.112	E94	"
    "0:*-18-20-*	Def\n"
    "4	0	18	20	0	0.000	0.000	0.112	E94	"
    "4:*-18-20-*	Def\n"
    "5	0	18	20	0	0.000	0.000	0.112	E94	"
    "5:*-18-20-*	Def\n"
    "0	0	18	22	0	0.000	0.000	0.112	E94	"
    "0:*-18-22-*	Def\n"
    "0	0	18	37	0	0.000	-1.200	-0.300	C94	"
    "0:*-18-37-*	Def\n"
    "0	32	18	37	37	-0.173	-0.965	-0.610	X94\n"
    "0	39	18	37	37	0.000	-0.760	0.227	X94\n"
    "0	43	18	37	37	0.228	-1.741	-0.371	X94\n"
    "0	0	18	39	0	0.000	0.000	0.500	C94	"
    "0:*-18-39-*	Def\n"
    "0	32	18	39	63	0.000	0.687	0.680	X94\n"
    "0	37	18	39	63	0.000	-0.513	0.357	X94\n"
    "0	0	18	43	0	0.000	0.000	0.350	C94	"
    "0:*-18-43-*	Def\n"
    "4	0	18	43	0	0.000	0.000	0.141	E94	"
    "4:*-18-43-*	Def\n"
    "5	0	18	43	0	0.000	0.000	0.141	E94	"
    "5:*-18-43-*	Def\n"
    "0	1	18	43	1	-0.914	-0.482	0.179	X94\n"
    "0	1	18	43	3	-0.392	-2.724	0.312	X94\n"
    "0	1	18	43	28	-1.508	-1.816	-0.175	X94\n"
    "0	1	18	43	37	0.823	-1.220	-0.770	X94\n"
    "0	32	18	43	1	1.588	1.499	1.410	X94\n"
    "0	32	18	43	3	0.653	0.254	0.000	X94\n"
    "0	32	18	43	28	0.528	0.342	0.000	X94\n"
    "0	32	18	43	37	0.812	1.513	1.266	X94\n"
    "0	37	18	43	1	-1.139	-0.703	1.088	X94\n"
    "0	37	18	43	28	-2.014	-1.646	-2.068	X94\n"
    "0	37	18	43	37	-1.519	-0.328	1.437	X94\n"
    "0	43	18	43	28	3.011	-1.405	2.038	X94\n"
    "0	0	18	48	0	0.000	0.000	0.400	C94	"
    "0:*-18-48-*	Def\n"
    "0	1	18	48	28	1.767	1.606	0.408	X94\n"
    "0	32	18	48	28	-1.463	-2.548	0.310	X94\n"
    "0	0	18	55	0	0.000	0.000	0.000	E94	"
    "0:*-18-55-*	Def\n"
    "0	0	18	58	0	0.000	0.000	0.000	E94	"
    "0:*-18-58-*	Def\n"
    "0	0	18	62	0	0.000	0.000	0.500	C94	"
    "0:*-18-62-*	Def\n"
    "0	1	18	62	1	-0.403	-0.273	0.440	X94\n"
    "0	32	18	62	1	0.291	0.385	0.582	X94\n"
    "0	0	18	63	0	0.000	0.000	0.000	E94	"
    "0:*-18-63-*	Def\n"
    "0	0	18	64	0	0.000	0.000	0.000	E94	"
    "0:*-18-64-*	Def\n"
    "0	0	18	80	0	0.000	0.000	0.000	E94	"
    "0:*-18-80-*	Def\n"
    "0	0	19	20	0	0.000	0.000	0.179	E94	"
    "0:*-19-20-*	Def\n"
    "4	0	19	20	0	0.000	0.000	0.179	E94	"
    "4:*-19-20-*	Def\n"
    "0	0	19	37	0	0.000	0.000	0.000	E94	"
    "0:*-19-37-*	Def\n"
    "0	0	19	40	0	0.000	0.000	0.225	E94	"
    "0:*-19-40-*	Def\n"
    "0	0	19	63	0	0.000	0.000	0.000	E94	"
    "0:*-19-63-*	Def\n"
    "0	0	19	75	0	0.000	0.000	0.000	E94	"
    "0:*-19-75-*	Def\n"
    "0	0	20	20	0	0.000	0.000	0.200	C94	"
    "0:*-20-20-*	Def\n"
    "4	0	20	20	0	0.000	0.000	0.000	C94	"
    "4:*-20-20-*	Def\n"
    "5	0	20	20	0	0.000	0.000	0.236	E94	"
    "5:*-20-20-*	Def\n"
    "0	1	20	20	5	0.067	0.081	0.347	C94\n"
    "0	1	20	20	20	-0.063	-0.064	0.140	C94\n"
    "0	3	20	20	5	0.000	0.000	0.083	C94\n"
    "0	3	20	20	20	0.000	0.000	0.000	C94\n"
    "0	5	20	20	5	0.000	0.000	0.424	C94\n"
    "0	5	20	20	6	0.000	0.000	-0.080	C94\n"
    "0	5	20	20	8	0.000	0.127	0.450	C94\n"
    "0	5	20	20	12	-0.072	-0.269	0.439	C94\n"
    "0	5	20	20	20	-0.057	0.000	0.307	C94\n"
    "4	6	20	20	20	0.000	0.000	0.000	C94\n"
    "4	8	20	20	20	0.000	-0.091	0.192	C94\n"
    "0	12	20	20	20	0.077	0.202	0.183	C94\n"
    "4	20	20	20	20	0.000	0.000	0.000	C94\n"
    "0	0	20	22	0	0.000	0.000	0.236	E94	"
    "0:*-20-22-*	Def\n"
    "4	0	20	22	0	0.000	0.000	0.236	E94	"
    "4:*-20-22-*	Def\n"
    "0	0	20	25	0	0.000	0.000	0.251	E94	"
    "0:*-20-25-*	Def\n"
    "4	0	20	25	0	0.000	0.000	0.251	E94	"
    "4:*-20-25-*	Def\n"
    "0	0	20	26	0	0.000	0.000	0.376	E94	"
    "0:*-20-26-*	Def\n"
    "4	0	20	26	0	0.000	0.000	0.376	E94	"
    "4:*-20-26-*	Def\n"
    "5	0	20	26	0	0.000	0.000	0.376	E94	"
    "5:*-20-26-*	Def\n"
    "0	0	20	30	0	0.000	0.000	0.000	E94	"
    "0:*-20-30-*	Def\n"
    "2	0	20	30	0	0.000	0.000	0.000	E94	"
    "2:*-20-30-*	Def\n"
    "4	0	20	30	0	0.000	0.000	0.000	E94	"
    "4:*-20-30-*	Def\n"
    "0	0	20	30	30	0.000	0.000	-0.500	C94	"
    "0:*-20-30=30	Def\n"
    "0	0	20	34	0	0.000	0.000	0.198	E94	"
    "0:*-20-34-*	Def\n"
    "4	0	20	34	0	0.000	0.000	0.198	E94	"
    "4:*-20-34-*	Def\n"
    "0	0	20	37	0	0.000	0.000	0.000	E94	"
    "0:*-20-37-*	Def\n"
    "4	0	20	37	0	0.000	0.000	0.000	E94	"
    "4:*-20-37-*	Def\n"
    "0	0	20	40	0	0.000	0.000	0.297	E94	"
    "0:*-20-40-*	Def\n"
    "0	0	20	41	0	0.000	0.000	0.000	E94	"
    "0:*-20-41-*	Def\n"
    "0	0	20	43	0	0.000	0.000	0.297	E94	"
    "0:*-20-43-*	Def\n"
    "4	0	20	43	0	0.000	0.000	0.297	E94	"
    "4:*-20-43-*	Def\n"
    "0	0	20	45	0	0.000	0.000	0.000	E94	"
    "0:*-20-45-*	Def\n"
    "0	0	22	22	0	0.000	0.000	0.236	E94	"
    "0:*-22-22-*	Def\n"
    "4	0	22	22	0	0.000	0.000	0.236	E94	"
    "4:*-22-22-*	Def\n"
    "5	0	22	22	0	0.000	0.000	0.236	E94	"
    "5:*-22-22-*	Def\n"
    "0	0	22	30	0	0.000	0.000	0.000	E94	"
    "0:*-22-30-*	Def\n"
    "4	0	22	30	0	0.000	0.000	0.000	E94	"
    "4:*-22-30-*	Def\n"
    "0	0	22	34	0	0.000	0.000	0.198	E94	"
    "0:*-22-34-*	Def\n"
    "0	0	22	37	0	0.000	0.000	0.000	E94	"
    "0:*-22-37-*	Def\n"
    "0	0	22	40	0	0.000	0.000	0.297	E94	"
    "0:*-22-40-*	Def\n"
    "0	0	22	41	0	0.000	0.000	0.000	E94	"
    "0:*-22-41-*	Def\n"
    "0	0	22	43	0	0.000	0.000	0.297	E94	"
    "0:*-22-43-*	Def\n"
    "5	0	22	43	0	0.000	0.000	0.297	E94	"
    "5:*-22-43-*	Def\n"
    "0	0	22	45	0	0.000	0.000	0.000	E94	"
    "0:*-22-45-*	Def\n"
    "0	0	25	25	0	0.000	0.000	0.267	E94	"
    "0:*-25-25-*	Def\n"
    "0	0	25	37	0	0.000	0.000	0.000	E94	"
    "0:*-25-37-*	Def\n"
    "5	0	25	37	0	0.000	0.000	0.000	E94	"
    "5:*-25-37-*	Def\n"
    "0	0	25	39	0	0.000	0.000	0.000	E94	"
    "0:*-25-39-*	Def\n"
    "0	0	25	40	0	0.000	0.000	0.316	E94	"
    "0:*-25-40-*	Def\n"
    "5	0	25	40	0	0.000	0.000	0.316	E94	"
    "5:*-25-40-*	Def\n"
    "0	0	25	43	0	0.000	0.000	0.250	C94	"
    "0:*-25-43-*	Def\n"
    "0	1	25	43	1	-2.686	-1.512	0.591	X94\n"
    "0	1	25	43	28	-3.730	-0.531	0.000	X94\n"
    "0	32	25	43	1	2.108	1.896	0.965	X94\n"
    "0	32	25	43	28	2.977	0.732	-0.502	X94\n"
    "0	0	25	57	0	0.000	0.000	0.000	E94	"
    "0:*-25-57-*	Def\n"
    "0	0	25	63	0	0.000	0.000	0.000	E94	"
    "0:*-25-63-*	Def\n"
    "0	0	26	26	0	0.000	0.000	0.600	E94	"
    "0:*-26-26-*	Def\n"
    "5	0	26	26	0	0.000	0.000	0.600	E94	"
    "5:*-26-26-*	Def\n"
    "0	0	26	34	0	0.000	0.000	0.316	E94	"
    "0:*-26-34-*	Def\n"
    "5	0	26	34	0	0.000	0.000	0.316	E94	"
    "5:*-26-34-*	Def\n"
    "0	0	26	37	0	0.000	1.423	0.000	E94	"
    "0:*-26-37-*	Def\n"
    "0	0	26	40	0	0.000	0.000	0.474	E94	"
    "0:*-26-40-*	Def\n"
    "0	0	30	30	0	0.000	12.000	0.000	E94	"
    "0:*-30-30-*	Def\n"
    "1	0	30	30	0	0.000	1.800	0.000	E94	"
    "1:*-30-30-*	Def\n"
    "4	0	30	30	0	0.000	1.800	0.000	E94	"
    "4:*-30-30-*	Def\n"
    "0	0	30	40	0	0.000	3.600	0.000	E94	"
    "0:*-30-40-*	Def\n"
    "1	0	30	67	0	0.000	1.800	0.000	E94	"
    "1:*-30-67-*	Def\n"
    "0	0	34	37	0	0.000	0.000	0.000	E94	"
    "0:*-34-37-*	Def\n"
    "0	0	34	43	0	0.000	0.000	0.250	E94	"
    "0:*-34-43-*	Def\n"
    "0	0	37	37	0	0.000	7.000	0.000	C94	"
    "0:*-37-37-*	Def\n"
    "1	0	37	37	0	0.000	2.000	0.000	#E94	"
    "1:*-37-37-*	Def\n"
    "4	0	37	37	0	0.000	6.000	0.000	E94	"
    "4:*-37-37-*	Def\n"
    "5	0	37	37	0	0.000	6.000	0.000	E94	"
    "5:*-37-37-*	Def\n"
    "0	1	37	37	5	0.000	7.000	0.000	C94\n"
    "0	1	37	37	37	0.000	7.000	0.000	C94\n"
    "0	2	37	37	5	0.000	7.000	0.000	C94\n"
    "0	2	37	37	37	0.000	7.000	0.000	C94\n"
    "0	3	37	37	5	0.000	7.000	0.000	C94\n"
    "0	3	37	37	37	0.000	7.000	0.000	C94\n"
    "0	5	37	37	5	0.000	7.000	0.000	C94\n"
    "0	5	37	37	6	0.000	7.000	0.000	C94\n"
    "0	5	37	37	15	0.000	7.000	0.000	C94\n"
    "0	5	37	37	18	0.000	7.000	0.000	X94\n"
    "0	5	37	37	37	0.000	7.000	0.000	C94\n"
    "0	5	37	37	40	0.000	7.000	0.000	C94\n"
    "0	5	37	37	43	0.000	7.000	0.000	X94\n"
    "0	6	37	37	37	0.000	7.000	0.000	C94\n"
    "0	15	37	37	37	0.000	7.000	0.000	C94\n"
    "0	18	37	37	37	0.000	7.000	0.000	X94\n"
    "0	37	37	37	37	0.000	7.000	0.000	C94\n"
    "0	37	37	37	40	0.000	7.000	0.000	C94\n"
    "0	37	37	37	43	0.000	7.000	0.000	X94\n"
    "0	0	37	38	0	0.000	7.000	0.000	C94	"
    "0:*-37-38-*	Def\n"
    "0	0	37	39	0	0.000	3.600	0.000	E94	"
    "0:*-37-39-*	Def\n"
    "1	0	37	39	0	0.000	6.000	0.000	E94	"
    "1:*-37-39-*	Def\n"
    "0	0	37	40	0	0.000	4.000	0.000	C94	"
    "0:*-37-40-*	Def\n"
    "5	0	37	40	0	0.000	3.600	0.000	E94	"
    "5:*-37-40-*	Def\n"
    "0	37	37	40	1	0.000	4.336	0.370	C94\n"
    "0	37	37	40	28	0.715	2.628	3.355	C94\n"
    "0	0	37	41	0	0.000	1.800	0.000	E94	"
    "0:*-37-41-*	Def\n"
    "0	0	37	43	0	0.000	2.000	1.800	C94	"
    "0:*-37-43-*	Def\n"
    "5	0	37	43	0	0.000	3.600	0.000	E94	"
    "5:*-37-43-*	Def\n"
    "0	37	37	43	18	0.372	2.284	2.034	X94\n"
    "0	37	37	43	28	0.000	1.694	1.508	X94\n"
    "0	0	37	45	0	0.000	1.800	0.000	E94	"
    "0:*-37-45-*	Def\n"
    "0	0	37	46	0	0.000	1.800	0.000	E94	"
    "0:*-37-46-*	Def\n"
    "0	0	37	55	0	0.000	4.800	0.000	E94	"
    "0:*-37-55-*	Def\n"
    "0	0	37	56	0	0.000	4.800	0.000	E94	"
    "0:*-37-56-*	Def\n"
    "1	0	37	57	0	0.000	1.800	0.000	E94	"
    "1:*-37-57-*	Def\n"
    "0	0	37	58	0	0.000	6.000	0.000	E94	"
    "0:*-37-58-*	Def\n"
    "1	0	37	58	0	0.000	4.800	0.000	E94	"
    "1:*-37-58-*	Def\n"
    "0	0	37	62	0	0.000	3.600	0.000	E94	"
    "0:*-37-62-*	Def\n"
    "0	0	37	63	0	0.000	7.000	0.000	C94	"
    "0:*-37-63-*	Def\n"
    "1	0	37	63	0	0.000	1.800	0.000	E94	"
    "1:*-37-63-*	Def\n"
    "0	0	37	64	0	0.000	7.000	0.000	C94	"
    "0:*-37-64-*	Def\n"
    "1	0	37	64	0	0.000	1.800	0.000	E94	"
    "1:*-37-64-*	Def\n"
    "1	0	37	67	0	0.000	1.800	0.000	E94	"
    "1:*-37-67-*	Def\n"
    "0	0	37	69	0	0.000	7.000	0.000	C94	"
    "0:*-37-69-*	Def\n"
    "0	0	37	78	0	0.000	6.000	0.000	E94	"
    "0:*-37-78-*	Def\n"
    "0	0	37	81	0	0.000	6.000	0.000	E94	"
    "0:*-37-81-*	Def\n"
    "1	0	37	81	0	0.000	4.800	0.000	E94	"
    "1:*-37-81-*	Def\n"
    "0	0	38	38	0	0.000	7.000	0.000	C94	"
    "0:*-38-38-*	Def\n"
    "0	0	38	58	0	0.000	7.000	0.000	C94	"
    "0:*-38-58-*	Def\n"
    "0	0	38	63	0	0.000	7.000	0.000	C94	"
    "0:*-38-63-*	Def\n"
    "0	0	38	64	0	0.000	7.000	0.000	C94	"
    "0:*-38-64-*	Def\n"
    "0	0	38	69	0	0.000	6.000	0.000	E94	"
    "0:*-38-69-*	Def\n"
    "0	0	38	78	0	0.000	6.000	0.000	E94	"
    "0:*-38-78-*	Def\n"
    "0	0	39	40	0	0.000	0.000	0.000	E94	"
    "0:*-39-40-*	Def\n"
    "0	0	39	45	0	0.000	6.000	0.000	E94	"
    "0:*-39-45-*	Def\n"
    "0	0	39	63	0	0.000	4.000	0.000	C94	"
    "0:*-39-63-*	Def\n"
    "1	0	39	63	0	0.000	6.000	0.000	E94	"
    "1:*-39-63-*	Def\n"
    "5	0	39	63	0	0.000	3.600	0.000	E94	"
    "5:*-39-63-*	Def\n"
    "0	1	39	63	5	0.000	4.000	0.000	C94\n"
    "0	1	39	63	64	0.000	4.000	0.000	C94\n"
    "0	18	39	63	5	0.000	4.000	0.000	X94\n"
    "0	18	39	63	64	0.000	4.000	0.000	X94\n"
    "0	63	39	63	5	0.000	4.000	0.000	C94\n"
    "0	63	39	63	64	0.000	4.000	0.000	C94\n"
    "0	0	39	64	0	0.000	3.600	0.000	E94	"
    "0:*-39-64-*	Def\n"
    "1	0	39	64	0	0.000	6.000	0.000	E94	"
    "1:*-39-64-*	Def\n"
    "0	0	39	65	0	0.000	4.000	0.000	C94	"
    "0:*-39-65-*	Def\n"
    "0	0	39	78	0	0.000	3.600	0.000	E94	"
    "0:*-39-78-*	Def\n"
    "0	0	40	40	0	0.000	0.000	0.375	E94	"
    "0:*-40-40-*	Def\n"
    "0	0	40	45	0	0.000	3.600	0.000	E94	"
    "0:*-40-45-*	Def\n"
    "0	0	40	46	0	0.000	3.600	0.000	E94	"
    "0:*-40-46-*	Def\n"
    "0	0	40	54	0	0.000	3.600	0.000	E94	"
    "0:*-40-54-*	Def\n"
    "2	0	40	54	0	0.000	3.600	0.000	E94	"
    "2:*-40-54-*	Def\n"
    "0	0	40	63	0	0.000	3.600	0.000	E94	"
    "0:*-40-63-*	Def\n"
    "0	0	40	64	0	0.000	3.600	0.000	E94	"
    "0:*-40-64-*	Def\n"
    "0	0	40	78	0	0.000	3.600	0.000	E94	"
    "0:*-40-78-*	Def\n"
    "0	0	41	41	0	0.000	1.800	0.000	E94	"
    "0:*-41-41-*	Def\n"
    "0	0	41	55	0	0.000	4.800	0.000	E94	"
    "0:*-41-55-*	Def\n"
    "0	0	41	62	0	0.000	3.600	0.000	E94	"
    "0:*-41-62-*	Def\n"
    "0	0	41	80	0	0.000	1.800	0.000	E94	"
    "0:*-41-80-*	Def\n"
    "0	0	43	43	0	0.000	0.000	0.375	E94	"
    "0:*-43-43-*	Def\n"
    "0	0	43	45	0	0.000	3.600	0.000	E94	"
    "0:*-43-45-*	Def\n"
    "0	0	43	64	0	0.000	3.600	0.000	E94	"
    "0:*-43-64-*	Def\n"
    "0	0	44	57	0	0.000	7.000	0.000	C94	"
    "0:*-44-57-*	Def\n"
    "0	0	44	63	0	0.000	7.000	0.000	C94	"
    "0:*-44-63-*	Def\n"
    "0	0	44	65	0	0.000	7.000	0.000	C94	"
    "0:*-44-65-*	Def\n"
    "0	0	44	78	0	0.000	2.846	0.000	E94	"
    "0:*-44-78-*	Def\n"
    "0	0	44	80	0	0.000	2.846	0.000	E94	"
    "0:*-44-80-*	Def\n"
    "0	0	45	63	0	0.000	1.800	0.000	E94	"
    "0:*-45-63-*	Def\n"
    "0	0	45	64	0	0.000	1.800	0.000	E94	"
    "0:*-45-64-*	Def\n"
    "0	0	45	78	0	0.000	1.800	0.000	E94	"
    "0:*-45-78-*	Def\n"
    "0	0	55	57	0	0.000	10.000	0.000	C94	"
    "0:*-55-57-*	Def\n"
    "2	0	55	57	0	0.000	4.800	0.000	E94	"
    "2:*-55-57-*	Def\n"
    "5	0	55	57	0	0.000	4.800	0.000	E94	"
    "5:*-55-57-*	Def\n"
    "0	1	55	57	5	0.423	12.064	0.090	C94\n"
    "0	1	55	57	55	-0.428	12.044	0.000	C94\n"
    "0	36	55	57	5	-0.268	8.077	-0.806	C94\n"
    "0	36	55	57	55	0.273	8.025	0.692	C94\n"
    "0	0	55	62	0	0.000	3.600	0.000	E94	"
    "0:*-55-62-*	Def\n"
    "0	0	55	64	0	0.000	4.800	0.000	E94	"
    "0:*-55-64-*	Def\n"
    "0	0	55	80	0	0.000	4.800	0.000	E94	"
    "0:*-55-80-*	Def\n"
    "0	0	56	57	0	0.000	6.000	0.000	C94	"
    "0:*-56-57-*	Def\n"
    "0	1	56	57	56	0.000	6.886	-0.161	C94\n"
    "0	36	56	57	56	0.000	4.688	0.107	C94\n"
    "0	0	56	63	0	0.000	4.800	0.000	E94	"
    "0:*-56-63-*	Def\n"
    "0	0	56	80	0	0.000	4.800	0.000	E94	"
    "0:*-56-80-*	Def\n"
    "1	0	57	63	0	0.000	1.800	0.000	E94	"
    "1:*-57-63-*	Def\n"
    "1	0	57	64	0	0.000	1.800	0.000	E94	"
    "1:*-57-64-*	Def\n"
    "0	0	58	63	0	0.000	6.000	0.000	E94	"
    "0:*-58-63-*	Def\n"
    "0	0	58	64	0	0.000	6.000	0.000	E94	"
    "0:*-58-64-*	Def\n"
    "0	0	59	63	0	0.000	7.000	0.000	C94	"
    "0:*-59-63-*	Def\n"
    "0	0	59	65	0	0.000	7.000	0.000	C94	"
    "0:*-59-65-*	Def\n"
    "0	0	59	78	0	0.000	3.600	0.000	E94	"
    "0:*-59-78-*	Def\n"
    "0	0	59	80	0	0.000	3.600	0.000	E94	"
    "0:*-59-80-*	Def\n"
    "0	0	59	82	0	0.000	3.600	0.000	E94	"
    "0:*-59-82-*	Def\n"
    "0	0	62	63	0	0.000	3.600	0.000	E94	"
    "0:*-62-63-*	Def\n"
    "0	0	62	64	0	0.000	3.600	0.000	E94	"
    "0:*-62-64-*	Def\n"
    "1	0	63	63	0	0.000	1.800	0.000	E94	"
    "1:*-63-63-*	Def\n"
    "0	0	63	64	0	0.000	7.000	0.000	C94	"
    "0:*-63-64-*	Def\n"
    "0	5	63	64	5	0.000	7.000	0.000	C94\n"
    "0	5	63	64	64	0.000	7.000	0.000	C94\n"
    "0	39	63	64	5	0.000	7.000	0.000	C94\n"
    "0	39	63	64	64	0.000	7.000	0.000	C94\n"
    "0	0	63	66	0	0.000	7.000	0.000	C94	"
    "0:*-63-66-*	Def\n"
    "0	0	63	78	0	0.000	6.000	0.000	E94	"
    "0:*-63-78-*	Def\n"
    "0	0	63	81	0	0.000	6.000	0.000	E94	"
    "0:*-63-81-*	Def\n"
    "0	0	64	64	0	0.000	7.000	0.000	C94	"
    "0:*-64-64-*	Def\n"
    "1	0	64	64	0	0.000	1.800	0.000	E94	"
    "1:*-64-64-*	Def\n"
    "0	5	64	64	5	0.000	7.000	0.000	C94\n"
    "0	5	64	64	63	0.000	7.000	0.000	C94\n"
    "0	63	64	64	63	0.000	7.000	0.000	C94\n"
    "0	0	64	65	0	0.000	7.000	0.000	C94	"
    "0:*-64-65-*	Def\n"
    "0	0	64	66	0	0.000	7.000	0.000	C94	"
    "0:*-64-66-*	Def\n"
    "0	0	64	78	0	0.000	6.000	0.000	E94	"
    "0:*-64-78-*	Def\n"
    "0	0	64	81	0	0.000	6.000	0.000	E94	"
    "0:*-64-81-*	Def\n"
    "5	0	64	81	0	0.000	6.000	0.000	E94	"
    "5:*-64-81-*	Def\n"
    "0	0	64	82	0	0.000	6.000	0.000	E94	"
    "0:*-64-82-*	Def\n"
    "0	0	65	66	0	0.000	7.000	0.000	C94	"
    "0:*-65-66-*	Def\n"
    "0	0	65	78	0	0.000	6.000	0.000	E94	"
    "0:*-65-78-*	Def\n"
    "0	0	65	81	0	0.000	6.000	0.000	E94	"
    "0:*-65-81-*	Def\n"
    "0	0	65	82	0	0.000	6.000	0.000	E94	"
    "0:*-65-82-*	Def\n"
    "0	0	66	66	0	0.000	7.000	0.000	C94	"
    "0:*-66-66-*	Def\n"
    "0	0	66	78	0	0.000	6.000	0.000	E94	"
    "0:*-66-78-*	Def\n"
    "0	0	66	81	0	0.000	6.000	0.000	E94	"
    "0:*-66-81-*	Def\n"
    "0	0	67	67	0	0.000	12.000	0.000	E94	"
    "0:*-67-67-*	Def\n"
    "5	0	67	67	0	0.000	12.000	0.000	E94	"
    "5:*-67-67-*	Def\n"
    "0	0	76	76	0	0.000	3.600	0.000	E94	"
    "0:*-76-76-*	Def\n"
    "0	0	76	78	0	0.000	3.600	0.000	E94	"
    "0:*-76-78-*	Def\n"
    "0	0	78	78	0	0.000	7.000	0.000	C94	"
    "0:*-78-78-*	Def\n"
    "0	0	78	79	0	0.000	6.000	0.000	E94	"
    "0:*-78-79-*	Def\n"
    "0	0	78	81	0	0.000	4.000	0.000	C94	"
    "0:*-78-81-*	Def\n"
    "0	0	79	79	0	0.000	6.000	0.000	E94	"
    "0:*-79-79-*	Def\n"
    "0	0	79	81	0	0.000	6.000	0.000	E94	"
    "0:*-79-81-*	Def\n"
    "0	0	80	81	0	0.000	4.000	0.000	C94	"
    "0:*-80-81-*	Def\n";
const std::string defaultMMFFsTor =
    "*\n"
    "*   Copyright (c) Merck and Co., Inc., 1994,1995,1996,1997,1998,1999\n"
    "*                         All Rights Reserved\n"
    "*\n"
    "* MMFF94s TORSION PARAMETERS\n"
    "* C94  - CORE MMFF94 parameter - from fits to conformational energies\n"
    "* C94S - CORE MMFF94s parameter - from fits to conformational energies\n"
    "* X94  - EXTD MMFF94 parameter - also from fits to conformational E's\n"
    "* E94  - EXTD MMFF94 parameter - from empirical rule\n"
    "* #E94 - Adjusted from empirical rule value\n"
    "*\n"
    "*      atom types       V1      V2      V3     Source\n"
    "0	0	1	1	0	0.000	0.000	0.300	C94	"
    "0:*-1-1-*	Def\n"
    "5	0	1	1	0	0.200	-0.800	1.500	C94	"
    "5:*-1-1-*	Def\n"
    "0	1	1	1	1	0.103	0.681	0.332	C94\n"
    "5	1	1	1	1	0.144	-0.547	1.126	C94\n"
    "0	1	1	1	2	-0.295	0.438	0.584	C94\n"
    "0	1	1	1	3	0.066	-0.156	0.143	C94\n"
    "0	1	1	1	5	0.639	-0.630	0.264	C94\n"
    "0	1	1	1	6	-0.688	1.757	0.477	C94\n"
    "5	1	1	1	6	0.000	0.000	0.054	C94\n"
    "0	1	1	1	8	-1.420	-0.092	1.101	C94\n"
    "5	1	1	1	8	0.000	-0.158	0.323	C94\n"
    "0	1	1	1	11	0.593	0.662	1.120	C94\n"
    "0	1	1	1	12	-0.678	0.417	0.624	C94\n"
    "0	1	1	1	15	-0.714	0.698	0.000	C94\n"
    "0	1	1	1	34	-0.647	0.550	0.590	C94\n"
    "0	2	1	1	5	0.321	-0.411	0.144	C94\n"
    "0	3	1	1	3	0.443	0.000	-1.140	C94\n"
    "0	3	1	1	5	-0.256	0.058	0.000	C94\n"
    "0	3	1	1	6	-0.679	-0.029	0.000	C94\n"
    "0	5	1	1	5	0.284	-1.386	0.314	C94\n"
    "0	5	1	1	6	-0.654	1.072	0.279	C94\n"
    "0	5	1	1	8	-0.744	-1.235	0.337	C94\n"
    "0	5	1	1	10	0.000	0.000	0.418	C94S\n"
    "0	5	1	1	11	0.000	0.516	0.291	C94\n"
    "0	5	1	1	12	0.678	-0.602	0.398	C94\n"
    "0	5	1	1	15	1.142	-0.644	0.367	C94\n"
    "0	5	1	1	25	0.000	0.000	0.295	X94\n"
    "0	5	1	1	34	0.692	-0.530	0.278	C94\n"
    "0	5	1	1	37	0.000	0.000	0.389	C94\n"
    "0	5	1	1	39	0.000	0.000	0.278	C94\n"
    "0	5	1	1	41	0.000	0.000	-0.141	C94\n"
    "0	5	1	1	56	0.000	0.000	0.324	C94\n"
    "0	5	1	1	68	0.000	0.000	0.136	C94\n"
    "0	6	1	1	6	0.408	1.397	0.961	C94\n"
    "5	6	1	1	6	0.313	-1.035	1.631	C94\n"
    "0	8	1	1	8	1.055	0.834	0.000	C94\n"
    "0	11	1	1	11	-0.387	-0.543	1.405	C94\n"
    "0	12	1	1	12	0.000	0.000	0.893	C94\n"
    "0	15	1	1	15	-0.177	0.000	0.049	C94\n"
    "0	0	1	2	0	0.000	0.000	0.000	C94	"
    "0:*-1-2-*	Def\n"
    "2	0	1	2	0	0.000	0.000	0.000	E94	"
    "2:*1-2-*	Def\n"
    "5	0	1	2	0	0.000	0.000	0.000	C94	"
    "5:*-1-2-*	Def\n"
    "0	0	1	2	2	0.000	0.000	-0.650	C94	"
    "0:*-1-2=2	Def\n"
    "5	0	1	2	2	0.000	0.000	-0.650	C94	"
    "5:*-1-2=2	Def\n"
    "0	1	1	2	1	0.419	0.296	0.282	C94\n"
    "0	1	1	2	2	-0.494	0.274	-0.630	C94\n"
    "0	1	1	2	5	0.075	0.000	0.358	C94\n"
    "0	2	1	2	2	-0.293	0.115	-0.508	C94\n"
    "0	2	1	2	5	0.301	0.104	0.507	C94\n"
    "0	3	1	2	1	0.565	-0.554	0.234	C94\n"
    "0	3	1	2	2	-0.577	-0.482	-0.427	C94\n"
    "0	3	1	2	5	0.082	0.000	0.123	C94\n"
    "0	5	1	2	1	0.000	-0.184	0.220	C94\n"
    "0	5	1	2	2	0.501	-0.410	-0.535	C94\n"
    "2	5	1	2	2	0.000	0.000	0.055	C94\n"
    "2	5	1	2	3	0.000	0.000	-0.108	C94\n"
    "0	5	1	2	5	-0.523	-0.228	0.208	C94\n"
    "2	5	1	2	37	0.000	0.000	0.000	C94\n"
    "0	6	1	2	1	-0.467	0.000	0.490	C94\n"
    "0	6	1	2	2	0.425	0.168	-0.875	C94\n"
    "0	6	1	2	5	0.000	0.136	0.396	C94\n"
    "0	8	1	2	1	-0.504	0.371	0.557	C94\n"
    "0	8	1	2	2	0.541	0.539	-1.009	C94\n"
    "0	8	1	2	5	0.000	0.204	0.464	C94\n"
    "0	0	1	3	0	0.000	0.400	0.300	C94	"
    "0:*-1-3-*	Def\n"
    "2	0	1	3	0	0.000	0.500	0.350	C94	"
    "2:*-1-3-*	Def\n"
    "5	0	1	3	0	0.000	0.000	0.000	E94	"
    "5:*1-3-*	Def\n"
    "0	0	1	3	1	0.000	0.000	0.550	C94	"
    "0:*-1-3-1	Def\n"
    "0	0	1	3	5	0.000	0.200	0.700	C94	"
    "0:*-1-3-5	Def\n"
    "0	0	1	3	7	0.000	0.400	0.400	C94	"
    "0:*-1-3-7	Def\n"
    "0	1	1	3	1	0.103	0.177	0.545	C94\n"
    "0	1	1	3	5	-0.072	0.316	0.674	C94\n"
    "0	1	1	3	6	-0.117	-0.333	0.202	C94\n"
    "0	1	1	3	7	0.825	0.139	0.325	C94\n"
    "0	1	1	3	10	-0.763	1.244	0.986	C94S\n"
    "0	2	1	3	5	0.663	-0.167	0.426	C94\n"
    "0	2	1	3	7	-0.758	0.112	0.563	C94\n"
    "0	5	1	3	1	-0.073	0.085	0.531	C94\n"
    "2	5	1	3	2	0.000	0.000	0.115	C94\n"
    "2	5	1	3	3	0.000	0.000	0.446	C94\n"
    "0	5	1	3	5	-0.822	0.501	1.008	C94\n"
    "0	5	1	3	6	0.000	-0.624	0.330	C94\n"
    "0	5	1	3	7	0.659	-1.407	0.308	C94\n"
    "0	5	1	3	10	-0.687	1.244	0.136	C94S\n"
    "2	5	1	3	37	0.000	0.000	0.056	C94\n"
    "0	5	1	3	43	0.000	1.027	0.360	X94\n"
    "0	5	1	3	51	0.000	1.543	0.350	X94\n"
    "0	5	1	3	53	0.000	0.501	0.000	X94\n"
    "0	5	1	3	74	0.000	0.513	-0.344	X94\n"
    "0	5	1	3	75	0.000	0.511	-0.186	X94\n"
    "0	6	1	3	6	0.447	0.652	0.318	C94\n"
    "0	6	1	3	7	-0.395	0.730	-0.139	C94\n"
    "0	10	1	3	7	0.530	2.905	2.756	C94S\n"
    "0	10	1	3	10	0.465	-0.241	1.850	C94S\n"
    "0	0	1	4	0	0.000	0.000	0.000	C94	"
    "0:*-1-4-*	Def\n"
    "0	0	1	6	0	0.000	0.000	0.200	C94	"
    "0:*-1-6-*	Def\n"
    "5	0	1	6	0	0.000	-0.200	0.400	C94	"
    "5:*-1-6-*	Def\n"
    "0	1	1	6	1	-0.681	0.755	0.755	C94\n"
    "5	1	1	6	1	0.000	0.243	-0.596	C94\n"
    "0	1	1	6	3	-0.547	0.000	0.320	C94\n"
    "0	1	1	6	21	0.000	0.270	0.237	C94\n"
    "0	2	1	6	21	0.102	0.460	-0.128	C94\n"
    "0	3	1	6	21	-1.652	-1.660	0.283	C94\n"
    "0	5	1	6	1	0.571	0.319	0.570	C94\n"
    "0	5	1	6	2	0.000	0.000	0.306	C94\n"
    "0	5	1	6	3	0.572	0.000	-0.304	C94\n"
    "0	5	1	6	21	0.596	-0.276	0.346	C94\n"
    "0	5	1	6	25	0.000	0.000	0.061	X94\n"
    "0	5	1	6	37	0.000	0.000	0.106	C94\n"
    "0	5	1	6	45	0.000	0.000	-0.174	X94\n"
    "0	6	1	6	1	0.229	-0.710	0.722	C94\n"
    "5	6	1	6	1	0.000	0.000	0.040	C94\n"
    "0	6	1	6	21	1.488	-3.401	-0.320	C94\n"
    "0	37	1	6	21	0.712	1.320	-0.507	C94\n"
    "0	0	1	8	0	0.000	-0.300	0.500	C94	"
    "0:*-1-8-*	Def\n"
    "5	0	1	8	0	0.000	0.000	0.297	E94	"
    "5:*1-8-*	Def\n"
    "0	1	1	8	1	-0.439	0.786	0.272	C94\n"
    "5	1	1	8	1	0.115	-0.390	0.658	C94\n"
    "0	1	1	8	6	-0.608	0.339	1.496	C94\n"
    "0	1	1	8	23	-0.428	0.323	0.280	C94\n"
    "0	2	1	8	23	0.594	-0.409	0.155	C94\n"
    "0	5	1	8	1	0.393	-0.385	0.562	C94\n"
    "0	5	1	8	6	0.598	-0.158	0.399	C94\n"
    "0	5	1	8	23	-0.152	-0.440	0.357	C94\n"
    "0	0	1	9	0	0.000	0.000	0.000	C94	"
    "0:*-1-9-*	Def\n"
    "5	0	1	9	0	0.000	0.000	0.000	E94	"
    "5:*1-9-*	Def\n"
    "0	5	1	9	3	0.204	-0.335	-0.352	C94\n"
    "0	5	1	9	53	0.000	0.000	0.097	X94\n"
    "0	0	1	10	0	0.000	0.000	0.300	C94	"
    "0:*-1-10-*	Def\n"
    "5	0	1	10	0	0.000	0.000	0.000	E94	"
    "5:*1-10-*	Def\n"
    "0	0	1	10	3	0.000	0.000	1.000	C94	"
    "0:*-1-10-3	Def\n"
    "0	1	1	10	3	-0.884	0.578	0.818	C94S\n"
    "0	1	1	10	6	0.000	-0.379	0.565	C94S\n"
    "0	1	1	10	28	0.750	-0.404	0.369	C94S\n"
    "0	3	1	10	3	3.219	-2.699	1.875	C94S\n"
    "0	3	1	10	28	0.207	0.461	0.324	C94S\n"
    "0	5	1	10	1	0.000	0.000	0.706	C94S\n"
    "0	5	1	10	3	-2.334	1.517	-0.065	C94S\n"
    "0	5	1	10	6	0.000	0.688	0.665	C94S\n"
    "0	5	1	10	28	-0.982	-0.207	0.166	C94S\n"
    "0	0	1	15	0	0.000	0.000	0.400	C94	"
    "0:*-1-15-*	Def\n"
    "5	0	1	15	0	0.000	0.000	0.336	E94	"
    "5:*1-15-*	Def\n"
    "0	1	1	15	1	-1.047	0.170	0.398	C94\n"
    "0	1	1	15	15	-1.438	0.263	0.501	C94\n"
    "0	1	1	15	71	-0.376	-0.133	0.288	C94\n"
    "0	5	1	15	1	1.143	-0.231	0.447	C94\n"
    "0	5	1	15	15	1.555	-0.323	0.456	C94\n"
    "0	5	1	15	37	0.000	0.000	0.459	C94\n"
    "0	5	1	15	71	0.229	0.203	0.440	C94\n"
    "0	0	1	17	0	0.000	0.000	0.350	C94	"
    "0:*-1-17-*	Def\n"
    "5	0	1	17	0	0.000	0.000	0.000	E94	"
    "5:*1-17-*	Def\n"
    "0	5	1	17	1	0.000	0.000	0.536	X94\n"
    "0	5	1	17	7	0.000	0.000	0.212	X94\n"
    "0	0	1	18	0	0.000	0.000	0.100	C94	"
    "0:*-1-18-*	Def\n"
    "5	0	1	18	0	0.000	0.000	0.112	E94	"
    "5:*1-18-*	Def\n"
    "0	5	1	18	1	0.000	0.000	0.000	X94\n"
    "0	5	1	18	6	0.000	0.000	0.099	X94\n"
    "0	5	1	18	32	0.000	0.585	0.388	X94\n"
    "0	5	1	18	43	0.000	-0.412	0.121	X94\n"
    "0	5	1	18	48	0.000	0.000	0.195	X94\n"
    "0	5	1	18	62	0.000	0.000	-0.088	X94\n"
    "0	0	1	19	0	0.000	0.000	0.150	C94	"
    "0:*-1-19-*	Def\n"
    "5	0	1	19	0	0.000	0.000	0.179	E94	"
    "5:*1-19-*	Def\n"
    "0	5	1	19	5	0.000	0.000	0.196	X94\n"
    "0	5	1	19	6	0.000	0.000	0.176	X94\n"
    "0	5	1	19	12	0.000	0.000	0.152	X94\n"
    "0	0	1	20	0	0.000	0.000	0.350	C94	"
    "0:*-1-20-*	Def\n"
    "5	0	1	20	0	0.000	0.000	0.350	C94	"
    "5:*-1-20-*	Def\n"
    "0	5	1	20	5	0.000	0.000	0.344	C94\n"
    "0	5	1	20	20	0.000	0.000	0.361	C94\n"
    "0	0	1	22	0	0.000	0.000	0.236	E94	"
    "0:*1-22-*	Def\n"
    "5	0	1	22	0	0.000	0.000	0.236	E94	"
    "5:*1-22-*	Def\n"
    "0	0	1	25	0	0.000	0.000	0.300	C94	"
    "0:*-1-25-*	Def\n"
    "5	0	1	25	0	0.000	0.000	0.251	E94	"
    "5:*1-25-*	Def\n"
    "0	1	1	25	1	0.000	-0.207	0.232	X94\n"
    "0	1	1	25	32	0.000	0.288	0.218	X94\n"
    "0	5	1	25	1	0.000	0.152	0.235	X94\n"
    "0	5	1	25	6	0.000	0.000	0.495	X94\n"
    "0	5	1	25	32	0.000	-0.130	0.214	X94\n"
    "0	5	1	25	43	0.000	0.000	0.466	X94\n"
    "0	5	1	25	72	0.000	0.000	0.243	X94\n"
    "0	0	1	26	0	0.000	0.000	0.450	C94	"
    "0:*-1-26-*	Def\n"
    "5	0	1	26	0	0.000	0.000	0.376	E94	"
    "5:*1-26-*	Def\n"
    "0	5	1	26	12	0.000	0.000	0.439	X94\n"
    "0	5	1	26	71	0.000	0.000	0.472	X94\n"
    "0	0	1	34	0	0.000	0.000	0.250	C94	"
    "0:*-1-34-*	Def\n"
    "5	0	1	34	0	0.000	0.000	0.198	E94	"
    "5:*1-34-*	Def\n"
    "0	1	1	34	36	0.000	0.000	0.187	C94\n"
    "0	5	1	34	1	0.000	0.000	0.247	C94\n"
    "0	5	1	34	36	0.000	0.000	0.259	C94\n"
    "0	0	1	37	0	0.000	0.000	0.200	C94	"
    "0:*-1-37-*	Def\n"
    "5	0	1	37	0	0.000	0.000	0.000	E94	"
    "5:*1-37-*	Def\n"
    "0	1	1	37	37	0.000	0.449	0.000	C94\n"
    "0	5	1	37	37	0.000	-0.420	0.391	C94\n"
    "0	6	1	37	37	0.000	0.000	0.150	C94\n"
    "0	0	1	39	0	0.000	0.000	0.000	C94	"
    "0:*-1-39-*	Def\n"
    "5	0	1	39	0	0.000	0.000	0.000	E94	"
    "5:*1-39-*	Def\n"
    "0	1	1	39	63	0.000	-0.080	-0.056	C94\n"
    "0	5	1	39	63	0.000	0.000	-0.113	C94\n"
    "0	0	1	40	0	0.000	0.000	0.250	C94	"
    "0:*-1-40-*	Def\n"
    "5	0	1	40	0	0.000	0.000	0.297	E94	"
    "5:*1-40-*	Def\n"
    "0	5	1	40	28	0.000	-0.105	0.000	C94S\n"
    "0	5	1	40	37	0.000	0.000	0.468	C94S\n"
    "0	0	1	41	0	0.000	0.600	0.000	C94	"
    "0:*-1-41-*	Def\n"
    "0	1	1	41	32	0.000	1.263	0.000	C94\n"
    "0	5	1	41	32	0.000	0.000	-0.106	C94\n"
    "0	5	1	41	72	0.000	0.632	0.000	X94\n"
    "0	0	1	43	0	0.000	0.000	0.150	C94	"
    "0:*-1-43-*	Def\n"
    "5	0	1	43	0	0.000	0.000	0.297	E94	"
    "5:*1-43-*	Def\n"
    "0	5	1	43	18	0.357	-0.918	0.000	X94\n"
    "0	5	1	43	25	0.000	0.000	0.061	X94\n"
    "0	5	1	43	28	-0.249	0.382	0.343	X94\n"
    "0	0	1	45	0	0.000	0.000	0.100	C94	"
    "0:*-1-45-*	Def\n"
    "0	5	1	45	32	0.000	0.000	0.125	X94\n"
    "0	0	1	46	0	0.000	0.000	-0.500	C94	"
    "0:*-1-46-*	Def\n"
    "0	5	1	46	7	0.000	0.000	-0.540	X94\n"
    "0	0	1	54	0	0.000	0.000	0.000	C94	"
    "0:*-1-54-*	Def\n"
    "2	0	1	54	0	0.000	0.000	0.000	E94	"
    "2:*1-54-*	Def\n"
    "5	0	1	54	0	0.000	0.000	0.000	E94	"
    "5:*1-54-*	Def\n"
    "0	5	1	54	3	0.000	0.000	-0.315	C94\n"
    "0	5	1	54	36	0.000	0.000	0.315	C94\n"
    "0	0	1	55	0	0.000	0.000	0.000	C94	"
    "0:*-1-55-*	Def\n"
    "5	0	1	55	0	0.000	0.000	0.000	E94	"
    "5:*1-55-*	Def\n"
    "0	5	1	55	36	0.000	-0.058	0.084	C94\n"
    "0	5	1	55	57	0.000	-0.058	-0.092	C94\n"
    "0	0	1	56	0	0.000	0.000	-0.300	C94	"
    "0:*-1-56-*	Def\n"
    "0	1	1	56	36	0.875	0.668	-0.015	C94\n"
    "0	1	1	56	57	-0.870	0.775	-0.406	C94\n"
    "0	5	1	56	36	-0.958	-0.629	-0.372	C94\n"
    "0	5	1	56	57	0.952	-0.715	-0.483	C94\n"
    "0	0	1	57	0	0.000	0.000	0.000	E94	"
    "0:*1-57-*	Def\n"
    "5	0	1	57	0	0.000	0.000	0.000	E94	"
    "5:*1-57-*	Def\n"
    "0	0	1	58	0	0.000	0.000	0.000	E94	"
    "0:*1-58-*	Def\n"
    "0	0	1	62	0	0.000	0.000	0.250	C94	"
    "0:*-1-62-*	Def\n"
    "0	5	1	62	18	0.000	0.000	0.270	X94\n"
    "0	0	1	63	0	0.000	0.000	0.000	E94	"
    "0:*1-63-*	Def\n"
    "5	0	1	63	0	0.000	0.000	0.000	E94	"
    "5:*1-63-*	Def\n"
    "0	0	1	64	0	0.000	0.000	0.000	E94	"
    "0:*1-64-*	Def\n"
    "5	0	1	64	0	0.000	0.000	0.000	E94	"
    "5:*1-64-*	Def\n"
    "0	0	1	67	0	0.000	0.000	0.000	E94	"
    "0:*1-67-*	Def\n"
    "5	0	1	67	0	0.000	0.000	0.000	E94	"
    "5:*1-67-*	Def\n"
    "0	0	1	68	0	0.000	0.000	0.400	C94	"
    "0:*-1-68-*	Def\n"
    "0	1	1	68	1	-0.117	0.090	0.751	C94\n"
    "0	1	1	68	23	0.373	0.153	0.635	C94\n"
    "0	1	1	68	32	-0.090	-0.169	0.075	C94\n"
    "0	5	1	68	1	0.134	-0.112	0.329	C94\n"
    "0	5	1	68	23	-0.361	-0.202	0.560	C94\n"
    "0	5	1	68	32	0.072	0.218	0.093	C94\n"
    "0	0	1	73	0	0.000	0.000	0.500	C94	"
    "0:*-1-73-*	Def\n"
    "0	5	1	73	32	0.000	0.000	0.509	X94\n"
    "0	5	1	73	72	0.000	0.000	0.443	X94\n"
    "0	0	1	75	0	0.000	0.000	0.000	E94	"
    "0:*1-75-*	Def\n"
    "0	0	1	78	0	0.000	0.000	0.000	E94	"
    "0:*1-78-*	Def\n"
    "0	0	1	80	0	0.000	0.000	0.000	E94	"
    "0:*1-80-*	Def\n"
    "0	0	1	81	0	0.000	0.000	0.000	E94	"
    "0:*1-81-*	Def\n"
    "0	0	2	2	0	0.000	12.000	0.000	C94	"
    "0:*-2=2-*	Def\n"
    "1	0	2	2	0	0.000	1.800	0.000	C94	"
    "1:*=2-2=*	Def\n"
    "5	0	2	2	0	0.000	12.000	0.000	C94	"
    "5:*-2=2-*	Def\n"
    "0	1	2	2	1	-0.403	12.000	0.000	C94\n"
    "0	1	2	2	2	0.000	12.000	0.000	C94\n"
    "1	1	2	2	2	-0.418	2.089	-0.310	C94\n"
    "0	1	2	2	5	0.000	12.000	0.000	C94\n"
    "1	1	2	2	5	0.412	2.120	0.269	C94\n"
    "1	2	2	2	2	0.094	1.621	0.877	C94\n"
    "0	2	2	2	5	0.000	12.000	0.000	C94\n"
    "1	2	2	2	5	0.317	1.421	-0.870	C94\n"
    "0	3	2	2	5	0.000	12.000	0.000	C94\n"
    "0	5	2	2	5	0.000	12.000	0.000	C94\n"
    "1	5	2	2	5	-0.406	1.767	0.000	C94\n"
    "0	5	2	2	6	0.000	12.000	0.000	C94\n"
    "0	5	2	2	37	0.000	12.000	0.000	C94\n"
    "0	5	2	2	40	0.000	12.000	0.000	C94\n"
    "0	5	2	2	41	0.000	12.000	0.000	C94\n"
    "0	5	2	2	45	0.000	12.000	0.000	X94\n"
    "0	5	2	2	62	0.000	12.000	0.000	X94\n"
    "1	0	2	3	0	0.000	2.500	0.000	#E94	"
    "0:*-2-3-*	Def\n"
    "1	1	2	3	1	0.136	1.798	0.630	C94\n"
    "1	1	2	3	5	0.497	2.405	0.357	C94\n"
    "1	1	2	3	6	-0.211	1.925	-0.131	C94\n"
    "1	1	2	3	7	-0.401	2.028	-0.318	C94\n"
    "1	1	2	3	10	0.000	2.237	-0.610	C94S\n"
    "1	2	2	3	1	-0.325	1.553	-0.487	C94\n"
    "1	2	2	3	5	-0.295	2.024	-0.590	C94\n"
    "1	2	2	3	6	-0.143	1.466	0.000	C94\n"
    "1	2	2	3	7	0.362	1.978	0.000	C94\n"
    "1	2	2	3	9	0.296	1.514	0.481	C94\n"
    "1	2	2	3	10	0.000	1.599	0.380	C94S\n"
    "1	5	2	3	1	0.213	1.728	-0.042	C94\n"
    "1	5	2	3	5	-0.208	1.622	0.223	C94\n"
    "1	5	2	3	6	0.359	1.539	0.194	C94\n"
    "1	5	2	3	7	0.000	2.046	0.000	C94\n"
    "1	5	2	3	9	-0.290	1.519	-0.470	C94\n"
    "1	5	2	3	10	0.000	1.409	0.254	C94S\n"
    "1	0	2	4	0	0.000	0.000	0.000	C94	"
    "0:*-2-4-*	Def\n"
    "0	0	2	6	0	0.000	3.100	0.000	C94	"
    "0:*-2-6-*	Def\n"
    "2	0	2	6	0	0.000	3.600	0.000	E94	"
    "2:*-2-6-*	Def\n"
    "5	0	2	6	0	0.000	3.600	0.000	E94	"
    "5:*-2-6-*	Def\n"
    "0	2	2	6	1	-1.953	3.953	-1.055	C94\n"
    "0	2	2	6	3	-1.712	2.596	-0.330	C94\n"
    "0	2	2	6	29	-0.215	2.810	-0.456	C94\n"
    "0	5	2	6	1	1.951	3.936	1.130	C94\n"
    "0	5	2	6	3	1.719	2.628	0.360	C94\n"
    "0	5	2	6	29	0.216	2.808	0.456	C94\n"
    "1	0	2	9	0	0.000	1.800	0.000	E94	"
    "1:*-2-9-*	Def\n"
    "0	0	2	10	0	0.000	6.000	0.000	E94	"
    "0:*-2-10-*	Def\n"
    "2	0	2	10	0	0.000	6.000	0.000	E94	"
    "2:*-2-10-*	Def\n"
    "5	0	2	10	0	0.000	6.000	0.000	E94	"
    "5:*-2-10-*	Def\n"
    "0	0	2	15	0	0.000	1.423	0.000	E94	"
    "0:*-2-15-*	Def\n"
    "2	0	2	15	0	0.000	1.423	0.000	E94	"
    "2:*-2-15-*	Def\n"
    "5	0	2	15	0	0.000	1.423	0.000	E94	"
    "5:*-2-15-*	Def\n"
    "0	0	2	17	0	0.000	1.423	0.000	E94	"
    "0:*-2-17-*	Def\n"
    "0	0	2	18	0	0.000	0.000	0.000	E94	"
    "0:*-2-18-*	Def\n"
    "2	0	2	18	0	0.000	0.000	0.000	E94	"
    "2:*-2-18-*	Def\n"
    "5	0	2	18	0	0.000	0.000	0.000	E94	"
    "5:*-2-18-*	Def\n"
    "0	0	2	19	0	0.000	0.000	0.000	E94	"
    "0:*-2-19-*	Def\n"
    "0	0	2	20	0	0.000	0.000	0.000	E94	"
    "0:*-2-20-*	Def\n"
    "2	0	2	20	0	0.000	0.000	0.000	E94	"
    "2:*-2-20-*	Def\n"
    "0	0	2	22	0	0.000	0.000	0.000	E94	"
    "0:*-2-22-*	Def\n"
    "2	0	2	22	0	0.000	0.000	0.000	E94	"
    "2:*-2-22-*	Def\n"
    "5	0	2	22	0	0.000	0.000	0.000	E94	"
    "5:*-2-22-*	Def\n"
    "0	0	2	25	0	0.000	0.000	0.000	E94	"
    "0:*-2-25-*	Def\n"
    "0	0	2	30	0	0.000	12.000	0.000	E94	"
    "0:*-2-30-*	Def\n"
    "0	0	2	34	0	0.000	0.000	0.000	E94	"
    "0:*-2-34-*	Def\n"
    "2	0	2	34	0	0.000	0.000	0.000	E94	"
    "2:*-2-34-*	Def\n"
    "1	0	2	37	0	0.000	2.000	0.000	C94	"
    "1:*-2-37-*	Def\n"
    "1	1	2	37	37	0.000	2.952	-0.079	C94\n"
    "1	2	2	37	37	0.000	1.542	0.434	C94\n"
    "1	5	2	37	37	0.000	1.308	-0.357	C94\n"
    "1	0	2	39	0	0.000	6.000	0.000	E94	"
    "1:*-2-39-*	Def\n"
    "0	0	2	40	0	0.000	3.700	0.000	C94	"
    "0:*-2-40-*	Def\n"
    "2	0	2	40	0	0.000	3.600	0.000	E94	"
    "2:*-2-40-*	Def\n"
    "5	0	2	40	0	0.000	3.600	0.000	E94	"
    "5:*-2-40-*	Def\n"
    "0	2	2	40	28	0.000	3.305	-0.530	C94S\n"
    "0	5	2	40	28	0.139	3.241	0.139	C94S\n"
    "0	0	2	41	0	0.000	1.200	0.000	C94	"
    "0:*-2-41-*	Def\n"
    "2	0	2	41	0	0.000	1.800	0.000	E94	"
    "2:*-2-41-*	Def\n"
    "0	2	2	41	32	0.000	1.235	0.000	C94\n"
    "0	5	2	41	32	0.000	1.231	0.000	C94\n"
    "0	0	2	43	0	0.000	3.600	0.000	E94	"
    "0:*-2-43-*	Def\n"
    "2	0	2	43	0	0.000	3.600	0.000	E94	"
    "2:*-2-43-*	Def\n"
    "0	0	2	45	0	0.000	2.200	0.000	C94	"
    "0:*-2-45-*	Def\n"
    "2	0	2	45	0	0.000	1.800	0.000	E94	"
    "2:*-2-45-*	Def\n"
    "0	2	2	45	32	0.000	2.212	0.000	X94\n"
    "0	5	2	45	32	0.000	2.225	0.000	X94\n"
    "0	0	2	46	0	0.000	1.800	0.000	E94	"
    "0:*-2-46-*	Def\n"
    "2	0	2	46	0	0.000	1.800	0.000	E94	"
    "2:*-2-46-*	Def\n"
    "0	0	2	55	0	0.000	4.800	0.000	E94	"
    "0:*-2-55-*	Def\n"
    "0	0	2	56	0	0.000	4.800	0.000	E94	"
    "0:*-2-56-*	Def\n"
    "0	0	2	62	0	0.000	8.000	0.000	C94	"
    "0:*-2-62-*	Def\n"
    "0	2	2	62	23	1.693	7.903	0.532	X94\n"
    "0	5	2	62	23	-1.696	7.897	-0.482	X94\n"
    "1	0	2	63	0	0.000	1.800	0.000	E94	"
    "1:*-2-63-*	Def\n"
    "1	0	2	64	0	0.000	1.800	0.000	E94	"
    "1:*-2-64-*	Def\n"
    "1	0	2	67	0	0.000	1.800	0.000	E94	"
    "1:*-2-67-*	Def\n"
    "1	0	2	81	0	0.000	4.800	0.000	E94	"
    "1:*-2-81-*	Def\n"
    "1	0	3	3	0	0.000	0.600	0.000	C94	"
    "0:*-3-3-*	Def\n"
    "4	0	3	3	0	0.000	1.800	0.000	E94	"
    "4:*-3-3-*	Def\n"
    "1	1	3	3	1	-0.486	0.714	0.000	C94\n"
    "1	1	3	3	6	-0.081	-0.125	0.132	C94\n"
    "1	1	3	3	7	1.053	1.327	0.000	C94\n"
    "1	5	3	3	6	0.000	0.188	0.436	C94\n"
    "1	5	3	3	7	0.000	0.177	-0.412	C94\n"
    "1	6	3	3	6	0.269	0.437	0.000	C94\n"
    "1	6	3	3	7	-0.495	0.793	-0.318	C94\n"
    "1	7	3	3	7	-0.260	1.084	0.193	C94\n"
    "0	0	3	6	0	0.000	5.500	0.000	C94	"
    "0:*-3-6-*	Def\n"
    "2	0	3	6	0	0.000	5.500	0.000	C94	"
    "2:*-3-6-*	Def\n"
    "4	0	3	6	0	0.000	3.600	0.000	E94	"
    "4:*-3-6-*	Def\n"
    "5	0	3	6	0	0.000	3.600	0.000	E94	"
    "5:*-3-6-*	Def\n"
    "0	1	3	6	1	-1.244	5.482	0.365	C94\n"
    "0	1	3	6	24	-1.166	5.078	-0.545	C94\n"
    "0	1	3	6	37	-0.677	5.854	0.521	C94\n"
    "2	2	3	6	24	0.256	4.519	0.258	C94\n"
    "2	3	3	6	24	1.663	4.073	0.094	C94\n"
    "0	5	3	6	1	0.526	5.631	0.691	C94\n"
    "0	5	3	6	2	0.159	6.586	0.216	C94\n"
    "0	5	3	6	24	-2.285	4.737	0.468	C94\n"
    "0	7	3	6	0	0.700	6.500	-0.400	C94	"
    "0:7-3-6-*	Def\n"
    "0	7	3	6	1	0.682	7.184	-0.935	C94\n"
    "0	7	3	6	2	-0.168	6.572	-0.151	C94\n"
    "0	7	3	6	24	1.662	6.152	-0.058	C94\n"
    "0	7	3	6	37	0.635	5.890	-0.446	C94\n"
    "2	37	3	6	24	0.000	3.892	-0.094	C94\n"
    "0	0	3	9	0	0.000	16.000	0.000	C94	"
    "0:*-3=9-*	Def\n"
    "1	0	3	9	0	0.000	1.800	0.000	E94	"
    "1:*-3-9-*	Def\n"
    "5	0	3	9	0	0.000	12.000	0.000	E94	"
    "5:*-3-9-*	Def\n"
    "0	2	3	9	27	0.000	16.000	0.000	C94\n"
    "0	5	3	9	1	0.687	16.152	0.894	C94\n"
    "0	5	3	9	27	0.000	16.000	0.000	C94\n"
    "0	40	3	9	1	-0.704	18.216	0.000	C94S\n"
    "0	40	3	9	27	0.000	16.000	0.178	C94S\n"
    "0	0	3	10	0	0.000	6.000	0.000	C94	"
    "0:*-3-10-*	Def\n"
    "2	0	3	10	0	0.000	6.000	0.000	C94	"
    "2:*-3-10-*	Def\n"
    "4	0	3	10	0	0.000	6.000	0.000	C94	"
    "4:*-3-10-*	Def\n"
    "5	0	3	10	0	0.000	6.000	0.000	E94	"
    "5:*-3-10-*	Def\n"
    "0	1	3	10	1	0.831	6.061	0.522	C94S\n"
    "0	1	3	10	6	-1.152	8.588	1.511	C94S\n"
    "0	1	3	10	28	-0.259	5.934	1.326	C94S\n"
    "2	2	3	10	28	0.000	6.561	0.294	C94S\n"
    "0	5	3	10	1	-0.195	6.304	1.722	C94S\n"
    "0	5	3	10	3	-0.705	5.383	0.234	C94S\n"
    "0	5	3	10	28	-0.417	5.981	0.511	C94S\n"
    "0	7	3	10	1	-0.491	6.218	0.000	C94S\n"
    "0	7	3	10	3	0.733	-0.543	-0.163	C94S\n"
    "0	7	3	10	6	1.234	8.372	-0.539	C94S\n"
    "0	7	3	10	28	1.168	4.857	-0.341	C94S\n"
    "0	10	3	10	28	0.000	3.706	1.254	C94S\n"
    "0	0	3	15	0	0.000	1.423	0.000	E94	"
    "0:*-3-15-*	Def\n"
    "2	0	3	15	0	0.000	1.423	0.000	E94	"
    "2:*-3-15-*	Def\n"
    "4	0	3	15	0	0.000	1.423	0.000	E94	"
    "4:*-3-15-*	Def\n"
    "5	0	3	15	0	0.000	1.423	0.000	E94	"
    "5:*-3-15-*	Def\n"
    "0	0	3	17	0	0.000	1.423	0.000	E94	"
    "0:*-3-17-*	Def\n"
    "5	0	3	17	0	0.000	1.423	0.000	E94	"
    "5:*-3-17-*	Def\n"
    "0	0	3	18	0	0.000	0.000	0.000	E94	"
    "0:*-3-18-*	Def\n"
    "2	0	3	18	0	0.000	0.000	0.000	E94	"
    "2:*-3-18-*	Def\n"
    "0	0	3	20	0	0.000	0.000	-0.300	C94	"
    "0:*-3-20-*	Def\n"
    "2	0	3	20	0	0.000	0.000	0.000	E94	"
    "2:*-3-20-*	Def\n"
    "4	0	3	20	0	0.000	0.000	-0.300	C94	"
    "4:*-3-20-*	Def\n"
    "5	0	3	20	0	0.000	0.000	0.000	E94	"
    "5:*-3-20-*	Def\n"
    "0	7	3	20	0	0.000	0.400	0.400	C94	"
    "0:7-3-20-*	Def\n"
    "0	7	3	20	5	0.000	0.000	-0.131	C94\n"
    "0	7	3	20	20	0.000	0.000	0.000	C94\n"
    "0	20	3	20	5	0.000	0.000	0.085	C94\n"
    "0	20	3	20	20	0.000	0.000	0.000	C94\n"
    "0	0	3	22	0	0.000	0.000	0.000	E94	"
    "0:*-3-22-*	Def\n"
    "2	0	3	22	0	0.000	0.000	0.000	E94	"
    "2:*-3-22-*	Def\n"
    "4	0	3	22	0	0.000	0.000	0.000	E94	"
    "4:*-3-22-*	Def\n"
    "5	0	3	22	0	0.000	0.000	0.000	E94	"
    "5:*-3-22-*	Def\n"
    "0	7	3	22	0	0.000	0.400	0.400	C94	"
    "0:7-3-22-*	Def\n"
    "0	0	3	25	0	0.000	0.000	0.000	E94	"
    "0:*-3-25-*	Def\n"
    "2	0	3	25	0	0.000	0.000	0.000	E94	"
    "2:*-3-25-*	Def\n"
    "1	0	3	30	0	0.000	1.800	0.000	E94	"
    "1:*-3-30-*	Def\n"
    "4	0	3	30	0	0.000	1.800	0.000	E94	"
    "4:*-3-30-*	Def\n"
    "1	0	3	37	0	0.000	2.500	0.000	#E94	"
    "1:*-3-37-*	Def\n"
    "4	0	3	37	0	0.000	1.800	0.000	E94	"
    "4:*-3-37-*	Def\n"
    "1	1	3	37	37	0.000	2.428	0.000	C94\n"
    "1	6	3	37	37	0.000	1.743	0.000	C94\n"
    "1	7	3	37	37	0.000	2.256	0.000	C94\n"
    "1	43	3	37	37	-0.241	3.385	-0.838	X94\n"
    "1	0	3	39	0	0.000	5.500	0.000	#E94	"
    "1:*-3-39-*	Def\n"
    "0	0	3	40	0	0.000	3.900	0.000	C94	"
    "0:*-3-40-*	Def\n"
    "2	0	3	40	0	0.000	3.600	0.000	E94	"
    "2:*-3-40-*	Def\n"
    "5	0	3	40	0	0.000	3.600	0.000	E94	"
    "5:*-3-40-*	Def\n"
    "0	5	3	40	28	-1.355	3.964	0.800	C94S\n"
    "0	9	3	40	28	1.045	3.785	-0.291	C94S\n"
    "0	40	3	40	28	0.508	2.985	0.809	C94S\n"
    "0	0	3	41	0	0.000	1.800	0.000	E94	"
    "0:*-3-41-*	Def\n"
    "2	0	3	41	0	0.000	1.800	0.000	E94	"
    "2:*-3-41-*	Def\n"
    "0	0	3	43	0	0.000	4.500	0.000	C94	"
    "0:*-3-43-*	Def\n"
    "2	0	3	43	0	0.000	3.600	0.000	E94	"
    "2:*-3-43-*	Def\n"
    "4	0	3	43	0	0.000	3.600	0.000	E94	"
    "4:*-3-43-*	Def\n"
    "5	0	3	43	0	0.000	3.600	0.000	E94	"
    "5:*-3-43-*	Def\n"
    "0	1	3	43	18	1.712	3.309	0.233	X94\n"
    "0	1	3	43	28	-0.414	4.168	-0.875	X94\n"
    "0	7	3	43	18	-0.880	5.091	-0.129	X94\n"
    "0	7	3	43	28	0.536	5.276	-0.556	X94\n"
    "2	37	3	43	18	-0.701	4.871	1.225	X94\n"
    "2	37	3	43	28	-0.086	5.073	0.878	X94\n"
    "0	0	3	45	0	0.000	1.800	0.000	E94	"
    "0:*-3-45-*	Def\n"
    "2	0	3	45	0	0.000	1.800	0.000	E94	"
    "2:*-3-45-*	Def\n"
    "0	0	3	48	0	0.000	0.000	0.892	E94	"
    "0:*-3-48-*	Def\n"
    "0	0	3	51	0	0.000	13.500	0.000	C94	"
    "0:*-3-51-*	Def\n"
    "0	1	3	51	52	0.000	13.549	0.000	X94\n"
    "0	0	3	54	0	0.000	8.000	0.000	C94	"
    "0:*-3-54-*	Def\n"
    "1	0	3	54	0	0.000	2.500	0.000	#E94	"
    "1:*-3-54-*	Def\n"
    "5	0	3	54	0	0.000	12.000	0.000	E94	"
    "5:*-3-54-*	Def\n"
    "0	5	3	54	1	0.000	8.000	0.000	C94\n"
    "0	5	3	54	36	0.000	8.000	0.000	C94\n"
    "0	0	3	55	0	0.000	4.800	0.000	E94	"
    "0:*-3-55-*	Def\n"
    "2	0	3	55	0	0.000	4.800	0.000	E94	"
    "2:*-3-55-*	Def\n"
    "0	0	3	56	0	0.000	4.800	0.000	E94	"
    "0:*-3-56-*	Def\n"
    "2	0	3	56	0	0.000	4.800	0.000	E94	"
    "2:*-3-56-*	Def\n"
    "1	0	3	57	0	0.000	2.500	0.000	#E94	"
    "1:*-3-57-*	Def\n"
    "1	0	3	58	0	0.000	4.800	0.000	E94	"
    "1:*-3-58-*	Def\n"
    "0	0	3	62	0	0.000	3.600	0.000	E94	"
    "0:*-3-62-*	Def\n"
    "2	0	3	62	0	0.000	3.600	0.000	E94	"
    "2:*-3-62-*	Def\n"
    "5	0	3	62	0	0.000	3.600	0.000	E94	"
    "5:*-3-62-*	Def\n"
    "1	0	3	63	0	0.000	2.500	0.000	#E94	"
    "1:*-3-63-*	Def\n"
    "1	0	3	64	0	0.000	2.500	0.000	#E94	"
    "1:*-3-64-*	Def\n"
    "0	0	3	67	0	0.000	12.000	0.000	E94	"
    "0:*-3-67-*	Def\n"
    "0	0	3	74	0	0.000	19.000	0.000	C94	"
    "0:*-3-74-*	Def\n"
    "0	1	3	74	7	0.000	19.349	0.000	X94\n"
    "0	0	3	75	0	0.000	19.000	0.000	C94	"
    "0:*-3-75-*	Def\n"
    "0	1	3	75	71	0.000	18.751	0.000	X94\n"
    "1	0	3	78	0	0.000	2.500	0.000	#E94	"
    "1:*-3-78-*	Def\n"
    "1	0	3	80	0	0.000	2.500	0.000	#E94	"
    "1:*-3-80-*	Def\n"
    "0	0	6	6	0	0.000	-2.000	0.000	E94	"
    "0:*-6-6-*	Def\n"
    "5	0	6	6	0	0.000	-2.000	0.000	E94	"
    "5:*-6-6-*	Def\n"
    "0	0	6	8	0	0.900	-1.100	-0.500	C94	"
    "0:*-6-8-*	Def\n"
    "5	0	6	8	0	0.000	0.000	0.274	E94	"
    "5:*-6-8-*	Def\n"
    "0	21	6	8	1	0.261	-0.330	-0.542	C94\n"
    "0	21	6	8	23	1.503	-1.853	-0.476	C94\n"
    "0	0	6	9	0	0.000	3.600	0.000	E94	"
    "0:*-6-9-*	Def\n"
    "5	0	6	9	0	0.000	3.600	0.000	E94	"
    "5:*-6-9-*	Def\n"
    "0	0	6	10	0	1.200	0.500	-1.000	C94	"
    "0:*-6-10-*	Def\n"
    "0	21	6	10	1	0.829	0.000	-0.730	C94S\n"
    "0	21	6	10	3	0.675	-0.185	-1.053	C94S\n"
    "0	0	6	15	0	0.000	-4.000	0.000	E94	"
    "0:*-6-15-*	Def\n"
    "0	0	6	17	0	0.000	1.423	0.000	E94	"
    "0:*-6-17-*	Def\n"
    "5	0	6	17	0	0.000	1.423	0.000	E94	"
    "5:*-6-17-*	Def\n"
    "0	0	6	18	0	0.000	0.000	0.100	C94	"
    "0:*-6-18-*	Def\n"
    "5	0	6	18	0	0.000	0.000	0.103	E94	"
    "5:*-6-18-*	Def\n"
    "0	33	6	18	1	-0.520	-0.471	-0.267	X94\n"
    "0	33	6	18	6	-1.623	0.204	0.438	X94\n"
    "0	33	6	18	32	1.616	0.425	0.191	X94\n"
    "0	0	6	19	0	0.000	0.000	0.150	C94	"
    "0:*-6-19-*	Def\n"
    "5	0	6	19	0	0.000	0.000	0.165	E94	"
    "5:*-6-19-*	Def\n"
    "0	21	6	19	1	-0.620	-0.329	0.303	X94\n"
    "0	21	6	19	5	0.683	0.220	0.000	X94\n"
    "0	0	6	20	0	0.000	0.000	0.400	C94	"
    "0:*-6-20-*	Def\n"
    "4	0	6	20	0	0.000	0.000	0.217	E94	"
    "4:*-6-20-*	Def\n"
    "5	0	6	20	0	0.000	0.000	0.217	E94	"
    "5:*-6-20-*	Def\n"
    "0	20	6	20	5	0.000	0.000	-0.079	C94\n"
    "4	20	6	20	20	0.000	0.000	0.000	C94\n"
    "0	0	6	22	0	0.000	0.000	0.217	E94	"
    "0:*-6-22-*	Def\n"
    "0	0	6	25	0	0.000	0.000	0.650	C94	"
    "0:*-6-25-*	Def\n"
    "5	0	6	25	0	0.000	0.000	0.231	E94	"
    "5:*-6-25-*	Def\n"
    "0	1	6	25	1	-1.704	-0.452	0.556	X94\n"
    "0	1	6	25	6	0.000	0.000	0.777	X94\n"
    "0	1	6	25	32	1.205	0.914	0.612	X94\n"
    "0	24	6	25	6	-3.209	-7.622	1.065	X94\n"
    "0	24	6	25	32	-5.891	-3.332	0.290	X94\n"
    "0	0	6	26	0	0.000	0.000	0.346	E94	"
    "0:*-6-26-*	Def\n"
    "0	0	6	30	0	0.000	3.600	0.000	E94	"
    "0:*-6-30-*	Def\n"
    "2	0	6	30	0	0.000	3.600	0.000	E94	"
    "2:*-6-30-*	Def\n"
    "0	0	6	37	0	0.000	3.200	0.000	C94	"
    "0:*-6-37-*	Def\n"
    "5	0	6	37	0	0.000	3.600	0.000	E94	"
    "5:*-6-37-*	Def\n"
    "0	1	6	37	37	0.000	4.382	0.000	C94\n"
    "0	3	6	37	37	0.000	2.576	0.000	C94\n"
    "0	29	6	37	37	0.000	2.801	0.000	C94\n"
    "0	0	6	39	0	0.000	0.000	0.000	E94	"
    "0:*-6-39-*	Def\n"
    "0	0	6	40	0	0.000	0.000	0.274	E94	"
    "0:*-6-40-*	Def\n"
    "0	0	6	41	0	0.000	3.600	0.000	E94	"
    "0:*-6-41-*	Def\n"
    "0	0	6	43	0	0.000	0.000	0.274	E94	"
    "0:*-6-43-*	Def\n"
    "0	0	6	45	0	0.000	6.000	0.000	C94	"
    "0:*-6-45-*	Def\n"
    "0	1	6	45	32	0.000	6.208	0.000	X94\n"
    "0	0	6	54	0	0.000	3.600	0.000	E94	"
    "0:*-6-54-*	Def\n"
    "0	0	6	55	0	0.000	3.600	0.000	E94	"
    "0:*-6-55-*	Def\n"
    "0	0	6	57	0	0.000	3.600	0.000	E94	"
    "0:*-6-57-*	Def\n"
    "0	0	6	58	0	0.000	3.600	0.000	E94	"
    "0:*-6-58-*	Def\n"
    "0	0	6	63	0	0.000	3.600	0.000	E94	"
    "0:*-6-63-*	Def\n"
    "0	0	6	64	0	0.000	3.600	0.000	E94	"
    "0:*-6-64-*	Def\n"
    "0	0	8	8	0	0.000	0.000	0.375	E94	"
    "0:*-8-8-*	Def\n"
    "5	0	8	8	0	0.000	0.000	0.375	E94	"
    "5:*-8-8-*	Def\n"
    "0	0	8	9	0	0.000	3.600	0.000	E94	"
    "0:*-8-9-*	Def\n"
    "5	0	8	9	0	0.000	3.600	0.000	E94	"
    "5:*-8-9-*	Def\n"
    "0	0	8	10	0	0.000	0.000	0.000	E94	"
    "0:*-8-10-*	Def\n"
    "4	0	8	10	0	0.000	0.000	0.000	E94	"
    "4:*-8-10-*	Def\n"
    "0	0	8	15	0	0.000	0.000	0.424	E94	"
    "0:*-8-15-*	Def\n"
    "0	0	8	17	0	0.000	1.423	0.000	E94	"
    "0:*-8-17-*	Def\n"
    "4	0	8	17	0	0.000	1.423	0.000	E94	"
    "4:*-8-17-*	Def\n"
    "5	0	8	17	0	0.000	1.423	0.000	E94	"
    "5:*-8-17-*	Def\n"
    "0	0	8	19	0	0.000	0.000	0.225	E94	"
    "0:*-8-19-*	Def\n"
    "0	0	8	20	0	0.000	0.000	0.350	C94	"
    "0:*-8-20-*	Def\n"
    "4	0	8	20	0	0.000	0.000	0.300	C94	"
    "4:*-8-20-*	Def\n"
    "5	0	8	20	0	0.000	0.000	0.297	E94	"
    "5:*-8-20-*	Def\n"
    "0	20	8	20	5	0.000	0.120	0.472	C94\n"
    "4	20	8	20	20	0.000	-0.097	0.200	C94\n"
    "0	23	8	20	5	-0.101	-0.324	0.371	C94\n"
    "0	23	8	20	20	0.107	0.253	0.151	C94\n"
    "0	0	8	22	0	0.000	0.000	0.297	E94	"
    "0:*-8-22-*	Def\n"
    "0	0	8	25	0	0.000	0.000	0.316	E94	"
    "0:*-8-25-*	Def\n"
    "5	0	8	25	0	0.000	0.000	0.316	E94	"
    "5:*-8-25-*	Def\n"
    "0	0	8	26	0	0.000	0.000	0.474	E94	"
    "0:*-8-26-*	Def\n"
    "5	0	8	26	0	0.000	0.000	0.474	E94	"
    "5:*-8-26-*	Def\n"
    "0	0	8	34	0	0.000	0.000	0.250	E94	"
    "0:*-8-34-*	Def\n"
    "0	0	8	39	0	0.000	0.000	0.000	E94	"
    "0:*-8-39-*	Def\n"
    "0	0	8	40	0	0.000	0.000	0.375	E94	"
    "0:*-8-40-*	Def\n"
    "0	0	8	43	0	0.000	0.000	0.375	E94	"
    "0:*-8-43-*	Def\n"
    "0	0	8	45	0	0.000	3.600	0.000	E94	"
    "0:*-8-45-*	Def\n"
    "0	0	8	46	0	0.000	3.600	0.000	E94	"
    "0:*-8-46-*	Def\n"
    "0	0	8	55	0	0.000	3.600	0.000	E94	"
    "0:*-8-55-*	Def\n"
    "0	0	8	56	0	0.000	3.600	0.000	E94	"
    "0:*-8-56-*	Def\n"
    "0	0	9	9	0	0.000	12.000	0.000	E94	"
    "0:*-9-9-*	Def\n"
    "1	0	9	9	0	0.000	1.800	0.000	E94	"
    "1:*-9-9-*	Def\n"
    "5	0	9	9	0	0.000	12.000	0.000	E94	"
    "5:*-9-9-*	Def\n"
    "0	0	9	10	0	0.000	6.000	0.000	E94	"
    "0:*-9-10-*	Def\n"
    "5	0	9	10	0	0.000	6.000	0.000	E94	"
    "5:*-9-10-*	Def\n"
    "0	0	9	15	0	0.000	1.423	0.000	E94	"
    "0:*-9-15-*	Def\n"
    "0	0	9	18	0	0.000	0.000	0.000	E94	"
    "0:*-9-18-*	Def\n"
    "0	0	9	19	0	0.000	0.000	0.000	E94	"
    "0:*-9-19-*	Def\n"
    "0	0	9	20	0	0.000	0.000	0.000	E94	"
    "0:*-9-20-*	Def\n"
    "0	0	9	25	0	0.000	0.000	0.000	E94	"
    "0:*-9-25-*	Def\n"
    "0	0	9	34	0	0.000	0.000	0.000	E94	"
    "0:*-9-34-*	Def\n"
    "5	0	9	34	0	0.000	0.000	0.000	E94	"
    "5:*-9-34-*	Def\n"
    "1	0	9	37	0	0.000	1.800	0.000	E94	"
    "1:*-9-37-*	Def\n"
    "1	0	9	39	0	0.000	6.000	0.000	E94	"
    "1:*-9-39-*	Def\n"
    "0	0	9	40	0	0.000	3.600	0.000	E94	"
    "0:*-9-40-*	Def\n"
    "0	0	9	41	0	0.000	4.800	0.000	E94	"
    "0:*-9-41-*	Def\n"
    "0	0	9	45	0	0.000	1.800	0.000	E94	"
    "0:*-9-45-*	Def\n"
    "0	0	9	54	0	0.000	12.000	0.000	E94	"
    "0:*-9-54-*	Def\n"
    "0	0	9	55	0	0.000	4.800	0.000	E94	"
    "0:*-9-55-*	Def\n"
    "0	0	9	56	0	0.000	4.800	0.000	E94	"
    "0:*-9-56-*	Def\n"
    "1	0	9	57	0	0.000	1.800	0.000	E94	"
    "1:*-9-57-*	Def\n"
    "0	0	9	62	0	0.000	3.600	0.000	E94	"
    "0:*-9-62-*	Def\n"
    "1	0	9	63	0	0.000	1.800	0.000	E94	"
    "1:*-9-63-*	Def\n"
    "1	0	9	64	0	0.000	1.800	0.000	E94	"
    "1:*-9-64-*	Def\n"
    "0	0	9	67	0	0.000	12.000	0.000	E94	"
    "0:*-9-67-*	Def\n"
    "1	0	9	78	0	0.000	1.800	0.000	E94	"
    "1:*-9-78-*	Def\n"
    "1	0	9	81	0	0.000	4.800	0.000	E94	"
    "1:*-9-81-*	Def\n"
    "0	0	10	10	0	0.000	0.000	0.000	E94	"
    "0:*-10-10-*	Def\n"
    "5	0	10	10	0	0.000	0.000	0.000	E94	"
    "5:*-10-10-*	Def\n"
    "0	0	10	15	0	0.000	0.000	0.000	E94	"
    "0:*-10-15-*	Def\n"
    "0	0	10	17	0	0.000	4.743	0.000	E94	"
    "0:*-10-17-*	Def\n"
    "0	0	10	20	0	0.000	0.000	0.000	E94	"
    "0:*-10-20-*	Def\n"
    "4	0	10	20	0	0.000	0.000	0.000	E94	"
    "4:*-10-20-*	Def\n"
    "5	0	10	20	0	0.000	0.000	0.000	E94	"
    "5:*-10-20-*	Def\n"
    "0	0	10	22	0	0.000	0.000	0.000	E94	"
    "0:*-10-22-*	Def\n"
    "0	0	10	25	0	0.000	0.000	0.000	E94	"
    "0:*-10-25-*	Def\n"
    "0	0	10	26	0	0.000	0.000	0.000	E94	"
    "0:*-10-26-*	Def\n"
    "5	0	10	26	0	0.000	0.000	0.000	E94	"
    "5:*-10-26-*	Def\n"
    "0	0	10	34	0	0.000	0.000	0.000	E94	"
    "0:*-10-34-*	Def\n"
    "0	0	10	37	0	0.000	6.000	0.000	E94	"
    "0:*-10-37-*	Def\n"
    "0	0	10	39	0	0.000	0.000	0.000	E94	"
    "0:*-10-39-*	Def\n"
    "0	0	10	40	0	0.000	0.000	0.000	E94	"
    "0:*-10-40-*	Def\n"
    "5	0	10	40	0	0.000	0.000	0.000	E94	"
    "5:*-10-40-*	Def\n"
    "0	0	10	41	0	0.000	6.000	0.000	E94	"
    "0:*-10-41-*	Def\n"
    "0	0	10	45	0	0.000	6.000	0.000	E94	"
    "0:*-10-45-*	Def\n"
    "0	0	10	63	0	0.000	6.000	0.000	E94	"
    "0:*-10-63-*	Def\n"
    "0	0	10	64	0	0.000	6.000	0.000	E94	"
    "0:*-10-64-*	Def\n"
    "0	0	15	15	0	-1.400	-8.300	1.000	C94	"
    "0:*-15-15-*	Def\n"
    "5	0	15	15	0	0.000	-8.000	0.000	E94	"
    "5:*-15-15-*	Def\n"
    "0	1	15	15	1	-1.663	-8.408	1.433	C94\n"
    "0	1	15	15	71	-1.088	-8.245	0.411	C94\n"
    "0	0	15	18	0	0.000	0.000	0.160	E94	"
    "0:*-15-18-*	Def\n"
    "0	0	15	19	0	0.000	0.000	0.255	E94	"
    "0:*-15-19-*	Def\n"
    "5	0	15	19	0	0.000	0.000	0.255	E94	"
    "5:*-15-19-*	Def\n"
    "0	0	15	20	0	0.000	0.000	0.336	E94	"
    "0:*-15-20-*	Def\n"
    "4	0	15	20	0	0.000	0.000	0.336	E94	"
    "4:*-15-20-*	Def\n"
    "0	0	15	22	0	0.000	0.000	0.336	E94	"
    "0:*-15-22-*	Def\n"
    "0	0	15	25	0	0.000	0.000	0.358	E94	"
    "0:*-15-25-*	Def\n"
    "4	0	15	25	0	0.000	0.000	0.358	E94	"
    "4:*-15-25-*	Def\n"
    "0	0	15	26	0	0.000	0.000	0.537	E94	"
    "0:*-15-26-*	Def\n"
    "0	0	15	30	0	0.000	1.423	0.000	E94	"
    "0:*-15-30-*	Def\n"
    "4	0	15	30	0	0.000	1.423	0.000	E94	"
    "4:*-15-30-*	Def\n"
    "0	0	15	37	0	0.000	1.300	0.000	C94	"
    "0:*-15-37-*	Def\n"
    "5	0	15	37	0	0.000	1.423	0.000	E94	"
    "5:*-15-37-*	Def\n"
    "0	1	15	37	37	0.000	2.177	0.000	C94\n"
    "0	71	15	37	37	0.000	0.505	0.333	C94\n"
    "0	0	15	40	0	0.000	0.000	0.424	E94	"
    "0:*-15-40-*	Def\n"
    "0	0	15	43	0	0.000	0.000	0.424	E94	"
    "0:*-15-43-*	Def\n"
    "0	0	15	57	0	0.000	1.423	0.000	E94	"
    "0:*-15-57-*	Def\n"
    "0	0	15	63	0	0.000	1.423	0.000	E94	"
    "0:*-15-63-*	Def\n"
    "0	0	15	64	0	0.000	1.423	0.000	E94	"
    "0:*-15-64-*	Def\n"
    "0	0	17	20	0	0.000	0.000	0.000	E94	"
    "0:*-17-20-*	Def\n"
    "4	0	17	20	0	0.000	0.000	0.000	E94	"
    "4:*-17-20-*	Def\n"
    "5	0	17	20	0	0.000	0.000	0.000	E94	"
    "5:*-17-20-*	Def\n"
    "0	0	17	22	0	0.000	0.000	0.000	E94	"
    "0:*-17-22-*	Def\n"
    "0	0	17	37	0	0.000	1.423	0.000	E94	"
    "0:*-17-37-*	Def\n"
    "0	0	17	43	0	0.000	3.795	0.000	E94	"
    "0:*-17-43-*	Def\n"
    "0	0	18	20	0	0.000	0.000	0.112	E94	"
    "0:*-18-20-*	Def\n"
    "4	0	18	20	0	0.000	0.000	0.112	E94	"
    "4:*-18-20-*	Def\n"
    "5	0	18	20	0	0.000	0.000	0.112	E94	"
    "5:*-18-20-*	Def\n"
    "0	0	18	22	0	0.000	0.000	0.112	E94	"
    "0:*-18-22-*	Def\n"
    "0	0	18	37	0	0.000	-1.200	-0.300	C94	"
    "0:*-18-37-*	Def\n"
    "0	32	18	37	37	-0.173	-0.965	-0.610	X94\n"
    "0	39	18	37	37	0.000	-0.760	0.227	X94\n"
    "0	43	18	37	37	0.228	-1.741	-0.371	X94\n"
    "0	0	18	39	0	0.000	0.000	0.500	C94	"
    "0:*-18-39-*	Def\n"
    "0	32	18	39	63	0.000	0.687	0.680	X94\n"
    "0	37	18	39	63	0.000	-0.513	0.357	X94\n"
    "0	0	18	43	0	0.000	0.000	0.350	C94	"
    "0:*-18-43-*	Def\n"
    "4	0	18	43	0	0.000	0.000	0.141	E94	"
    "4:*-18-43-*	Def\n"
    "5	0	18	43	0	0.000	0.000	0.141	E94	"
    "5:*-18-43-*	Def\n"
    "0	1	18	43	1	-0.914	-0.482	0.179	X94\n"
    "0	1	18	43	3	-0.392	-2.724	0.312	X94\n"
    "0	1	18	43	28	-1.508	-1.816	-0.175	X94\n"
    "0	1	18	43	37	0.823	-1.220	-0.770	X94\n"
    "0	32	18	43	1	1.588	1.499	1.410	X94\n"
    "0	32	18	43	3	0.653	0.254	0.000	X94\n"
    "0	32	18	43	28	0.528	0.342	0.000	X94\n"
    "0	32	18	43	37	0.812	1.513	1.266	X94\n"
    "0	37	18	43	1	-1.139	-0.703	1.088	X94\n"
    "0	37	18	43	28	-2.014	-1.646	-2.068	X94\n"
    "0	37	18	43	37	-1.519	-0.328	1.437	X94\n"
    "0	43	18	43	28	3.011	-1.405	2.038	X94\n"
    "0	0	18	48	0	0.000	0.000	0.400	C94	"
    "0:*-18-48-*	Def\n"
    "0	1	18	48	28	1.767	1.606	0.408	X94\n"
    "0	32	18	48	28	-1.463	-2.548	0.310	X94\n"
    "0	0	18	55	0	0.000	0.000	0.000	E94	"
    "0:*-18-55-*	Def\n"
    "0	0	18	58	0	0.000	0.000	0.000	E94	"
    "0:*-18-58-*	Def\n"
    "0	0	18	62	0	0.000	0.000	0.500	C94	"
    "0:*-18-62-*	Def\n"
    "0	1	18	62	1	-0.403	-0.273	0.440	X94\n"
    "0	32	18	62	1	0.291	0.385	0.582	X94\n"
    "0	0	18	63	0	0.000	0.000	0.000	E94	"
    "0:*-18-63-*	Def\n"
    "0	0	18	64	0	0.000	0.000	0.000	E94	"
    "0:*-18-64-*	Def\n"
    "0	0	18	80	0	0.000	0.000	0.000	E94	"
    "0:*-18-80-*	Def\n"
    "0	0	19	20	0	0.000	0.000	0.179	E94	"
    "0:*-19-20-*	Def\n"
    "4	0	19	20	0	0.000	0.000	0.179	E94	"
    "4:*-19-20-*	Def\n"
    "0	0	19	37	0	0.000	0.000	0.000	E94	"
    "0:*-19-37-*	Def\n"
    "0	0	19	40	0	0.000	0.000	0.225	E94	"
    "0:*-19-40-*	Def\n"
    "0	0	19	63	0	0.000	0.000	0.000	E94	"
    "0:*-19-63-*	Def\n"
    "0	0	19	75	0	0.000	0.000	0.000	E94	"
    "0:*-19-75-*	Def\n"
    "0	0	20	20	0	0.000	0.000	0.200	C94	"
    "0:*-20-20-*	Def\n"
    "4	0	20	20	0	0.000	0.000	0.000	C94	"
    "4:*-20-20-*	Def\n"
    "5	0	20	20	0	0.000	0.000	0.236	E94	"
    "5:*-20-20-*	Def\n"
    "0	1	20	20	5	0.067	0.081	0.347	C94\n"
    "0	1	20	20	20	-0.063	-0.064	0.140	C94\n"
    "0	3	20	20	5	0.000	0.000	0.083	C94\n"
    "0	3	20	20	20	0.000	0.000	0.000	C94\n"
    "0	5	20	20	5	0.000	0.000	0.424	C94\n"
    "0	5	20	20	6	0.000	0.000	-0.080	C94\n"
    "0	5	20	20	8	0.000	0.127	0.450	C94\n"
    "0	5	20	20	12	-0.072	-0.269	0.439	C94\n"
    "0	5	20	20	20	-0.057	0.000	0.307	C94\n"
    "4	6	20	20	20	0.000	0.000	0.000	C94\n"
    "4	8	20	20	20	0.000	-0.091	0.192	C94\n"
    "0	12	20	20	20	0.077	0.202	0.183	C94\n"
    "4	20	20	20	20	0.000	0.000	0.000	C94\n"
    "0	0	20	22	0	0.000	0.000	0.236	E94	"
    "0:*-20-22-*	Def\n"
    "4	0	20	22	0	0.000	0.000	0.236	E94	"
    "4:*-20-22-*	Def\n"
    "0	0	20	25	0	0.000	0.000	0.251	E94	"
    "0:*-20-25-*	Def\n"
    "4	0	20	25	0	0.000	0.000	0.251	E94	"
    "4:*-20-25-*	Def\n"
    "0	0	20	26	0	0.000	0.000	0.376	E94	"
    "0:*-20-26-*	Def\n"
    "4	0	20	26	0	0.000	0.000	0.376	E94	"
    "4:*-20-26-*	Def\n"
    "5	0	20	26	0	0.000	0.000	0.376	E94	"
    "5:*-20-26-*	Def\n"
    "0	0	20	30	0	0.000	0.000	0.000	E94	"
    "0:*-20-30-*	Def\n"
    "2	0	20	30	0	0.000	0.000	0.000	E94	"
    "2:*-20-30-*	Def\n"
    "4	0	20	30	0	0.000	0.000	0.000	E94	"
    "4:*-20-30-*	Def\n"
    "0	0	20	30	30	0.000	0.000	-0.500	C94	"
    "0:*-20-30=30	Def\n"
    "0	0	20	34	0	0.000	0.000	0.198	E94	"
    "0:*-20-34-*	Def\n"
    "4	0	20	34	0	0.000	0.000	0.198	E94	"
    "4:*-20-34-*	Def\n"
    "0	0	20	37	0	0.000	0.000	0.000	E94	"
    "0:*-20-37-*	Def\n"
    "4	0	20	37	0	0.000	0.000	0.000	E94	"
    "4:*-20-37-*	Def\n"
    "0	0	20	40	0	0.000	0.000	0.297	E94	"
    "0:*-20-40-*	Def\n"
    "0	0	20	41	0	0.000	0.000	0.000	E94	"
    "0:*-20-41-*	Def\n"
    "0	0	20	43	0	0.000	0.000	0.297	E94	"
    "0:*-20-43-*	Def\n"
    "4	0	20	43	0	0.000	0.000	0.297	E94	"
    "4:*-20-43-*	Def\n"
    "0	0	20	45	0	0.000	0.000	0.000	E94	"
    "0:*-20-45-*	Def\n"
    "0	0	22	22	0	0.000	0.000	0.236	E94	"
    "0:*-22-22-*	Def\n"
    "4	0	22	22	0	0.000	0.000	0.236	E94	"
    "4:*-22-22-*	Def\n"
    "5	0	22	22	0	0.000	0.000	0.236	E94	"
    "5:*-22-22-*	Def\n"
    "0	0	22	30	0	0.000	0.000	0.000	E94	"
    "0:*-22-30-*	Def\n"
    "4	0	22	30	0	0.000	0.000	0.000	E94	"
    "4:*-22-30-*	Def\n"
    "0	0	22	34	0	0.000	0.000	0.198	E94	"
    "0:*-22-34-*	Def\n"
    "0	0	22	37	0	0.000	0.000	0.000	E94	"
    "0:*-22-37-*	Def\n"
    "0	0	22	40	0	0.000	0.000	0.297	E94	"
    "0:*-22-40-*	Def\n"
    "0	0	22	41	0	0.000	0.000	0.000	E94	"
    "0:*-22-41-*	Def\n"
    "0	0	22	43	0	0.000	0.000	0.297	E94	"
    "0:*-22-43-*	Def\n"
    "5	0	22	43	0	0.000	0.000	0.297	E94	"
    "5:*-22-43-*	Def\n"
    "0	0	22	45	0	0.000	0.000	0.000	E94	"
    "0:*-22-45-*	Def\n"
    "0	0	25	25	0	0.000	0.000	0.267	E94	"
    "0:*-25-25-*	Def\n"
    "0	0	25	37	0	0.000	0.000	0.000	E94	"
    "0:*-25-37-*	Def\n"
    "5	0	25	37	0	0.000	0.000	0.000	E94	"
    "5:*-25-37-*	Def\n"
    "0	0	25	39	0	0.000	0.000	0.000	E94	"
    "0:*-25-39-*	Def\n"
    "0	0	25	40	0	0.000	0.000	0.316	E94	"
    "0:*-25-40-*	Def\n"
    "5	0	25	40	0	0.000	0.000	0.316	E94	"
    "5:*-25-40-*	Def\n"
    "0	0	25	43	0	0.000	0.000	0.250	C94	"
    "0:*-25-43-*	Def\n"
    "0	1	25	43	1	-2.686	-1.512	0.591	X94\n"
    "0	1	25	43	28	-3.730	-0.531	0.000	X94\n"
    "0	32	25	43	1	2.108	1.896	0.965	X94\n"
    "0	32	25	43	28	2.977	0.732	-0.502	X94\n"
    "0	0	25	57	0	0.000	0.000	0.000	E94	"
    "0:*-25-57-*	Def\n"
    "0	0	25	63	0	0.000	0.000	0.000	E94	"
    "0:*-25-63-*	Def\n"
    "0	0	26	26	0	0.000	0.000	0.600	E94	"
    "0:*-26-26-*	Def\n"
    "5	0	26	26	0	0.000	0.000	0.600	E94	"
    "5:*-26-26-*	Def\n"
    "0	0	26	34	0	0.000	0.000	0.316	E94	"
    "0:*-26-34-*	Def\n"
    "5	0	26	34	0	0.000	0.000	0.316	E94	"
    "5:*-26-34-*	Def\n"
    "0	0	26	37	0	0.000	1.423	0.000	E94	"
    "0:*-26-37-*	Def\n"
    "0	0	26	40	0	0.000	0.000	0.474	E94	"
    "0:*-26-40-*	Def\n"
    "0	0	30	30	0	0.000	12.000	0.000	E94	"
    "0:*-30-30-*	Def\n"
    "1	0	30	30	0	0.000	1.800	0.000	E94	"
    "1:*-30-30-*	Def\n"
    "4	0	30	30	0	0.000	1.800	0.000	E94	"
    "4:*-30-30-*	Def\n"
    "0	0	30	40	0	0.000	3.600	0.000	E94	"
    "0:*-30-40-*	Def\n"
    "1	0	30	67	0	0.000	1.800	0.000	E94	"
    "1:*-30-67-*	Def\n"
    "0	0	34	37	0	0.000	0.000	0.000	E94	"
    "0:*-34-37-*	Def\n"
    "0	0	34	43	0	0.000	0.000	0.250	E94	"
    "0:*-34-43-*	Def\n"
    "0	0	37	37	0	0.000	7.000	0.000	C94	"
    "0:*-37-37-*	Def\n"
    "1	0	37	37	0	0.000	2.000	0.000	#E94	"
    "1:*-37-37-*	Def\n"
    "4	0	37	37	0	0.000	6.000	0.000	E94	"
    "4:*-37-37-*	Def\n"
    "5	0	37	37	0	0.000	6.000	0.000	E94	"
    "5:*-37-37-*	Def\n"
    "0	1	37	37	5	0.000	7.000	0.000	C94\n"
    "0	1	37	37	37	0.000	7.000	0.000	C94\n"
    "0	2	37	37	5	0.000	7.000	0.000	C94\n"
    "0	2	37	37	37	0.000	7.000	0.000	C94\n"
    "0	3	37	37	5	0.000	7.000	0.000	C94\n"
    "0	3	37	37	37	0.000	7.000	0.000	C94\n"
    "0	5	37	37	5	0.000	7.000	0.000	C94\n"
    "0	5	37	37	6	0.000	7.000	0.000	C94\n"
    "0	5	37	37	15	0.000	7.000	0.000	C94\n"
    "0	5	37	37	18	0.000	7.000	0.000	X94\n"
    "0	5	37	37	37	0.000	7.000	0.000	C94\n"
    "0	5	37	37	40	0.000	7.000	0.000	C94\n"
    "0	5	37	37	43	0.000	7.000	0.000	X94\n"
    "0	6	37	37	37	0.000	7.000	0.000	C94\n"
    "0	15	37	37	37	0.000	7.000	0.000	C94\n"
    "0	18	37	37	37	0.000	7.000	0.000	X94\n"
    "0	37	37	37	37	0.000	7.000	0.000	C94\n"
    "0	37	37	37	40	0.000	7.000	0.000	C94\n"
    "0	37	37	37	43	0.000	7.000	0.000	X94\n"
    "0	0	37	38	0	0.000	7.000	0.000	C94	"
    "0:*-37-38-*	Def\n"
    "0	0	37	39	0	0.000	3.600	0.000	E94	"
    "0:*-37-39-*	Def\n"
    "1	0	37	39	0	0.000	6.000	0.000	E94	"
    "1:*-37-39-*	Def\n"
    "0	0	37	40	0	0.000	4.000	0.000	C94	"
    "0:*-37-40-*	Def\n"
    "5	0	37	40	0	0.000	3.600	0.000	E94	"
    "5:*-37-40-*	Def\n"
    "0	37	37	40	1	0.000	4.095	0.382	C94S\n"
    "0	37	37	40	28	0.698	2.542	3.072	C94S\n"
    "0	0	37	41	0	0.000	1.800	0.000	E94	"
    "0:*-37-41-*	Def\n"
    "0	0	37	43	0	0.000	2.000	1.800	C94	"
    "0:*-37-43-*	Def\n"
    "5	0	37	43	0	0.000	3.600	0.000	E94	"
    "5:*-37-43-*	Def\n"
    "0	37	37	43	18	0.372	2.284	2.034	X94\n"
    "0	37	37	43	28	0.000	1.694	1.508	X94\n"
    "0	0	37	45	0	0.000	1.800	0.000	E94	"
    "0:*-37-45-*	Def\n"
    "0	0	37	46	0	0.000	1.800	0.000	E94	"
    "0:*-37-46-*	Def\n"
    "0	0	37	55	0	0.000	4.800	0.000	E94	"
    "0:*-37-55-*	Def\n"
    "0	0	37	56	0	0.000	4.800	0.000	E94	"
    "0:*-37-56-*	Def\n"
    "1	0	37	57	0	0.000	1.800	0.000	E94	"
    "1:*-37-57-*	Def\n"
    "0	0	37	58	0	0.000	6.000	0.000	E94	"
    "0:*-37-58-*	Def\n"
    "1	0	37	58	0	0.000	4.800	0.000	E94	"
    "1:*-37-58-*	Def\n"
    "0	0	37	62	0	0.000	3.600	0.000	E94	"
    "0:*-37-62-*	Def\n"
    "0	0	37	63	0	0.000	7.000	0.000	C94	"
    "0:*-37-63-*	Def\n"
    "1	0	37	63	0	0.000	1.800	0.000	E94	"
    "1:*-37-63-*	Def\n"
    "0	0	37	64	0	0.000	7.000	0.000	C94	"
    "0:*-37-64-*	Def\n"
    "1	0	37	64	0	0.000	1.800	0.000	E94	"
    "1:*-37-64-*	Def\n"
    "1	0	37	67	0	0.000	1.800	0.000	E94	"
    "1:*-37-67-*	Def\n"
    "0	0	37	69	0	0.000	7.000	0.000	C94	"
    "0:*-37-69-*	Def\n"
    "0	0	37	78	0	0.000	6.000	0.000	E94	"
    "0:*-37-78-*	Def\n"
    "0	0	37	81	0	0.000	6.000	0.000	E94	"
    "0:*-37-81-*	Def\n"
    "1	0	37	81	0	0.000	4.800	0.000	E94	"
    "1:*-37-81-*	Def\n"
    "0	0	38	38	0	0.000	7.000	0.000	C94	"
    "0:*-38-38-*	Def\n"
    "0	0	38	58	0	0.000	7.000	0.000	C94	"
    "0:*-38-58-*	Def\n"
    "0	0	38	63	0	0.000	7.000	0.000	C94	"
    "0:*-38-63-*	Def\n"
    "0	0	38	64	0	0.000	7.000	0.000	C94	"
    "0:*-38-64-*	Def\n"
    "0	0	38	69	0	0.000	6.000	0.000	E94	"
    "0:*-38-69-*	Def\n"
    "0	0	38	78	0	0.000	6.000	0.000	E94	"
    "0:*-38-78-*	Def\n"
    "0	0	39	40	0	0.000	0.000	0.000	E94	"
    "0:*-39-40-*	Def\n"
    "0	0	39	45	0	0.000	6.000	0.000	E94	"
    "0:*-39-45-*	Def\n"
    "0	0	39	63	0	0.000	4.000	0.000	C94	"
    "0:*-39-63-*	Def\n"
    "1	0	39	63	0	0.000	6.000	0.000	E94	"
    "1:*-39-63-*	Def\n"
    "5	0	39	63	0	0.000	3.600	0.000	E94	"
    "5:*-39-63-*	Def\n"
    "0	1	39	63	5	0.000	4.000	0.000	C94\n"
    "0	1	39	63	64	0.000	4.000	0.000	C94\n"
    "0	18	39	63	5	0.000	4.000	0.000	X94\n"
    "0	18	39	63	64	0.000	4.000	0.000	X94\n"
    "0	63	39	63	5	0.000	4.000	0.000	C94\n"
    "0	63	39	63	64	0.000	4.000	0.000	C94\n"
    "0	0	39	64	0	0.000	3.600	0.000	E94	"
    "0:*-39-64-*	Def\n"
    "1	0	39	64	0	0.000	6.000	0.000	E94	"
    "1:*-39-64-*	Def\n"
    "0	0	39	65	0	0.000	4.000	0.000	C94	"
    "0:*-39-65-*	Def\n"
    "0	0	39	78	0	0.000	3.600	0.000	E94	"
    "0:*-39-78-*	Def\n"
    "0	0	40	40	0	0.000	0.000	0.375	E94	"
    "0:*-40-40-*	Def\n"
    "0	0	40	45	0	0.000	3.600	0.000	E94	"
    "0:*-40-45-*	Def\n"
    "0	0	40	46	0	0.000	3.600	0.000	E94	"
    "0:*-40-46-*	Def\n"
    "0	0	40	54	0	0.000	3.600	0.000	E94	"
    "0:*-40-54-*	Def\n"
    "2	0	40	54	0	0.000	3.600	0.000	E94	"
    "2:*-40-54-*	Def\n"
    "0	0	40	63	0	0.000	3.600	0.000	E94	"
    "0:*-40-63-*	Def\n"
    "0	0	40	64	0	0.000	3.600	0.000	E94	"
    "0:*-40-64-*	Def\n"
    "0	0	40	78	0	0.000	3.600	0.000	E94	"
    "0:*-40-78-*	Def\n"
    "0	0	41	41	0	0.000	1.800	0.000	E94	"
    "0:*-41-41-*	Def\n"
    "0	0	41	55	0	0.000	4.800	0.000	E94	"
    "0:*-41-55-*	Def\n"
    "0	0	41	62	0	0.000	3.600	0.000	E94	"
    "0:*-41-62-*	Def\n"
    "0	0	41	80	0	0.000	1.800	0.000	E94	"
    "0:*-41-80-*	Def\n"
    "0	0	43	43	0	0.000	0.000	0.375	E94	"
    "0:*-43-43-*	Def\n"
    "0	0	43	45	0	0.000	3.600	0.000	E94	"
    "0:*-43-45-*	Def\n"
    "0	0	43	64	0	0.000	3.600	0.000	E94	"
    "0:*-43-64-*	Def\n"
    "0	0	44	57	0	0.000	7.000	0.000	C94	"
    "0:*-44-57-*	Def\n"
    "0	0	44	63	0	0.000	7.000	0.000	C94	"
    "0:*-44-63-*	Def\n"
    "0	0	44	65	0	0.000	7.000	0.000	C94	"
    "0:*-44-65-*	Def\n"
    "0	0	44	78	0	0.000	2.846	0.000	E94	"
    "0:*-44-78-*	Def\n"
    "0	0	44	80	0	0.000	2.846	0.000	E94	"
    "0:*-44-80-*	Def\n"
    "0	0	45	63	0	0.000	1.800	0.000	E94	"
    "0:*-45-63-*	Def\n"
    "0	0	45	64	0	0.000	1.800	0.000	E94	"
    "0:*-45-64-*	Def\n"
    "0	0	45	78	0	0.000	1.800	0.000	E94	"
    "0:*-45-78-*	Def\n"
    "0	0	55	57	0	0.000	10.000	0.000	C94	"
    "0:*-55-57-*	Def\n"
    "2	0	55	57	0	0.000	4.800	0.000	E94	"
    "2:*-55-57-*	Def\n"
    "5	0	55	57	0	0.000	4.800	0.000	E94	"
    "5:*-55-57-*	Def\n"
    "0	1	55	57	5	0.423	12.064	0.090	C94\n"
    "0	1	55	57	55	-0.428	12.044	0.000	C94\n"
    "0	36	55	57	5	-0.268	8.077	-0.806	C94\n"
    "0	36	55	57	55	0.273	8.025	0.692	C94\n"
    "0	0	55	62	0	0.000	3.600	0.000	E94	"
    "0:*-55-62-*	Def\n"
    "0	0	55	64	0	0.000	4.800	0.000	E94	"
    "0:*-55-64-*	Def\n"
    "0	0	55	80	0	0.000	4.800	0.000	E94	"
    "0:*-55-80-*	Def\n"
    "0	0	56	57	0	0.000	6.000	0.000	C94	"
    "0:*-56-57-*	Def\n"
    "0	1	56	57	56	0.000	6.886	-0.161	C94\n"
    "0	36	56	57	56	0.000	4.688	0.107	C94\n"
    "0	0	56	63	0	0.000	4.800	0.000	E94	"
    "0:*-56-63-*	Def\n"
    "0	0	56	80	0	0.000	4.800	0.000	E94	"
    "0:*-56-80-*	Def\n"
    "1	0	57	63	0	0.000	1.800	0.000	E94	"
    "1:*-57-63-*	Def\n"
    "1	0	57	64	0	0.000	1.800	0.000	E94	"
    "1:*-57-64-*	Def\n"
    "0	0	58	63	0	0.000	6.000	0.000	E94	"
    "0:*-58-63-*	Def\n"
    "0	0	58	64	0	0.000	6.000	0.000	E94	"
    "0:*-58-64-*	Def\n"
    "0	0	59	63	0	0.000	7.000	0.000	C94	"
    "0:*-59-63-*	Def\n"
    "0	0	59	65	0	0.000	7.000	0.000	C94	"
    "0:*-59-65-*	Def\n"
    "0	0	59	78	0	0.000	3.600	0.000	E94	"
    "0:*-59-78-*	Def\n"
    "0	0	59	80	0	0.000	3.600	0.000	E94	"
    "0:*-59-80-*	Def\n"
    "0	0	59	82	0	0.000	3.600	0.000	E94	"
    "0:*-59-82-*	Def\n"
    "0	0	62	63	0	0.000	3.600	0.000	E94	"
    "0:*-62-63-*	Def\n"
    "0	0	62	64	0	0.000	3.600	0.000	E94	"
    "0:*-62-64-*	Def\n"
    "1	0	63	63	0	0.000	1.800	0.000	E94	"
    "1:*-63-63-*	Def\n"
    "0	0	63	64	0	0.000	7.000	0.000	C94	"
    "0:*-63-64-*	Def\n"
    "0	5	63	64	5	0.000	7.000	0.000	C94\n"
    "0	5	63	64	64	0.000	7.000	0.000	C94\n"
    "0	39	63	64	5	0.000	7.000	0.000	C94\n"
    "0	39	63	64	64	0.000	7.000	0.000	C94\n"
    "0	0	63	66	0	0.000	7.000	0.000	C94	"
    "0:*-63-66-*	Def\n"
    "0	0	63	78	0	0.000	6.000	0.000	E94	"
    "0:*-63-78-*	Def\n"
    "0	0	63	81	0	0.000	6.000	0.000	E94	"
    "0:*-63-81-*	Def\n"
    "0	0	64	64	0	0.000	7.000	0.000	C94	"
    "0:*-64-64-*	Def\n"
    "1	0	64	64	0	0.000	1.800	0.000	E94	"
    "1:*-64-64-*	Def\n"
    "0	5	64	64	5	0.000	7.000	0.000	C94\n"
    "0	5	64	64	63	0.000	7.000	0.000	C94\n"
    "0	63	64	64	63	0.000	7.000	0.000	C94\n"
    "0	0	64	65	0	0.000	7.000	0.000	C94	"
    "0:*-64-65-*	Def\n"
    "0	0	64	66	0	0.000	7.000	0.000	C94	"
    "0:*-64-66-*	Def\n"
    "0	0	64	78	0	0.000	6.000	0.000	E94	"
    "0:*-64-78-*	Def\n"
    "0	0	64	81	0	0.000	6.000	0.000	E94	"
    "0:*-64-81-*	Def\n"
    "5	0	64	81	0	0.000	6.000	0.000	E94	"
    "5:*-64-81-*	Def\n"
    "0	0	64	82	0	0.000	6.000	0.000	E94	"
    "0:*-64-82-*	Def\n"
    "0	0	65	66	0	0.000	7.000	0.000	C94	"
    "0:*-65-66-*	Def\n"
    "0	0	65	78	0	0.000	6.000	0.000	E94	"
    "0:*-65-78-*	Def\n"
    "0	0	65	81	0	0.000	6.000	0.000	E94	"
    "0:*-65-81-*	Def\n"
    "0	0	65	82	0	0.000	6.000	0.000	E94	"
    "0:*-65-82-*	Def\n"
    "0	0	66	66	0	0.000	7.000	0.000	C94	"
    "0:*-66-66-*	Def\n"
    "0	0	66	78	0	0.000	6.000	0.000	E94	"
    "0:*-66-78-*	Def\n"
    "0	0	66	81	0	0.000	6.000	0.000	E94	"
    "0:*-66-81-*	Def\n"
    "0	0	67	67	0	0.000	12.000	0.000	E94	"
    "0:*-67-67-*	Def\n"
    "5	0	67	67	0	0.000	12.000	0.000	E94	"
    "5:*-67-67-*	Def\n"
    "0	0	76	76	0	0.000	3.600	0.000	E94	"
    "0:*-76-76-*	Def\n"
    "0	0	76	78	0	0.000	3.600	0.000	E94	"
    "0:*-76-78-*	Def\n"
    "0	0	78	78	0	0.000	7.000	0.000	C94	"
    "0:*-78-78-*	Def\n"
    "0	0	78	79	0	0.000	6.000	0.000	E94	"
    "0:*-78-79-*	Def\n"
    "0	0	78	81	0	0.000	4.000	0.000	C94	"
    "0:*-78-81-*	Def\n"
    "0	0	79	79	0	0.000	6.000	0.000	E94	"
    "0:*-79-79-*	Def\n"
    "0	0	79	81	0	0.000	6.000	0.000	E94	"
    "0:*-79-81-*	Def\n"
    "0	0	80	81	0	0.000	4.000	0.000	C94	"
    "0:*-80-81-*	Def\n";

class std::unique_ptr<MMFFVdWCollection> MMFFVdWCollection::ds_instance = nullptr;

extern const std::string defaultMMFFVdW;

MMFFVdWCollection *MMFFVdWCollection::getMMFFVdW(const std::string &mmffVdW) {
  if (!ds_instance || !mmffVdW.empty()) {
    ds_instance.reset(new MMFFVdWCollection(mmffVdW));
  }
  return ds_instance.get();
}

MMFFVdWCollection::MMFFVdWCollection(std::string mmffVdW) {
  if (mmffVdW.empty()) {
    mmffVdW = defaultMMFFVdW;
  }
  std::istringstream inStream(mmffVdW);
  bool firstLine = true;
  std::string inLine = RDKit::getLine(inStream);
  while (!(inStream.eof())) {
    if (inLine[0] != '*') {
      boost::char_separator<char> tabSep("\t");
      tokenizer tokens(inLine, tabSep);
      tokenizer::iterator token = tokens.begin();
      if (firstLine) {
        firstLine = false;
        this->power = boost::lexical_cast<double>(*token);
        ++token;
        this->B = boost::lexical_cast<double>(*token);
        ++token;
        this->Beta = boost::lexical_cast<double>(*token);
        ++token;
        this->DARAD = boost::lexical_cast<double>(*token);
        ++token;
        this->DAEPS = boost::lexical_cast<double>(*token);
        ++token;
      } else {
        MMFFVdW mmffVdWObj;
#ifdef RDKIT_MMFF_PARAMS_USE_STD_MAP
        unsigned int atomType = boost::lexical_cast<unsigned int>(*token);
#else
        d_atomType.push_back(
            (std::uint8_t)(boost::lexical_cast<unsigned int>(*token)));
#endif
        ++token;
        mmffVdWObj.alpha_i = boost::lexical_cast<double>(*token);
        ++token;
        mmffVdWObj.N_i = boost::lexical_cast<double>(*token);
        ++token;
        mmffVdWObj.A_i = boost::lexical_cast<double>(*token);
        ++token;
        mmffVdWObj.G_i = boost::lexical_cast<double>(*token);
        ++token;
        mmffVdWObj.DA = (boost::lexical_cast<std::string>(*token)).at(0);
        ++token;
        mmffVdWObj.R_star =
            mmffVdWObj.A_i * pow(mmffVdWObj.alpha_i, this->power);
#ifdef RDKIT_MMFF_PARAMS_USE_STD_MAP
        d_params[atomType] = mmffVdWObj;
#else
        d_params.push_back(mmffVdWObj);
#endif
      }
    }
    inLine = RDKit::getLine(inStream);
  }
}

const std::string defaultMMFFVdW =
    "*\n"
    "*          Copyright (c) Merck and Co., Inc., 1994, 1995, 1996\n"
    "*                         All Rights Reserved\n"
    "*\n"
    "* E94  - From empirical rule (JACS 1992, 114, 7827) \n"
    "* C94  - Adjusted in fit to HF/6-31G* dimer energies and geometries\n"
    "* X94  - Chosen in the extension of the paratererization for MMFF \n"
    "*        by analogy to other, similar atom types or, for ions, by\n"
    "*        fitting to atomic radii (and sometimes to association energies\n"
    "*        for hydrates)\n"
    "*\n"
    "*  power      B       Beta     DARAD      DAEPS\n"
    "0.25	0.2	12.	0.8	0.5\n"
    "*\n"
    "*  type  alpha-i     N-i       A-i       G-i DA Symb   Origin\n"
    "*------------------------------------------------------------\n"
    "1	1.050	2.490	3.890	1.282	-	CR	E94\n"
    "2	1.350	2.490	3.890	1.282	-	C=C	E94\n"
    "3	1.100	2.490	3.890	1.282	-	C=O	E94\n"
    "4	1.300	2.490	3.890	1.282	-	CSP	E94\n"
    "5	0.250	0.800	4.200	1.209	-	HC	C94\n"
    "6	0.70	3.150	3.890	1.282	A	OR	C94\n"
    "7	0.65	3.150	3.890	1.282	A	O=C	C94\n"
    "8	1.15	2.820	3.890	1.282	A	NR	C94\n"
    "9	0.90	2.820	3.890	1.282	A	N=C	C94\n"
    "10	1.000	2.820	3.890	1.282	A	NC=O	E94\n"
    "11	0.35	3.480	3.890	1.282	A	F	C94\n"
    "12	2.300	5.100	3.320	1.345	A	CL	E94\n"
    "13	3.400	6.000	3.190	1.359	A	BR	E94\n"
    "14	5.500	6.950	3.080	1.404	A	I	E94\n"
    "15	3.00	4.800	3.320	1.345	A	S	C94\n"
    "16	3.900	4.800	3.320	1.345	A	S=C	E94\n"
    "17	2.700	4.800	3.320	1.345	-	SO	E94\n"
    "18	2.100	4.800	3.320	1.345	-	SO2	E94\n"
    "19	4.500	4.200	3.320	1.345	-	SI	E94\n"
    "20	1.050	2.490	3.890	1.282	-	CR3R	E94\n"
    "21	0.150	0.800	4.200	1.209	D	HOR	C94\n"
    "22	1.100	2.490	3.890	1.282	-	CR3R	E94\n"
    "23	0.150	0.800	4.200	1.209	D	HNR	C94\n"
    "24	0.150	0.800	4.200	1.209	D	HOCO	C94\n"
    "25	1.600	4.500	3.320	1.345	-	PO4	E94\n"
    "26	3.600	4.500	3.320	1.345	A	P	E94\n"
    "27	0.150	0.800	4.200	1.209	D	HN=C	C94\n"
    "28	0.150	0.800	4.200	1.209	D	HNCO	C94\n"
    "29	0.150	0.800	4.200	1.209	D	HOCC	C94\n"
    "30	1.350	2.490	3.890	1.282	-	CE4R	E94\n"
    "31	0.150	0.800	4.200	1.209	D	HOH	C94\n"
    "32	0.75	3.150	3.890	1.282	A	O2CM	C94\n"
    "33	0.150	0.800	4.200	1.209	D	HOS	C94\n"
    "34	1.00	2.820	3.890	1.282	-	NR+	C94\n"
    "35	1.50	3.150	3.890	1.282	A	OM	X94\n"
    "36	0.150	0.800	4.200	1.209	D	HNR+	C94\n"
    "37	1.350	2.490	3.890	1.282	-	CB	E94\n"
    "38	0.85	2.820	3.890	1.282	A	NPYD	C94\n"
    "39	1.10	2.820	3.890	1.282	-	NPYL	C94\n"
    "40	1.00	2.820	3.890	1.282	A	NC=C	E94\n"
    "41	1.100	2.490	3.890	1.282	-	CO2M	C94\n"
    "42	1.000	2.820	3.890	1.282	A	NSP	E94\n"
    "43	1.000	2.820	3.890	1.282	A	NSO2	E94\n"
    "44	3.00	4.800	3.320	1.345	A	STHI	C94\n"
    "45	1.150	2.820	3.890	1.282	-	NO2	E94\n"
    "46	1.300	2.820	3.890	1.282	-	N=O	E94\n"
    "47	1.000	2.820	3.890	1.282	A	NAZT	X94\n"
    "48	1.200	2.820	3.890	1.282	A	NSO	X94\n"
    "49	1.00	3.150	3.890	1.282	-	O+	X94\n"
    "50	0.150	0.800	4.200	1.209	D	HO+	C94\n"
    "51	0.400	3.150	3.890	1.282	-	O=+	E94\n"
    "52	0.150	0.800	4.200	1.209	D	HO=+	C94\n"
    "53	1.000	2.820	3.890	1.282	-	=N=	X94\n"
    "54	1.30	2.820	3.890	1.282	-	N+=C	C94\n"
    "55	0.80	2.820	3.890	1.282	-	NCN+	E94\n"
    "56	0.80	2.820	3.890	1.282	-	NGD+	E94\n"
    "57	1.000	2.490	3.890	1.282	-	CNN+	E94\n"
    "58	0.80	2.820	3.890	1.282	-	NPD+	E94\n"
    "59	0.65	3.150	3.890	1.282	A	OFUR	C94\n"
    "60	1.800	2.490	3.890	1.282	A	C%-	E94\n"
    "61	0.800	2.820	3.890	1.282	A	NR%	E94\n"
    "62	1.300	2.820	3.890	1.282	A	NM	X94\n"
    "63	1.350	2.490	3.890	1.282	-	C5A	E94\n"
    "64	1.350	2.490	3.890	1.282	-	C5B	E94\n"
    "65	1.000	2.820	3.890	1.282	A	N5A	E94\n"
    "66	0.75	2.820	3.890	1.282	A	N5B	C94\n"
    "67	0.950	2.82	3.890	1.282	A	N2OX	X94\n"
    "68	0.90	2.82	3.890	1.282	A	N3OX	C94\n"
    "69	0.950	2.82	3.890	1.282	A	NPOX	C94\n"
    "70	0.87	3.150	3.890	1.282	A	OH2	C94\n"
    "71	0.150	0.800	4.200	1.209	D	HS	C94\n"
    "72	4.000	4.800	3.320	1.345	A	SM	X94\n"
    "73	3.000	4.800	3.320	1.345	-	SMO2	X94\n"
    "74	3.000	4.800	3.320	1.345	-	=S=O	X94\n"
    "75	4.000	4.500	3.320	1.345	A	-P=C	X94\n"
    "76	1.200	2.820	3.890	1.282	A	N5M	X94\n"
    "77	1.500	5.100	3.320	1.345	A	CLO4	X94\n"
    "78	1.350	2.490	3.890	1.282	-	C5	X94\n"
    "79	1.000	2.820	3.890	1.282	A	N5	X94\n"
    "80	1.000	2.490	3.890	1.282	-	CIM+	C94\n"
    "81	0.80	2.820	3.890	1.282	-	NIM+	C94\n"
    "82	0.950	2.82	3.890	1.282	A	N5OX	X94\n"
    "87	0.45	6.0	4.0	1.4	-	FE+2	X94\n"
    "88	0.55	6.0	4.0	1.4	-	FE+3	X94\n"
    "89	1.4	3.48	3.890	1.282	A	F-	X94\n"
    "90	4.5	5.100	3.320	1.345	A	CL-	X94\n"
    "91	6.0	6.000	3.190	1.359	A	BR-	X94\n"
    "92	0.15	2.0	4.0	1.3	-	LI+	X94\n"
    "93	0.4	3.5	4.0	1.3	-	NA+	X94\n"
    "94	1.0	5.0	4.0	1.3	-	K+	X94\n"
    "95	0.43	6.0	4.0	1.4	-	ZN+2	X94\n"
    "96	0.9	5.0	4.0	1.4	-	CA+2	X94\n"
    "97	0.35	6.0	4.0	1.4	-	CU+1	X94\n"
    "98	0.40	6.0	4.0	1.4	-	CU+2	X94\n"
    "99	0.35	3.5	4.0	1.3	-	MG+2	X94\n";
}  // end of namespace MMFF
}  // end of namespace ForceFields
