/*
 * Do rate lookups and set destination route and maximum call length based on that
 * 
 * Copyright (C) 2003 by Troll Phone Networks AS
 *
 * This program is distributed under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2, or (at your
 * option) any later version.
 *
 *	$Id: rate_engine.c,v 1.24 2004/01/22 15:04:02 tholo Exp $
 */

#ifdef __linux__
#define _GNU_SOURCE
#endif

#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <pthread.h>

#include <mysql/mysql.h>

#include <pcre.h>
#include "pcre_subst.h"

#include <asterisk/file.h>
#include <asterisk/config.h>
#include <asterisk/logger.h>
#include <asterisk/channel.h>
#include <asterisk/pbx.h>
#include <asterisk/module.h>
#include <asterisk/options.h>
#include <asterisk/cli.h>
#include <asterisk/manager.h>

static char *tdesc = "Call Routing and Rating Application";
static char *cdr_name = "ratecall";
static char *routecall_app = "RouteCall";
static char *routecall_synopsis = "Do cost based routing of a call and set call time limit";
static char *routecall_descrip =
"  RouteCall(extension[|flags]): Does Least Cost Routing of a call, and\n"
"sets  an  absolute timeout  on  call length  based on customers credit\n"
"worthiness.  If there exists a priority n + 101, and no route could be\n"
"found,  then the  channel  will be set up to continue at that priority\n"
"level.\n"
"  If a route  is  found, a dialstring suitable  for  use with the Dial\n"
"application is created in the variable ${DESTINATIONnn}, to be used as\n"
"Dial(${DESTINATIONnn}) where 'nn' is a number  from 1 and up, with the\n"
"route in ${DESTINATION1} being the lowest-cost route found,  the route\n"
"in  ${DESTINATION2} being  the  next  lowest rate found and so on.  An\n"
"AbsoluteTimeout limit may also be set for the call.\n"
"  Always returns 0.\n";

STANDARD_LOCAL_USER;

LOCAL_USER_DECL;

static ast_mutex_t ratelock = AST_MUTEX_INITIALIZER;
static MYSQL mysql;
static char *systemname;

static pthread_t poster_thread;
static int cancel_poster = 0;
static pthread_mutex_t poster_lock = PTHREAD_MUTEX_INITIALIZER;
static pthread_cond_t poster_cond = PTHREAD_COND_INITIALIZER;

static int warn_threshold = 500;	/* Start giving warnings if CDR queue is this long */
static int drop_threshold = 1000;	/* Start dropping entries if CDR queue is this long */
static int warn_freq = 25;		/* Warn every this many entries only */
static int queuelen = 0;

typedef struct cdr_queue_t {
	struct cdr_queue_t *next;
	char *cdrquery;
	size_t cdrlen;
} cdr_queue_t;

static cdr_queue_t *queue_head = NULL, *queue_tail = NULL;

static char *dbhost = NULL, *dbuser = NULL, *dbpass = NULL, *dbname = NULL, *dbsock = NULL;
static char *table_cdr = NULL, *table_error = NULL, *table_egress = NULL, *table_cost = NULL;
static int dbport = 0, connected = 0, avg_call_length = 0;
static int reloadevery = 0;
static time_t lastreload = 0;
static int num_egress = 0, num_rate = 0;

typedef struct egress_t {
    int route_id;
    char *technology;
    char *peer;
    pcre *pcre;
    pcre_extra *extra;
    char *substitute;
    char *description;
} egress_t;

static egress_t *new_egress(void);
static void free_egress(egress_t *);

typedef struct rate_t {
    int rate_id;
    int route_id;
    egress_t *egress;
    char iso[3];
    char type[4];
    char *country;
    char *extra;
    char prefix[11];
    time_t active_date;
    time_t expires_date;
    int firstperiod;
    int periods;
    int startcost;
    int periodcost;
    int trialcost;
} rate_t;

static rate_t *new_rate(void);
static void free_rate(rate_t *);

typedef struct splayval_t {
    struct splayval_t *next;
    egress_t *egress;
    rate_t *rate;
} splayval_t;

typedef struct splay_t {
    char *key;
    void *value;
    int sentinel;
    struct splay_t *left;
    struct splay_t *right;
    struct splay_t *forward;
    struct splay_t *back;
    struct splay_t *parent;
} splay_t;

static splay_t *rates = NULL;
static splay_t *egresses = NULL;

static splay_t *new_splay(void);
static void free_splay(splay_t *);
static splay_t *insert_splay(splay_t *tree, const char *skey, void *value);
static splay_t *search_splay(splay_t *tree, const char *skey);
static splay_t *splay_root(splay_t *tree);
static splay_t *splay_first(splay_t *tree);
#if 0
static splay_t *splay_last(splay_t *tree);
#endif
static splay_t *splay_next(splay_t *node);
static splay_t *splay_prev(splay_t *node);
static splay_t *splay_null(splay_t *tree);
static void splay_delete(splay_t *node);
static splay_t *splay_find_nearest(splay_t *tree, const char *skey, int *cmpval);
static void splay(splay_t *node);

static void clear_egresses(void);
static void inject_egress(egress_t *);
static void clear_rates(void);
static void inject_rate(rate_t *);
static int load_rates(int);

/*
 * Find the Least Cost Route for a given number, and construct
 * a dial string suitable for use with the Dial application to
 * call using that route.  Also make available any other routes
 * sorted by least expensive to most expensive
 *
 * The dial strings are put in channel variables DESTINATIONxx
 * where "xx" is a number from 1 to N.  Also the cost values
 * are put in a matching FIRSTLEN, RESTLEN, STARTCOST and
 * PERIODCOST variables to be picked up by the CDR part to
 * calculate actual call cost.
 */
static int routecall_exec(struct ast_channel *chan, void *data)
{
    char destvar[20], firstvar[20], restvar[20], startcost[20], periodcost[20];
    char trialcost[20];
    char *args, *number, *flags, *destination;
    int res = 0, maxcall = 0, i, cmpval;
    struct localuser *u;
    char tempstr[16];
    splayval_t *val;
    splay_t *s;
    time_t now;

    /*
     * If we should load rates automatically, see if they have
     * expired, and if so, try to reload
     */
    if (reloadevery &&
	lastreload + reloadevery <= time(NULL)) {
	load_rates(0);
    }

    /*
     * Verify that arguments were supplied
     */
    if (data == NULL || *(char *)data == '\0') {
	ast_log(LOG_WARNING, "RouteCall requires an argument (number[|flags])\n");
	return -1;
    }

    LOCAL_USER_ADD(u);

    /*
     * Parse out arguments to number and (optional) flags
     */
    args = ast_strdupa((char *)data);
    number = strsep(&args, "|");
    flags = strsep(&args, "\0");

    /*
     * Make empty strings be NULL pointers instead
     */
    if (number && *number == '\0')
	number = NULL;
    if (flags && *flags == '\0')
	flags = NULL;

    /*
     * We can't do any routing if we don't have any rates or
     * egress routes loaded
     */
    if (rates == NULL || egresses == NULL) {
	ast_log(LOG_WARNING, "RouteCall requires that rates and egress routes exists\n");

	/*
	 * Jump to priority + 101, since we got an error
	 */
	if (ast_exists_extension(chan, chan->context, chan->exten, chan->priority + 101, chan->callerid))
	    chan->priority += 100;
	else
	    res = -1;

	LOCAL_USER_REMOVE(u);

	return res;
    }

    /*
     * Make sure only one thread access the database at a time
     */
    ast_mutex_lock(&ratelock);

    /*
     * If no phone number was supplied, error out
     */
    if (!number)
	goto errout;

    /*
     * Search the in-memory splay tree for a prefix match
     */
    s = splay_find_nearest(rates, number, &cmpval);

    /*
     * cmpval != 0 means inexact match, make sure it is a
     * prefix match if at all possible
     */
    if (cmpval && s) {
	/*
	 * We got a non-matchin key back -- if it is not a prefix,
	 * retrieve the next smaller until we run out of prefixes
	 * to try or we find a usable prefix (we are guaranteed
	 * that the returned key is larger than or equal to the
	 * one we searched for if such a key exist)
	 */
	while (s != splay_null(rates) &&
	       strncasecmp(number, s->key, strlen(s->key)) != 0)
	    s = splay_prev(s);
	if (s != splay_null(rates) &&
	    strncasecmp(number, s->key, strlen(s->key)) == 0) {
	    cmpval = 0;
	}
    }
    
    /*
     * If we could not find a match for the number, we should
     * log an entry to the error table so that there will be
     * a record of a non-matched number for future reference,
     * and then return failure
     */
    if (cmpval != 0) {
	ast_log(LOG_WARNING, "No match for phone number %s in rate tables\n", number);
	goto errout;
    }

    /*
     * Optimize by moving matched prefixes towards the top
     * of the splay tree
     */
    splay(s);

    /*
     * Get current time, so we can skip rate entries that
     * are expired or not yet valid
     */
    time(&now);

    /*
     * The in-memory rates are already sorted from least to
     * most expensive
     */
    for (i = 0, val = s->value; val; val = val->next) {
	/*
	 * If we do not have the actual egress route, skip the rate
	 */
	if (val->rate->egress == NULL) {
	    ast_log(LOG_WARNING, "Rate without egress route for prefix '%s'\n", s->key);
	    continue;
	}

	/*
	 * If the rate is not yet active, or if it has expired,
	 * skip it
	 */
	if ((val->rate->active_date &&
	     val->rate->active_date > now) ||
	    (val->rate->expires_date &&
	     val->rate->expires_date > now))
	    continue;

	snprintf(destvar, sizeof(destvar), "DESTINATION%d", i + 1);
	snprintf(firstvar, sizeof(firstvar), "FIRSTLEN%d", i + 1);
	snprintf(restvar, sizeof(restvar), "RESTLEN%d", i + 1);
	snprintf(startcost, sizeof(startcost), "STARTCOST%d", i + 1);
	snprintf(periodcost, sizeof(periodcost), "PERIODCOST%d", i + 1);
	snprintf(trialcost, sizeof(trialcost), "TRIALCOST%d", i + 1);

	destination = pcre_subst(val->rate->egress->pcre, val->rate->egress->extra, number, strlen(number), 0, 0, val->rate->egress->substitute);
	res = 0;
	pbx_builtin_setvar_helper(chan, destvar, destination);
	snprintf(tempstr, sizeof(tempstr), "%d", val->rate->firstperiod);
	pbx_builtin_setvar_helper(chan, firstvar, tempstr);
	snprintf(tempstr, sizeof(tempstr), "%d", val->rate->periods);
	pbx_builtin_setvar_helper(chan, restvar, tempstr);
	snprintf(tempstr, sizeof(tempstr), "%d", val->rate->startcost);
	pbx_builtin_setvar_helper(chan, startcost, tempstr);
	snprintf(tempstr, sizeof(tempstr), "%d", val->rate->periodcost);
	pbx_builtin_setvar_helper(chan, periodcost, tempstr);
	snprintf(tempstr, sizeof(tempstr), "%d", val->rate->trialcost);
	pbx_builtin_setvar_helper(chan, trialcost, tempstr);
	if (option_verbose > 3) {
	    if (i)
		ast_verbose( VERBOSE_PREFIX_4 "Alternate route as %s\n", destination);
	    else
		ast_verbose( VERBOSE_PREFIX_4 "Route as %s\n", destination);
	}
	pcre_free(destination);
	i++;
    }

    /*
     * If no prefix rates actually had egress routes or were valid,
     * we need to error out here -- should be try a shorter prefix
     * for this case?
     */
    if (i == 0) {
	ast_log(LOG_WARNING, "No valid egress routes for prefix '%s'\n", s->key);
	goto errout;
    }

    /*
     * Release the lock
     */
    ast_mutex_unlock(&ratelock);

    /*
     * If we could not find a match for the number, we should
     * log an entry to the error table so that there will be
     * a record of a non-matched number for future reference,
     * and then return failure
     */
    if (res < 0) {
	res = 0;
	ast_log(LOG_WARNING, "No route could be made for phone number %s\n", number);
	goto errout;
    }

    /*
     * Set an AbsoluteTimeout on the call, based on customer
     * credit worthiness and call cost for the destination
     * we have found a route for
     */
    ast_channel_setwhentohangup(chan, maxcall);
    if (option_verbose > 3)
	ast_verbose( VERBOSE_PREFIX_4 "Set AbsoluteTimeout to %d\n", maxcall);

    goto out;
errout:
    /*
     * Release the lock
     */
    ast_mutex_unlock(&ratelock);

    /*
     * Jump to priority + 101, since we got an error
     */
    if (res == 0 && ast_exists_extension(chan, chan->context, chan->exten, chan->priority + 101, chan->callerid))
	chan->priority += 100;
    else
	res = -1;

out:
    LOCAL_USER_REMOVE(u);

    return res;
}

/*
 * Called when a call is completed, to log call details.  We
 * extend this to calculate actual call costs based on the
 * route used, by using saved rate information from the LCR
 * routine.
 *
 * We also pick up ANI from the channel structure, to get
 * slightly better CDR data.
 */
static int cdr_ratecall(struct ast_cdr *cdr)
{
    char *ani, *src, *dst, *dcontext, *channel, *dstchannel, *lastapp, *lastdata;
    char starttime[32], answertime[32], endtime[32], destvar[32], firstvar[32];
    char trialvar[32];
    int len, i, first = 0, start = 0, each = 0, rest = 0, trial = 0;
    char *account, *uniqueid, *destination = "", *end;
    char restvar[32], startvar[32], periodvar[32];
    struct ast_channel *chan;
    long long cost = 0;
    cdr_queue_t *q;
    struct tm tm;
    time_t t;

    /*
     * If we should load rates automatically, see if they have
     * expired, and if so, try to reload
     */
    if (reloadevery &&
	lastreload + reloadevery <= time(NULL)) {
	load_rates(0);
    }

    /*
     * Find which channel this CDR entry was generated from -- this
     * will allow us to retrieve channel variables and other info
     * that is needed for accurate logging
     */
    for (chan = ast_channel_walk(NULL); chan && chan->cdr != cdr; chan = ast_channel_walk(chan))
	;
    if (chan) {
	for (i = 1;; i++) {
	    snprintf(destvar, sizeof(destvar), "DESTINATION%d", i);
	    snprintf(firstvar, sizeof(firstvar), "FIRSTLEN%d", i);
	    snprintf(restvar, sizeof(restvar), "RESTLEN%d", i);
	    snprintf(startvar, sizeof(startvar), "STARTCOST%d", i);
	    snprintf(periodvar, sizeof(periodvar), "PERIODCOST%d", i);
	    snprintf(trialvar, sizeof(trialvar), "TRIALCOST%d", i);

	    dst = pbx_builtin_getvar_helper(chan, destvar);
	    if (dst == NULL)
		break;
	    if (strcmp(dst, cdr->lastdata) == 0) {
		dst = pbx_builtin_getvar_helper(chan, firstvar);
		first = strtoul(dst, &end, 10);
		dst = pbx_builtin_getvar_helper(chan, restvar);
		rest = strtoul(dst, &end, 10);
		dst = pbx_builtin_getvar_helper(chan, startvar);
		start = strtoul(dst, &end, 10);
		dst = pbx_builtin_getvar_helper(chan, periodvar);
		each = strtoul(dst, &end, 10);
		dst = pbx_builtin_getvar_helper(chan, trialvar);
		trial = strtoul(dst, &end, 10);
		break;
	    }
	}
    }

    /*
     * Make sure only one thread access the database at a time
     */
    ast_mutex_lock(&ratelock);

    /*
     * Make MySQL-safe copies of all strings
     */
    if ((src = alloca((len = strlen(cdr->src)) * 2 + 1)) != NULL)
	mysql_real_escape_string(&mysql, src, cdr->src, len);

    /*
     * Try to pick up real ANI if that exists
     */
    if (chan && chan->ani) {
	if ((ani = alloca((len = strlen(chan->ani)) * 2 + 1)) != NULL)
	    mysql_real_escape_string(&mysql, ani, chan->ani, len);
    }
    else
	ani = src;
    if ((dst = alloca((len = strlen(cdr->dst)) * 2 + 1)) != NULL)
	mysql_real_escape_string(&mysql, dst, cdr->dst, len);
    if ((dcontext = alloca((len = strlen(cdr->dcontext)) * 2 + 1)) != NULL)
	mysql_real_escape_string(&mysql, dcontext, cdr->dcontext, len);
    if ((channel = alloca((len = strlen(cdr->channel)) * 2 + 1)) != NULL)
	mysql_real_escape_string(&mysql, channel, cdr->channel, len);
    if ((dstchannel = alloca((len = strlen(cdr->dstchannel)) * 2 + 1)) != NULL)
	mysql_real_escape_string(&mysql, dstchannel, cdr->dstchannel, len);
    if ((lastapp = alloca((len = strlen(cdr->lastapp)) * 2 + 1)) != NULL)
	mysql_real_escape_string(&mysql, lastapp, cdr->lastapp, len);
    if ((lastdata = alloca((len = strlen(cdr->lastdata)) * 2 + 1)) != NULL)
	mysql_real_escape_string(&mysql, lastdata, cdr->lastdata, len);
    if ((account = alloca((len = strlen(cdr->accountcode)) * 2 + 1)) != NULL)
	mysql_real_escape_string(&mysql, account, cdr->accountcode, len);
    if ((uniqueid = alloca((len = strlen(cdr->uniqueid)) * 2 + 1)) != NULL)
	mysql_real_escape_string(&mysql, uniqueid, cdr->uniqueid, len);

    /*
     * Release database lock
     */
    ast_mutex_unlock(&ratelock);

    /*
     * If any allocations failed, bail with an error
     */
    if (ani == NULL || dcontext == NULL || channel == NULL || dstchannel == NULL ||
	lastapp == NULL || lastdata == NULL || uniqueid == NULL) {
	ast_log(LOG_ERROR, "Out of memory!\n");
	return -1;
    }

    t = cdr->start.tv_sec;
    localtime_r(&t,&tm);
    strftime(starttime, sizeof(starttime), "%Y-%m-%d %T", &tm);
    t = cdr->answer.tv_sec;
    localtime_r(&t,&tm);
    strftime(answertime, sizeof(answertime), "%Y-%m-%d %T", &tm);
    t = cdr->end.tv_sec;
    localtime_r(&t,&tm);
    strftime(endtime, sizeof(endtime), "%Y-%m-%d %T", &tm);

    /*
     * Charge for answered calls
     */
    if (cdr->disposition == AST_CDR_ANSWERED &&
	(first || rest) && (start || each)) {
	/*
	 * Billable seconds is really the number of whole
	 * periods, plus the length of the first period
	 * which often is a different length, and/or has
	 * a different price
	 */
	cdr->billsec = (cdr->billsec - first < 0 ? 0 : cdr->billsec - first + rest - 1) / rest * rest;

	/*
	 * Cost of the call is the starting cost (for the
	 * first period), plus the price for each full
	 * period thereafter
	 */
	cost = start + cdr->billsec / rest * each;

	/*
	 * Don't forget to actually add inn the length of
	 * that first period
	 */
	cdr->billsec += first;
    }
    /*
     * Charge for unanswered or busy calls
     */
    else if (cdr->disposition == AST_CDR_NOANSWER ||
	     cdr->disposition == AST_CDR_BUSY) {
	cost = trial;
    }

    /*
     * Allocate a CDR queue entry
     */
    if ((q = malloc(sizeof(cdr_queue_t))) == NULL) {
	ast_log(LOG_WARNING, "Out of memory\n");
	return -1;
    }
    q->next = NULL;

    q->cdrlen = asprintf(&q->cdrquery,
			 "INSERT INTO %s "
			 "(callDate,endDate,ANI,callerID,calledNumber,calledContext,"
			 "sourceChannel,destinationChannel,lastApplication,lastData,"
			 "callDuration,billableDuration,callDisposition,amaFlags,"
			 "accountCode,destinationDescription,callCost,uniqueID,"
			 "handlingSystem) VALUES "
			 "('%s','%s','%s','%s','%s','%s','%s','%s','%s',"
			 "'%s',%i,%i,'%s','%s','%s','%s',%lld,'%s','%s')",
			 table_cdr,
			 starttime, endtime, ani, src, dst, dcontext,
			 channel, dstchannel, lastapp, lastdata, cdr->duration,
			 cdr->billsec, ast_cdr_disp2str(cdr->disposition),
			 ast_cdr_flags2str(cdr->amaflags), account,
			 destination, cost, uniqueid, systemname);

    /*
     * Acquire shared data lock
     */
    pthread_mutex_lock(&poster_lock);

    /*
     * Only add to queue if we are below the maximum threshold
     */
    if (queuelen < drop_threshold) {
	/*
	 * Append CDR entry to queue
	 */
	if (queue_tail != NULL)
	    queue_tail-> next =q;
	queue_tail = q;
	if (queue_head == NULL)
	    queue_head = q;

	/*
	 * Warn if we are getting too long a queue (but only once every
	 * warn_freq entries)
	 */
	if (++queuelen > warn_threshold &&
	    warn_freq &&
	    ((queuelen - warn_threshold) % warn_freq) == 0)
	    ast_log(LOG_WARNING, "Rating Engine CDR Queue now at %d entries!\n", queuelen);
    }
    else {
	ast_log(LOG_ERROR, "Rating Engine CDR entry dropped!\n");
	free(q->cdrquery);
	free(q);
    }

    /*
     * Signal worker thread that there is data available
     */
    pthread_cond_signal(&poster_cond);

    /*
     * Release shared data lock
     */
    pthread_mutex_unlock(&poster_lock);

    return 0;
}

/*
 * Worker thread to actually post the CDR data to a
 * Rating Engine -- will not remove the entries from
 * queue if connecting to the Rating Engine fails,
 * or if sending the entries to it fails
 */
static void *poster_worker(void *arg)
{
    int isconnected = 0, fail;
    struct timespec ts;
    MYSQL poster_mysql;
    cdr_queue_t *q;

    ast_log(LOG_DEBUG, "Rating Engine poster thread started\n");

    /*
     * Get lock on shared data
     */
    pthread_mutex_lock(&poster_lock);

    /*
     * We need a timeout if there is something on the queue
     */
    ts.tv_sec = time(NULL) + 30;
    ts.tv_nsec = 0;

    /*
     * Initialize MySQL
     */
    mysql_init(&poster_mysql);
    if (mysql_real_connect(&poster_mysql, dbhost, dbuser, dbpass, dbname, dbport, dbsock, 0))
	isconnected = 1;
    else {
	ast_log(LOG_ERROR, "Failed to connect to MySQL database '%s': %s\n", dbname, mysql_error(&poster_mysql));
	isconnected = 0;
    }

    /*
     * Wait for [more] data to be available
     */
    for (;; (queue_head ?
	     (ts.tv_sec = time(NULL) + 30, ts.tv_nsec = 0, pthread_cond_timedwait(&poster_cond, &poster_lock, &ts)) :
	     pthread_cond_wait(&poster_cond, &poster_lock))) {

	ast_log(LOG_DEBUG, "Rating Engine poster thread processing\n");

	/*
	 * Check if we should stop processing
	 */
	if (cancel_poster) {
	    cancel_poster = 0;
	    break;
	}

	/*
	 * If we have to (re-)connect to a database, that might
	 * take a little while, so release the lock while we do
	 * that
	 */
	pthread_mutex_unlock(&poster_lock);

	/*
	 * If we are not connected to a database, try to connect
	 */
	if (!isconnected) {
	    if (mysql_real_connect(&poster_mysql, dbhost, dbuser, dbpass, dbname, dbport, dbsock, 0))
		isconnected = 1;
	    else {
		ast_log(LOG_ERROR, "Failed to connect to MySQL database '%s': %s\n", dbname, mysql_error(&poster_mysql));

		/*
		 * Reacquire the lock before we restart loop
		 */
		pthread_mutex_lock(&poster_lock);

		continue;
	    }
	}

	/*
	 * Try to verify that our database connection is still alive
	 */
	if (mysql_ping(&poster_mysql)) {
	    ast_log(LOG_ERROR, "Cannot talk to MySQL database '%s': %s\n",
		    dbname, mysql_error(&poster_mysql));
	    isconnected = 0;

	    /*
	     * Reacquire the lock before we restart loop
	     */
	    pthread_mutex_lock(&poster_lock);

	    continue;
	}

	/*
	 * We do have to reacquire the lock before we can actually
	 * touch the queue, however...
	 */
	pthread_mutex_lock(&poster_lock);

	/*
	 * Process CDR records on queue
	 */
	for (q = queue_head; q; q = queue_head) {

	    ast_log(LOG_DEBUG, "Attempting to write queue entry to database\n");

	    /*
	     * Update the queue head and tail
	     */
	    queue_head = q->next;
	    if (queue_head == NULL)
		queue_tail = NULL;
	    queuelen--;

	    /*
	     * Release the lock before doing potentially long operation
	     */
	    pthread_mutex_unlock(&poster_lock);

	    /*
	     * Perform query
	     */
	    if (mysql_real_query(&poster_mysql, q->cdrquery, q->cdrlen)) {
		fail = 1;
		ast_log(LOG_ERROR,"Failed to insert into database: %s\n", mysql_error(&poster_mysql));
	    }
	    else
		fail = 0;

	    /*
	     * Re-acquire the lock
	     */
	    pthread_mutex_lock(&poster_lock);

	    /*
	     * If we errored out of sending the CDR entry, put it back
	     * on the head of the queue, updating the queue as needed;
	     * be careful -- if we emptied the queue above, we have a
	     * new queue_head, and if not we have to set the tail too
	     */
	    if (fail) {
		ast_log(LOG_DEBUG, "Prepending failed CDR entry to Rating Engine queue\n");
		q->next = queue_head;
		queue_head = q;
		if (queue_tail == NULL)
		    queue_tail = q;
		queuelen++;
		break;
	    }

	    /*
	     * Otherwise, we successfully sent a queue entry, and should
	     * free the memory it used
	     */
	    free(q->cdrquery);
	    free(q);
	}
    }

    ast_log(LOG_DEBUG, "Rating Engine poster thread exiting\n");

    /*
     * Release the lock before exiting thread
     */
    pthread_mutex_unlock(&poster_lock);

    /*
     * Close connection if it is open
     */
    mysql_close(&poster_mysql);

    /*
     * Exit thread
     */
    return NULL;
}

/*
 * Load configuration settings from configuration file,
 * and do some minimal sanity checks on the information
 * loaded
 */
static int load_config(void)
{
    struct ast_variable *var;
    struct ast_config *cfg;
    char *cat, *e;

    /*
     * Lock the configuration, since we are about to change it
     */
    ast_mutex_lock(&ratelock);

    cfg = ast_load("rate_engine.conf");
    if (cfg) {
	cat = ast_category_browse(cfg, NULL);
	while (cat) {
	    var = ast_variable_browse(cfg, cat);

	    /*
	     * Global settings
	     */
	    if (strcasecmp(cat, "general") == 0) {
		while (var) {
		    /*
		     * Our system name -- used only for logging
		     */
		    if (strcasecmp(var->name, "system") == 0) {
			if (systemname)
			    free(systemname);
			systemname = strdup(var->value);
			if (option_verbose > 2)
			    ast_verbose(VERBOSE_PREFIX_3 "System Name: %s\n", var->value);
		    }
		    /*
		     * Host where our database is
		     */
		    else if (strcasecmp(var->name, "host") == 0) {
			if (dbhost)
			    free(dbhost);
			dbhost = strdup(var->value);
			if (option_verbose > 3)
			    ast_verbose(VERBOSE_PREFIX_4 "DB Host: %s\n", var->value);
		    }
		    /*
		     * Port for the database -- used only when
		     * host is set, and is not "localhost"
		     */
		    else if (strcasecmp(var->name, "port") == 0) {
			dbport = strtol(var->value, &e, 10);
			if (option_verbose > 3)
			    ast_verbose(VERBOSE_PREFIX_4 "DB Port: %s\n", var->value);
		    }
		    /*
		     * Unix domain socket -- used only when
		     * host is not set, or is set to "localhost"
		     */
		    else if (strcasecmp(var->name, "socket") == 0) {
			if (dbsock)
			    free(dbsock);
			dbsock = strdup(var->value);
			if (option_verbose > 3)
			    ast_verbose(VERBOSE_PREFIX_4 "DB Socket: %s\n", var->value);
		    }
		    /*
		     * Username to authenticate to the database
		     */
		    else if (strcasecmp(var->name, "username") == 0) {
			if (dbuser)
			    free(dbuser);
			dbuser = strdup(var->value);
			if (option_verbose > 3)
			    ast_verbose(VERBOSE_PREFIX_4 "DB Username: %s\n", var->value);
		    }
		    /*
		     * Password to authenticate to the database
		     */
		    else if (strcasecmp(var->name, "password") == 0) {
			if (dbpass)
			    free(dbpass);
			dbpass = strdup(var->value);
			if (option_verbose > 3)
			    ast_verbose(VERBOSE_PREFIX_4 "DB Password: [set]\n");
		    }
		    /*
		     * Name of database to be used
		     */
		    else if (strcasecmp(var->name, "database") == 0) {
			if (dbname)
			    free(dbname);
			dbname = strdup(var->value);
			if (option_verbose > 2)
			    ast_verbose(VERBOSE_PREFIX_3 "Database: %s\n", var->value);
		    }
		    /*
		     * How often to reload rate and egress information
		     */
		    else if (strcasecmp(var->name, "reloadevery") == 0) {
			reloadevery = strtol(var->value, &e, 10) * 60;
			if (option_verbose > 3) {
			    if (reloadevery)
				ast_verbose(VERBOSE_PREFIX_4 "Reload Rates: every %ss\n", var->value);
			    else
				ast_verbose(VERBOSE_PREFIX_4 "Reload Rates: manually\n");
			}
		    }
		    else {
			ast_log(LOG_WARNING, "Unrecognized variable '%s' in category '%s'\n",
				var->name, cat);
		    }
		    var = var->next;
		}
	    }

	    /*
	     * Settings for logging CDR data
	     */
	    else if (strcasecmp(cat, "cdr") == 0) {
		while (var) {
		    /*
		     * Table to log CDR entries to
		     */
		    if (strcasecmp(var->name, "table") == 0) {
			if (table_cdr)
			    free(table_cdr);
			table_cdr = strdup(var->value);
			if (option_verbose > 3)
			    ast_verbose(VERBOSE_PREFIX_4 "CDR Table: %s\n", var->value);
		    }
		    /*
		     * Backlog threshold before we start dropping
		     * CDR entries (if database unavailable)
		     */
		    else if (strcasecmp(var->name, "dropthres") == 0) {
			drop_threshold = strtol(var->value, &e, 10);
			if (option_verbose > 3)
			    ast_verbose(VERBOSE_PREFIX_4 "Drop Threshold: %s entries\n", var->value);
		    }
		    /*
		     * Backlog threshold before we start warning
		     * about database being backed up or unavailable
		     */
		    else if (strcasecmp(var->name, "warnthres") == 0) {
			warn_threshold = strtol(var->value, &e, 10);
			if (option_verbose > 3)
			    ast_verbose(VERBOSE_PREFIX_4 "Warning Threshold: %s entries\n", var->value);
		    }
		    /*
		     * Frequency of warnings once we are at or above
		     * the warning threshold
		     */
		    else if (strcasecmp(var->name, "warnfreq") == 0) {
			warn_freq = strtol(var->value, &e, 10);
			if (option_verbose > 3)
			    ast_verbose(VERBOSE_PREFIX_4 "Warning Frequency: every %s entries\n", var->value);
		    }
		    else {
			ast_log(LOG_WARNING, "Unrecognized variable '%s' in category '%s'\n",
				var->name, cat);
		    }
		    var = var->next;
		}
	    }

	    /*
	     * Configuration information for Least Cost Routing
	     */
	    else if (strcasecmp(cat, "route") == 0) {
		while (var) {
		    /*
		     * Table for logging when rate information
		     * for a prefix could not be found
		     */
		    if (strcasecmp(var->name, "errors") == 0) {
			if (table_error)
			    free(table_error);
			table_error = strdup(var->value);
			if (option_verbose > 3)
			    ast_verbose(VERBOSE_PREFIX_4 "Error Table: %s\n", var->value);
		    }
		    /*
		     * Table holding egress routes
		     */
		    else if (strcasecmp(var->name, "egress") == 0) {
			if (table_egress)
			    free(table_egress);
			table_egress = strdup(var->value);
			if (option_verbose > 3)
			    ast_verbose(VERBOSE_PREFIX_4 "Egress Table: %s\n", var->value);
		    }
		    /*
		     * Table holding rate information for prefixes
		     */
		    else if (strcasecmp(var->name, "cost") == 0) {
			if (table_cost)
			    free(table_cost);
			table_cost = strdup(var->value);
			if (option_verbose > 3)
			    ast_verbose(VERBOSE_PREFIX_4 "Cost Table: %s\n", var->value);
		    }
		    /*
		     * Average length of a phone call -- used to calculate
		     * lowest rates, due to imbalances with starting cost
		     * and cost per time unit on a call
		     */
		    else if (strcasecmp(var->name, "avglen") == 0) {
			avg_call_length = strtol(var->value, NULL, 10);
			if (option_verbose > 3)
			    ast_verbose(VERBOSE_PREFIX_4 "Average Call Length: %ss\n", var->value);
		    }
		    else {
			ast_log(LOG_WARNING, "Unrecognized variable '%s' in category '%s'\n",
				var->name, cat);
		    }
		    var = var->next;
		}
	    }
	    else {
		ast_log(LOG_WARNING, "Unrecognized category '%s'\n", cat);
	    }
	    cat = ast_category_browse(cfg, cat);
	}
    }
    ast_destroy(cfg);

    /*
     * Do some quick sanity checks
     */
    if (dbhost == NULL)
	ast_log(LOG_WARNING, "MySQL host not specified -- assuming localhost\n");
    if (dbname == NULL) {
	ast_log(LOG_WARNING, "MySQL database not specified -- assuming rating\n");
	dbname = strdup("rating");
    }
    if (dbuser == NULL) {
	ast_log(LOG_WARNING, "MySQL username not specified -- assuming root\n");
	dbuser = strdup("root");
    }
    if (table_egress == NULL) {
	ast_log(LOG_WARNING, "egress table not specified -- assuming 'egress'\n");
	table_egress = strdup("egress");
    }
    if (table_cost == NULL) {
	ast_log(LOG_WARNING, "cost table not specified -- assuming 'rate'\n");
	table_cost = strdup("rate");
    }
    if (table_cdr == NULL) {
	ast_log(LOG_WARNING, "cdr table not specified -- assuming 'cdr'\n");
	table_cdr = strdup("cdr");
    }
    if (avg_call_length == 0) {
	ast_log(LOG_WARNING, "Average Call Length not specified -- assuming 180s\n");
	avg_call_length = 180;
    }
    if (drop_threshold <= warn_threshold) {
	ast_log(LOG_WARNING, "Drop threshold must be larger than warning threshold, adjusting...\n");
	drop_threshold = warn_threshold * 2;
    }

    /*
     * Configuration has been loaded -- it is now safe to use
     */
    ast_mutex_unlock(&ratelock);

    return 0;
}

static void splay_rotate(splay_t *node)
{
    splay_t *parent, *grandparent;

    parent = node->parent;

    if (parent->sentinel)
	return;
  
    grandparent = parent->parent;

    if (parent->left == node) {
	parent->left = node->right;
	if (parent->left)
	    parent->left->parent = parent;
	node->right = parent;
    }
    else if (parent->right == node) {
	parent->right = node->left;
	if (parent->right)
	    parent->right->parent = parent;
	node->left = parent;
    }
    else {
	fprintf(stderr, "rotate: error: parent's children are not right\n");
	exit(1);
    }

    parent->parent = node;
    node->parent = grandparent;

    if (grandparent->sentinel) {
	grandparent->parent = node;
    }
    else if (grandparent->left == parent) {
	grandparent->left = node;
    }
    else if (grandparent->right == parent) {
	grandparent->right = node;
    }
    else {
	fprintf(stderr, "rotate: error: grandparent's children are not right\n");
	exit(1);
    }
}

static void splay(splay_t *node)
{
    splay_t *parent, *grandparent;
  
    if (node->sentinel)
	return;

    for (;;) {
	parent = node->parent;

	if (parent->sentinel)
	    return;
  
	grandparent = parent->parent;

	/*
	 * If the node's parent is the root of the tree, do one rotation
	 */
	if (grandparent->sentinel)
	    splay_rotate(node);
	/*
	 * If we have a zig-zig, then rotate my parent, then rotate me
	 */
	else if ((parent->left  == node && grandparent->left  == parent) ||
		 (parent->right == node && grandparent->right == parent)) {
	    splay_rotate(parent);
	    splay_rotate(node);
	}
	/*
	 * If we have a zig-zag, then rotate me twice
	 */
	else {
	    splay_rotate(node);
	    splay_rotate(node);
	}
    }
}

static splay_t *new_splay(void)
{
    splay_t *tree;

    tree = (splay_t *) malloc(sizeof(splay_t));

    tree->key = NULL;
    tree->value = NULL;
    tree->sentinel = 1;
    tree->forward = tree;
    tree->back = tree;
    tree->left = NULL;
    tree->right = NULL;
    tree->parent = NULL;

    return tree;
}

static splay_t *splay_root(splay_t *tree)
{
    return tree->parent;
}

static splay_t *splay_first(splay_t *tree)
{
    return tree->forward;
}

#if 0
static splay_t *splay_last(splay_t *tree)
{
    return tree->back;
}
#endif

static splay_t *splay_next(splay_t *node)
{
    return node->forward;
}

static splay_t *splay_prev(splay_t *node)
{
    return node->back;
}

static splay_t *splay_null(splay_t *tree)
{
    return tree;
}

static void free_splay(splay_t *tree)
{
    splay_t *ptr;

    for (;;) {
	ptr = splay_first(tree);
	if (!ptr->sentinel)
	    splay_delete(ptr);
	else {
	    free(ptr);
	    return;
	}
    }
}

static splay_t *splay_find_nearest(splay_t *tree, const char *skey, int *cmpval) 
{
    splay_t *s, *last;
    int cmp;

    last = tree;
    s = splay_root(tree);
    cmp = 1;

    while (s) {
	last = s;
	cmp = strcmp(skey, s->key);
	if (cmp == 0) {
	    *cmpval = 0;
	    return s;
	}
	else if (cmp < 0)
	    s = s->left;
	else
	    s = s->right;
    }

    *cmpval = cmp;
    return last;
}
 
static splay_t *search_splay(splay_t *tree, const char *skey)
{
    int cmpval;
    splay_t *s;

    s = splay_find_nearest(tree, skey, &cmpval);
    splay(s);
    if (cmpval == 0)
	return s;
    else
	return NULL;
}

static splay_t *splay_insert(const char *skey, void *val, splay_t *parent, int cmpval)
{
    splay_t *s;

    s = (splay_t *) malloc(sizeof(splay_t));
    s->sentinel = 0;
    s->parent = parent;
    s->left = NULL;
    s->right = NULL;
    s->key = strdup(skey);
    s->value = val;

    /*
     * Set the parent's correct child pointer.  The only
     * subtle case here is when the key is already in 
     * the tree -- then we need to find a leaf node 
     * to use as a parent
     *
     * When we're done here, parent should point to the
     * new node's successor in the linked list
     */
    if (parent->sentinel)
	parent->parent = s;
    else {
	if (cmpval == 0) {   /* If the key is already in the
			      * tree, try to insert a new one as the
			      * node's right child.  If the node already
			      * has a right child, then try to insert the
			      * new one as a left child.  If there is already
			      * a left child, then go to parent-flink and 
			      * insert the node as its left child.
			      */
	    if (parent->right == NULL)
		cmpval = 1;
	    else if (parent->left == NULL)
		cmpval = -1;
	    else {
		parent = parent->forward;
		s->parent = parent;
		cmpval = -1;
	    }
	}
	if (cmpval > 0) {   /* Insert as right child */
	    if (parent->right != NULL) {
		fprintf(stderr, "splay_insert error: parent->right != NULL");
		exit(1);
	    }
	    parent->right = s;
	    parent = parent->forward;
	}
	else {
	    if (parent->left != NULL) {
		fprintf(stderr, "splay_insert error: parent->left != NULL");
		exit(1);
	    }
	    parent->left = s;
	}
    }

    s->forward = parent;
    s->back = parent->back;
    s->forward->back = s;
    s->back->forward = s;
    splay(s);
    return s;
}

static splay_t *insert_splay(splay_t *tree, const char *skey, void *val)
{
    splay_t *parent;
    int cmpval;

    parent = splay_find_nearest(tree, skey, &cmpval);
    return splay_insert(skey, val, parent, cmpval);
}

static void splay_delete(splay_t *node)
{
    splay_t *left, *right, *tree, *newroot;
    splayval_t *val;

    splay(node);

    tree = node->parent;

    left = node->left;
    right = node->right;
    newroot = node->forward;

    node->forward->back = node->back;
    node->back->forward = node->forward;

    if (node->key)
	free(node->key);
    val = node->value;
    while (val) {
	if (val->rate)
	    free_rate(val->rate);
	if (val->egress)
	    free_egress(val->egress);
	node->value = val->next;
	free(val);
	val = node->value;
    }
    free(node);

    if (right == NULL && left == NULL) {
	tree->parent = NULL;
    }
    else if (right == NULL) {
	tree->parent = left;
	left->parent = tree;
    }
    else if (left == NULL) {
	tree->parent = right;
	right->parent = tree;
    }
    else {
	tree->parent = right;
	right->parent = tree;
	splay(newroot);
	newroot->left = left;
	left->parent = newroot;
    }
}

/*
 * Clear in-memory egress table completely
 */
static void clear_egresses(void)
{
    splayval_t *val;
    splay_t *s;

    if (egresses) {
	free_splay(egresses);
	egresses = NULL;

	/*
	 * Make sure to clear egress pointers when egress routes are removed
	 */
	for (s = splay_first(rates); s != splay_null(rates); s = splay_next(s))
	    for (val = s->value; val; val = val->next)
		if (val->rate)
		    val->rate->egress = NULL;
	ast_log(LOG_DEBUG, "Cleared egress route from all rates\n");
    }
}

/*
 * Inject an egress route into the in-memory table
 */
static void inject_egress(egress_t *egress)
{
    splayval_t *val;
    char id[12];
    splay_t *s;

    if (egresses == NULL)
	egresses = new_splay();

    snprintf(id, sizeof(id), "%d", egress->route_id);

    val = malloc(sizeof(splayval_t));
    val->egress = egress;
    val->rate = NULL;

    s = search_splay(egresses, id);
    if (s) {
	ast_log(LOG_ERROR, "Duplicate egress route for %s[%s]\n", egress->technology, egress->peer);
	free_egress(egress);
	free(val);
    }
    else {
	val->next = NULL;
	s = insert_splay(egresses, id, val);

	/*
	 * If rate information is loaded, attempt to populate egress pointer
	 */
	if (rates) {
	    for (s = splay_first(rates); s != splay_null(rates); s = splay_next(s))
		for (val = s->value; val; val = val->next)
		    if (val->rate->route_id == egress->route_id)
			val->rate->egress = egress;
	    ast_log(LOG_DEBUG, "Added egress route to applicable rates\n");
	}
    }
}

/*
 * Clear in-memory rate table completely
 */
static void clear_rates(void)
{
    if (rates)
	free_splay(rates);
    rates = NULL;
}

/*
 * Inject a rate entry into the in-memory table -- when adding
 * a rate to an existing prefix, we should have a linked list
 * with least expensive first and most expensive last
 */
static void inject_rate(rate_t *rate)
{
    splayval_t *val;
    char id[12];
    splay_t *s;

    if (rates == NULL)
	rates = new_splay();

    snprintf(id, sizeof(id), "%d", rate->route_id);
    s = search_splay(egresses, id);
    if (s) {
	val = s->value;
	rate->egress = val->egress;
    }

    val = malloc(sizeof(splayval_t));
    val->egress = NULL;
    val->rate = rate;

    s = search_splay(rates, rate->prefix);
    if (s) {
	val->next = s->value;
	s->value = val;
    }
    else {
	val->next = NULL;
	insert_splay(rates, rate->prefix, val);
    }
}

/*
 * Allocate and initialize a new egress entry
 */
static egress_t *new_egress(void)
{
    egress_t *egress = malloc(sizeof(egress_t));

    egress->route_id = -1;
    egress->technology = NULL;
    egress->peer = NULL;
    egress->pcre = NULL;
    egress->extra = NULL;
    egress->substitute = NULL;
    egress->description = NULL;

    return egress;
}

/*
 * Free the resources used by an egress entry
 */
static void free_egress(egress_t *egress)
{
    if (egress->technology)
	free(egress->technology);
    if (egress->peer)
	free(egress->peer);
    if (egress->extra)
	pcre_free(egress->extra);
    if (egress->pcre)
	pcre_free(egress->pcre);
    if (egress->substitute)
	free(egress->substitute);
    if (egress->description)
	free(egress->description);
    free(egress);
}

/*
 * Allocate and initialize a new rate entry
 */
static rate_t *new_rate(void)
{
    rate_t *rate = malloc(sizeof(rate_t));

    rate->rate_id = -1;
    rate->route_id = -1;
    memset(rate->iso, '\0', sizeof(rate->iso));
    memset(rate->type, '\0', sizeof(rate->type));
    rate->country = NULL;
    rate->extra = NULL;
    memset(rate->prefix, '\0', sizeof(rate->prefix));
    rate->active_date = 0;
    rate->expires_date = 0;
    rate->firstperiod = 0;
    rate->periods = 0;
    rate->startcost = 0;
    rate->periodcost = 0;
    rate->trialcost = 0;

    return rate;
}

/*
 * Free the resources used by a rate entry
 */
static void free_rate(rate_t *rate)
{
    if (rate->country)
	free(rate->country);
    if (rate->extra)
	free(rate->extra);
    free(rate);
}

/*
 * Load egress and rate information from file into memory
 */
static int load_rates(int cli)
{
    char *query, *end;
    MYSQL_RES *result;
    egress_t *egress;
    int len, erroff;
    const char *err;
    MYSQL_ROW row;
    rate_t *rate;

    /*
     * Make sure we have exclusive access to MySQL and to the
     * in-memory rate / egress information
     */
    ast_mutex_lock(&ratelock);

    /*
     * Re-connect to MySQL if we are not currently connected
     */
    if (connected == 0) {
	if (mysql_real_connect(&mysql, dbhost, dbuser, dbpass, dbname, dbport, dbsock, 0)) {
	    if (option_verbose > 2)
		ast_verbose(VERBOSE_PREFIX_3 "Connected to database: %s\n", dbname);
	    connected = 1;
	}
	else {
	    connected = 0;
	    ast_mutex_unlock(&ratelock);
	    ast_log(LOG_ERROR, "Failed to connect to MySQL database '%s': %s\n", dbname, mysql_error(&mysql));
	    return -1;
	}
    }
    /*
     * Verify that our connection to MySQL is still alive,
     * reconnecting as needed
     */
    else if (mysql_ping(&mysql)) {
	connected = 0;
	ast_mutex_unlock(&ratelock);
	ast_log(LOG_ERROR, "Cannot talk to MySQL database '%s': %s\n", dbname, mysql_error(&mysql));
	return -1;
    }

    /*
     * Log an event so everyone knows what happened
     */
    ast_log(LOG_EVENT, "Reloading rates\n");

    /*
     * Load all egress routes into memory
     */
    len = asprintf(&query,
		   "SELECT route_id, technology, peer, pattern, substitute, description "
		   "FROM %s",
		   table_egress);
    if (mysql_real_query(&mysql, query, len)) {
	ast_mutex_unlock(&ratelock);
	ast_log(LOG_ERROR, "MySQL query failed: %s\n", mysql_error(&mysql));
	free(query);
	return -1;
    }
    else if ((result = mysql_store_result(&mysql)) == NULL) {
	ast_mutex_unlock(&ratelock);
	ast_log(LOG_ERROR, "Failed to retrieve MySQL result: %s\n", mysql_error(&mysql));
	free(query);
	return -1;
    }
    else {
	/*
	 * Now that we know the query succeeded, clear existing
	 * egress routes from memory and set the count to zero
	 */
	clear_egresses();
	num_egress = 0;

	/*
	 * Load each egress route into memory, increasing the
	 * count for each route successfully loaded
	 */
	while ((row = mysql_fetch_row(result)) != NULL) {
	    egress = new_egress();
	    egress->route_id = strtol(row[0], &end, 10);
	    egress->technology = strdup(row[1]);
	    egress->peer = strdup(row[2]);

	    /*
	     * Compile the matching pattern since we will use
	     * this quite a bit for active egress routes
	     */
	    if ((egress->pcre = pcre_compile(row[3], 0, &err, &erroff, NULL)) == NULL) {
		ast_log(LOG_WARNING, "Bad search pattern for egress route %d: %s\n", egress->route_id, err);
		free_egress(egress);
		continue;
	    }

	    /*
	     * We need the additional information for the
	     * substitute operation that we will be doing
	     * on egress route usage, too
	     */
	    egress->extra = pcre_study(egress->pcre, 0, &err);
	    if (err) {
		ast_log(LOG_WARNING, "Search pattern problem for egress route %d: %s\n", egress->route_id, err);
		free_egress(egress);
		continue;
	    }

	    egress->substitute = strdup(row[4]);
	    if (row[5])
		egress->description = strdup(row[5]);

	    /*
	     * Log egress routes on the console, but only when
	     * called during startup or from the CLI handler
	     */
	    if (cli && option_verbose > 3)
		ast_verbose(VERBOSE_PREFIX_4 "Egress route exists to %s[%s] (%s)\n",
			    egress->technology, egress->peer, egress->description);

	    num_egress++;
	    inject_egress(egress);
	}
	mysql_free_result(result);
    }
    free(query);

    /*
     * Load all rates into memory
     */
    len = asprintf(&query,
		   "SELECT rate_id, route_id, iso, type, country, extra, prefix, "
			  "UNIX_TIMESTAMP(active_date), UNIX_TIMESTAMP(expires_date), "
			  "firstperiod, periods, startcost, periodcost, trialcost "
		   "FROM %s",
		   table_cost);
    if (mysql_real_query(&mysql, query, len)) {
	ast_mutex_unlock(&ratelock);
	ast_log(LOG_ERROR, "MySQL query failed: %s\n", mysql_error(&mysql));
	free(query);
	return -1;
    }
    else if ((result = mysql_store_result(&mysql)) == NULL) {
	ast_mutex_unlock(&ratelock);
	ast_log(LOG_ERROR, "Failed to retrieve MySQL result: %s\n", mysql_error(&mysql));
	free(query);
	return -1;
    }
    else {
	/*
	 * Now that we know the query succeeded, clear existing
	 * egress routes from memory and set the count to zero
	 */
	clear_rates();
	num_rate = 0;

	/*
	 * Load each rate entry into memory, increasing the
	 * count for each rate successfully loaded
	 */
	while ((row = mysql_fetch_row(result)) != NULL) {
	    rate = new_rate();
	    rate->rate_id = strtol(row[0], &end, 10);
	    rate->route_id = strtol(row[1], &end, 10);
	    strncpy(rate->iso, row[2], sizeof(rate->iso) - 1);
	    if (row[3])
		strncpy(rate->type, row[3], sizeof(rate->type) - 1);
	    rate->country = strdup(row[4]);
	    if (row[5])
		rate->extra = strdup(row[5]);
	    strncpy(rate->prefix, row[6], sizeof(rate->prefix) - 1);
	    if (row[7])
		rate->active_date = strtol(row[7], &end, 10);
	    if (row[8])
		rate->expires_date = strtol(row[8], &end, 10);
	    rate->firstperiod = strtol(row[9], &end, 10);
	    rate->periods = strtol(row[10], &end, 10);
	    rate->startcost = strtol(row[11], &end, 10);
	    rate->periodcost = strtol(row[12], &end, 10);
	    rate->trialcost = strtol(row[13], &end, 10);

	    num_rate++;
	    inject_rate(rate);
	}
	mysql_free_result(result);
    }
    free(query);

    /*
     * Note when we last reloaded rate information
     */
    time(&lastreload);

    /*
     * And release lock, now that we are done updating the
     * in-memory tables and are no longer using MySQL
     */
    ast_mutex_unlock(&ratelock);

    return 0;
}

/*
 * CLI routine called to reload our rate and egress information
 */
static int rates_reload(int fd, int argc, char *argv[])
{
    if (load_rates(1))
	ast_cli(fd, "Error loading rates.\n");
    else
	ast_cli(fd, "Rates and egress routes reloaded.\n");
    return 0;
}

/*
 * CLI routine called to get rate and egress statistics
 */
static int rates_status(int fd, int argc, char *argv[])
{
    ast_cli(fd, "There are a total of %d rates, with %d egress routes.\n", num_rate, num_egress);
    return 0;
}

/*
 * Manager routine called to reload our rate and egress information
 */
static int manager_rates_reload(struct mansession *s, struct message *m)
{
    char *id = astman_get_header(m,"ActionID");
    char *idText = "";
    int res;

    if (id)
	asprintf(&idText, "ActionID: %s\r\n", id);
    else
	idText = NULL;

    res = load_rates(0);

    ast_cli(s->fd,
	    "Event: RatesReload\r\n"
	    "Reloaded: %s\r\n"
	    "Rates: %d\r\n"
	    "Routes: %d\r\n"
	    "%s"
	    "\r\n",
	    res ? "no" : "yes",
	    num_rate,
	    num_egress,
	    idText);
    if (id)
	free(idText);
    return 0;
}

/*
 * Manager routine called to get rate and egress statistics
 */
static int manager_rates_status(struct mansession *s, struct message *m)
{
    char *id = astman_get_header(m,"ActionID");
    char *idText = "";

    if (id)
	asprintf(&idText, "ActionID: %s\r\n", id);
    else
	idText = NULL;

    ast_cli(s->fd,
	    "Event: RatesStatus\r\n"
	    "Rates: %d\r\n"
	    "Routes: %d\r\n"
	    "%s"
	    "\r\n",
	    num_rate,
	    num_egress,
	    idText);
    if (id)
	free(idText);
    return 0;
}

static char rates_reload_usage[] =
"Usage: rates reload\n"
"       Reloads all rates from storage\n";

static char rates_status_usage[] =
"Usage: rates status\n"
"       Display statistics on rates in memory\n";

static struct ast_cli_entry cli_rates_reload = 
{ { "rates", "reload", NULL }, rates_reload, "Reloads all rates from storage", rates_reload_usage, NULL, NULL, 0 };

static struct ast_cli_entry cli_rates_status = 
{ { "rates", "status", NULL }, rates_status, "Display statistics on rates in memory", rates_status_usage, NULL, NULL, 0 };

/*
 * Called to reload configuration, rate and egress information
 */
int reload(void)
{
    int res;

    /*
     * Reload our configuration
     */
    res = load_config();

    /*
     * If successful, reload rates
     */
    if (res == 0)
	res = load_rates(1);

    return res;
}

int unload_module(void)
{
    int res;

    /*
     * Hang up on any channel that is connected to us
     */
    STANDARD_HANGUP_LOCALUSERS;

    /*
     * Unregister CDR handler
     */
    ast_cdr_unregister(cdr_name);

    /*
     * Unregister application entry point
     */
    res = ast_unregister_application(routecall_app);

    /*
     * Unregister manager entry points
     */
    ast_manager_unregister("RatesStatus");
    ast_manager_unregister("RatesReload");

    /*
     * Unregister CLI entry points
     */
    ast_cli_unregister(&cli_rates_reload);
    ast_cli_unregister(&cli_rates_status);

    /*
     * If we have a worker thread, cancel it
     */
    if (!pthread_equal(poster_thread, (pthread_t) -1)) {

	/*
	 * Get lock on shared data
	 */
	pthread_mutex_lock(&poster_lock);

	/*
	 * Tell posting thread to exit and signal a change
	 */
	cancel_poster = 1;
	pthread_cond_signal(&poster_cond);

	/*
	 * Release lock on shared data
	 */
	pthread_mutex_unlock(&poster_lock);

	/*
	 * Wait for posting thread to exit
	 */
	pthread_join(poster_thread, NULL);
	poster_thread = (pthread_t) -1;
    }

    /*
     * Close database connection
     */
    mysql_close(&mysql);
    connected = 0;

    /*
     * Free resources
     */
    if (dbhost) {
	free(dbhost);
	dbhost = NULL;
    }
    if (dbuser) {
	free(dbuser);
	dbuser = NULL;
    }
    if (dbpass) {
	free(dbpass);
	dbpass = NULL;
    }
    if (dbname) {
	free(dbname);
	dbname = NULL;
    }
    if (dbsock) {
	free(dbsock);
	dbsock = NULL;
    }
    if (table_cdr) {
	free(table_cdr);
	table_cdr = NULL;
    }
    if (table_error) {
	free(table_error);
	table_error = NULL;
    }
    if (table_egress) {
	free(table_egress);
	table_egress = NULL;
    }
    if (table_cost) {
	free(table_cost);
	table_cost = NULL;
    }
    clear_rates();
    clear_egresses();

    return res;
}

int load_module(void)
{
    int res;

    /*
     * Load configuration settings
     */
    res = load_config();
    if (res)
	return res;

    /*
     * Initialize database connection
     */
    mysql_init(&mysql);

    /*
     * Load route and rate information into memory
     */
    load_rates(1);

    /*
     * Register CLI entry points
     */
    ast_cli_register(&cli_rates_reload);
    ast_cli_register(&cli_rates_status);

    /*
     * Register manager entry points
     */
    ast_manager_register("RatesReload", 0, manager_rates_reload, "Rates Reload");
    ast_manager_register("RatesStatus", 0, manager_rates_status, "Rates Status");

    /*
     * Register application entry point
     */
    res = ast_register_application(routecall_app, routecall_exec, routecall_synopsis, routecall_descrip);
    if (res) {
	ast_log(LOG_ERROR, "Unable to register RouteCall application\n");
	return res;
    }

    /*
     * Create worker thread to post CDR data to the Rating Engine
     */
    if (pthread_create(&poster_thread, NULL, poster_worker, NULL)) {
	ast_log(LOG_ERROR, "Unable to create CDR Rating Engine Poster thread.\n");
	return -1;
    }

    /*
     * Register us as a CDR consumer
     */
    res = ast_cdr_register(cdr_name, tdesc, cdr_ratecall);
    if (res) {
	ast_log(LOG_ERROR, "Unable to register CDR handling\n");
	return res;
    }

    return res;
}

char *description(void)
{
    /*
     * Return a terse description of the module
     */
    return tdesc;
}

int usecount(void)
{
    int res;

    /*
     * Return the number of current users of this module --
     * count our worker thread as a user in this context
     */
    STANDARD_USECOUNT(res);
    return res + pthread_equal(poster_thread, (pthread_t) -1) ? 0 : 1;
}

char *key()
{
    return ASTERISK_GPL_KEY;
}
