/*
 * RageIRCd: an advanced Internet Relay Chat daemon (ircd).
 * (C) 2000-2005 the RageIRCd Development Team, all rights reserved.
 *
 * This software is free, licensed under the General Public License.
 * Please refer to doc/LICENSE and doc/README for further details.
 *
 * $Id: memory.c,v 1.11.2.1 2004/12/07 03:05:14 pneumatus Exp $
 */

#include "struct.h"
#include "common.h"
#include "sys.h"
#include "h.h"
#include "numeric.h"
#include "blalloc.h"
#include "setup.h"
#include "memory.h"
#include <stdlib.h>
#include <string.h>

static void mem_fail(char *file, int line)
{
#ifdef ABORT_ON_MALLOC_FAILURE
	ircdlog(LOG_ERROR, "FATAL: malloc failed at %s:%d, aborting!", file, line);
	abort();
#else
	ircdlog(LOG_ERROR, "malloc failed at %s:%d, restarting server...", file, line);
	outofmemory();
#endif
}

void *_MyMalloc(size_t size, char *file, int line)
{
	void *m;

	if ((m = calloc(1, size)) == NULL) {
		mem_fail(file, line);
	}

	return m;
}

void *_MyRealloc(void *p, size_t size, char *file, int line)
{
	void *ret;

	if (p == NULL) {
		return _MyMalloc(size, file, line);
	}
	if (!size) {
		_MyFree(p, file, line);
		return NULL;
	}
	if ((ret = realloc(p, size)) == NULL) {
		mem_fail(file, line);
	}
	return ret;
}

void _MyFree(void *p, char *file, int line)
{
	if (p != NULL) {
		free(p);
	}
}

void _DupString(char **p, const char *s, char *file, int line)
{
	*p = _MyMalloc(strlen(s) + 1, file, line);
	strcpy(*p, s);
}

void *_BlockHeapAlloc(BlockHeap *heap, char *file, int line)
{
#ifdef USE_BLALLOC
	void *ret;

	if ((ret = BlockHeapALLOC(heap)) == NULL) {
		mem_fail(file, line);
	}
	return ret;
#else
	return _MyMalloc((size_t)heap, file, line);
#endif
}

void _BlockHeapFree(BlockHeap *heap, void *p, char *file, int line)
{
#ifdef USE_BLALLOC
	ASSERT(heap != NULL);
	if (p != NULL) {
		BlockHeapFREE(heap, p);
	}
#else
	if (p != NULL) {
		_MyFree(p, file, line);
	}
#endif
}

void outofmemory(void)
{
	ircdlog(LOG_ERROR, "FATAL: out of memory, restarting server.");
	Debug((DEBUG_FATAL, "Out of memory: restarting server..."));
	restart("Out of memory");
}
