/*
 * RageIRCd: an advanced Internet Relay Chat daemon (ircd).
 * (C) 2000-2005 the RageIRCd Development Team, all rights reserved.
 *
 * This software is free, licensed under the General Public License.
 * Please refer to doc/LICENSE and doc/README for further details.
 *
 * $Id: list.c,v 1.59.2.1 2004/12/07 03:05:13 pneumatus Exp $
 */

#include "struct.h"
#include "common.h"
#include "sys.h"
#include "h.h"
#include "numeric.h"
#include "blalloc.h"
#include "zlink.h"
#include "setup.h"
#include "memory.h"
#include "xmode.h"

BlockHeap *slink_heap = NULL;

void init_link()
{
	slink_heap = BlockHeapCreate(sizeof(SLink), SLINK_HEAP_SIZE);
}

LocalClient *make_localclient()
{
	return (LocalClient *)BlockHeapAlloc(localclient_heap);
}

void free_localclient(LocalClient *lcptr)
{
	BlockHeapFree(localclient_heap, lcptr);
}

aClient *make_client(aClient *from, aClient *uplink)
{
	aClient *cptr = (aClient *)BlockHeapAlloc(client_heap);

	cptr->status = STAT_UNKNOWN;
	cptr->uplink = uplink;
	cptr->id.id = 0xFFFFFFFF;

	strcpy(cptr->username, "unknown");

	if (from == NULL) {
		cptr->from = cptr;
		cptr->lasttime = cptr->firsttime = cptr->since = timeofday;
		cptr->localClient = make_localclient();
		cptr->localClient->sockerr = -1;
		cptr->localClient->class = NULL;
		cptr->localClient->fd = -1;
		dlink_add_node(&lunknown_list, &cptr->localClient->self, cptr);
		cptr->localUser = NULL;
	}
	else {
		ASSERT(from->localClient != NULL);
		cptr->from = from;
		cptr->localClient = NULL;
		cptr->localUser = NULL;
	}

	return cptr;
}

void free_client(aClient *cptr)
{
	if (cptr->localClient != NULL) {
		free_localclient(cptr->localClient);
	}
	BlockHeapFree(client_heap, cptr);
}

LocalUser *make_localuser(aClient *cptr)
{
	ASSERT(cptr->localClient != NULL);
	if (cptr->localUser == NULL) {
		cptr->localUser = (LocalUser *)BlockHeapAlloc(localuser_heap);
	}
	return cptr->localUser;
}

void free_localuser(LocalUser *luser)
{
	ASSERT(luser->invited == NULL);
	BlockHeapFree(localuser_heap, luser);
}

anUser *make_user(aClient *cptr)
{
	if (cptr->user == NULL) {
		cptr->user = (anUser *)BlockHeapAlloc(user_heap);
	}
	return cptr->user;
}

aServer *make_server(aClient *cptr)
{
	if (cptr->serv == NULL) {
		cptr->serv = (aServer *)MyMalloc(sizeof(aServer));
	}
	return cptr->serv;
}

void free_user(anUser *user)
{
	ASSERT(user->joined == 0);
	ASSERT(user->channel == NULL);

	if (user->away != NULL) {
		MyFree(user->away);
	}
	BlockHeapFree(user_heap, user);
}

aChannel *make_channel()
{
	return (aChannel *)BlockHeapAlloc(channel_heap);
}

void free_channel(aChannel *chptr)
{
	BlockHeapFree(channel_heap, chptr);
}

channelBan *make_channelban()
{
	return (channelBan *)BlockHeapAlloc(channelban_heap);
}

void free_channelban(channelBan *ban)
{
	BlockHeapFree(channelban_heap, ban);
}

void remove_client_from_list(aClient *cptr)
{
	if (IsServer(cptr)) {
		Count.server--;
	}
	else if (IsClient(cptr)) {
		Count.total--;
		if (HasMode(cptr, UMODE_OPER)) {
			Count.oper--;
		}
		if (HasMode(cptr, UMODE_INVISIBLE)) {
			Count.invisi--;
		}
	}
	if (cptr->prev != NULL) {
		cptr->prev->next = cptr->next;
	}
	else {
		client = cptr->next;
		client->prev = NULL;
	}
	if (cptr->next != NULL) {
		cptr->next->prev = cptr->prev;
	}
	if (IsPerson(cptr) && cptr->user != NULL) {
		add_history(cptr, 0);
		off_history(cptr);
	}
#ifdef FLUD
	if (MyConnect(cptr)) {
		free_fluders(cptr, NULL);
	}
	free_fludees(cptr);
#endif
	if (cptr->user != NULL) {
		free_user(cptr->user);
	}
	if (cptr->localUser != NULL) {
		free_localuser(cptr->localUser);
	}
	if (cptr->serv != NULL) {
#ifdef USE_OPENSSL
		if (cptr->serv->sessioninfo_in != NULL) {
			dh_end_session(cptr->serv->sessioninfo_in);
		}
		if (cptr->serv->sessioninfo_out != NULL) {
			dh_end_session(cptr->serv->sessioninfo_out);
		}
		if (cptr->serv->rc4_in != NULL) {
			rc4_destroystate(cptr->serv->rc4_in);
		}
		if (cptr->serv->rc4_out != NULL) {
			rc4_destroystate(cptr->serv->rc4_out);
		}
#endif
		if (cptr->serv->zip_in != NULL) {
			zip_destroy_input_session(cptr->serv->zip_in);
		}
		if (cptr->serv->zip_out != NULL) {
			zip_destroy_output_session(cptr->serv->zip_out);
		}
		MyFree(cptr->serv);
	}
	free_client(cptr);
}

void add_client_to_list(aClient *cptr)
{
	cptr->next = client;
	client = cptr;
	if (cptr->next != NULL) {
		cptr->next->prev = cptr;
	}
}

chanMember *find_user_member(chanMember *cm, aChannel *chptr)
{
	if (chptr != NULL) {
		while (cm != NULL) {
			if (cm->chptr == chptr) {
				return cm;
			}
			cm = cm->nextchan;
		}
	}
	return NULL;
}

SLink *find_str_link(SLink *lp, char *charptr)
{
	if (charptr != NULL) {
		for (; lp != NULL; lp = lp->next) {
			if (!match(lp->value.cp, charptr)) {
				return lp;
			}
		}
	}
	return NULL;
}

SLink *make_slink()
{
	SLink *lp = (SLink *)BlockHeapAlloc(slink_heap);
	lp->next = NULL;
	return lp;
}

void free_slink(SLink *lp)
{
	BlockHeapFree(slink_heap, lp);
}

chanMember *make_chanmember()
{
	return (chanMember *)BlockHeapAlloc(chanmember_heap);
}

void free_chanmember(chanMember *cm)
{
	BlockHeapFree(chanmember_heap, cm);
}

#ifdef USE_THROTTLE
throttle *make_throttle()
{
	return (throttle *)BlockHeapAlloc(throttle_heap);
}

void free_throttle(throttle *tp)
{
	BlockHeapFree(throttle_heap, tp);
}

hashent *make_hashent()
{
	return (hashent *)BlockHeapAlloc(hashent_heap);
}

void free_hashent(hashent *h)
{
	BlockHeapFree(hashent_heap, h);
}
#endif

dlink_node *make_dlink_node()
{
	return (dlink_node *)BlockHeapAlloc(dlinknode_heap);
}

void free_dlink_node(dlink_node *node)
{
	BlockHeapFree(dlinknode_heap, node);
}
