/*
 * RageIRCd: an advanced Internet Relay Chat daemon (ircd).
 * (C) 2000-2005 the RageIRCd Development Team, all rights reserved.
 *
 * This software is free, licensed under the General Public License.
 * Please refer to doc/LICENSE and doc/README for further details.
 *
 * $Id: engine.c,v 1.53.2.2 2004/12/07 03:05:09 pneumatus Exp $
 */

#include "setup.h"
#include "config.h"
#include "struct.h"
#include "common.h"
#include "sys.h"
#include "h.h"
#include "fd.h"
#include "zlink.h"
#include "memory.h"
#include "res.h"
#include "ssl.h"
#include "hook.h"
#include "xmode.h"
#include <sys/types.h>
#include <unistd.h>
#include <stdio.h>

#ifdef INCREASE_SOCK_BUFS
extern char *readbuf;
extern int rcvbufmax, sndbufmax;
#else
extern char readbuf[MAX_CLIENT_RECVQ];
#endif

#ifdef USE_OPENSSL
#define secure_accept(x) safe_SSL_accept((x)->localClient->ssl, (x)->localClient->fd, (x)->localClient->sockhost)
#endif

static int engine_parse_queued(aClient *);

static const char *engine_error_table[] = {
	"success",
	"DNS lookup error",
	"bind error",
	"connect error",
	"connect timed out",
	"died during handshake",
	"flushed",
	"internal engine error"
};

static char *engine_get_sockerr(aClient *cptr)
{
	ASSERT(cptr != NULL);
	ASSERT(cptr->localClient != NULL);

	if (cptr->localClient->sockerr > 0) {
		return strerror(cptr->localClient->sockerr);
	}
	else if (cptr->localClient->sockerr < 0) {
		switch (cptr->localClient->sockerr) {
			case -1:
				return "unknown socket error (this is a bug)";
			case IRCERR_BUFALLOC:
				return "sbuf allocation error";
			case IRCERR_ZIP:
				return "compression failure";
			case IRCERR_SSL:
				return "SSL error";
			default:
				break;
		}
	}

	return "No error";
}

int engine_ignore_errno(int errtmp)
{
	switch (errtmp) {
		case EALREADY:
		case EINTR:
		case EINPROGRESS:
		case EWOULDBLOCK:
#if EAGAIN != EWOULDBLOCK
		case EAGAIN:
#endif
#ifdef ERESTART
		case ERESTART:
#endif
			return 1;
		default:
			break;
	}

	return 0;
}

static void engine_flush_queued(int fd, void *data, int engine_status_unused)
{
	aClient *cptr = (aClient *)data;

	ASSERT(cptr != NULL);
	ASSERT(cptr->localClient != NULL);

	if (DeadSocket(cptr)) {
		return;
	}

	ASSERT(HasRecvQ(cptr));
	ASSERT(SBufLength(&cptr->localClient->recvQ) > 0);

	if (engine_parse_queued(cptr) != FLUSH_BUFFER) {
		/* The client has not been exited, so flush again if needed */

		if (SBufLength(&cptr->localClient->recvQ) && !NoNewLine(cptr)) {
			engine_set_flush(fd, 1000, engine_flush_queued, cptr);
		}
		else {
			ClearRecvQ(cptr);
		}
	}
}

static int engine_parse_queued(aClient *cptr)
{
	int len = 0;

	ASSERT(cptr != NULL);
	ASSERT(cptr->localClient != NULL);

	while (SBufLength(&cptr->localClient->recvQ) && !NoNewLine(cptr)
	  && ((cptr->status < STAT_UNKNOWN) || (cptr->since - timeofday < 10) || DoingDKEY(cptr))) {
		if (IsServer(cptr) || DoingDKEY(cptr)) {
#ifdef INCREASE_SOCK_BUFS
			len = sbuf_get(&cptr->localClient->recvQ, readbuf, rcvbufmax * sizeof(char));
#else
			len = sbuf_get(&cptr->localClient->recvQ, readbuf, sizeof(readbuf));
#endif

			if (!len) {
				break;
			}

			if (serv_dopacket(cptr, readbuf, len) == FLUSH_BUFFER) {
				/* cptr has been exited! */
				return FLUSH_BUFFER;
			}

			break;
		}

#ifdef INCREASE_SOCK_BUFS
		len = sbuf_getmsg(&cptr->localClient->recvQ, readbuf, rcvbufmax * sizeof(char));
#else
		len = sbuf_getmsg(&cptr->localClient->recvQ, readbuf, sizeof(readbuf));
#endif

		if (!len) {
			if (SBufLength(&cptr->localClient->recvQ) < 510) {
				SetNoNewLine(cptr);
			}
			else {
				SBufClear(&cptr->localClient->recvQ);
			}
			break;
		}

		if (user_dopacket(cptr, readbuf, len) == FLUSH_BUFFER) {
			/* cptr has been exited! */
			return FLUSH_BUFFER;
		}
	}

	if (!HasRecvQ(cptr) && SBufLength(&cptr->localClient->recvQ) && !NoNewLine(cptr)) {
		/* We have more data, register a flush ... */
		SetRecvQ(cptr);
		engine_set_flush(cptr->localClient->fd, 1000, engine_flush_queued, cptr);
	}

	return 0;
}

static void engine_read_error(aClient *cptr, int len)
{
	char *errmsg, errbuf[512];

	ASSERT(cptr != NULL);
	ASSERT(cptr->localClient != NULL);

	errmsg = engine_get_sockerr(cptr);

	Debug((DEBUG_ERROR, "READ ERROR: fd = %d len = %d err_no = %d (%s)", cptr->localClient->fd,
		len, cptr->localClient->sockerr, errmsg));

	if (IsAnyServer(cptr)) {
		char *path = get_client_name(cptr, HIDE_IP);

		if (!len) {
			send_gnotice("Server %s closed the connection.", path);
			sendto_serv_msg_butone(cptr, &me, &CMD_GNOTICE,
				":Server %s closed the connection.", path);
		}
		else if (IsConnecting(cptr) || IsHandshake(cptr)) {
			send_gnotice("Connect error to %s (%s)", path, errmsg);
			sendto_serv_msg_butone(cptr, &me, &CMD_GNOTICE,
				"Connect error to %s (%s)", path, errmsg);
		}
		else {
			send_gnotice("Read error from %s (%s), closing link", path, errmsg);
			sendto_serv_msg_butone(cptr, &me, &CMD_GNOTICE,
				":Read error from %s (%s), closing link", path, errmsg);
		}
	}

	if (len == -1) {
		ircsprintf(errbuf, "Read error: %s", errmsg);
	}
	else {
		ircsprintf(errbuf, "Client closed connection");
	}

	exit_client(cptr, cptr, &me, errbuf);
}

void engine_write_error(aClient *cptr, char *error, int sockerr)
{
	char *errmsg, errbuf[512];

	ASSERT(cptr != NULL);
	ASSERT(cptr->localClient != NULL);

	cptr->localClient->sockerr = sockerr;
	SetDeadSocket(cptr);

	SBufClear(&cptr->localClient->recvQ);
	SBufClear(&cptr->localClient->sendQ);

	errmsg = SendQEx(cptr) ? "SendQ exceeded" : engine_get_sockerr(cptr);

	if (!IsPerson(cptr) && !IsUnknown(cptr) && !IsClosing(cptr)) {
		char *path = get_client_name(cptr, HIDE_IP);

		ircsprintf(errbuf, "%s", error);
		send_gnotice(errbuf, path, errmsg);
		sendto_serv_msg_butone(cptr, &me, &CMD_GNOTICE, errbuf, path, errmsg);
	}

	ircsprintf(errbuf, "Write error: %s", errmsg);
	exit_client(cptr, cptr, &me, errbuf);
}

void engine_connect_error(aClient *cptr, int engine_status, int err_no)
{
	const char *engine_errmsg;
	char *socket_errmsg = NULL, errbuf[512];

	ASSERT(cptr != NULL);
	ASSERT(cptr->localClient != NULL);

	engine_errmsg = engine_strerror(engine_status);
	if (engine_status != ENGINE_ERR_TIMEOUT) {
		socket_errmsg = strerror(err_no);
	}

	if (IsConnecting(cptr) || IsHandshake(cptr)) {
		if (socket_errmsg == NULL) {
			send_gnotice("Connect to %s failed: %s",
				get_client_name(cptr, SHOW_IP), engine_errmsg);
		}
		else {
			send_gnotice("Connect to %s failed: %s (%s)",
				get_client_name(cptr, SHOW_IP), engine_errmsg, socket_errmsg);
		}
	}

	ircsprintf(errbuf, "Connect error: %s", socket_errmsg);
	exit_client(cptr, cptr, &me, errbuf);
}

static void engine_send_unblock(int fd, void *data, int engine_status_unused)
{
	/* Next transmit will occur on the IO loop */
	ClearBlocked((aClient *)data);
}

void engine_send_queued(aClient *cptr)
{
	int len = 0, ret = 0, more = 0;
	char *buf;
	
	Debug((DEBUG_DEBUG, "ENGINE: send queued for [%s]", cptr->name));

	ASSERT(cptr != NULL);
	ASSERT(cptr->localClient != NULL);

#ifdef USE_OPENSSL
	if (IsSecure(cptr) && !SSL_is_init_finished(cptr->localClient->ssl)) {
		if (DeadSocket(cptr) || secure_accept(cptr) == -1) {
			engine_write_error(cptr, "SSL initialisation failure", IRCERR_SSL);
		}
		return;
	}
#endif

	if (DeadSocket(cptr)) {
		return;
	}

	if (zip_out(cptr)) {
		if (SBufLength(&cptr->localClient->sendQ)) {
			more = 1;
		}
		else {
			int ldata = SendingBurst(cptr);

			buf = zip_output(cptr->serv->zip_out, NULL, &len, 1, &ldata);
			if (len == -1) {
				sendto_realops("Zip output error for %s: (%d) %s", cptr->name, ldata, buf);
				engine_write_error(cptr, "Zip output error for %s", IRCERR_ZIP);
				return;
			}

#ifdef USE_OPENSSL
			if (OutputRC4(cptr)) {
				rc4_process_stream(cptr->serv->rc4_out, buf, len);
			}
#endif

			if (sbuf_put(&cptr->localClient->sendQ, buf, len) == -1) {
				engine_write_error(cptr, "Buffer allocation error for %s", IRCERR_BUFALLOC);
				return;
			}
		}
	}

	while (SBufLength(&cptr->localClient->sendQ)) {
		buf = sbuf_map(&cptr->localClient->sendQ, &len);
		ret = ircsend(cptr, buf, len);

		if (ret == -1) {
			if (engine_ignore_errno(errno)) {
				/* Block it and just re-register the write */
				SetBlocked(cptr);
				engine_set_call(cptr->localClient->fd, FDEV_WRITE, engine_send_unblock, cptr, 0);
			}
			else {
				/* Fatal error, exit the client */
				engine_write_error(cptr, "Write error to %s, closing link (%s)", errno);
			}
			return;
		}
		if (ret == 0) {
			/* EOF... */
			/* FIXME -- this is a disconnect */
			engine_write_error(cptr, "Write error (EOF) to %s, closing link", 0);
			return;
		}

		sbuf_delete(&cptr->localClient->sendQ, ret);
		cptr->localClient->last_sendq = (SBufLength(&cptr->localClient->sendQ) >> 10);

		cptr->localClient->sendB += ret;
		me.localClient->sendB += ret;

		if (cptr->localClient->sendB > 1023) {
			cptr->localClient->sendK += (cptr->localClient->sendB >> 10);
			cptr->localClient->sendB &= 0x03ff;
		}
		if (me.localClient->sendB > 1023) {
			me.localClient->sendK += (me.localClient->sendB >> 10);
			me.localClient->sendB &= 0x03ff;
		}

		if (more && !SBufLength(&cptr->localClient->sendQ)) {
			int ldata = SendingBurst(cptr);

			more = 0;

			buf = zip_output(cptr->serv->zip_out, NULL, &len, 1, &ldata);
			if (len == -1) {
				sendto_realops("Zip output error (more) for %s: (%d) %s",
					cptr->name, ldata, buf);
				engine_write_error(cptr, "Zip output error (more) for %s", IRCERR_ZIP);
				return;
			}

#ifdef USE_OPENSSL
			if (OutputRC4(cptr)) {
				rc4_process_stream(cptr->serv->rc4_out, buf, len);
			}
#endif

			if (sbuf_put(&cptr->localClient->sendQ, buf, len) == -1) {
				engine_write_error(cptr, "Buffer allocation error (more) for %s",
					IRCERR_BUFALLOC);
				return;
			}
		}
	}
	
	if (IsServer(cptr) && CapBURST(cptr) && SentSOB(cptr)
	  && (SBufLength(&cptr->localClient->sendQ) < 20480)) {
		/* Bursting complete, send EOB */
		if (!SendingBurst(cptr)) {
			ClearSentSOB(cptr);
			sendto_one_client_nopostfix(cptr, NULL, &CMD_BURST, "%d", SBufLength(&cptr->localClient->sendQ));
		}
	}
}

void engine_read_packet(int fd, void *data, int engine_status_unused)
{
	aClient *cptr = (aClient *)data;
	int len = 0;

	ASSERT(cptr != NULL);
	ASSERT(cptr->localClient != NULL);

#ifdef USE_OPENSSL
	if (IsSecure(cptr) && !SSL_is_init_finished(cptr->localClient->ssl)) {
		if (DeadSocket(cptr) || secure_accept(cptr) == -1) {
			engine_read_error(cptr, 0);
		}
		else {
			engine_set_call(cptr->localClient->fd, FDEV_READ, engine_read_packet, cptr, 0);
		}
		return;
	}
#endif

	if (DeadSocket(cptr)) {
		return;
	}

#ifdef INCREASE_SOCK_BUFS
	len = ircrecv(cptr, readbuf,
		IsPerson(cptr) ? (MAX_CLIENT_RECVQ * sizeof(char)) : (rcvbufmax * sizeof(char)));
#else
	len = ircrecv(cptr, readbuf, sizeof(readbuf));
#endif

	if (len <= 0) {
		if ((len == -1) && engine_ignore_errno(errno)) {
			/* Just re-register the read */
			Debug((DEBUG_DEBUG, "SETCALL: fd=%d engine_read_packet() re-registering, len %d", fd, len));
			engine_set_call(cptr->localClient->fd, FDEV_READ, engine_read_packet, cptr, 0);
		}
		else {
			cptr->localClient->sockerr = (len == -1) ? errno : 0;
			engine_read_error(cptr, len);
		}
		return;
	}

	if ((cptr->lasttime = timeofday) > cptr->since) {
		cptr->since = timeofday;
	}

	ClearPingSent(cptr);
	ClearNoNewLine(cptr);

	/* If it's a server we process the data instantly */
	if (IsAnyServer(cptr) || DoingDKEY(cptr)) {
		if (serv_dopacket(cptr, readbuf, len) != FLUSH_BUFFER) {
			/* Done! Next read... */
			Debug((DEBUG_DEBUG, "SETCALL: fd=%d engine_read_packet() anyserver, re-registering", fd));
			engine_set_call(cptr->localClient->fd, FDEV_READ, engine_read_packet, cptr, 0);
		}
		return;
	}

	if (sbuf_put(&cptr->localClient->recvQ, readbuf, len) == -1) {
		exit_client(cptr, cptr, cptr, "sbuf_put() fail");
		return;
	}

	if (IsPerson(cptr)) {
		int limit = FloodConfig.user_recvq_limit;

		if (FloodConfig.increase_oper_recvq && HasMode(cptr, UMODE_OPER)) {
			limit *= 2; /* double the RecvQ limit */
		}

		if (SBufLength(&cptr->localClient->recvQ) > limit) {
			sendto_realops_lev(FLOOD_LEV, "%s!%s@%s exceeded RecvQ limit: %d > %d",
				cptr->name, cptr->username, MaskedHost(cptr),
				SBufLength(&cptr->localClient->recvQ), limit);
			exit_client(cptr, cptr, cptr, "Excess Flood");
			return;
		}
	}

	if (engine_parse_queued(cptr) != FLUSH_BUFFER) {
		/* We still have the client, so register the next read */
		engine_set_call(cptr->localClient->fd, FDEV_READ, engine_read_packet, cptr, 0);
	}
}

const char *engine_strerror(int error)
{
	if ((error < ENGINE_OK) || (error > ENGINE_ERROR)) {
		error = ENGINE_ERROR;
	}
	return engine_error_table[error];
}

int engine_open(int sock_type, int proto)
{
	int fd;

	if (fds.count >= MASTER_MAX) {
		errno = ENFILE;
		return -1;
	}
	if ((fd = socket(AF_INET, sock_type, proto)) < 0) {
		return -1;
	}
	if (!set_non_blocking(fd)) {
		ircdlog(LOG_ERROR, "engine_open() fd %d set_non_blocking() failure", fd);
		close(fd);
		return -1;
	}

	fd_open(fd);
	return fd;
}

int engine_accept(int fd, struct sockaddr_in *addr)
{
	int newfd, addrlen = sizeof(struct sockaddr_in);

	if (fds.count >= MASTER_MAX) {
		errno = ENFILE;
		return -1;
	}
	if ((newfd = accept(fd, (struct sockaddr *)addr, &addrlen)) == -1) {
		return -1;
	}
	if (!set_non_blocking(newfd)) {
		ircdlog(LOG_ERROR, "engine_accept() fd %d set_non_blocking() failure", newfd);
		close(newfd);
		return -1;
	}

	fd_open(newfd);
	return newfd;
}

void engine_set_timeout(int fd, time_t timeout, FDCB *cb, void *data)
{
	fd_entry *fde = &fd_table[fd];
	SetCallback(fde->timeout, cb, data);
	fde->timeout_time = timeofday + (timeout / 1000);
}

void engine_set_flush(int fd, time_t timeout, FDCB *cb, void *data)
{
	fd_entry *fde = &fd_table[fd];
	SetCallback(fde->flush, cb, data);
	fde->flush_time = timeofday + (timeout / 1000);
}

void engine_check_timeouts()
{
	int i;
	fd_entry *fde;
	FDCB *cb;
	void *data;

	for (i = 0; i <= fds.highest; i++) {
		fde = &fd_table[i];

		if (!fde->open) {
			continue;
		}
		ASSERT(fde->fd > FD_UNUSED);

		if ((fde->flush_time > 0) && (timeofday >= fde->flush_time)) {
			GetCallback(fde->flush, cb, data);
			ClrCallback(fde->flush);
			fde->flush_time = 0;

			if (cb != NULL) {
				Debug((DEBUG_DEBUG, "Engine flush on fd %d", fde->fd));
				cb(fde->fd, data, ENGINE_ERR_FLUSH);
			}
		}

		if ((fde->timeout_time > 0) && (timeofday >= fde->timeout_time)) {
			GetCallback(fde->timeout, cb, data);
			ClrCallback(fde->timeout);
			fde->timeout_time = 0;

			if (cb != NULL) {
				Debug((DEBUG_DEBUG, "Engine timeout on fd %d", fde->fd));
				cb(fde->fd, data, ENGINE_ERR_TIMEOUT);
			}
		}
	}
}

static void engine_connect_callback(int fd, int status)
{
	fd_entry *fde = &fd_table[fd];
	FDCB *cb;
	void *data;

	GetCallback(fde->connect, cb, data);
	ClrCallback(fde->connect);

	if (cb != NULL) {
		ClrCallback(fde->timeout);
		cb(fde->fd, data, status);
	}
}

static void engine_connect_try(int fd, void *data, int engine_status_unused)
{
	fd_entry *fde = &fd_table[fd];

	if (connect(fd, (struct sockaddr *)&fde->ip, sizeof(struct sockaddr_in)) == -1) {
		if (errno == EISCONN) {
			Debug((DEBUG_DEBUG, "engine_connect_try() connect failure, errno EISCONN, state:ENGINE_OK"));
			engine_connect_callback(fd, ENGINE_OK);
		}
		else if (engine_ignore_errno(errno)) {
			Debug((DEBUG_DEBUG, "engine_connect_try() connect failure, errno %d(%s) ignored, re-register",
				errno, strerror(errno)));
			engine_set_call(fd, FDEV_WRITE, engine_connect_try, NULL, 0);
		}
		else {
			Debug((DEBUG_DEBUG, "engine_connect_try() connect failure, errno %d, state:ENGINE_ERR_CONNECT", errno));
			engine_connect_callback(fd, ENGINE_ERR_CONNECT);
		}
	}
	else {
		Debug((DEBUG_DEBUG, "engine_connect_try() connect success? state:ENGINE_OK"));
		engine_connect_callback(fd, ENGINE_OK);
	}
}

static void engine_connect_timeout(int fd, void *data, int engine_status_unused)
{
	engine_connect_callback(fd, ENGINE_ERR_TIMEOUT);
}

static void engine_connect_dns_callback(void *vfd, adns_answer *answer)
{
	fd_entry *fde = (fd_entry *)vfd;

	if (answer == NULL) {
		engine_connect_callback(fde->fd, ENGINE_ERR_DNS);
		return;
	}
	if (answer->status != adns_s_ok) {
		engine_connect_callback(fde->fd, ENGINE_ERR_DNS);
		MyFree(answer);
		MyFree(fde->dns_query);
		return;
	}

	fde->ip.sin_addr.s_addr = answer->rrs.addr->addr.inet.sin_addr.s_addr;
	engine_set_timeout(fde->fd, 30 * 1000, engine_connect_timeout, NULL);

	MyFree(answer);
	MyFree(fde->dns_query);

	engine_connect_try(fde->fd, NULL, ENGINE_OK);
}

void engine_connect_tcp(int fd, const char *host, int port, struct sockaddr *sa, int salen, FDCB *cb, void *data, int timeout)
{
	fd_entry *fde = &fd_table[fd];

	SetCallback(fde->connect, cb, data);
	fde->ip.sin_family = AF_INET;
	fde->ip.sin_port = htons(port);

	if ((sa != NULL) && (bind(fd, sa, salen) == -1)) {
		engine_connect_callback(fd, ENGINE_ERR_BIND);
		return;
	}

	if (!inetpton(host, &fde->ip.sin_addr.s_addr)) {
		Debug((DEBUG_DEBUG, "engine_connect_tcp() to %s:%d submitting DNS lookup", host, port));
		fde->dns_query = (DNSQuery *)MyMalloc(sizeof(DNSQuery));
		fde->dns_query->ptr = fde;
		fde->dns_query->callback = engine_connect_dns_callback;
		dns_gethost(host, fde->dns_query);
	}
	else {
		Debug((DEBUG_DEBUG, "engine_connect_tcp() attempting connection on %s:%d", host, port));
		engine_set_timeout(fd, timeout * 1000, engine_connect_timeout, NULL);
		engine_connect_try(fd, NULL, ENGINE_OK);
	}
}

void engine_preinit()
{
	add_event("engine_check_timeouts", engine_check_timeouts, NULL, 1, 1);
}
