/*
 * RageIRCd: an advanced Internet Relay Chat daemon (ircd).
 * (C) 2000-2005 the RageIRCd Development Team, all rights reserved.
 *
 * This software is free, licensed under the General Public License.
 * Please refer to doc/LICENSE and doc/README for further details.
 *
 * $Id: m_oper.c,v 1.68.2.2 2005/01/15 23:53:31 amcwilliam Exp $
 */

#include "config.h"
#include "struct.h"
#include "common.h"
#include "sys.h"
#include "numeric.h"
#include "msg.h"
#include "channel.h"
#include "h.h"
#include "memory.h"
#include "hook.h"
#include "modules.h"
#include "xmode.h"
#include <time.h>
#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>

Hook *h_post_oper = NULL;

struct error_msg {
	int numeric;
	char *msg;
};
static struct error_msg errors[] = {
	{ ERR_NOOPERHOST, "NO CONFIGURATION" },
	{ ERR_NOOPERHOST, "USERHOST MISMATCH" },
	{ ERR_PASSWDMISMATCH, "AUTHENTICATION FAILURE" }
};

Module MOD_HEADER(m_oper) = {
	"m_oper",
	"/OPER command",
	6, "$Revision: 1.68.2.2 $"
};

int MOD_LOAD(m_oper)()
{
	if ((h_post_oper = register_hook(&MOD_HEADER(m_oper), "h_post_oper")) == NULL) {
		return MOD_FAILURE;
	}
	if (register_command(&MOD_HEADER(m_oper), &CMD_OPER, m_oper) == NULL) {
		return MOD_FAILURE;
	}
	return MOD_SUCCESS;
}

int MOD_UNLOAD(m_oper)()
{
	return MOD_SUCCESS;
}

/*
 * m_oper
 *	parv[0] = sender prefix
 *	parv[1] = login
 *	parv[2] = password
 */
int m_oper(aClient *cptr, aClient *sptr, int parc, char *parv[])
{
	dlink_node *node;
	ConfigItem_oper *oper = NULL;
	int old = 0, send_error = 0;
	struct error_msg *error = NULL;
	char *from;
	HookData hdata = HOOKDATA_INIT;

	if (!MyClient(sptr)) {
		return 0;
	}
	if (parc < 3 || *parv[2] == '\0') {
		send_me_numeric(sptr, ERR_NEEDMOREPARAMS, "OPER");
		return 0;
	}
	if (HasMode(sptr, UMODE_OPER)) {
		send_me_numericNA(sptr, RPL_YOUREOPER);
		return 0;
	}

	from = make_user_host(sptr->username, sptr->host);
	DLINK_FOREACH_DATA(conf_oper_list.head, node, oper, ConfigItem_oper) {
		if (oper->item.temp || irccmp(oper->name, parv[1])) {
			continue;
		}
		if (!conf_check_from(&oper->from, from)) {
			send_error = 1;
			continue;
		}
		if (!check_auth(oper->auth, parv[2])) {
			send_error = 2;
			continue;
		}
		break;
	}
	if (oper == NULL) {
		error = &errors[send_error];
		ircdlog(LOG_OPER, "failed oper attempt by %s [%s]", get_client_name(cptr, FALSE),
			error->msg);

		send_me_numericNA(sptr, error->numeric);
		if (GeneralConfig.failed_oper_notice) {
			sendto_realops_lev(SPY_LEV, "Failed OPER attempt by %s (%s@%s) [%s]", sptr->name,
				sptr->username, MaskedHost(sptr), error->msg);
		}
		return 0;
	}

	ircdlog(LOG_OPER, "%s successfully opered as %s", get_client_name(sptr, FALSE), oper->name);

	attach_oper(sptr, oper);
	attach_class(sptr);

#ifdef USE_THROTTLE
	throttle_remove(sptr->hostip);
#endif

	old = (sptr->umode & ALL_UMODES);

	OPAddFlag(sptr, oper->flags);
	if (OPHasFlag(sptr, OFLAG_SADMIN)) {
		AddMode(sptr, UMODE_SADMIN);
	}
	if (OPHasFlag(sptr, OFLAG_RSTAFF)) {
		AddMode(sptr, UMODE_RSTAFF);
	}
	if (OPHasFlag(sptr, OFLAG_NETADMIN) && GeneralConfig.enable_netadmins) {
		AddMode(sptr, UMODE_NETADMIN);
	}
	if (OPHasFlag(sptr, OFLAG_ADMIN)) {
		AddMode(sptr, UMODE_ADMIN);
	}
	if (OPHasFlag(sptr, OFLAG_FNOTICE)) {
		AddMode(sptr, UMODE_FNOTICE);
	}
	if (OPHasFlag(sptr, OFLAG_LCLICONN)) {
		AddMode(sptr, UMODE_LCLICONN);
	}
	if (OPHasFlag(sptr, OFLAG_GCLICONN)) {
		AddMode(sptr, UMODE_GCLICONN);
	}
	AddMode(sptr, AUTOOPER_UMODES);

	send_umode_out(cptr, sptr, old);

	Count.oper++;
	dlink_add(&oper_list, sptr);

	send_me_numericNA(sptr, RPL_YOUREOPER);

	sendto_realops("%s (%s@%s) is now %s [%s]", sptr->name, sptr->username, MaskedHost(sptr),
		HasMode(sptr, UMODE_NETADMIN) ? "a Network Administrator" :
		HasMode(sptr, UMODE_ADMIN) ? "a Server Administrator" : "an Operator", oper->name);
	if (HasMode(sptr, UMODE_NETADMIN)) {
		sendto_serv_msg_butone(NULL, &me, &CMD_GLOBOPS, ":%s (%s@%s) is now a "
			"Network Administrator", sptr->name, sptr->username, MaskedHost(sptr));
	}

	if (!HasVhost(sptr)) {
		strncpyzt(sptr->user->maskedhost, maskme(sptr), HOSTLEN + 1);
	}

	if (OPHasFlag(sptr, OFLAG_OPERMOTD)) {
		send_message_file(sptr, opermotd);
	}

	hdata.sptr = sptr;
	hdata.v = (void *)oper;
	hook_run(h_post_oper, &hdata);

	return 0;
}
