/*
 * RageIRCd: an advanced Internet Relay Chat daemon (ircd).
 * (C) 2000-2005 the RageIRCd Development Team, all rights reserved.
 *
 * This software is free, licensed under the General Public License.
 * Please refer to doc/LICENSE and doc/README for further details.
 *
 * $Id: conf2.h,v 1.122.2.2 2005/02/11 21:53:49 amcwilliam Exp $
 */

#ifndef __conf2_include__
#define __conf2_include__ 

#include "setup.h"
#include "dlink.h"
#include <time.h>
#include <sys/types.h>
#include <netinet/in.h>

struct dns_query;
struct Client;
struct listener;

#define MAXCONFHOSTS 15

typedef struct _configfile ConfigFile;
typedef struct _configentry ConfigEntry;
typedef struct _configunknown ConfigUnknown;
typedef struct _configdirective ConfigDirective;
typedef struct _configblock ConfigBlock;
typedef struct _configoption ConfigOption;
typedef struct _configauth ConfigAuth;
typedef struct _configfrom ConfigFrom;
typedef struct _configitem ConfigItem;

typedef struct _configitem_servinfo ConfigItem_servinfo;
typedef struct _configitem_admin ConfigItem_admin;
typedef struct _configitem_class ConfigItem_class;
typedef struct _configitem_allow ConfigItem_allow;
typedef struct _configitem_oper ConfigItem_oper;
typedef struct _configitem_link ConfigItem_link;
typedef struct _configitem_listen ConfigItem_listen;
typedef struct _configitem_include ConfigItem_include;

typedef struct _configitem_network ConfigItem_network;
typedef struct _configitem_masking ConfigItem_masking;
typedef struct _configitem_flood ConfigItem_flood;
typedef struct _configitem_general ConfigItem_general;

struct _configfile {
	ConfigFile *next;
	char *filename;
	ConfigEntry *entries;
};

struct _configentry {
	ConfigEntry *entries, *next, *prev;
	ConfigFile *file;
	int varlinenum, sectlinenum;
	char *varname;
	char *vardata;
};

struct _configunknown {
	ConfigEntry *entry;
	void *item;
	short type;
};

struct _configdirective {
	char *directive;
	void *(*parse_func)(ConfigEntry *);
	int (*test_func)(ConfigFile *, void *);
	void (*free_func)(void *);
};

struct _configblock {
	ConfigFile *file;
	ConfigDirective *dir;
	void *item;
};

struct _configoption {
	char *token;
	unsigned int flag;
};

struct _configauth {
	char *string;
	short type;
	char *(*func)(char *, char *, short);
};

struct _configfrom {
	char *hosts[MAXCONFHOSTS + 1];
	int host_count;
};

struct _configitem {
	unsigned temp : 1;
	unsigned perm : 1;
	int refcount; /* How many times this item is referenced */
};

struct _configitem_servinfo {
	char *name;
	char *desc;
#ifdef USE_OPENSSL
	char *ssl_certificate;
	char *ssl_private_key;
#endif
	char *default_bind_ip;
	unsigned int identity;
	char *kline_address;
	int max_clients;
	unsigned int logs;
	unsigned hub : 1;
};

struct _configitem_admin {
	char *name;
	char *desc;
	char *email;
};

struct _configitem_class {
	ConfigItem item;
	char *name;
	time_t ping_time;
	int max_clients;
	long sendq_length;
	int clients;
};

struct _configitem_allow {
	ConfigItem item;
	char *hostname;
	char *ipaddr;
	int port;
	int max_per_ip;
	ConfigItem_class *class;
	ConfigAuth *auth;
	char *redir_serv;
	int redir_port;
	char *spoof_mask;
	unsigned int flags;
};

struct _configitem_oper {
	ConfigItem item;
	char *name;
	ConfigAuth *auth;
	ConfigFrom from;
	ConfigItem_class *class;
	char *join_on_oper;
	unsigned int flags;
};

struct _configitem_link {
	ConfigItem item;
	char *servername;
	char *host;
	char *bind_ip;
	ConfigAuth *auth;
	int port;
	ConfigItem_class *class;
	unsigned int flags;
	int servers;
	struct in_addr ip;
	time_t next_connect;
	struct dns_query *dns_query;
};

struct _configitem_listen {
	ConfigItem item;
	char *ipaddr;
	int port;
	unsigned int flags;
	struct listener *l;
};

struct _configitem_include {
	char *filename;
	unsigned loaded : 1;
};

struct _configitem_network {
	char *name;
	char *name_005;
	char *kline_address;
	char *services_server;
	char *stats_server;
	int max_link_depth;
};

struct _configitem_masking {
	char *netadmin_mask;
	char *admin_mask;
	char *oper_mask;
	char *user_mask_prefix;
	long mask_key1;
	long mask_key2;
	long mask_key3;
};

struct _configitem_flood {
	time_t knock_delay;
	time_t accept_notice_time;
	time_t antispam_quit_msg_time;
	unsigned anti_nick_flood : 1;
	unsigned anti_away_flood : 1;
	time_t pace_wait_simple;
	time_t pace_wait_intense;
#ifdef USE_THROTTLE
	time_t min_connect_time;
	int max_connect_count;
	unsigned throttle_rejected_clients : 1;
#endif
	int user_recvq_limit;
	time_t min_join_part_time;
	int max_join_part_count;
	time_t spambot_squelch_time;
	unsigned increase_oper_recvq : 1;
};

struct _configitem_general {
	int max_chans_per_user;
	time_t connauth_timeout;
	time_t statsfile_save_freq;
	unsigned int modes_on_connect;
	char *join_on_connect;
	int compression_level;
	time_t auto_connect_freq;
	time_t ts_delta_warn;
	time_t ts_delta_max;
	int max_accept;
	int max_dccallow;
	int max_bans;
	time_t default_kline_time;
	int max_watch;
	int max_kills;
	int max_targets;
	int max_who_replies;

	unsigned short_motd : 1;
	unsigned hide_super_servers : 1;
	unsigned check_identd : 1;
	unsigned show_headers : 1;
	unsigned failed_oper_notice : 1;
	unsigned enable_knock : 1;
	unsigned custom_quit_msgs : 1;
	unsigned show_invisible_lusers : 1;
	unsigned flatten_links : 1;
	unsigned enable_map : 1;
	unsigned spy_notices : 1;
	unsigned resolve_hostnames : 1;
	unsigned custom_channels : 1;
	unsigned show_cliconn_quit_msgs : 1;
	unsigned no_oper_accept : 1;
	unsigned enable_netadmins : 1;
	unsigned restrict_chan_override : 1;
	unsigned allow_fake_channels : 1;
	unsigned ignore_remote_motd : 1;
	unsigned ignore_remote_rules : 1;
	unsigned ignore_remote_stats : 1;
	unsigned no_mixed_versions : 1;
};

enum {
	CONF_SERVINFO,
	CONF_ADMIN,
	CONF_CLASS,
	CONF_ALLOW,
	CONF_OPER,
	CONF_LINK,
	CONF_NETWORK,
	CONF_MASKING,
	CONF_FLOOD,
	CONF_GENERAL
};

enum {
	CONF_FALLBACK = -2,
	CONF_FAILURE = -1,
	CONF_NOMATCH,
	CONF_SUCCESS
};

#define LOG_DEFAULT			0x0001
#define LOG_ERROR			0x0002
#define LOG_KILL			0x0004
#define LOG_CLIENT			0x0008
#define LOG_SERVER			0x0010
#define LOG_OPER			0x0020
#define LOG_DEBUG			0x0040
#define LOG_OVERRIDE			0x0080

#define ALLOW_HOSTNAME_AT		0x0001
#define ALLOW_IPADDR_AT			0x0002
#define ALLOW_KLINEEXEMPT		0x0004
#define ALLOW_NOTILDE			0x0008
#define ALLOW_NEEDIDENTD		0x0010
#ifdef USE_THROTTLE
#define ALLOW_THROTTLEEXEMPT		0x0020
#endif

#define LINK_AUTOCONNECT		0x0001
#define LINK_COMPRESS			0x0002
#ifdef USE_OPENSSL
#define LINK_SECURE			0x0004
#endif

#define LISTEN_BOUND			0x0001
#define LISTEN_CLIENTONLY		0x0002
#define LISTEN_SERVERONLY		0x0004
#ifdef USE_OPENSSL
#define LISTEN_SECURE			0x0008
#endif

extern dlink_list conf_file_list;
extern dlink_list conf_block_list;
extern dlink_list conf_unknown_list;

extern ConfigItem_network NetworkConfig, tmpNetworkConfig;
extern ConfigItem_masking MaskingConfig, tmpMaskingConfig;
extern ConfigItem_flood FloodConfig, tmpFloodConfig;
extern ConfigItem_general GeneralConfig, tmpGeneralConfig;

extern ConfigItem_servinfo *ServerInfo;
extern ConfigItem_admin *AdminInfo;
extern ConfigItem_class *DefaultClass;

extern dlink_list conf_class_list;
extern dlink_list conf_allow_list;
extern dlink_list conf_oper_list;
extern dlink_list conf_super_list;
extern dlink_list conf_link_list;
extern dlink_list conf_listen_list;
extern dlink_list conf_include_list;
#ifndef STATIC_MODULES
extern dlink_list conf_modulefile_list;
extern dlink_list conf_modulepath_list;
#endif

extern ConfigDirective conf_directives[];
extern ConfigOption servinfo_logs[];
extern ConfigOption oper_can_perform[];
extern ConfigOption oper_can_see[];

extern void *parse_servinfo(ConfigEntry *);
extern void *parse_admin(ConfigEntry *);
extern void *parse_class(ConfigEntry *);
extern void *parse_allow(ConfigEntry *);
extern void *parse_oper(ConfigEntry *);
extern void *parse_super(ConfigEntry *);
extern void *parse_link(ConfigEntry *);
extern void *parse_listen(ConfigEntry *);
extern void *parse_restrict(ConfigEntry *);
extern void *parse_kill(ConfigEntry *);
extern void *parse_include(ConfigEntry *);
extern void *parse_modules(ConfigEntry *);
extern void *parse_network(ConfigEntry *);
extern void *parse_masking(ConfigEntry *);
extern void *parse_flood(ConfigEntry *);
extern void *parse_general(ConfigEntry *);

extern int test_servinfo(ConfigFile *, void *);
extern int test_class(ConfigFile *, void *);
extern int test_allow(ConfigFile *, void *);
extern int test_oper(ConfigFile *, void *);
extern int test_link(ConfigFile *, void *);
extern int test_network(ConfigFile *, void *);
extern int test_masking(ConfigFile *, void *);
extern int test_flood(ConfigFile *, void *);
extern int test_general(ConfigFile *, void *);

extern void free_servinfo(void *);
extern void free_class(void *);
extern void free_allow(void *);
extern void free_oper(void *);
extern void free_link(void *);
extern void free_network(void *);
extern void free_masking(void *);
extern void free_flood(void *);
extern void free_general(void *);

extern ConfigItem_class *find_class(char *);
extern ConfigItem_oper *find_oper(char *);
extern char *find_super(char *);
extern ConfigItem_link *find_link(char *, char *);
extern ConfigItem_listen *find_listen(char *, int);

extern void report(short, char *, ...);
extern void init_conf();
extern void conf_rehash();
extern int conf_check_from(ConfigFrom *, char *);

extern int attach_allow(struct Client *, ConfigItem_allow *);
extern void detach_allow(struct Client *);
extern void attach_class(struct Client *);
extern void detach_class(struct Client *);
extern void attach_oper(struct Client *, ConfigItem_oper *);
extern void detach_oper(struct Client *);
extern void attach_link(struct Client *, ConfigItem_link *);
extern void detach_link(struct Client *);
extern void attach_listen(struct listener *, ConfigItem_listen *);
extern void detach_listen(struct listener *);
extern void detach_confs(struct Client *);

enum {
#ifdef USE_OPENSSL
	AUTH_RIPEMD_160 = 0,
	AUTH_SHA1_160,
	AUTH_MD5_128,
	AUTH_MD5_56,
#else
	AUTH_MD5_56 = 0,
#endif
	AUTH_DES_56,
	AUTH_PLAINTEXT
};

ConfigAuth *parse_auth(ConfigEntry *, char *);
void destroy_auth(ConfigAuth *);
int check_auth(ConfigAuth *, char *);
void make_auth_passwd(struct Client *, char *, char *);

#define cDupString(xx, yy)	if ((xx) != NULL) MyFree((xx)); DupString((xx), (yy))
#define cMyFree(xx)		if ((xx) != NULL) MyFree((xx)); (xx) = NULL

#define EACH_ENTRY(xx, yy)	(xx) = (yy)->entries; (xx) != NULL; (xx) = (xx)->next

/* Check we have a block name, or return NULL */
#define BLOCK_NAME(xx, yy, zz)	if (BadPtr((xx)->vardata)) { report(0, "%s:%d: %s: missing %s", \
					(xx)->file->filename, (xx)->varlinenum, (yy), (zz)); \
				return NULL; }

/* Check block has entries, or return NULL */
#define BLOCK_ENTRIES(xx, yy)	if ((xx)->entries == NULL) { report(0, "%s:%d: %s: empty block", \
					(xx)->file->filename, (xx)->varlinenum, (yy)); \
				return NULL; }

/* Check section has entries, or continue */
#define SECTION(xx, yy)		if ((xx)->entries == NULL) { report(0, "%s:%d: %s: empty section", \
					(xx)->file->filename, (xx)->varlinenum, (yy)); \
				continue; }

/* Check variable, or continue */
#define VARIABLE(xx, yy)	if (BadPtr((xx)->varname)) { report(0, "%s:%d: %s: missing variable name", \
					(xx)->file->filename, (xx)->varlinenum, (yy)); \
				continue; }

/* Check parameter, or continue */
#define PARAMETER(xx, yy)	if (BadPtr((xx)->vardata)) { report(0, "%s:%d: %s: missing variable parameter", \
					(xx)->file->filename, (xx)->varlinenum, (yy)); \
				continue; }

#define UNKNOWN_VAR(xx, yy)	report(0, "%s:%d: %s: unknown variable %s", (xx)->file->filename, \
					(xx)->varlinenum, (yy), (xx)->varname)

#define UNKNOWN_VARNA(xx)	report(0, "%s:%d: unknown variable %s", (xx)->file->filename, \
					(xx)->varlinenum, (xx)->varname)

#endif /* __conf2_include__ */
