\name{gnlmix}
\title{Generalized Nonlinear Regression with a Random Parameter}
\alias{gnlmix}
\usage{
gnlmix(y=NULL, distribution="normal", mixture="normal",
	random=NULL, nest=NULL, mu=NULL, shape=NULL, linear=NULL,
	pmu=NULL, pshape=NULL, pmix=NULL, delta=1, common=FALSE,
	envir=parent.frame(), print.level=0, typsiz=abs(p),
	ndigit=10, gradtol=0.00001, stepmax=10*sqrt(p%*%p), steptol=0.00001,
	iterlim=100, fscale=1, eps=1.0e-4, points=5, steps=10)
}
\description{
\code{gnlmix} fits user-specified nonlinear regression equations to one
or both parameters of the common one and two parameter distributions.
One parameter of the location regression is random with some
specified mixing distribution. 

It is recommended that initial estimates for \code{pmu} and
\code{pshape} be obtained from \code{gnlr}.

These nonlinear regression models must be supplied as formulae where
parameters are unknowns. (See \code{\link[rmutil]{finterp}}.)
}
\arguments{
\item{y}{A response vector of uncensored data, a two column matrix
for binomial data, or an object of class, \code{response} (created by
\code{\link[rmutil]{restovec}}) or \code{repeated} (created by
\code{\link[rmutil]{rmna}} or \code{\link[rmutil]{lvna}}). If the
\code{repeated} data object contains more than one response variable,
give that object in \code{envir} and give the name of the response
variable to be used here.}
\item{distribution}{The distribution for the response: binomial, beta
binomial, double binomial, mult(iplicative) binomial,
Poisson, negative binomial, double Poisson, mult(iplicative) Poisson,
gamma count, Consul generalized Poisson, logarithmic series,
geometric, normal, inverse Gauss, logistic, exponential, gamma,
Weibull, extreme value, Cauchy, Pareto, Laplace, Levy, beta,
simplex, or two-sided power. (For definitions of distributions, see
the corresponding [dpqr]distribution help.)}
\item{mixture}{The mixing distribution for the random parameter:
normal, Cauchy, logistic, Laplace, inverse Gauss, gamma, inverse gamma,
Weibull, beta, simplex, or two-sided power. The first four have zero location
parameter, the next three have unit location parameter, and the last
two have location parameter set to 0.5.}
\item{random}{The name of the random parameter in the \code{mu} formula.}
\item{nest}{The variable classifying observations by the unit upon
which they were observed. Ignored if \code{y} or \code{envir} has class,
\code{response} or \code{repeated}.}
\item{mu}{A user-specified formula containing named unknown
parameters, giving the regression equation for the location
parameter. This may contain the keyword, \code{linear} referring to a
linear part.}
\item{shape}{A user-specified formula containing named unknown
parameters, giving the regression equation for the shape
parameter. This may contain the keyword, \code{linear} referring to a
linear part. If nothing is supplied, this parameter is taken to
be constant. This parameter is the logarithm of the usual one.}
\item{linear}{A formula beginning with ~ in W&R notation, specifying
the linear part of the regression function for the location parameter
or list of two such expressions for the location and/or shape parameters.}
\item{pmu}{Vector of initial estimates for the location parameters.
These must be supplied either in their order of appearance in the formula
or in a named list.}
\item{pshape}{Vector of initial estimates for the shape parameters.
These must be supplied either in their order of appearance in the
expression or in a named list.}
\item{pmix}{Initial estimate for the logarithm of the dispersion
parameter of the mixing distribution.}
\item{delta}{Scalar or vector giving the unit of measurement (always
one for discrete data) for each response value, set to unity by
default. For example, if a response is measured to two decimals,
\code{delta=0.01}. If the response is transformed, this must be multiplied by
the Jacobian. The transformation cannot contain unknown
parameters. For example, with a log transformation,
\code{delta=1/y}. (The delta values for the censored response are
ignored.)}
\item{common}{If TRUE, the formulae with unknowns for the location and
shape have names in common. All parameter estimates must
be supplied in \code{pmu}.}
\item{envir}{Environment in which model formulae are to be
interpreted or a data object of class, \code{repeated}, \code{tccov},
or \code{tvcov}; the name of the response variable should be given in
\code{y}. If \code{y} has class \code{repeated}, it is used as
the environment.}
\item{eps}{Precision of the Romberg integration.}
\item{steps}{For the Romberg integration, the maximum number of steps,
by default set to 10.}
\item{points}{For the Romberg integration, the number of extrapolation
points so that 2*points is the order of integration, by default set to 5;
points=2 is Simpson's rule.}
\item{others}{Arguments controlling \code{\link{nlm}}.}
}
\value{
A list of class \code{gnlm} is returned that contains all of the
relevant information calculated, including error codes.
}
\author{J.K. Lindsey}
\seealso{
\code{\link[growth]{carma}}, \code{\link[rmutil]{finterp}},
\code{\link[growth]{elliptic}}, \code{\link[repeated]{glmm}},
\code{\link[repeated]{gnlmm}}, \code{\link[gnlm]{gnlr}},
\code{\link[repeated]{hnlmix}}, \code{\link[repeated]{kalseries}},
\code{\link[gnlm]{nlr}}, \code{\link[nls]{nls}}.
}
\examples{
library(growth)
dose <- c(9,12,4,9,11,10,2,11,12,9,9,9,4,9,11,9,14,7,9,8)
#y <- rgamma(20,shape=2+0.3*dose,scale=2)+rep(rnorm(4,0,4),rep(5,4))
y <- c(8.674419, 11.506066, 11.386742, 27.414532, 12.135699,  4.359469,
       1.900681, 17.425948,  4.503345,  2.691792,  5.731100, 10.534971,
      11.220260,  6.968932,  4.094357, 16.393806, 14.656584,  8.786133,
      20.972267, 17.178012)
resp <- restovec(matrix(y, nrow=4, byrow=TRUE), name="y")
reps <- rmna(resp, tvcov=tvctomat(matrix(dose, nrow=4, byrow=TRUE), name="dose"))

# same linear normal model with random normal intercept fitted four ways
elliptic(reps, model=~dose, preg=c(0,0.6), pre=4)
glmm(y~dose, nest=individuals, data=reps)
gnlmm(reps, mu=~dose, pmu=c(8.7,0.25), psh=3.5, psd=3)
gnlmix(reps, mu=~a+b*dose+rand, random="rand", pmu=c(8.7,0.25),
	pshape=3.44, pmix=2.3)

# gamma model with log link and random normal intercept fitted three ways
glmm(y~dose, family=Gamma(link=log), nest=individuals, data=reps, points=8)
gnlmm(reps, distribution="gamma", mu=~exp(a+b*dose), pmu=c(2,0.03),
	psh=1, psd=0.3)
gnlmix(reps, distribution="gamma", mu=~exp(a+b*dose+rand), random="rand",
	pmu=c(2,0.04), pshape=1, pmix=-2)

# gamma model with log link and random gamma mixtures
gnlmix(reps, distribution="gamma", mixture="gamma",
	mu=~exp(a*rand+b*dose), random="rand", pmu=c(2,0.04),
	pshape=1.24, pmix=3.5)
gnlmix(reps, distribution="gamma", mixture="gamma",
	mu=~exp(a+b*dose)*rand, random="rand", pmu=c(2,0.04),
	pshape=1.24, pmix=2.5)
}
\keyword{models}
