% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/proteinToX.R
\name{proteinToTranscript}
\alias{proteinToTranscript}
\title{Map protein-relative coordinates to positions within the transcript}
\usage{
proteinToTranscript(x, db, id = "name", idType = "protein_id")
}
\arguments{
\item{x}{\code{IRanges} with the coordinates within the protein(s). The
object has also to provide some means to identify the protein (see
details).}

\item{db}{\code{EnsDb} object to be used to retrieve genomic coordinates of
encoding transcripts.}

\item{id}{\code{character(1)} specifying where the protein identifier can be
found. Has to be either \code{"name"} or one of \code{colnames(mcols(prng))}.}

\item{idType}{\code{character(1)} defining what type of IDs are provided. Has to
be one of \code{"protein_id"} (default), \code{"uniprot_id"} or \code{"tx_id"}.}
}
\value{
\code{IRangesList}, each element being the mapping results for one of the input
ranges in \code{x}. Each element is a \code{IRanges} object with the positions within
the encoding transcript (relative to the start of the transcript, which
includes the 5' UTR). The transcript ID is reported as the name of each
\code{IRanges}. The \code{IRanges} can be of length > 1 if the provided
protein identifier is annotated to more than one Ensembl protein ID (which
can be the case if Uniprot IDs are provided). If the coordinates can not be
mapped (because the protein identifier is unknown to the database) an
\code{IRanges} with negative coordinates is returned.

The following metadata columns are available in each \code{IRanges} in the result:
\itemize{
\item \code{"protein_id"}: the ID of the Ensembl protein for which the within-protein
coordinates were mapped to the genome.
\item \code{"tx_id"}: the Ensembl transcript ID of the encoding transcript.
\item \code{"cds_ok"}: contains \code{TRUE} if the length of the CDS matches the length
of the amino acid sequence and \code{FALSE} otherwise.
\item \code{"protein_start"}: the within-protein sequence start coordinate of the
mapping.
\item \code{"protein_end"}: the within-protein sequence end coordinate of the mapping.
}
}
\description{
\code{proteinToTranscript} maps protein-relative coordinates to positions within
the encoding transcript. Note that the returned positions are relative to
the complete transcript length, which includes the 5' UTR.

Similar to the \code{\link[=proteinToGenome]{proteinToGenome()}} function, \code{proteinToTranscript} compares
for each protein whether the length of its sequence matches the length of
the encoding CDS and throws a warning if that is not the case. Incomplete
3' or 5' CDS of the encoding transcript are the most common reasons for a
mismatch between protein and transcript sequences.
}
\details{
Protein identifiers (supported are Ensembl protein IDs or Uniprot IDs) can
be passed to the function as \code{names} of the \code{x} \code{IRanges} object, or
alternatively in any one of the metadata columns (\code{mcols}) of \code{x}.
}
\note{
While mapping of Ensembl protein IDs to Ensembl transcript IDs is 1:1, a
single Uniprot identifier can be annotated to several Ensembl protein IDs.
\code{proteinToTranscript} calculates in such cases transcript-relative
coordinates for each annotated Ensembl protein.
}
\examples{

library(EnsDb.Hsapiens.v86)
## Restrict all further queries to chromosome x to speed up the examples
edbx <- filter(EnsDb.Hsapiens.v86, filter = ~ seq_name == "X")

## Define an IRange with protein-relative coordinates within a protein for
## the gene SYP
syp <- IRanges(start = 4, end = 17)
names(syp) <- "ENSP00000418169"
res <- proteinToTranscript(syp, edbx)
res
## Positions 4 to 17 within the protein span are encoded by the region
## from nt 23 to 64.

## Perform the mapping for multiple proteins identified by their Uniprot
## IDs.
ids <- c("O15266", "Q9HBJ8", "unexistant")
prngs <- IRanges(start = c(13, 43, 100), end = c(21, 80, 100))
names(prngs) <- ids

res <- proteinToTranscript(prngs, edbx, idType = "uniprot_id")

## The result is a list, same length as the input object
length(res)
names(res)

## No protein/encoding transcript could be found for the last one
res[[3]]

## The first protein could be mapped to multiple Ensembl proteins. The
## region within all transcripts encoding the region in the protein are
## returned
res[[1]]

## The result for the region within the second protein
res[[2]]
}
\seealso{
Other coordinate mapping functions: \code{\link{cdsToTranscript}},
  \code{\link{genomeToProtein}},
  \code{\link{genomeToTranscript}},
  \code{\link{proteinToGenome}},
  \code{\link{transcriptToCds}},
  \code{\link{transcriptToGenome}},
  \code{\link{transcriptToProtein}}
}
\author{
Johannes Rainer
}
