#include "qutdes.h"

//For a given in buffer, out buffer and table this function maps bits 
//which are true given that the entries in the table show the new 
//arrangement of the bits from their initial order
void QutDES::MapTrueBits(
    const unsigned char * lpMap, unsigned int cbMap, 
    const unsigned char * lpIn, unsigned char * const lpOut)
{
    unsigned char chBitIndex;

    //translate bits according to table
	for (chBitIndex=0; chBitIndex<cbMap; chBitIndex++)
    {
        if (GETBIT(lpIn, lpMap[chBitIndex]))
        {
            SETBIT(lpOut, chBitIndex);
        }
    }

}// MapTrueBits

//For a given key, create the permuted K+ based on table PC1. </P>
void QutDES::CreateKeyPlus(
    const unsigned char Key[8], unsigned char KeyPlus[7])
{    
    static unsigned char tblPC1[] =
    {
        56, 48, 40, 32, 24, 16,  8,
         0, 57, 49, 41, 33, 25, 17, 
         9,  1, 58, 50, 42, 34, 26, 
        18, 10,  2, 59, 51, 43, 35, 
        62, 54, 46, 38, 30, 22, 14, 
         6, 61, 53, 45, 37, 29, 21, 
        13,  5, 60, 52, 44, 36, 28, 
        20, 12,  4, 27, 19, 11,  3
    };

    memset(KeyPlus, 0, 7);
    MapTrueBits(tblPC1, sizeof(tblPC1), Key, KeyPlus);

}// CreateKeyPlus

//Create the Initial permuation from the message
void QutDES::CreateInitialPermutation(
    const unsigned char Message[8], unsigned char IP[8])
{
    static unsigned char tblIP[] =
    {
        57, 49, 41, 33, 25, 17,  9,  1, 
        59, 51, 43, 35, 27, 19, 11,  3,
        61, 53, 45, 37, 29, 21, 13,  5,
        63, 55, 47, 39, 31, 23, 15,  7,
        56, 48, 40, 32, 24, 16,  8,  0,
        58, 50, 42, 34, 26, 18, 10,  2,
        60, 52, 44, 36, 28, 20, 12,  4,
        62, 54, 46, 38, 30, 22, 14,  6
    };

    memset(IP, 0, 8);
    MapTrueBits(tblIP, sizeof(tblIP), Message, IP);

}// CreateInitialPermutation

//Create array of 16 subkeys from the key
void QutDES::CreateSubKeys(
    const unsigned char Key[8], unsigned char SubKeys[16][6])
{
    //table which dicates the number of left shifts required for the pairing
	static unsigned char lShifts[] = 
    {
        1, 1, 2, 2, 2, 2, 2, 2, 1, 2, 2, 2, 2, 2, 2, 1
    };

    //table used to permutate 
	static unsigned char tblPC2[] =
    {
        13, 16, 10, 23,  0,  4, 
         2, 27, 14,  5, 20,  9, 
        22, 18, 11,  3, 25,  7, 
        15,  6, 26, 19, 12,  1, 
        40, 51, 30, 36, 46, 54, 
        29, 39, 50, 44, 32, 47, 
        43, 48, 38, 55, 33, 52, 
        45, 41, 49, 35, 28, 31
    };

    long lPair[17][2];
    int n;
    unsigned char bytePair[7];
    unsigned char KeyPlus[7];

    //create permutated key (K+)
	CreateKeyPlus(Key, KeyPlus);

    //Obtain first pair in long format (C0 in [0][0], D0 in [0][1])
	lPair[0][0] = (KeyPlus[0] << 20) | (KeyPlus[1] << 12) | 
        (KeyPlus[2] << 4) | ((KeyPlus[3] & 0xF0) >> 4);

    lPair[0][1] = ((KeyPlus[3] & 0x0F) << 24) | (KeyPlus[4] << 16) | 
        (KeyPlus[5] << 8) | KeyPlus[6] ;

    //Create pairings (Cx, Dx) by left shifting bits from the previous pair
    //by either one or two shifts (depending on lShifts table)
	for (n = 1; n < 17; n++)
    {
        if (lShifts[n-1] == 2)
        {
            Rotate26BitsLeft2(lPair[n-1][0], lPair[n][0]);
            Rotate26BitsLeft2(lPair[n-1][1], lPair[n][1]);
        }
        else
        {
            Rotate26BitsLeft1(lPair[n-1][0], lPair[n][0]);
            Rotate26BitsLeft1(lPair[n-1][1], lPair[n][1]);
        }
    }

    //clear the subkeys array
	memset(SubKeys, 0, 16*6);
    
    //create the subkeys array using the pairing permutated with table PC2
    for (n=0; n < 16; n++)
    {
        //convert long pair into byte pair
		GetKeyPair(bytePair, lPair[n+1][0], lPair[n+1][1]);

        //permutate the bits from the byte pair into the subkey
		MapTrueBits(tblPC2, sizeof(tblPC2), bytePair, SubKeys[n]);
    }
    
}// CreateSubKeys

//Obtain nibble from the SBox table
unsigned char QutDES::GetSBoxNibble(
    unsigned char sbox, unsigned int idxSBox)
{
    //SBox used for to convert the 48 bits into the 32 bit output.  The 48 
    //bits are used as 8 bunches of 6 bits.  The first index used for the 
    //table is dictated by the number of the 6bit bunch (ie the first 6bits 
    //refer to tblSBox[0], the last refers to tblSBox[7]).  Then the msb and 
    //lsb are combined together to obtain the row within the SBox.  Bits
    //2 to 5 are used to obtain the column within the sbox.
    //If the FIRST 6 bytes are 010001 then the reference to the SBox will 
    //be: tblSBox[0][1][8]
    //            |  |  |
    //         FIRST |  |
    //               |  |
    //   MSB and LSB 01 |
    //                  |
    //     Bits 2 to 5 1000 (8)
	static unsigned char tblSBox[8][4][16] = 
    {
        {
            {14,4,13,1,2,15,11,8,3,10,6,12,5,9,0,7},
            {0,15,7,4,14,2,13,1,10,6,12,11,9,5,3,8},
            {4,1,14,8,13,6,2,11,15,12,9,7,3,10,5,0},
            {15,12,8,2,4,9,1,7,5,11,3,14,10,0,6,13}
        },

        { 
            {15,1,8,14,6,11,3,4,9,7,2,13,12,0,5,10},
            {3,13,4,7,15,2,8,14,12,0,1,10,6,9,11,5},
            {0,14,7,11,10,4,13,1,5,8,12,6,9,3,2,15},
            {13,8,10,1,3,15,4,2,11,6,7,12,0,5,14,9}
        },

        {
            {10,0,9,14,6,3,15,5,1,13,12,7,11,4,2,8},
            {13,7,0,9,3,4,6,10,2,8,5,14,12,11,15,1},
            {13,6,4,9,8,15,3,0,11,1,2,12,5,10,14,7},
            {1,10,13,0,6,9,8,7,4,15,14,3,11,5,2,12}
        },

        {
            {7,13,14,3,0,6,9,10,1,2,8,5,11,12,4,15},
            {13,8,11,5,6,15,0,3,4,7,2,12,1,10,14,9},
            {10,6,9,0,12,11,7,13,15,1,3,14,5,2,8,4},
            {3,15,0,6,10,1,13,8,9,4,5,11,12,7,2,14}
        },

        {
            {2,12,4,1,7,10,11,6,8,5,3,15,13,0,14,9},
            {14,11,2,12,4,7,13,1,5,0,15,10,3,9,8,6},
            {4,2,1,11,10,13,7,8,15,9,12,5,6,3,0,14},
            {11,8,12,7,1,14,2,13,6,15,0,9,10,4,5,3}
        },

        {
            {12,1,10,15,9,2,6,8,0,13,3,4,14,7,5,11},
            {10,15,4,2,7,12,9,5,6,1,13,14,0,11,3,8},
            {9,14,15,5,2,8,12,3,7,0,4,10,1,13,11,6},
            {4,3,2,12,9,5,15,10,11,14,1,7,6,0,8,13}
        },

        {
            {4,11,2,14,15,0,8,13,3,12,9,7,5,10,6,1},
            {13,0,11,7,4,9,1,10,14,3,5,12,2,15,8,6},
            {1,4,11,13,12,3,7,14,10,15,6,8,0,5,9,2},
            {6,11,13,8,1,4,10,7,9,5,0,15,14,2,3,12}
        },

        {
            {13,2,8,4,6,15,11,1,10,9,3,14,5,0,12,7},
            {1,15,13,8,10,3,7,4,12,5,6,11,0,14,9,2},
            {7,11,4,1,9,12,14,2,0,6,10,13,15,3,5,8},
            {2,1,14,7,4,10,8,13,15,12,9,0,3,5,6,11}
        }
    };

    unsigned char row, col, nibble;

    row = ((sbox & 0x20) >> 4) | (sbox & 0x01);
    col = ((sbox & 0x1E) >> 1);

    nibble = tblSBox[idxSBox][row][col];

    return nibble;

}// GetSBoxNibble

//Function for encrypting each RIGHT half of each block of encryption 
//data by expanding the in data from 32 bits to 48; exclusively ORing (XOR)
//these 48 bits with the Subkey, passing this data through the SBOX to 
//return back to 32 bits and applying a final table permutation
void QutDES::MapThroughSBox(
    const unsigned char Right[4], const unsigned char SubKey[6], 
    unsigned char output32[4])
{
    //selection table used to expand 32 bits to 48 prior to XORing with the
    //subkey
	static unsigned char tblEBitSelection[48] =
    {
        31,  0,  1,  2,  3,  4,
         3,  4,  5,  6,  7,  8,
         7,  8,  9, 10, 11, 12,
        11, 12, 13, 14, 15, 16,
        15, 16, 17, 18, 19, 20, 
        19, 20, 21, 22, 23, 24, 
        23, 24, 25, 26, 27, 28,
        27, 28, 29, 30, 31,  0
    };

    //permutation table applied after running through the sbox
	static unsigned char tblfP[32] = 
    {
        15,  6, 19, 20, 
        28, 11, 27, 16, 
         0, 14, 22, 25, 
         4, 17, 30,  9, 
         1,  7, 23, 13, 
        31, 26,  2,  8, 
        18, 12, 29,  5, 
        21, 10,  3, 24
    };

    unsigned char output48[6];    
    unsigned char sbox32[4];
    unsigned char nSboxIndex;
    int n;

    //clear the 32 bits 
	memset(sbox32, 0, 4);

    //create E(n) via using Ebit selection permutation table (effectively
    //creating 48 bits from 32)
	memset(output48, 0, 6);
    MapTrueBits(tblEBitSelection, sizeof(tblEBitSelection), Right, 
        output48);

    //Exclusively or (XOR) 48 bit output with the subkey
	for(n = 0; n < 6; n++)
    {
        output48[n] ^= SubKey[n];
    }

    //Obtain each six bits in turn (eight on total) to create the 32b output.
    //Each six bit set represents a location in the sbox table.  This is 
    //defined as the first and last bit for the row and the middle for bits 
    //comprise the column.  These six bits are then used to obtain a nibble
    //which are catenated into the resulting 32 bit output buffer
    nSboxIndex = (output48[0] & 0xFC) >> 2;
    sbox32[0] |= GetSBoxNibble(nSboxIndex, 0) << 4;        
    
    nSboxIndex = ((output48[0] & 0x03) << 4) | ((output48[1] & 0xFC) >> 4);
    sbox32[0] |= GetSBoxNibble(nSboxIndex, 1);

    nSboxIndex = ((output48[1] & 0x0F) << 2) | ((output48[2] & 0xC0) >> 6);
    sbox32[1] |= GetSBoxNibble(nSboxIndex, 2) << 4;    

    nSboxIndex = (output48[2] & 0x3F);
    sbox32[1] |= GetSBoxNibble(nSboxIndex, 3);

    nSboxIndex = (output48[3] & 0xFC) >> 2;
    sbox32[2] |= GetSBoxNibble(nSboxIndex, 4) << 4;

    nSboxIndex = ((output48[3] & 0x03) << 4) | ((output48[4] & 0xFC) >> 4);
    sbox32[2] |= GetSBoxNibble(nSboxIndex, 5);

    nSboxIndex = ((output48[4] & 0x0F) << 2) | ((output48[5] & 0xC0) >> 6);
    sbox32[3] |= GetSBoxNibble(nSboxIndex, 6) << 4;

    nSboxIndex = (output48[5] & 0x3F);
    sbox32[3] |= GetSBoxNibble(nSboxIndex, 7);

    //Apply the permutation to the sbox to obtain the output
	memset(output32, 0, 4);
    MapTrueBits(tblfP, sizeof(tblfP), sbox32, output32);

}// MapThroughSBox
   
//Encode 64 bits of data
void QutDES::Encode64Bits(
    const unsigned char in[8], unsigned char out[8], 
    const unsigned char SubKeys[16][6], int nEncrypt)
{    
    //Table used for applying final permutation to the 64 bit block of data
	static unsigned char tblIPmin1[64] = 
    {                             
        39, 7, 47, 15, 55, 23, 63, 31,
        38, 6, 46, 14, 54, 22, 62, 30,
        37, 5, 45, 13, 53, 21, 61, 29,
        36, 4, 44, 12, 52, 20, 60, 28,
        35, 3, 43, 11, 51, 19, 59, 27,
        34, 2, 42, 10, 50, 18, 58, 26,
        33, 1, 41,  9, 49, 17, 57, 25,
        32, 0, 40,  8, 48, 16, 56, 24
    };

    //table use to create the initial permutation
	static unsigned char tblIP[] =
    {
        57, 49, 41, 33, 25, 17,  9,  1, 
        59, 51, 43, 35, 27, 19, 11,  3,
        61, 53, 45, 37, 29, 21, 13,  5,
        63, 55, 47, 39, 31, 23, 15,  7,
        56, 48, 40, 32, 24, 16,  8,  0,
        58, 50, 42, 34, 26, 18, 10,  2,
        60, 52, 44, 36, 28, 20, 12,  4,
        62, 54, 46, 38, 30, 22, 14,  6
    };

    //buffers
	unsigned char data[17][8];
    unsigned char output32[4];   
    unsigned char swapped64[8];    
    int n, n2;

	//create the initial permutation from the message
    memset(data, 0, 8);
    MapTrueBits(tblIP, sizeof(tblIP), in, data[0]);

    //Encryption
	if (nEncrypt != 0)
    {
        for(n = 1; n < 17; n++)
        {        
            //Ln = R(n-1)
            *(unsigned long*)(data[n]) = *(unsigned long*)(&(data[n-1][4]));

            //Call function to pass key through the sbox f(R(n-1), K(n))
            MapThroughSBox(&(data[n-1][4]), SubKeys[n-1], output32);

            //Rn = L(n-1) ^ f(R(n-1), K(n))
            for(n2 = 0; n2 < 4; n2++)
            {
                data[n][n2+4] = data[n-1][n2] ^ output32[n2];
            }          
        }
    }

    //Decryption
    else 
    {
        for(n = 1; n < 17; n++)
        {        
            //Ln = R(n-1)
            *(unsigned long*)(data[n]) = *(unsigned long*)(&(data[n-1][4]));

            //Call function to pass key through the sbox f(R(n-1), K(16-n))
            MapThroughSBox(&(data[n-1][4]), SubKeys[16-n], output32);

            //Rn = L(n-1) ^ f(R(n-1), K(n))
            for(n2 = 0; n2 < 4; n2++)
            {
                data[n][n2+4] = data[n-1][n2] ^ output32[n2];
            }          
        }
   		
    }

    //reverse order of bits in the last 64 bit block L(16)R(16) -> R(16)L(16)
	for(n = 0; n < 4; n++)
    {
        swapped64[n] = data[16][n+4];
        swapped64[n+4] = data[16][n];
    }

    //Apply the FINAL permutation to obtain the output
	memset(out, 0, 8);
    MapTrueBits(tblIPmin1, sizeof(tblIPmin1), swapped64, out);

}// Encode64Bits

//Performs single/triple DES encryption/decryption on a given buffer 
//and places the results in an out buffer.  Triple DES is just DES done 
//two times with two keys in a particular order. 
//If padding is enabled (nPadding != 0) then the PKCS#5 standard will be
//used for padding.  When data is padded in this way, the EXACT size of the
//original data will be obtained when the cryptogram is decrypted rather 
//than returning a multiple of eight - the only shortcoming with using this
//padding method is that an extra eight bytes will be added to the 
//cryptogram when encrypted if the size of the original data ends is a 
//multiple of eight.  For further details see the PKCS#5 comments near the 
//start of this function
bool QutDES::DoDES(const unsigned char * lpIn, 
    unsigned long cbIn, bool bEncrypt)
{        
    bool bBuffTooSmall;
    unsigned char SubKeys[2][16][6]; 
    unsigned char in[8], pad, bIsPadded;
    unsigned long cbBlock, cbOut, cbCrypt, n;
    unsigned long cbRemaining;
    int nEncryptMode;
    unsigned char *lpTmp1, *lpTmp2;
    unsigned char * lpOut;

    //Check the punter hasn't passed us anything duff.
	assert(lpIn);
    assert(cbIn);

    //Calculate the size of the out buffer based on the encryption method and
    //whether padding is used.
    if (bEncrypt)
    {
        if (GetCBC())
        {
            cbOut = CRYPTSIZE(GetPadding(), 1, cbIn);
            cbCrypt = cbOut - 8;
        }
        else
        {
            cbOut = CRYPTSIZE(GetPadding(), 0, cbIn);
            cbCrypt = cbOut;
        }
    }
    else
    {        
        cbOut = TEXTSIZEFROMCRYPT(GetPadding(), GetCBC(), cbIn);
        cbCrypt = cbOut;
    }

    //Check the output buffer is large enough for the results
	bBuffTooSmall = GetOutputSize() < cbOut;    

    //If we don't own the output buffer and it is too small then return false
    //to the punter, otherwise resize the output buffer to ensure it is large 
    //enough
	if (m_cbOutput < cbOut)
    {
        if (!GetOutputIsOwned())
        {            
            return false;
        }
        else
        {
            ResizeOutput(cbOut);			
        }
    }

    //Cache the output size.
	m_cbResult = cbOut;

    //Create the 16 subkeys for each key
	CreateSubKeys(GetKey1(), SubKeys[0]);

    if (GetIterations() > 1)
    {
        CreateSubKeys(GetKey2(), SubKeys[1]);
    }

    //Point at the beginning of the output buffer.
	lpOut = m_lpOutput;

    //Create the initialisation vector (IV) used for the first block if we 
    //are using CBC
	if (GetCBC()) 
    {
        if (bEncrypt)
        {
            //Create the random part of the IV
			for (n = 0; n < 8; n++)
            {
                in[n] = (rand() >> 4) % 256;
            }   

            //Encrypt the random values to get the IV
            nEncryptMode = bEncrypt == true;

            for(n = 0; n < GetIterations(); n++)
            {
                //Encode/decode the 64 bit block with the key
                Encode64Bits(in, lpOut, 
                    (const unsigned char (*)[6])SubKeys[n % 2 == 0 ? 0 : 1], 
                    nEncryptMode);

                //swap the encryption mode
                nEncryptMode = !nEncryptMode;
            }

            //Point to the next block
            lpOut += 8;
        }

        //If we are decrypting
        else 
        {
            //Point past the IV
			lpIn += 8;
            cbIn -= 8;
        }
    }

    //Encrypt/decrypt each 64 bit block 
	cbBlock = 0;
    cbRemaining = cbIn;
    for(cbBlock = 0; cbBlock < cbCrypt; cbBlock += 8)
    {
        //If we are encrypting with padding then the last iteration of this 
        //loop may end on an exact eight bytes boundary of the in block.  If 
        //this occurs then we need to populate an extra byte for padding to
        //the PKCS#5 standard
		if (cbRemaining == 0)
        {
            memset(in, 8, 8);            
        }

        //If we have less than eight bytes remaining on the in buffer then
        //popluate the remaining characters with bytes as according to the
        //PKCS#5 standard
		else if (cbRemaining < 8)
        {
            memset(in+cbRemaining, 8-cbRemaining, 8-cbRemaining);
            memcpy(in, lpIn+cbBlock, cbRemaining);
        }
        else
        {
            memcpy(in, lpIn+cbBlock, 8);
        }

        //For encryption, Chain Block Cipher (CBC) works like this:
        //Where 'C' is cryptogram and 'M' is unencrypted message.
        //1.  Create a random block (Initialisation Vector - IV) seed with 
        //    key to get C 0->7.
        //2.  XOR C 0->7 with M 0->7, then encrypt to get C 8->15.
        //3.  XOR C 8->15 with M 8->15, then encrypt to get C 16->23.     
        //n.  XOR C n->n+7 with M n+8->n+15, then encrypt to get C n+8->n+15
		if (GetCBC() && bEncrypt)
        {
            lpTmp1 = lpOut+cbBlock-8;            

            for (n = 0; n < 8; n++)
            {
                in[n] ^= lpTmp1[n];
            }
        }

        //Perform the encryption/decryption the specified number of times
        nEncryptMode = bEncrypt == true;

		for(n = 0; n < GetIterations(); n++)
        {
            //Encode/decode the 64 bit block with the key
            Encode64Bits(in, lpOut+cbBlock, 
                (const unsigned char (*)[6])SubKeys[n % 2 == 0 ? 0 : 1], 
                nEncryptMode);

            //Swap the encryption mode
			nEncryptMode = !nEncryptMode;

            //Take copy of results for next iteration
			memcpy(in, lpOut+cbBlock, 8);
        }

        //For decryption, Chain Block Cipher (CBC) works like this:
        //Where 'C' is cryptogram and 'M' is unencrypted message.
        //For decryption:
        //1.  Decrypt C 0->7 for M 0->7.
        //2.  Decrypt C 8->15, exclusively OR with C 0->7 to get M 8->15.
        //3.  Decrypt C 16->23, exclusively OR with C 8->15 to get M 16->23.
        //4.  Decrypt C n+8->n+15, exclusively OR with C n->n+7 to get 
        //    M n+8-n+15.
		if (GetCBC() && !bEncrypt)
        {
            lpTmp1 = ((unsigned char*)lpIn)+cbBlock-8;
            lpTmp2 = lpOut+cbBlock;

            for (n = 0; n < 8; n++)
            {
                lpTmp2[n] ^= lpTmp1[n];
            }
        }

        cbRemaining-=8;
    }

    //If we are decrypting with padding then we need to calculate the actual
    //size of the message
	if (!bEncrypt && GetPadding())
    {		
        //First ensure PKCS#5 padding is enabled
		pad = lpOut[cbOut-1];

        //Pad value must be 8 or less inorder to be valid
		if (pad <= 8)
        {
            bIsPadded = 1;

            //Check that each byte in padding part of the byte are all set to 
            //the value of the pad.
			for(n=cbOut-2; n>(cbOut-pad); n--)
            {
                if (lpOut[n] != pad)
                {
                    bIsPadded = 0;
                    break;
                }
            }

            //If this is still true then we ARE using PKCS#5 padding
			if (bIsPadded)
            {
                //Take the last byte which dictates the number of spare bytes 
                //used then subtract this from the number of bytes in the out 
                //message
                if (pad)
                {
                    memset(&lpOut[cbOut - pad], 0, pad);
		            m_cbResult -= pad;
                }
            }
        }
    }

    return true;

}// DoDES