// Copyright 2016 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "third_party/blink/renderer/core/editing/editor.h"

#include "third_party/blink/renderer/core/clipboard/system_clipboard.h"
#include "third_party/blink/renderer/core/editing/commands/editor_command.h"
#include "third_party/blink/renderer/core/editing/testing/editing_test_base.h"
#include "third_party/blink/renderer/core/html/forms/html_input_element.h"
#include "third_party/blink/renderer/platform/testing/unit_test_helpers.h"

namespace blink {

class EditorTest : public EditingTestBase {
 public:
  void TearDown() override {
    SystemClipboard::GetInstance().WritePlainText(String(""));
    EditingTestBase::TearDown();
  }

  void ExecuteCopy() {
    Editor& editor = GetDocument().GetFrame()->GetEditor();
    editor.CreateCommand("Copy").Execute();
    test::RunPendingTasks();
  }
};

TEST_F(EditorTest, copyGeneratedPassword) {
  // Checks that if the password field has the value generated by Chrome
  // (HTMLInputElement::shouldRevealPassword will be true), copying the field
  // should be available.
  const char* body_content = "<input type='password' id='password'></input>";
  SetBodyContent(body_content);

  HTMLInputElement& element =
      ToHTMLInputElement(*GetDocument().getElementById("password"));

  const String kPasswordValue = "secret";
  element.focus();
  element.setValue(kPasswordValue);
  element.SetSelectionRange(0, kPasswordValue.length());

  Editor& editor = GetDocument().GetFrame()->GetEditor();
  EXPECT_FALSE(editor.CanCopy());

  element.SetShouldRevealPassword(true);
  EXPECT_TRUE(editor.CanCopy());
}

TEST_F(EditorTest, CopyVisibleSelection) {
  const char* body_content = "<input id=hiding value=HEY>";
  SetBodyContent(body_content);

  HTMLInputElement& text_control =
      ToHTMLInputElement(*GetDocument().getElementById("hiding"));
  text_control.select();

  ExecuteCopy();

  const String copied = SystemClipboard::GetInstance().ReadPlainText();
  EXPECT_EQ("HEY", copied);
}

TEST_F(EditorTest, DontCopyHiddenSelections) {
  const char* body_content =
      "<input type=checkbox id=checkbox>"
      "<input id=hiding value=HEY>";
  SetBodyContent(body_content);

  HTMLInputElement& text_control =
      ToHTMLInputElement(*GetDocument().getElementById("hiding"));
  text_control.select();

  HTMLInputElement& checkbox =
      ToHTMLInputElement(*GetDocument().getElementById("checkbox"));
  checkbox.focus();

  ExecuteCopy();

  const String copied = SystemClipboard::GetInstance().ReadPlainText();
  EXPECT_TRUE(copied.IsEmpty()) << copied << " was copied.";
}

}  // namespace blink
