// Copyright 2017 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "third_party/blink/renderer/core/css/offscreen_font_selector.h"

#include "build/build_config.h"
#include "third_party/blink/renderer/core/css/css_segmented_font_face.h"
#include "third_party/blink/renderer/core/css/css_value_list.h"
#include "third_party/blink/renderer/core/css/resolver/style_resolver.h"
#include "third_party/blink/renderer/core/frame/local_frame.h"
#include "third_party/blink/renderer/core/frame/settings.h"
#include "third_party/blink/renderer/core/frame/use_counter.h"
#include "third_party/blink/renderer/core/loader/frame_loader.h"
#include "third_party/blink/renderer/platform/fonts/font_cache.h"
#include "third_party/blink/renderer/platform/fonts/font_selector_client.h"
#include "third_party/blink/renderer/platform/fonts/simple_font_data.h"
#include "third_party/blink/renderer/platform/wtf/text/atomic_string.h"

namespace blink {

OffscreenFontSelector::OffscreenFontSelector(ExecutionContext* context)
    : execution_context_(context) {
  FontCache::GetFontCache()->AddClient(this);
}

OffscreenFontSelector::~OffscreenFontSelector() = default;

void OffscreenFontSelector::UpdateGenericFontFamilySettings(
    const GenericFontFamilySettings& settings) {
  generic_font_family_settings_ = settings;
}

void OffscreenFontSelector::RegisterForInvalidationCallbacks(
    FontSelectorClient* client) {}

void OffscreenFontSelector::UnregisterForInvalidationCallbacks(
    FontSelectorClient* client) {}

scoped_refptr<FontData> OffscreenFontSelector::GetFontData(
    const FontDescription& font_description,
    const AtomicString& family_name) {
  if (CSSSegmentedFontFace* face =
          font_face_cache_.Get(font_description, family_name)) {
    return face->GetFontData(font_description);
  }

  AtomicString settings_family_name = FamilyNameFromSettings(
      generic_font_family_settings_, font_description, family_name);
  if (settings_family_name.IsEmpty())
    return nullptr;

  return FontCache::GetFontCache()->GetFontData(font_description,
                                                settings_family_name);
}

void OffscreenFontSelector::WillUseFontData(
    const FontDescription& font_description,
    const AtomicString& family,
    const String& text) {
  CSSSegmentedFontFace* face = font_face_cache_.Get(font_description, family);
  if (face)
    face->WillUseFontData(font_description, text);
}

void OffscreenFontSelector::WillUseRange(
    const FontDescription& font_description,
    const AtomicString& family,
    const FontDataForRangeSet& range_set) {
  CSSSegmentedFontFace* face = font_face_cache_.Get(font_description, family);
  if (face)
    face->WillUseRange(font_description, range_set);
}

bool OffscreenFontSelector::IsPlatformFamilyMatchAvailable(
    const FontDescription& font_description,
    const AtomicString& passed_family) {
  AtomicString family = FamilyNameFromSettings(generic_font_family_settings_,
                                               font_description, passed_family);
  if (family.IsEmpty())
    family = passed_family;
  return FontCache::GetFontCache()->IsPlatformFamilyMatchAvailable(
      font_description, family);
}

void OffscreenFontSelector::ReportNotDefGlyph() const {}

void OffscreenFontSelector::FontCacheInvalidated() {
  font_face_cache_.IncrementVersion();
}

void OffscreenFontSelector::FontFaceInvalidated() {
  FontCacheInvalidated();
}

void OffscreenFontSelector::Trace(blink::Visitor* visitor) {
  visitor->Trace(execution_context_);
  visitor->Trace(font_face_cache_);
  FontSelector::Trace(visitor);
}

}  // namespace blink
