/***************************************************************************
**
** Copyright (C) 2012 Research In Motion
** Contact: http://www.qt-project.org/legal
**
** This file is part of the QtBluetooth module of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:LGPL$
** Commercial License Usage
** Licensees holding valid commercial Qt licenses may use this file in
** accordance with the commercial license agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and Digia.  For licensing terms and
** conditions see http://qt.digia.com/licensing.  For further information
** use the contact form at http://qt.digia.com/contact-us.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Digia gives you certain additional
** rights.  These rights are described in the Digia Qt LGPL Exception
** version 1.1, included in the file LGPL_EXCEPTION.txt in this package.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 3.0 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU General Public License version 3.0 requirements will be
** met: http://www.gnu.org/copyleft/gpl.html.
**
**
** $QT_END_LICENSE$
**
****************************************************************************/

#include "qrfcommserver.h"
#include "qrfcommserver_p.h"
#include "qbluetoothsocket.h"
#include "qbluetoothsocket_p.h"
#include "qbluetoothlocaldevice.h"

#include <QSocketNotifier>

#include <QCoreApplication>

QT_BEGIN_NAMESPACE_BLUETOOTH

extern QHash<QRfcommServerPrivate*, int> __fakeServerPorts;

QRfcommServerPrivate::QRfcommServerPrivate()
    : socket(0),maxPendingConnections(1),securityFlags(QBluetooth::NoSecurity)
{
    socket = new QBluetoothSocket(QBluetoothSocket::RfcommSocket);
    ppsRegisterControl();
}

QRfcommServerPrivate::~QRfcommServerPrivate()
{
    Q_Q(QRfcommServer);
    if (socket)
        delete socket;
    q->close();
    __fakeServerPorts.remove(this);
    ppsUnregisterControl(this);
}

void QRfcommServerPrivate::controlReply(ppsResult result)
{
    Q_Q(QRfcommServer);

    if (result.msg == QStringLiteral("register_server")) {
        qBBBluetoothDebug() << "SPP: Server registration succesfull";

    } else if (result.msg == QStringLiteral("get_mount_point_path")) {
        qBBBluetoothDebug() << "SPP: Mount point for server" << result.dat.first();

        int socketFD = ::open(result.dat.first().toStdString().c_str(), O_RDWR | O_NONBLOCK);
        if (socketFD == -1) {
            qWarning() << Q_FUNC_INFO << "RFCOMM Server: Could not open socket FD" << errno;
        } else {
            socket->setSocketDescriptor(socketFD, QBluetoothSocket::RfcommSocket,
                                           QBluetoothSocket::ConnectedState);
            socket->connectToService(QBluetoothAddress(nextClientAddress), m_uuid);
            activeSockets.append(socket);
            socket = new QBluetoothSocket(QBluetoothSocket::RfcommSocket, this);
            socket->setSocketState(QBluetoothSocket::ListeningState);
            emit q->newConnection();
        }
    }
}

void QRfcommServerPrivate::controlEvent(ppsResult result)
{
    if (result.msg == QStringLiteral("service_connected")) {
        qBBBluetoothDebug() << "SPP: Server: Sending request for mount point path";
        qBBBluetoothDebug() << result.dat;
        for (int i=0; i<result.dat.size(); i++) {
            qBBBluetoothDebug() << result.dat.at(i);
        }

        if (result.dat.contains(QStringLiteral("addr")) && result.dat.contains(QStringLiteral("uuid"))
                && result.dat.contains(QStringLiteral("subtype"))) {
            nextClientAddress = result.dat.at(result.dat.indexOf(QStringLiteral("addr")) + 1);
            m_uuid = QBluetoothUuid(result.dat.at(result.dat.indexOf(QStringLiteral("uuid")) + 1));
            int subtype = result.dat.at(result.dat.indexOf(QStringLiteral("subtype")) + 1).toInt();
            qBBBluetoothDebug() << "Getting mount point path" << m_uuid << nextClientAddress<< subtype;
            ppsSendControlMessage("get_mount_point_path", 0x1101, m_uuid, nextClientAddress,
                                  m_serviceName, this, BT_SPP_SERVER_SUBTYPE);
        } else {
            qWarning() << Q_FUNC_INFO << "address not specified in service connect reply";
        }
    }
}

void QRfcommServer::close()
{
    Q_D(QRfcommServer);
    if (!d->socket) {
        // there is no way to propagate the error to user
        // so just ignore the problem ;)
        return;
    }
    d->socket->close();
    d->socket = 0;
    ppsSendControlMessage("deregister_server", 0x1101, d->m_uuid, QString(), QString(), 0);
    // force active object (socket) to run and shutdown socket.
    qApp->processEvents(QEventLoop::ExcludeUserInputEvents);
}

bool QRfcommServer::listen(const QBluetoothAddress &address, quint16 port)
{
    Q_UNUSED(address)
    Q_D(QRfcommServer);
    // listen has already been called before
    if (d->socket->state() == QBluetoothSocket::ListeningState)
        return true;

    //We can not register an actual Rfcomm port, because the platform does not allow it
    //but we need a way to associate a server with a service

    if (port == 0) { //Try to assign a non taken port id
        for (int i=1; ; i++){
            if (__fakeServerPorts.key(i) == 0) {
                port = i;
                break;
            }
        }
    }

    if (__fakeServerPorts.key(port) == 0) {
        __fakeServerPorts[d] = port;
        qBBBluetoothDebug() << "Port" << port << "registered";
    } else {
        qWarning() << "server with port" << port << "already registered or port invalid";
        return false;
    }

    d->socket->setSocketState(QBluetoothSocket::ListeningState);

    ppsRegisterForEvent(QStringLiteral("service_connected"),d);
    return true;
}

void QRfcommServer::setMaxPendingConnections(int numConnections)
{
    Q_D(QRfcommServer);
    d->maxPendingConnections = numConnections; //Currently not used
}

QBluetoothAddress QRfcommServer::serverAddress() const
{
    Q_D(const QRfcommServer);
    if (d->socket)
        return d->socket->localAddress();
    else
        return QBluetoothAddress();
}

quint16 QRfcommServer::serverPort() const
{
    //Currently we do not have access to the port
    Q_D(const QRfcommServer);
    return __fakeServerPorts.value((QRfcommServerPrivate*)d);
}

bool QRfcommServer::hasPendingConnections() const
{
    Q_D(const QRfcommServer);
    return !d->activeSockets.isEmpty();
}

QBluetoothSocket *QRfcommServer::nextPendingConnection()
{
    Q_D(QRfcommServer);
    if (d->activeSockets.isEmpty())
        return 0;

    return d->activeSockets.takeFirst();
}

void QRfcommServer::setSecurityFlags(QBluetooth::SecurityFlags security)
{
    Q_D(QRfcommServer);
    d->securityFlags = security; //not used
}

QBluetooth::SecurityFlags QRfcommServer::securityFlags() const
{
    Q_D(const QRfcommServer);
    return d->securityFlags; //not used
}

QT_END_NAMESPACE_BLUETOOTH

