/*
 * CP3D.h
 * $Id: CP3D.h,v 1.5 2003/06/24 14:50:02 anxo Exp $
 *
 * Copyright (C) 1999, 2000 Michael Meissner
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * As a special exception to the GPL, the QGLViewer authors (Markus
 * Janich, Michael Meissner, Richard Guenther, Alexander Buck and Thomas
 * Woerner) give permission to link this program with Qt (non-)commercial
 * edition, and distribute the resulting executable, without including
 * the source code for the Qt (non-)commercial edition in the source
 * distribution.
 *
 */

//  Description : Class CP3D
//  Purpose     : Provides  funcionality


#ifndef __CP3D_H
#define __CP3D_H


// System
///////////
#include <math.h>
#ifdef _MSC_VER
#if _MSC_VER >= 1300
#include <iostream>
#endif
#else
#include <iostream.h>
#endif


// Own
///////////
#include "CV3D.h"
#include "CP4D.h"

// forward declarations
/////////////////////////
class CV3D;




/** This class provides a interface to 3D point 
  *
  * @author Michael Meissner
  */
class CP3D 
{   
public:
  static double epsilon;

  /** Default constructor.
   * The default value of the instantiated point will be (0.0,0.0,0.0). */
  CP3D() { m_ard[0] = 0.0;
           m_ard[1] = 0.0;
           m_ard[2] = 0.0; };

  /** Construct new point.
   * The value of the point will be (rdX, rdY, rdZ). */
  CP3D(double rdX, double rdY, double rdZ) { m_ard[0] = rdX;
                                             m_ard[1] = rdY;
		 			     m_ard[2] = rdZ; }

  /** Copy constructor.
   * The parameters will simply be copied. */
  CP3D(const CP3D& Point) { m_ard[0] = Point[0];
                            m_ard[1] = Point[1];
			    m_ard[2] = Point[2]; }


  //////////////////////////
  // OVERLOADED OPERATORS //
  //////////////////////////
 
  /** Cast operator to convert CP3D points to CP4D points.
   * Each component is devided by the fourth component. */
  operator CP4D() const;

  /** Assign one point to another.*/
  const CP3D& operator=(const CP3D&);

  /** Compares to points for being equal.
   * The result will be 'true'(1) if the two point are indentically
   * up to <= CP3D::epsilon for each component. Otherwise 'false'(0) 
   * will be returned. */
  int operator==(const CP3D&) const;

  /** Compares to points for not being equal.
   * Same as operator== but inverted.
   * @see operator==() */
  int operator!=(const CP3D&) const;

  /** Adds a vector to this point. */
  CP3D& operator+=(const CV3D&);

  /** Subtracts a vector from this point. */
  CP3D& operator-=(const CV3D&);
  
  /** Multiplies a point by a vector. */
  CP3D& operator*=(const CV3D&);

  /** Multiplies a point by a scalar. */
  CP3D& operator*=(double);

  /** Divides a point by a scalar. */
  CP3D& operator/=(double);

  /** Adds a vector to a point. */
  CP3D operator+(const CV3D&) const;

  /** Adds a point to a point. */
  CP3D operator+(const CP3D&) const;  // eigentlich nur fuer affine Punkte
  
  /** Substracts a vector from a point. */
  CP3D operator-(const CV3D&) const;

  /** Substracts a point from a point. */
  CV3D operator-(const CP3D&) const ;

  /** Multiplies a point by a vector. */
  CP3D operator*(const CV3D&) const;  // scaling

  /** Multiplies a point by a scalar. */
  CP3D operator*(double) const;

  /** Divides a point by a vector. */
  CP3D operator/(const CV3D&) const;

  /** Divides a point by a scalar. */
  CP3D operator/(double) const;

  /** Returns the i-th component of the point.
   * The index goes from 0 to 2, 0 stands for the
   * x-coordinate, 1 for the y-coordinate and so on. */
  double& operator [] (int i) { return m_ard[i]; };

  /** Same as above but does not alter anything. */
  double operator[](int i) const { return m_ard[i]; };


  /////////////
  // METHODS //
  /////////////
  
  /** Returns the value of the minimal point component. */
  double getMinComponent(void) const    { return m_ard[getMinComponentCoord()]; };
 
  /** Returns the value of the minimal point component. */
  double getAbsMinComponent(void) const { return m_ard[getAbsMinComponentCoord()]; };
 
  /** Returns the value of the maximal point component. */
  double getMaxComponent(void) const    { return m_ard[getMaxComponentCoord()]; };
 
  /** Returns the value of the maximal point component. */
  double getAbsMaxComponent(void) const { return m_ard[getAbsMaxComponentCoord()]; };
 
  /** Returns the coordinate index of the minial point component. */
  int getMinComponentCoord(void) const;

  /** Returns the coordinate index of the minial point component (using fabs). */
  int getAbsMinComponentCoord(void) const;

  /** Returns the coordinate index of the maximum point component. */
  int getMaxComponentCoord(void) const;

  /** Returns the coordinate index of the maximum point component (using fabs). */
  int getAbsMaxComponentCoord(void) const;

  /** Converts a point to a vector.
   * It's implemented as 'get'-method to prevent
   * implicit casting by the compiler. */
  CV3D getCV3D() const;

  /** Returns the x-coordinate of the point. */
  double getX(void) const  { return m_ard[0]; };

  /** Returns the y-coordinate of the point. */
  double getY(void) const  { return m_ard[1]; };

  /** Returns the z-coordinate of the point. */
  double getZ(void) const  { return m_ard[2]; };

  /** Sets the x-coordinate of the point to 'rdX'. */
  void setX(double rdX)    { m_ard[0] = rdX; return; };

  /** Sets the y-coordinate of the point to 'rdX'. */
  void setY(double rdY)    { m_ard[1] = rdY; return; };

  /** Sets the z-coordinate of the point to 'rdX'. */
  void setZ(double rdZ)    { m_ard[2] = rdZ; return; };

  /** Set the values of the point.
   * The value of the point will be (rdX, rdY, rdZ). */
  void setCoord(double rdX, double rdY, double rdZ) { m_ard[0] = rdX; 
                                                      m_ard[1] = rdY;
                                                      m_ard[2] = rdZ; 
                                                      return; };

  /////////////
  // FRIENDS //
  /////////////
  
  /** Returns the affine combination of the points and vectors. */
  friend CP3D AffinComb(const CP3D&, double, const CP3D&);

  /** Returns the affine combination of the points and vectors. */
  friend CP3D AffinComb3(double r, const CP3D& R, 
                         double s, const CP3D& S,
                         double t, const CP3D T) { 
                                  return CP3D(r*R[0] + s*S[0] + t*T[0],
                                              r*R[1] + s*S[1] + t*T[1],
                                              r*R[2] + s*S[2] + t*T[2]); };

  /** Returns the distance between two points. */
  friend double dist(const CP3D&, const CP3D&);

  /** Returns the square of the distance between two points. */
  friend double quaddist(const CP3D&, const CP3D&);

  /** Returns the minimum of all components of two points. */
  friend CP3D Min(const CP3D&, const CP3D&);

  /** Returns the maximum of all components of two points. */
  friend CP3D Max(const CP3D&, const CP3D&);

  /** Returns a point being the result of multiplying a scalar and a point. */
  friend CP3D operator*(double, const CP3D&);

  /** Returns the point in the middle between two points. */
  friend CP3D MidPoint(const CP3D&, const CP3D&);


  // output to stderr
  /////////////////////
  /** Prints a point to the standard output. */
  void print() const;

  /** Same as above. But more useful for streams. */
  friend inline ostream& operator<<(ostream&, const CP3D&);

  /** Reads a point from the given stream. */
  friend inline istream& operator>>(istream&, CP3D&);

protected:
  double m_ard[3];
};



// Function   : operator=
// Parameters : const CP3D& p1
// Purpose    : assign another point to this point
// Comments   : 
inline const CP3D& CP3D::operator=(const CP3D& p1)
/*******************************************************************/
{
  m_ard[0] = p1.m_ard[0];
  m_ard[1] = p1.m_ard[1];
  m_ard[2] = p1.m_ard[2];
  return *this;
}



// Function   : getCV3D
// Parameters : 
// Purpose    : Convert CP3D to CV3D
// Comments   : 
inline CV3D CP3D::getCV3D() const
/*******************************************************************/
{
  return CV3D(m_ard[0], m_ard[1], m_ard[2]);
}



// Function   : operator<<
// Parameters : ostream& s, const CP3D& pnt
// Purpose    : 
// Comments   : 
inline ostream& operator<<(ostream& s, const CP3D& pnt)
/*******************************************************************/
{   
  return s << "(" << pnt.m_ard[0] << "," << pnt.m_ard[1] << "," << pnt.m_ard[2] << ")"; 
}



// Function   : operator>>
// Parameters : istream& s, CP3D& pnt
// Purpose    : 
// Comments   : 
inline istream& operator>>(istream& s, CP3D& pnt)
/*******************************************************************/
{   
  char c;
  return s >> c >> pnt.m_ard[0] >> c >> pnt.m_ard[1] >> c >> pnt.m_ard[2] >> c;
}

#endif
