/*
	Description: patch viewer window

	Author: Marco Costalba (C) 2005-2006

	Copyright: See COPYING file that comes with this distribution

*/
#include <qtextedit.h>
#include <qlineedit.h>
#include <qapplication.h>
#include <qsyntaxhighlighter.h>
#include <qradiobutton.h>
#include "common.h"
#include "git.h"
#include "domain.h"
#include "patchview.h"

// diff types options, corresponds to buttons id
#define DIFF_TO_PARENT	0
#define DIFF_TO_HEAD	1
#define DIFF_TO_SHA		2

class DiffHighlighter : public QSyntaxHighlighter {
public:
	DiffHighlighter(PatchView* p, QTextEdit* te)
	: QSyntaxHighlighter(te), pv(p), combinedLenght(0) {}

	void setCombinedLength(uint cl) { combinedLenght = cl; }
	virtual int highlightParagraph (const QString& text, int) {

		QColor myColor;
		const char firstChar = text[0].latin1();
		switch (firstChar) {
		case '@':
			myColor = Qt::darkMagenta;
			break;
		case '+':
			myColor = Qt::darkGreen;
			break;
		case '-':
			myColor = Qt::red;
			break;
		case 'd':
			if (text.startsWith("diff --git a/"))
				myColor = Qt::darkBlue;
			else if (combinedLenght > 0 && text.startsWith("diff --combined"))
				myColor = Qt::darkBlue;
			break;
		case ' ':
			if (combinedLenght > 0) {
				if (text.left(combinedLenght).contains('+'))
					myColor = Qt::darkGreen;
				else if (text.left(combinedLenght).contains('-'))
					myColor = Qt::red;
			}
			break;
		}
		if (myColor.isValid())
			setFormat(0, text.length(), myColor);

		if (pv->matches.count() > 0) {
			int indexFrom, indexTo;
			if (pv->getMatch(currentParagraph(), &indexFrom, &indexTo)) {
				QFont f = textEdit()->currentFont();
				f.setUnderline(true);
				f.setBold(true);
				if (indexTo == 0)
					indexTo = text.length();
				setFormat(indexFrom, indexTo - indexFrom, f, Qt::blue);
			}
		}
		return 0;
	}
private:
	PatchView* pv;
	uint combinedLenght;
};

PatchView::PatchView(Domain* dm, Git* g, QTextEdit* p, QLineEdit* l, QRadioButton* r) :
		QObject(dm), d(dm), git(g), textEditDiff(p),lineEditDiff(l),radioButtonSha(r) {

	diffHighlighter = new DiffHighlighter(this, textEditDiff);
	seekTarget = diffLoaded = false;
	pickAxeRE.setMinimal(true);
	pickAxeRE.setCaseSensitive(false);

	st = &(d->st);

	connect(lineEditDiff, SIGNAL(returnPressed()),
		   this, SLOT(on_lineEditDiff_returnPressed()));
}

PatchView::~PatchView() {

	git->cancelProcess(proc);
	delete diffHighlighter;
}

void PatchView::clear() {

	matches.clear();
	diffLoaded = false;
	textEditDiff->clear();
	seekTarget = !target.isEmpty();
	partialParagraphs = "";
}

void PatchView::centerOn(const QString& t) {

	target = t;
	seekTarget = !target.isEmpty();
	if (seekTarget)
		centerTarget();
}

void PatchView::on_fileSelected(const QString& text) {
// center diffViewer on selected file name

	bool combined = (st->isMerge() && !st->allMergeFiles());
	QString prefix((combined) ? "diff --combined " : "diff --git a/");
	centerOn(prefix + text);
}

void PatchView::centerTarget() {

	textEditDiff->setCursorPosition(0, 0);
	if (!textEditDiff->find(target, true, true)) // updates cursor position
		return;

	// target found
	seekTarget = false;
	int para, index;
	textEditDiff->getCursorPosition(&para, &index);
	QPoint p = textEditDiff->paragraphRect(para).topLeft();
	textEditDiff->setContentsPos(p.x(), p.y());
	textEditDiff->removeSelection();
}

void PatchView::centerMatch(uint id) {

	if (matches.count() <= id)
		return;

	textEditDiff->setSelection(matches[id].paraFrom, matches[id].indexFrom,
						  matches[id].paraTo, matches[id].indexTo);
}

void PatchView::on_procDataReady(const QString& data) {

	int X = textEditDiff->contentsX();
	int Y = textEditDiff->contentsY();

	bool targetInNewChunk = false;
	if (seekTarget)
		targetInNewChunk = (data.find(target) != -1);

	// QTextEdit::append() adds a new paragraph, i.e. inserts a LF
	// if not already present. For performance reasons we cannot use
	// QTextEdit::text() + QString::append() + QTextEdit::setText()
	// so we append only \n terminating text
	//
	// NOTE: last char of diff data MUST always be '\n' for this to work
	int pos = data.findRev('\n');
	if (pos == -1) {
		partialParagraphs.append(data);
		return;
	}
	textEditDiff->append(partialParagraphs + data.left(pos + 1));
	partialParagraphs = data.right(data.length() - pos - 1);
	if (targetInNewChunk)
		centerTarget();
	else {
		textEditDiff->setContentsPos(X, Y);
		textEditDiff->sync();
	}
}

void PatchView::on_eof() {

	diffLoaded = true;
	computeMatches();
	diffHighlighter->rehighlight();
	centerMatch();
}

int PatchView::doSearch(SCRef txt, int pos) {

	if (isRegExp)
		return pickAxeRE.search(txt, pos);

	return txt.find(pickAxeRE.pattern(), pos, true);
}

void PatchView::computeMatches() {

	matches.clear();
	if (pickAxeRE.isEmpty())
		return;

	SCRef txt = textEditDiff->text();
	int pos, lastPos = 0, lastPara = 0;

	// must be at the end to catch patterns across more the one chunk
	while ((pos = doSearch(txt, lastPos)) != -1) {

		matches.append(MatchSelection());
		MatchSelection& s = matches.last();

		s.paraFrom = txt.mid(lastPos, pos - lastPos).contains('\n');
		s.paraFrom += lastPara;
		s.indexFrom = pos - txt.findRev('\n', pos) - 1; // index starts from 0

		lastPos = pos;
		pos += (isRegExp) ? pickAxeRE.matchedLength() : pickAxeRE.pattern().length();
		pos--;

		s.paraTo = s.paraFrom + txt.mid(lastPos, pos - lastPos).contains('\n');
		s.indexTo = pos - txt.findRev('\n', pos) - 1;
		s.indexTo++; // in QTextEdit::setSelection() indexTo is not included

		lastPos = pos;
		lastPara = s.paraTo;
	}
}

bool PatchView::getMatch(int para, int* indexFrom, int* indexTo) {

	for (uint i = 0; i < matches.count(); i++)
		if (matches[i].paraFrom <= para && matches[i].paraTo >= para) {

			*indexFrom = (para == matches[i].paraFrom) ? matches[i].indexFrom : 0;
			*indexTo = (para == matches[i].paraTo) ? matches[i].indexTo : 0;
			return true;
		}
	return false;
}

void PatchView::on_highlightPatch(const QString& exp, bool re) {

	pickAxeRE.setPattern(exp);
	isRegExp = re;
	if (diffLoaded)
		on_eof();
}

void PatchView::on_lineEditDiff_returnPressed() {

	if (lineEditDiff->text().isEmpty())
		return;

	radioButtonSha->setChecked(true); // could be called by code
	on_buttonGroupDiff_clicked(DIFF_TO_SHA);
}

void PatchView::on_buttonGroupDiff_clicked(int diffType) {

	QString sha;
	switch(diffType) {
	case DIFF_TO_PARENT:
		break;
	case DIFF_TO_HEAD:
		sha = "HEAD";
		break;
	case DIFF_TO_SHA:
		sha = lineEditDiff->text();
		break;
	}
	if (sha == QGit::ZERO_SHA)
		return;

	// check for a ref name or an abbreviated form
	normalizedSha = (sha.length() != 40 && !sha.isEmpty()) ? git->getRefSha(sha) : sha;

	if (normalizedSha != st->diffToSha()) { // avoid looping
		st->setDiffToSha(normalizedSha); // could be empty
		UPDATE_DOMAIN(d);
	}
}

void PatchView::update(bool force) {

	if (st->sha().isEmpty())
		return;

	bool combined = (st->isMerge(true) && !st->allMergeFiles(true));
	bool prevCombined = (st->isMerge(false) && !st->allMergeFiles(false));

	bool isChanged = ((st->sha(true) != st->sha(false)) 	||
			(st->diffToSha(true) != st->diffToSha(false))||
			(combined != prevCombined));

	if (!isChanged && !force)
		return;

	git->cancelProcess(proc);
	clear();

	if (combined) {
		const Rev* r = git->revLookup(st->sha());
		diffHighlighter->setCombinedLength(r->parentsCount());
	} else
		diffHighlighter->setCombinedLength(0);

	if (normalizedSha != st->diffToSha()) {
		if (!st->diffToSha().isEmpty()) {
			lineEditDiff->setText(st->diffToSha());
			on_lineEditDiff_returnPressed();
		} else
			radioButtonSha->setChecked(false);
	}
	proc = git->getDiff(st->sha(), this, st->diffToSha(), combined); // non blocking
}
