/*
	Description: file viewer window

	Author: Marco Costalba (C) 2005-2006

	Copyright: See COPYING file that comes with this distribution

*/
#include <qtextedit.h>
#include <qsyntaxhighlighter.h>
#include <qlistview.h>
#include <qspinbox.h>
#include <qlineedit.h>
#include <qmessagebox.h>
#include <qsettings.h>
#include <qstatusbar.h>
#include <qlabel.h>
#include <qsplitter.h>
#include <qapplication.h>
#include <qeventloop.h>
#include <qcursor.h>
#include <qregexp.h>
#include <qclipboard.h>
#include <qtoolbutton.h>
#include <qtabwidget.h>
#include "mainimpl.h"
#include "git.h"
#include "annotate.h"
#include "listview.h"
#include "fileview.h"
#include "filecontent.h"

#define MAX_LINE_NUM 5

FileView::FileView(MainImpl* mi, Git* g, QWidget* t) :
                   Domain(mi, g), tabPage(t) {

	EM_INIT(exDtorCalled, "Closing file viewer");

	histListView = new ListView(this, git, m()->histListView, true);
	textEditFile = new FileContent(this, git, m()->textEditFile);

	// cannot be set directly in the .ui file
	m()->spinBoxRevision->setSpecialValueText(" ");

	connect(git, SIGNAL(histLoadCompleted()),this, SLOT(on_histLoadCompleted()));
	connect(git, SIGNAL(newHistRevsAdded(const QValueVector<QString>&)),
		   histListView, SLOT(on_newRevsAdded(const QValueVector<QString>&)));

	connect(histListView, SIGNAL(contextMenu(const QString&, int)),
		   d, SLOT(on_contextMenu(const QString&, int)));

	connect(textEditFile, SIGNAL(annotationAvailable(bool)),
		   this, SLOT(on_annotationAvailable(bool)));
	connect(textEditFile, SIGNAL(fileAvailable(bool)),
		   this, SLOT(on_fileAvailable(bool)));
	connect(textEditFile, SIGNAL(revIdSelected(int)),
		   this, SLOT(on_revIdSelected(int)));

	connect(m()->toolButtonCopy, SIGNAL(clicked()),
		this, SLOT(on_toolButtonCopy_clicked()));

	connect(m()->toolButtonShowAnnotate, SIGNAL(toggled(bool)),
		this, SLOT(on_toolButtonShowAnnotate_toggled(bool)));

	connect(m()->toolButtonFindAnnotate, SIGNAL(toggled(bool)),
    		this, SLOT(on_toolButtonFindAnnotate_toggled(bool)));

    	connect(m()->toolButtonRangeFilter, SIGNAL(toggled(bool)),
		this, SLOT(toolButtonRangeFilter_toggled(bool)));

	connect(m()->toolButtonPin, SIGNAL(toggled(bool)),
		this, SLOT(on_toolButtonPin_toggled(bool)));

	connect(m()->spinBoxRevision, SIGNAL(valueChanged(int)),
		this, SLOT(on_spinBoxRevision_valueChanged(int)));
}

FileView::~FileView() {

	EM_RAISE(exDtorCalled);

	delete histListView;
	delete textEditFile;

	QApplication::restoreOverrideCursor();
}

void FileView::clear(bool complete) {

	if (complete) {
		st.clear();
		m()->tabWdg->changeTab(tabPage, "File");
		m()->toolButtonCopy->setEnabled(false);
	}
	histListView->clear();
	textEditFile->clear();

	m()->toolButtonPin->setEnabled(false);
	m()->toolButtonPin->setOn(false); // should not trigger an update
	m()->spinBoxRevision->setEnabled(false);
	m()->spinBoxRevision->setValue(m()->spinBoxRevision->minValue()); // clears the box
}

bool FileView::goToCurrentAnnotation() {

	QListViewItem* item = m()->histListView->currentItem();
	int id = (item) ? item->text(QGit::ANN_ID_COL).toInt() : 0;
	textEditFile->goToAnnotation(id);
	return (id != 0);
}

void FileView::updateSpinBoxValue() {

	QListViewItem* item = m()->histListView->currentItem();
	if (!item || !m()->spinBoxRevision->isEnabled())
		return;

	int id = item->text(QGit::ANN_ID_COL).toInt();
	m()->spinBoxRevision->setValue(id); // triggers on_spinBoxRevision_valueChanged
}

bool FileView::doUpdate() {

	if (st.fileName().isEmpty())
		return false;

	try {
		EM_REGISTER(exDtorCalled);

		if (st.fileName(true) != st.fileName(false)) {

			clear(false); // blocking

			m()->tabWdg->changeTab(tabPage, st.fileName());

			QApplication::restoreOverrideCursor();
			QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));

			m()->statusBar()->message("Retrieving history of '" +
					st.fileName() + "'...");

			git->getFileHistory(st.fileName()); // non blocking
		} else {
			bool found = histListView->update();
			if (found) {
				updateSpinBoxValue();
				m()->statusBar()->message(git->getRevInfo(st.sha(), true));
			}
		}
		if (!m()->toolButtonPin->isOn())
			textEditFile->update();

		EM_REMOVE(exDtorCalled);

		return true; // always accept new state

	} catch (int i) {
		EM_REMOVE(exDtorCalled);

		if (EM_MATCH(i, exDtorCalled, "updating file viewer")) {
			EM_CHECK_PENDING;
			return false;
		}
		const QString info("Exception \'" + EM_DESC(i) + "\' "
				"not handled in file viewer...re-throw");
		dbp("%1", info);
		throw i;
	}
}

// ************************************ SLOTS ********************************

void FileView::on_toolButtonCopy_clicked() {

	textEditFile->copySelection();
}

void FileView::on_toolButtonShowAnnotate_toggled(bool b) {

	textEditFile->setShowAnnotate(b);

	m()->toolButtonFindAnnotate->setEnabled(b);

	if (b && m()->toolButtonFindAnnotate->isOn())
		goToCurrentAnnotation();
}

void FileView::on_toolButtonFindAnnotate_toggled(bool b) {

	if (b)
		goToCurrentAnnotation();
}

void FileView::on_toolButtonPin_toggled(bool b) {
// button is enabled and togglable only if st.sha() is found

	m()->spinBoxRevision->setDisabled(b);

	if (!b) {
		updateSpinBoxValue();
		textEditFile->update(true);
	}
}

void FileView::on_spinBoxRevision_valueChanged(int id) {

	if (id != m()->spinBoxRevision->minValue()) {

		SCRef selRev(histListView->getSha(id));
		if (st.sha() != selRev) { // to avoid looping
			st.setSha(selRev);
			st.setSelectItem(true);
			UPDATE_DOMAIN(d);
		}
	}
}

void FileView::on_histLoadCompleted() {

	QApplication::restoreOverrideCursor();
	histListView->updateIdValues();
	int maxId = m()->histListView->childCount();
	if (maxId == 0) {
		m()->statusBar()->clear();
		return;
	}
	m()->spinBoxRevision->setMaxValue(maxId);
	m()->toolButtonPin->setEnabled(true);
	m()->spinBoxRevision->setEnabled(true);

	UPDATE_DOMAIN(d);

	updateProgressBar(0);
	textEditFile->startAnnotate();
}

void FileView::addAnnotation() {

	if (!m()->toolButtonPin->isOn()			&&
		m()->toolButtonShowAnnotate->isEnabled()&&
		m()->toolButtonShowAnnotate->isOn()) {

		textEditFile->addAnnotation();

		if (m()->toolButtonFindAnnotate->isEnabled() &&
			m()->toolButtonFindAnnotate->isOn())
			goToCurrentAnnotation();
	}
}

void FileView::on_annotationAvailable(bool b) {

	m()->toolButtonRangeFilter->setEnabled(b);

	if (!b && m()->toolButtonRangeFilter->isOn())
		m()->toolButtonRangeFilter->toggle();

	m()->toolButtonShowAnnotate->setEnabled(b);
	m()->toolButtonFindAnnotate->setEnabled(b && m()->toolButtonShowAnnotate->isOn());

	if (b)
		addAnnotation(); // in case annotation arrived after file
}

void FileView::on_fileAvailable(bool b) {

	if (b) { // avoid flickering when disable only temporary
		m()->toolButtonCopy->setEnabled(b);

		// code range is indipendent from annotation
		if (m()->toolButtonRangeFilter->isOn())
			textEditFile->goToRangeStart();

		addAnnotation(); // in case file arrived after annotation
	}
}

void FileView::on_revIdSelected(int id) {

	if (id != 0) {
		if (m()->spinBoxRevision->isEnabled())
			m()->spinBoxRevision->setValue(id);
	}
}

void FileView::toolButtonRangeFilter_toggled(bool b) {

	if (b) {
		if (!textEditFile->annotateAvailable()) {
			dbs("ASSERT in toolButtonRangeFilter_toggled: annotate not available");
			return;
		}
 		if (!m()->textEditFile->hasSelectedText()) {
 			m()->statusBar()->message("Please select some text "
 					"to activate range filter");
 			return;
 		}
 	}
	bool rangeFilterActive = textEditFile->rangeFilter(b);
	filterOnRange(rangeFilterActive);
}

// ******************************* data events ****************************

void FileView::customEvent(QCustomEvent* e) {

	if (e->type() == (int)QGit::ANN_PRG_EV)
		updateProgressBar(((AnnotateProgressEvent*)e)->data());
	else
		Domain::customEvent(e);
}

void FileView::updateProgressBar(int annotatedNum) {

	uint tot = m()->histListView->childCount();
	if (tot == 0)
		return;

	int cc = (annotatedNum * 100) / tot;
	int idx = (annotatedNum * 40) / tot;
	QString head("Annotating '" + st.fileName() + "' [");
	QString tail("] " + QString::number(cc) + " %");
	QString done, toDo;
	done.fill('.', idx);
	toDo.fill(' ', 40 - idx);
	m()->statusBar()->message(head + done + toDo + tail);
}

void FileView::filterOnRange(bool isOn) {
// TODO integrate with mainimpl function

	QListViewItemIterator it(m()->histListView);
	bool evenLine = false;
	int visibleCnt = 0;
	RangeInfo r;
	while (it.current()) {
		ListViewItem* item = static_cast<ListViewItem*>(it.current());

		if (isOn) {
			if (!textEditFile->getRange(item->text(QGit::COMMIT_COL), &r))
				continue;

			if (r.modified) {
				item->setEven(evenLine);
				evenLine = !evenLine;
				visibleCnt++;
			} else
				item->setVisible(false);
		} else {
			item->setEven(evenLine);
			evenLine = !evenLine;
			if (!item->isVisible())
				item->setVisible(true);
		}
		++it;
	}
	if (isOn) {
		const QString tmp(QString("Found %1 matches. Toggle filter "
				"button to remove the filter").arg(visibleCnt));
		d->m()->statusBar()->message(tmp);
	} else {
		m()->histListView->ensureItemVisible(m()->histListView->currentItem());
		d->m()->statusBar()->clear();
	}
}
